// Exception.h: interface for the Exception class.
//
//////////////////////////////////////////////////////////////////////

#if !defined(AFX_EXCEPTION_H__F7EF395C_C6C8_473C_BAF5_1743CDEB9BDC__INCLUDED_)
#define AFX_EXCEPTION_H__F7EF395C_C6C8_473C_BAF5_1743CDEB9BDC__INCLUDED_

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

//////////////////////////////////////////////////////////////////////
// Static definitions related to exceptions are gathered up in the class
// "Exception"
// Don't create "Exception" objects, this class is all static

class Exception 
{
private:
	class Any_ErrorDescr
	{
	public:
		virtual const char* name() = 0;
		char *message;
	};

	class RendezvousErrorDescr : public Any_ErrorDescr
	{
		virtual const char* name() = 0;
	};

	class AcceptErrorDescr : public RendezvousErrorDescr
	{
	public:
		AcceptErrorDescr(char* message);
		virtual const char* name();
	};
	class NotATaskErrorDescr : public RendezvousErrorDescr
	{
	public:
		NotATaskErrorDescr(char* message);
		virtual const char* name();
	};
	class ForeignEntryErrorDescr : public RendezvousErrorDescr
	{
	public:
		ForeignEntryErrorDescr(char* message);
		virtual const char* name();
	};
	class ParameterErrorDescr : public RendezvousErrorDescr
	{
	public:
		ParameterErrorDescr(char* message);
		virtual const char* name();
	};
	class EntryKindErrorDescr : public RendezvousErrorDescr
	{
	public:
		EntryKindErrorDescr(char* message);
		virtual const char* name();
	};
	
	class TaskingErrorDescr : public Any_ErrorDescr
	{
		virtual const char* name() = 0;
	};

	class EntryCallErrorDescr : public TaskingErrorDescr
	{
	public:
		EntryCallErrorDescr(char* message);
		virtual const char* name();
	};
	class ActivationErrorDescr : public TaskingErrorDescr
	{
	public:
		ActivationErrorDescr(char* message);
		virtual const char* name();
	};

	class Any_EscapeDescr
	{
	public:
		virtual const char* name() = 0;
	};
	
	class TerminateEscapeDescr : public Any_EscapeDescr
	{
	public:
		virtual const char* name();
	};

public:
	// When an operation cannot complete normally and cannot return a 
	// reasonable result, some of the task and synchronization operations 
	// throw exceptions
	// A user program can catch exceptions either individually, or as a 
	// group
	// All exceptions are included in the exception type "Any_Error"

	typedef Any_ErrorDescr				*Any_Error;
	
	static const char* name(Any_Error e);
	static char* message(Any_Error e);

	// A task throws a RendezvousError if it cannot perform a rendezvous as 
	// intended due to an inconsistency in the rendezvous parameters
	// This is most likely the result of a programming error
	// The exceptions defined below are included in the exception type
	// "RendezvousError"

	typedef RendezvousErrorDescr		*RendezvousError;
	
	typedef AcceptErrorDescr			*AcceptError;
	typedef NotATaskErrorDescr			*NotATaskError;
	typedef ForeignEntryErrorDescr		*ForeignEntryError;
	typedef ParameterErrorDescr			*ParameterError;
	typedef EntryKindErrorDescr			*EntryKindError;

	static void throw_AcceptError(char* message);
	static void throw_NotATaskError(char* message);
	static void throw_ForeignEntryError(char* message);
	static void throw_ParameterError(char* message);
	static void throw_EntryKindError(char* message);

	// A task throws a TaskingError if a task interaction completes 
	// abnormally
	// This is most likely the result of unexpected task behavior, 
	// especially premature termination or uncoordinated activation
	// The exceptions defined below are included in the exception type
	// "TaskingError"
	
	typedef TaskingErrorDescr			*TaskingError;

	typedef EntryCallErrorDescr			*EntryCallError;
	typedef ActivationErrorDescr		*ActivationError;

	static void throw_EntryCallError(char* message);
	static void throw_ActivationError(char* message);

	// The exception type "Any_Escape" includes all escape exceptions 
	// the tasking implementation uses internally
	// An application program SHOULD NOT catch any escape exceptions
	// If it does the handler MUST re-throw them
	// An application program therefore should not use the "catch (...)" 
	// syntax for an exception handler but the macro "CATCH_ANY" defined
	// below
	
	typedef Any_EscapeDescr			*Any_Escape;
	typedef TerminateEscapeDescr	*TerminateEscape;

	static void throw_TerminateEscape();

};

//////////////////////////////////////////////////////////////////////
// An application program must use the macro "CATCH_ALL" to catch
// exceptions not explicitly specified in a handler
// An application program MUST NOT USE the ellipsis notation "catch (...)"

#define CATCH_ALL catch (Exception::Any_Escape) {throw;} catch (...)

#define CATCH_ANY CATCH_ALL
#define CATCH_OTHER CATCH_ALL


#endif // !defined(AFX_EXCEPTION_H__F7EF395C_C6C8_473C_BAF5_1743CDEB9BDC__INCLUDED_)
