// Synchronization.h: interface for the Entry and Rendezvous classes.
//
//////////////////////////////////////////////////////////////////////

#if !defined(TASK_SYNCHRONIZATION_H__INCLUDED_)
#define TASK_SYNCHRONIZATION_H__INCLUDED_

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

#include <atask\EntrySynchronizationBase.h>

//////////////////////////////////////////////////////////////////////
// All timeouts and delays are measured in milliseconds

typedef DWORD Milliseconds;


//////////////////////////////////////////////////////////////////////
// The abstract class Any_Entry is the base class of all entry classes

class Any_Entry : protected EntrySynchronizationBase
{
public:
	int count();
	static Any_Entry* null_pe();
protected:
	Any_Entry(EntryKind kind = normal_Entry, int signal_number = 0);
	friend class Rendezvous;
};

//////////////////////////////////////////////////////////////////////
// EntryPtrTable is a NULL terminated array of pointers to Any_Entry

typedef Any_Entry** EntryPtrTable;


//////////////////////////////////////////////////////////////////////
// A task uses Rendezvous objects to accept calls to its entries and 
// synchronize with their callers
// The rendezvous of a task with a caller starts at the declaration of the 
// Rendezvous object, and ends when the Rendezvous object is destroyed
// If the owner leaves the local block via a goto, return, break statement or
// exception, the rendezvous ends at that point
// Inside the rendezvous region a task may access the actual parameters passed
// by the caller and read or modify their values
// The caller remains suspended until the rendezvous ends

Any_Entry* guarded(Any_Entry* pe, bool guard);
// An entry specified in a Rendezvous constructor can be guarded 

#define UNCONDITIONALLY				(DWORD)INFINITE
#define OR_FAIL						(DWORD)0
#define OR_DELAY(milliseconds)		(DWORD)(milliseconds)
#define OR_TERMINATE				(DWORD)(INFINITE-1)
// A task can perform four different kinds of rendezvous
// - UNCONDITIONALLY indicates a normal rendezvous, which is the default
// - OR_FAIL indicates a conditional rendezvous
// - OR_DELAY(milliseconds) indicates a timed rendezvous
// - OR_TERMINATE indicates a terminable rendezvous

class Rendezvous  
{
public:
	Rendezvous(Any_Entry *pe0, 
		DWORD alternative = UNCONDITIONALLY);
	Rendezvous(Any_Entry *pe0, Any_Entry *pe1, 
		DWORD alternative = UNCONDITIONALLY);
	Rendezvous(Any_Entry *pe0, Any_Entry *pe1, Any_Entry *pe2, 
		DWORD alternative = UNCONDITIONALLY);
	Rendezvous(Any_Entry *pe0, Any_Entry *pe1, Any_Entry *pe2, Any_Entry *pe3, 
		DWORD alternative = UNCONDITIONALLY);
	Rendezvous(EntryPtrTable pe_table, 
		DWORD alternative = UNCONDITIONALLY);
	bool successful();
	int entry_selector();
	void* entry_parameters(Any_Entry* pe = NULL);
	~Rendezvous();
protected:
	typedef EntrySynchronizationBase::RendezvousData RendezvousData;
	RendezvousData data;
	int accepted_ix;
	Rendezvous() {}
};


//////////////////////////////////////////////////////////////////////
// Tasks interact by means of entries
// Each entry itself is an object of an Entry class - one of the Entry classes
// declared below

template <class Parameters> 
class Entry : public Any_Entry
{
// Normal Entry class with arbitrary parameters
// Parameters are passed as a struct of type "Parameters"
public:
	bool operator() (Parameters &params, Milliseconds timeout = INFINITE);
	Parameters &actual_parameters(Rendezvous &rv);
	Parameters* actual_parameters_pointer(Rendezvous &rv);
};

class Entry0 : public Any_Entry
{
// Normal Entry class with no (0) parameters
public:
	bool operator() (Milliseconds timeout = INFINITE);
};

template <class Parameter1> 
class Entry1 : public Any_Entry
{
// Normal Entry class with with 1 parameter
public:
	bool operator() (Parameter1 &p1, 
					 Milliseconds timeout = INFINITE);
	Parameter1 &actual_parameter1(Rendezvous &rv);
};

template <class Parameter1, class Parameter2> 
class Entry2 : public Any_Entry
{
// Normal Entry class with with 2 parameters
public:
	bool operator() (Parameter1 &p1, Parameter2 &p2, 
					 Milliseconds timeout = INFINITE);
	Parameter1 &actual_parameter1(Rendezvous &rv);
	Parameter2 &actual_parameter2(Rendezvous &rv);
};

template <class Parameter1, class Parameter2, class Parameter3> 
class Entry3 : public Any_Entry
{
// Normal Entry class with with 3 parameters
public:
	bool operator() (Parameter1 &p1, Parameter2 &p2, Parameter3 &p3, 
					 Milliseconds timeout = INFINITE);
	Parameter1 &actual_parameter1(Rendezvous &rv);
	Parameter2 &actual_parameter2(Rendezvous &rv);
	Parameter3 &actual_parameter3(Rendezvous &rv);
};

template <class Parameter1, class Parameter2, class Parameter3, class Parameter4> 
class Entry4 : public Any_Entry
{
// Normal Entry class with with 4 parameters
public:
	bool operator() (Parameter1 &p1, Parameter2 &p2, Parameter3 &p3, Parameter4 &p4, 
					 Milliseconds timeout = INFINITE);
	Parameter1 &actual_parameter1(Rendezvous &rv);
	Parameter2 &actual_parameter2(Rendezvous &rv);
	Parameter3 &actual_parameter3(Rendezvous &rv);
	Parameter4 &actual_parameter4(Rendezvous &rv);
};

class MsgEntry : public Any_Entry
{
// Msg Entry class
public:
	MSG &actual_msg(Rendezvous &rv);
	MSG* actual_msg_pointer(Rendezvous &rv);
	MsgEntry();
};

template <int sig>
class SignalEntry : public Any_Entry
{
// Signal Entry class
public:
	SignalEntry();
};

class AIOEntry : public Any_Entry
{
// Asynchronous I/O Entry class
public:
	struct CompletionData {
		DWORD dwErrorCode;
		DWORD dwNumberOfBytesTransfered;
		LPOVERLAPPED lpOverlapped;
	};
	CompletionData &actual_data(Rendezvous &rv);
	CompletionData* actual_data_pointer(Rendezvous &rv);
	static void CALLBACK completion_routine
		(DWORD dwErrorCode,
		 DWORD dwNumberOfBytesTransfered,
		 LPOVERLAPPED lpOverlapped);
	AIOEntry();
private:
	CompletionData aio_data;
};


//////////////////////////////////////////////////////////////////////
// Entry member definitions

template <class Parameters>
bool Entry<Parameters>::operator() (Parameters &params, Milliseconds timeout)
{
	return entry_call(&params, timeout);
}
template <class Parameters>
Parameters &Entry<Parameters>::actual_parameters(Rendezvous &rv)
{
	return *actual_parameters_pointer(rv);
}
template <class Parameters>
Parameters* Entry<Parameters>::actual_parameters_pointer(Rendezvous &rv)
{
	return (Parameters*)rv.entry_parameters(this);
}

//////////////////////////////////////////////////////////////////////
// Entry1 member definitions

template <class Parameter1>
bool Entry1<Parameter1>::operator() (Parameter1 &p1, Milliseconds timeout)
{
	return entry_call(&p1, timeout);
}
template <class Parameter1>
Parameter1 &Entry1<Parameter1>::actual_parameter1(Rendezvous &rv)
{
	Parameter1 *pp1 = (Parameter1*)rv.entry_parameters(this);
	return *pp1;
}

//////////////////////////////////////////////////////////////////////
// Entry2 member definitions

template <class Parameter1, class Parameter2>
bool Entry2<Parameter1, Parameter2>::operator() 
	(Parameter1 &p1, Parameter2 &p2, Milliseconds timeout)
{
	void *params[2];
	params[0] = &p1;
	params[1] = &p2;
	return entry_call(params, timeout);
}
template <class Parameter1, class Parameter2>
Parameter1 &Entry2<Parameter1, Parameter2>::actual_parameter1(Rendezvous &rv)
{
	void **pparams = (void**)rv.entry_parameters(this);
	Parameter1 *pp1 = (Parameter1*)pparams[0];
	return *pp1;
}
template <class Parameter1, class Parameter2>
Parameter2 &Entry2<Parameter1, Parameter2>::actual_parameter2(Rendezvous &rv)
{
	void **pparams = (void**)rv.entry_parameters(this);
	Parameter2 *pp2 = (Parameter2*)pparams[1];
	return *pp2;
}

//////////////////////////////////////////////////////////////////////
// Entry3 member definitions

template <class Parameter1, class Parameter2, class Parameter3>
bool Entry3<Parameter1, Parameter2, Parameter3>::operator() 
	(Parameter1 &p1, Parameter2 &p2, Parameter3 &p3, Milliseconds timeout)
{
	void *params[3];
	params[0] = &p1;
	params[1] = &p2;
	params[2] = &p3;
	return entry_call(params, timeout);
}
template <class Parameter1, class Parameter2, class Parameter3>
Parameter1 &Entry3<Parameter1, Parameter2, Parameter3>::actual_parameter1(Rendezvous &rv)
{
	void **pparams = (void**)rv.entry_parameters(this);
	Parameter1 *pp1 = (Parameter1*)pparams[0];
	return *pp1;
}
template <class Parameter1, class Parameter2, class Parameter3>
Parameter2 &Entry3<Parameter1, Parameter2, Parameter3>::actual_parameter2(Rendezvous &rv)
{
	void **pparams = (void**)rv.entry_parameters(this);
	Parameter2 *pp2 = (Parameter2*)pparams[1];
	return *pp2;
}
template <class Parameter1, class Parameter2, class Parameter3>
Parameter3 &Entry3<Parameter1, Parameter2, Parameter3>::actual_parameter3(Rendezvous &rv)
{
	void **pparams = (void**)rv.entry_parameters(this);
	Parameter3 *pp3 = (Parameter3*)pparams[2];
	return *pp3;
}

//////////////////////////////////////////////////////////////////////
// Entry4 member definitions

template <class Parameter1, class Parameter2, class Parameter3, class Parameter4>
bool Entry4<Parameter1, Parameter2, Parameter3, Parameter4>::operator() 
	(Parameter1 &p1, Parameter2 &p2, Parameter3 &p3, Parameter4 &p4, 
	 Milliseconds timeout)
{
	void *params[4];
	params[0] = &p1;
	params[1] = &p2;
	params[2] = &p3;
	params[3] = &p4;
	return entry_call(params, timeout);
}
template <class Parameter1, class Parameter2, class Parameter3, class Parameter4>
Parameter1 &Entry4<Parameter1, Parameter2, Parameter3, Parameter4>::actual_parameter1
	(Rendezvous &rv)
{
	void **pparams = (void**)rv.entry_parameters(this);
	Parameter1 *pp1 = (Parameter1*)pparams[0];
	return *pp1;
}
template <class Parameter1, class Parameter2, class Parameter3, class Parameter4>
Parameter2 &Entry4<Parameter1, Parameter2, Parameter3, Parameter4>::actual_parameter2
	(Rendezvous &rv)
{
	void **pparams = (void**)rv.entry_parameters(this);
	Parameter2 *pp2 = (Parameter2*)pparams[1];
	return *pp2;
}
template <class Parameter1, class Parameter2, class Parameter3, class Parameter4>
Parameter3 &Entry4<Parameter1, Parameter2, Parameter3, Parameter4>::actual_parameter3
	(Rendezvous &rv)
{
	void **pparams = (void**)rv.entry_parameters(this);
	Parameter3 *pp3 = (Parameter3*)pparams[2];
	return *pp3;
}
template <class Parameter1, class Parameter2, class Parameter3, class Parameter4>
Parameter4 &Entry4<Parameter1, Parameter2, Parameter3, Parameter4>::actual_parameter4
	(Rendezvous &rv)
{
	void **pparams = (void**)rv.entry_parameters(this);
	Parameter4 *pp4 = (Parameter4*)pparams[3];
	return *pp4;
}

//////////////////////////////////////////////////////////////////////
// SignalEntry member definitions

template <int sig>
SignalEntry<sig>::SignalEntry() : Any_Entry(signal_Entry, sig)
{
}

#endif // !defined(TASK_SYNCHRONIZATION_H__INCLUDED_)

