/*
 * Translated from Andrew Tuman's Pascal sources.
 */

#ifndef __DATE_H
#define __DATE_H

typedef unsigned DayTy;
typedef unsigned MonthTy;
typedef unsigned YearTy;
typedef unsigned long JulTy;

typedef struct {
	unsigned short year;
	unsigned char month;
	unsigned char day;
} BDateTy;

#define DATE_BUFFER_LEN 11

extern "C" {
	extern const YearTy century;
	extern const DayTy daysInMonth[];

	YearTy get_full_year(YearTy year);

	int LeapYear(YearTy);

	DayTy DaysToDate(YearTy, MonthTy, DayTy);
	JulTy DaysToYear(YearTy);
    
	void Encode(YearTy, MonthTy, DayTy, JulTy&);
	void Decode(YearTy&, MonthTy&, DayTy&, const JulTy);

	int DayOfWeek(YearTy, MonthTy, DayTy);    

	char* d2xbase(YearTy year, MonthTy month, DayTy day, char* buffer);

	char* d2h(const char* value, char* buffer);
	char* h2d(const char* value, char* buffer);
}

class DateTy {
public:
/*
 *	USA      (mm-dd-yy)
 *	American (mm/dd/yy)
 *	British  (dd/mm/yy)
 *	German   (dd.mm.yy)
 *	Italian  (dd-mm-yy)
 *	Japanese (yy/mm/dd)
 *	ANSI     (yy.mm.dd)
 *	xBase    (YYYYMMDD)
 */
	enum Formats {
		AMERICAN = 0, USA = 1, ANSI = 2, JAPANESE = 3, BRITISH = 4,
		GERMAN = 5, ITALIAN = 6, XBASE = 7, PACKED = 8, DEFAULT = 9,
		SWIFT32A = 10
	};

	DateTy(bool use_global = true);
	DateTy(YearTy, MonthTy, DayTy);
	DateTy(JulTy);
	DateTy(const char *, DateTy::Formats = DEFAULT);

	const char* c_str(DateTy::Formats = DEFAULT, bool century = true) const;

	int IsLeap() const;
	int WeekDay() const;
	DayTy Day() const;
	MonthTy Month() const;
	YearTy Year(bool century = true) const;
	DayTy DaysInMonth() const;
	DayTy LastWorkDay() const;   

	DateTy& FirstMonthDay();
	DateTy& LastMonthDay();
	DateTy& DecMonth(unsigned = 1);
	DateTy& IncMonth(unsigned = 1);

	operator JulTy() const {
		return JulNum;
	}
	bool operator<(const DateTy& rhs) const {
		return JulNum < rhs.JulNum;
	}
	bool operator==(const DateTy& rhs) const {
		return JulNum == rhs.JulNum;
	}
	DateTy& operator++() {
		JulNum += 1;
		return *this;
	}
	DateTy& operator--() {
		JulNum -= 1;
		return *this;
	}
	DateTy& operator+=(int delta) {
		JulNum += delta;
		return *this;
	}
	DateTy& operator-=(int delta) {
		JulNum -= delta;
		return *this;
	}
	JulTy operator-(const DateTy& rhs) const {
		return JulNum - rhs.JulNum;
	}
	JulTy operator+(int rhs) const {
		return JulNum + rhs;
	}

	const char* ShortDayName(DayTy weekDayNumber);
	const char* MonthName(MonthTy monthNumber);

	static void PrintMode(Formats = DateTy::BRITISH);

protected:
	static char GetDelimiter();

	void IncOneMonth();
	void DecOneMonth();

	void d2h(const char* value) const;
	void h2d(const char* value) const;

private:
	JulTy JulNum;
	static Formats PrintOption;
	mutable char buffer[16];
};

class DateMY {
private:
	MonthTy month;
	YearTy year;
   
public:
	DateMY(JulTy date) : month(DateTy(date).Month()), year(DateTy(date).Year()) {}
	DateMY(const DateTy& date) : month(date.Month()), year(date.Year()) {}

	bool operator<(const DateMY& rhs) const {
		if(year < rhs.year) return true;
		if(year > rhs.year) return false;
		return month < rhs.month;
	}
	bool operator==(const DateMY& rhs) const {
		return month == rhs.month && year == rhs.year;
	}

	MonthTy Month() const { return month; }
	YearTy Year() const { return year; }
};

extern DateTy global_date_value;

#endif // __DATE_H

