/** \file DS_Container.h
    Generic container

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.
Permission is granted to use and copy this file for non-commercial use only.
Please contact the author concerning commercial usage.
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/
#ifndef H_DS_CONTAINER
#define H_DS_CONTAINER

#include <DS_SpecContainable.h>

class Enumeration;

/** Abstract Container. 
    This class will not be very useful directly.
    Most of it's implementation is either pure or has
    a "brainless" default implementation.

    put(data, index), get(index), remove(index), and size()
    are pure and must be defined by classes that implement Container.
*/
class Container : public Serial_Containable
{
public:
   enum {DESTRUCTIVE = 0x00000010};

   /** Default Constructor: Construct an empty container. */
   Container(const unsigned flags=0) : m_Flags(flags) {}

   /** Copy Constructor: Construct a copy of a container. */
   Container(const Container& c)     : m_Flags(0)     { *this = c; }

   /** Construct a container from an enumeration.
       Throws DestructiveCopy if the enumeration isDestructive(). */
   Container(const Enumeration& e)   : m_Flags(0)     { load((Enumeration&)e, 0); }

   /** Destructor */
   virtual ~Container()                               {}

   /** Assignment: Copy a container.
       Throws DestructiveCopy if either this container or
       the source container isDestructive(). */
   Container& operator=(const Container& c);


/** Base interface. */
   
   /** Put a data item to this container.
       Exact behavior varies depending on the container.

       If index == -1 (the default) or index == size(), the element is
          added to this container.  1 is returned if a new item
          was added.  If an item was replaced, zero is returned.

       If index >= 0 and index < size(), the item at that index is
          removed (and deleted if this container isDestructive()).
          The new item is inserted at the given index.  Note that
          some containers may ignore the index during the insertion
          operation.  If the container only supports unique entries,
          an additional item (with the same key) may be removed from 
          the container (and deleted is this container isDestructive())
          In this case, -1 is returned (net loss in container size); 
          otherwise, 0 is returned.

       If index < -1 or index > size(), throws IndexOutOfBounds.
   
       Note that for some containers, indices are not stable.  Some containers
       may shift the item (or others) to a different location as part of the put
       operation.  An indexOf(item) immediately followed by either put(item, index)
       or get(index) will always be safe. */
   virtual int         put(const Data item, const index=-1) = 0;

   /** Get a data item from this container by index.
       Exact behavior varies depending on the container.
      
       If index >= 0 and index < size(), the item at that index
          is returned (as determined by this container's own numbering
          scheme).
      
       If index == -1 (the default) (peek) this container decides which 
          item to return (this will be same item as that returned by pop()).
          If isEmpty(), NULL is returned.

       If index < -1 or index >= size(), IndexOutOfBounds is thrown.

       Note: the index used by this function is not stable.  Other Container 
       functions such as put() or remove() may (and probably will, depending on 
       the particular container implementation) cause items to be relocated. */
   virtual Data        get(const int index=-1) const = 0;

   /** Get (retrieve) a data item from this container by key.
       If an item is found in this container with the same key
       (as determined by Containable::equals()), it is returned;
       otherwise, (not found) NULL is returned.

       A Serial_Containable instance cannot be retrieved by this
       function without its actual pointer, because it only equals another
       Containable which has in the same memory address exactly. */
   virtual Data        get(const Data item) const;

   /** Pop: Get and remove a data item from this container.  The particular
          item chosen to be returned is dependent on the container implementation.
          If this container isDestructive(), this function will return garbage,
          because the item will have been deleted.
          If isEmpty(), pop() returns NULL. */
   virtual Data        pop();
   
   /** Remove a data item by index.
       If this container isDestructive(), the item is deleted.
       Throws IndexOutOfBounds if index < 0 or index >= size(). */
   virtual void        remove(const int index) = 0;

   /** Remove a data item by key.
       The item must conform to Container::get(item) rules.  If the data item exists 
       in this container, it is removed and non-zero is returned; otherwise, (not 
       found) zero is returned.  Either way, if this container isDestructive(), 
       the item passed in is deleted. */
   virtual DS_BOOL     remove(const Data item);

   /** Remove all elements from this container.   This has a simple
       default implemetation which does remove(0) until size() == 0. */
   virtual  void       clear();

   /** Returns the number of elements in this container. */
   virtual int         size() const = 0;

   /** Returns non-zero if there is no data in this container. */
           DS_BOOL     isEmpty() const                { return size()==0; }

   /** Returns an enumeration of the container's elements.  As this is a copy 
       (snapshot) of the elements, this container can be modified while 
       iterating through the enumeration. */
   virtual Enumeration elements() const;


/** Utility functions. */

   /** Load this container with the elements of another container.
       Each element of the source container is added to this container
       with put(item).  The number of additional elements added to the 
       container is returned.
       load() has a cumulative effect (this container is not cleared first).
       Throws DestructiveCopy if either this container or the
       source container isDestructive(). */
   virtual int     load(const Container&);

   /** Load this container with an enumeration.  See load(container). */
           int     load(const Enumeration& e)         { return load((Enumeration&)e, 0); }

   /** Load this container with a non-const enumeration and optionally discard source
       if temp is non-zero. Invaildates e if temp. */
   virtual int     load(Enumeration&, int temp=0);

   /** Retrieve the index of a data item.  If the item is found in
       this container, its index is returned; otherwise, (-1) is
       returned.  See the comments for get(index) and put()
       for cautions regarding usage of these "index" functions. */
   virtual int     indexOf(const Data item) const;

   /** Returns non-zero if the item is in this container. */
   virtual DS_BOOL contains(const Data item) const     { return indexOf(item) != -1; }

   /** operator[].  Same as get(index).  
       Usage of this for an L-value is not permitted. */
   Data operator[](const int index) const              { return get(index); }

   Data    peek() const                                { return get(-1); }

   /* Returns non-zero if this container is DESTRUCTIVE (constructed
      with the DESTRUCTIVE flag set). */
   DS_BOOL isDestructive() const                       { return (m_Flags & DESTRUCTIVE)!=0; }

   /** Checks the validity of an element index.  
       Returns non-zero if the index is valid. */
   DS_BOOL isValidIndex(const int index) const         { return index>=0 && index<size(); }

protected:
   unsigned m_Flags;
};


#endif // H_DS_CONTAINER