/* Panodome - Panoramic imersion engine
 *
 * (c) Copyright 1998-2000, ITB CompuPhase
 */
#ifndef _PANODOME_H
#define _PANODOME_H

/* Every compiler uses different "default" macros to indicate the mode
 * it is in. Throughout the source, I use the Borland C++ macros, so
 * the macros of Watcom C/C++ and Microsoft Visual C/C++ are mapped to
 * those of Borland C++.
 */
#if defined(__WATCOMC__)
#  if defined(__WINDOWS__) || defined(__NT__)
#    define _Windows
#  endif
#  ifdef __386__
#    define __32BIT__
#  endif
#  if defined(_Windows) && defined(__32BIT__)
#    define __WIN32__
#  endif
#elif defined(_MSC_VER)
#  if defined(_WINDOWS) || defined(_WIN32)
#    define _Windows
#  endif
#  ifdef _WIN32
#    define __WIN32__
#    define __32BIT__
#  endif
#endif

#if !defined(_INC_WINDOWS) && !defined(__WINDOWS_H)
#  include <windows.h>
#endif

#ifdef __cplusplus
extern "C"
{
#endif

enum {
  PAN_ERR_NONE,
  PAN_ERR_IMGSIZE,      /* width of source image must be a multiple of 4-bytes */
  PAN_ERR_VPSIZE,       /* viewport is too big */
  PAN_ERR_MEMORY,
  PAN_ERR_WRONGFORMAT,  /* general format in the source image */
  PAN_ERR_PARAM,        /* general parameter error */
  PAN_ERR_ANGLE,        /* invalid panorama angle */
  PAN_ERR_COORDS,       /* invalid coordinates */
  PAN_ERR_INDEX,        /* invalid index */
};

enum {
  PAN_VALUE_HORIZON,
  PAN_VALUE_ANGLE,
  PAN_VALUE_RADIUS,
  PAN_VALUE_NUMHOTSPOTS,
  PAN_VALUE_ID,         /* user selectable id */
  PAN_VALUE_XPOS,
  PAN_VALUE_YPOS,
  PAN_VALUE_ZPOS,
  PAN_VALUE_MSGNUM,     /* notification message number */
  PAN_VALUE_ORIENTATION,
  PAN_VALUE_SRCWIDTH,
  PAN_VALUE_SRCHEIGHT,
  PAN_VALUE_VPWIDTH,
  PAN_VALUE_VPHEIGHT,
  PAN_VALUE_MAXZOOM,
  PAN_VALUE_SPEED,

  PAN_DATA_IMAGE = 100,
  PAN_DATA_VIEWPORT,    /* DIB of the current "view" */
  PAN_DATA_BITMAPINFO,  /* BITMAPINFOHDR of the current "view" */
  PAN_DATA_LENS,
  PAN_DATA_HOTSPOTLBL,  /* label of the hotspot */
  PAN_DATA_HOTSPOTTEXT, /* tooltip text of the hotspot */
  PAN_DATA_HOTSPOTLIST, /* raw list of the hotspots */

  PAN_HANDLE_PALETTE = 200,
  PAN_HANDLE_VIEWERWND, /* handle of the "viewer window" */
  PAN_HANDLE_NOTIFYWND, /* window where notification messages are sent */

  PAN_RECT_HOTSPOT = 300,
};

#define PAN_BOUND_X     0x01    /* bit mask */
#define PAN_BOUND_Y     0x02
#define PAN_BOUND_Z     0x04

#define PAN_FLAG_OWNHOTSPOTS    0x0001  /* the library owns the hotspot list */

#define PAN_COMMAND     (WM_USER+78)
#define PAN_NOTIFY      (WM_USER+79)
enum {                  /* values for wParam upon receipt of PAN_NOTIFY */
  PN_SELHOTSPOT,        /* click on hotspot, low word = index */
  PN_HITHOTSPOT,        /* mouse cursor above hotspot */
  PN_MOVE,              /* pan or tilt */
  PN_HALT,              /* stop moving */
};
enum {                  /* values for wParam for PAN_COMMAND */
  PC_ORIENTATION,       /* set the compass needle */
  PC_TIMERTICK,         /* issue a timer tick (for user timers) */
};

struct PANHOTSPOT__ {
  int left, top, right, bottom; /* the rectangle */
  char label[128];              /* label (or action) */
  char text[128];               /* general purpose text */
};
typedef struct PANHOTSPOT__ PANHOTSPOT;
typedef struct PANHOTSPOT__ FAR *LPPANHOTSPOT;

typedef struct PANATTRIB__ {
  int angle;
  int horizon;
  int orientation;
  int numhotspots;
  LPPANHOTSPOT hotspots;
  char userstring[128];
  int flags;
  long reserved;        /* should be zero */
};
typedef struct PANATTRIB__ PANATTRIB;
typedef struct PANATTRIB__ FAR *LPPANATTRIB;


#if !defined _PANODOME_DLL

#  if defined __WIN32__
#    define PANAPI      __declspec(dllimport) __stdcall
#  else
#    define PANAPI      WINAPI
#  endif

   typedef struct PANINFO__ {
     long reserved;
   };
   typedef const struct PANINFO__ FAR *PANINFO;

#endif


LPVOID  PANAPI pan_AllocResource(long Size);
BOOL    PANAPI pan_AttribFile(LPCSTR Filename, LPPANATTRIB Attribs);
BOOL    PANAPI pan_AttribString(LPCSTR CmdString, int Length, LPPANATTRIB Attribs);
BOOL    PANAPI pan_CheckHotSpot(PANINFO Pan, int X, int Y, LPINT Index);
PANINFO PANAPI pan_Create(LPBITMAPINFO BitsInfo, LPVOID ImageBits,
                          LPRECT ViewPort, LPPANATTRIB Attribs);
HWND    PANAPI pan_CreateCompass(int X, int Y, HWND hwndParent, BOOL Child);
HWND    PANAPI pan_CreateWindow(PANINFO Pan, DWORD dwStyle, HWND hwndParent);
BOOL    PANAPI pan_Delete(PANINFO Pan, BOOL DeleteImage);
int     PANAPI pan_Error(PANINFO Pan);
LPVOID  PANAPI pan_FreeResource(LPVOID Resource);
LPVOID  PANAPI pan_GetData(PANINFO Pan, int Code, int Index);
HANDLE  PANAPI pan_GetHandle(PANINFO Pan, int Code);
LPRECT  PANAPI pan_GetRect(PANINFO Pan, int Code, int Index);
int     PANAPI pan_GetValue(PANINFO Pan, int Code);
LPVOID  PANAPI pan_GetView(PANINFO Pan, LPBITMAPINFO BitsInfo);
LPVOID  PANAPI pan_LoadDIB(LPCSTR Filename, LPBITMAPINFO BitsInfo,
                           HINSTANCE hInstance, LPCSTR ResourceType);
BOOL    PANAPI pan_MapCoordinates(PANINFO Pan, int X, int Y, LPPOINT Point);
BOOL    PANAPI pan_MinImageSize(LPINT Width, LPINT Height, LPRECT viewport, int Angle);
WORD    PANAPI pan_Move(PANINFO Pan, int DeltaX, int DeltaY, int DeltaZ);
BOOL    PANAPI pan_Replace(PANINFO Pan, LPBITMAPINFO BitsInfo, LPVOID ImageBits,
                           LPPANATTRIB Attribs, BOOL DeleteDIB);
LPVOID  PANAPI pan_ScaleDIB(LPBITMAPINFO SrcInfo, LPVOID SrcBits,
                            LPBITMAPINFO DstInfo, int Width, int Height,
                            BOOL DeleteSrc);
BOOL    PANAPI pan_SetHandle(PANINFO Pan, int Code, HANDLE Handle);
WORD    PANAPI pan_SetPos(PANINFO Pan, int X, int Y, int Z);
BOOL    PANAPI pan_SetValue(PANINFO Pan, int Code, int Value);


/* Utility macros */

#if !defined DIBLINESIZE
  #define DIBLINESIZE(BitCount, Width)  \
        ((((long)(Width) * (BitCount) + 31L) & ~31uL) >> 3)
#endif
#if !defined DIBSIZE
  #define DIBSIZE(w,h)  ( (((w) + 3L) & ~3L) * (h) )
#endif

#ifdef __cplusplus
}
#endif


#ifdef __cplusplus
class PanAttrib {
  PANATTRIB Attrib;
public:
  PanAttrib()
    { memset(&Attrib, 0, sizeof Attrib); }
  PanAttrib(LPCSTR string, int Length=-1)
    { if (!pan_AttribString(string, &Attrib, Length))
        pan_AttribFile(string, &Attrib); }

  BOOL File(LPCSTR string)
    { return pan_AttribFile(string, &Attrib); }
  BOOL String(LPCSTR string, int Length=-1)
    { return pan_AttribString(string, &Attrib); }
};

class PanView {
  PANINFO Pan;
  BOOL ownimage;        // used only for the default destructor
  // class assignment is not supported; make the operator private so
  // that the compiler won't allow the assignment operator
  PanView& operator=(PanView&)
    { pan_Delete(Pan); Pan=NULL; return *this; }
  // copy constructor is private too (i.e. it is unsupported)
  PanView(PanView&)
    { Pan=NULL; }
public:
  PanView()
    { Pan=NULL; ownimage=FALSE; }
  PanView(LPBITMAPINFO BitsInfo, LPVOID Image, LPRECT ViewPort,
          PanAttrib *Attribs=NULL)
    { Pan = pan_Create(BitsInfo, lpImage, ViewPort, Attribs); ownimage = FALSE; }
  PanView(LPCSTR Filename,LPRECT ViewPort, PanAttrib *Attribs=NULL)
    { LPBITMAPINFO BitsInfo = pan_LoadDIB(Filename); ownimage = TRUE;
      Pan = (BitsInfo!=NULL) ? pan_Create(BitsInfo,NULL,ViewPort,Attribs) : NULL; }
  ~PanView()
    { pan_Delete(Pan,ownimage); }
  // this lets you test whether the panorama was opened correctly
  operator int()
    { return Pan!=NULL; }

  BOOL CheckHotSpot(int X, int Y, LPINT Index)
    { return pan_CheckHotSpot(Pan, X, Y, Index); }
  BOOL Create(LPBITMAPINFO BitsInfo, LPVOID Image, LPRECT ViewPort,
              PanAttrib *Attribs=NULL)
    { return (Pan=pan_Create(BitsInfo,lpImage,ViewPort,Attribs)) != NULL; }
  HWND CreateWindow(DWORD dwStyle, HWND hwndParent=NULL)
    { return pan_CreateWindow(Pan, dwStyle, hwndParent); }
  BOOL Delete(BOOL DeleteImage=FALSE)
    { return pan_Delete(Pan,DeleteImage) ? Pan=NULL,1 : 0; }
  int  Error()
    { return pan_Error(Pan); }
  LPVOID GetData(int Code, int Index=0)
    { return pan_GetData(Pan, Code, Index); }
  HANDLE  GetHandle(int Code)
    { return pan_GetHandle(Pan, Code); }
  LPRECT GetRect(int Code, int Index=0)
    { return pan_GetRect(Pan, Code, Index); }
  int GetValue(int Code)
    { return pan_GetValue(Pan, Code); }
  LPVOID GetView(LPBITMAPINFO BitsInfo)
    { pan_GetView(Pan, BitsInfo); }
  BOOL MapCoordinates(LPPOINT Point, int X, int Y)
    { return pan_MapCoordinates(Pan, Point, X, Y); }
  WORD Move(int DeltaX, int DeltaY, int DeltaZ)
    { return pan_Move(Pan, DeltaX, DeltaY, DeltaZ); }
  BOOL Replace(LPBITMAPINFO BitsInfo, LPVOID ImageBits, PanAttrib *Attribs=NULL,
               BOOL DeleteDIB=FALSE)
    { return pan_Replace(Pan, BitsInfo, lpImage, Attribs, DeleteDIB); }
  BOOL SetHandle(int Code, HANDLE Handle)
    { return pan_SetHandle(Pan, Code, Handle); }
  WORD SetPos(int X, int Y, int Z)
    { return pan_SetPos(Pan, X, Y, Z); }
  BOOL SetValue(int Code, int Value)
    { return pan_SetValue(Pan, Code, Value); }
};

class PanCompass {
  HWND hwnd;
  // class assignment is not supported; make the operator private so
  // that the compiler won't allow the assignment operator
  PanCompass& operator=(Panorama&)
    { DestroyWindow(hwnd); hwnd=NULL; return *this; }
  // copy constructor is private too (i.e. it is unsupported)
  PanCompass(PanCompass&)
    { hwnd=NULL; }
public:
  PanCompass()
    { hwnd=NULL; }
  PanCompass(int X, int Y, HWND hwndParent=NULL, BOOL Child=FALSE)
    { hwnd = pan_CreateCompass(X, Y, hwndParent, Child); }
  ~PanCompass()
    { DestroyWindow(hwnd); }
  // this lets you test whether the panorama was opened correctly
  operator int()
    { return hwnd!=NULL; }

  HWND Create(int X, int Y, HWND hwndParent=NULL, BOOL Child=FALSE)
    { return (hwnd=pan_CreateCompass(X, Y, hwndParent, Child)); }
  HWND GetHWND()
    { return hwnd; }
  void SetOrientation(int orientation)
    { if (hwnd!=NULL) SendMessage(hwnd, PAN_COMMAND, PC_ORIENTATION, orientation); }
};
#endif

#endif /* _PANODOME_H */
