/*
**  AFINGR.C
**
**  Asynchronous FINGER program.
**
**  FINGER program comparison
**
**    (1) AFINGR -- asyncronous version.
**    (2) BFINGR -- asyncronous version (using ASYNC.C functions).
**    (3) SFINGR -- synchronous version.
**    (4) CFINGR -- synchronous version (Win32 CONSOLE program).
**    (5) FINGER -- state driven version.
*/

#include <windows.h>
#include <winsock.h>

#include "wil.h"
#include "message.h"
#include "paint.h"
#include "about.h"
#include "str.h"


#ifdef WIN32
#define USE_INS HINSTANCE
#define USE_PTR PSTR
#else
#define USE_INS HANDLE
#define USE_PTR LPSTR
#endif

LRESULT CALLBACK MainWndProc(HWND, UINT, WPARAM, LPARAM);

/* globals */

HWND hMainWnd;            /* main window handle */

#define BS            8
#define LF           10
#define CR           13
#define MAX_BUF     512
#define MAX_STR      50

#define ONE_SEC    1000
#define TEN_SEC   10000

#define AFINGR_PORT  79

static HMENU hMenu;
static USE_INS hInstance;
static int WinWidth = 8 * NCOLS;
static int WinHeight = 12 * NROWS + 48;
static char Temp[MAX_BUF+8];
static int  InBufLen = 0;
static char InBuffer[MAX_BUF+1];
static char UserName[MAX_STR] = "";
static char HostName[MAX_STR] = "";
static SOCKET Socket = 0;
static ULONG  HostAddr = 0;
static LPSTR  Ptr;
static HCURSOR ArrowCursor;
static HCURSOR WaitCursor;
static int CompletedFlag = FALSE;

/* add character to buffer */

static void Add2Buffer(char Chr)
{/* add char to input buffer */
 switch(Chr)
   {case BS:
      if(InBufLen>0)
        {/* backup on screen */
         DisplayChar(BS);
         /* remove last char from buffer */
         InBufLen--;
        }
      break;
    default:
      /* display char */
      DisplayChar(Chr);
      /* put into buffer */
      if(InBufLen<MAX_BUF) InBuffer[InBufLen++] = Chr;
      break;
   }
}

/* display error message */

static void DisplayError(int Code, LPSTR Msg)
{wilAwaitCancel(Socket,hMainWnd);
 DisplayString("ERROR: ");
 if(Msg) DisplayString(Msg);
 if(Code)
   {wilErrorText(Code,(LPSTR)Temp,50);
    DisplayLine((LPSTR)Temp);
   }
 SetCursor(ArrowCursor);
}

/* WinMain */

#ifdef WIN32
int WINAPI
#else
int PASCAL
#endif
WinMain(USE_INS hInst, USE_INS hPrevInstance,
        USE_PTR szCmdLine,  int nCmdShow)
{WNDCLASS  wc;
 MSG msg;
 BOOL Result;
 if(!hPrevInstance)
   {/* register main window class */
    wc.style = CS_HREDRAW | CS_VREDRAW;
    wc.lpfnWndProc = MainWndProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
    wc.hInstance = hInst;
    wc.hIcon = LoadIcon(hInst, "HostIcon");
    wc.hCursor = NULL;
    wc.hbrBackground = (HBRUSH)GetStockObject(WHITE_BRUSH);
    wc.lpszMenuName =  "HostMenu";
    wc.lpszClassName = "HostWClass";
    Result = RegisterClass(&wc);
    if(!Result) return FALSE;
   }

 /* create main window */
 hInstance = hInst;
 hMainWnd = CreateWindow(
        "HostWClass",   "FINGER",    WS_OVERLAPPEDWINDOW,
        CW_USEDEFAULT,  CW_USEDEFAULT,
        WinWidth,       WinHeight,
        NULL,           NULL,
        hInstance,      NULL);
 ShowWindow(hMainWnd, nCmdShow);
 UpdateWindow(hMainWnd);
 hMenu = GetMenu(hMainWnd);

 /* window control loop */

 while(GetMessage(&msg,NULL,0,0))
   {
    TranslateMessage(&msg);
    DispatchMessage(&msg);
   }
 return (msg.wParam);
} /* end WinMain */

#ifdef WIN32
LRESULT CALLBACK
#else
long FAR PASCAL
#endif
MainWndProc(HWND hWindow,UINT iMsg,WPARAM wParam,LPARAM lParam)
{int Code;
 HDC hDC;
 PAINTSTRUCT ps;
#ifdef WIN32
#else
 static FARPROC lpfnAboutDlgProc;
#endif
 hMainWnd = hWindow;
 switch (iMsg)
    {case WM_CREATE:
      /* create cursors */
      ArrowCursor = LoadCursor(NULL, IDC_ARROW);
      WaitCursor = LoadCursor(NULL, IDC_WAIT);
      SetCursor(ArrowCursor);
#ifdef WIN32
#else
       /* create thunk for Win16 */
       lpfnAboutDlgProc = MakeProcInstance(AboutDlgProc,hInstance);
#endif
      /* initialize paint module */
      PaintInit();
      /* attach WINSOCK */
      DisplayString("Attaching WINSOCK...");
      Code = wilAttach();
      DisplayLine("OK");
      if(Code<0) DisplayError(Code,"wilAttach fails:");
      else
        {DisplayString(" Description: ");
         wilGetDescription((LPSTR)Temp, MAX_BUF);
         DisplayLine((LPSTR)Temp);
         DisplayString("My Host Name: ");
         wilGetMyHostName((LPSTR)Temp, MAX_BUF);
         DisplayLine((LPSTR)Temp);
         DisplayString("My Host Addr: ");
         wilGetMyHostDotted(0,(LPSTR)Temp, MAX_BUF);
         DisplayLine((LPSTR)Temp);
        }
      break;

     case WM_COMMAND:
         switch(wParam)
           {case MSG_ABOUT :
#ifdef WIN32
               DialogBox(hInstance, "AboutBox", hMainWnd, AboutDlgProc);
#else
               DialogBox(hInstance, "AboutBox", hMainWnd, lpfnAboutDlgProc);
#endif
               return 0;

            case MSG_EXIT:
              wilRelease();
              DestroyWindow(hMainWnd);
              break;

            case MSG_DEBUG:
              Code = wilDebug(0);
              wsprintf((LPSTR)Temp,"Debug(0) returned %d",Code);
              DisplayLine((LPSTR)Temp);
              break;
            case MSG_BREAK:
              wilCancelBlocking();
              wilAwaitCancel(Socket,hMainWnd);
              break;

            case MSG_FINGER:
              InBufLen = 0;
              DisplayString("Enter user (usr@domain):");
              break;
           }
         break;

    case WM_PAINT:
      HideCaret(hMainWnd);
      hDC = BeginPaint(hMainWnd, &ps);
      SetMapMode(hDC,MM_ANISOTROPIC);
      SelectObject(hDC, GetStockObject(OEM_FIXED_FONT) );
      PaintMain(hDC,&ps);
      EndPaint(hMainWnd,&ps);
      SetCaretPos(PaintGetColPos(),PaintGetRowPos());
      ShowCaret(hMainWnd);
      break;

    case WM_DESTROY:
      PostQuitMessage(0);
      break;

    case WM_USER:

#if 0
wsprintf((LPSTR)Temp,"[%ld]", lParam);
DisplayString((LPSTR)Temp);
#endif

      /* received message from wilAwaitEvent */
      switch( LOWORD(lParam) )
        {case FD_CONNECT:
           /* we are now connected. send user name to server */
           wsprintf((LPSTR)Temp,"%s\r\n",(LPSTR)UserName);
           Code = wilWriteString(Socket,(LPSTR)Temp);
           if(Code<0)
             {DisplayError(Code, NULL);
              break;
             }
           /* await 1st response */
           wilAwaitRead(Socket,hMainWnd);
           break;

         case FD_READ:
           if(CompletedFlag) break;
           Code = wilReadLine(Socket,(LPSTR)InBuffer,MAX_BUF);
           if(Code==WIL_EOF)
             {/* we are done */
              wilAwaitCancel(Socket,hMainWnd);
              DisplayLine("<EOF>");
              CompletedFlag = TRUE;
              wilCloseSocket(Socket,500);
              SetCursor(ArrowCursor);
              DisplayLine("\r\nFINGER completed.");
              break;
             }
           if(Code>0)
             {/* display line from server */
              DisplayString((LPSTR)InBuffer);
              /* await next line */
              break;
             }
           if(Code<0)
             {/* error */
              DisplayError(Code,"wilRead fails:");
              break;
             }
           /* Code == 0 */
#if 0
           DisplayChar('!');
#endif
        }
      break;

    case WM_CHAR:
      if(wParam==CR)
        {/* do the CR */
         DisplayChar((char)wParam);
         /* user has completed input */
         DisplayChar(LF);
         InBuffer[InBufLen] = '\0';
         /* execute command */
         wsprintf((LPSTR)Temp,"User@Host: %s",(LPSTR)InBuffer);
         DisplayLine((LPSTR)Temp);
         /* extract user & host names */
         Ptr = StringChar((LPSTR)InBuffer,'@');
         if(Ptr==NULL)
           {DisplayError(Code, "Cannot recognize User@Host");
            break;
           }
         *Ptr = '\0';
         lstrcpy((LPSTR)UserName, (LPSTR)InBuffer);
         lstrcpy((LPSTR)HostName, (LPSTR)(++Ptr));
         wsprintf((LPSTR)Temp,"User ='%s'", (LPSTR)UserName);
         DisplayLine(Temp);
         wsprintf((LPSTR)Temp,"Host ='%s'", (LPSTR)HostName);
         DisplayLine(Temp);
         SetCursor(WaitCursor);
         /* get host name */
         if(wilIsDotted((LPSTR)HostName))
           {/* already have IP address */
            HostAddr = wilMakeAddr((LPSTR)HostName);
           }
         else
           {/* put up hour glass */
            SetCursor(WaitCursor);
            /* ask DNS for IP address */
            HostAddr = wilGetHostAddr((LPSTR)HostName,0);
            if(HostAddr==0)
              {DisplayError(0, "Cannot get IP addess");
               break;
              }
           }
         /* display remote host address */
         DisplayString("HostAddr = ");
         wilMakeDotted(HostAddr,(LPSTR)Temp,MAX_BUF);
         DisplayLine((LPSTR)Temp);
         /* create socket */
         Socket = wilTcpSocket();
         if((int)Socket<0)
           {DisplayError((int)Code, NULL);
            break;
           }
         CompletedFlag = FALSE;
         /* WM_USER message will be sent when connect is done */
         wilAwaitConnect(Socket,hMainWnd);
         /* initiate connect to remote host */
         Code = wilConnect(Socket,HostAddr,AFINGR_PORT);
         if(Code<0)
           {DisplayError(Code, "Connect:");
            break;
           }
        }
      else
        {/* add char to input buffer */
         Add2Buffer((char)wParam);
        }
      break;

    default:
      return (DefWindowProc(hMainWnd, iMsg, wParam, lParam));
   }
 return 0;
}

