/*
**  FINGER.C
**
**  State driven FINGER program.
**
**  The state variable <State> is used with a timer to implement
**  an asynchronous, state driven FINGER program.
**
**  FINGER program comparison
**
**    (1) AFINGR -- asyncronous version.
**    (2) BFINGR -- asyncronous version (using ASYNC.C functions).
**    (3) SFINGR -- synchronous version.
**    (4) CFINGR -- synchronous version (Win32 CONSOLE program).
**    (5) FINGER -- state driven version.
*/

#include <windows.h>
#include <winsock.h>

#include "wil.h"
#include "message.h"
#include "paint.h"
#include "about.h"
#include "str.h"


#ifdef WIN32
#define USE_INS HINSTANCE
#define USE_PTR PSTR
#else
#define USE_INS HANDLE
#define USE_PTR LPSTR
#endif

LRESULT CALLBACK MainWndProc(HWND, UINT, WPARAM, LPARAM);

/* globals */

HWND hMainWnd;            /* main window handle */

#define BS            8
#define LF           10
#define CR           13
#define MAX_BUF     256
#define MAX_STR      40

#define ONE_SEC    1000
#define TEN_SEC   10000

#define FINGER_PORT  79

#define S_CONNECT           1001
#define S_WAIT_FOR_CONNECT  1002
#define S_SEND_USER         1003
#define S_AWAIT_INCOMING    1004
#define S_SUCCESS           1005

static HMENU hMenu;
static USE_INS hInstance;
static int WinWidth = 8 * NCOLS;
static int WinHeight = 12 * NROWS + 48;
static char Temp[MAX_BUF+8];
static int  InBufLen = 0;
static char InBuffer[MAX_BUF+1];
static char UserName[MAX_STR] = "";
static char HostName[MAX_STR] = "";
static SOCKET Socket = 0;
static ULONG  HostAddr = 0;
static LPSTR  Ptr;
static HCURSOR ArrowCursor;
static HCURSOR WaitCursor;
static UINT idTimer;

/* miscellaneous functions */

static void Add2Buffer(char Chr)
{/* add char to input buffer */
 switch(Chr)
   {case BS:
      if(InBufLen>0)
        {/* backup on screen */
         DisplayChar(BS);
         /* remove last char from buffer */
         InBufLen--;
        }
      break;
    default:
      /* display char */
      DisplayChar(Chr);
      /* put into buffer */
      if(InBufLen<MAX_BUF) InBuffer[InBufLen++] = Chr;
      break;
   }
}

static void DisplayError(int Code, LPSTR Msg)
{DisplayString("ERROR: ");
 if(Msg) DisplayString(Msg);
 if(Code)
   {wilErrorText(Code,(LPSTR)Temp,50);
    DisplayString((LPSTR)Temp);
    wsprintf((LPSTR)Temp," (%d)",Code);
    DisplayLine((LPSTR)Temp);
   }
 SetCursor(ArrowCursor);
}

/* FINGER driver */

void wilDriver(int FirstState)
{int Code;
 static Lines = 0;
 static int  DriverState = 0;   /* state variable */
 static ULONG TimeMark = 0;     /* future time value */
 /* initialize <DriverState> ? */
 if(FirstState>0) DriverState = FirstState;
 /* execute next state */
 switch(DriverState)
   {case 0:
      /* state driver is idle */
      return;

    case S_CONNECT:
      DriverState = 0;
      /* put up hour glass */
      SetCursor(WaitCursor);
      /* get host IP address */
      if(wilIsDotted((LPSTR)HostName))
        {/* already have IP address */
         HostAddr = wilMakeAddr((LPSTR)HostName);
        }
      else
        {
         HostAddr = wilGetHostAddr((LPSTR)HostName,0);
         if(HostAddr==0)
           {DisplayLine("No such host name.");
            return;
           }
        }
      DisplayString("Host IP = ");
      wilMakeDotted(HostAddr,(LPSTR)Temp,MAX_BUF);
      DisplayLine((LPSTR)Temp);
      /* create socket */
      Socket = wilTcpSocket();
      if((int)Socket<0)
        {DisplayError((int)Code, "wilTcpSocket: ");
         return;
        }
      /* attempt to connect to remote host */
      Code = wilConnect(Socket,HostAddr,FINGER_PORT);
      if(Code<0)
        {DisplayError(Code, NULL);
         return;
        }
      /* wait up to 30 seconds for connection */
      TimeMark = GetCurrentTime() + 30000;
      DriverState = S_WAIT_FOR_CONNECT;
      return;

    case S_WAIT_FOR_CONNECT:
      /* wait for connection */
      if(wilIsConnected(Socket,0))
        {DisplayLine("Connected to FINGER server");
         DriverState = S_SEND_USER;
         return;
        }
      if(GetCurrentTime() > TimeMark)
        {DisplayError(0, "Cannot CONNECT");
         DriverState = 0;
        }
      return;

    case S_SEND_USER:
      /* send user name */
      wsprintf((LPSTR)Temp,"%s\r\n",(LPSTR)UserName);
      Code = wilWriteString(Socket,(LPSTR)Temp);
      if(Code<0)
        {/* an error has occurred while writing to the socket */
         DisplayError(Code, "wilWriteString: ");
         DriverState = 0;
        }
      Lines = 0;
      DriverState = S_AWAIT_INCOMING;
      return;

   case S_AWAIT_INCOMING:
      if(wilDataIsReady(Socket,0)) DriverState = S_SUCCESS;
      return;

   case S_SUCCESS:
      while(1)
        {Code = wilReadLine(Socket,(LPSTR)InBuffer,MAX_BUF);
         if(Code==WIL_EOF)
           {/* all done */
            DriverState = 0;
            wilCloseSocket(Socket,500);
            SetCursor(ArrowCursor);
            DisplayLine("\r\nFINGER completed.");
            return;
           }
         if(Code<0)
           {/* an error has occurred while reading socket */
            DriverState = 0;
            DisplayError(Code, "wilReadLine: ");
            return;
           }
         if(Code==0)
           {/* line not ready */
            return;
           }
         /* (Code>0) display this line */
         Lines++;
         wsprintf((LPSTR)Temp,"> %s", (LPSTR)InBuffer);
         DisplayString((LPSTR)Temp);
        } /*end-while*/
      return;
   } /* end-switch(DriverState) */
}


/* WinMain */


#ifdef WIN32
int WINAPI
#else
int PASCAL
#endif
WinMain(USE_INS hInst, USE_INS hPrevInstance,
        USE_PTR szCmdLine,  int nCmdShow)
{WNDCLASS  wc;
 MSG msg;
 BOOL Result;
 if(!hPrevInstance)
   {/* register main window class */
    wc.style = CS_HREDRAW | CS_VREDRAW;
    wc.lpfnWndProc = MainWndProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
    wc.hInstance = hInst;
    wc.hIcon = LoadIcon(hInst, "HostIcon");
    wc.hCursor = NULL;
    wc.hbrBackground = (HBRUSH)GetStockObject(WHITE_BRUSH);
    wc.lpszMenuName =  "HostMenu";
    wc.lpszClassName = "HostWClass";
    Result = RegisterClass(&wc);
    if(!Result) return FALSE;
   }

 /* create main window */
 hInstance = hInst;
 hMainWnd = CreateWindow(
        "HostWClass",   "FINGER",    WS_OVERLAPPEDWINDOW,
        CW_USEDEFAULT,  CW_USEDEFAULT,
        WinWidth,       WinHeight,
        NULL,           NULL,
        hInstance,      NULL);
 ShowWindow(hMainWnd, nCmdShow);
 UpdateWindow(hMainWnd);
 hMenu = GetMenu(hMainWnd);

 /* window control loop */

 while(GetMessage(&msg,NULL,0,0))
   {
    TranslateMessage(&msg);
    DispatchMessage(&msg);
   }
 return (msg.wParam);
} /* end WinMain */

#ifdef WIN32
LRESULT CALLBACK
#else
long FAR PASCAL
#endif
MainWndProc(HWND hWindow,UINT iMsg,WPARAM wParam,LPARAM lParam)
{int Code;
 HDC hDC;
 PAINTSTRUCT ps;
#ifdef WIN32
#else
 static FARPROC lpfnAboutDlgProc;
#endif
 hMainWnd = hWindow;
 switch (iMsg)
    {case WM_CREATE:
      /* create cursors */
      ArrowCursor = LoadCursor(NULL, IDC_ARROW);
      WaitCursor = LoadCursor(NULL, IDC_WAIT);
      SetCursor(ArrowCursor);
#ifdef WIN32
#else
      /* create thunk for Win16 */
      lpfnAboutDlgProc = MakeProcInstance(AboutDlgProc,hInstance);
#endif
      /* initialize paint module */
      PaintInit();
      /* attach WINSOCK */
      DisplayString("Attaching WINSOCK...");
      Code = wilAttach();
      DisplayLine("OK");
      if(Code<0) DisplayError(Code,"wilAttach fails:");
      else
        {DisplayString(" Description: ");
         wilGetDescription((LPSTR)Temp, MAX_BUF);
         DisplayLine((LPSTR)Temp);
         DisplayString("My Host Name: ");
         wilGetMyHostName((LPSTR)Temp, MAX_BUF);
         DisplayLine((LPSTR)Temp);
         DisplayString("My Host Addr: ");
         wilGetMyHostDotted(0,(LPSTR)Temp, MAX_BUF);
         DisplayLine((LPSTR)Temp);
        }
       /* start timer for state driver (0.50 second) */
       idTimer = SetTimer(hMainWnd,1,500,NULL);
       if(idTimer==0) DisplayLine("ERROR: No timers remaining!");
       break;

     case WM_COMMAND:
         switch(wParam)
           {case MSG_ABOUT :
#ifdef WIN32
               DialogBox(hInstance, "AboutBox", hMainWnd, AboutDlgProc);
#else
               DialogBox(hInstance, "AboutBox", hMainWnd, lpfnAboutDlgProc);
#endif
               return 0;

            case MSG_EXIT:
              wilRelease();
              DestroyWindow(hMainWnd);
              break;

            case MSG_DEBUG:
              Code = wilDebug(0);
              wsprintf((LPSTR)Temp,"Debug(0) returned %xH",Code);
              DisplayLine((LPSTR)Temp);
              break;

            case MSG_FINGER:
              InBufLen = 0;
              DisplayString("Enter user (usr@domain):");
              break;

           }
         break;

    case WM_PAINT:
      HideCaret(hMainWnd);
      hDC = BeginPaint(hMainWnd, &ps);
      SetMapMode(hDC,MM_ANISOTROPIC);
      SelectObject(hDC, GetStockObject(OEM_FIXED_FONT) );
      PaintMain(hDC,&ps);
      EndPaint(hMainWnd,&ps);
      SetCaretPos(PaintGetColPos(),PaintGetRowPos());
      ShowCaret(hMainWnd);
      break;

    case WM_DESTROY:
      PostQuitMessage(0);
      break;

    case WM_TIMER:
      /* execute next state */
      wilDriver(0);
      break;

    case WM_CHAR:
      if(wParam==CR)
        {/* do the CR */
         DisplayChar((char)wParam);
         /* user has completed input */
         DisplayChar(LF);
         InBuffer[InBufLen] = '\0';
         /* execute command */
         wsprintf((LPSTR)Temp,"User@Host: %s",(LPSTR)InBuffer);
         DisplayLine((LPSTR)Temp);
         /* extract user & host names */
         Ptr = StringChar((LPSTR)InBuffer,'@');
         if(Ptr==NULL)
           {DisplayError(Code, "Cannot recognize User@Host");
            break;
           }
         *Ptr = '\0';
         lstrcpy((LPSTR)UserName, (LPSTR)InBuffer);
         lstrcpy((LPSTR)HostName, (LPSTR)(++Ptr));
         wsprintf((LPSTR)Temp,"User ='%s'", (LPSTR)UserName);
         DisplayLine(Temp);
         wsprintf((LPSTR)Temp,"Host ='%s'", (LPSTR)HostName);
         DisplayLine(Temp);
         /* start state driver */
         wilDriver(S_CONNECT);
        }
      else
        {/* add char to input buffer */
         Add2Buffer((char)wParam);
        }
      break;

    default:
      return (DefWindowProc(hMainWnd, iMsg, wParam, lParam));
   }
 return 0;
}


