#
# webcalng - a web based calendar program.
# Copyright 2003 - webcalng software solutions
#

#
# This file contains perl subroutines used for doing authentication i/o
# when using a flat file database backend.  Depending on the authentication
# type, not all of these subroutines will be used.
#

package webcalng_auth_io;
use strict;
use Fcntl ':flock';

#
# Retrieve session information for a given key.  This is only used when session based
# authentication is being used.
#
sub get_session_info {
	my ($given_session_key) = (@_);
	my ($session_key,$username,$time,$found_session);
	my $session_file  = $::db_dir . "/sessions";
	$found_session = 0;
	open FILE, "<$session_file" or webcalng_subs::hard_error("Could not read $session_file: $!\n");
	flock FILE, LOCK_SH or webcalng_subs::hard_error("Could not lock file $session_file: $!\n");
	while (<FILE>) {
		chomp;
		($session_key,$username,$time) = split /:/, $_, 3;
		$found_session = 1, last if ($session_key eq $given_session_key);
	}
	close FILE;
	if (! $found_session) {
		$username = "";
		$time     = "";
	} 
	return ($username,$time);
}

#
# Remove any old session keys.  If a specific session key is passed in, just 
# remove that one.  This is only used when session based authentication is being used.
#
sub remove_expired_sessions {
	my ($given_session_key) = (@_);
	my ($session_file,$now,$time,$session_key,@data);
	$session_file = $::db_dir . "/sessions";
	$now          = time;
	open FILE, "+>>$session_file" or webcalng_subs::hard_error("Could not write $session_file: $!\n");
	flock FILE, LOCK_EX or webcalng_subs::hard_error("Could not lock file $session_file: $!\n");
	seek FILE, 0, 0;
	if ($given_session_key) {
		while (<FILE>) {
			($session_key) = (split /:/, $_, 3)[0];
			push(@data,$_) unless ($session_key eq $given_session_key);
		}
	} else {
		while (<FILE>) {
			($time) = (split /:/, $_, 3)[2];
			chomp $time;
			push(@data,$_) unless (($now - $time) > $::webcalng_conf{'SESSION_LENGTH'});
		}
	}
	seek FILE, 0, 0;
	truncate FILE, 0;
	print FILE @data;
	close FILE;
	return 1;
}

#
# Save a new session key.  This is only used when session based authentication is being used.
#
sub save_session_key {
	my ($session_key,$username,$time) = (@_);
	my $session_file  = $::db_dir . "/sessions";
	my ($line);
	$line = join ':', $session_key, $username, $time;
	open FILE, ">>$session_file" or webcalng_subs::hard_error("Could not read $session_file: $!\n");
	flock FILE, LOCK_EX or webcalng_subs::hard_error("Could not lock file $session_file: $!\n");
	print FILE "$line\n";
	close FILE;
	return 1;
}

#
# Retrieve the salt for a given password.
#
sub get_salt {
	my ($given_username) = (@_);
	my $password_file = $::db_dir . "/passwords";
	my ($username,$password,$salt,$found_user);
	$found_user = $salt = 0;
	open FILE, "<$password_file" or webcalng_subs::hard_error("Could not read $password_file: $!\n");
	flock FILE, LOCK_SH or webcalng_subs::hard_error("Could not lock file $password_file: $!\n");
	while (<FILE>) {
		chomp;
		($username,$password)  = split /:/, $_, 2;
		$found_user = 1, last if ($username eq $given_username);
	}
	$salt = substr($password, 0, 2) if ($found_user);
	close FILE;
	return $salt;
}

#
# See if a given user already exists.  Should return 1 if user exists, 0 otherwise.
#
sub lookup_user_io {
	my ($given_username) = (@_);
	my $password_file = $::db_dir . "/passwords";
	my ($username,$user_exists);
	$user_exists = 0;
	open FILE, "<$password_file" or webcalng_subs::hard_error("Could not read $password_file: $!\n");
	flock FILE, LOCK_SH or webcalng_subs::hard_error("Could not lock file $password_file: $!\n");
	while (<FILE>) {
		($username)  = (split /:/, $_, 2)[0];
		$user_exists = 1, last if ($username eq $given_username);
	}
	close FILE;
	return $user_exists;
}

#
# See if the username/password combination given is correct.  This should return 1 if
# the username/password is correct, and 0 otherwise.  The given_password that is passed
# in needs to be encrypted.
#
sub validate_login_io {
	my ($given_username,$given_password) = (@_);
	my $password_file = $::db_dir . "/passwords";
	my ($valid,$username,$password);
	$valid = 0;
	open FILE, "<$password_file" or webcalng_subs::hard_error("Could not read $password_file: $!\n");
	flock FILE, LOCK_SH or webcalng_subs::hard_error("Could not lock file $password_file: $!\n");
	while (<FILE>) {
		chomp;
		($username,$password) = split /:/, $_, 2;
		$valid = 1, last if (($username eq $given_username) && ($given_password eq $password));
	}
	close FILE;
	return $valid;
}

#
# Get a list of users.
#
sub get_user_list_io {
	my $password_file = $::db_dir . "/passwords";
	my ($user,@users);
	open FILE, "<$password_file" or webcalng_subs::hard_error("Could not read $password_file: $!\n");
	flock FILE, LOCK_SH or webcalng_subs::hard_error("Could not lock file $password_file: $!\n");
	while (<FILE>) {
		($user)  = (split /:/, $_, 2)[0];
		push(@users,$user);
	}
	close FILE;
	return @users;
}

#
# Create a new account.
#
sub create_user_io {
	my ($username,$password) = (@_);
	my $password_file = $::db_dir . "/passwords";
	webcalng_subs::hard_error("Empty username andor password passed to create_user.") unless ($password && $username);
	open FILE, ">>$password_file" or webcalng_subs::hard_error("Could not write $password_file: $!\n");
	flock FILE, LOCK_EX or webcalng_subs::hard_error("Could not lock file $password_file: $!\n");
	print FILE "$username:$password\n";
	close FILE;
	return 1;
}

#
# Remove an account.
#
sub remove_user_io {
	my ($user) = (@_);
	my ($password_file,@data);

	# Set up variables and do sanity checks.
	$password_file = $::db_dir . "/passwords";
	webcalng_subs::hard_error("Empty username passed to remove_user.") unless ($user);

	# Open filehandles.
	open FILE, "+>>$password_file" or webcalng_subs::hard_error("Could not write $password_file: $!\n");
	flock FILE, LOCK_SH or webcalng_subs::hard_error("Could not lock file $password_file: $!\n");
	seek FILE, 0, 0;

	# Remove the account.
	while (<FILE>) {
		push(@data,$_) unless /^${user}:/;
	}

	# Close filehandles and put new password file in place
	seek FILE, 0, 0;
	truncate FILE, 0;
	print FILE @data;
	close FILE;

	return 1;
}

1;
