#
# webcalng - a web based calendar program.
# Copyright 2003 - webcalng software solutions
#

#
# This file contains perl subroutines used for doing authentication i/o
# when using a relational database backend.  Depending on the authentication
# type, not all of these subroutines will be used.
#
# Whenever we insert or update rows in a table, the operations are wrapped
# in an eval statement and any errors cause an exception to be raised.  This
# is done so that we can rollback any partially made changes if the operation
# fails at some point.  If the database does not support transactions with
# rollbacks, the DBD module will just ignore the rollback call.
#

package webcalng_auth_io;
use strict;
use Fcntl ':flock';

#
# Retrieve session information for a given key.  This is only used when session based
# authentication is being used.
#
sub get_session_info {
	my ($given_session_key) = (@_);
	my ($session_key,$username,$time,$found_session,$session_table,$sql,$sth,$ref);

	$found_session = 0;
	$session_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "sessions";
	$sql           = qq{ SELECT webcalng_username,webcalng_time FROM $session_table WHERE webcalng_sessionkey=? };
	$sth           = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	$sth->execute($given_session_key) or webcalng_subs::hard_error($DBI::errstr);

	$ref = $sth->fetchrow_hashref('NAME_lc');
	$sth->finish();
	if ($ref->{'webcalng_username'}) {
		$username = $ref->{'webcalng_username'};
		$time     = $ref->{'webcalng_time'};
	} else {
		$username = "";
		$time     = "";
	}

	return ($username,$time);
}

#
# Remove any old session keys.  If a specific session key is passed in, just 
# remove that one.  This is only used when session based authentication is being used.
#
sub remove_expired_sessions {
	my ($given_session_key) = (@_);
	my ($session_table,$now,$time,$target_time,$session_key,@data,$sql,$sth,$ref);

	$session_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "sessions";
	if ($given_session_key) {
		$sql = qq{ DELETE FROM $session_table WHERE webcalng_sessionkey=? };
		$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
		eval {
			$sth->execute($given_session_key) or die $sth->errstr;
			$::dbh->commit();
		};
		if ($@) {
			$::dbh->rollback();
			webcalng_subs::hard_error($@);
		}
		$sth->finish();
	} else {
		$now           = time;
		$target_time   = $now - $::webcalng_conf{'SESSION_LENGTH'};
		$sql           = qq{ DELETE FROM $session_table WHERE $target_time > webcalng_time };
		$sth           = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
		eval {
			$sth->execute() or die $sth->errstr;
			$::dbh->commit();
		};
		if ($@) {
			$::dbh->rollback();
			webcalng_subs::hard_error($@);
		}
		$sth->finish();
	}

	return 1;
}

#
# Save a new session key.  This is only used when session based authentication is being used.
#
sub save_session_key {
	my ($session_key,$username,$time) = (@_);
	my ($session_table,$sth,$sql);

	$session_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "sessions";
	$sql           = qq{ INSERT INTO $session_table VALUES (?,?,?) };
	$sth           = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	eval {
		$sth->execute($session_key,$username,$time) or die $sth->errstr;
		$::dbh->commit();
	};
	if ($@) {
		$::dbh->rollback();
		webcalng_subs::hard_error($@);
	}
	$sth->finish();

	return 1;
}

#
# Retrieve the salt for a given password.
#
sub get_salt {
	my ($given_username) = (@_);
	my ($password_table,$username,$password,$salt,$sql,$sth,$ref);

	$password_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "passwords";
	$sql            = qq{ SELECT webcalng_password FROM $password_table WHERE webcalng_username=? };
	$sth            = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	$sth->execute($given_username) or webcalng_subs::hard_error($DBI::errstr);
	$ref = $sth->fetchrow_hashref('NAME_lc');
	$sth->finish();

	if ($ref->{'webcalng_password'}) {
		$salt = substr($ref->{'webcalng_password'}, 0, 2);
	} else {
		$salt = "";
	}

	return $salt;
}

#
# See if a given user already exists.  Should return 1 if user exists, 0 otherwise.
#
sub lookup_user_io {
	my ($given_username) = (@_);
	my ($password_table,$username,$user_exists,$sql,$sth,$ref);

	$user_exists    = 0;
	$password_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "passwords";
	$sql            = qq{ SELECT webcalng_username FROM $password_table WHERE webcalng_username=? };
	$sth            = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	$sth->execute($given_username) or webcalng_subs::hard_error($DBI::errstr);
	$ref = $sth->fetchrow_hashref('NAME_lc');
	$sth->finish();
	$user_exists++ if ($ref->{'webcalng_username'});
	
	return $user_exists;
}

#
# See if the username/password combination given is correct.  This should return 1 if
# the username/password is correct, and 0 otherwise.  The given_password that is passed
# in needs to be encrypted.
#
sub validate_login_io {
	my ($given_username,$given_password) = (@_);
	my ($password_table,$valid,$username,$password,$sql,$sth,$ref);

	$valid          = 0;
	$password_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "passwords";
	$sql            = qq{ SELECT webcalng_username FROM $password_table WHERE webcalng_username=? AND webcalng_password=? };
	$sth            = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	$sth->execute($given_username,$given_password) or webcalng_subs::hard_error($DBI::errstr);

	$ref = $sth->fetchrow_hashref('NAME_lc');
	$sth->finish();
	$valid++ if ($ref->{'webcalng_username'});

	return $valid;
}

#
# Get a list of users.
#
sub get_user_list_io {
	my ($password_table,@users,$sql,$sth,$ref);

	$password_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "passwords";
	$sql            = qq{ SELECT webcalng_username FROM $password_table };
	$sth            = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	$sth->execute() or webcalng_subs::hard_error($DBI::errstr);
	while($ref = $sth->fetchrow_hashref('NAME_lc')) {
		push(@users,$ref->{'webcalng_username'});
	}
	$sth->finish();

	return @users;
}

#
# Create a new account.
#
sub create_user_io {
	my ($username,$password) = (@_);
	my ($password_table,$sql,$sth);

	$password_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "passwords";
	$sql            = qq{ INSERT INTO $password_table VALUES (?,?) };
	$sth            = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	eval {
		$sth->execute($username,$password) or die $sth->errstr;
		$::dbh->commit();
	};
	if ($@) {
		$::dbh->rollback();
		webcalng_subs::hard_error($@);
	}
	$sth->finish();

	# Keep the htaccess password file in synch if needed.
	create_user_io_flat($username,$password) if (! $::webcalng_conf{'USE_COOKIES'});

	return 1;
}

#
# Create a new account.  This is called if we are using htaccess style authentication,
# as that relies on the flat password file to authenticate users.  We just keep the
# flat and and database table in synch if using htaccess style authentication.
#
sub create_user_io_flat {
	my ($username,$password) = (@_);
	my $password_file = $::db_dir . "/passwords";
	webcalng_subs::hard_error("Empty username andor password passed to create_user.") unless ($password && $username);
	open FILE, ">>$password_file" or webcalng_subs::hard_error("Could not write $password_file: $!\n");
	flock FILE, LOCK_EX or webcalng_subs::hard_error("Could not lock file $password_file: $!\n");
	print FILE "$username:$password\n";
close FILE;
	return 1;
}

#
# Remove an account.
#
sub remove_user_io {
	my ($user) = (@_);
	my ($password_table,$sql,$sth);

	$password_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "passwords";
	$sql            = qq{ DELETE FROM $password_table WHERE webcalng_username=? };
	$sth            = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	eval {
		$sth->execute($user) or die $sth->errstr;
		$::dbh->commit();
	};
	if ($@) {
		$::dbh->rollback();
		webcalng_subs::hard_error($@);
	}
	$sth->finish();

	# Keep the htaccess password file in synch if needed.
	remove_user_io_flat($user) if (! $::webcalng_conf{'USE_COOKIES'});

	return 1;
}

#
# Remove an account.  This is called if we are using htaccess style authentication,
# as that relies on the flat password file to authenticate users.  We just keep the
# flat and and database table in synch if using htaccess style authentication.
#
sub remove_user_io_flat {
	my ($user) = (@_);
	my ($password_file,@data);

	# Set up variables and do sanity checks.
	$password_file = $::db_dir . "/passwords";
	webcalng_subs::hard_error("Empty username passed to remove_user.") unless ($user);

	# Open filehandles.
	open FILE, "+>>$password_file" or webcalng_subs::hard_error("Could not write $password_file: $!\n");
	flock FILE, LOCK_SH or webcalng_subs::hard_error("Could not lock file $password_file: $!\n");
	seek FILE, 0, 0;

        # Remove the account.
	while (<FILE>) {
		push(@data,$_) unless /^${user}:/;
	}

	# Close filehandles and put new password file in place
	seek FILE, 0, 0;
	truncate FILE, 0;
	print FILE @data;
	close FILE;

        return 1;
}

1;
