#
# webcalng - a web based calendar program.
# Copyright 2003 - webcalng software solutions
#

#
# This file contains perl subroutines used for data input and output.
# These subroutine are for use with a relational database backend.
#
# Whenever we insert or update rows in a table, the operations are wrapped
# in an eval statement and any errors cause an exception to be raised.  This
# is done so that we can rollback any partially made changes if the operation
# fails at some point.  If the database does not support transactions with 
# rollbacks, the DBD module will just ignore the rollback call.
#

package webcalng_io;
use strict;
use DBI;

#
# Get a hash of the fields for admin preferences.
#
sub get_admin_preferences {
	my ($preferences_table,%data,$sql,$sth,$ref);

	$preferences_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "preferences";
	$sql = qq{ SELECT * FROM $preferences_table WHERE webcalng_calendar=? };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	$sth->execute("admin") or webcalng_subs::hard_error($DBI::errstr);
	while ($ref = $sth->fetchrow_hashref('NAME_lc')) {
		my $key                 = $ref->{'webcalng_key'};
		$data{$key}{'value'}    = $ref->{'webcalng_value'} || "";
		$data{$key}{'options'}  = $ref->{'webcalng_options'} || "";
		$data{$key}{'category'} = $ref->{'webcalng_category'};
		$data{$key}{'usermod'}  = $ref->{'webcalng_usermod'};
		$data{$key}{'order'}    = $ref->{'webcalng_order'};
	}
	$sth->finish();

	return \%data;
}

#
# Get a hash of the fields for user preferences.
#
sub get_calendar_preferences {
	my ($calendar) = (@_);
	my ($preferences_table,%data,$sql,$sth,$ref);

	$preferences_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "preferences";
	$sql = qq{ SELECT webcalng_key,webcalng_value FROM $preferences_table WHERE webcalng_calendar=? };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	$sth->execute($calendar) or webcalng_subs::hard_error($DBI::errstr);
	while ($ref = $sth->fetchrow_hashref('NAME_lc')) {
		my $key     = $ref->{'webcalng_key'};
		$data{$key} = $ref->{'webcalng_value'} || "";

	}
	$sth->finish();

	return \%data;
}

#
# Subroutine to write changes for the admin preferences to the preference table.
# The value and/or the usermod functionality can change for any given preference key.

# 
sub save_admin_preferences {
	my ($dataref) = (@_);
	my ($preferences_table,$sql,$sth,$key);

	$preferences_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "preferences";
	$sql = qq{ UPDATE $preferences_table SET webcalng_value=?,webcalng_usermod=? WHERE webcalng_calendar=? AND webcalng_key=? };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	for $key (%$dataref) {
		if (defined $dataref->{$key}{'value'}) {
			eval {
				$sth->execute($dataref->{$key}{'value'},$dataref->{$key}{'usermod'},"admin",$key) or die $sth->errstr;
			};
			if ($@) {
				$::dbh->rollback();
				webcalng_subs::hard_error($@);
			}
		}
	}
	$::dbh->commit();
	$sth->finish();

	return 1;
}

#
# Subroutine to write changes for a user calendar to the preference table.
# Users can only update the value for a preference key in their calendar.
# 
sub save_calendar_preferences {
	my ($dataref) = (@_);
	my ($preferences_table,$sql1,$sql2,$sth1,$sth2,$key);

	$preferences_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "preferences";
	$sql1 = qq{ DELETE FROM $preferences_table WHERE webcalng_calendar=? AND webcalng_key=? };
	$sql2 = qq{ INSERT INTO $preferences_table (webcalng_calendar,webcalng_key,webcalng_value) VALUES (?,?,?) };
	$sth1 = $::dbh->prepare($sql1) or webcalng_subs::hard_error($DBI::errstr);
	$sth2 = $::dbh->prepare($sql2) or webcalng_subs::hard_error($DBI::errstr);
	for $key (%$dataref) {
		if ((defined $dataref->{$key}{'value'}) && ($::preferences{$key}{'usermod'})) {
			eval {
				$sth1->execute($::calendar,$key) or die $sth1->errstr;
			};
			if ($@) {
				$::dbh->rollback();
				webcalng_subs::hard_error($@);
			}
			eval {
				$sth2->execute($::calendar,$key,$dataref->{$key}{'value'}) or die $sth2->errstr;
			};
			if ($@) {
				$::dbh->rollback();
				webcalng_subs::hard_error($@);
			}
		}
	}
	$::dbh->commit();
	$sth1->finish();
	$sth2->finish();

	return 1;
}

#
# Get a list of available calendars to display.  If they are
# searching for a calendar, send back a list of all calendars
# that match - or all available calendars if there are not matches.
#
sub get_calendar_list {
	my ($dir,@dir,@list,@all,$calendars_table,$sql,$sth,$ref);

	$calendars_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "calendars";
	$sql = qq{ SELECT webcalng_calendar FROM $calendars_table };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	$sth->execute() or webcalng_subs::hard_error($DBI::errstr);
	while ($ref = $sth->fetchrow_hashref('NAME_lc')) {
		if ($::calendar) {
			push(@list, $ref->{'webcalng_calendar'}) if ($ref->{'webcalng_calendar'} =~ /$::calendar/i);
			return $ref->{'webcalng_calendar'} if ($ref->{'webcalng_calendar'} eq $::calendar);
		} else {
			push(@list, $ref->{'webcalng_calendar'});
		}
		push(@all,$ref->{'webcalng_calendar'});
	}
	$sth->finish();

	if ($#list == -1) {
		return @all;	
	} else {
		return @list;	
	}
}

#
# Find out who owns and has read and/or write access to a calendar.
#
sub get_calendar_permissions {
	my ($calendar) = (@_);
	my (%data,$calendars_table,$sql,$sth,$ref);
	return unless ($calendar);

	# Get data from calendars table.
	$calendars_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "calendars";
	$sql = qq{ SELECT webcalng_type,webcalng_owner FROM $calendars_table WHERE webcalng_calendar=? };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	$sth->execute($calendar) or webcalng_subs::hard_error($DBI::errstr);
	$ref = $sth->fetchrow_hashref('NAME_lc');
	$sth->finish();

	# Find calendar owner.
	$data{'owner'} = $ref->{'webcalng_owner'};

	# Find type of calendar.
	$data{'type'} = $ref->{'webcalng_type'};

	# Create regexp of users who have read/write access, including the calendar owner.
	# Format of regexp is: user1|user2|user3|user4
	if ($data{'owner'}) {
		$data{'read'}  = $data{'owner'} . " " . $::preferences{'read_access'}{'value'};	
		$data{'write'} = $data{'owner'} . " " . $::preferences{'write_access'}{'value'};	
		$data{'read'}  =~ s/\s+/|/g;
		$data{'write'} =~ s/\s+/|/g;
		$data{'read'}  =~ s/\,/|/g;
		$data{'write'} =~ s/\,/|/g;
		$data{'read'}  =~ s/\|$//;
		$data{'write'} =~ s/\|$//;
		$data{'read'}  =~ s/^\|//;
		$data{'write'} =~ s/^\|//;
	} else {
		$data{'read'}  = "";
		$data{'write'} = "";
	}

	return \%data;
}

#
# See if a given calendar name already exists.  This should return 0 if the calendar does
# not exist, and 1 if it does.
#
sub calendar_exists {
	my ($calendar) = (@_);
	my ($calendar_exists,$calendars_table,$sql,$sth,$ref);

	# Get data from calendars table.
	$calendars_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "calendars";
	$sql = qq{ SELECT webcalng_calendar FROM $calendars_table WHERE webcalng_calendar=? };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	$sth->execute($calendar) or webcalng_subs::hard_error($DBI::errstr);
	$ref = $sth->fetchrow_hashref('NAME_lc');
	$sth->finish();

	$calendar_exists = 0;
	$calendar_exists++ if ($ref->{'webcalng_calendar'});

	return $calendar_exists;
}

#
# Create a new calendar.
#
sub create_calendar {
	my ($calendar,$type,$username) = (@_);
	my ($calendars_table,$sql,$sth);

	$calendars_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "calendars";
	$sql = qq{ INSERT INTO $calendars_table VALUES (?,?,?) };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	eval {
		$sth->execute($calendar,$username,$type) or die $sth->errstr;
		$::dbh->commit();
	};
	if ($@) {
		$::dbh->rollback();
		webcalng_subs::hard_error($@);
	}
	$sth->finish();

	return 1;
}

#
# Remove all evidence of a calendar's existence!
#
sub remove_calendar {
	my ($calendar) = (@_);
	my ($calendars_table,$items_table,$exceptions_table,$meetings_table,$addresses_table,$tasks_table,$preferences_table);
	my ($sql1,$sql2,$sql3,$sql4,$sql5,$sql6,$sql7,$sth1,$sth2,$sth3,$sth4,$sth5,$sth6,$sth7);

	return 1 if ((! defined $calendar) || ($calendar eq ""));
	$calendars_table   = $::webcalng_conf{'DB_TABLE_PREFIX'} . "calendars";
	$items_table       = $::webcalng_conf{'DB_TABLE_PREFIX'} . "items";
	$exceptions_table  = $::webcalng_conf{'DB_TABLE_PREFIX'} . "exceptions";
	$meetings_table    = $::webcalng_conf{'DB_TABLE_PREFIX'} . "meetings";
	$addresses_table   = $::webcalng_conf{'DB_TABLE_PREFIX'} . "addresses";
	$tasks_table       = $::webcalng_conf{'DB_TABLE_PREFIX'} . "tasks";
	$preferences_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "preferences";
	$sql1 = qq{ DELETE FROM $calendars_table WHERE webcalng_calendar=? };
	$sql2 = qq{ DELETE FROM $items_table WHERE webcalng_calendar=? };
	$sql3 = qq{ DELETE FROM $exceptions_table WHERE webcalng_calendar=? };
	$sql4 = qq{ DELETE FROM $meetings_table WHERE webcalng_calendar=? };
	$sql5 = qq{ DELETE FROM $addresses_table WHERE webcalng_calendar=? };
	$sql6 = qq{ DELETE FROM $tasks_table WHERE webcalng_calendar=? };
	$sql7 = qq{ DELETE FROM $preferences_table WHERE webcalng_calendar=? };
	$sth1 = $::dbh->prepare($sql1) or webcalng_subs::hard_error($DBI::errstr);
	$sth2 = $::dbh->prepare($sql2) or webcalng_subs::hard_error($DBI::errstr);
	$sth3 = $::dbh->prepare($sql3) or webcalng_subs::hard_error($DBI::errstr);
	$sth4 = $::dbh->prepare($sql4) or webcalng_subs::hard_error($DBI::errstr);
	$sth5 = $::dbh->prepare($sql5) or webcalng_subs::hard_error($DBI::errstr);
	$sth6 = $::dbh->prepare($sql6) or webcalng_subs::hard_error($DBI::errstr);
	$sth7 = $::dbh->prepare($sql7) or webcalng_subs::hard_error($DBI::errstr);
	eval {
		$sth1->execute($calendar) or die $sth1->errstr;
		$sth2->execute($calendar) or die $sth2->errstr;
		$sth3->execute($calendar) or die $sth3->errstr;
		$sth4->execute($calendar) or die $sth4->errstr;
		$sth5->execute($calendar) or die $sth5->errstr;
		$sth6->execute($calendar) or die $sth6->errstr;
		$sth7->execute($calendar) or die $sth7->errstr;
		$::dbh->commit();
	};
	if ($@) {
		$::dbh->rollback();
		webcalng_subs::hard_error($@);
	}
	$sth1->finish();
	$sth2->finish();
	$sth3->finish();
	$sth4->finish();
	$sth5->finish();
	$sth6->finish();
	$sth7->finish();

	return 1;
}

#
# Update the owner and/or type of a calendar.
#
sub update_calinfo {
	my ($calendar,$owner,$type) = (@_);
	my ($calendars_table,$sql,$sth);

	$calendars_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "calendars";
	$sql = qq{ UPDATE $calendars_table SET webcalng_owner=?,webcalng_type=? WHERE webcalng_calendar=? };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	eval {
		$sth->execute($owner,$type,$calendar) or die $sth->errstr;
		$::dbh->commit();
	};
	if ($@) {
		$::dbh->rollback();
		webcalng_subs::hard_error($@);
	}
	$sth->finish();
	
	return 1;
}


#
# Change the name of a calendar.
#
sub update_calname {
	my ($oldcal,$cal,$type,$owner) = (@_);
	my ($calendars_table,$items_table,$exceptions_table,$meetings_table,$addresses_table,$tasks_table,$preferences_table);
	my ($sql1,$sql2,$sql3,$sql4,$sql5,$sql6,$sql7,$sth1,$sth2,$sth3,$sth4,$sth5,$sth6,$sth7);

	return 1 if ((! defined $cal) || (! defined $oldcal) || ($cal eq "") || ($oldcal eq ""));
	$calendars_table   = $::webcalng_conf{'DB_TABLE_PREFIX'} . "calendars";
	$items_table       = $::webcalng_conf{'DB_TABLE_PREFIX'} . "items";
	$exceptions_table  = $::webcalng_conf{'DB_TABLE_PREFIX'} . "exceptions";
	$meetings_table    = $::webcalng_conf{'DB_TABLE_PREFIX'} . "meetings";
	$addresses_table   = $::webcalng_conf{'DB_TABLE_PREFIX'} . "addresses";
	$tasks_table       = $::webcalng_conf{'DB_TABLE_PREFIX'} . "tasks";
	$preferences_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "preferences";
	$sql1 = qq{ UPDATE $calendars_table SET webcalng_calendar=? WHERE webcalng_calendar=? };
	$sql2 = qq{ UPDATE $items_table SET webcalng_calendar=? WHERE webcalng_calendar=? };
	$sql3 = qq{ UPDATE $exceptions_table SET webcalng_calendar=? WHERE webcalng_calendar=? };
	$sql4 = qq{ UPDATE $meetings_table SET webcalng_calendar=? WHERE webcalng_calendar=? };
	$sql5 = qq{ UPDATE $addresses_table SET webcalng_calendar=? WHERE webcalng_calendar=? };
	$sql6 = qq{ UPDATE $tasks_table SET webcalng_calendar=? WHERE webcalng_calendar=? };
	$sql7 = qq{ UPDATE $preferences_table SET webcalng_calendar=? WHERE webcalng_calendar=? };
	$sth1 = $::dbh->prepare($sql1) or webcalng_subs::hard_error($DBI::errstr);
	$sth2 = $::dbh->prepare($sql2) or webcalng_subs::hard_error($DBI::errstr);
	$sth3 = $::dbh->prepare($sql3) or webcalng_subs::hard_error($DBI::errstr);
	$sth4 = $::dbh->prepare($sql4) or webcalng_subs::hard_error($DBI::errstr);
	$sth5 = $::dbh->prepare($sql5) or webcalng_subs::hard_error($DBI::errstr);
	$sth6 = $::dbh->prepare($sql6) or webcalng_subs::hard_error($DBI::errstr);
	$sth7 = $::dbh->prepare($sql7) or webcalng_subs::hard_error($DBI::errstr);
	eval {
		$sth1->execute($cal,$oldcal) or die $sth1->errstr;
		$sth2->execute($cal,$oldcal) or die $sth2->errstr;
		$sth3->execute($cal,$oldcal) or die $sth3->errstr;
		$sth4->execute($cal,$oldcal) or die $sth4->errstr;
		$sth5->execute($cal,$oldcal) or die $sth5->errstr;
		$sth6->execute($cal,$oldcal) or die $sth6->errstr;
		$sth7->execute($cal,$oldcal) or die $sth7->errstr;
		$::dbh->commit();
	};
	if ($@) {
		$::dbh->rollback();
		webcalng_subs::hard_error($@);
	}
	$sth1->finish();
	$sth2->finish();
	$sth3->finish();
	$sth4->finish();
	$sth5->finish();
	$sth6->finish();
	$sth7->finish();

	return 1;
}


#
# Get the info for a given addressid, or all addressids.
#
sub get_address {
	my ($addressid) = (@_);
	my ($address_table,%data,$sql,$sth,$ref);

	$address_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "addresses";

	if ($addressid) {
		$sql = qq{ SELECT * FROM $address_table WHERE webcalng_calendar=? AND webcalng_addressid=? };
		$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
		$sth->execute($::calendar,$addressid) or webcalng_subs::hard_error($DBI::errstr);
	} else {
		$sql = qq{ SELECT * FROM $address_table WHERE webcalng_calendar=? };
		$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
		$sth->execute($::calendar) or webcalng_subs::hard_error($DBI::errstr);
	}

	while ($ref = $sth->fetchrow_hashref('NAME_lc')) {
		if ($addressid) {
			$data{'displayname'}  = $ref->{'webcalng_displayname'}  || "";
			$data{'emailaddress'} = $ref->{'webcalng_emailaddress'} || "";
			$data{'phonenumber'}  = $ref->{'webcalng_phonenumber'}  || "";
			$data{'phonenumber2'} = $ref->{'webcalng_phonenumber2'} || "";
		} else {
			my $tmpaddressid = $ref->{'webcalng_addressid'};
			$data{$tmpaddressid}{'displayname'}  = $ref->{'webcalng_displayname'}  || "";
			$data{$tmpaddressid}{'emailaddress'} = $ref->{'webcalng_emailaddress'} || "";
			$data{$tmpaddressid}{'phonenumber'}  = $ref->{'webcalng_phonenumber'}  || "";
			$data{$tmpaddressid}{'phonenumber2'} = $ref->{'webcalng_phonenumber2'} || "";
		}	
	}
	$sth->finish();

	return \%data;
}

#
# Determine if a given address already exists.
#
sub address_exists {
	my ($displayname) = (@_);
	my ($exists,$address_table,%data,$sql,$sth,$ref);

	$address_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "addresses";
	$sql = qq{ SELECT webcalng_displayname FROM $address_table WHERE webcalng_calendar=? AND webcalng_displayname=? };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	$sth->execute($::calendar,$displayname) or webcalng_subs::hard_error($DBI::errstr);
	$ref = $sth->fetchrow_hashref('NAME_lc');
	$sth->finish();
	$exists = 0;
	$exists++ if ($ref->{'webcalng_displayname'});

	return $exists;
}

#
# Add a new address record to the address table.
#
sub addaddress_io {
	my ($dataref) = (@_);
	my ($address_table,$sql,$sth,$addressid);

	$address_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "addresses";
	$sql = qq{ INSERT INTO $address_table (webcalng_calendar,webcalng_displayname,webcalng_emailaddress,webcalng_phonenumber,webcalng_phonenumber2) VALUES (?,?,?,?,?) };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	eval {
		$sth->execute($::calendar,$dataref->{'displayname'},$dataref->{'emailaddress'},$dataref->{'phonenumber'},$dataref->{'phonenumber2'}) or die $sth->errstr;
		$::dbh->commit();
	};
	if ($@) {
		$::dbh->rollback();
		webcalng_subs::hard_error($@);
	}
	$sth->finish();	

	return 1;
}

#
# Update an address record.
#
sub editaddress_io {
	my ($dataref) = (@_);
	my ($address_table,$sql,$sth);

	$address_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "addresses";
	$sql = qq{ UPDATE $address_table SET webcalng_displayname=?,webcalng_emailaddress=?,webcalng_phonenumber=?,webcalng_phonenumber2=? WHERE webcalng_calendar=? AND webcalng_addressid=? };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	eval {
		$sth->execute($dataref->{'displayname'},$dataref->{'emailaddress'},$dataref->{'phonenumber'},$dataref->{'phonenumber2'},$::calendar,$dataref->{'addressid'}) or die $sth->errstr;
		$::dbh->commit();
	};
	if ($@) {
		$::dbh->rollback();
		webcalng_subs::hard_error($@);
	}
	$sth->finish();	

	return 1;
}

#
# Delete an address record.
#
sub deladdress_io {
	my ($addressid) = (@_);
	my ($address_table,$sql,$sth);

	$address_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "addresses";
	$sql = qq{ DELETE FROM $address_table WHERE webcalng_calendar=? AND webcalng_addressid=? };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	eval {
		$sth->execute($::calendar,$addressid) or die $sth->errstr;
		$::dbh->commit();
	};
	if ($@) {
		$::dbh->rollback();
		webcalng_subs::hard_error($@);
	}
	$sth->finish();	

	return 1;
}

#
# Get the info for a given task id, or all taskids.
#
sub get_task {
	my ($taskid) = (@_);
	my ($task_table,%data,$sql,$sth,$ref);

	$task_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "tasks";

	if ($taskid) {
		$sql = qq{ SELECT * FROM $task_table WHERE webcalng_calendar=? AND webcalng_taskid=? };
		$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
		$sth->execute($::calendar,$taskid) or webcalng_subs::hard_error($DBI::errstr);
	} else {
		$sql = qq{ SELECT * FROM $task_table WHERE webcalng_calendar=? };
		$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
		$sth->execute($::calendar) or webcalng_subs::hard_error($DBI::errstr);
	}

	while ($ref = $sth->fetchrow_hashref('NAME_lc')) {
		if ($taskid) {
			$data{'taskdata'}  = $ref->{'webcalng_taskdata'}  || "";
			$data{'taskdone'} = $ref->{'webcalng_taskdone'}   || "";
		} else {
			my $tmptaskid = $ref->{'webcalng_taskid'};
			$data{$tmptaskid}{'taskdata'}  = $ref->{'webcalng_taskdata'}  || "";
			$data{$tmptaskid}{'taskdone'} = $ref->{'webcalng_taskdone'}   || "";
		}	
	}
	$sth->finish();

	return \%data;
}

#
# Add a new task to the task file.
#
sub addtask_io {
	my ($dataref) = (@_);
	my ($task_table,$sql,$sth);

	$task_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "tasks";
	$sql = qq{ INSERT INTO $task_table (webcalng_calendar,webcalng_taskdata,webcalng_taskdone) VALUES (?,?,?) };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	eval {
		$sth->execute($::calendar,$dataref->{'taskdata'},$dataref->{'taskdone'}) or die $sth->errstr;
		$::dbh->commit();
	};
	if ($@) {
		$::dbh->rollback();
		webcalng_subs::hard_error($@);
	}
	$sth->finish();	

	return 1;
}

#
# Update an task record.
#
sub edittask_io {
	my ($dataref) = (@_);
	my ($task_table,$sql,$sth);

	$task_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "tasks";
	$sql = qq{ UPDATE $task_table SET webcalng_taskdata=?,webcalng_taskdone=? WHERE webcalng_calendar=? AND webcalng_taskid=? };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	eval {
		$sth->execute($dataref->{'taskdata'},$dataref->{'taskdone'},$::calendar,$dataref->{'taskid'}) or die $sth->errstr;
		$::dbh->commit();
	};
	if ($@) {
		$::dbh->rollback();
		webcalng_subs::hard_error($@);
	}
	$sth->finish();	

	return 1;
}

#
# Mark a task complete.
#
sub marktask_io {
	my ($taskid) = (@_);
	my ($task_table,$sql,$sth);

	$task_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "tasks";
	$sql = qq{ UPDATE $task_table SET webcalng_taskdone=1 WHERE webcalng_calendar=? AND webcalng_taskid=? };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	eval {
		$sth->execute($::calendar,$taskid) or die $sth->errstr;
		$::dbh->commit();
	};
	if ($@) {
		$::dbh->rollback();
		webcalng_subs::hard_error($@);
	}
	$sth->finish();	

	return 1;
}

#
# Delete a task.
#
sub deltask_io {
	my ($taskid) = (@_);
	my ($task_table,$sql,$sth);

	$task_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "tasks";
	$sql = qq{ DELETE FROM $task_table WHERE webcalng_calendar=? AND webcalng_taskid=? };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	eval {
		$sth->execute($::calendar,$taskid) or die $sth->errstr;
		$::dbh->commit();
	};
	if ($@) {
		$::dbh->rollback();
		webcalng_subs::hard_error($@);
	}
	$sth->finish();	

	return 1;
}

#
# Add a new calendar item.
#
sub additem_io {
	my ($dataref,$itemid,$exceptionid,$exception,$calendar) = (@_);
	my ($items_table,$exceptions_table,$sql,$sth);

	# Set table names.
	$items_table      = $::webcalng_conf{'DB_TABLE_PREFIX'} . "items";
	$exceptions_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "exceptions";

	# Insert the data. 
	if ($exception) {

		$sql = qq{ INSERT INTO $exceptions_table (webcalng_calendar,webcalng_itemid,webcalng_modified,webcalng_user,webcalng_description,webcalng_exceptiondate,webcalng_starttime,webcalng_endtime,webcalng_visibility,webcalng_hyperlink,webcalng_notes) VALUES (?,?,?,?,?,?,?,?,?,?,?) };
		$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
		eval {
			$sth->execute($calendar,$itemid,$dataref->{'modified'},$::username,$dataref->{'description'},$dataref->{'exceptiondate'},$dataref->{'starttime'},$dataref->{'endtime'},$dataref->{'visibility'},$dataref->{'hyperlink'},$dataref->{'notes'}) or die $sth->errstr;
			$::dbh->commit();
		};
		if ($@) {
			$::dbh->rollback();
			webcalng_subs::hard_error($@);
		}
		$sth->finish();	

	} else {

		$sql = qq{ INSERT INTO $items_table (webcalng_calendar,webcalng_modified,webcalng_user,webcalng_description,webcalng_dateregexp,webcalng_startdate,webcalng_enddate,webcalng_starttime,webcalng_endtime,webcalng_visibility,webcalng_repeat,webcalng_repeatdays,webcalng_repeatend,webcalng_remindwho,webcalng_remindwhen,webcalng_remindersent,webcalng_meetingid,webcalng_hyperlink,webcalng_notes) VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?) };
		$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
		eval {
			$sth->execute($calendar,$dataref->{'modified'},$::username,$dataref->{'description'},$dataref->{'dateregexp'},$dataref->{'startdate'},$dataref->{'enddate'},$dataref->{'starttime'},$dataref->{'endtime'},$dataref->{'visibility'},$dataref->{'repeat'},$dataref->{'repeatdays'},$dataref->{'repeatend'},$dataref->{'remindwho'},$dataref->{'remindwhen'},$dataref->{'remindersent'},$dataref->{'meetingid'},$dataref->{'hyperlink'},$dataref->{'notes'}) or die $sth->errstr;
			$::dbh->commit();
		};

		if ($@) {
			$::dbh->rollback();
			webcalng_subs::hard_error($@);
		}
		$sth->finish();	

	}

	return 1;
}

#
# Delete an item.
#
sub delitem_io {
	my ($itemid,$exceptionid,$calendar) = (@_);
	my ($items_table,$exceptions_table,$sql1,$sth1,$sql2,$sth2);

	# If there is an exceptionid, just remove that id from the exceptions table.  If there
	# is no exceptionid, remove all instances of this id from both items and exceptions tables.
	$items_table      = $::webcalng_conf{'DB_TABLE_PREFIX'} . "items";
	$exceptions_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "exceptions";

	if ($exceptionid) {
		$sql1 = qq{ DELETE FROM $exceptions_table WHERE webcalng_calendar=? AND webcalng_exceptionid=? AND webcalng_itemid=? };
		$sth1 = $::dbh->prepare($sql1) or webcalng_subs::hard_error($DBI::errstr);
		eval {
			$sth1->execute($calendar,$exceptionid,$itemid) or die $sth1->errstr;
			$::dbh->commit();
		};
		if ($@) {
			$::dbh->rollback();
			webcalng_subs::hard_error($@);
		}
		$sth1->finish();	
	} else {
		$sql1 = qq{ DELETE FROM $items_table WHERE webcalng_calendar=? AND webcalng_itemid=? };
		$sth1 = $::dbh->prepare($sql1) or webcalng_subs::hard_error($DBI::errstr);
		$sql2 = qq{ DELETE FROM $exceptions_table WHERE webcalng_calendar=? AND webcalng_itemid=? };
		$sth2 = $::dbh->prepare($sql2) or webcalng_subs::hard_error($DBI::errstr);
		eval {
			$sth1->execute($calendar,$itemid) or die $sth1->errstr;
			$sth2->execute($calendar,$itemid) or die $sth2->errstr;
			$::dbh->commit();
		};
		if ($@) {
			$::dbh->rollback();
			webcalng_subs::hard_error($@);
		}
		$sth1->finish();	
		$sth2->finish();	
	}

	return 1;
}

#
# Create a data structure of all the items for a group of calendars during the
# given time period.
#
sub get_items {
	my ($startmark,$endmark,$searchtext,$targetitemid,$immediate,@calendars) = (@_);
	my ($calendar,$itemid,$exceptionid,%items,%exceptions,$items_table,$exceptions_table,$sql,$sth,$ref,@itemids);

	# Define table names.
	$items_table      = $::webcalng_conf{'DB_TABLE_PREFIX'} . "items";
	$exceptions_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "exceptions";

	# Clean parameters that might end up in the SQL statement.
	$startmark    =~ s/'//g if ($startmark);
	$endmark      =~ s/'//g if ($endmark);
	$searchtext   =~ s/'//g if ($searchtext);
	$targetitemid =~ s/'//g if ($targetitemid);
	$immediate    =~ s/'//g if ($immediate);
	for (@calendars) { s/'/\'/g; }
	
	# We do not use placeholders in this subroutine, as there are too many
	# variations of the select statement for me to keep track of it all.
	if ($targetitemid) {
		$sql = "SELECT * FROM $items_table WHERE webcalng_itemid='$targetitemid'";
	} else {
		$sql = "SELECT * FROM $items_table WHERE (";
		for $calendar (@calendars) {
			$sql .= "webcalng_calendar='$calendar' OR ";
		}
		$sql =~ s/\s+OR\s+$//;
		$sql .= ") AND ((webcalng_startdate > $startmark AND webcalng_startdate < $endmark) OR (webcalng_startdate <= $startmark AND webcalng_enddate >= $startmark))";
		if ($searchtext) {
			$sql .= " AND webcalng_description LIKE '%$searchtext%'";
		}
	}

	# First, get the items in the items table.
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	$sth->execute() or webcalng_subs::hard_error($DBI::errstr);
	while ($ref = $sth->fetchrow_hashref('NAME_lc')) {
		next unless ((! $immediate) || (($ref->{'webcalng_remindwhen'}) && (($ref->{'webcalng_remindwhen'} =~ /^1,/) || ($ref->{'webcalng_remindwhen'} =~ /^1$/))));
		$calendar                                   = $ref->{'webcalng_calendar'};
		$itemid                                     = $ref->{'webcalng_itemid'};
		$items{$calendar}{$itemid}{'modified'}      = $ref->{'webcalng_modified'} || "";
		$items{$calendar}{$itemid}{'user'}          = $ref->{'webcalng_user'} || "";
		$items{$calendar}{$itemid}{'description'}   = $ref->{'webcalng_description'} || "";
		$items{$calendar}{$itemid}{'dateregexp'}    = $ref->{'webcalng_dateregexp'} || "";
		$items{$calendar}{$itemid}{'startdate'}     = $ref->{'webcalng_startdate'} || "";
		$items{$calendar}{$itemid}{'enddate'}       = $ref->{'webcalng_enddate'} || "";
		$items{$calendar}{$itemid}{'starttime'}     = $ref->{'webcalng_starttime'} || "";
		$items{$calendar}{$itemid}{'endtime'}       = $ref->{'webcalng_endtime'} || "";
		$items{$calendar}{$itemid}{'visibility'}    = $ref->{'webcalng_visibility'};
		$items{$calendar}{$itemid}{'repeat'}        = $ref->{'webcalng_repeat'} || "";
		$items{$calendar}{$itemid}{'repeatdays'}    = $ref->{'webcalng_repeatdays'} || "";
		$items{$calendar}{$itemid}{'repeatend'}     = $ref->{'webcalng_repeatend'} || "";
		$items{$calendar}{$itemid}{'remindwho'}     = $ref->{'webcalng_remindwho'};
		$items{$calendar}{$itemid}{'remindwhen'}    = $ref->{'webcalng_remindwhen'} || "";
		$items{$calendar}{$itemid}{'remindersent'}  = $ref->{'webcalng_remindersent'};
		$items{$calendar}{$itemid}{'meetingid'}     = $ref->{'webcalng_meetingid'};
		$items{$calendar}{$itemid}{'hyperlink'}     = $ref->{'webcalng_hyperlink'} || "";
		$items{$calendar}{$itemid}{'notes'}         = $ref->{'webcalng_notes'} || "";
		$items{$calendar}{$itemid}{'exceptiondate'} = "";
		if ($calendar eq $::calendar) {
			$items{$calendar}{$itemid}{'origcalendar'} = 0;
		} else {
			$items{$calendar}{$itemid}{'origcalendar'} = $calendar;
		}
		push(@itemids,$itemid);
	}
	$sth->finish();

	# Now, get the exceptions for all the items that matched the search above.
	if ($#itemids >= 0) {
		$sql = "SELECT * FROM $exceptions_table WHERE ( "; 
		for $itemid (@itemids) {
			$sql .= "webcalng_itemid=$itemid OR ";
		}
		$sql  =~ s/\s+OR\s+$//;
		$sql .= ")";
		$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
		$sth->execute() or webcalng_subs::hard_error($DBI::errstr);
		while ($ref = $sth->fetchrow_hashref('NAME_lc')) {
			$calendar                                                      = $ref->{'webcalng_calendar'};
			$exceptionid                                                   = $ref->{'webcalng_exceptionid'};
			$itemid                                                        = $ref->{'webcalng_itemid'};
			$exceptions{$calendar}{$itemid}{$exceptionid}{'modified'}      = $ref->{'webcalng_modified'} || "";
			$exceptions{$calendar}{$itemid}{$exceptionid}{'user'}          = $ref->{'webcalng_user'} || "";
			$exceptions{$calendar}{$itemid}{$exceptionid}{'description'}   = $ref->{'webcalng_description'} || "";
			$exceptions{$calendar}{$itemid}{$exceptionid}{'dateregexp'}    = $items{$calendar}{$itemid}{'dateregexp'} || "";
			$exceptions{$calendar}{$itemid}{$exceptionid}{'startdate'}     = $items{$calendar}{$itemid}{'startdate'} || "";
			$exceptions{$calendar}{$itemid}{$exceptionid}{'enddate'}       = $items{$calendar}{$itemid}{'enddate'} || "";
			$exceptions{$calendar}{$itemid}{$exceptionid}{'exceptiondate'} = $ref->{'webcalng_exceptiondate'} || "";
			$exceptions{$calendar}{$itemid}{$exceptionid}{'starttime'}     = $ref->{'webcalng_starttime'} || "";
			$exceptions{$calendar}{$itemid}{$exceptionid}{'endtime'}       = $ref->{'webcalng_endtime'} || "";
			$exceptions{$calendar}{$itemid}{$exceptionid}{'visibility'}    = $ref->{'webcalng_visibility'};
			$exceptions{$calendar}{$itemid}{$exceptionid}{'repeat'}        = $items{$calendar}{$itemid}{'repeat'} || "";
			$exceptions{$calendar}{$itemid}{$exceptionid}{'repeatdays'}    = $items{$calendar}{$itemid}{'repeatdays'} || "";
			$exceptions{$calendar}{$itemid}{$exceptionid}{'repeatend'}     = $items{$calendar}{$itemid}{'repeatend'} || "";
			$exceptions{$calendar}{$itemid}{$exceptionid}{'remindwho'}     = $items{$calendar}{$itemid}{'remindwho'};
			$exceptions{$calendar}{$itemid}{$exceptionid}{'remindwhen'}    = $items{$calendar}{$itemid}{'remindwhen'} || "";
			$exceptions{$calendar}{$itemid}{$exceptionid}{'remindersent'}  = $items{$calendar}{$itemid}{'remindersent'};
			$exceptions{$calendar}{$itemid}{$exceptionid}{'meetingid'}     = $items{$calendar}{$itemid}{'meetingid'};
			$exceptions{$calendar}{$itemid}{$exceptionid}{'hyperlink'}     = $ref->{'webcalng_hyperlink'} || "";
			$exceptions{$calendar}{$itemid}{$exceptionid}{'notes'}         = $ref->{'webcalng_notes'} || "";
			if ($calendar eq $::calendar) {
				$exceptions{$calendar}{$itemid}{$exceptionid}{'origcalendar'} = 0;
			} else {
				$exceptions{$calendar}{$itemid}{$exceptionid}{'origcalendar'} = $calendar;
			}
		}
		$sth->finish();
	}

	return (\%items,\%exceptions);
}

#
# Create a data structure of a single given item.
#
sub get_single_item {
	my ($itemid,$exceptionid) = (@_);
	my (%item,$items_table,$exceptions_table,$sql,$sth,$ref);

	# Define table names.
	$items_table      = $::webcalng_conf{'DB_TABLE_PREFIX'} . "items";
	$exceptions_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "exceptions";

	# First, get the item from the items table.
	$sql = qq{ SELECT * FROM $items_table WHERE webcalng_calendar=? AND webcalng_itemid=? };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	$sth->execute($::calendar,$itemid) or webcalng_subs::hard_error($DBI::errstr);
	$ref = $sth->fetchrow_hashref('NAME_lc');
	$sth->finish();
	$item{'modified'}      = $ref->{'webcalng_modified'}      || "";
	$item{'user'}          = $ref->{'webcalng_user'}          || "";
	$item{'description'}   = $ref->{'webcalng_description'}   || "";
	$item{'dateregexp'}    = $ref->{'webcalng_dateregexp'}    || "";
	$item{'startdate'}     = $ref->{'webcalng_startdate'}     || "";
	$item{'enddate'}       = $ref->{'webcalng_enddate'}       || "";
	$item{'starttime'}     = $ref->{'webcalng_starttime'}     || "";
	$item{'endtime'}       = $ref->{'webcalng_endtime'}       || "";
	$item{'visibility'}    = $ref->{'webcalng_visibility'};
	$item{'repeat'}        = $ref->{'webcalng_repeat'}        || "0";
	$item{'repeatdays'}    = $ref->{'webcalng_repeatdays'}    || "";
	$item{'repeatend'}     = $ref->{'webcalng_repeatend'}     || "";
	$item{'remindwho'}     = $ref->{'webcalng_remindwho'};
	$item{'remindwhen'}    = $ref->{'webcalng_remindwhen'}    || "";
	$item{'remindersent'}  = $ref->{'webcalng_remindersent'};
	$item{'meetingid'}     = $ref->{'webcalng_meetingid'};
	$item{'hyperlink'}     = $ref->{'webcalng_hyperlink'}     || "";
	$item{'notes'}         = $ref->{'webcalng_notes'}         || "";
	$item{'exceptiondate'} = "";
	
	# If this is an exception item, fill in that information.
	if ($exceptionid) {
		$sql = qq{ SELECT * FROM $exceptions_table WHERE webcalng_calendar=? AND webcalng_exceptionid=? AND webcalng_itemid=? };
		$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
		$sth->execute($::calendar,$exceptionid,$itemid) or webcalng_subs::hard_error($DBI::errstr);
		$ref = $sth->fetchrow_hashref('NAME_lc');
		$sth->finish();
		$item{'modified'}      = $ref->{'webcalng_modified'}      || "";
		$item{'user'}          = $ref->{'webcalng_user'}          || "";
		$item{'description'}   = $ref->{'webcalng_description'}   || "";
		$item{'exceptiondate'} = $ref->{'webcalng_exceptiondate'} || "";
		$item{'starttime'}     = $ref->{'webcalng_starttime'}     || "";
		$item{'endtime'}       = $ref->{'webcalng_endtime'}       || "";
		$item{'visibility'}    = $ref->{'webcalng_visibility'};
		$item{'hyperlink'}     = $ref->{'webcalng_hyperlink'}     || "";
		$item{'notes'}         = $ref->{'webcalng_notes'}         || "";
	}

	return \%item;
}

#
# Get a hash of calendar names which the current calendar has access to import.
# Also, get the email address and auto_add_event information since we are already
# looking at the preferences table.
#
sub get_import_permissions {
	my ($targetcalendar) = (@_);
	my ($calendars_table,$preferences_table,$sql,$sth,$ref,$calendar,%import,%type,@export_list,$email,$auto_add_event,$currentcal);

	$calendars_table   = $::webcalng_conf{'DB_TABLE_PREFIX'} . "calendars";
	$preferences_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "preferences";
	
	# First, get a list of all calendars and what type they are.
	$sql = "SELECT webcalng_calendar,webcalng_type FROM $calendars_table";
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	$sth->execute() or webcalng_subs::hard_error($DBI::errstr);
	while ($ref = $sth->fetchrow_hashref('NAME_lc')) {
		$calendar        = $ref->{'webcalng_calendar'};
		$type{$calendar} = $ref->{'webcalng_type'};
		# Open and public calendar are automatically fair game for importing.
		$import{$calendar}{'import'}++ if ($ref->{'webcalng_type'} eq "open" || $ref->{'webcalng_type'} eq "public");
	}
	$sth->finish();

	# Now, read the preferences table to get export and meeting information.
	$sql = "SELECT webcalng_calendar,webcalng_key,webcalng_value FROM $preferences_table WHERE webcalng_key='export_calendars' OR webcalng_key='email' OR webcalng_key='auto_add_event' ORDER BY webcalng_calendar";
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	$sth->execute() or webcalng_subs::hard_error($DBI::errstr);
	$currentcal = "";
	while ($ref = $sth->fetchrow_hashref('NAME_lc')) {
		$calendar = $ref->{'webcalng_calendar'};
		next if ($calendar eq "admin");
		if ($currentcal ne $calendar) {
			@export_list    = ();
			$email          = 0;
			$auto_add_event = 0;
			$currentcal     = $calendar;
		} 
		if (($ref->{'webcalng_key'} eq "export_calendars") && ($ref->{'webcalng_value'})) {
			@export_list = split /,/, $ref->{'webcalng_value'};
		}
		$email          = $ref->{'webcalng_value'} if (($ref->{'webcalng_key'} eq "email") && ($ref->{'webcalng_value'}));
		$auto_add_event = 1 if (($ref->{'webcalng_key'} eq "auto_add_event") && ($ref->{'webcalng_value'}));
		if (($type{$calendar} eq "open") || ($type{$calendar} eq "public")) {
			$import{$calendar}{'auto'}  = $auto_add_event;
			$import{$calendar}{'email'} = $email;
		} else {
			for (@export_list) {
				if ($_ eq $targetcalendar) {
					$import{$calendar}{'import'}++;
					$import{$calendar}{'auto'}  = $auto_add_event;
					$import{$calendar}{'email'} = $email;
					last;
				}
			}
		}
	}
	$sth->finish();

	return \%import;
}

#
# Get information about all attendees of a meeting.
#
sub get_meeting_data {
	my ($meetingid,$calendar) = (@_);
	my ($meeting_table,%data,$sql,$sth,$ref);
	
	$meeting_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "meetings";
	$sql = qq{ SELECT * FROM $meeting_table WHERE webcalng_calendar=? AND webcalng_meetingid=? };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	$sth->execute($calendar,$meetingid) or webcalng_subs::hard_error($DBI::errstr);
	while ($ref = $sth->fetchrow_hashref('NAME_lc')) {
		my $id                 = $ref->{'webcalng_id'};
		$data{$id}{'password'} = $ref->{'webcalng_password'} || "";
		$data{$id}{'response'} = $ref->{'webcalng_response'} || 0;
		$data{$id}{'note'}     = $ref->{'webcalng_note'}     || "";
	}
	$sth->finish();

	return \%data;
}

#
# This subroutine just determines what itemid a given meetingid is
# associated with.
#
sub get_itemid_for_meetingid {
	my ($meetingid,$calendar) = (@_);
	my ($itemid,$items_table,$sql,$sth,$ref);

	$items_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "items";
	$sql = qq{ SELECT webcalng_itemid FROM $items_table WHERE webcalng_calendar=? AND webcalng_meetingid=? };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	$sth->execute($calendar,$meetingid) or webcalng_subs::hard_error($DBI::errstr);
	$ref = $sth->fetchrow_hashref('NAME_lc');
	$sth->finish();
	if ($ref->{'webcalng_itemid'}) {
		$itemid = $ref->{'webcalng_itemid'};
	} else {
		$itemid = 0;
	}

	return $itemid;
}

#
# Subroutine to create a meeting in the meetings table.
#
sub create_meeting_io {
	my ($dataref) = (@_);
	my ($meeting_table,$meetingid,$id,$sql,$sth);

	$meeting_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "meetings";
	$sql = qq{ INSERT INTO $meeting_table VALUES (?,?,?,?,?,?) };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	eval {
		for $meetingid (keys %$dataref) {
			for $id (keys %{ $dataref->{$meetingid} }) {
				$sth->execute($::calendar,$meetingid,$id,$dataref->{$meetingid}{$id}{'password'},$dataref->{$meetingid}{$id}{'response'},$dataref->{$meetingid}{$id}{'note'}) or die $sth->errstr;
			}
		}
		$::dbh->commit();
	};
	if ($@) {
		$::dbh->rollback();
		webcalng_subs::hard_error($@);
	}
	$sth->finish();	

	return 1;
}

#
# Remove a item from a calendar that was added due to a meeting, since the meeting is now cancelled.
#
sub remove_meetingitem {
	my ($calendar,$meetingid) = (@_);
	my ($items_table,$exceptions_table,$sql1,$sth1,$sql2,$sth2,$itemid);

	$itemid           = get_itemid_for_meetingid($meetingid,$calendar);
	$items_table      = $::webcalng_conf{'DB_TABLE_PREFIX'} . "items";
	$exceptions_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "exceptions";
	$sql1 = qq{ DELETE FROM $items_table WHERE webcalng_calendar=? AND webcalng_meetingid=? };
	$sth1 = $::dbh->prepare($sql1) or webcalng_subs::hard_error($DBI::errstr);
	$sql2 = qq{ DELETE FROM $exceptions_table WHERE webcalng_calendar=? AND webcalng_itemid=? };
	$sth2 = $::dbh->prepare($sql2) or webcalng_subs::hard_error($DBI::errstr);
	eval {
		$sth1->execute($calendar,$meetingid) or die $sth1->errstr;
		$sth2->execute($calendar,$itemid) or die $sth2->errstr;
		$::dbh->commit();
	};
	if ($@) {
		$::dbh->rollback();
		webcalng_subs::hard_error($@);
	}
	$sth1->finish();	
	$sth2->finish();	

	return 1;
}

#
# Remove all entries from the meetings table for a given meetingid.
#
sub remove_meeting_io {
	my ($meetingid,$targetid) = (@_);
	my ($meeting_table,$sql,$sth);

	$meeting_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "meetings";
	if ($targetid) {
		$sql = qq{ DELETE FROM $meeting_table WHERE webcalng_calendar=? AND webcalng_meetingid=? AND webcalng_id=? };
		$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
		eval {
			$sth->execute($::calendar,$meetingid,$targetid) or die $sth->errstr;
			$::dbh->commit();
		};
		if ($@) {
			$::dbh->rollback();
			webcalng_subs::hard_error($@);
		}
	} else {
		$sql = qq{ DELETE FROM $meeting_table WHERE webcalng_calendar=? AND webcalng_meetingid=? };
		$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
		eval {
			$sth->execute($::calendar,$meetingid) or die $sth->errstr;
			$::dbh->commit();
		};
		if ($@) {
			$::dbh->rollback();
			webcalng_subs::hard_error($@);
		}
	}
	$sth->finish();	

	return 1;
}

#
# This subroutine marks off a reminder as being sent.  This is only called from
# the webcalng_remind.pl script.
#
sub mark_remindersent {
	my ($calendar,$itemid,$x,$today) = (@_);
	my ($items_table,$remindersent,@remindersent,$sql,$sth,$ref);

	# Get the current remindersent list.
	$items_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "items";
	$sql = qq{ SELECT webcalng_remindersent FROM $items_table WHERE webcalng_calendar=? AND webcalng_itemid=? };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	$sth->execute($calendar,$itemid) or webcalng_subs::hard_error($DBI::errstr);
	$ref = $sth->fetchrow_hashref('NAME_lc');
	$sth->finish();	

	# Create a new remindersent list, with the given reminder spot updated.
	$remindersent     = $ref->{'webcalng_remindersent'};
	@remindersent     = split /,/, $remindersent;
	$remindersent[$x] = $today;
	$remindersent     = join ',', @remindersent;	

	# Update the table with the changes.
	$sql = qq{ UPDATE $items_table SET webcalng_remindersent=? WHERE webcalng_calendar=? AND webcalng_itemid=? };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	eval {
		$sth->execute($remindersent,$calendar,$itemid) or die $sth->errstr;
		$::dbh->commit();
	};
	if ($@) {
		$::dbh->rollback();
		webcalng_subs::hard_error($@);
	}
	$sth->finish();	

	return 1;
}

#
# This clears the remindersent flag for a given reminder, or all of them.
#
sub clear_remindersent {
	my ($calendar,$itemid,$x) = (@_);
	my ($items_table,$sql,$sth,$ref,$remindersent,@remindersent);

	# Get the current remindersent list.
	$items_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "items";
	$sql = qq{ SELECT webcalng_remindersent FROM $items_table WHERE webcalng_calendar=? AND webcalng_itemid=? };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	$sth->execute($calendar,$itemid) or webcalng_subs::hard_error($DBI::errstr);
	$ref = $sth->fetchrow_hashref('NAME_lc');
	$sth->finish();	

	# Create a new remindersent list, with the given reminder spot cleared.
	$remindersent     = $ref->{'webcalng_remindersent'};
	@remindersent     = split /,/, $remindersent;
	$remindersent[$x] = 0;
	$remindersent     = join ',', @remindersent;	

	# Update the table with the changes.
	$sql = qq{ UPDATE $items_table SET webcalng_remindersent=? WHERE webcalng_calendar=? AND webcalng_itemid=? };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	eval {
		$sth->execute($remindersent,$calendar,$itemid) or die $sth->errstr;
		$::dbh->commit();
	};
	if ($@) {
		$::dbh->rollback();
		webcalng_subs::hard_error($@);
	}
	$sth->finish();	

	return 1;
}

#
# This subroutine schedules a one time message to be sent when the reminder script runs next.
#
sub schedule_message {
	my ($email,$message) = (@_);
	my ($message_table,$sql,$sth);

	$message_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "messagequeue";
	$sql = qq{ INSERT INTO $message_table (webcalng_email,webcalng_message) VALUES (?,?) };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	eval {
		$sth->execute($email,$message) or die $sth->errstr;
		$::dbh->commit();
	};
	if ($@) {
		$::dbh->rollback();
		webcalng_subs::hard_error($@);
	}
	$sth->finish();	

	return 1;
}

#
# This subroutine reads entries from the message queue, and them removes them as they are on their way out.
#
sub read_messagequeue {
	my ($message_table,$sql,$sth,$ref,%messages,$id,$key,$counter,@indexes);

	$message_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "messagequeue";

	# First, read in the current messagequeue.
	$sql = qq{ SELECT * FROM $message_table };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	$sth->execute() or webcalng_subs::hard_error($DBI::errstr);
	$counter = 0;
	while ($ref = $sth->fetchrow_hashref('NAME_lc')) {
		$id  = $ref->{'webcalng_email'};
		$key = $counter . ";" . $id;
		$messages{$key} = $ref->{'webcalng_message'};
		$counter++;
		push(@indexes,$ref->{'webcalng_index'});
	}
	$sth->finish();
	
	return (\%messages,@indexes);
}

#
# Remove a set of indexes from the messagequeue.
#
sub empty_messagequeue {
	my (@indexes) = (@_);
	my ($message_table,$index,$sql,$sth);

	$message_table = $::webcalng_conf{'DB_TABLE_PREFIX'} . "messagequeue";
	$sql = qq{ DELETE FROM $message_table WHERE webcalng_index=? };
	$sth = $::dbh->prepare($sql) or webcalng_subs::hard_error($DBI::errstr);
	for $index (@indexes) {
		eval {
			$sth->execute($index) or die $sth->errstr;
		};
		if ($@) {
			$::dbh->rollback();
			webcalng_subs::hard_error($@);
		}
	}
	$::dbh->commit();
	$sth->finish();	

	return 1;
}

1;
