// ==============================================================
//
//  Copyright (c) 2003 by Alex Vinokur.
//
//  For conditions of distribution and use, see
//  copyright notice in version.h
//
// ==============================================================


// ##############################################################
//
//  SOFTWARE : Turing Machine with faults, failures and recovery
//             C++ Simulator
//
//  FILE     : rules.cpp
//
//  DESCRIPTION :
//         Classes CurSituation & NextSituation (Implementation)
//
// ##############################################################


// ==============
#include "rules.h"
// ==============

// =========
vector<state_t> CurSituation::allowed_states_s;
vector<state_t> CurSituation::allowed_neutral_cur_states_s;
vector<symbol_mt> CurSituation::allowed_symbols_s;

vector<state_t> NextSituation::allowed_states_s;
vector<symbol_mt> NextSituation::allowed_symbols_s;
vector<symbol_mt> NextSituation::write_nothing_pseudo_symbols_s;


// =========
// =========
// Constructor-0
CurSituation::CurSituation ()
{
} 


// =========
// Constructor-1
CurSituation::CurSituation (
		const state_t&		cur_state_i, 
		const state_t&		cur_substate1_i, 
		const state_t&		cur_substate2_i, 
		const vector<symbol_mt>&	cur_symbols_i,
		const vector<SymbolTypes>&	cur_symbol_mtypes_i,
		bool			controlling_flag_i
		)
	:
	cur_state_ (cur_state_i),
	cur_substate1_ (cur_substate1_i),
	cur_substate2_ (cur_substate2_i),
	cur_symbols_ (cur_symbols_i),
	cur_symbol_mtypes_ (cur_symbol_mtypes_i),
	controlling_flag_ (controlling_flag_i)	
{
  if (!cur_symbol_mtypes_i.empty())
  {
    assert (cur_symbols_.size() == cur_symbol_mtypes_.size());
    assert (static_cast<size_t>(count (cur_symbol_mtypes_.begin(), cur_symbol_mtypes_.end(), UNDEF_SymbolTypes)) == 0);
  }

  if (!allowed_states_s.empty())
  {
    assert (belongs_to (cur_state_, allowed_states_s));
    assert (belongs_to (cur_substate1_, allowed_states_s));
    assert (belongs_to (cur_substate2_, allowed_states_s));
    // assert (find (allowed_states_s.begin(), allowed_states_s.end(), cur_state_) != allowed_states_s.end());
  }

  if (!allowed_symbols_s.empty())
  {
    for (size_t i = 0; i < cur_symbols_.size(); i++)
    {
      assert (belongs_to (cur_symbols_[i], allowed_symbols_s));
      // assert (find (allowed_symbols_s.begin(), allowed_symbols_s.end(), cur_symbols_[i]) != allowed_symbols_s.end());
    }
  }

} 

// =========
// Destructor
CurSituation::~CurSituation ()
{
}


// =========
void CurSituation::set_shared_data1_s (
	const vector<symbol_mt>&	allowed_symbols_i,
	const vector<state_t>&	allowed_states_i
	)
{
  assert (allowed_symbols_s.empty());
  assert (allowed_states_s.empty());

  allowed_symbols_s = allowed_symbols_i;
  allowed_states_s = allowed_states_i;

  assert (!allowed_symbols_s.empty());
  assert (allowed_states_s.empty());

  allowed_neutral_cur_states_s.clear();
  assert (allowed_neutral_cur_states_s.empty());


}


// =========
void CurSituation::set_shared_data2_s (
	const vector<state_t>&	allowed_states_i,
	const vector<state_t>&	allowed_neutral_cur_states_i
	)
{
  assert (allowed_states_s.empty());
  assert (allowed_neutral_cur_states_s.empty());

  allowed_states_s = allowed_states_i;
  allowed_neutral_cur_states_s = allowed_neutral_cur_states_i;

  assert (!allowed_states_s.empty());
  assert (!allowed_neutral_cur_states_s.empty());

}


// =========
void CurSituation::clear_shared_data_s ()
{
  allowed_symbols_s.clear();
  allowed_states_s.clear();

  assert (allowed_symbols_s.empty());
  assert (allowed_states_s.empty());
}

// =========
state_t CurSituation::get_state () const
{
  return cur_state_;
} 


// =========
state_t CurSituation::get_substate1 () const
{
  return cur_substate1_;
} 


// =========
state_t CurSituation::get_substate2 () const
{
  return cur_substate2_;
} 


// =========
void CurSituation::set_substate1 (const state_t& state_i)
{
  assert (!(state_i == state_t()));
  cur_substate1_ = state_i;
} 


// =========
void CurSituation::set_substate2 (const state_t& state_i)
{
  assert (!(state_i == state_t()));
  cur_substate2_ = state_i;
} 



// =========
bool CurSituation::get_controlling_flag () const
{
  return controlling_flag_;
} 



// =========
symbol_mt CurSituation::get_symbol (size_t index_i) const
{
  assert (index_i < cur_symbols_.size());
  return cur_symbols_[index_i];
}


// =========
size_t	CurSituation::get_total_symbols () const
{
  return cur_symbols_.size();
}

// =========
string	CurSituation::getstr_symbols (size_t max_symbol_size_i) const
{
ostringstream oss;
  for (size_t i = 0; i < cur_symbols_.size(); i++)
  {
    oss << setw(max_symbol_size_i) << left << to_string (cur_symbols_[i]).c_str() << " ";
  }
  return oss.str();
}


// =========
string	CurSituation::getstr_show (const string& msg_i) const
{
ostringstream oss;
  if (!msg_i.empty())
  {
    oss << msg_i << " : ";
  }
const size_t max_size = 2;
  oss << "("
      << setw (max_size)
      << left
      << get_state().c_str()
      << " "
      << setw (max_size)
      << left
      << get_substate1().c_str()
      << " "
      << setw (max_size)
      << left
      << get_substate2().c_str()
      << ")  ["
      << getstr_symbols ()
      << "]";
  return oss.str();
}

// =========
vector<symbol_mt> CurSituation::get_symbols () const
{
  return cur_symbols_;
}


// =========
bool operator< (const CurSituation& inst1_i, const CurSituation& inst2_i)
{
  return pre_operator (inst1_i, inst2_i, true, false, false);
}



// =========
bool operator== (const CurSituation& inst1_i, const CurSituation& inst2_i)
{
  return pre_operator (inst1_i, inst2_i, false, false, true);
}



// =========
// =========
void SymbolAndShift::set_symbol (const symbol_mt& symbol_i)
{
  symbol_ = symbol_i;
}

// =========
void SymbolAndShift::set_no_shift ()
{
  set_shift (NO_SHIFT);
}

// =========
void SymbolAndShift::set_shift (const shift_t& shift_i)
{
  assert ((shift_i == LEFT_SHIFT) || (shift_i == RIGHT_SHIFT) || (shift_i == NO_SHIFT));
  shift_ = shift_i;
}



// =========
// =========
// Constructor-0
NextSituation::NextSituation ()
{
} 

// =========
// Constructor-1
NextSituation::NextSituation (
		const state_t& next_state_i, 
		const state_t& next_substate1_i, 
		const state_t& next_substate2_i, 
		const vector<SymbolAndShift>& next_symbols_and_shifts_i
		)
	:
	next_state_ (next_state_i),
	next_substate1_ (next_substate1_i),
	next_substate2_ (next_substate2_i),
	next_symbols_and_shifts_ (next_symbols_and_shifts_i),
	next_symbol_mtypes_ (next_symbols_and_shifts_i.size(), UNDEF_SymbolTypes)
{
  assert (next_symbols_and_shifts_.size() == next_symbol_mtypes_.size());
  assert (static_cast<size_t>(count (next_symbol_mtypes_.begin(), next_symbol_mtypes_.end(), UNDEF_SymbolTypes)) == next_symbol_mtypes_.size());

  if (!allowed_states_s.empty())
  {
    assert (belongs_to (next_state_, allowed_states_s));
    assert (belongs_to (next_substate1_, allowed_states_s));
    assert (belongs_to (next_substate2_, allowed_states_s));
    // assert (find (allowed_states_s.begin(), allowed_states_s.end(), next_state_) != allowed_states_s.end());
  }

  if (!allowed_symbols_s.empty())
  {
    for (size_t i = 0; i < next_symbols_and_shifts_.size(); i++)
    {
      assert (belongs_to (next_symbols_and_shifts_[i].get_symbol(), allowed_symbols_s));
      // assert (find (allowed_symbols_s.begin(), allowed_symbols_s.end(), next_symbols_and_shifts_[i].get_symbol()) != allowed_symbols_s.end());
    }
  }


} 


// =========
// Destructor
NextSituation::~NextSituation ()
{
} 

// =========
state_t NextSituation::get_state () const
{
  return next_state_;
} 


// =========
state_t NextSituation::get_substate1 () const
{
  return next_substate1_;
} 


// =========
state_t NextSituation::get_substate2 () const
{
  return next_substate2_;
} 



// =========
void NextSituation::set_substate1 (const state_t& state_i)
{
  assert (!(state_i == state_t()));
  next_substate1_ = state_i;
} 


// =========
void NextSituation::set_substate2 (const state_t& state_i)
{
  assert (!(state_i == state_t()));
  next_substate2_ = state_i;
} 


// =========
symbol_mt	NextSituation::get_symbol (size_t index_i) const
{
  assert (index_i < next_symbols_and_shifts_.size());
  return next_symbols_and_shifts_[index_i].symbol_;
}



// =========
void NextSituation::set_symbol_mtype (size_t index_i, SymbolTypes type_i)
{
  assert (type_i != UNDEF_SymbolTypes);
  assert (type_i < SIZE_OF_SymbolTypes);
  assert (index_i < next_symbol_mtypes_.size());

  next_symbol_mtypes_[type_i];

}


// =========
size_t	NextSituation::get_total_symbols () const
{
  return next_symbols_and_shifts_.size();
}

// =========
shift_t	NextSituation::get_shift (size_t index_i) const
{
  assert (index_i < next_symbols_and_shifts_.size());
  return next_symbols_and_shifts_[index_i].shift_;
}

// =========
string	NextSituation::getstr_symbols_and_shifts(size_t max_symbol_size_i) const
{
ostringstream oss;
  for (size_t i = 0; i < next_symbols_and_shifts_.size(); i++)
  {
    oss << "(" << setw (max_symbol_size_i) << left << to_string (next_symbols_and_shifts_[i].symbol_).c_str() << ", " << next_symbols_and_shifts_[i].shift_ << ") ";
  }
  return oss.str();
}


// =========
vector<SymbolAndShift>	NextSituation::get_symbols_and_shifts() const
{
  return next_symbols_and_shifts_; 
}


// =========
string	NextSituation::getstr_show (const string& msg_i) const
{
ostringstream oss;
  if (!msg_i.empty())
  {
    oss << msg_i << " : ";
  }
const size_t max_size = 2;
  oss << "("
      << setw (max_size)
      << left
      << get_state().c_str()
      << " "
      << setw (max_size)
      << left
      << get_substate1().c_str()
      << " "
      << setw (max_size)
      << left
      << get_substate2().c_str()
      << ")  ["
      << getstr_symbols_and_shifts ()
      << "]";
  return oss.str();
}



// =========
void NextSituation::set_shared_data1_s (
	const vector<symbol_mt>&	write_nothing_pseudo_symbols_i,
	const vector<symbol_mt>&	allowed_symbols_i,
	const vector<state_t>&	allowed_states_i
	)
{
  assert (write_nothing_pseudo_symbols_s.empty());
  assert (allowed_symbols_s.empty());
  assert (allowed_states_s.empty());

  write_nothing_pseudo_symbols_s = write_nothing_pseudo_symbols_i;
  allowed_symbols_s = allowed_symbols_i;
  allowed_states_s = allowed_states_i;

  assert (!write_nothing_pseudo_symbols_s.empty());
  assert (!allowed_symbols_s.empty());

}


// =========
void NextSituation::set_shared_data2_s (
	const vector<state_t>&	allowed_states_i
	)
{
  assert (allowed_states_s.empty());

  allowed_states_s = allowed_states_i;

  assert (!allowed_states_s.empty());

}


// =========
void NextSituation::clear_shared_data_s ()
{
  write_nothing_pseudo_symbols_s.clear();
  allowed_symbols_s.clear();
  allowed_states_s.clear();

  assert (write_nothing_pseudo_symbols_s.empty());
  assert (allowed_symbols_s.empty());
  assert (allowed_states_s.empty());
}




// =========
// =========
bool pre_operator (
		const CurSituation& inst1_i, 
		const CurSituation& inst2_i,
		bool	ret_if_less_i,
		bool	ret_if_greater_i,
		bool	ret_if_end_i
		)
{
  // ---------------------
  if (inst1_i.cur_state_ < inst2_i.cur_state_) 
  {
    return ret_if_less_i;
  }
  if (inst1_i.cur_state_ > inst2_i.cur_state_) 
  {
    return ret_if_greater_i;
  }
 
  // -------------------------------------

  assert (inst1_i.cur_symbols_.size() == inst2_i.cur_symbols_.size()); 

const size_t the_size = inst1_i.cur_symbols_.size();
  for (size_t i = 0; i < the_size; i++)
  {
    assert (
	     (inst1_i.cur_symbol_mtypes_.empty() && inst2_i.cur_symbol_mtypes_.empty())
	     ||
	     (!inst1_i.cur_symbol_mtypes_.empty() && !inst2_i.cur_symbol_mtypes_.empty())
	   );

    // ------------------------------------------------
    if (!inst1_i.cur_symbol_mtypes_.empty())
    {
      assert (inst1_i.cur_symbol_mtypes_[i] != UNDEF_SymbolTypes);
    }

    if (!inst2_i.cur_symbol_mtypes_.empty())
    {
      assert (inst2_i.cur_symbol_mtypes_[i] != UNDEF_SymbolTypes);
    }

    // ---------------------
    if (inst1_i.cur_symbols_[i] == inst2_i.cur_symbols_[i]) 
    {
      continue;
    }


    const bool symbol_mtypes_empty_flag = inst1_i.cur_symbol_mtypes_.empty();
    if (!symbol_mtypes_empty_flag)
    {

      // ------
      if (inst1_i.cur_symbol_mtypes_[i] == NEVER_MIND_PseudoSymbol) 
      {
        continue;
      }
      if (inst2_i.cur_symbol_mtypes_[i] == NEVER_MIND_PseudoSymbol) 
      {
        continue;
      }

      // ------
      if (
           (inst1_i.cur_symbol_mtypes_[i] == ANY_NOT_MARKER_PseudoSymbol)
           &&
           (
             (inst2_i.cur_symbol_mtypes_[i] == BLANK_Symbol) 
             || 
             (inst2_i.cur_symbol_mtypes_[i] == INPUT_Symbol) 
             || 
             (inst2_i.cur_symbol_mtypes_[i] == INTERNAL_NOT_MARKER_Symbol)
           )
         )
      {
        continue;
      }

      if (
           (inst2_i.cur_symbol_mtypes_[i] == ANY_NOT_MARKER_PseudoSymbol)
           &&
           (
             (inst1_i.cur_symbol_mtypes_[i] == BLANK_Symbol) 
             || 
             (inst1_i.cur_symbol_mtypes_[i] == INPUT_Symbol) 
             || 
             (inst1_i.cur_symbol_mtypes_[i] == INTERNAL_NOT_MARKER_Symbol)
           )
         )
      {
        continue;
      }

      // ------
      if (
           (inst1_i.cur_symbol_mtypes_[i] == ANY_NOT_MARKER_AND_NOT_BLANK_PseudoSymbol)
           &&
           (
             (inst2_i.cur_symbol_mtypes_[i] == INPUT_Symbol) 
             || 
             (inst2_i.cur_symbol_mtypes_[i] == INTERNAL_NOT_MARKER_Symbol)
           )
         )
      {
        continue;
      }

      if (
           (inst2_i.cur_symbol_mtypes_[i] == ANY_NOT_MARKER_AND_NOT_BLANK_PseudoSymbol)
           &&
           (
             (inst1_i.cur_symbol_mtypes_[i] == INPUT_Symbol) 
             || 
             (inst1_i.cur_symbol_mtypes_[i] == INTERNAL_NOT_MARKER_Symbol)
           )
         )
      {
        continue;
      }

    } // if (!symbol_mtypes_empty_flag)

    // ---------------------
    // ---------------------
    if (inst1_i.cur_symbols_[i] < inst2_i.cur_symbols_[i])
    {
      return ret_if_less_i;
    }
    if (inst1_i.cur_symbols_[i] > inst2_i.cur_symbols_[i])
    {
      return ret_if_greater_i;
    }

    // ------ asserts ------
    if (inst1_i.cur_substate1_ == state_t()) assert (0);
    if (inst1_i.cur_substate2_ == state_t()) assert (0);

    if (inst2_i.cur_substate1_ == state_t()) assert (0);
    if (inst2_i.cur_substate2_ == state_t()) assert (0);


    if (belongs_to (inst1_i.cur_substate1_, CurSituation::allowed_neutral_cur_states_s)) assert (0);
    if (belongs_to (inst1_i.cur_substate2_, CurSituation::allowed_neutral_cur_states_s)) assert (0);

    if (belongs_to (inst2_i.cur_substate1_, CurSituation::allowed_neutral_cur_states_s)) assert (0);
    if (belongs_to (inst2_i.cur_substate2_, CurSituation::allowed_neutral_cur_states_s)) assert (0);


    assert (!(inst1_i.cur_substate1_ == state_t()));
    assert (!(inst1_i.cur_substate2_ == state_t()));

    assert (!(inst2_i.cur_substate1_ == state_t()));
    assert (!(inst2_i.cur_substate2_ == state_t()));


    assert (!belongs_to (inst1_i.cur_substate1_, CurSituation::allowed_neutral_cur_states_s));
    assert (!belongs_to (inst1_i.cur_substate2_, CurSituation::allowed_neutral_cur_states_s));

    assert (!belongs_to (inst2_i.cur_substate1_, CurSituation::allowed_neutral_cur_states_s));
    assert (!belongs_to (inst2_i.cur_substate2_, CurSituation::allowed_neutral_cur_states_s));

    // ---------------------
  } // for

  // -------------------------------------
  // --- cur_substate1_ and cur_substate2_
  // -------------------------------------
const bool flag_substate1 = (
			(inst1_i.cur_substate1_ == state_t())
			||
			(inst2_i.cur_substate1_ == state_t())
			||
			(
			  belongs_to (inst1_i.cur_substate1_, CurSituation::allowed_neutral_cur_states_s)
			  ||
			  belongs_to (inst2_i.cur_substate1_, CurSituation::allowed_neutral_cur_states_s)
			)
			);

const bool flag_substate2 = (
			(inst1_i.cur_substate2_ == state_t())
			||
			(inst2_i.cur_substate2_ == state_t())
			||
			(
			  belongs_to (inst1_i.cur_substate2_, CurSituation::allowed_neutral_cur_states_s)
			  ||
			  belongs_to (inst2_i.cur_substate2_, CurSituation::allowed_neutral_cur_states_s)
			)
			);


  if (!flag_substate1)
  {
    if (inst1_i.cur_substate1_ < inst2_i.cur_substate1_) return ret_if_less_i;
    if (inst1_i.cur_substate1_ > inst2_i.cur_substate1_) return ret_if_greater_i;
  }
  
  if (!flag_substate2)
  {
    if (inst1_i.cur_substate2_ < inst2_i.cur_substate2_) return ret_if_less_i;
    if (inst1_i.cur_substate2_ > inst2_i.cur_substate2_) return ret_if_greater_i;
  }

  // ----------
  return ret_if_end_i;

} // pre_operator





// =========
// =========  
// Constructor
CurOutsideSituation::CurOutsideSituation (
		const state_t&	cur_daemon_state_i,
		const state_t&	cur_tester_state_i,
		const state_t&	cur_apparatus_state_i,
		OusideRuleTypes	non_deterministic_rule_no_i
		)
		:
		cur_daemon_state_ (cur_daemon_state_i),
		cur_tester_state_ (cur_tester_state_i),
		cur_apparatus_state_ (cur_apparatus_state_i),
		non_deterministic_rule_no_ (non_deterministic_rule_no_i)
{
}


// =========  
// Destructor
CurOutsideSituation::~CurOutsideSituation ()
{
}

// =========  
state_t CurOutsideSituation::get_cur_daemon_state () const
{
  return cur_daemon_state_;
}


// =========  
state_t CurOutsideSituation::get_cur_tester_state () const
{
  return cur_tester_state_;
}



// =========  
state_t CurOutsideSituation::get_cur_apparatus_state () const
{
  return cur_apparatus_state_;
}


// =========  
OusideRuleTypes CurOutsideSituation::get_non_deterministic_rule_no () const
{
  return non_deterministic_rule_no_;
}



// =========
bool operator< (const CurOutsideSituation& inst1_i, const CurOutsideSituation& inst2_i)
{
  if (inst1_i.cur_daemon_state_ < inst2_i.cur_daemon_state_) return true;
  if (inst1_i.cur_daemon_state_ > inst2_i.cur_daemon_state_) return false;

  if (inst1_i.cur_tester_state_ < inst2_i.cur_tester_state_) return true;
  if (inst1_i.cur_tester_state_ > inst2_i.cur_tester_state_) return false;

  if (inst1_i.cur_apparatus_state_ < inst2_i.cur_apparatus_state_) return true;
  if (inst1_i.cur_apparatus_state_ > inst2_i.cur_apparatus_state_) return false;

  if (inst1_i.non_deterministic_rule_no_ < inst2_i.non_deterministic_rule_no_) return true;
  if (inst1_i.non_deterministic_rule_no_ > inst2_i.non_deterministic_rule_no_) return false;

  return false;
}


// =========
// =========
// Constructor
NextOutsideSituation::NextOutsideSituation (
		const state_t&	next_daemon_state_i,
		const state_t&	next_tester_state_i,
		const state_t&	next_apparatus_state_i,
		bool		program_state_allowed_flag_i

		)
		:
		next_daemon_state_ (next_daemon_state_i),
		next_tester_state_ (next_tester_state_i),
		next_apparatus_state_ (next_apparatus_state_i),
		program_state_allowed_flag_ (program_state_allowed_flag_i)
{
}



// =========  
// Destructor
NextOutsideSituation::~NextOutsideSituation ()
{
}


// =========  
state_t NextOutsideSituation::get_next_daemon_state () const
{
  return next_daemon_state_;
}


// =========  
state_t NextOutsideSituation::get_next_tester_state () const
{
  return next_tester_state_;
}



// =========  
state_t NextOutsideSituation::get_next_apparatus_state () const
{
  return next_apparatus_state_;
}



// =========  
bool NextOutsideSituation::get_program_state_allowed_flag () const
{
  return program_state_allowed_flag_;
}


