// ==============================================================
//
//  Copyright (c) 2003 by Alex Vinokur.
//
//  For conditions of distribution and use, see
//  copyright notice in version.h
//
// ==============================================================


// ##############################################################
//
//  SOFTWARE : Turing Machine with faults, failures and recovery
//             C++ Simulator
//
//  FILE     : t-manage.cpp
//
//  DESCRIPTION :
//         Class TuringMachine : basic handling methods (Implementation)
//
// ##############################################################


// =================
#include "turing-s.h"
// =================


// =========
// =========
// =========
AuxTuringMachine	aux_turing_machine;
const vector<state_t>	TuringMachine::daemon_passive_states_s (aux_turing_machine.daemon_passive_states_);
const vector<state_t>	TuringMachine::daemon_active_states_s (aux_turing_machine.daemon_active_states_);
const vector<state_t>	TuringMachine::daemon_aggressive_states_s (aux_turing_machine.daemon_aggressive_states_);
const vector<state_t>	TuringMachine::tester_tracking_states_s (aux_turing_machine.tester_tracking_states_);
const vector<state_t>	TuringMachine::tester_stabilizing_states_s (aux_turing_machine.tester_stabilizing_states_);
const vector<state_t>	TuringMachine::apparatus_normal_states_s (aux_turing_machine.apparatus_normal_states_);
const vector<state_t>	TuringMachine::apparatus_emergency_states_s (aux_turing_machine.apparatus_emergency_states_);


const vector<symbol_mt>	TuringMachine::embedded_blank_synchronous_tape_alphabet_s (aux_turing_machine.embedded_blank_synchronous_tape_alphabet_);
const vector<symbol_mt>	TuringMachine::embedded_internal_synchronous_tape_alphabet_s (aux_turing_machine.embedded_internal_synchronous_tape_alphabet_);


const size_t	TuringMachine::max_size_of_daemon_states_s(aux_turing_machine.get_max_size_of_daemon_states());
const size_t	TuringMachine::max_size_of_tester_states_s(aux_turing_machine.get_max_size_of_tester_states());
const size_t	TuringMachine::max_size_of_apparatus_states_s(aux_turing_machine.get_max_size_of_apparatus_states());


const vector<string>	TuringMachine::descr_of_tape_kinds_s (aux_turing_machine.descr_of_tape_kinds_);
const size_t		TuringMachine::max_size_of_descr_of_tape_kinds_s (aux_turing_machine.max_size_of_descr_of_tape_kinds_);

const vector<string>	TuringMachine::descr_of_program_state_top_kinds_s (aux_turing_machine.descr_of_program_state_top_kinds_);
const size_t		TuringMachine::max_size_of_descr_of_program_state_top_kinds_s (aux_turing_machine.max_size_of_descr_of_program_state_top_kinds_);

const vector<string>	TuringMachine::descr_of_user_defined_program_state_kinds_s (aux_turing_machine.descr_of_user_defined_program_state_kinds_);
const size_t		TuringMachine::max_size_of_descr_of_user_defined_program_state_kinds_s = aux_turing_machine.max_size_of_descr_of_user_defined_program_state_kinds_;
const vector<vector<string> >	TuringMachine::descr_of_user_defined_program_states_s (aux_turing_machine.descr_of_user_defined_program_states_);
const vector<size_t>	TuringMachine::max_size_of_descr_of_user_defined_program_states_s (aux_turing_machine.max_size_of_descr_of_user_defined_program_states_);



const vector<string>	TuringMachine::descr_of_neutral_program_state_kinds_s (aux_turing_machine.descr_of_neutral_program_state_kinds_);
const size_t		TuringMachine::max_size_of_descr_of_neutral_program_state_kinds_s = aux_turing_machine.max_size_of_descr_of_neutral_program_state_kinds_;
const vector<vector<string> >	TuringMachine::descr_of_neutral_program_states_s (aux_turing_machine.descr_of_neutral_program_states_);
const vector<size_t>	TuringMachine::max_size_of_descr_of_neutral_program_states_s (aux_turing_machine.max_size_of_descr_of_neutral_program_states_);


const vector<string>	TuringMachine::descr_of_pre_initial_program_state_kinds_s (aux_turing_machine.descr_of_pre_initial_program_state_kinds_);
const size_t		TuringMachine::max_size_of_descr_of_pre_initial_program_state_kinds_s = aux_turing_machine.max_size_of_descr_of_pre_initial_program_state_kinds_;
const vector<vector<string> >	TuringMachine::descr_of_pre_initial_program_states_s (aux_turing_machine.descr_of_pre_initial_program_states_);
const vector<size_t>	TuringMachine::max_size_of_descr_of_pre_initial_program_states_s (aux_turing_machine.max_size_of_descr_of_pre_initial_program_states_);


const vector<string>	TuringMachine::descr_of_post_halting_program_state_kinds_s (aux_turing_machine.descr_of_post_halting_program_state_kinds_);
const size_t		TuringMachine::max_size_of_descr_of_post_halting_program_state_kinds_s = aux_turing_machine.max_size_of_descr_of_post_halting_program_state_kinds_;
const vector<vector<string> >	TuringMachine::descr_of_post_halting_program_states_s (aux_turing_machine.descr_of_post_halting_program_states_);
const vector<size_t>	TuringMachine::max_size_of_descr_of_post_halting_program_states_s (aux_turing_machine.max_size_of_descr_of_post_halting_program_states_);


const vector<string>	TuringMachine::descr_of_user_required_program_state_kinds_s (aux_turing_machine.descr_of_user_required_program_state_kinds_);
const size_t		TuringMachine::max_size_of_descr_of_user_required_program_state_kinds_s = aux_turing_machine.max_size_of_descr_of_user_required_program_state_kinds_;
const vector<vector<string> >	TuringMachine::descr_of_user_required_program_states_s (aux_turing_machine.descr_of_user_required_program_states_);
const vector<size_t>	TuringMachine::max_size_of_descr_of_user_required_program_states_s (aux_turing_machine.max_size_of_descr_of_user_required_program_states_);

const vector<string>	TuringMachine::descr_of_embedded_program_state_kinds_s (aux_turing_machine.descr_of_embedded_program_state_kinds_);
const size_t		TuringMachine::max_size_of_descr_of_embedded_program_state_kinds_s = aux_turing_machine.max_size_of_descr_of_embedded_program_state_kinds_;
const vector<vector<string> >	TuringMachine::descr_of_embedded_program_states_s (aux_turing_machine.descr_of_embedded_program_states_);
const vector<size_t>	TuringMachine::max_size_of_descr_of_embedded_program_states_s (aux_turing_machine.max_size_of_descr_of_embedded_program_states_);

const size_t		TuringMachine::max_size_of_descr_of_program_state_kinds_s (aux_turing_machine.max_size_of_descr_of_program_state_kinds_);

size_t			TuringMachine::counter_s = 0;


// =========
// =========
// Constructor-0
TuringMachine::TuringMachine (
	size_t	number_of_tapes_i,
	// ---
        const vector<vector<string> >& descr_i,
	const vector<state_t>& user_defined_initial_program_states_i, 
	const vector<state_t>& user_defined_halting_program_states_i, 
	const vector<state_t>& user_defined_internal_program_states_i,
	// ---
	const vector<symbol_mt>& user_defined_blank_alphabet_i, 
	const vector<symbol_mt>& user_defined_internal_alphabet_i,
	const vector<symbol_mt>& user_defined_input_alphabet_i,
	// ---
	const Transitions_t& user_defined_transitions_i,
	const Transitions_t& daemon_defined_fault_transitions_i,
	// ---
	const string& msg_i
	)
	:
	descr_ (descr_i),
     	user_defined_initial_program_states_ (user_defined_initial_program_states_i),
	user_defined_halting_program_states_ (user_defined_halting_program_states_i),
	user_defined_internal_program_states_ (user_defined_internal_program_states_i),
        embedded_program_states_ (vector<vector<state_t> > (NUMBER_OF_EMBEDDED_PROGRAM_STATE_KINDS)),
	user_defined_blank_alphabet_ (user_defined_blank_alphabet_i), 
	user_defined_internal_alphabet_ (user_defined_internal_alphabet_i),
	user_defined_input_alphabet_ (user_defined_input_alphabet_i),
	user_defined_transitions_ (user_defined_transitions_i),
	daemon_defined_fault_transitions_ (daemon_defined_fault_transitions_i),
	set_marker_flag_ (true),
	check_process_results_ (true),
	machine_serial_no_ (++counter_s),
	input_set_no_ (0),
	alert_counter_ (0),
	max_size_of_user_defined_program_states_ (0),
	max_size_of_user_defined_and_required_states_ (0),
	max_size_of_extra_program_states_ (0),
	max_size_of_program_states_ (0),
	max_size_of_user_defined_alphabet_symbol_ (0)
{
  IF_NOT_EMPTY (msg_i, 5, '=');
		 
  // --------------------------
  if (number_of_tapes_i == 0)
  {
    check_process_results_ = false;
    FATAL_MSG	("Illegal number of tapes : "
		<< number_of_tapes_i
		);
    return;
  }

  if (number_of_tapes_i > UNBELIEVABLE_NUMBER_OF_TAPES)
  {
    WARNING_MSG	("Unbelievable number of tapes : "
		<< number_of_tapes_i
		);
  }


  for (size_t i = 0; i < NUMBER_OF_TAPE_KINDS; i++)
  {
    set_tape_alphabet (
	number_of_tapes_i,
	static_cast<TapeKinds> (i), 
	user_defined_blank_alphabet_i, 
	user_defined_internal_alphabet_i, 
	user_defined_input_alphabet_i
	); 
  } 

  for (size_t i = 0; i < NUMBER_OF_TAPE_KINDS; i++)
  {
    assert (tapes_[i].size() == number_of_tapes_i);
  }
 
  // ------------------------------------
  CurSituation::clear_shared_data_s ();
  NextSituation::clear_shared_data_s ();
  Tape::clear_shared_data_s ();

  set_max_user_defined_program_state_size();
  set_max_size_of_user_defined_alphabet_symbol();

  set_pseudo_symbols_alphabet ();
  share_information1 ();
  elongate_user_defined_transitions();
  set_extra_program_states ();
  elongate_daemon_defined_fault_transitions();
  share_information2 ();

  set_max_size_of_extra_program_states_size();
  set_max_size_of_user_defined_and_required_states();
  set_max_size_of_program_states();

  // -------------------------
  check1_assert();


  // ---------------
  show1_env();

  build_transition ();

  show2_env();

  // show_all_tapes_alphabets_combinations ();

  // ----------------------
  assert (check_daemon_states_s ());
  assert (check_tester_states_s ());
  assert (check_apparatus_states_s ());

  if (check_process_results_)	check_process_results_ = check_user_defined_program_states ();
  if (check_process_results_)	check_process_results_ = check_extra_program_states ();
  if (check_process_results_)	check_process_results_ = check_daemon_states_s ();
  if (check_process_results_)	check_process_results_ = check_tester_states_s ();
  if (check_process_results_)	check_process_results_ = check_apparatus_states_s ();
  if (check_process_results_)	check_process_results_ = check_alphabets ();
  if (check_process_results_)	check_process_results_ = check_user_defined_transition ();

} 

// =========
// Destructor
TuringMachine::~TuringMachine ()
{
} 



// =========
void TuringMachine::check1_assert () const
{
  // -------------------------
  assert (max_size_of_user_defined_program_states_ > 0);
  assert (max_size_of_user_defined_and_required_states_ > 0);
  assert (max_size_of_program_states_ > 0);
  assert (max_size_of_extra_program_states_ > 0);
  assert (max_size_of_daemon_states_s > 0);
  assert (max_size_of_tester_states_s > 0);
  assert (max_size_of_apparatus_states_s > 0);
  assert (max_size_of_user_defined_alphabet_symbol_ > 0);


  // ---------
  assert (!descr_of_tape_kinds_s.empty());
  for (size_t i = 0; i < descr_of_tape_kinds_s.size(); i++)
  {
    assert (!descr_of_tape_kinds_s[i].empty());
  }
  assert (max_size_of_descr_of_tape_kinds_s > 0);


  // ---------
  assert (!descr_of_program_state_top_kinds_s.empty());
  for (size_t i = 0; i < descr_of_program_state_top_kinds_s.size(); i++)
  {
    assert (!descr_of_program_state_top_kinds_s[i].empty());
  }
  assert (max_size_of_descr_of_program_state_top_kinds_s > 0);

  // ---------
  assert (!descr_of_user_defined_program_state_kinds_s.empty());
  for (size_t i = 0; i < descr_of_user_defined_program_state_kinds_s.size(); i++)
  {
    assert (!descr_of_user_defined_program_state_kinds_s[i].empty());
  }
  assert (max_size_of_descr_of_user_defined_program_state_kinds_s > 0);

  // ---------

  assert (!descr_of_user_defined_program_states_s.empty());
  for (size_t i = 0; i < descr_of_user_defined_program_states_s.size(); i++)
  {
    assert (!descr_of_user_defined_program_states_s[i].empty());
    for (size_t j = 0; j < descr_of_user_defined_program_states_s[i].size(); j++)
    {
      assert (!descr_of_user_defined_program_states_s[i][j].empty());
    } 
  }
  for (size_t i = 0; i < max_size_of_descr_of_user_defined_program_states_s.size(); i++)
  {
    assert (max_size_of_descr_of_user_defined_program_states_s[i] > 0);
  }

  // ---------
  assert (!descr_of_user_required_program_state_kinds_s.empty());
  for (size_t i = 0; i < descr_of_user_required_program_state_kinds_s.size(); i++)
  {
    assert (!descr_of_user_required_program_state_kinds_s[i].empty());
  }
  assert (max_size_of_descr_of_user_required_program_state_kinds_s > 0);



  // ---------
  assert (!descr_of_neutral_program_states_s.empty());
  for (size_t i = 0; i < descr_of_neutral_program_states_s.size(); i++)
  {
    assert (!descr_of_neutral_program_states_s[i].empty());
    for (size_t j = 0; j < descr_of_neutral_program_states_s[i].size(); j++)
    {
      assert (!descr_of_neutral_program_states_s[i][j].empty());
    }
  }
  assert (!max_size_of_descr_of_neutral_program_states_s.empty());
  for (size_t i = 0; i < max_size_of_descr_of_neutral_program_states_s.size(); i++)
  {
    assert (max_size_of_descr_of_neutral_program_states_s[i] > 0);
  }


  // ---------
  assert (!descr_of_pre_initial_program_states_s.empty());
  for (size_t i = 0; i < descr_of_pre_initial_program_states_s.size(); i++)
  {
    assert (!descr_of_pre_initial_program_states_s[i].empty());
    for (size_t j = 0; j < descr_of_pre_initial_program_states_s[i].size(); j++)
    {
      assert (!descr_of_pre_initial_program_states_s[i][j].empty());
    }
  }
  assert (!max_size_of_descr_of_pre_initial_program_states_s.empty());
  for (size_t i = 0; i < max_size_of_descr_of_pre_initial_program_states_s.size(); i++)
  {
    assert (max_size_of_descr_of_pre_initial_program_states_s[i] > 0);
  }



  // ---------
  assert (!descr_of_post_halting_program_states_s.empty());
  for (size_t i = 0; i < descr_of_post_halting_program_states_s.size(); i++)
  {
    assert (!descr_of_post_halting_program_states_s[i].empty());
    for (size_t j = 0; j < descr_of_post_halting_program_states_s[i].size(); j++)
    {
      assert (!descr_of_post_halting_program_states_s[i][j].empty());
    }
  }
  assert (!max_size_of_descr_of_post_halting_program_states_s.empty());
  for (size_t i = 0; i < max_size_of_descr_of_post_halting_program_states_s.size(); i++)
  {
    assert (max_size_of_descr_of_post_halting_program_states_s[i] > 0);
  }


  // ---------
  assert (!descr_of_user_required_program_states_s.empty());
  for (size_t i = 0; i < descr_of_user_required_program_states_s.size(); i++)
  {
    assert (!descr_of_user_required_program_states_s[i].empty());
    for (size_t j = 0; j < descr_of_user_required_program_states_s[i].size(); j++)
    {
      assert (!descr_of_user_required_program_states_s[i][j].empty());
    }
  }
  assert (!max_size_of_descr_of_user_required_program_states_s.empty());
  for (size_t i = 0; i < max_size_of_descr_of_user_required_program_states_s.size(); i++)
  {
    assert (max_size_of_descr_of_user_required_program_states_s[i] > 0);
  }


  // ---------
  assert (!descr_of_embedded_program_state_kinds_s.empty());
  for (size_t i = 0; i < descr_of_embedded_program_state_kinds_s.size(); i++)
  {
    assert (!descr_of_embedded_program_state_kinds_s[i].empty());
  }
  assert (max_size_of_descr_of_embedded_program_state_kinds_s > 0);

  // ---------
  assert (!descr_of_embedded_program_states_s.empty());
  for (size_t i = 0; i < descr_of_embedded_program_states_s.size(); i++)
  {
    assert (!descr_of_embedded_program_states_s[i].empty());
    for (size_t j = 0; j < descr_of_embedded_program_states_s[i].size(); j++)
    {
      assert (!descr_of_embedded_program_states_s[i][j].empty());
    }
  }
  assert (!max_size_of_descr_of_embedded_program_states_s.empty());
  for (size_t i = 0; i < max_size_of_descr_of_embedded_program_states_s.size(); i++)
  {
    assert (max_size_of_descr_of_embedded_program_states_s[i] > 0);
  }

  // ---------
  assert (max_size_of_descr_of_program_state_kinds_s > 0);

}


// =========
void TuringMachine::set_tape_alphabet (
			size_t		number_of_tapes_i,
			TapeKinds	tape_kind_i,
			const vector<symbol_mt>& user_defined_blank_alphabet_i,
			const vector<symbol_mt>& user_defined_internal_alphabet_i,
			const vector<symbol_mt>& user_defined_input_alphabet_i
			)

{
typedef Tapes_t::value_type value_type;
pair<Tapes_t::iterator, bool> couple;

  assert (number_of_tapes_i > 0);
  for (size_t i = 0; i < number_of_tapes_i; i++)
  {
    switch (tape_kind_i)
    {
      case MASTER_COMPUTATION_TAPE :
      case BACKUP_COMPUTATION_TAPE :
      case TESTER_TAPE :
        couple = tapes_[tape_kind_i].insert (value_type(i, Tape (user_defined_blank_alphabet_i, user_defined_internal_alphabet_i, user_defined_input_alphabet_i)));
        assert (couple.second);
        break;

      case SYNCHRONOUS_COMPUTATION_TAPE :
      case SYNCHRONOUS_BACKUP_COMPUTATION_TAPE :
        couple = tapes_[tape_kind_i].insert (value_type(i, Tape (embedded_blank_synchronous_tape_alphabet_s, embedded_internal_synchronous_tape_alphabet_s)));
        assert (couple.second);
        break;

      default :
        assert (0);
        break;
    }

    if (!couple.second)
    {
      check_process_results_ = false;
      FATAL_MSG	("Unable to add tape");
      break;
    }
    assert (couple.second);

    if (!couple.first->second.check_results_)
    {
      check_process_results_ = false;
      FATAL_MSG	("Tape#" << i << " has problems");
      break;
    }

  }

}
	     


// =========
bool	TuringMachine::find_appropriate_rule (
			const CurSituation&	 inst_i, 
			Transitions_t::iterator& iter_o
			)
{
bool	ret_bool = false;
Transitions_t::iterator	pos_iter (built_transitions_.end());

int counter = 0;
  // ----------------
  for (pos_iter = built_transitions_.begin();
       pos_iter != built_transitions_.end();
       pos_iter++)
  {
    if (inst_i == pos_iter->first)
    {
      counter++;
      assert (ret_bool == false);
      ret_bool = true;
      iter_o = pos_iter;
    }  
  }
  return ret_bool;
}


// =========
size_t TuringMachine::get_internal_index_for_tape (TapeKinds kind_i, size_t external_index_i) const
{
  assert (kind_i < NUMBER_OF_TAPE_KINDS);

size_t ret_value = 0;
  for (size_t i = 0; i < static_cast<size_t> (kind_i); i++)
  {
    ret_value += tapes_[i].size();
  }
  ret_value += external_index_i;

  return ret_value;

}


// =========
void TuringMachine::set_input (const vector<vector<symbol_mt> >& input_words_i)
{
ExtendedState	initial_state (
			// user_defined_initial_program_states_.front(),  
			pre_initial_program_states_.front(),  
			get_neutral_cur_substate_alphabet_symbol(),
			get_neutral_cur_substate_alphabet_symbol(),
			daemon_passive_states_s.front(),
			daemon_passive_states_s.front(),
			tester_tracking_states_s.front(),
			apparatus_normal_states_s.front()
			);
 
  assert (process_extended_states_.empty());
  assert (process_tester_states_.empty());

  process_extended_states_.push_back (initial_state);
  process_tester_states_.push_back (user_defined_initial_program_states_.front());

  // -----------------------------------
  for (size_t s = 0; s < NUMBER_OF_TAPE_KINDS; s++) 
  {
    for (size_t i = 0; i < tapes_[s].size(); i++) 
    {
      // tapes_[s][i].tape_.clear();
      assert (tapes_[s][i].tape_.size() <= 1);
    }
  }

  // -----------------------------------
  for (size_t s = 0; s < NUMBER_OF_TAPE_KINDS; s++) 
  {
    assert (tapes_[s].size() == input_words_i.size());
    for (size_t i = 0; i < input_words_i.size(); i++) 
    {
      for (size_t j = 0; j < input_words_i[i].size(); j++) 
      {
        switch (s)
        {
          case MASTER_COMPUTATION_TAPE :
          case TESTER_TAPE :
            tapes_[s][i].tape_.push_back (input_words_i[i][j]);
            break;
  
          case BACKUP_COMPUTATION_TAPE :
            tapes_[s][i].tape_.push_back (user_defined_blank_alphabet_.front());
            break;
  
          case SYNCHRONOUS_COMPUTATION_TAPE :
          case SYNCHRONOUS_BACKUP_COMPUTATION_TAPE :
            tapes_[s][i].tape_.push_back (embedded_blank_synchronous_tape_alphabet_s.front());
            break;
  
          default :
            assert (0);
            break;

        } // switch (s)

      }	// for (size_t j = 0

    } // for (size_t i = 0

  } // for (size_t s = 0

}



// =========
void TuringMachine::set_marker ()
{
  assert (set_marker_flag_ == true);
  for (size_t s = 0; s < NUMBER_OF_TAPE_KINDS; s++) 
  {
    for (size_t i = 0; i < tapes_[s].size(); i++) 
    {
      assert (tapes_[s][i].tape_.empty());

      switch (s)
      {
        case MASTER_COMPUTATION_TAPE :
        case BACKUP_COMPUTATION_TAPE :
        case TESTER_TAPE :
          tapes_[s][i].tape_.push_back (get_user_defined_marker());
          break;
    
        case SYNCHRONOUS_COMPUTATION_TAPE :
        case SYNCHRONOUS_BACKUP_COMPUTATION_TAPE :
          tapes_[s][i].tape_.push_back (get_embedded_synchronous_tape_marker_s());
          break;
    
        default :
          assert (0);
          break;
  
      } // switch (s)

    } // for (size_t i = 0

  } // for (size_t s = 0
}

// =========
void TuringMachine::clear_process_data ()
{
  process_extended_states_.clear();
  process_tester_states_.clear();
  process_transitions_.clear();
  process_tester_transitions_.clear();
  input_set_no_++;
  alert_counter_ = 0;
    
  assert (!process_statistics_.empty());
  for (size_t i = 0; i < process_statistics_.size(); i++)
  {
    process_statistics_[i] = 0;
  } 

map<CurSituation, count_t>::iterator pos1_iter;
  for (pos1_iter = count_daemon_transitions_.begin(); pos1_iter != count_daemon_transitions_.end(); pos1_iter++)
  {
    pos1_iter->second = 0;
  }
  // --------------------
  
Tapes_t::iterator pos2_iter;
  for (size_t i = 0; i < NUMBER_OF_TAPE_KINDS; i++)
  {
    for (pos2_iter = tapes_[i].begin(); pos2_iter != tapes_[i].end(); pos2_iter++)
    {
      pos2_iter->second.clear_it (set_marker_flag_);
    }
  }

  if (set_marker_flag_)	set_marker ();

} // clear_process_data




// =========
bool	TuringMachine::check_input_words (const vector<vector<symbol_mt> >& input_words_i) const
{
Tapes_t::const_iterator	pos_iter;

  if (input_words_i.empty())
  {
    FATAL_MSG	("No input words");
    return false;
  }

  if (input_words_i.size() != tapes_[MASTER_COMPUTATION_TAPE].size())
  {
    FATAL_MSG	("Number of input words ("
		<< input_words_i.size()
		<< ") is NOT equal number of tapes ("
		<< tapes_[MASTER_COMPUTATION_TAPE].size()
		<< ")"
		);
    return false;  
  }

size_t i;
  for (i = 0, pos_iter = tapes_[MASTER_COMPUTATION_TAPE].begin(); i < input_words_i.size(); i++, pos_iter++)
  {
    assert (pos_iter->first == i);
    if (input_words_i[i].empty())
    {
      FATAL_MSG	("No input word for tape#"
		<< i
		);
      return false;
    }

    for (size_t j = 0; j < input_words_i[i].size(); j++)
    {
      if (!(pos_iter->second.is_input_symbol (input_words_i[i][j])))
      {
        if (!((j == 0) && (input_words_i[i].size() == 1) && (pos_iter->second.is_empty_symbol (input_words_i[i][j]))))
        { 
          FATAL_MSG	("Tape#"
			<< i
			<< ", symbol#"
			<< j
			<< " : "
			<< input_words_i[i][j]
			<< " is NOT input symbol"
			);
          return false;
        }
      }
    } // for (size_t j = 0; ...
  } // for (size_t i = 0; ...

  return true;
}


// =========
bool TuringMachine::perform_machine_step (
		state_t&	program_state_o, 
		bool&		controlling_flag_o, 
		string&		str_cur_situation_o,
		string&		str_next_situation_o,
		const string&	msg_i
		)
{
  IF_NOT_EMPTY (msg_i, 5, '=');
vector<symbol_mt>	scanned_symbols;
  for (size_t s = 0; s < NUMBER_OF_TAPE_KINDS; s++)
  {
    for (size_t i = 0; i < tapes_[s].size(); i++)
    {
      scanned_symbols.push_back(tapes_[s][i].get_position_symbol ());
    }
  }

const CurSituation cur_situation (
			process_extended_states_.rbegin()->program_state_, 
			process_extended_states_.rbegin()->program_substate1_, 
			process_extended_states_.rbegin()->program_substate2_, 
			scanned_symbols,
			get_symbol_mtypes (scanned_symbols),
			false	// Never Mind ???
			);

  str_cur_situation_o = cur_situation.getstr_show();
  str_next_situation_o.clear();

// --------- old ---------
// const Transitions_t::iterator find_iter = built_transitions_.find (cur_situation);
// if (find_iter == built_transitions_.end())	return false;
// -----------------------

Transitions_t::iterator find_iter;
const bool find_flag = find_appropriate_rule (cur_situation, find_iter);
  if (!find_flag) return false;

  assert (find_iter != built_transitions_.end());

  // =============================
  assert (built_transitions_.count(find_iter->first) == 1);
  controlling_flag_o = find_iter->first.get_controlling_flag();

size_t dist = distance (built_transitions_.begin(), find_iter);
  assert (dist < built_transitions_.size());
  process_statistics_[dist]++;

  process_transitions_.push_back (find_iter);

  // ------------------------------------
  if (
       (find_iter->second.get_state() == EMBSTATE_V1)
       &&
       (find_iter->first.get_state() != EMBSTATE_V1)
     )
  {
     alert_counter_++;
     ostringstream oss;
     oss << getstr_id_info () << ": ALERT-" << alert_counter_;
     const string text1 ("Tester data and Machine data are not identical");
     const string text2 ("Data recovery to be perform");
     const size_t setw1 = MAX_VALUE (text1.size(), text2.size());
     const size_t setw0_left = (setw1 - oss.str().size() - 2)/2;
     const size_t setw0_right = setw1 - oss.str().size() - 2 - setw0_left;
      
     cout << endl;
     cout << endl;
     cout << "\t   " << string (setw1, '=') << endl;
     cout << "\t   " 
          << string (setw0_left, '-')
          << " "
          << oss.str()
          << " "
          << string (setw0_right, '-')
          << endl;
     cout << "\t   " << text1 << endl;
     cout << "\t   " << text2 << endl;
     cout << "\t   " << string (setw1, '=') << endl;
     cout << endl;

  }

  // -----------------
state_t	daemon_state_on_tact1_to_be;
state_t	tester_state_on_tact1_to_be;
state_t	apparatus_state_on_tact1_to_be;


  // =============================
  // === Tact1
  // =============================

Transitions_t::const_iterator daemon_iter;
OusideRuleTypes daemon_behavior = get_daemon_behavior (*find_iter, daemon_iter);

map<CurOutsideSituation, NextOutsideSituation>::const_iterator	find_outside_tact1_iter = 
	outside_tact1_transitions_.find (
			CurOutsideSituation (
				process_extended_states_.rbegin()->daemon_state_on_tact2_,
				process_extended_states_.rbegin()->tester_state_,
				process_extended_states_.rbegin()->apparatus_state_,
				daemon_behavior
  				)
  				);
  assert (find_outside_tact1_iter != outside_tact1_transitions_.end());

  daemon_state_on_tact1_to_be = find_outside_tact1_iter->second.get_next_daemon_state();
  assert (
           (belongs_to (daemon_state_on_tact1_to_be, daemon_passive_states_s))
           ||
           (belongs_to (daemon_state_on_tact1_to_be, daemon_active_states_s))
           ||
           (belongs_to (daemon_state_on_tact1_to_be, daemon_aggressive_states_s))
         );

  tester_state_on_tact1_to_be = find_outside_tact1_iter->second.get_next_tester_state();
  assert (
           belongs_to (tester_state_on_tact1_to_be, tester_tracking_states_s)
           ||
           belongs_to (tester_state_on_tact1_to_be, tester_stabilizing_states_s)
         );


  apparatus_state_on_tact1_to_be = find_outside_tact1_iter->second.get_next_apparatus_state();
  assert (
           belongs_to (apparatus_state_on_tact1_to_be, apparatus_normal_states_s)
           ||
           belongs_to (apparatus_state_on_tact1_to_be, apparatus_emergency_states_s)
         );


Transitions_t::const_iterator actual_iter ((daemon_behavior == ACTIVE_NON_DETERMINISTIC_OUTSIDE_RULE_TYPE) ? daemon_iter : find_iter);

  switch (daemon_behavior)
  {
    case DETERMINISTIC_OUTSIDE_RULE_TYPE :
      assert (0);
      break;

    case PASSIVE_NON_DETERMINISTIC_OUTSIDE_RULE_TYPE :
      // Do nothing
      break;

    case ACTIVE_NON_DETERMINISTIC_OUTSIDE_RULE_TYPE :
      {
        assert (daemon_iter->first == find_iter->first);
        assert (actual_iter->first == find_iter->first);
        assert (!(daemon_iter->second.getstr_symbols_and_shifts() == find_iter->second.getstr_symbols_and_shifts()));
        assert (actual_iter->second.getstr_symbols_and_shifts() == daemon_iter->second.getstr_symbols_and_shifts());
    
        ostringstream oss, oss1, oss2, oss3;
        alert_counter_++;
        oss3 << getstr_id_info () << ": ALERT-" << alert_counter_;
        oss << "Daemon invoked to create FAULT";
        oss1 << "  User rule   : "
             << find_iter->first.getstr_show() 
             << " ---> " 
             << find_iter->second.getstr_show();
        oss2 << "  Daemon rule : "
             << actual_iter->first.getstr_show() 
             << " ---> " 
             << actual_iter->second.getstr_show();
    
        size_t sizew1 = 0;
        sizew1 = MAX_VALUE (sizew1, oss.str().size());
        sizew1 = MAX_VALUE (sizew1, oss1.str().size());
        sizew1 = MAX_VALUE (sizew1, oss2.str().size());
        assert (oss3.str().size() < sizew1);
        const size_t setw3_left = (sizew1 - oss3.str().size() - 2)/2;
        const size_t setw3_right = sizew1 - oss3.str().size() - 2 - setw3_left;
    
        cout << endl;
        cout << "\t   " << string (sizew1, '=') << endl;
        cout << "\t   " 
             << string (setw3_left, '-')
             << " "
             << oss3.str()
             << " "
             << string (setw3_right, '-')
             << endl;

        cout << "\t   " << oss.str() << endl;
        cout << "\t   " << oss1.str() << endl;
        cout << "\t   " << oss2.str() << endl;
        cout << "\t   " << string (sizew1, '=') << endl;
        cout << endl;
      }
      break;

    case AGGRESSIVE_NON_DETERMINISTIC_OUTSIDE_RULE_TYPE :
      {
        ostringstream oss, oss3;
        alert_counter_++;
        oss3 << getstr_id_info () << ": ALERT-" << alert_counter_;
        oss << "Daemon invoked to create FAILURE";
        size_t sizew1 = 0;
        sizew1 = MAX_VALUE (sizew1, oss.str().size());
        assert (oss3.str().size() < sizew1);
        const size_t setw3_left = (sizew1 - oss3.str().size() - 2)/2;
        const size_t setw3_right = sizew1 - oss3.str().size() - 2 - setw3_left;
    
        cout << endl;
        cout << "\t   " << string (sizew1, '=') << endl;
        cout << "\t   " 
             << string (setw3_left, '-')
             << " "
             << oss3.str()
             << " "
             << string (setw3_right, '-')
             << endl;

        cout << "\t   " << oss.str() << endl;
        cout << "\t   " << string (sizew1, '=') << endl;
        cout << endl;
      }
      break;
   
    default :
      assert (0);
      break;
  }


  // =============================
  // =============================
  // === Tact2
  // =============================
state_t	daemon_state_on_tact2_to_be;
state_t	tester_state_on_tact2_to_be;
state_t	apparatus_state_on_tact2_to_be;


map<CurOutsideSituation, NextOutsideSituation>::const_iterator	find_outside_tact2_iter =
	outside_tact2_transitions_.find (
			CurOutsideSituation (
				daemon_state_on_tact1_to_be,
				tester_state_on_tact1_to_be,
				apparatus_state_on_tact1_to_be
  				)
  				);
  assert (find_outside_tact2_iter != outside_tact2_transitions_.end());
  daemon_state_on_tact2_to_be = find_outside_tact2_iter->second.get_next_daemon_state();
  assert (
           (belongs_to (daemon_state_on_tact2_to_be, daemon_passive_states_s))
         );

  tester_state_on_tact2_to_be = find_outside_tact2_iter->second.get_next_tester_state();
  assert (
           belongs_to (tester_state_on_tact2_to_be, tester_tracking_states_s)
           ||
           belongs_to (tester_state_on_tact2_to_be, tester_stabilizing_states_s)
         );


  apparatus_state_on_tact2_to_be = find_outside_tact2_iter->second.get_next_apparatus_state();
  assert (
           belongs_to (apparatus_state_on_tact2_to_be, apparatus_normal_states_s)
           ||
           belongs_to (apparatus_state_on_tact2_to_be, apparatus_emergency_states_s)
         );



  assert (!process_extended_states_.rbegin()->program_substate1_.empty());


state_t state_to_be;
state_t substate1_to_be;
state_t substate2_to_be;

  translate_logical_rule_to_actual_program_triple_state_s (
		actual_iter->second,
		process_extended_states_.rbegin()->program_substate1_,
		process_extended_states_.rbegin()->program_substate2_,
		state_to_be,
		substate1_to_be,
		substate2_to_be
		);

   assert (state_to_be != state_t());
   assert (substate1_to_be != state_t());
   assert (substate2_to_be != state_t());

  if (daemon_behavior == AGGRESSIVE_NON_DETERMINISTIC_OUTSIDE_RULE_TYPE)
  {
    state_to_be	= EMBSTATE_E1;
    substate1_to_be = get_neutral_cur_substate_alphabet_symbol();
    substate2_to_be = get_neutral_cur_substate_alphabet_symbol();
  }

ExtendedState	the_extended_state (
			state_to_be,
			substate1_to_be,
			substate2_to_be,
			daemon_state_on_tact1_to_be,
			daemon_state_on_tact2_to_be,
			tester_state_on_tact2_to_be,
			apparatus_state_on_tact2_to_be
			);

  // -------------------------------------------
  process_extended_states_.push_back (the_extended_state);
  // -------------------------------------------

  if (!(daemon_behavior == AGGRESSIVE_NON_DETERMINISTIC_OUTSIDE_RULE_TYPE))
  {
    for (size_t s = 0; s < NUMBER_OF_TAPE_KINDS; s++)
    {
      for (size_t i = 0; i < tapes_[s].size(); i++)
      {
        tapes_[s][i].set_position_symbol (actual_iter->second.get_symbol (get_internal_index_for_tape (static_cast<TapeKinds>(s), i)));
        tapes_[s][i].shift_position (actual_iter->second.get_shift(get_internal_index_for_tape (static_cast<TapeKinds>(s), i)));
      }
    }
  }
  else
  {
    for (size_t s = 0; s < NUMBER_OF_TAPE_KINDS; s++)
    {
      if (s == TESTER_TAPE) continue;
      for (size_t i = 0; i < tapes_[s].size(); i++)
      {
        tapes_[s][i].position_ = 0;
        tapes_[s][i].tape_.erase(tapes_[s][i].tape_.begin() + 1, tapes_[s][i].tape_.end()); // yyy
      }
    }
  }
 

  program_state_o = process_extended_states_.rbegin()->program_state_;

  // -------------------------------------
vector<symbol_mt>	scanned2_symbols;
  for (size_t s = 0; s < NUMBER_OF_TAPE_KINDS; s++)
  {
    for (size_t i = 0; i < tapes_[s].size(); i++)
    {
      scanned2_symbols.push_back(tapes_[s][i].get_position_symbol ());
    }
  }

  if (!is_embedded_shutting_down_program_state (process_extended_states_.rbegin()->program_state_))
  {
    const CurSituation as_cur_situation (
			process_extended_states_.rbegin()->program_state_, 
			process_extended_states_.rbegin()->program_substate1_, 
			process_extended_states_.rbegin()->program_substate2_, 
			scanned2_symbols,
			get_symbol_mtypes (scanned2_symbols),
			false	// Never Mind ???
			);

    str_next_situation_o = as_cur_situation.getstr_show();
  }
  else
  {
    str_next_situation_o = string();
  }
 
  // -------------------------------------

  if (daemon_behavior == AGGRESSIVE_NON_DETERMINISTIC_OUTSIDE_RULE_TYPE)
  {
    return false;
  }
  return true;
  

} // perform_machine_step



// =========
void TuringMachine::translate_logical_rule_to_actual_program_triple_state_s (
			const NextSituation& next_rule_situation_i,
			const state_t& last_process_substate1_i,
			const state_t& last_process_substate2_i,
			state_t& process_state_to_be_o,
			state_t& process_substate1_to_be_o,
			state_t& process_substate2_to_be_o
			)
{

  process_state_to_be_o	= next_rule_situation_i.get_state();

  process_substate1_to_be_o.clear();
  process_substate2_to_be_o.clear();

  // ------
const bool flag_next_substate1 = (
		(next_rule_situation_i.get_substate1() == get_neutral_next_the_same_substate_alphabet_symbol())
		||
		(next_rule_situation_i.get_substate1() == get_neutral_next_not_the_same_substate_alphabet_symbol())
		);
  if (next_rule_situation_i.get_substate1() == get_neutral_next_the_same_substate_alphabet_symbol())
  {
    assert (flag_next_substate1);
    process_substate1_to_be_o = last_process_substate1_i;
  }
  if (next_rule_situation_i.get_substate1() == get_neutral_next_not_the_same_substate_alphabet_symbol())
  {
    assert (flag_next_substate1);
    process_substate1_to_be_o = get_neutral_cur_substate_alphabet_symbol();
  }
  if (!flag_next_substate1)
  {
    process_substate1_to_be_o = next_rule_situation_i.get_substate1();
  }

  // ------
const bool flag_next_substate2 = (
		(next_rule_situation_i.get_substate2() == get_neutral_next_the_same_substate_alphabet_symbol())
		||
		(next_rule_situation_i.get_substate2() == get_neutral_next_not_the_same_substate_alphabet_symbol())
		);
  if (next_rule_situation_i.get_substate2() == get_neutral_next_the_same_substate_alphabet_symbol())
  {
    assert (flag_next_substate2);
    process_substate2_to_be_o = last_process_substate2_i;
  }
  if (next_rule_situation_i.get_substate2() == get_neutral_next_not_the_same_substate_alphabet_symbol())
  {
    assert (flag_next_substate2);
    process_substate2_to_be_o = get_neutral_cur_substate_alphabet_symbol();
  }
  if (!flag_next_substate2)
  {
    process_substate2_to_be_o = next_rule_situation_i.get_substate2();
  }

  assert (process_state_to_be_o != state_t());
  assert (process_substate1_to_be_o != state_t());
  assert (process_substate2_to_be_o != state_t());

} // translate_logical_rule_to_actual_rule_s


// =========
bool TuringMachine::perform_tester_step (
		state_t&	program_state_o, 
		bool&		controlling_flag_o, 
		const string&	msg_i
		)
{
  IF_NOT_EMPTY (msg_i, 5, '=');
  
vector<symbol_mt>	scanned_symbols;
  for (size_t i = 0; i < tapes_[TESTER_TAPE].size(); i++)
  {
    scanned_symbols.push_back(tapes_[TESTER_TAPE][i].get_position_symbol ());
  }

  assert (!neutral_program_states_.empty());
const CurSituation cur_situation (
		   	*process_tester_states_.rbegin(), 
			get_neutral_cur_substate_alphabet_symbol(),
			get_neutral_cur_substate_alphabet_symbol(),
			scanned_symbols,
			vector<SymbolTypes> (),
			false	// Never Mind ???
			);
  // XXX ???  assert (!is_halting_program_state (*process_tester_states_.rbegin()));

const Transitions_t::const_iterator iter = user_defined_transitions_.find (cur_situation);

  if (iter == user_defined_transitions_.end()) return false;


  // =========================
  assert (user_defined_transitions_.count(iter->first) == 1);
  controlling_flag_o = iter->first.get_controlling_flag();

size_t dist = distance (user_defined_transitions_.begin(), iter);
  assert (dist < user_defined_transitions_.size());

// -----------------
  process_tester_transitions_.push_back (iter);

state_t	the_state (iter->second.get_state ());

  process_tester_states_.push_back (the_state);
  for (size_t i = 0; i < tapes_[TESTER_TAPE].size(); i++)
  {
    tapes_[TESTER_TAPE][i].set_position_symbol (iter->second.get_symbol (i));
    tapes_[TESTER_TAPE][i].shift_position (iter->second.get_shift(i));
  }
 
  program_state_o = *process_tester_states_.rbegin();
  return true;

} // perform_tester_step


// =========
bool TuringMachine::process_input (
		const vector<vector<symbol_mt> >& input_words_i,
		const string& msg_i
		)
{
  // --------------------------------------
  IF_NOT_EMPTY (msg_i, 5, '=');

  if (!check_process_results_)
  {
    FATAL_MSG	("Invalid Turing Machine definition");
    return false;
  }

  cout << endl;
  if (!check_input_words(input_words_i))
  {
    show_input (input_words_i, false);
    FATAL_MSG	("Invalid input words");
    return false;
  }
  clear_process_data ();
  show_input (input_words_i);
  set_input (input_words_i);


  // --------------------------------------
  // --------------------------------------
state_t	program_state;
bool	ret_bool_value;

  cout << endl;
  cout << endl;

  cout << "\t"
       << string (5, '#')
       << " "
       << getstr_id_info ()
       << "Machine-Processing"
       << " "
       << string (5, '#')
       << endl;

  cout << endl;
  cout << endl;

size_t serial_no = 1;
size_t tester_serial_no = 1;

TesterCalculationResults	ret_tester_code;
MachineCalculationResults	ret_machine_code = UNDEF_MachineCalculationResults;

bool	success_tester_flag;
bool	success_machine_flag;

bool	check_flag;

  // ---------------------
  show_situation ("Initial");

Tapes_t	tester_tape (tapes_[TESTER_TAPE]);
  for (size_t i = 0; ; i++)
  {
    const string text0 (3, '=');
    const string text1 (" Check-interval-" + to_string (i + 1));

    cout << endl;
    cout << endl;
    cout << endl;
    cout << "\t" << text0 << text1 << " : START " << text0 << endl;

    if (ret_machine_code != MACHINE_CALCULATION__CHECK_POINT_AFTER_RECOVERY)
    {
      tapes_[TESTER_TAPE] = tester_tape;
      tester_tape.clear();
      ret_tester_code = execute_tester_checked_interval (i + 1, tester_serial_no, success_tester_flag);
      tester_tape = tapes_[TESTER_TAPE];
    }
    else
    {
      ostringstream oss;
      oss << "Tester skips Interval-" << (i + 1) << " after Recovery";

      cout << endl;
      cout << "\t   " << string (oss.str().size(), '=') << endl;
      cout << "\t   " << oss.str() << endl;
      cout << "\t   " << string (oss.str().size(), '=') << endl;
      cout << endl;

    }
    assert (!tapes_[TESTER_TAPE].empty());

    ret_machine_code = execute_machine_checked_interval (i + 1, serial_no, success_machine_flag);

    check_flag = (
		(
		  (ret_tester_code == TESTER_CALCULATION__USER_REQUIRED_CHECK_POINT) 
		  && 
		  (ret_machine_code == MACHINE_CALCULATION__CHECK_POINT_AFTER_BACKING_UP)
		)
		||
		(
		  (ret_tester_code == TESTER_CALCULATION__USER_REQUIRED_CHECK_POINT) 
		  && 
		  (ret_machine_code == MACHINE_CALCULATION__CHECK_POINT_AFTER_RECOVERY)
		)
		||
		(
		  (ret_tester_code == TESTER_CALCULATION__AUTOMATIC_HALTING_CHECK_POINT) 
		  && 
		  (ret_machine_code == MACHINE_CALCULATION__SHUTTING_DOWN_STATE)
		  )
		||
		(
		  (ret_tester_code == TESTER_CALCULATION__AUTOMATIC_NOT_APPLICABLE_CHECK_POINT) 
		  && 
		  (ret_machine_code == MACHINE_CALCULATION__CHECK_POINT_AFTER_NO_APPLICABLE_RULE)
		)
		);

    cout << endl;
    cout << "\t" << text0 << text1 << " : Check results - ";
    if (check_flag)
    {
      cout << "OK";
    }
    else
    {
      cout << "Error";
    }
    cout << " " << text0 << endl;
    
const bool exit_flag = (
		(ret_machine_code == MACHINE_CALCULATION__SHUTTING_DOWN_STATE)
		||
		(ret_machine_code == MACHINE_CALCULATION__CHECK_POINT_AFTER_NO_APPLICABLE_RULE)
		||
		(ret_machine_code == MACHINE_CALCULATION__EMERGENCY_STATE)
		);

    if (exit_flag) 
    {
      cout << endl;
      cout << "\t" << text0 << " End of checking ---> " << text1 << " is last checked interval" << endl;
      break;
    }

    cout << "\t" << text0 << text1 << " : FINISH " << text0 << endl;

    if (ret_machine_code != MACHINE_CALCULATION__CHECK_POINT_AFTER_RECOVERY)
    {
      tapes_[TESTER_TAPE].clear();
    }

  }

  // --------------------------------------
  // ------------ Show Summary ------------
const size_t the2_setw = 3;
const string prefix1 (string (the2_setw, '-') + " " + getstr_id_info (true) + " ");
vector<string>	summary_info;

ostringstream oss;

  oss.str("");
  oss  << "" 
       << "Summary Line : " 
       << (check_flag ? "OK   " : "ERROR")
       << "  "
       << "T="
       << (success_tester_flag ? "Acp" : "Rej") 
       << "  "
       << "M="
       << (success_machine_flag ? "Acp" : "Rej");
  summary_info.push_back (oss.str());  

  summary_info.push_back ("");  

  oss.str("");
  oss  << "" 
       << "Check   : Tester vs. Machine -> " 
       << (check_flag ? "OK" : "ERROR");
  summary_info.push_back (oss.str());  

  oss.str("");
  oss  << "" 
       << "Tester  : Input word(s) " 
       << (success_tester_flag ? "ACCEPTED" : "REJECTED") 
       << " "
       << "( "
       << process_tester_transitions_.size()
       << " steps"
       << ")"; 
  summary_info.push_back (oss.str());  

  oss.str("");
  oss  << "" 
       << "Machine : Input word(s) " 
       << (success_machine_flag ? "ACCEPTED" : "REJECTED") 
       << " " 
       << "( "
       << process_transitions_.size()
       << " steps"
       << ")"; 
  summary_info.push_back (oss.str());  

  summary_info.push_back ("");  

  oss.str("");
  oss  << "" 
       << "User-defined program states   : " 
       << (user_defined_initial_program_states_.size() + user_defined_halting_program_states_.size() + user_defined_internal_program_states_.size());
  summary_info.push_back (oss.str());  

  oss.str("");
  oss  << "" 
       << "User-defined transition rules : " 
       << user_defined_transitions_.size();
  summary_info.push_back (oss.str());  

  oss.str("");
  oss  << ""
       << "User-defined check-points     : " 
       << user_required_check_point_program_states_.size();
  summary_info.push_back (oss.str());  

  oss.str("");
  oss  << ""
       << "Daemon transition rules       : " 
       << daemon_defined_fault_transitions_.size();
  summary_info.push_back (oss.str());  


  oss.str("");
  oss  << ""
       << "Constructed transition rules  : " 
       << built_transitions_.size();
  summary_info.push_back (oss.str());  

const size_t max_size1 (get_max_size(summary_info) + prefix1.size());

  cout << endl;
  cout << "\t" << string (max_size1, '-') << endl;
  for (size_t id1 = 0; id1 < summary_info.size(); id1++)
  {
    cout << "\t";
    if (summary_info[id1].empty())
    {
      cout << string (max_size1, '.');
    }
    else
    {
      cout << prefix1 << summary_info[id1];
    }
    cout << endl;
  }
  cout << "\t" << string (max_size1, '-') << endl;
  cout <<  endl;

  // -----------------------------
  show_statistics ();

  ret_bool_value = success_machine_flag;
  return ret_bool_value;
}


// =========
MachineCalculationResults TuringMachine::execute_machine_checked_interval (
			size_t		interval_no_i, 
			size_t&		serial_no_io, 
			bool&		success_flag_o
			)
{
const string	prefix3 (3, ' ');

  if (!process_extended_states_.empty())
  {
    assert (!is_halting_program_state (process_extended_states_.rbegin()->program_state_));
    assert (!is_embedded_shutting_down_program_state (process_extended_states_.rbegin()->program_state_));
  }

  assert (!tapes_[TESTER_TAPE].empty());


state_t	program_state;
MachineCalculationResults ret_enum_value = UNDEF_MachineCalculationResults;

bool	controlling_flag;

  // ---------------------
  program_state = state_t();

  cout << endl;
  cout << endl;
  for (int i = 0; ; i++)
  {
    const state_t prev_program_state (program_state);
    string str_cur_situation;
    string str_as_cur_situation;
    success_flag_o = perform_machine_step (
			program_state, 
			controlling_flag, 
			str_cur_situation, 
			str_as_cur_situation
			);
    cout << endl << "\t" << "==M==| Current Configuration : " << str_cur_situation << endl;

    if (success_flag_o)
    {
      cout << endl;
      cout << endl;
      show_situation (string(), interval_no_i, (i + 1), serial_no_io++);

      if (is_embedded_shutting_down_program_state (program_state))
      {

        assert (prev_program_state == EMBSTATE_U2);
        cout << endl;
        cout << endl;
        cout << "\t" << "Successfully : Current state is embedded shutting down state" << endl;
        ret_enum_value = MACHINE_CALCULATION__SHUTTING_DOWN_STATE;
        break;
      }


      if (prev_program_state == EMBSTATE_T4)
      {
        if (!((program_state == EMBSTATE_T4) || (program_state == EMBSTATE_S1)))
        {
          cout << prefix3 << "Machine-Required Check-Point : Data Checked (Backing Up)" << endl;
          ret_enum_value = MACHINE_CALCULATION__CHECK_POINT_AFTER_BACKING_UP;
          break;
        }
      }

      if (prev_program_state == EMBSTATE_W4)
      {
        if (!((program_state == EMBSTATE_W4) || (program_state == EMBSTATE_V1)))
        {
          cout << prefix3 << "Machine-Required Check-Point : Data Checked (Recovery)" << endl;
          ret_enum_value = MACHINE_CALCULATION__CHECK_POINT_AFTER_RECOVERY;
          break;
        }
      }

    }
    else
    {
      if (program_state == EMBSTATE_E1)
      {
        cout << endl;
        cout << endl;
        cout << endl;
        cout << "\t" << "--- Machine broken ---" << endl;
        cout << endl;
        cout << "\t" << "Failure : Unable to continue calculation" << endl;
        cout << "\t" << "==M==| Emergency Configuration : " << str_as_cur_situation << endl;
        cout << endl;
        cout << endl;
        show_situation ("Emergency");
        cout << endl;
        cout << endl;
 
        ret_enum_value = MACHINE_CALCULATION__EMERGENCY_STATE;
        break;
      }

      cout << "\t" << "--- Applicable rule not found ---" << endl;
      cout << endl;
      cout << "\t" << "Unsuccessfully : 1) There exists no applicable rule" << endl;
      cout << "\t" << "                 2) Current state is not embedded shutting down state" << endl;
      ret_enum_value = MACHINE_CALCULATION__CHECK_POINT_AFTER_NO_APPLICABLE_RULE;
      break;
    }

  } // for (int i = 


  assert (ret_enum_value != UNDEF_MachineCalculationResults);

  // ------------------
  switch (ret_enum_value)
  {
    case MACHINE_CALCULATION__CHECK_POINT_AFTER_BACKING_UP :
    case MACHINE_CALCULATION__CHECK_POINT_AFTER_RECOVERY :
    case MACHINE_CALCULATION__CHECK_POINT_AFTER_NO_APPLICABLE_RULE :
    case MACHINE_CALCULATION__SHUTTING_DOWN_STATE :
    case MACHINE_CALCULATION__EMERGENCY_STATE :
      // Do nothing
      break;

    case UNDEF_MachineCalculationResults :
      assert (0);
      break;

    default :
      assert (0);
      break;
  }

  return ret_enum_value;

} // execute_machine_checked_interval


// =========
TesterCalculationResults TuringMachine::execute_tester_checked_interval (
		size_t	call_no_i, 
		size_t& serial_no_io,
		bool&	succces_tester_flag_o	
		)
{
TesterCalculationResults ret_enum_value = UNDEF_TesterCalculationResults;
state_t			state;
const CurSituation	cur_situation;
const string		prefix3 (3, ' ');
bool			controlling_flag;

  assert (!tapes_[TESTER_TAPE].empty());

ostringstream oss;

  oss  << ""
       << getstr_id_info ()
       << "Tester-Processing : Interval-"
       << call_no_i;


  cout << endl << endl;
  cout << "\t"
       << string (5, '#')
       << " "
       << oss.str()
       << " "
       << string (5, '#')
       << endl;


const size_t size6 = 6;

  if (!process_tester_states_.empty())
  {
    // XXX ???    assert (!is_halting_program_state (*process_tester_states_.rbegin()));
  }

  cout << endl;
  cout << endl;
  cout << prefix3 << string (size6, '=') << " " << oss.str() << " : BEGIN" << " " << string (size6, '=') << endl;
  show_tester_situation (true);
  for (size_t i = 0; ; i++)
  {
    succces_tester_flag_o = perform_tester_step (state, controlling_flag);  

    show_tester_situation (false, call_no_i, (i + 1), serial_no_io++);


    if (succces_tester_flag_o && controlling_flag)
    {
      cout << prefix3 << "User-Required Check-Point : Data to Be Checked" << endl;
      cout << endl;
      ret_enum_value = TESTER_CALCULATION__USER_REQUIRED_CHECK_POINT;
      break;
    }

    // ----------------- 
    if (!succces_tester_flag_o)
    { 
      cout << prefix3 << "Automatic Check-Point : Data to Be Checked ( 1. There exists no applicable rule;  2. Current state is not user-defined halting state )" << endl;
      cout << endl;
      ret_enum_value = TESTER_CALCULATION__AUTOMATIC_NOT_APPLICABLE_CHECK_POINT;
      break;
    }

    // ------------------
    assert (succces_tester_flag_o);

    if (is_halting_program_state (state))
    {
      cout << prefix3 << "Automatic Check-Point : Data to Be Checked (Current state is user-defined halting state)" << endl;
      cout << endl;
      ret_enum_value = TESTER_CALCULATION__AUTOMATIC_HALTING_CHECK_POINT;
      break;
    }

  } // for (size_t i = 0; ; i++)

  // -----------------------------
  cout << prefix3 << string (size6, '=') << " " << oss.str() << " : END  " << " " << string (size6, '=') << endl;
  cout << endl;

  assert (ret_enum_value != UNDEF_TesterCalculationResults);

  // ------------------
  switch (ret_enum_value)
  {
    case TESTER_CALCULATION__USER_REQUIRED_CHECK_POINT :
    case TESTER_CALCULATION__AUTOMATIC_HALTING_CHECK_POINT :
    case TESTER_CALCULATION__AUTOMATIC_NOT_APPLICABLE_CHECK_POINT :
      // Do nothing
      break;

    case UNDEF_TesterCalculationResults :
      assert (0);
      break;

    default :
      assert (0);
      break;
  }

  return ret_enum_value;
   
} // execute_tester_checked_interval


// =========			 
bool TuringMachine::check_user_defined_program_states () const
{
bool	ret_bool_value = true;

state_t	phisical_empty_program_state = string();
vector<state_t>::const_iterator	find_iter;

  assert (!user_defined_initial_program_states_.empty());
  assert (!user_defined_halting_program_states_.empty());
  assert (!user_defined_internal_program_states_.empty());

  // ---------
  find_iter = find (user_defined_initial_program_states_.begin(), user_defined_initial_program_states_.end(), phisical_empty_program_state); 
  if (find_iter != user_defined_initial_program_states_.end())
  {
    ret_bool_value = false;
    FATAL_MSG	("Initial program state#"
		<< distance (user_defined_initial_program_states_.begin(), find_iter)
		<< " is empty : <"
		<< *find_iter
		<< ">"
		);
  }

  find_iter = find (user_defined_halting_program_states_.begin(), user_defined_halting_program_states_.end(), phisical_empty_program_state); 
  if (find_iter != user_defined_halting_program_states_.end())
  {
    ret_bool_value = false;
    FATAL_MSG	("Halting program state#"
		<< distance (user_defined_halting_program_states_.begin(), find_iter)
		<< " is empty : <"
		<< *find_iter
		<< ">"
		);
  }


  find_iter = find (user_defined_internal_program_states_.begin(), user_defined_internal_program_states_.end(), phisical_empty_program_state); 
  if (find_iter != user_defined_internal_program_states_.end())
  {
    ret_bool_value = false;
    FATAL_MSG	("Internal program state#"
		<< distance (user_defined_internal_program_states_.begin(), find_iter)
		<< " is empty : <"
		<< *find_iter
		<< ">"
		);
  }

  // ------
vector<state_t> tmp_all_program_states (get_all_user_defined_program_states());
vector<state_t>::iterator	pos_iter;
  for (pos_iter = tmp_all_program_states.begin(); pos_iter != tmp_all_program_states.end(); pos_iter++)
  {
    assert (count (pos_iter, tmp_all_program_states.end(), *pos_iter));
    if (count (pos_iter, tmp_all_program_states.end(), *pos_iter) > 1)
    {
      ret_bool_value = false;
      FATAL_MSG	("User-Defined Program State "
		<< "<"
		<< (*pos_iter)
	 	<< ">"
		<< " occurs more than once"
		);
    }
  }

  // -----------
  return ret_bool_value;
} // check_user_defined_program_states



// =========			 
bool TuringMachine::check_extra_program_states () const
{
bool	ret_bool_value = true;

  assert (!neutral_program_states_.empty());
  assert (!pre_initial_program_states_.empty());
  assert (!post_halting_program_states_.empty());

  if (user_required_check_point_program_states_.empty())
  {
    WARNING_MSG ("No check points defined !"
                << endl 
                << string (23, ' ')
                << "Note! Check points are defined in transitions file"
                << endl
                );  
  }

  assert (!embedded_program_states_.empty());

  for (size_t i = 0; i < embedded_program_states_.size(); i++)
  {
    assert (!embedded_program_states_[i].empty());
  }

  // -----------
  return ret_bool_value;
} // check_extra_program_states



// =========			 
void TuringMachine::set_pseudo_symbols_alphabet ()
{
const vector<symbol_mt>	special_notation (get_notation_of_pseudo_symbols_alphabet());
size_t counter = 0;
  assert (!special_notation.empty());
  pseudo_never_mind_symbols_alphabet_.push_back (special_notation[counter++]);

  assert (counter < special_notation.size());
  pseudo_not_marker_symbols_alphabet_.push_back (special_notation[counter++]);

  assert (counter < special_notation.size());
  pseudo_not_marker_and_not_blank_symbols_alphabet_.push_back (special_notation[counter++]);

  assert (counter < special_notation.size());
  pseudo_write_nothing_symbols_alphabet_.push_back (special_notation[counter++]);

}


// =========			 
void TuringMachine::share_information1 () const
{
vector<state_t> allowed_states;
vector<symbol_mt> allowed_symbols;

  // ------ CurSituation ------
vector<symbol_mt> cur_vect1 (get_all_master_tape_alphabet_symbols ());
vector<symbol_mt> cur_vect2 (get_all_synchronous_tape_alphabet_symbols_s ());

  assert (!cur_vect1.empty());
  assert (!cur_vect2.empty());
  assert (!pseudo_never_mind_symbols_alphabet_.empty());

  allowed_states.clear();  
  allowed_symbols.clear();  

  copy (cur_vect1.begin(), cur_vect1.end(), back_inserter(allowed_symbols));
  copy (cur_vect2.begin(), cur_vect2.end(), back_inserter(allowed_symbols));
  copy (pseudo_never_mind_symbols_alphabet_.begin(), pseudo_never_mind_symbols_alphabet_.end(), back_inserter(allowed_symbols));
  copy (pseudo_not_marker_symbols_alphabet_.begin(), pseudo_not_marker_symbols_alphabet_.end(), back_inserter(allowed_symbols));
  copy (pseudo_not_marker_and_not_blank_symbols_alphabet_.begin(), pseudo_not_marker_and_not_blank_symbols_alphabet_.end(), back_inserter(allowed_symbols));


  assert (!allowed_symbols.empty());

  CurSituation::set_shared_data1_s (
		allowed_symbols,
		allowed_states	// empty
		);


  // ------ NextSituation ------

vector<symbol_mt> next_vect1 (get_all_master_tape_alphabet_symbols ());
vector<symbol_mt> next_vect2 (get_all_synchronous_tape_alphabet_symbols_s ());

  assert (!next_vect1.empty());
  assert (!next_vect2.empty());
  assert (!pseudo_write_nothing_symbols_alphabet_.empty());

  allowed_states.clear();  
  allowed_symbols.clear();  

  copy (next_vect1.begin(), next_vect1.end(), back_inserter(allowed_symbols));
  copy (next_vect2.begin(), next_vect2.end(), back_inserter(allowed_symbols));
  copy (pseudo_write_nothing_symbols_alphabet_.begin(), pseudo_write_nothing_symbols_alphabet_.end(), back_inserter(allowed_symbols));


  assert (!allowed_symbols.empty());

  NextSituation::set_shared_data1_s (
		pseudo_write_nothing_symbols_alphabet_,
		allowed_symbols,
		allowed_states	// empty
		);


  // ------ Tape ------
  Tape::set_shared_data_s (pseudo_write_nothing_symbols_alphabet_);

}


// =========			 
void TuringMachine::share_information2 () const
{
vector<state_t> allowed_states;
vector<state_t> allowed_neutral_cur_states;

  // ------ CurSituation ------
     // --- allowed_states ---
vector<state_t> cur_vect1 (get_all_user_defined_program_states ());
vector<state_t> cur_vect2 (get_all_extra_program_states ());

  assert (!cur_vect1.empty());
  assert (!cur_vect2.empty());

  allowed_states.clear();
  copy (cur_vect1.begin(), cur_vect1.end(), back_inserter(allowed_states));
  copy (cur_vect2.begin(), cur_vect2.end(), back_inserter(allowed_states));

  assert (belongs_to (EMBSTATE_E1, allowed_states));
  allowed_states.push_back(EMBSTATE_E1);

  assert (!allowed_states.empty());

const size_t prev_size = allowed_states.size();
  assert (!embedded_program_states_[EMBEDDED_PROGRAM_STATE_KIND__SHUTTING_DOWN].empty());

vector<state_t>::iterator find_iter;
  for (size_t i = 0; i < embedded_program_states_[EMBEDDED_PROGRAM_STATE_KIND__SHUTTING_DOWN].size(); i++)
  {
    find_iter = find (allowed_states.begin(), allowed_states.end(), embedded_program_states_[EMBEDDED_PROGRAM_STATE_KIND__SHUTTING_DOWN][i]);
    if (find_iter != allowed_states.end())
    {
      allowed_states.erase(find_iter);
    }
  }
  
  assert (prev_size == (allowed_states.size() + embedded_program_states_[EMBEDDED_PROGRAM_STATE_KIND__SHUTTING_DOWN].size()));

     // --- allowed_neutral_cur_states ---
vector<state_t> cur_vect3 (1, get_neutral_cur_substate_alphabet_symbol());

  assert (!cur_vect3.empty());

  allowed_neutral_cur_states.clear();
  copy (cur_vect3.begin(), cur_vect3.end(), back_inserter(allowed_neutral_cur_states));

  assert (!allowed_neutral_cur_states.empty());


     // --- set_shared_data2_s ---
  CurSituation::set_shared_data2_s (allowed_states, allowed_neutral_cur_states);



  // ------ NextSituation ------
vector<state_t> next_vect1 (get_all_user_defined_program_states ());
vector<state_t> next_vect2 (get_all_extra_program_states ());

  assert (!next_vect1.empty());
  assert (!next_vect2.empty());

  allowed_states.clear();
  copy (next_vect1.begin(), next_vect1.end(), back_inserter(allowed_states));
  copy (next_vect2.begin(), next_vect2.end(), back_inserter(allowed_states));

  assert (!allowed_states.empty());


     // --- set_shared_data2_s ---
  NextSituation::set_shared_data2_s (allowed_states);

}


// =========			 
void TuringMachine::set_extra_program_states ()
{

  assert (neutral_program_states_.empty());
  assert (pre_initial_program_states_.empty());
  assert (post_halting_program_states_.empty());
  assert (user_required_check_point_program_states_.empty());

  for (size_t i = 0; i < embedded_program_states_.size(); i++)
  {
    assert (!embedded_program_states_.empty());
    for (size_t j = 0; j < embedded_program_states_[i].size(); j++)
    {
      assert (embedded_program_states_[i].empty());
    }
  }

Transitions_t::const_iterator pos_iter;
  for (pos_iter = built_transitions_.begin();
       pos_iter != built_transitions_.end();
       pos_iter++
       )
  {
    if (pos_iter->first.get_controlling_flag ()) 
    {	
      assert (user_required_check_point_program_states_.count(pos_iter->first) == 0);
      user_required_check_point_program_states_[pos_iter->first] = state_t();
    }
  }

const size_t	number_of_neutral_program_states = NUMBER_OF_NEUTRAL_PROGRAM_STATE_KINDS;
const size_t	number_of_pre_initial_program_states = NUMBER_OF_PRE_INITIAL_PROGRAM_STATE_KINDS;
const size_t	number_of_post_halting_program_states = NUMBER_OF_POST_HALTING_PROGRAM_STATE_KINDS;

const vector<state_t>	special_notation (
				get_notation_of_extra_program_states (
					(
					  user_required_check_point_program_states_.size() + number_of_neutral_program_states 
					  + 
					  number_of_pre_initial_program_states
					  + 
					  number_of_post_halting_program_states
					)
					)
					);


size_t	counter = 0;
  assert (neutral_program_states_.empty());
  for (size_t i = 0; i < number_of_neutral_program_states; i++)
  {
    neutral_program_states_.push_back(special_notation [counter++]);  
  }
  assert (!neutral_program_states_.empty());

  assert (pre_initial_program_states_.empty());
  for (size_t i = 0; i < number_of_pre_initial_program_states; i++)
  {
    pre_initial_program_states_.push_back(special_notation [counter++]);  
  }
  assert (!pre_initial_program_states_.empty());


  assert (post_halting_program_states_.empty());
  for (size_t i = 0; i < number_of_post_halting_program_states; i++)
  {
    post_halting_program_states_.push_back(special_notation [counter++]);  
  }
  assert (!post_halting_program_states_.empty());

map<CurSituation, state_t>::iterator pos2_iter;
  for (pos2_iter = user_required_check_point_program_states_.begin(); 
       pos2_iter != user_required_check_point_program_states_.end(); 
       pos2_iter++
       )
  {
    assert (counter < special_notation.size());
    pos2_iter->second = special_notation [counter++];
  }  

  // -----------
  // -----------
  assert (!embedded_program_states_.empty());
  assert (embedded_program_states_.size() == NUMBER_OF_EMBEDDED_PROGRAM_STATE_KINDS);

  // -----------
  embedded_program_states_[EMBEDDED_PROGRAM_STATE_KIND__PRE_INITIAL_BACKUP].resize (NUMBER_OF_PRE_INITIAL_BACKUP_STATES);
  embedded_program_states_[EMBEDDED_PROGRAM_STATE_KIND__COMPUTATION_CONTROL].resize (NUMBER_OF_COMPUTATION_CONTROL_STATES);
  embedded_program_states_[EMBEDDED_PROGRAM_STATE_KIND__BACKUP].resize (NUMBER_OF_BACKUP_STATES);
  embedded_program_states_[EMBEDDED_PROGRAM_STATE_KIND__BACKUP_CONTROL].resize (NUMBER_OF_BACKUP_CONTROL_STATES);
  embedded_program_states_[EMBEDDED_PROGRAM_STATE_KIND__RECOVERY].resize (NUMBER_OF_RECOVERY_STATES);
  embedded_program_states_[EMBEDDED_PROGRAM_STATE_KIND__RECOVERY_CONTROL].resize (NUMBER_OF_RECOVERY_CONTROL_STATES);
  embedded_program_states_[EMBEDDED_PROGRAM_STATE_KIND__SUMMARY_CONTROL].resize (NUMBER_OF_SUMMARY_CONTROL_STATES);
  embedded_program_states_[EMBEDDED_PROGRAM_STATE_KIND__EMERGENCY].resize (NUMBER_OF_EMBEDDED_EMERGENCY_STATES);
  embedded_program_states_[EMBEDDED_PROGRAM_STATE_KIND__SHUTTING_DOWN].resize (NUMBER_OF_EMBEDDED_HALTING_STATES);


  // ------
  for (size_t s = 0; s < embedded_program_states_.size(); s++)
  {
    assert (!embedded_program_states_[s].empty());
    assert (embedded_program_states_[s].size() == descr_of_embedded_program_states_s[s].size());
    assert (static_cast<size_t>(count (embedded_program_states_[s].begin(), embedded_program_states_[s].end(), state_t())) == embedded_program_states_[s].size());
    for (size_t i = 0; i < embedded_program_states_[s].size(); i++)
    {
      assert (counter < special_notation.size());
      embedded_program_states_[s][i] = special_notation [counter++];
    }  
  }

} // set_extra_program_states



// =========
vector<state_t> TuringMachine::get_notation_of_extra_program_states(size_t add_size_i) const
{
vector<state_t> ret_vector;
state_t		tmp_state ("");
  
  while (ret_vector.size() <= (total_embedded_program_states_CNS + add_size_i))
  {
    tmp_state = get_next_size_ordered_word (tmp_state);
    if (is_user_defined_program_state (tmp_state)) continue;
    ret_vector.push_back(tmp_state);
  }
  return ret_vector;
}


// =========
vector<symbol_mt> TuringMachine::get_notation_of_pseudo_symbols_alphabet () const
{
vector<symbol_mt>	ret_vector;
symbol_mt		tmp_symbol ("");
vector<symbol_mt>	master_alphabet (get_all_master_tape_alphabet_symbols ());	
vector<symbol_mt>	synhro_alphabet (get_all_synchronous_tape_alphabet_symbols_s());
  
  while (ret_vector.size() <= (master_alphabet.size() + synhro_alphabet.size()))
  {
    tmp_symbol = symbol_mt (get_next_size_ordered_word (tmp_symbol.getstr_symbol()));
    if (count (master_alphabet.begin(), master_alphabet.end(), tmp_symbol)) continue;
    if (count (synhro_alphabet.begin(), synhro_alphabet.end(), tmp_symbol)) continue;
    ret_vector.push_back(tmp_symbol);
  }
  return ret_vector;
}


// =========
bool TuringMachine::check_daemon_states_s ()
{
bool	ret_bool_value = true;

state_t	phisical_empty_daemon_state = string();
vector<state_t>::const_iterator	find_iter;

  assert (!daemon_passive_states_s.empty());
  assert (!daemon_active_states_s.empty());
  assert (!daemon_aggressive_states_s.empty());

  // ---------
  find_iter = find (daemon_passive_states_s.begin(), daemon_passive_states_s.end(), phisical_empty_daemon_state); 
  if (find_iter != daemon_passive_states_s.end())
  {
    ret_bool_value = false;
    BUG_MSG	("Passive daemon state#"
		<< distance (daemon_passive_states_s.begin(), find_iter)
		<< " is empty : <"
		<< *find_iter
		<< ">"
		);
  }

  find_iter = find (daemon_active_states_s.begin(), daemon_active_states_s.end(), phisical_empty_daemon_state); 
  if (find_iter != daemon_active_states_s.end())
  {
    ret_bool_value = false;
    BUG_MSG	("Active daemon state#"
		<< distance (daemon_active_states_s.begin(), find_iter)
		<< " is empty : <"
		<< *find_iter
		<< ">"
		);
  }


  find_iter = find (daemon_aggressive_states_s.begin(), daemon_aggressive_states_s.end(), phisical_empty_daemon_state); 
  if (find_iter != daemon_aggressive_states_s.end())
  {
    ret_bool_value = false;
    BUG_MSG	("Aggressive daemon state#"
		<< distance (daemon_aggressive_states_s.begin(), find_iter)
		<< " is empty : <"
		<< *find_iter
		<< ">"
		);
  }

  // ------
vector<state_t> tmp_all_daemon_states (get_all_daemon_states_s ());
vector<state_t>::iterator	pos_iter;
  for (pos_iter = tmp_all_daemon_states.begin(); pos_iter != tmp_all_daemon_states.end(); pos_iter++)
  {
    assert (count (pos_iter, tmp_all_daemon_states.end(), *pos_iter));
    if (count (pos_iter, tmp_all_daemon_states.end(), *pos_iter) > 1)
    {
      ret_bool_value = false;
      BUG_MSG	("Daemon State "
		<< "<"
		<< (*pos_iter)
	 	<< ">"
		<< " occurs more than once"
		);
    }
  }

  // -----------
  assert (ret_bool_value);
  return ret_bool_value;
} // check_daemon_states_S



// =========
bool TuringMachine::check_tester_states_s ()
{
bool	ret_bool_value = true;

state_t	phisical_empty_tester_state = string();
vector<state_t>::const_iterator	find_iter;

  assert (!tester_tracking_states_s.empty());
  assert (!tester_stabilizing_states_s.empty());

  // ---------
  find_iter = find (tester_tracking_states_s.begin(), tester_tracking_states_s.end(), phisical_empty_tester_state); 
  if (find_iter != tester_tracking_states_s.end())
  {
    ret_bool_value = false;
    BUG_MSG	("Tracking tester state#"
		<< distance (tester_tracking_states_s.begin(), find_iter)
		<< " is empty : <"
		<< *find_iter
		<< ">"
		);
  }

  find_iter = find (tester_stabilizing_states_s.begin(), tester_stabilizing_states_s.end(), phisical_empty_tester_state); 
  if (find_iter != tester_stabilizing_states_s.end())
  {
    ret_bool_value = false;
    BUG_MSG	("Stabilizing tester state#"
		<< distance (tester_stabilizing_states_s.begin(), find_iter)
		<< " is empty : <"
		<< *find_iter
		<< ">"
		);
  }


  // ------
vector<state_t> tmp_all_tester_states (get_all_tester_states_s ());
vector<state_t>::iterator	pos_iter;
  for (pos_iter = tmp_all_tester_states.begin(); pos_iter != tmp_all_tester_states.end(); pos_iter++)
  {
    assert (count (pos_iter, tmp_all_tester_states.end(), *pos_iter));
    if (count (pos_iter, tmp_all_tester_states.end(), *pos_iter) > 1)
    {
      ret_bool_value = false;
      BUG_MSG	("Tester State "
		<< "<"
		<< (*pos_iter)
	 	<< ">"
		<< " occurs more than once"
		);
    }
  }

  // -----------
  assert (ret_bool_value);
  return ret_bool_value;
} // check_tester_states_S



// =========
bool TuringMachine::check_apparatus_states_s ()
{
bool	ret_bool_value = true;

state_t	phisical_empty_apparatus_state = string();
vector<state_t>::const_iterator	find_iter;

  assert (!apparatus_normal_states_s.empty());
  assert (!apparatus_emergency_states_s.empty());

  // ---------
  find_iter = find (apparatus_normal_states_s.begin(), apparatus_normal_states_s.end(), phisical_empty_apparatus_state); 
  if (find_iter != apparatus_normal_states_s.end())
  {
    ret_bool_value = false;
    BUG_MSG	("Normal apparatus state#"
		<< distance (apparatus_normal_states_s.begin(), find_iter)
		<< " is empty : <"
		<< (*find_iter)
		<< ">"
		);
  }

  find_iter = find (apparatus_emergency_states_s.begin(), apparatus_emergency_states_s.end(), phisical_empty_apparatus_state); 
  if (find_iter != apparatus_emergency_states_s.end())
  {
    ret_bool_value = false;
    BUG_MSG	("Emergency apparatus state#"
		<< distance (apparatus_emergency_states_s.begin(), find_iter)
		<< " is empty : <"
		<< (*find_iter)
		<< ">"
		);
  }


  // ------
vector<state_t> tmp_all_apparatus_states (get_all_apparatus_states_s ());
vector<state_t>::iterator	pos_iter;
  for (pos_iter = tmp_all_apparatus_states.begin(); pos_iter != tmp_all_apparatus_states.end(); pos_iter++)
  {
    assert (count (pos_iter, tmp_all_apparatus_states.end(), *pos_iter));
    if (count (pos_iter, tmp_all_apparatus_states.end(), *pos_iter) > 1)
    {
      ret_bool_value = false;
      BUG_MSG	("Emergency State "
		<< "<"
		<< (*pos_iter)
	 	<< ">"
		<< " occurs more than once"
		);
    }
  }

  // -----------
  assert (ret_bool_value);
  return ret_bool_value;
} // check_apparatus_states_S



// =========
bool TuringMachine::check_alphabets () const
{
bool	ret_bool_value = true;

Tapes_t::const_iterator	pos_iter;

  for (size_t i = 0; i < NUMBER_OF_TAPE_KINDS; i++)
  {
    for (pos_iter = tapes_[i].begin(); pos_iter != tapes_[i].end(); pos_iter++)
    {
      assert (pos_iter->first == static_cast<size_t> (distance (tapes_[i].begin(), pos_iter)));
      if (!(pos_iter->second.check_alphabet()))
      {
        // XXX
        ret_bool_value = false;	
      }
    }
  }

  return ret_bool_value;
}


// =========
bool TuringMachine::check_user_defined_transition () const
{
bool	ret_bool_value = true;

Transitions_t::const_iterator	pos1_iter;
Tapes_t::const_iterator		pos2_iter;
size_t		i;
state_t		cur_program_state;
size_t		cur_total_symbols;
symbol_mt	cur_symbol;

state_t		next_program_state;
size_t		next_total_symbols;
symbol_mt	next_symbol;
shift_t	shift;

  if (user_defined_transitions_.empty())
  {
    ret_bool_value = false;
    FATAL_MSG	("No user-defined transition function");
  }

  for (pos1_iter = user_defined_transitions_.begin(); pos1_iter != user_defined_transitions_.end(); pos1_iter++)
  {
    const string user_defined_transitions_line_info ("User-Defined Transition Rule#" + to_string (distance (user_defined_transitions_.begin(), pos1_iter)));
    const string user_defined_transitions_line_and_tape_no_info (user_defined_transitions_line_info + ", tape#");

    // --- first ---
    cur_program_state = pos1_iter->first.get_state();
    cur_total_symbols = pos1_iter->first.get_total_symbols();

    if (!((is_pre_initial_program_state (cur_program_state)) || (is_initial_program_state (cur_program_state)) || (is_internal_program_state (cur_program_state))))
    {
      ret_bool_value = false;
      FATAL_MSG	(user_defined_transitions_line_info
		<< " : illegal cur-program_state = <"
		<< cur_program_state
		<< ">"
		);
    }

    if (cur_total_symbols != tapes_[MASTER_COMPUTATION_TAPE].size())
    {
      ret_bool_value = false;
      FATAL_MSG	(user_defined_transitions_line_info
		<< " : number-of-cur-symbols = "
		<< cur_total_symbols
		<< " is not equal number-of-tapes = "
		<< tapes_[MASTER_COMPUTATION_TAPE].size()
		);
    }


    for (i = 0, pos2_iter = tapes_[MASTER_COMPUTATION_TAPE].begin(); i < cur_total_symbols; i++, pos2_iter++)
    {
      assert (pos2_iter->first == i);
      cur_symbol = pos1_iter->first.get_symbol(i);

      if (!(pos2_iter->second.is_valid_symbol(cur_symbol)))
      {
        ret_bool_value = false;
        FATAL_MSG	(user_defined_transitions_line_and_tape_no_info
			<< i
			<< " : illegal cur-symbol = <"
			<< cur_symbol
			<< ">"
			);
      }
    }

    
    // --- second ---
    next_program_state = pos1_iter->second.get_state();
    next_total_symbols = pos1_iter->second.get_total_symbols();

    //  if (!((is_halting_program_state (next_program_state)) || (is_internal_program_state (next_program_state))))
    //  {
    //    ret_bool_value = false;
    //    FATAL_MSG	(user_defined_transitions_line_info
    //  		<< " : illegal next-program_state = <"
    //  		<< next_program_state
    //  		<< ">"
    //  		);
    //  }

    if (next_total_symbols != tapes_[MASTER_COMPUTATION_TAPE].size())
    {
      ret_bool_value = false;
      FATAL_MSG	(user_defined_transitions_line_info
		<< " : number-of-next-symbols = "
		<< next_total_symbols
		<< " is not equal number-of-tapes = "
		<< tapes_[MASTER_COMPUTATION_TAPE].size()
		);
    }

    for (i = 0, pos2_iter = tapes_[MASTER_COMPUTATION_TAPE].begin(); i < next_total_symbols; i++, pos2_iter++)
    {
      assert (pos2_iter->first == i);
      next_symbol = pos1_iter->second.get_symbol(i);
      shift = pos1_iter->second.get_shift(i);

      if (!(pos2_iter->second.is_valid_symbol(next_symbol)))
      {
        ret_bool_value = false;
        FATAL_MSG	(user_defined_transitions_line_and_tape_no_info
			<< i
			<< " : illegal next-symbol = <"
			<< next_symbol
			<< ">"
			);
      }

      if (!(pos2_iter->second.is_valid_shift(shift)))
      {
        ret_bool_value = false;
        FATAL_MSG	(user_defined_transitions_line_and_tape_no_info
			<< i
			<< " : illegal shift = <"
			<< shift
			<< ">"
			);
      }

    }

  } // for (pos1_iter = user_defined_transitions_.begin(); ...
  return ret_bool_value;
}


// =========
void TuringMachine::set_never_mind_symbol (TapeKinds kind_i, vector<symbol_mt>& vect_o) const
{
const size_t number_of_types = tapes_[kind_i].size();
  assert (vect_o.size() == NUMBER_OF_TAPE_KINDS	* number_of_types);

  for (size_t i = 0; i < number_of_types; i++)
  {
    vect_o [kind_i * number_of_types + i] = pseudo_never_mind_symbols_alphabet_.front();
  }
}


// =========
void TuringMachine::set_write_nothing_symbol (TapeKinds kind_i, vector<SymbolAndShift>& vect_o) const
{
const size_t number_of_types = tapes_[kind_i].size();
  assert (vect_o.size() == NUMBER_OF_TAPE_KINDS	* number_of_types);

  for (size_t i = 0; i < number_of_types; i++)
  {
    vect_o [kind_i * number_of_types + i].set_symbol (pseudo_write_nothing_symbols_alphabet_.front());
  }
}


// =========
void TuringMachine::set_no_shift_s (vector<SymbolAndShift>& vect_o)
{
  for (size_t i = 0; i < vect_o.size(); i++)
  {
    vect_o[i].set_no_shift();
  }
}

