/*
    Chip3voi.c

    Routines for detecting and playing sampled sound on the 3-voice chip.
*/

#include <stdio.h>      /* Standard I/O functions header file. */
#include "play3voi.h"   /* Global constants and variables */
#include "chip3voi.h"   /* Routines for accessing the 3-voice chip */


/*************************************************************************/
/*                                                                       */
/* Global variables.                                                     */
/*                                                                       */
/*************************************************************************/

    /* I/O port for 3-voice chip, set by Detect_3_Voice(), below.  1E0h is 
       in fact the 3-voice chip address on the 2500-series.  On most other 
       machines, it is unused. */
unsigned IOport = 0x1e0;

    /* Level 0 amplification table.  This table leaves the samples 
       unmodified. */
static char Level_0_Table[256] = {
    0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,
    16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,
    32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,
    48,49,50,51,52,53,54,55,56,57,58,59,60,61,62,63,
    64,65,66,67,68,69,70,71,72,73,74,75,76,77,78,79,
    80,81,82,83,84,85,86,87,88,89,90,91,92,93,94,95,
    96,97,98,99,100,101,102,103,104,105,106,107,108,109,110,111,
    112,113,114,115,116,117,118,119,120,121,122,123,124,125,126,127,
    128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
    144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
    160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,
    176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,
    192,193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,
    208,209,210,211,212,213,214,215,216,217,218,219,220,221,222,223,
    224,225,226,227,228,229,230,231,232,233,234,235,236,237,238,239,
    240,241,242,243,244,245,246,247,248,249,250,251,252,253,254,255,
};

    /* Level 1 amplification table.  This table amplifies the samples by 
       applying a quadratic function. */
static char Level_1_Table[256] = {
    0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,
    2,2,2,2,3,3,3,4,4,4,5,5,6,6,7,7,
    8,8,9,9,10,10,11,11,12,13,13,14,15,15,16,17,
    18,18,19,20,21,21,22,23,24,25,26,27,28,29,30,31,
    32,33,34,35,36,37,38,39,40,41,42,43,45,46,47,48,
    50,51,52,53,55,56,57,59,60,61,63,64,66,67,69,70,
    72,73,75,76,78,79,81,82,84,86,87,89,91,92,94,96,
    98,99,101,103,105,106,108,110,112,114,116,118,120,122,124,126,
    128,129,131,133,135,137,139,141,143,145,147,149,150,152,154,156,
    158,159,161,163,164,166,168,169,171,173,174,176,177,179,180,182,
    184,185,186,188,189,191,192,194,195,196,198,199,200,202,203,204,
    206,207,208,209,210,212,213,214,215,216,217,218,219,220,221,222,
    224,224,225,226,227,228,229,230,231,232,233,234,234,235,236,237,
    238,238,239,240,240,241,242,242,243,244,244,245,245,246,246,247,
    248,248,248,249,249,250,250,251,251,251,252,252,252,253,253,253,
    254,254,254,254,254,255,255,255,255,255,255,255,255,255,255,255,
};

    /* Level 2 amplification table.  This table amplifies the samples by 
       applying a cubic function. */
static char Level_2_Table[256] = {
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,
    2,2,2,2,2,3,3,3,3,4,4,4,5,5,5,6,
    6,7,7,8,8,9,9,10,10,11,11,12,13,13,14,15,
    16,16,17,18,19,20,20,21,22,23,24,25,26,27,28,30,
    31,32,33,34,36,37,38,40,41,43,44,45,47,49,50,52,
    54,55,57,59,61,62,64,66,68,70,72,74,76,79,81,83,
    85,88,90,92,95,97,100,102,105,108,110,113,116,119,122,125,
    128,130,133,136,139,142,145,147,150,153,155,158,160,163,165,167,
    170,172,174,176,179,181,183,185,187,189,191,193,194,196,198,200,
    202,203,205,206,208,210,211,212,214,215,217,218,219,221,222,223,
    224,225,227,228,229,230,231,232,233,234,235,235,236,237,238,239,
    240,240,241,242,242,243,244,244,245,245,246,246,247,247,248,248,
    249,249,250,250,250,251,251,251,252,252,252,252,253,253,253,253,
    254,254,254,254,254,254,254,255,255,255,255,255,255,255,255,255,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
};

    /* Level 3 amplification table.  This table amplifies the samples by 
       applying a quartic function. */
static char Level_3_Table[256] = {
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,1,1,1,1,1,1,1,2,2,
    2,2,2,3,3,3,4,4,4,5,5,5,6,6,7,7,
    8,8,9,9,10,10,11,12,12,13,14,15,15,16,17,18,
    19,20,21,22,23,24,26,27,28,29,31,32,34,35,37,38,
    40,42,43,45,47,49,51,53,55,57,60,62,64,67,69,72,
    75,77,80,83,86,89,92,95,98,102,105,109,112,116,120,124,
    128,131,135,139,143,146,150,153,157,160,163,166,169,172,175,178,
    180,183,186,188,191,193,195,198,200,202,204,206,208,210,212,213,
    215,217,218,220,221,223,224,226,227,228,229,231,232,233,234,235,
    236,237,238,239,240,240,241,242,243,243,244,245,245,246,246,247,
    248,248,248,249,249,250,250,250,251,251,251,252,252,252,253,253,
    253,253,253,254,254,254,254,254,254,254,255,255,255,255,255,255,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
};

    /* Level 4 amplification table.  This table amplifies the samples by 
       applying a quintic function. */
static char Level_4_Table[256] = {
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,1,1,1,1,1,1,1,2,2,2,2,2,3,3,3,
    4,4,4,5,5,5,6,6,7,7,8,8,9,10,10,11,
    12,12,13,14,15,16,17,18,19,20,21,23,24,25,27,28,
    30,31,33,35,37,39,41,43,45,47,49,52,54,57,59,62,
    65,68,71,74,78,81,85,88,92,96,100,104,109,113,118,123,
    128,132,137,142,146,151,155,159,163,167,170,174,177,181,184,187,
    190,193,196,198,201,203,206,208,210,212,214,216,218,220,222,224,
    225,227,228,230,231,232,234,235,236,237,238,239,240,241,242,243,
    243,244,245,245,246,247,247,248,248,249,249,250,250,250,251,251,
    252,252,252,252,253,253,253,253,253,254,254,254,254,254,254,254,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
};

    /* Level 5 amplification table.  This table amplifies the samples by 
       applying a 6th-degree function. */
static char Level_5_Table[256] = {
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,
    2,2,2,2,2,3,3,3,4,4,4,5,5,6,6,7,
    7,8,8,9,10,10,11,12,13,14,15,16,17,18,20,21,
    22,24,25,27,29,30,32,34,36,39,41,43,46,48,51,54,
    57,60,63,67,70,74,78,82,86,91,95,100,105,111,116,122,
    128,133,139,144,150,155,160,164,169,173,177,181,185,188,192,195,
    198,201,204,207,209,212,214,216,219,221,223,225,226,228,230,231,
    233,234,235,237,238,239,240,241,242,243,244,245,245,246,247,247,
    248,248,249,249,250,250,251,251,251,252,252,252,253,253,253,253,
    254,254,254,254,254,254,254,255,255,255,255,255,255,255,255,255,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
};

    /* Level 6 amplification table.  This table amplifies the samples by 
       applying a 7th-degree function. */
static char Level_6_Table[256] = {
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    1,1,1,1,1,1,1,2,2,2,2,3,3,3,3,4,
    4,5,5,6,6,7,7,8,9,10,10,11,12,13,14,15,
    17,18,19,21,22,24,26,27,29,31,34,36,38,41,44,47,
    50,53,56,60,64,68,72,76,81,86,91,96,102,108,114,121,
    128,134,141,147,153,159,164,169,174,179,183,187,191,195,199,202,
    205,208,211,214,217,219,221,224,226,228,229,231,233,234,236,237,
    238,240,241,242,243,244,245,245,246,247,248,248,249,249,250,250,
    251,251,252,252,252,252,253,253,253,253,254,254,254,254,254,254,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
};

    /* Level 7 amplification table.  This table amplifies the samples by 
       applying an 8th-degree function. */
static char Level_7_Table[256] = {
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,1,1,1,1,1,1,1,2,2,2,
    2,3,3,4,4,4,5,5,6,6,7,8,9,9,10,11,
    12,13,15,16,17,19,20,22,24,26,28,30,32,35,38,40,
    43,47,50,54,58,62,66,71,76,81,87,93,99,105,112,120,
    128,135,143,150,156,162,168,174,179,184,189,193,197,201,205,208,
    212,215,217,220,223,225,227,229,231,233,235,236,238,239,240,242,
    243,244,245,246,246,247,248,249,249,250,250,251,251,251,252,252,
    253,253,253,253,254,254,254,254,254,254,254,255,255,255,255,255,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
};

    /* Level 8 amplification table.  This table amplifies the samples by 
       applying a 9th-degree function. */
static char Level_8_Table[256] = {
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,
    1,2,2,2,2,3,3,3,4,4,5,5,6,7,7,8,
    9,10,11,12,13,15,16,18,19,21,23,25,27,30,32,35,
    38,41,45,48,52,57,61,66,71,77,83,89,96,103,111,119,
    128,136,144,152,159,166,172,178,184,189,194,198,203,207,210,214,
    217,220,223,225,228,230,232,234,236,237,239,240,242,243,244,245,
    246,247,248,248,249,250,250,251,251,252,252,252,253,253,253,253,
    254,254,254,254,254,254,255,255,255,255,255,255,255,255,255,255,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
};

    /* Level 9 amplification table.  This table amplifies the samples by 
       applying a 10th-degree function. */
static char Level_9_Table[256] = {
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,
    1,1,1,1,1,2,2,2,3,3,3,4,4,5,5,6,
    7,7,8,9,10,11,13,14,16,17,19,21,23,25,28,30,
    33,36,40,43,47,52,56,61,67,72,79,85,93,100,109,118,
    128,137,146,155,162,170,176,183,188,194,199,203,208,212,215,219,
    222,225,227,230,232,234,236,238,239,241,242,244,245,246,247,248,
    248,249,250,250,251,251,252,252,252,253,253,253,254,254,254,254,
    254,254,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
    255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
};

    /* List of available amplification tables. */
static char *AmpTableList[MAX_AMPLIFICATION+1] = {
    Level_0_Table,
    Level_1_Table,
    Level_2_Table,
    Level_3_Table,
    Level_4_Table,
    Level_5_Table,
    Level_6_Table,
    Level_7_Table,
    Level_8_Table,
    Level_9_Table,
};

    /* Selected amplification table. */
char *AmplifyTable = Level_0_Table;

    /* Values to be written to the SN76496 port to play samples. */
static char PortVals[4*256] = {
    0x9f,0xbf,0xdf,0x00,
    0x9f,0xbf,0xde,0x00,
    0x9e,0xbe,0xde,0x00,
    0x9e,0xbe,0xdd,0x00,
    0x9e,0xbd,0xdd,0x00,
    0x9f,0xbe,0xdb,0x00,
    0x9f,0xbf,0xda,0x00,
    0x9f,0xbc,0xdb,0x00,
    0x9f,0xbf,0xd9,0x00,
    0x9f,0xbe,0xd9,0x00,
    0x9d,0xbb,0xdb,0x00,
    0x9f,0xbc,0xd9,0x00,
    0x9f,0xba,0xda,0x00,
    0x9c,0xbb,0xda,0x00,
    0x9d,0xbb,0xd9,0x00,
    0x9e,0xba,0xd9,0x00,
    0x9e,0xbb,0xd8,0x00,
    0x9c,0xba,0xd9,0x00,
    0x9a,0xba,0xda,0x00,
    0x9f,0xbb,0xd7,0x00,
    0x9f,0xbe,0xd6,0x00,
    0x9d,0xb9,0xd8,0x00,
    0x9f,0xbc,0xd6,0x00,
    0x9e,0xb8,0xd8,0x00,
    0x9d,0xb8,0xd8,0x00,
    0x9f,0xbf,0xd5,0x00,
    0x9e,0xba,0xd6,0x00,
    0x9d,0xba,0xd6,0x00,
    0x9c,0xba,0xd6,0x00,
    0x9a,0xb9,0xd7,0x00,
    0x9b,0xb8,0xd7,0x00,
    0x9f,0xba,0xd5,0x00,
    0x9d,0xb8,0xd6,0x00,
    0x9f,0xbe,0xd4,0x00,
    0x9f,0xb7,0xd6,0x00,
    0x9f,0xbc,0xd4,0x00,
    0x9c,0xb9,0xd5,0x00,
    0x98,0xb8,0xd7,0x00,
    0x9d,0xbb,0xd4,0x00,
    0x9e,0xba,0xd4,0x00,
    0x9a,0xb7,0xd6,0x00,
    0x9c,0xba,0xd4,0x00,
    0x9f,0xbe,0xd3,0x00,
    0x9f,0xbd,0xd3,0x00,
    0x9f,0xbc,0xd3,0x00,
    0x9e,0xb6,0xd5,0x00,
    0x9c,0xb8,0xd4,0x00,
    0x9c,0xb6,0xd5,0x00,
    0x99,0xb7,0xd5,0x00,
    0x9d,0xba,0xd3,0x00,
    0x9c,0xba,0xd3,0x00,
    0x9b,0xb7,0xd4,0x00,
    0x9e,0xb6,0xd4,0x00,
    0x9d,0xb6,0xd4,0x00,
    0x9d,0xb8,0xd3,0x00,
    0x9e,0xbd,0xd2,0x00,
    0x9a,0xb5,0xd5,0x00,
    0x9d,0xbc,0xd2,0x00,
    0x98,0xb7,0xd4,0x00,
    0x97,0xb6,0xd5,0x00,
    0x9c,0xb5,0xd4,0x00,
    0x9b,0xbb,0xd2,0x00,
    0x9a,0xb7,0xd3,0x00,
    0x9a,0xb5,0xd4,0x00,
    0x99,0xb7,0xd3,0x00,
    0x9e,0xb4,0xd4,0x00,
    0x9d,0xb4,0xd4,0x00,
    0x9f,0xbf,0xd1,0x00,
    0x9b,0xb8,0xd2,0x00,
    0x9d,0xb7,0xd2,0x00,
    0x9c,0xb7,0xd2,0x00,
    0x9f,0xbb,0xd1,0x00,
    0x99,0xb8,0xd2,0x00,
    0x9f,0xba,0xd1,0x00,
    0x9c,0xbb,0xd1,0x00,
    0x9d,0xb4,0xd3,0x00,
    0x9e,0xb9,0xd1,0x00,
    0x9b,0xba,0xd1,0x00,
    0x9c,0xb9,0xd1,0x00,
    0x9a,0xba,0xd1,0x00,
    0x99,0xb6,0xd2,0x00,
    0x97,0xb5,0xd3,0x00,
    0x97,0xb7,0xd2,0x00,
    0x98,0xb6,0xd2,0x00,
    0x9d,0xb3,0xd3,0x00,
    0x9f,0xbf,0xd0,0x00,
    0x9b,0xb7,0xd1,0x00,
    0x9e,0xb6,0xd1,0x00,
    0x97,0xb4,0xd3,0x00,
    0x9c,0xb6,0xd1,0x00,
    0x99,0xb7,0xd1,0x00,
    0x9f,0xba,0xd0,0x00,
    0x9c,0xbb,0xd0,0x00,
    0x99,0xb4,0xd2,0x00,
    0x9e,0xb9,0xd0,0x00,
    0x9d,0xb9,0xd0,0x00,
    0x9c,0xb9,0xd0,0x00,
    0x9a,0xba,0xd0,0x00,
    0x9b,0xb3,0xd2,0x00,
    0x9c,0xb8,0xd0,0x00,
    0x9e,0xb4,0xd1,0x00,
    0x97,0xb6,0xd1,0x00,
    0x99,0xb3,0xd2,0x00,
    0x98,0xb5,0xd1,0x00,
    0x96,0xb4,0xd2,0x00,
    0x9e,0xb6,0xd0,0x00,
    0x9e,0xb2,0xd2,0x00,
    0x9c,0xb6,0xd0,0x00,
    0x99,0xb7,0xd0,0x00,
    0x9e,0xb3,0xd1,0x00,
    0x9f,0xb5,0xd0,0x00,
    0x9c,0xb3,0xd1,0x00,
    0x9b,0xb3,0xd1,0x00,
    0x96,0xb3,0xd2,0x00,
    0x97,0xb4,0xd1,0x00,
    0x94,0xb3,0xd3,0x00,
    0x9a,0xb5,0xd0,0x00,
    0x94,0xb4,0xd2,0x00,
    0x9e,0xb4,0xd0,0x00,
    0x97,0xb6,0xd0,0x00,
    0x97,0xb2,0xd2,0x00,
    0x98,0xb5,0xd0,0x00,
    0x9c,0xb2,0xd1,0x00,
    0x97,0xb3,0xd1,0x00,
    0x95,0xb4,0xd1,0x00,
    0x97,0xb5,0xd0,0x00,
    0x94,0xb3,0xd2,0x00,
    0x9e,0xb3,0xd0,0x00,
    0x9d,0xb3,0xd0,0x00,
    0x9c,0xb3,0xd0,0x00,
    0x95,0xb2,0xd2,0x00,
    0x94,0xb4,0xd1,0x00,
    0x97,0xb4,0xd0,0x00,
    0x95,0xb3,0xd1,0x00,
    0x9f,0xb1,0xd1,0x00,
    0x95,0xb5,0xd0,0x00,
    0x96,0xb4,0xd0,0x00,
    0x9c,0xb1,0xd1,0x00,
    0x9b,0xb1,0xd1,0x00,
    0x96,0xb2,0xd1,0x00,
    0x9a,0xb1,0xd1,0x00,
    0x97,0xb3,0xd0,0x00,
    0x95,0xb4,0xd0,0x00,
    0x9a,0xb2,0xd0,0x00,
    0x9a,0xb2,0xd0,0x00,
    0x98,0xb1,0xd1,0x00,
    0x99,0xb2,0xd0,0x00,
    0x93,0xb2,0xd2,0x00,
    0x97,0xb1,0xd1,0x00,
    0x98,0xb2,0xd0,0x00,
    0x93,0xb3,0xd1,0x00,
    0x95,0xb3,0xd0,0x00,
    0x9f,0xb1,0xd0,0x00,
    0x96,0xb1,0xd1,0x00,
    0x9d,0xb1,0xd0,0x00,
    0x9c,0xb1,0xd0,0x00,
    0x9b,0xb1,0xd0,0x00,
    0x96,0xb2,0xd0,0x00,
    0x9a,0xb1,0xd0,0x00,
    0x95,0xb1,0xd1,0x00,
    0x99,0xb1,0xd0,0x00,
    0x93,0xb2,0xd1,0x00,
    0x95,0xb2,0xd0,0x00,
    0x98,0xb1,0xd0,0x00,
    0x98,0xb1,0xd0,0x00,
    0x94,0xb1,0xd1,0x00,
    0x94,0xb1,0xd1,0x00,
    0x97,0xb1,0xd0,0x00,
    0x93,0xb3,0xd0,0x00,
    0x94,0xb2,0xd0,0x00,
    0x9f,0xb0,0xd0,0x00,
    0x96,0xb1,0xd0,0x00,
    0x92,0xb2,0xd1,0x00,
    0x9c,0xb0,0xd0,0x00,
    0x9b,0xb0,0xd0,0x00,
    0x93,0xb1,0xd1,0x00,
    0x9a,0xb0,0xd0,0x00,
    0x95,0xb1,0xd0,0x00,
    0x99,0xb0,0xd0,0x00,
    0x93,0xb2,0xd0,0x00,
    0x98,0xb0,0xd0,0x00,
    0x98,0xb0,0xd0,0x00,
    0x98,0xb0,0xd0,0x00,
    0x94,0xb1,0xd0,0x00,
    0x94,0xb1,0xd0,0x00,
    0x97,0xb0,0xd0,0x00,
    0x92,0xb1,0xd1,0x00,
    0x92,0xb1,0xd1,0x00,
    0x96,0xb0,0xd0,0x00,
    0x96,0xb0,0xd0,0x00,
    0x92,0xb2,0xd0,0x00,
    0x92,0xb2,0xd0,0x00,
    0x93,0xb1,0xd0,0x00,
    0x93,0xb1,0xd0,0x00,
    0x93,0xb1,0xd0,0x00,
    0x95,0xb0,0xd0,0x00,
    0x95,0xb0,0xd0,0x00,
    0x95,0xb0,0xd0,0x00,
    0x91,0xb1,0xd1,0x00,
    0x91,0xb1,0xd1,0x00,
    0x91,0xb1,0xd1,0x00,
    0x91,0xb1,0xd1,0x00,
    0x94,0xb0,0xd0,0x00,
    0x94,0xb0,0xd0,0x00,
    0x92,0xb1,0xd0,0x00,
    0x92,0xb1,0xd0,0x00,
    0x92,0xb1,0xd0,0x00,
    0x92,0xb1,0xd0,0x00,
    0x93,0xb0,0xd0,0x00,
    0x93,0xb0,0xd0,0x00,
    0x93,0xb0,0xd0,0x00,
    0x93,0xb0,0xd0,0x00,
    0x93,0xb0,0xd0,0x00,
    0x93,0xb0,0xd0,0x00,
    0x93,0xb0,0xd0,0x00,
    0x93,0xb0,0xd0,0x00,
    0x91,0xb1,0xd0,0x00,
    0x91,0xb1,0xd0,0x00,
    0x91,0xb1,0xd0,0x00,
    0x91,0xb1,0xd0,0x00,
    0x91,0xb1,0xd0,0x00,
    0x92,0xb0,0xd0,0x00,
    0x92,0xb0,0xd0,0x00,
    0x92,0xb0,0xd0,0x00,
    0x92,0xb0,0xd0,0x00,
    0x92,0xb0,0xd0,0x00,
    0x92,0xb0,0xd0,0x00,
    0x92,0xb0,0xd0,0x00,
    0x92,0xb0,0xd0,0x00,
    0x92,0xb0,0xd0,0x00,
    0x91,0xb0,0xd0,0x00,
    0x91,0xb0,0xd0,0x00,
    0x91,0xb0,0xd0,0x00,
    0x91,0xb0,0xd0,0x00,
    0x91,0xb0,0xd0,0x00,
    0x91,0xb0,0xd0,0x00,
    0x91,0xb0,0xd0,0x00,
    0x91,0xb0,0xd0,0x00,
    0x91,0xb0,0xd0,0x00,
    0x91,0xb0,0xd0,0x00,
    0x91,0xb0,0xd0,0x00,
    0x91,0xb0,0xd0,0x00,
    0x91,0xb0,0xd0,0x00,
    0x91,0xb0,0xd0,0x00,
    0x91,0xb0,0xd0,0x00,
    0x91,0xb0,0xd0,0x00,
    0x90,0xb0,0xd0,0x00,
    0x90,0xb0,0xd0,0x00,
    0x90,0xb0,0xd0,0x00,
    0x90,0xb0,0xd0,0x00,
    0x90,0xb0,0xd0,0x00,
    0x90,0xb0,0xd0,0x00,
    0x90,0xb0,0xd0,0x00,
    0x90,0xb0,0xd0,0x00,
    0x90,0xb0,0xd0,0x00,
    0x90,0xb0,0xd0,0x00,
};

    /* Values to be written to the SN76496 port to play samples.  This is a 
       "smoothed" table that will hopefully give less static during 
       playback.  It is a little less accurate than the table above. */
static char PortVals2[4*256] = {
    0x9f,0xbf,0xdf,0x00,
    0x9e,0xbf,0xdf,0x00,
    0x9e,0xbe,0xde,0x00,
    0x9d,0xbe,0xde,0x00,
    0x9d,0xbd,0xde,0x00,
    0x9d,0xbd,0xdd,0x00,
    0x9c,0xbd,0xdd,0x00,
    0x9c,0xbc,0xdd,0x00,
    0x9c,0xbc,0xdc,0x00,
    0x9c,0xbc,0xdc,0x00,
    0x9b,0xbc,0xdc,0x00,
    0x9b,0xbb,0xdc,0x00,
    0x9b,0xbb,0xdb,0x00,
    0x9b,0xbb,0xdb,0x00,
    0x9a,0xbb,0xdb,0x00,
    0x9a,0xbb,0xdb,0x00,
    0x9a,0xba,0xdb,0x00,
    0x9a,0xba,0xdb,0x00,
    0x9a,0xba,0xda,0x00,
    0x9a,0xba,0xda,0x00,
    0x99,0xba,0xda,0x00,
    0x99,0xba,0xda,0x00,
    0x99,0xb9,0xda,0x00,
    0x99,0xb9,0xda,0x00,
    0x99,0xb9,0xd9,0x00,
    0x99,0xb9,0xd9,0x00,
    0x99,0xb9,0xd9,0x00,
    0x98,0xb9,0xd9,0x00,
    0x98,0xb9,0xd9,0x00,
    0x98,0xb9,0xd9,0x00,
    0x98,0xb8,0xd9,0x00,
    0x98,0xb8,0xd9,0x00,
    0x98,0xb8,0xd8,0x00,
    0x98,0xb8,0xd8,0x00,
    0x98,0xb8,0xd8,0x00,
    0x98,0xb8,0xd8,0x00,
    0x97,0xb8,0xd8,0x00,
    0x97,0xb8,0xd8,0x00,
    0x97,0xb8,0xd8,0x00,
    0x97,0xb7,0xd8,0x00,
    0x97,0xb7,0xd8,0x00,
    0x97,0xb7,0xd8,0x00,
    0x97,0xb7,0xd8,0x00,
    0x97,0xb7,0xd7,0x00,
    0x97,0xb7,0xd7,0x00,
    0x97,0xb7,0xd7,0x00,
    0x97,0xb7,0xd7,0x00,
    0x96,0xb7,0xd7,0x00,
    0x96,0xb7,0xd7,0x00,
    0x96,0xb7,0xd7,0x00,
    0x96,0xb7,0xd7,0x00,
    0x96,0xb7,0xd7,0x00,
    0x96,0xb6,0xd7,0x00,
    0x96,0xb6,0xd7,0x00,
    0x96,0xb6,0xd7,0x00,
    0x96,0xb6,0xd7,0x00,
    0x96,0xb6,0xd6,0x00,
    0x96,0xb6,0xd6,0x00,
    0x96,0xb6,0xd6,0x00,
    0x96,0xb6,0xd6,0x00,
    0x96,0xb6,0xd6,0x00,
    0x95,0xb6,0xd6,0x00,
    0x95,0xb6,0xd6,0x00,
    0x95,0xb6,0xd6,0x00,
    0x95,0xb6,0xd6,0x00,
    0x95,0xb6,0xd6,0x00,
    0x95,0xb6,0xd6,0x00,
    0x95,0xb5,0xd6,0x00,
    0x95,0xb5,0xd6,0x00,
    0x95,0xb5,0xd6,0x00,
    0x95,0xb5,0xd6,0x00,
    0x95,0xb5,0xd6,0x00,
    0x95,0xb5,0xd6,0x00,
    0x95,0xb5,0xd5,0x00,
    0x95,0xb5,0xd5,0x00,
    0x95,0xb5,0xd5,0x00,
    0x95,0xb5,0xd5,0x00,
    0x95,0xb5,0xd5,0x00,
    0x95,0xb5,0xd5,0x00,
    0x94,0xb5,0xd5,0x00,
    0x94,0xb5,0xd5,0x00,
    0x94,0xb5,0xd5,0x00,
    0x94,0xb5,0xd5,0x00,
    0x94,0xb5,0xd5,0x00,
    0x94,0xb5,0xd5,0x00,
    0x94,0xb5,0xd5,0x00,
    0x94,0xb4,0xd5,0x00,
    0x94,0xb4,0xd5,0x00,
    0x94,0xb4,0xd5,0x00,
    0x94,0xb4,0xd5,0x00,
    0x94,0xb4,0xd5,0x00,
    0x94,0xb4,0xd5,0x00,
    0x94,0xb4,0xd5,0x00,
    0x94,0xb4,0xd4,0x00,
    0x94,0xb4,0xd4,0x00,
    0x94,0xb4,0xd4,0x00,
    0x94,0xb4,0xd4,0x00,
    0x94,0xb4,0xd4,0x00,
    0x94,0xb4,0xd4,0x00,
    0x94,0xb4,0xd4,0x00,
    0x94,0xb4,0xd4,0x00,
    0x94,0xb4,0xd4,0x00,
    0x93,0xb4,0xd4,0x00,
    0x93,0xb4,0xd4,0x00,
    0x93,0xb4,0xd4,0x00,
    0x93,0xb4,0xd4,0x00,
    0x93,0xb4,0xd4,0x00,
    0x93,0xb4,0xd4,0x00,
    0x93,0xb4,0xd4,0x00,
    0x93,0xb4,0xd4,0x00,
    0x93,0xb4,0xd4,0x00,
    0x93,0xb3,0xd4,0x00,
    0x93,0xb3,0xd4,0x00,
    0x93,0xb3,0xd4,0x00,
    0x93,0xb3,0xd4,0x00,
    0x93,0xb3,0xd4,0x00,
    0x93,0xb3,0xd4,0x00,
    0x93,0xb3,0xd4,0x00,
    0x93,0xb3,0xd4,0x00,
    0x93,0xb3,0xd4,0x00,
    0x93,0xb3,0xd3,0x00,
    0x93,0xb3,0xd3,0x00,
    0x93,0xb3,0xd3,0x00,
    0x93,0xb3,0xd3,0x00,
    0x93,0xb3,0xd3,0x00,
    0x93,0xb3,0xd3,0x00,
    0x93,0xb3,0xd3,0x00,
    0x93,0xb3,0xd3,0x00,
    0x93,0xb3,0xd3,0x00,
    0x93,0xb3,0xd3,0x00,
    0x92,0xb3,0xd3,0x00,
    0x92,0xb3,0xd3,0x00,
    0x92,0xb3,0xd3,0x00,
    0x92,0xb3,0xd3,0x00,
    0x92,0xb3,0xd3,0x00,
    0x92,0xb3,0xd3,0x00,
    0x92,0xb3,0xd3,0x00,
    0x92,0xb3,0xd3,0x00,
    0x92,0xb3,0xd3,0x00,
    0x92,0xb3,0xd3,0x00,
    0x92,0xb3,0xd3,0x00,
    0x92,0xb2,0xd3,0x00,
    0x92,0xb2,0xd3,0x00,
    0x92,0xb2,0xd3,0x00,
    0x92,0xb2,0xd3,0x00,
    0x92,0xb2,0xd3,0x00,
    0x92,0xb2,0xd3,0x00,
    0x92,0xb2,0xd3,0x00,
    0x92,0xb2,0xd3,0x00,
    0x92,0xb2,0xd3,0x00,
    0x92,0xb2,0xd3,0x00,
    0x92,0xb2,0xd3,0x00,
    0x92,0xb2,0xd3,0x00,
    0x92,0xb2,0xd2,0x00,
    0x92,0xb2,0xd2,0x00,
    0x92,0xb2,0xd2,0x00,
    0x92,0xb2,0xd2,0x00,
    0x92,0xb2,0xd2,0x00,
    0x92,0xb2,0xd2,0x00,
    0x92,0xb2,0xd2,0x00,
    0x92,0xb2,0xd2,0x00,
    0x92,0xb2,0xd2,0x00,
    0x92,0xb2,0xd2,0x00,
    0x92,0xb2,0xd2,0x00,
    0x92,0xb2,0xd2,0x00,
    0x91,0xb2,0xd2,0x00,
    0x91,0xb2,0xd2,0x00,
    0x91,0xb2,0xd2,0x00,
    0x91,0xb2,0xd2,0x00,
    0x91,0xb2,0xd2,0x00,
    0x91,0xb2,0xd2,0x00,
    0x91,0xb2,0xd2,0x00,
    0x91,0xb2,0xd2,0x00,
    0x91,0xb2,0xd2,0x00,
    0x91,0xb2,0xd2,0x00,
    0x91,0xb2,0xd2,0x00,
    0x91,0xb2,0xd2,0x00,
    0x91,0xb2,0xd2,0x00,
    0x91,0xb2,0xd2,0x00,
    0x91,0xb2,0xd2,0x00,
    0x91,0xb1,0xd2,0x00,
    0x91,0xb1,0xd2,0x00,
    0x91,0xb1,0xd2,0x00,
    0x91,0xb1,0xd2,0x00,
    0x91,0xb1,0xd2,0x00,
    0x91,0xb1,0xd2,0x00,
    0x91,0xb1,0xd2,0x00,
    0x91,0xb1,0xd2,0x00,
    0x91,0xb1,0xd2,0x00,
    0x91,0xb1,0xd2,0x00,
    0x91,0xb1,0xd2,0x00,
    0x91,0xb1,0xd2,0x00,
    0x91,0xb1,0xd2,0x00,
    0x91,0xb1,0xd2,0x00,
    0x91,0xb1,0xd1,0x00,
    0x91,0xb1,0xd1,0x00,
    0x91,0xb1,0xd1,0x00,
    0x91,0xb1,0xd1,0x00,
    0x91,0xb1,0xd1,0x00,
    0x91,0xb1,0xd1,0x00,
    0x91,0xb1,0xd1,0x00,
    0x91,0xb1,0xd1,0x00,
    0x91,0xb1,0xd1,0x00,
    0x91,0xb1,0xd1,0x00,
    0x91,0xb1,0xd1,0x00,
    0x91,0xb1,0xd1,0x00,
    0x91,0xb1,0xd1,0x00,
    0x91,0xb1,0xd1,0x00,
    0x91,0xb1,0xd1,0x00,
    0x91,0xb1,0xd1,0x00,
    0x90,0xb1,0xd1,0x00,
    0x90,0xb1,0xd1,0x00,
    0x90,0xb1,0xd1,0x00,
    0x90,0xb1,0xd1,0x00,
    0x90,0xb1,0xd1,0x00,
    0x90,0xb1,0xd1,0x00,
    0x90,0xb1,0xd1,0x00,
    0x90,0xb1,0xd1,0x00,
    0x90,0xb1,0xd1,0x00,
    0x90,0xb1,0xd1,0x00,
    0x90,0xb1,0xd1,0x00,
    0x90,0xb1,0xd1,0x00,
    0x90,0xb1,0xd1,0x00,
    0x90,0xb1,0xd1,0x00,
    0x90,0xb1,0xd1,0x00,
    0x90,0xb1,0xd1,0x00,
    0x90,0xb1,0xd1,0x00,
    0x90,0xb1,0xd1,0x00,
    0x90,0xb0,0xd1,0x00,
    0x90,0xb0,0xd1,0x00,
    0x90,0xb0,0xd1,0x00,
    0x90,0xb0,0xd1,0x00,
    0x90,0xb0,0xd1,0x00,
    0x90,0xb0,0xd1,0x00,
    0x90,0xb0,0xd1,0x00,
    0x90,0xb0,0xd1,0x00,
    0x90,0xb0,0xd1,0x00,
    0x90,0xb0,0xd1,0x00,
    0x90,0xb0,0xd1,0x00,
    0x90,0xb0,0xd1,0x00,
    0x90,0xb0,0xd1,0x00,
    0x90,0xb0,0xd1,0x00,
    0x90,0xb0,0xd1,0x00,
    0x90,0xb0,0xd1,0x00,
    0x90,0xb0,0xd1,0x00,
    0x90,0xb0,0xd1,0x00,
    0x90,0xb0,0xd0,0x00,
    0x90,0xb0,0xd0,0x00,
    0x90,0xb0,0xd0,0x00,
    0x90,0xb0,0xd0,0x00,
    0x90,0xb0,0xd0,0x00,
    0x90,0xb0,0xd0,0x00,
    0x90,0xb0,0xd0,0x00,
    0x90,0xb0,0xd0,0x00,
    0x90,0xb0,0xd0,0x00,
};

    /* List of available tables of I/O port values. */
static char *PortTableList[MAX_TRANSLATE+1] = {
    PortVals,
    PortVals2,
};

    /* Selected I/O port value table. */
char *PortTable = PortVals;


/*************************************************************************/
/*                                                                       */
/* Function prototypes.                                                  */
/*                                                                       */
/*************************************************************************/



/*************************************************************************/
/*                                                                       */
/* Detect_3_Voice() function.  This function detects the 3-voice chip    */
/* and determines the I/O port for it.  If the chip is present, 0 is     */
/* returned; otherwise, -1 is returned.                                  */
/*                                                                       */
/*************************************************************************/

int Detect_3_Voice( void )
{
    int retcod;     /* return code for caller */

    /* Start out pessimistic. */
    retcod = -1;

    /* Detect the chip. */
    _asm {
        push es
        ;
        ; Detect PCMCIA Socket Services.  If present, skip detecting the
        ; DAC.  Note:  This sets the sound multiplexer on the PCjr to the
        ; system beep (8253 channel 2) as a side effect.
        ;
        xor cx,cx
        mov ax,8000h
        int 1ah
        cmp cx,5353h
        je detect_nodac
        ;
        ; Detect the Tandy DAC (can't do this if PCMCIA Socket Services is
        ; present).
        ;
        mov ax,8100h
        int 1ah
        cmp ax,8000h
        jae detect_nodac
        ;
        ; Tandy DAC is present.  The SN76496 is located 4 ports below the
        ; the base I/O port of the DAC.
        ;
        sub ax,4
        mov word ptr IOport,ax
        mov word ptr retcod,0
        jmp detect_done
        ;
        ; Tandy DAC is not present.  Check for a PCjr.
        ;
    detect_nodac:
        mov ax,0ffffh
        mov es,ax
        cmp byte ptr es:[0eh],0fdh
        jne detect_nopcjr
        ;
        ; The PCjr has the 3-voice chip at port 0C0h.
        ;
        mov word ptr IOport,0c0h
        mov word ptr retcod,0
        jmp detect_done
        ;
        ; There's no Tandy DAC, and it's not a PCjr.  Check for a Tandy
        ; 1000-series.
        ;
    detect_nopcjr:
        cmp byte ptr es:[0eh],0ffh
        jne detect_done
        mov ax,0fc00h
        mov es,ax
        cmp byte ptr es:[0],21h
        jne detect_done
        ;
        ; Older Tandy 1000-series without DAC.  The 3-voice chip is at
        ; 0C0h, as it is on the PCjr.
        ;
        mov word ptr IOport,0c0h
        mov word ptr retcod,0
    detect_done:
        pop es
    }

    /* Return result. */
    return( retcod );
}


/*************************************************************************/
/*                                                                       */
/* Play_Sound() function.  This function plays the sound in buffer       */
/* SoundBuf, whose length is given by SoundLen, at SampRate Hertz, using */
/* amplification factor Amplification.  It returns nothing.              */
/*     This function assumes the small memory model (so, for example,    */
/* SS = DS).                                                             */
/*                                                                       */
/*************************************************************************/

void Play_Sound( void )
{
    /* Set amplification table pointer. */
    AmplifyTable = AmpTableList[Amplification];

    /* Set the I/O port translation table pointer. */
    PortTable = PortTableList[Translation];

    /* Do the rest in assembler. */
    Play_Internal();
}
