/*----------------------------------------------------------------------------
	Set_up_sound -	Makes the calls necessary to start the Sound Toolkit
			2-Jul-94

			This module is used in most of the Sound Toolkit
			demonstration programs.  It is responsible for
			allocating memory that will be used and informing the
			sound toolkit of the areas that were allocated.  It
			also displays the version information provided by the
			sound toolkit (a totally optional operation) and makes
			the calls that initialize the sound toolkit functions.

			The parameters this routine accepts are:

			nobufs -	number of buffers to allocate that
					will be used by the sound toolkit.
					The minimum value of buffers (nobufs)
					for non-file play operation is 5.
					For file-play or file-record operation,
					at least 20 are needed.  The more
					buffers, the lower the chance of
					under- or over-run.  The buffers are
					allocated using far malloc.

			trigger -	number of buffers to fill before
					starting playback (0 causes all
					buffers to be filled before playback
					starts.)

			dma_channel -	specifies which DMA channel to use.
					(u_short) 0 causes the default to be
					used.

			irq -		specified which interrupt channel
					(IRQ) to use. (u_short) 0 causes the
					default to be used.
		
			options -	is used for the various option flags
					that snd_init() accepts.  See the
					snd_init() description for
					information.

	DSETUP is part of the PSSJ Digital Sound Toolkit.
	Copyright 1994 Frank Durda IV, all rights reserved.
	Commercial use is restricted.  See intro(PSSJ) for more information.
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <malloc.h>
#include <stdlib.h>
#include "sound.h"
#include "dsetup.h"

#ifdef __STDC__
void far set_up_sound(u_short nobufs, u_short trigger, u_short dma_channel, 
	u_short irq, u_short options)
#else
void far set_up_sound(nobufs, trigger, dma_channel, irq, options)
u_short nobufs, trigger, dma_channel, irq, options;
#endif
{
	int	bufno;			/*buffers allocated counter */
	int	len;			/*appropriate size for buffers */
	char far *cp;			/*pointer to last allocated buffer */
	int	ret;			/*snd_init return code */

/*	Print the version number */

	printf("Sound Toolkit ");
	for (cp = snd_version()->version; *cp; putchar(*cp++));
	printf("\n");

/*	This statement determines what size buffers this version of
	the sound toolkit requires.  Avoid hardcoding this test.
*/

	len = snd_addbuf((SOUND far *) 0, (short) 0);	

/*	loop for however many buffers are needed */

	for (bufno = 0; bufno < nobufs; bufno++) {
		cp = _fmalloc(len);	/*Allocate a buffer in far data space*/
		if (cp == NULL)		/*Was the allocation successful?*/
		{
			fprintf(stderr, "Cannot allocate sound buffers\n");
			exit (1);
		}
		snd_addbuf(cp,(short)1);/*pass one buffer to snd toolkit*/
	}				/* loop for the next buffer */

/*	The specified number of buffers have been successfully allocated.
	Now the sound toolkit is initialized by calling snd_init().
*/
	ret = snd_init(trigger, dma_channel, irq, options);
	switch(ret) {
		case NOERROR:		/*No Errors*/
			return;
		case INITBUF:		/*Insufficient buffers*/
			printf("snd_init:  Insufficient buffers\n");
			break;
		case INIT64K:		/*All buffers span 64k boundary*/
			printf("snd_init:  all buffers span 64k boundary\n");
			break;
		case BADDMA:		/*DMA channel requested is invalid*/
			printf("snd_init: DMA channel requested is invalid\n");
			break;
		case BADIRQ:		/*IRQ number requested is invalid*/
			printf("snd_init: IRQ number requested is invalid\n");
			break;
		case NOIRON:		/*Can't find sound hardware*/
			printf("snd_init: Cannot find sound hardware\n");
			break;
		default:		/*unknown error code*/
			printf("snd_init: unknown error code %d\n", ret);
			break;
	}
	exit(1);
}

