/*
 * linux/drivers/video/geodefb.c -- Geode frame buffer device driver
 *
 * This file is based on the S3 Virge frame buffer device (virgefb.c):
 *
 *    Copyright (C) 2000
 *    National semiconductor corporation
 *
 * This file is based on the CyberVision frame buffer device (cyberfb.c):
 *
 *    Copyright (C) 1996 Martin Apel
 *                       Geert Uytterhoeven
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file COPYING in the main directory of this archive
 * for more details.
 */

#define GEODEFBDEBUG

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/tty.h>
#include <linux/malloc.h>
#include <linux/delay.h>
#include <linux/fb.h>
#include <linux/console.h>
#include <linux/selection.h>
#include <linux/ioport.h>
#include <linux/init.h>
#include <linux/devfs_fs_kernel.h>

#include <asm/io.h>
#include <asm/mtrr.h>

#include <video/fbcon.h>
#include <video/fbcon-cfb8.h>
#include <video/fbcon-cfb16.h>

#include "geodeioctl.h"

#include "gfx/gfx_rtns.h"
#include "gfx/gfx_defs.h"
#include "gfx/gfx_regs.h"

#include "/usr/src/nsm/galapi.h"

#ifdef GEODEFBDEBUG
#define DPRINTK(fmt, args...) printk(KERN_INFO ## fmt, args)
#else
#define DPRINTK(fmt, args...)
#endif

#define arraysize(x)    (sizeof(x)/sizeof(*(x)))

/*
 *	may be needed when we initialize the board?
 *	8bit: flag = 2, 16 bit: flag = 1, 24/32bit: flag = 0 
 *	_when_ the board is initialized, depth doesnt matter, we allways write
 *	to the same address, aperture seems not to matter on Z2.
 */

struct geodefb_par {
    int xres;
    int yres;
    int bpp;
    int Refresh;
    int accel;
};

static struct geodefb_par current_par;

static int current_par_valid = 0;
static int currcon = 0;

static struct display disp;
static struct fb_info fb_info;

static union {
#ifdef FBCON_HAS_CFB16
    u16 cfb16[16];
#endif
} fbcon_cmap;

/*
 *    Switch for Chipset Independency
 */

static struct fb_hwswitch {

      /* Initialisation */

    int (*init)(void);

      /* Display Control */

    int (*encode_fix)(struct fb_fix_screeninfo *fix, struct geodefb_par *par);
    int (*decode_var)(struct fb_var_screeninfo *var, struct geodefb_par *par);
    int (*encode_var)(struct fb_var_screeninfo *var, struct geodefb_par *par);
    int (*getcolreg)(u_int regno, u_int *red, u_int *green, u_int *blue,
                     u_int *transp, struct fb_info *info);
    int (*setcolreg)(u_int regno, u_int red, u_int green, u_int blue,
                     u_int transp, struct fb_info *info);
    void (*blank)(int blank);
} *fbhw;

static int blit_maybe_busy = 0;

/*
 *    Frame Buffer Name
 */

static char geodefb_name[16] = "Geode VGA";


/*
 *    Geodevision Graphics Board
 */

#define GEODE8_WIDTH 1152
#define GEODE8_HEIGHT 886
#define GEODE8_PIXCLOCK 12500    /* ++Geert: Just a guess */

#if 1
#define GEODE16_WIDTH 800
#define GEODE16_HEIGHT 600
#endif
#define GEODE16_PIXCLOCK 25000   /* ++Geert: Just a guess */


static struct
{
    unsigned short red;
    unsigned short green;
    unsigned short blue;
} Geode_palette [256];
static unsigned long GeodeMem_phys;
static unsigned long GeodeRegs_phys;
static unsigned long GeodeRegs_phys_size;
static unsigned long GeodeMem_virt;
static unsigned long GeodeSize;
	
#define CYBMEM_OFFSET_8  0x800000	/* offsets from start of video - */ 
#define CYBMEM_OFFSET_16 0x400000	/* ram to appropriate aperture */

#define FB_ACCEL_NSC_GEODE   40

#define dac_reg	(0x3c8)
#define dac_val	(0x3c9)


/*
 *    Predefined Video Modes
 */

static struct {
    const char *name;
    struct fb_var_screeninfo var;
} geodefb_predefined[] __initdata = {
    {
        "640x480-8", {		/* Geodevision 8 bpp */
            640, 480, 640, 480, 0, 0, 8, 0,
            {0, 8, 0}, {0, 8, 0}, {0, 8, 0}, {0, 0, 0},
            0, 0, -1, -1, FB_ACCELF_TEXT, GEODE8_PIXCLOCK, 64, 96, 35, 12, 112, 2,
            FB_SYNC_COMP_HIGH_ACT|FB_SYNC_VERT_HIGH_ACT, FB_VMODE_NONINTERLACED
        }
    }, {
        "800x600-8", {		/* Geodevision 8 bpp */
            800, 600, 800, 600, 0, 0, 8, 0,
            {0, 8, 0}, {0, 8, 0}, {0, 8, 0}, {0, 0, 0},
            0, 0, -1, -1, FB_ACCELF_TEXT, GEODE8_PIXCLOCK, 64, 96, 35, 12, 112, 2,
            FB_SYNC_COMP_HIGH_ACT|FB_SYNC_VERT_HIGH_ACT, FB_VMODE_NONINTERLACED
        }
    }, {
        "1024x768-8", {		/* Geodevision 8 bpp */
            1024, 768, 1024, 768, 0, 0, 8, 0,
            {0, 8, 0}, {0, 8, 0}, {0, 8, 0}, {0, 0, 0},
            0, 0, -1, -1, FB_ACCELF_TEXT, GEODE8_PIXCLOCK, 64, 96, 35, 12, 112, 2,
            FB_SYNC_COMP_HIGH_ACT|FB_SYNC_VERT_HIGH_ACT, FB_VMODE_NONINTERLACED
        }
    }, {
        "1152x886-8", {		/* Geodevision 8 bpp */
            1152, 886, 1152, 886, 0, 0, 8, 0,
            {0, 8, 0}, {0, 8, 0}, {0, 8, 0}, {0, 0, 0},
            0, 0, -1, -1, FB_ACCELF_TEXT, GEODE8_PIXCLOCK, 64, 96, 35, 12, 112, 2,
            FB_SYNC_COMP_HIGH_ACT|FB_SYNC_VERT_HIGH_ACT, FB_VMODE_NONINTERLACED
        }
    }, {
        "1280x1024-8", {	/* Geodevision 8 bpp */
            1280, 1024, 1280, 1024, 0, 0, 8, 0,
            {0, 8, 0}, {0, 8, 0}, {0, 8, 0}, {0, 0, 0},
            0, 0, -1, -1, FB_ACCELF_TEXT, GEODE8_PIXCLOCK, 64, 96, 35, 12, 112, 2,
            FB_SYNC_COMP_HIGH_ACT|FB_SYNC_VERT_HIGH_ACT, FB_VMODE_NONINTERLACED
        }
    }, {
        "1600x1200-8", {	/* Geodevision 8 bpp */
            1600, 1200, 1600, 1200, 0, 0, 8, 0,
            {0, 8, 0}, {0, 8, 0}, {0, 8, 0}, {0, 0, 0},
            0, 0, -1, -1, FB_ACCELF_TEXT, GEODE8_PIXCLOCK, 64, 96, 35, 12, 112, 2,
            FB_SYNC_COMP_HIGH_ACT|FB_SYNC_VERT_HIGH_ACT, FB_VMODE_NONINTERLACED
        }
    }, {
        "640x480-16", {		/* Geodevision 16 bpp */
            640, 480, 640, 480, 0, 0, 16, 0,
            {11, 5, 0}, {5, 6, 0}, {0, 5, 0}, {0, 0, 0},
            0, 0, -1, -1, FB_ACCELF_TEXT, GEODE16_PIXCLOCK, 64, 96, 35, 12, 112, 2,
            FB_SYNC_COMP_HIGH_ACT|FB_SYNC_VERT_HIGH_ACT, FB_VMODE_NONINTERLACED
        }
    }, {
        "800x600-16", {		/* Geodevision 16 bpp */
            800, 600, 800, 600, 0, 0, 16, 0,
            {11, 5, 0}, {5, 6, 0}, {0, 5, 0}, {0, 0, 0},
            0, 0, -1, -1, FB_ACCELF_TEXT, GEODE16_PIXCLOCK, 64, 96, 35, 12, 112, 2,
            FB_SYNC_COMP_HIGH_ACT|FB_SYNC_VERT_HIGH_ACT, FB_VMODE_NONINTERLACED
        }
    }, {
        "1024x768-16", {         /* Geodevision 16 bpp */
            1024, 768, 1024, 768, 0, 0, 16, 0,
            {11, 5, 0}, {5, 6, 0}, {0, 5, 0}, {0, 0, 0},
            0, 0, -1, -1, FB_ACCELF_TEXT, GEODE16_PIXCLOCK, 64, 96, 35, 12, 112, 2,
            FB_SYNC_COMP_HIGH_ACT|FB_SYNC_VERT_HIGH_ACT, FB_VMODE_NONINTERLACED
        }
    }, {
        "1152x886-16", {         /* Geodevision 16 bpp */
            1152, 886, 1152, 886, 0, 0, 16, 0,
            {11, 5, 0}, {5, 6, 0}, {0, 5, 0}, {0, 0, 0},
            0, 0, -1, -1, FB_ACCELF_TEXT, GEODE16_PIXCLOCK, 64, 96, 35, 12, 112, 2,
            FB_SYNC_COMP_HIGH_ACT|FB_SYNC_VERT_HIGH_ACT, FB_VMODE_NONINTERLACED
        }
    }, {
        "1280x1024-16", {         /* Geodevision 16 bpp */
            1280, 1024, 1280, 1024, 0, 0, 16, 0,
            {11, 5, 0}, {5, 6, 0}, {0, 5, 0}, {0, 0, 0},
            0, 0, -1, -1, FB_ACCELF_TEXT, GEODE16_PIXCLOCK, 64, 96, 35, 12, 112, 2,
            FB_SYNC_COMP_HIGH_ACT|FB_SYNC_VERT_HIGH_ACT, FB_VMODE_NONINTERLACED
        }
    }, {
        "1600x1200-16", {         /* Geodevision 16 bpp */
            1600, 1200, 1600, 1200, 0, 0, 16, 0,
            {11, 5, 0}, {5, 6, 0}, {0, 5, 0}, {0, 0, 0},
            0, 0, -1, -1, FB_ACCELF_TEXT, GEODE16_PIXCLOCK, 64, 96, 35, 12, 112, 2,
            FB_SYNC_COMP_HIGH_ACT|FB_SYNC_VERT_HIGH_ACT, FB_VMODE_NONINTERLACED
        }
    }
};


#define NUM_TOTAL_MODES    arraysize(geodefb_predefined)


static int Geodefb_inverse = 0;

/*
 *    Some default modes
 */

#define GEODE8_DEFMODE     (1)
#define GEODE16_DEFMODE    (7)

static struct fb_var_screeninfo geodefb_default;


/*
 *    Interface used by the world
 */

int geodefb_setup(char*);

static int geodefb_open(struct fb_info *info, int user);
static int geodefb_release(struct fb_info *info, int user);
static int geodefb_get_fix(struct fb_fix_screeninfo *fix, int con, struct
                           fb_info *info);
static int geodefb_get_var(struct fb_var_screeninfo *var, int con, struct
                           fb_info *info);
static int geodefb_set_var(struct fb_var_screeninfo *var, int con, struct
                           fb_info *info);
static int geodefb_get_cmap(struct fb_cmap *cmap, int kspc, int con,
                            struct fb_info *info);
static int geodefb_set_cmap(struct fb_cmap *cmap, int kspc, int con,
                            struct fb_info *info);
static int geodefb_pan_display(struct fb_var_screeninfo *var, int con,
                               struct fb_info *info);
static int geodefb_ioctl(struct inode *inode, struct file *file, u_int cmd,
                         u_long arg, int con, struct fb_info *info);


/*
 *    Interface to the low level console driver
 */

int geodefb_init(void);
static int Geodefb_switch(int con, struct fb_info *info);
static int Geodefb_updatevar(int con, struct fb_info *info);
static void Geodefb_blank(int blank, struct fb_info *info);


/*
 *    Text console acceleration
 */

#ifdef FBCON_HAS_CFB8
static struct display_switch fbcon_geode8;
#endif

#ifdef FBCON_HAS_CFB16
static struct display_switch fbcon_geode16;
#endif

/*
 *   Hardware Specific Routines
 */

static int Geode_init(void);
static int Geode_encode_fix(struct fb_fix_screeninfo *fix,
                            struct geodefb_par *par);
static int Geode_decode_var(struct fb_var_screeninfo *var,
                            struct geodefb_par *par);
static int Geode_encode_var(struct fb_var_screeninfo *var,
                            struct geodefb_par *par);
static int Geode_getcolreg(u_int regno, u_int *red, u_int *green, u_int *blue,
                           u_int *transp, struct fb_info *info);
static int Geode_setcolreg(u_int regno, u_int red, u_int green, u_int blue,
                           u_int transp, struct fb_info *info);
static void Geode_blank(int blank);
static void Geode_WaitBusy(void);

/*
 *    Internal routines
 */

static void geodefb_get_par(struct geodefb_par *par);
static void geodefb_set_par(struct geodefb_par *par);
static int do_fb_set_var(struct fb_var_screeninfo *var, int isactive);
static void do_install_cmap(int con, struct fb_info *info);
static void geodefb_set_disp(int con, struct fb_info *info);
static int get_video_mode(const char *name);


/* -------------------- Hardware specific routines ------------------------- */


static void Geode_WaitBusy(void)
{
    gfx_wait_until_idle();
    blit_maybe_busy = 0;
}

/*
 *    Initialization
 *
 *    Set the default video mode for this chipset. If a video mode was
 *    specified on the command line, it will override the default mode.
 */

static int Geode_init(void)
{
    int i;

    for (i = 0; i < 256; i++)
    {
        Geode_palette[i].red = i;
        Geode_palette[i].green = i;
        Geode_palette[i].blue = i;
    }

      /*
       * Just clear the thing for the biggest mode.
       *
       * ++Andre, TODO: determine size first, then clear all memory
       *                (the 3D penguin might need texture memory :-) )
       */

    memset ((char*)GeodeMem_virt, 0, GeodeSize);
    
#if 0
      /* Disable hardware cursor */
    vgawb_3d(0x3c8, 255);
    vgawb_3d(0x3c9, 56);
    vgawb_3d(0x3c9, 100);
    vgawb_3d(0x3c9, 160);

    vgawb_3d(0x3c8, 254);
    vgawb_3d(0x3c9, 0);
    vgawb_3d(0x3c9, 0);
    vgawb_3d(0x3c9, 0);

      /* Disable hardware cursor */
    vgawb_3d(S3_CRTC_ADR, S3_REG_LOCK2);
    vgawb_3d(S3_CRTC_DATA, 0xa0);
    vgawb_3d(S3_CRTC_ADR, S3_HGC_MODE);
    vgawb_3d(S3_CRTC_DATA, 0x00);
    vgawb_3d(S3_CRTC_ADR, S3_HWGC_DX);
    vgawb_3d(S3_CRTC_DATA, 0x00);
    vgawb_3d(S3_CRTC_ADR, S3_HWGC_DY);
    vgawb_3d(S3_CRTC_DATA, 0x00);
#endif

    return 0; /* TODO: hardware cursor for CV64/3D */
}


/*
 *    This function should fill in the `fix' structure based on the
 *    values in the `par' structure.
 */

static int Geode_encode_fix(struct fb_fix_screeninfo *fix,
                            struct geodefb_par *par)
{
    memset(fix, 0, sizeof(struct fb_fix_screeninfo));
    strcpy(fix->id, geodefb_name);

    fix->smem_start = GeodeMem_phys;
    fix->smem_len = GeodeSize;
    fix->mmio_start = GeodeRegs_phys;
    fix->mmio_len = GeodeRegs_phys_size;
    
    fix->type = FB_TYPE_PACKED_PIXELS;
    fix->type_aux = 0;
    if (par->bpp == 8)
        fix->visual = FB_VISUAL_PSEUDOCOLOR;
    else
        fix->visual = FB_VISUAL_TRUECOLOR;
    
    fix->xpanstep = 0;
    fix->ypanstep = 0;
    fix->ywrapstep = 0;
    fix->line_length = gfx_get_display_pitch();
    fix->accel = FB_ACCEL_NSC_GEODE;
    return(0);
}


/*
 *    Get the video params out of `var'. If a value doesn't fit, round
 *    it up, if it's too big, return -EINVAL.
 */

static int Geode_decode_var(struct fb_var_screeninfo *var,
                            struct geodefb_par *par)
{
#if 1
    par->xres = var->xres;
    par->yres = var->yres;
    par->bpp = var->bits_per_pixel;
    if (var->accel_flags & FB_ACCELF_TEXT)
        par->accel = FB_ACCELF_TEXT;
    else
        par->accel = 0;
#else
    if (Geodefb_Geode8) {
        par->xres = GEODE8_WIDTH;
        par->yres = GEODE8_HEIGHT;
        par->bpp = 8;
    } else {
        par->xres = GEODE16_WIDTH;
        par->yres = GEODE16_HEIGHT;
        par->bpp = 16;
    }
#endif
    return(0);
}


/*
 *    Fill the `var' structure based on the values in `par' and maybe
 *    other values read out of the hardware.
 */

static int Geode_encode_var(struct fb_var_screeninfo *var,
                            struct geodefb_par *par)
{
    memset(var, 0, sizeof(struct fb_var_screeninfo));
    var->xres = par->xres;
    var->yres = par->yres;
    var->xres_virtual = par->xres;
    var->yres_virtual = par->yres;
    var->xoffset = 0;
    var->yoffset = 0;

    var->bits_per_pixel = par->bpp;
    var->grayscale = 0;

    switch (var->bits_per_pixel) {
        case 8:		/* CLUT */
            var->red.offset = 0;
            var->red.length = 6;
            var->red.msb_right = 0;
            var->blue = var->green = var->red;
            break;
        case 16:	/* RGB 565 */
            var->red.offset = 11;
            var->red.length = 5;
            var->green.offset = 5;
            var->green.length = 6;
            var->blue.offset = 0;
            var->blue.length = 5;
            var->transp.offset = 0;
            var->transp.length = 0;
            break;
    }
    var->red.msb_right = 0;
    var->green.msb_right = 0;
    var->blue.msb_right = 0;
    var->transp.msb_right = 0;

    var->nonstd = 0;
    var->activate = 0;

    var->height = -1;
    var->width = -1;

    var->accel_flags = (par->accel &&
                        ((par->bpp == 8) || (par->bpp == 16))) ? FB_ACCELF_TEXT : 0;

/*	printk("CV64/3D : %s\n",(var->accel_flags ? "accel" : "no accel")); */

    var->vmode = FB_VMODE_NONINTERLACED;

      /* Clock is a 16:16 fixed point number */
    var->pixclock = (gfx_get_clock_frequency() & 0xFFFF0000) >> 6;
    if (var->pixclock)
        var->pixclock = 1000000000 / var->pixclock;
    
    var->sync = 0;
    var->hsync_len = gfx_get_hsync_end() - gfx_get_hsync_start();
    var->vsync_len = gfx_get_vsync_end() - gfx_get_vsync_start();
    var->right_margin = var->left_margin = 
        (gfx_get_htotal() - gfx_get_hactive() - var->hsync_len) / 2;

    var->upper_margin = var->lower_margin = 
        (gfx_get_vtotal() - gfx_get_vactive() - var->vsync_len) / 2;

    return(0);
}


/*
 *    Set a single color register. The values supplied are already
 *    rounded down to the hardware's capabilities (according to the
 *    entries in the var structure). Return != 0 for invalid regno.
 */

static int Geode_setcolreg(u_int regno, u_int red, u_int green, u_int blue,
                           u_int transp, struct fb_info *info)
{
    if (((current_par.bpp==8) && (regno>255)) ||
        ((current_par.bpp!=8) && (regno>15)))
        return (1);

    if (((current_par.bpp==8) && (regno<256)) || ((current_par.bpp==16) &&(regno<16))) {
        Geode_palette[regno].red = red >> 8;
        Geode_palette[regno].green = green >> 8;
        Geode_palette[regno].blue = blue >> 8;
    }

    switch (current_par.bpp) {
#ifdef FBCON_HAS_CFB8
        case 8:
#warning FIXME: Write to MMIO registers
              /* without protected mode interface, try VGA registers... */
            outb_p(regno,       dac_reg);
            outb_p(red   >> 10, dac_val);
            outb_p(green >> 10, dac_val);
            outb_p(blue  >> 10, dac_val);
            break;
#endif
#ifdef FBCON_HAS_CFB16
        case 16:
            fbcon_cmap.cfb16[regno] =
                ((red  & 0xf800) |
                 ((green & 0xfc00) >> 5) |
                 ((blue  & 0xf800) >> 11));
            break;
#endif
    }
    return (0);
}


/*
 *    Read a single color register and split it into
 *    colors/transparent. Return != 0 for invalid regno.
 */

static int Geode_getcolreg(u_int regno, u_int *red, u_int *green, u_int *blue,
                           u_int *transp, struct fb_info *info)
{
    int t;

    if (regno > 255)
        return (1);

    if (((current_par.bpp==8) && (regno<256)) || ((current_par.bpp==16) && (regno<16))) {

        t = Geode_palette[regno].red;
        *red    = (t<<8) | t;
        t = Geode_palette[regno].green;
        *green    = (t<<8) | t;
        t = Geode_palette[regno].blue;
        *blue    = (t<<8) | t;
    }
    *transp = 0;
    return (0);
}


/*
 *    (Un)Blank the screen
 *
 * 0 unblank, 1 blank, 2 no vsync, 3 no hsync, 4 off */

void Geode_blank(int blank)
{
    unsigned int iDispCfg;
    
    switch(blank)
    {
        case VESA_VSYNC_SUSPEND:
            iDispCfg = READ_VID32(CS5530_DISPLAY_CONFIG);
            iDispCfg &= (~0x4);
            WRITE_VID32(CS5530_DISPLAY_CONFIG, iDispCfg);
            break;
        case VESA_HSYNC_SUSPEND:
            iDispCfg = READ_VID32(CS5530_DISPLAY_CONFIG);
            iDispCfg &= (~0x2);
            WRITE_VID32(CS5530_DISPLAY_CONFIG, iDispCfg);
            break;
        case VESA_POWERDOWN:
            iDispCfg = READ_VID32(CS5530_DISPLAY_CONFIG);
            iDispCfg &= (~0x6);
            WRITE_VID32(CS5530_DISPLAY_CONFIG, iDispCfg);
            break;
        case VESA_NO_BLANKING:
        default:
            iDispCfg = READ_VID32(CS5530_DISPLAY_CONFIG);
            iDispCfg |= 0x6;
            WRITE_VID32(CS5530_DISPLAY_CONFIG, iDispCfg);
            break;
    }
}

/* -------------------- Interfaces to hardware functions -------------------- */


static struct fb_hwswitch Geode_switch = {
    Geode_init, Geode_encode_fix, Geode_decode_var, Geode_encode_var,
    Geode_getcolreg, Geode_setcolreg, Geode_blank
};


/* -------------------- Generic routines ------------------------------------ */


/*
 *    Fill the hardware's `par' structure.
 */

static void geodefb_get_par(struct geodefb_par *par)
{
    if (current_par_valid)
    {
        *par = current_par;
    }
    else
    {
        fbhw->decode_var(&geodefb_default, par);
    }
}


static void geodefb_set_par(struct geodefb_par *par)
{
    current_par = *par;
    current_par_valid = 1;
}


static int do_fb_set_var(struct fb_var_screeninfo *var, int isactive)
{
    int err, activate;
    struct geodefb_par par;

    if ((err = fbhw->decode_var(var, &par)))
        return(err);
    activate = var->activate;
    if ((var->activate & FB_ACTIVATE_MASK) == FB_ACTIVATE_NOW && isactive)
        geodefb_set_par(&par);
    fbhw->encode_var(var, &par);
    var->activate = activate;

    return 0;
}


static void do_install_cmap(int con, struct fb_info *info)
{
    int cmap_len = 0;
    
    if (info->var.bits_per_pixel == 8)
        cmap_len = 256;
    else
        cmap_len = 16;

    if (con != currcon)
        return;
    
    if (fb_display[con].cmap.len)
        fb_set_cmap(&fb_display[con].cmap, 1, fbhw->setcolreg, info);
    else
        fb_set_cmap(fb_default_cmap(cmap_len),
                    1, fbhw->setcolreg, info);
}


/*
 *  Open/Release the frame buffer device
 */

static int geodefb_open(struct fb_info *info, int user)
{
      /*
       * Nothing, only a usage count for the moment
       */

    MOD_INC_USE_COUNT;
    return(0);
}

static int geodefb_release(struct fb_info *info, int user)
{
    MOD_DEC_USE_COUNT;
    return(0);
}


/*
 *    Get the Fixed Part of the Display
 */

static int geodefb_get_fix(struct fb_fix_screeninfo *fix, int con,
                           struct fb_info *info)
{
    struct geodefb_par par;
    int error = 0;

    if (con == -1)
        geodefb_get_par(&par);
    else
        error = fbhw->decode_var(&fb_display[con].var, &par);
    return(error ? error : fbhw->encode_fix(fix, &par));
}


/*
 *    Get the User Defined Part of the Display
 */

static int geodefb_get_var(struct fb_var_screeninfo *var, int con,
                           struct fb_info *info)
{
    struct geodefb_par par;
    int error = 0;

    if (con == -1)
    {
        geodefb_get_par(&par);
        error = fbhw->encode_var(var, &par);
        disp.var = *var;   /* ++Andre: don't know if this is the right place */
    }
    else
    {
        *var = fb_display[con].var;
    }

    return(error);
}


static void geodefb_set_disp(int con, struct fb_info *info)
{
    struct fb_fix_screeninfo fix;
    struct display *display;

    if (con >= 0)
        display = &fb_display[con];
    else
        display = &disp;	/* used during initialization */

    geodefb_get_fix(&fix, con, info);
    if (con == -1)
        con = 0;

    display->screen_base = (char*) GeodeMem_virt;
    
    display->visual = fix.visual;
    display->type = fix.type;
    display->type_aux = fix.type_aux;
    display->ypanstep = fix.ypanstep;
    display->ywrapstep = fix.ywrapstep;
    display->can_soft_blank = 1;
    display->inverse = Geodefb_inverse;
    display->line_length = fix.line_length;
    display->next_line = fix.line_length;

    switch (display->var.bits_per_pixel) {
#ifdef FBCON_HAS_CFB8
        case 8:
            if (display->var.accel_flags & FB_ACCELF_TEXT) {
                display->dispsw = &fbcon_geode8;
/* #warning FIXME: We should reinit the graphics engine here */
            } else
                display->dispsw = &fbcon_cfb8;
            break;
#endif
#ifdef FBCON_HAS_CFB16
        case 16:
            if (display->var.accel_flags & FB_ACCELF_TEXT) {
                display->dispsw = &fbcon_geode16;
            } else
                display->dispsw = &fbcon_cfb16;
            display->dispsw_data = &fbcon_cmap.cfb16;
            break;
#endif
        default:
            display->dispsw = &fbcon_dummy;
            break;
    }
}


/*
 *    Set the User Defined Part of the Display
 */

static int geodefb_set_var(struct fb_var_screeninfo *var, int con,
                           struct fb_info *info)
{
    int err, oldxres, oldyres, oldvxres, oldvyres, oldbpp, oldaccel;


      // Check to see if we support the mode
      // we really need to check our mode table here
    if ((var->activate & FB_ACTIVATE_MASK) == FB_ACTIVATE_TEST)
    {
        if (gfx_is_display_mode_supported(var->xres, var->yres,
                                          var->bits_per_pixel,
                                          75))
        {
            return 0;
        }
        else
            return -EINVAL;
    }
  
    if ((err = do_fb_set_var(var, con == currcon)))
        return(err);
  
    if ((var->activate & FB_ACTIVATE_MASK) == FB_ACTIVATE_NOW) {
        oldxres = fb_display[con].var.xres;
        oldyres = fb_display[con].var.yres;
        oldvxres = fb_display[con].var.xres_virtual;
        oldvyres = fb_display[con].var.yres_virtual;
        oldbpp = fb_display[con].var.bits_per_pixel;
        oldaccel = fb_display[con].var.accel_flags;
        fb_display[con].var = *var;
        if (oldxres != var->xres || oldyres != var->yres ||
            oldvxres != var->xres_virtual ||
            oldvyres != var->yres_virtual ||
            oldbpp != var->bits_per_pixel ||
            oldaccel != var->accel_flags) {
      
              /* Switch the mode here */
            gfx_disable_softvga();
            gfx_set_bpp(var->bits_per_pixel);
            gfx_set_display_mode(var->xres, var->yres, 
                                 var->bits_per_pixel, 75);

            geodefb_set_disp(con, info);
            
            if (info->changevar)
                (*info->changevar) (con);
      
            fb_alloc_cmap(&fb_display[con].cmap, 0, 0);
            do_install_cmap(con, info);
        }
    }
    var->activate = 0;
    return(0);
}


/*
 *    Get the Colormap
 */

static int geodefb_get_cmap(struct fb_cmap *cmap, int kspc, int con,
                            struct fb_info *info)
{
    if (con == currcon) /* current console? */
        return(fb_get_cmap(cmap, kspc, fbhw->getcolreg, info));
    else if (fb_display[con].cmap.len) /* non default colormap? */
        fb_copy_cmap(&fb_display[con].cmap, cmap, kspc ? 0 : 2);
    else
        fb_copy_cmap(fb_default_cmap(1<<fb_display[con].var.bits_per_pixel),
                     cmap, kspc ? 0 : 2);
    return(0);
}


/*
 *    Set the Colormap
 */

static int geodefb_set_cmap(struct fb_cmap *cmap, int kspc, int con,
                            struct fb_info *info)
{
    int err;
    int cmap_len = 0;
    
    if (info->var.bits_per_pixel == 8)
        cmap_len = 256;
    else
        cmap_len = 16;

    if (!fb_display[con].cmap.len) {       /* no colormap allocated? */
        if ((err = fb_alloc_cmap(&fb_display[con].cmap,
                                 cmap_len, 0)))
            return(err);
    }
    if (con == currcon)		 /* current console? */
        return(fb_set_cmap(cmap, kspc, fbhw->setcolreg, info));
    else
        fb_copy_cmap(cmap, &fb_display[con].cmap, kspc ? 0 : 1);
    return(0);
}


/*
 *    Pan or Wrap the Display
 *
 *    This call looks only at xoffset, yoffset and the FB_VMODE_YWRAP flag
 */

static int geodefb_pan_display(struct fb_var_screeninfo *var, int con,
                               struct fb_info *info)
{
    return(-EINVAL);
}


/*
 *	 Geodevision Frame Buffer Specific ioctls
 */

static int geodefb_ioctl(struct inode *inode, struct file *file,
                         u_int cmd, u_long arg, int con, struct fb_info *info)
{
    if (cmd == FBIOGAL_API)
    {
        return geodefb_ioctl_ext(inode, file, cmd, arg, con, info);
    }

    return(-EINVAL);
}


static struct fb_ops geodefb_ops = {
    geodefb_open, geodefb_release, geodefb_get_fix, geodefb_get_var,
    geodefb_set_var, geodefb_get_cmap, geodefb_set_cmap,
    geodefb_pan_display, geodefb_ioctl
};


int __init geodefb_setup(char *options)
{
    char *this_opt;

    fb_info.fontname[0] = '\0';

    if (!options || !*options)
        return 0;

    for (this_opt = strtok(options, ","); this_opt; this_opt = strtok(NULL, ","))
        if (!strcmp(this_opt, "inverse")) {
            Geodefb_inverse = 1;
            fb_invert_cmaps();
        } else if (!strncmp(this_opt, "font:", 5))
            strcpy(fb_info.fontname, this_opt+5);
        else if (!strcmp (this_opt, "geode8")){
            geodefb_default = geodefb_predefined[GEODE8_DEFMODE].var;
        }
        else if (!strcmp (this_opt, "geode16")){
            geodefb_default = geodefb_predefined[GEODE16_DEFMODE].var;
        }
        else
            get_video_mode(this_opt);

    DPRINTK("default mode: xres=%d, yres=%d, bpp=%d\n",geodefb_default.xres,
            geodefb_default.yres,
            geodefb_default.bits_per_pixel);
    return 0;
}

static int
nscgal_open(struct inode *inode, struct file *file)
{
    return 0;
}

static int
nscgal_release(struct inode *inode, struct file *file)
{
    return 0;
}

static int 
nscgal_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
         unsigned long arg)
{
    return geodefb_ioctl_ext(inode, file, cmd, arg, 0, NULL);
}

static struct file_operations nscgal_fops = {
    ioctl:		nscgal_ioctl,
    open:		nscgal_open,
    release:	nscgal_release,
};

static devfs_handle_t devfs_handle = NULL;
void geode_create_galdevice(void)
{
      /* We get a major device handle out. */
    if ((devfs_handle = devfs_register_chrdev(UNNAMED_MAJOR,"nscgal",&nscgal_fops)) <= 0)
        printk("unable to get major %d for fb devs\n", FB_MAJOR);
}

/*
 *    Initialization
 */

int __init geodefb_init(void)
{
    struct geodefb_par par;
    unsigned int cpu_version, address, address_size;

    cpu_version = gfx_detect_cpu();
    if (!cpu_version)
    {
          /* NO GEODE PROCESSOR DETECTED */
        return -ENXIO;
    }

      /* DISPLAY DETECTION MESSAGE */

    DPRINTK("Geode Processor Detected (Type = %d, Version = %d.%d).\n",
           cpu_version & 0xFF, (cpu_version >> 8) & 0xFF, (cpu_version >> 16) & 0xFF);
    
      /* Initialize durango API here */
      /* SET DURANGO REGISTER POINTERS */
      /* The method of mapping from a physical address to a linear address */
      /* is operating system independent.  Set variables to linear address. */
    
    GeodeRegs_phys = address = gfx_get_cpu_register_base();
    GeodeRegs_phys_size = address_size = 0x10000;
#if 0 
      /* Fails for geode. Investigate later */
    if (!request_mem_region(address, address_size, "geodefb")) {
        DPRINTK("Geodefb: abort, cannot reserve Gfx registers at 0x%lx\n",
                address);
        return -EBUSY;
    }
#endif
    gfx_regptr = (unsigned char *)ioremap(address, address_size);
    DPRINTK("Geodefb: Gfx Registers at 0x%x, mapped to 0x%p, size %dk\n",
            address, gfx_regptr, address_size/1024);
    

    address = gfx_get_vid_register_base();
    address_size = 0x1000;
#if 0
      /* Fails for geode. Investigate later */
    if (!request_mem_region(address, address_size, "geodefb")) {
        DPRINTK("Geodefb: abort, cannot reserve 5530 mem space at 0x%lx\n",
                address);
        return -EBUSY;
    }
#endif
    gfx_vidptr = (unsigned char *)ioremap(address, address_size);
    DPRINTK("Geodefb: 5530 registers at 0x%x, mapped to 0x%p, size %dk\n",
            address, (unsigned int *)gfx_vidptr, address_size/1024);
    

    GeodeMem_phys = address = gfx_get_frame_buffer_base();
    GeodeSize = address_size = gfx_get_frame_buffer_size();
#if 0 
      /* Fails for geode. Investigate later */
    if (!request_mem_region(address, address_size, "geodefb")) {
        DPRINTK("Geodefb: abort, cannot reserve video mem at 0x%lx\n",
                GeodeMem_phys);
        return -EBUSY;
    }    
#endif
    gfx_fbptr = (unsigned char *)ioremap(address, address_size);
    GeodeMem_virt = (unsigned int) gfx_fbptr;
    DPRINTK("Geodefb: framebuffer at 0x%x, mapped to 0x%p, size %dk\n",
            address, (unsigned char *)gfx_fbptr, address_size/1024);


      /* CHECK IF REGISTERS WERE MAPPED SUCCESSFULLY */

    if (!((unsigned)gfx_regptr & (unsigned)gfx_vidptr & (unsigned)gfx_fbptr))
    {	
        DPRINTK("%s", "Geodefb: Could not map hardware registers.\n");
        return -EBUSY;
    }

    geode_create_galdevice();
    
      /* If we are in hires mode, get that mode */
    if (screen_info.orig_video_isVGA != VIDEO_TYPE_VLFB)
        return -EBUSY;

    par.xres = screen_info.lfb_width;
    par.yres = screen_info.lfb_height;
    par.bpp = screen_info.lfb_depth;
    par.Refresh = 75; /* Strap to 75hz */

    fbhw = &Geode_switch;

    strcpy(fb_info.modename, geodefb_name);
    fb_info.changevar = NULL;
    fb_info.node = -1;
    fb_info.fbops = &geodefb_ops;
    fb_info.disp = &disp;
    fb_info.switch_con = &Geodefb_switch;
    fb_info.updatevar = &Geodefb_updatevar;
    fb_info.blank = &Geodefb_blank;
    fb_info.flags = FBINFO_FLAG_DEFAULT;

/*    fbhw->decode_var(&geodefb_default, &par); */
    fbhw->encode_var(&geodefb_default, &par);

      /* Let us not trust the Bios, go thru a mode set again */
    if (screen_info.orig_video_isVGA == VIDEO_TYPE_VLFB)
    {
        gfx_disable_softvga();
        gfx_set_bpp(geodefb_default.bits_per_pixel);
        gfx_set_display_mode(geodefb_default.xres, 
                             geodefb_default.yres, 
                             geodefb_default.bits_per_pixel, 
                             75);
        fbhw->init();
    }

    do_fb_set_var(&geodefb_default, 1);
    geodefb_get_var(&fb_display[0].var, -1, &fb_info);
    geodefb_set_disp(-1, &fb_info);

    do_install_cmap(0, &fb_info);

    if (register_framebuffer(&fb_info) < 0) {
        DPRINTK("%s", "geodefb.c: register_framebuffer failed\n");
        return -EINVAL;
    }

    DPRINTK("fb%d: %s frame buffer device, using %ldK of "
            "video memory\n", GET_FB_IDX(fb_info.node),
            fb_info.modename, GeodeSize>>10);
    
      /* TODO: This driver cannot be unloaded yet */
    MOD_INC_USE_COUNT;
    return 0;	
}


static int Geodefb_switch(int con, struct fb_info *info)
{
      /* Do we have to save the colormap? */
    if (fb_display[currcon].cmap.len)
        fb_get_cmap(&fb_display[currcon].cmap, 1, fbhw->getcolreg,
                    info);

    do_fb_set_var(&fb_display[con].var, 1);
    currcon = con;
      /* Install new colormap */
    do_install_cmap(con, info);
    return(0);
}


/*
 *    Update the `var' structure (called by fbcon.c)
 *
 *    This call looks only at yoffset and the FB_VMODE_YWRAP flag in `var'.
 *    Since it's called by a kernel driver, no range checking is done.
 */

static int Geodefb_updatevar(int con, struct fb_info *info)
{
    return(0);
}


/*
 *    Blank the display.
 */

static void Geodefb_blank(int blank, struct fb_info *info)
{
    fbhw->blank(blank);
}


/*
 *    Get a Video Mode
 */

static int __init get_video_mode(const char *name)
{
    int i;

    for (i = 0; i < NUM_TOTAL_MODES; i++) {
        if (!strcmp(name, geodefb_predefined[i].name)) {
            geodefb_default = geodefb_predefined[i].var;
            return(i);
        }
    }
      /* ++Andre: set geodefb default mode */
    geodefb_default = geodefb_predefined[GEODE8_DEFMODE].var;
    return(0);
}


/*
 *    Text console acceleration
 */

static void geodefb_bmove(struct display *p, int sy, int sx, int dy, int dx,
                          int height, int width)
{
    sx *= fontwidth(p); 
    dx *= fontwidth(p); 
    width *= fontwidth(p);
    
    sy *= fontheight(p);
    dy *= fontheight(p);
    height *= fontheight(p);

    gfx_set_solid_pattern(0);
    gfx_set_raster_operation(0xCC);
    gfx_screen_to_screen_blt(sx, sy, 
                             dx, dy,
                             width, height);
    blit_maybe_busy = 1;
}

static void geodefb_clear(struct vc_data *conp, struct display *p, int sy, int sx,
                          int height, int width)
{
    unsigned long bg = 0;
    
    sx *= fontwidth(p); 
    width *= fontwidth(p);
    
    sy *= fontheight(p);
    height *= fontheight(p);
    
    if (p->var.bits_per_pixel == 8)
    {
        bg = attr_bgcol_ec(p,conp);
    }
    else
    {
        bg = ((u16 *)p->dispsw_data)[attr_bgcol_ec(p, conp)];
    }

    gfx_set_solid_pattern(bg);
    gfx_set_raster_operation(0xF0);
    gfx_pattern_fill(sx, sy, width, height);
    blit_maybe_busy = 1;
}

#ifdef FBCON_HAS_CFB8

static void fbcon_geode8_putc(struct vc_data *conp, struct display *p, int c, int yy,
                              int xx)
{
    if (blit_maybe_busy)
        Geode_WaitBusy();
    fbcon_cfb8_putc(conp, p, c, yy, xx);
}

static void fbcon_geode8_putcs(struct vc_data *conp, struct display *p,
                               const unsigned short *s, int count, int yy, int xx)
{
    if (blit_maybe_busy)
        Geode_WaitBusy();
    fbcon_cfb8_putcs(conp, p, s, count, yy, xx);
}

static void fbcon_geode8_revc(struct display *p, int xx, int yy)
{
    if (blit_maybe_busy)
        Geode_WaitBusy();
    fbcon_cfb8_revc(p, xx, yy);
}

static void fbcon_geode8_clear_margins(struct vc_data *conp, struct display *p,
                                       int bottom_only)
{
    if (blit_maybe_busy)
        Geode_WaitBusy();
    fbcon_cfb8_clear_margins(conp, p, bottom_only);
}

static struct display_switch fbcon_geode8 = {
    fbcon_cfb8_setup, geodefb_bmove, geodefb_clear, fbcon_geode8_putc,
    fbcon_geode8_putcs, fbcon_geode8_revc, NULL, NULL, fbcon_geode8_clear_margins,
    FONTWIDTH(4)|FONTWIDTH(8)|FONTWIDTH(12)|FONTWIDTH(16)
};
#endif

#ifdef FBCON_HAS_CFB16
   
static void fbcon_geode16_putc(struct vc_data *conp, struct display *p, int c, int yy,
                               int xx)
{
    if (blit_maybe_busy)
        Geode_WaitBusy();
    fbcon_cfb16_putc(conp, p, c, yy, xx);
}

static void fbcon_geode16_putcs(struct vc_data *conp, struct display *p,
                                const unsigned short *s, int count, int yy, int xx)
{
    if (blit_maybe_busy)
        Geode_WaitBusy();
    fbcon_cfb16_putcs(conp, p, s, count, yy, xx);
}

static void fbcon_geode16_revc(struct display *p, int xx, int yy)
{
    if (blit_maybe_busy)
        Geode_WaitBusy();
    fbcon_cfb16_revc(p, xx, yy);
}

static void fbcon_geode16_clear_margins(struct vc_data *conp, struct display *p,
                                        int bottom_only)
{
    if (blit_maybe_busy)
        Geode_WaitBusy();
    fbcon_cfb16_clear_margins(conp, p, bottom_only);
}

static struct display_switch fbcon_geode16 = {
    fbcon_cfb16_setup, geodefb_bmove, geodefb_clear, fbcon_geode16_putc,
    fbcon_geode16_putcs, fbcon_geode16_revc, NULL, NULL, fbcon_geode16_clear_margins,
    FONTWIDTH(4)|FONTWIDTH(8)|FONTWIDTH(12)|FONTWIDTH(16)
};
#endif

#ifdef MODULE
int init_module(void)
{
    return geodefb_init();
}

void cleanup_module(void)
{
      /* Not reached because the usecount will never be
         decremented to zero */
    unregister_framebuffer(&fb_info);
      /* TODO: clean up ... */
}
#endif /* MODULE */
