/*****************************************************************************
 *
 *                  CX5530  Native Audio driver for Linux 2.[23].x
 *
 *
 *  Supported devices:
 *  /dev/dsp0-3    standard /dev/dsp device, (mostly) OSS compatible
 *  /dev/mixer  standard /dev/mixer device, (mostly) OSS compatible
*
*****************************************************************************/
#define DRIVER_VERSION "1.0.3"
/*
#define M_DEBUG 
#define STARTUP_DEBUG
#define READ_DEBUG
#define WRITE_DEBUG 
#define IOCTL_DEBUG
#define ALLOC_DEBUG
#define INT_DEBUG
*/
#include "includes.h"
#include "cx5530_debug.h"

#define NUMBER_OF_FRAGS_BEFORE_WAIT 8

//--------------------------------
//              DURAUDIO Prototypes
//--------------------------------

unsigned char  DURAUDIO_Initialize   (unsigned int Irq, unsigned int IoBase, unsigned char DmaChan);
void           DURAUDIO_Deinitialize (void);

MMRESULT       DURAUDIO_WaveOpen     (WAPI_INOUT, LPWAVEFORMATEX, unsigned char);
void           DURAUDIO_WaveClose    (void);
PWAVEHDR       DURAUDIO_WaveStart    (PWAVEHDR, unsigned long);
PWAVEHDR       DURAUDIO_WaveContinue (PWAVEHDR pwh);
void           DURAUDIO_WaveStop     (void);
unsigned char  DURAUDIO_ClearStatus  (void);

void           DURAUDIO_SetCodecRate       (unsigned long);           
void           DURAUDIO_CodecWrite      (unsigned short codec_port, unsigned short codec_data);
unsigned short DURAUDIO_CodecDirectRegRead (unsigned short codec_port );

//--------------------------------
//                   Local prototypes
//--------------------------------

PWAVEHDR     AddNode             (unsigned long);
void         cx5530_WaveOpen     (size_t count);
void         RemoveHEAD          (void);
void         EmptyList           (void);
void         CalculateBufferSize (size_t count);
unsigned int BytesToUse          (void);
void         StartDMA            (void);
static void  cx5530_interrupt    (int irq, void *dev_id, struct pt_regs *regs);
void 		  PrintGlobalVariables (void);
//
//        Global variables
//

unsigned long flags;

volatile PWAVEHDR pWaveHeader_HEAD, pWaveHeader_TAIL, pWaveHeaderLastBuffered;
WAVEFORMATEX    WaveFormat;

volatile unsigned long BytesAllocated, DMABufferSize;
volatile unsigned int  WaveStarted, WaveOpened, FragSizeInBytes, OSSFragSize;
volatile unsigned int  BytesRecorded;
WAPI_INOUT WaveDirection;

DECLARE_WAIT_QUEUE_HEAD(REC_wait);
DECLARE_WAIT_QUEUE_HEAD(PLAY_wait);

volatile int EndOfBuffer;

spinlock_t LockWave;				 //  This locks around the oss Wave Operations

//---------------------------------------------------------
//          Structures
//---------------------------------------------------------
enum card_types_t 
{
	TYPE_CX5530,
	TYPE_GEODE
};

static const char *card_names[]=
{
	[TYPE_CX5530] = "CX5530",
	[TYPE_GEODE]  = "GEODE"
};


//---------------------------------------------------------
//                      CX5530 Structures
//---------------------------------------------------------
struct cx5530_state 
{
	struct cx5530_card *card;   	 //  Card info
	spinlock_t lock;				 //  This locks around the oss state in the driver 
	wait_queue_head_t open_wait;  //  Only let 1 be opening at a time 
	mode_t open_mode;
	int dev_audio;				 //  Soundcore stuff 
};

struct cx5530_card 
{
	int dev_mixer;
	int card_type;
	
   //   As most of this is static,
   //  perhaps it should be a pointer to a global struct 
	struct mixer_data 
	{
		int modcnt;
		int supported_mixers;
		int stereo_mixers;
		int record_sources;
		
      // the caller must guarantee arg sanity before calling these 
		
		void (*write_mixer)(struct cx5530_card *card,int mixer, unsigned int left,unsigned int right);
		int (*recmask_io)(struct cx5530_card *card,int rw,int mask);
		unsigned int mixer_state[SOUND_MIXER_NRDEVICES];
	} mix;
	
	struct cx5530_state state;
	
   //  This locks around the physical registers on the card 
	spinlock_t lock;
	
   //  Hardware resources 
	struct pci_dev pcidev; 
	u32    iobase;
	u32    irq;
};

static struct cx5530_card *devs = NULL;

//
//    LM4548 - AC97 codec programming interface.
//

static void cx5530_ac97_set(int io, u8 cmd, u16 val)
{
	unsigned short   reg;
	
	CHECK_SUSPEND;
	
	reg=cmd;
	reg=reg<<8;
	DURAUDIO_CodecWrite (reg, val);
}

static u16 cx5530_ac97_get(int io, u8 cmd)
{
	u16 data;
	
	CHECK_SUSPEND;
	data = DURAUDIO_CodecDirectRegRead (cmd);
	
	return data;
}

//
//   OSS interface to the ac97s.. 
//
#define AC97_STEREO_MASK (SOUND_MASK_VOLUME|SOUND_MASK_PCM|SOUND_MASK_LINE|SOUND_MASK_CD|SOUND_MASK_VIDEO|SOUND_MASK_LINE1|SOUND_MASK_IGAIN)
#define AC97_SUPPORTED_MASK (AC97_STEREO_MASK | SOUND_MASK_BASS|SOUND_MASK_TREBLE|SOUND_MASK_MIC|SOUND_MASK_SPEAKER)
#define AC97_RECORD_MASK (SOUND_MASK_MIC|SOUND_MASK_CD| SOUND_MASK_VIDEO| SOUND_MASK_LINE1| SOUND_MASK_LINE|SOUND_MASK_PHONEIN)

#define supported_mixer(CARD,FOO) ( CARD->mix.supported_mixers & (1<<FOO) )

//---------------------------------------------------------
//  This table has default mixer values for all OSS mixers.
//  be sure to fill it in if you add oss mixers
//  to anyone's supported mixer defines 
//---------------------------------------------------------

unsigned int mixer_defaults[SOUND_MIXER_NRDEVICES] = 
{
	[SOUND_MIXER_VOLUME] =  0x3232,
	[SOUND_MIXER_BASS] =    0x3232,
	[SOUND_MIXER_TREBLE] =  0x3232,
	[SOUND_MIXER_SPEAKER] = 0x3232,
	[SOUND_MIXER_MIC] =     0x8000,  
	[SOUND_MIXER_LINE] =    0x3232,
	[SOUND_MIXER_CD] =      0x3232,
	[SOUND_MIXER_VIDEO] =   0x3232,
	[SOUND_MIXER_LINE1] =   0x3232,
	[SOUND_MIXER_PCM] =     0x3232,
	[SOUND_MIXER_IGAIN] =   0x3232
};

static struct ac97_mixer_hw 
{
	unsigned char offset;
	int scale;
} ac97_hw[SOUND_MIXER_NRDEVICES]= 
{
	[SOUND_MIXER_VOLUME]    =   {0x02,63},
	[SOUND_MIXER_BASS]      =   {0x08,15},
	[SOUND_MIXER_TREBLE]    =   {0x08,15},
	[SOUND_MIXER_SPEAKER]   =   {0x0a,15},
	[SOUND_MIXER_MIC]       =   {0x0e,31},
	[SOUND_MIXER_LINE]      =   {0x10,31},
	[SOUND_MIXER_CD]        =   {0x12,31},
	[SOUND_MIXER_VIDEO]     =   {0x14,31},
	[SOUND_MIXER_LINE1]     =   {0x16,31},
	[SOUND_MIXER_PCM]       =   {0x18,31},
	[SOUND_MIXER_IGAIN]     =   {0x1c,15}
};

//---------------------------------------------------------
//  Write the OSS encoded volume to the given OSS encoded mixe2r,
//  again caller's job to make sure all is well in arg land,
//  call with spinlock held 
//---------------------------------------------------------

//  linear scale -> log 

unsigned char lin2log[101] = 
{
	0, 0 , 15 , 23 , 30 , 34 , 38 , 42 , 45 , 47 ,
	50 , 52 , 53 , 55 , 57 , 58 , 60 , 61 , 62 ,
	63 , 65 , 66 , 67 , 68 , 69 , 69 , 70 , 71 ,
	72 , 73 , 73 , 74 , 75 , 75 , 76 , 77 , 77 ,
	78 , 78 , 79 , 80 , 80 , 81 , 81 , 82 , 82 ,
	83 , 83 , 84 , 84 , 84 , 85 , 85 , 86 , 86 ,
	87 , 87 , 87 , 88 , 88 , 88 , 89 , 89 , 89 ,
	90 , 90 , 90 , 91 , 91 , 91 , 92 , 92 , 92 ,
	93 , 93 , 93 , 94 , 94 , 94 , 94 , 95 , 95 ,
	95 , 95 , 96 , 96 , 96 , 96 , 97 , 97 , 97 ,
	97 , 98 , 98 , 98 , 98 , 99 , 99 , 99 , 99 , 99 
};

static void ac97_write_mixer(struct cx5530_card *card,int mixer, unsigned int left, unsigned int right)
{
	u16 val=0;
	struct ac97_mixer_hw *mh = &ac97_hw[mixer];
	
	if(AC97_STEREO_MASK & (1<<mixer)) 
	{
		//  Stereo mixers, mute them if we can 
		
		if (mixer == SOUND_MIXER_IGAIN) 
		{
			//   igain's slider is reversed.. 
			right = (right * mh->scale) / 100;
			left = (left * mh->scale) / 100;
			
			if ((left == 0) && (right == 0))
				val |= 0x8000;
		} 
		else 
		{
			//   log conversion for the stereo controls 
			if((left == 0) && (right == 0))
				val = 0x8000;
			right = ((100 - lin2log[right]) * mh->scale) / 100;
			left = ((100 - lin2log[left]) * mh->scale) / 100;
		}
		
		val |= (left << 8) | right;
		
	} 
	else 
	if (mixer == SOUND_MIXER_SPEAKER) 
	{
		val = (((100 - left) * mh->scale) / 100) << 1;
	} 
	else 
	if (mixer == SOUND_MIXER_MIC) 
	{
		val = cx5530_ac97_get(card->iobase , mh->offset) & ~0x801f;
		val |= (((100 - left) * mh->scale) / 100);
		//   the low bit is optional in the tone sliders and masking
		//   it lets is avoid the 0xf 'bypass'.. 
	} 
	else 
	if (mixer == SOUND_MIXER_BASS) 
	{
		val = cx5530_ac97_get(card->iobase , mh->offset) & ~0x0f00;
		val |= ((((100 - left) * mh->scale) / 100) << 8) & 0x0e00;
	} 
	else 
	if (mixer == SOUND_MIXER_TREBLE)  
	{
		val = cx5530_ac97_get(card->iobase , mh->offset) & ~0x000f;
		val |= (((100 - left) * mh->scale) / 100) & 0x000e;
	}
	
	cx5530_ac97_set(card->iobase , mh->offset, val);
}

//---------------------------------------------------------
//   The following tables allow us to go from 
//   OSS <-> ac97 quickly. 
//---------------------------------------------------------

enum ac97_recsettings {
	AC97_REC_MIC=0,
	AC97_REC_CD,
	AC97_REC_VIDEO,
	AC97_REC_AUX,
	AC97_REC_LINE,
	AC97_REC_STEREO,   // combination of all enabled outputs..  
	AC97_REC_MONO,       // ... or the mono equivalent 
	AC97_REC_PHONE        
};

static unsigned int ac97_oss_mask[] = {
	[AC97_REC_MIC] = SOUND_MASK_MIC, 
	[AC97_REC_CD] = SOUND_MASK_CD, 
	[AC97_REC_VIDEO] = SOUND_MASK_VIDEO, 
	[AC97_REC_AUX] = SOUND_MASK_LINE1, 
	[AC97_REC_LINE] = SOUND_MASK_LINE, 
	[AC97_REC_PHONE] = SOUND_MASK_PHONEIN
};

//  indexed by bit position 
static unsigned int ac97_oss_rm[] = {
	[SOUND_MIXER_MIC] = AC97_REC_MIC,
	[SOUND_MIXER_CD] = AC97_REC_CD,
	[SOUND_MIXER_VIDEO] = AC97_REC_VIDEO,
	[SOUND_MIXER_LINE1] = AC97_REC_AUX,
	[SOUND_MIXER_LINE] = AC97_REC_LINE,
	[SOUND_MIXER_PHONEIN] = AC97_REC_PHONE
};

//---------------------------------------------------------
//   read or write the recmask.
//   the ac97 can really have left and right recording
//   inputs independantly set, but OSS doesn't seem to 
//   want us to express that to the user. 
//   the caller guarantees that we have a supported bit set,
//   and they must be holding the card's spinlock 
//---------------------------------------------------------

static int 
ac97_recmask_io(struct cx5530_card *card, int read, int mask) 
{
	unsigned int val = ac97_oss_mask[ cx5530_ac97_get(card->iobase, 0x1a) & 0x7 ];
	
	if (read) return val;
	
	//   OSS can have many inputs, cx5530 cant.  try
	//  to pick the 'new' one 
	
	if (mask != val) mask &= ~val;
	
	val = ffs(mask) - 1; 
	val = ac97_oss_rm[val];
	val |= val << 8;  /* set both channels */
	
	M_printk("cx5530: setting ac97 recmask to 0x%x\n",val);
	
	cx5530_ac97_set(card->iobase,0x1a,val);
	
	return 0;
};

static u16 cx5530_ac97_init(struct cx5530_card *card, int iobase)
{
	u16 vend1, vend2, caps;
	
	card->mix.supported_mixers = AC97_SUPPORTED_MASK;
	card->mix.stereo_mixers = AC97_STEREO_MASK;
	card->mix.record_sources = AC97_RECORD_MASK;
	card->mix.write_mixer = ac97_write_mixer;
	card->mix.recmask_io = ac97_recmask_io;
	
	vend1 = cx5530_ac97_get(iobase, 0x7c);
	vend2 = cx5530_ac97_get(iobase, 0x7e);
	
	caps = cx5530_ac97_get(iobase, 0x00);
	
	STARTUP_printk(KERN_INFO "cx5530: AC97 Codec detected: v: 0x%2x%2x caps: 0x%x pwr: 0x%x\n",
		vend1,vend2,caps,cx5530_ac97_get(iobase,0x26) & 0xf);
	
	if (! (caps & 0x4) )
		card->mix.supported_mixers &= ~(SOUND_MASK_BASS|SOUND_MASK_TREBLE);  //  no bass/treble nobs 
	
	cx5530_ac97_set(iobase, 0x1E, 0x0404);
	cx5530_ac97_set(iobase, 0x20, 0x0000);
	
	return 0;
}

void PrintGlobalVariables(void)
{
	M_printk("************************************\n");
	M_printk("*         Global Variables         *\n");
	M_printk("************************************\n");
	//	M_printk("BytesAllocated :%08X\n", BytesAllocated);
	// 	M_printk("DMABufferSize  :%08X\n", DMABufferSize);
	M_printk("BytesRecorded  :%08X\n", BytesRecorded);
	M_printk("WaveStarted    :%08X\n", WaveStarted);
	M_printk("WaveOpened     :%08X\n", WaveOpened);
	M_printk("FragSizeInBytes:%08X\n", FragSizeInBytes);
	M_printk("OSSFragSize    :%08X\n", OSSFragSize);
	
}

static void cx5530_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{      
	M_printk("INT! ");   
	
	INT_printk("pWaveHeaderLastBuffered (IN):%08X\n", (unsigned int) pWaveHeaderLastBuffered);
	
	if (pWaveHeaderLastBuffered==NULL)
	{               
		INT_printk("HEAD:%08X  TAIL:%08X  BytesAllocated:%d  BytesRecorded:%d\n", pWaveHeader_HEAD,pWaveHeader_TAIL, BytesAllocated, BytesRecorded);
		INT_printk("No more data. List is NULL ");
		
		if(WaveStarted)
		{
			M_printk("EOP ***\n");
			DURAUDIO_WaveStop(); 
			WaveStarted=FALSE;
		}
		else
		{
			M_printk("EOT ***\n");
			M_printk("BytesAllocated:%d   BytesRecorded:%d\n", BytesAllocated, BytesRecorded);
			EmptyList();			
			DURAUDIO_ClearStatus();
			DURAUDIO_WaveClose();
			WaveOpened=FALSE;		
		}
	}
	else
	{		
		M_printk("pWaveHeaderLastBuffered:%08X  BytesRecorded:%08X\n", pWaveHeaderLastBuffered, BytesRecorded);
		pWaveHeaderLastBuffered=DURAUDIO_WaveContinue(pWaveHeaderLastBuffered);		
	}
	
	EndOfBuffer=TRUE;
	
	if (WaveDirection==WAPI_OUT)
		wake_up_interruptible(&PLAY_wait);
	else
	{
		BytesRecorded+=FragSizeInBytes;
		wake_up_interruptible(&REC_wait);
	}
	
	INT_printk("pWaveHeaderLastBuffered (OUT):%08X\n", pWaveHeaderLastBuffered);
}

static void set_mixer(struct cx5530_card *card,unsigned int mixer, unsigned int val ) 
{
	unsigned int left,right;
	
	right = ((val >> 8)  & 0xff) ;
	left = (val  & 0xff) ;
	
	if(right > 100) right = 100;
	if(left > 100) left = 100;
	
	card->mix.mixer_state[mixer]=(right << 8) | left;
	card->mix.write_mixer(card,mixer,left,right);
}

static void mixer_push_state(struct cx5530_card *card)
{
	int i;
	for(i = 0 ; i < SOUND_MIXER_NRDEVICES ; i++) 
	{
		if( ! supported_mixer(card,i)) continue;
		
		set_mixer(card,i,card->mix.mixer_state[i]);
	}
}

static int mixer_ioctl(struct cx5530_card *card, unsigned int cmd, unsigned long arg)
{
	int i, val=0;
	
	if (cmd == SOUND_MIXER_INFO) 
	{
		mixer_info info;
		strncpy(info.id, card_names[card->card_type], sizeof(info.id));
		strncpy(info.name,card_names[card->card_type],sizeof(info.name));
		info.modify_counter = card->mix.modcnt;
		if (copy_to_user((void *)arg, &info, sizeof(info)))
			return -EFAULT;
		return 0;
	}
	
	if (cmd == SOUND_OLD_MIXER_INFO) 
	{
		_old_mixer_info info;
		strncpy(info.id, card_names[card->card_type], sizeof(info.id));
		strncpy(info.name,card_names[card->card_type],sizeof(info.name));
		if (copy_to_user((void *)arg, &info, sizeof(info)))
			return -EFAULT;
		return 0;
	}
	
	if (cmd == OSS_GETVERSION)
		return put_user(SOUND_VERSION, (int *)arg);
	
	if (_IOC_TYPE(cmd) != 'M' || _IOC_SIZE(cmd) != sizeof(int))
		return -EINVAL;
	
	if (_IOC_DIR(cmd) == _IOC_READ) {
		switch (_IOC_NR(cmd)) {
			case SOUND_MIXER_RECSRC: /* give them the current record source */
				
				if(!card->mix.recmask_io) {
					val = 0;
				} else {
					spin_lock(&card->lock);
					val = card->mix.recmask_io(card,1,0);
					spin_unlock(&card->lock);
				}
				break;
				
			case SOUND_MIXER_DEVMASK: //  Give them the supported mixers 
				val = card->mix.supported_mixers;
				break;
				
			case SOUND_MIXER_RECMASK: //  Arg contains a bit for each supported recording source 
				val = card->mix.record_sources;
				break;
				
			case SOUND_MIXER_STEREODEVS: // Mixer channels supporting stereo
				val = card->mix.stereo_mixers;
				break;
				
			case SOUND_MIXER_CAPS:
				val = SOUND_CAP_EXCL_INPUT;
				break;
				
			default: // read a specific mixer 
				i = _IOC_NR(cmd);
				
				if ( ! supported_mixer(card,i)) 
					return -EINVAL;
				
				val = card->mix.mixer_state[i];
				break;
		}
		
		return put_user(val,(int *)arg);
	}
	
	if (_IOC_DIR(cmd) != (_IOC_WRITE|_IOC_READ))
		return -EINVAL;
	
	card->mix.modcnt++;
	
	get_user_ret(val, (int *)arg, -EFAULT);
	
	switch (_IOC_NR(cmd)) 
	{
		case SOUND_MIXER_RECSRC: // Arg contains a bit for each recording source 
			
			if (!card->mix.recmask_io) return -EINVAL;
			if(!val) return 0;
			if(! (val &= card->mix.record_sources)) return -EINVAL;
			
			spin_lock(&card->lock);
			card->mix.recmask_io(card,0,val);
			spin_unlock(&card->lock);
			return 0;
			
		default:
			i = _IOC_NR(cmd);
			
			if ( ! supported_mixer(card,i)) 
				return -EINVAL;
			
			spin_lock(&card->lock);
			set_mixer(card,i,val);
			spin_unlock(&card->lock);
			
			return 0;
	}
}

static loff_t cx5530_llseek(struct file *file, loff_t offset, int origin)
{
	printk("cx5530_llseek:\n");
	return -ESPIPE;
}

static int cx5530_open_mixdev(struct inode *inode, struct file *file)
{
	struct cx5530_card *card = devs;
	
	file->private_data = card;
	MOD_INC_USE_COUNT;
	
	return 0;
}

static int cx5530_release_mixdev(struct inode *inode, struct file *file)
{
	MOD_DEC_USE_COUNT;
	return 0;
}

static int cx5530_ioctl_mixdev(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
	struct cx5530_card *card = (struct cx5530_card *)file->private_data;
	
	return mixer_ioctl(card, cmd, arg);
}


#if LINUX_22
static struct file_operations cx5530_mixer_fops = 
{
	&cx5530_llseek,
	NULL,  //read 
	NULL,  // write 
	NULL,  // readdir 
	NULL,  // poll 
	&cx5530_ioctl_mixdev,
	NULL,  // mmap 
	&cx5530_open_mixdev,
	NULL,   // flush 
	&cx5530_release_mixdev,
	NULL,  // fsync 
	NULL,  // fasync 
	NULL,  // check_media_change 
	NULL,  // revalidate 
	NULL,  // lock 
};
#else
static struct file_operations cx5530_mixer_fops = 
{
	owner:		THIS_MODULE,
	llseek:	cx5530_llseek,
	ioctl:		cx5530_ioctl_mixdev,
	open:		cx5530_open_mixdev,
	release:	cx5530_release_mixdev,
};
#endif

static int cx5530_open(struct inode *inode, struct file *file)
{
	struct          cx5530_card *c = devs;
	struct          cx5530_state *s = NULL;
	
	STARTUP_printk("cx5530_open - BEGIN\n");
	
	s=&c->state;
	
	file->private_data = s;
	
	WaveOpened=FALSE;
	pWaveHeader_HEAD=NULL;
	pWaveHeader_TAIL=NULL;
	BytesRecorded=0;
	
	while (s->open_mode & file->f_mode) 
	{
		if (file->f_flags & O_NONBLOCK) 
			return -EWOULDBLOCK;
		
		interruptible_sleep_on(&s->open_wait);
		
		if (signal_pending(current))
			return -ERESTARTSYS;
	}
	
	s->open_mode |= file->f_mode & (FMODE_READ | FMODE_WRITE);
	
	MOD_INC_USE_COUNT;
	
	STARTUP_printk("cx5530_open - END\n");
	
	return 0;
}

static ssize_t cx5530_read(struct file *file, char *buffer, size_t count, loff_t *ppos)
{
	unsigned long 			BytesToReadFromThisBuffer;
	int 						cnt;
	unsigned char 			*TempBuffer = NULL;
	DECLARE_WAITQUEUE(wait,current);
	
	if (ppos != &file->f_pos)
		return -ESPIPE;
	if (!access_ok(VERIFY_WRITE, buffer, count))
		return -EFAULT;
	if(!(TempBuffer = kmalloc(count,GFP_KERNEL)))
		return -ENOMEM;
	
	M_printk("READ - count:%d --------------------------------\n", count);
	
	memset(TempBuffer,0,count);
	
	if (!WaveOpened)
	{
		WaveDirection=WAPI_IN;
		cx5530_WaveOpen(count);
	}
	
	//READ_printk("count: %d  FragSizeInBytes:%d BytesAllocated:%d\n", count, FragSizeInBytes, BytesAllocated);
	
	while(BytesToUse()<(8*FragSizeInBytes) || BytesToUse()<(2*count)) 
	{
		if (!AddNode(FragSizeInBytes))
			return -EFAULT;
	}	
	
	if(!WaveStarted)
	{	
		StartDMA();
	}
	
	while(BytesRecorded<count && WaveStarted && !(file->f_flags & O_NONBLOCK))
	{
		if (!EndOfBuffer)
		{
			add_wait_queue(&REC_wait, &wait);
			M_printk("Waiting...");
			
			while (!EndOfBuffer && WaveStarted)
			{
				current->state=TASK_INTERRUPTIBLE;
				schedule_timeout(HZ);
			}
			EndOfBuffer=FALSE;
			remove_wait_queue(&REC_wait,&wait);
			M_printk("Finished Fragment..\n");
		}	
		EndOfBuffer=FALSE;
		
		READ_printk("On While: BytesRecorded:%d   BytesAllocated:%d\n", BytesRecorded, BytesAllocated);
		if (!AddNode(FragSizeInBytes))
			return -EFAULT;
	}
	
	cnt=0;
	
	while(cnt<count && WaveStarted)
	{
		if(pWaveHeader_HEAD!=NULL)
		{
			BytesToReadFromThisBuffer=pWaveHeader_HEAD->dwBytesRecorded - pWaveHeader_HEAD->dwBytesRead;
			
			if (BytesToReadFromThisBuffer>=(count-cnt))
				BytesToReadFromThisBuffer=(count-cnt);
			
			M_printk("Copying %d bytes from %08X+%d to TempBuffer(%08X Size:%d)+cnt(%d)\n", 
				BytesToReadFromThisBuffer, pWaveHeader_HEAD, pWaveHeader_HEAD->dwBytesRead, TempBuffer, count, cnt);	
			
			memcpy(TempBuffer+cnt, pWaveHeader_HEAD->lpData+pWaveHeader_HEAD->dwBytesRead, BytesToReadFromThisBuffer);
			
			pWaveHeader_HEAD->dwBytesRead+=BytesToReadFromThisBuffer;
			cnt+=BytesToReadFromThisBuffer;
			
			BytesRecorded-=BytesToReadFromThisBuffer;
			
			while (pWaveHeader_HEAD != NULL 
				&& pWaveHeader_HEAD != pWaveHeaderLastBuffered 
				&& pWaveHeader_HEAD->dwBytesRead >= pWaveHeader_HEAD->dwBufferLength)
				RemoveHEAD();					
		}
		else
		{			
			copy_to_user(buffer, TempBuffer, cnt);
			M_printk("pWaveHeader_HEAD is NULL - Returning to user\n");
			BytesRecorded-=cnt;
			kfree(TempBuffer);
			return cnt;
		}
	}
	
	//memset(TempBuffer,0,count);
	
	if (copy_to_user(buffer, TempBuffer, count)!=0) 
	{
		printk("Failed copying buffer to user !!!!!!!!!!!  \n");
		return -EFAULT;
	}
	
	if(TempBuffer) 
		kfree(TempBuffer);
	
	return count;	
}

static ssize_t 
cx5530_write(struct file *file, const char *buffer, size_t count, loff_t *ppos)
{
	ssize_t    ret;
	DECLARE_WAITQUEUE(wait,current);
	PWAVEHDR	pwh;
	
	M_printk("WRITE - count:%d  ***************: BytesAllocated:%d \n", count, BytesAllocated);
	
	if (!count)
		return count;
	
	if (ppos != &file->f_pos)
		ret = -ESPIPE;
	else if (!access_ok(VERIFY_READ, buffer, count))
		ret = -EFAULT;
	else
	{		
		if (!WaveOpened)
		{
			WaveDirection=WAPI_OUT;
			cx5530_WaveOpen(count);
		}
		
		if(!(pwh=AddNode(count)))
			return -EFAULT;
		
		if (copy_from_user(pwh->lpData, buffer, count))
		{
			printk("Failed copying from user !!!!!!!!!!!  \n");
			return -EFAULT;
		}
		
		if(!WaveStarted)
		{
			if(BytesAllocated>=((NUMBER_OF_FRAGS_BEFORE_WAIT-2)*FragSizeInBytes))
				StartDMA();
		}
		else
		{
			WRITE_printk("WAVE IS STARTED !!!!!: BytesAllocated:%d \n", BytesAllocated);
			if (file->f_flags & O_NONBLOCK) 
				return count;
			
			while (pWaveHeader_HEAD != NULL 
				&& pWaveHeader_HEAD->dwBytesRecorded >= pWaveHeader_HEAD->dwBufferLength
				&& pWaveHeader_HEAD != pWaveHeaderLastBuffered 
				&& BytesAllocated>0)
				RemoveHEAD();
			
			while(BytesToUse()>(NUMBER_OF_FRAGS_BEFORE_WAIT*FragSizeInBytes) && WaveStarted) 
			{
				if (!EndOfBuffer)
				{
					add_wait_queue(&PLAY_wait, &wait);
					WRITE_printk("Waiting...");
					while (!EndOfBuffer && WaveStarted)
					{
						current->state=TASK_INTERRUPTIBLE;
						schedule_timeout(HZ);
					}
					remove_wait_queue(&PLAY_wait,&wait);
					READ_printk("Finished Fragment..");
				}
				
				EndOfBuffer=FALSE;
			}
		}
		
		ret=count;
	}
	
	M_printk("cx5530_WRITE *******  END\n");	
	return ret;
}

void StartDMA (void)
{
	M_printk("STARTING DMA ***\n");
	PrintGlobalVariables();
	pWaveHeaderLastBuffered=DURAUDIO_WaveStart(pWaveHeader_HEAD,DMABufferSize);
	EndOfBuffer=FALSE;		
	
	WaveStarted=TRUE;
}

unsigned int BytesToUse(void)
{
	PWAVEHDR		pwh;
	unsigned int  BytesUsed;
	
	pwh=pWaveHeader_HEAD;
	BytesUsed=0;
	
	while(pwh!=NULL)
	{      
		BytesUsed+=pwh->dwBytesRecorded;
		pwh=pwh->lpNext;                  
	}      
	
	return (BytesAllocated-BytesUsed);
}

void CalculateBufferSize(size_t count)
{
	if(WaveFormat.nSamplesPerSec 
		&& WaveFormat.nChannels 
		&& WaveFormat.wBitsPerSample 
		&& !WaveOpened)
	{
		DMABufferSize=OSSFragSize * ( (32 / WaveFormat.nChannels) / WaveFormat.wBitsPerSample);
		
		if (WaveFormat.nSamplesPerSec>=22050 || DMABufferSize>DMA_BUFFER_SIZE)
		{
			DMABufferSize=DMA_BUFFER_SIZE;	
			FragSizeInBytes=DMABufferSize/( (32 / WaveFormat.nChannels) / WaveFormat.wBitsPerSample);
		}
		
		IOCTL_printk("CalculateBufferSize: DMABufferSize:%d  FragSizeInBytes:%d\n", (unsigned int) DMABufferSize, (unsigned int) FragSizeInBytes);		
	}
}

void cx5530_WaveOpen(size_t count)
{
	CalculateBufferSize(count);
	
	DURAUDIO_WaveOpen(WaveDirection, &WaveFormat, 0);
	WaveOpened=TRUE;
	WaveStarted=FALSE;
	
	pWaveHeader_HEAD=NULL;
	pWaveHeader_TAIL=NULL;
	EmptyList();
	
	BytesRecorded=0;
	BytesAllocated=0;
	EndOfBuffer=FALSE;
}

void EmptyList(void)
{
	STARTUP_printk("EmptyList***\n");
	
	while (pWaveHeader_HEAD!=NULL)
		RemoveHEAD();	
	
	pWaveHeader_TAIL=NULL;
	pWaveHeaderLastBuffered=NULL;	
	BytesAllocated=0;
	BytesRecorded=0;
	STARTUP_printk("EmptyList***   END\n");	
}

void RemoveHEAD(void)
{
	PWAVEHDR   pwh;
	
	pwh=pWaveHeader_HEAD->lpNext;
	
	kfree((char *)   pWaveHeader_HEAD->lpData);
	kfree((PWAVEHDR) pWaveHeader_HEAD);	
	
	BytesAllocated-=pWaveHeader_HEAD->dwBufferLength;
	
	ALLOC_printk("REMOVED - %08X  BytesAllocated:%d  dwBytesRecorded:%d    dwBufferLength:%d\n", (unsigned int) pWaveHeader_HEAD, BytesAllocated, pWaveHeader_HEAD->dwBytesRecorded, pWaveHeader_HEAD->dwBufferLength);
	pWaveHeader_HEAD=pwh;
	
	if (pWaveHeader_HEAD==NULL)
		pWaveHeader_TAIL=NULL;
}

static unsigned int cx5530_poll(struct file *file, struct poll_table_struct *wait)
{
	printk("cx5530_poll:\n");
	return 0;
}

static int cx5530_mmap(struct file *file, struct vm_area_struct *vma)
{
	printk("cx5530_mmap:\n");
	return 0;
}

static int cx5530_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
	audio_buf_info        abinfo;
	int                   val; 
	
	//IOCTL_printk("cmd:%08X   CX5530: cx5530_ioctl: ", cmd);
	
	switch (cmd) 
	{
		case OSS_GETVERSION:
			IOCTL_printk("OSS_GETVERSION\n");
			return put_user(SOUND_VERSION, (int *)arg);
			
		case SNDCTL_DSP_SYNC:
			while(WaveOpened) 
			{
				if(WaveDirection==WAPI_OUT)
					interruptible_sleep_on_timeout(&PLAY_wait, HZ);
				else
					interruptible_sleep_on_timeout(&REC_wait, HZ);
			}
			
			IOCTL_printk("SNDCTL_DSP_SYNC\n");
			return 0;
			
		case SNDCTL_DSP_SETDUPLEX:
			IOCTL_printk("SNDCTL_DSP_SETDUPLEX\n");
			return 0;
			
		case SNDCTL_DSP_GETCAPS:
			IOCTL_printk("SNDCTL_DSP_GETCAPS\n");
			return put_user(DSP_CAP_DUPLEX | DSP_CAP_REALTIME | DSP_CAP_TRIGGER | DSP_CAP_MMAP, (int *)arg);
			
		case SNDCTL_DSP_RESET:
			M_printk("SNDCTL_DSP_RESET\n");
			synchronize_irq();
			return 0;
			
		case SNDCTL_DSP_SPEED:
			IOCTL_printk("SNDCTL_DSP_SPEED\n");
			get_user_ret(val, (int *)arg, -EFAULT);
			if (val >= 0) 
			{
				if (file->f_mode & FMODE_READ) 
				{
					IOCTL_printk("RECORD MODE\n");
					DURAUDIO_SetCodecRate ((unsigned long) val);
				}
				if (file->f_mode & FMODE_WRITE) 
				{
					IOCTL_printk("PLAYBACK MODE\n");
					DURAUDIO_SetCodecRate ((unsigned long) val);
				}
			}
			
			WaveFormat.nSamplesPerSec=(unsigned long) val;
			WaveFormat.wFormatTag=WAVE_FORMAT_PCM;
			
			IOCTL_printk("SPEED: %d\n", val);
			
			return put_user(val, (int *)arg);			
			
		case SNDCTL_DSP_STEREO:
			IOCTL_printk("SNDCTL_DSP_STEREO\n");
			
			get_user_ret(val, (int *)arg, -EFAULT);
			if (val)
				WaveFormat.nChannels=2;
			else
				WaveFormat.nChannels=1;
			
			return put_user(val, (int *)arg);
			
		case SNDCTL_DSP_CHANNELS:
			IOCTL_printk("SNDCTL_DSP_CHANNELS\n");
			get_user_ret(val, (int *)arg, -EFAULT);
			
			if (val != 0) 
			{
				if (val >= 2)
					WaveFormat.nChannels=2;
				else
					WaveFormat.nChannels=1;
			}
			IOCTL_printk("WaveFormat.nChannels:%d\n", WaveFormat.nChannels);
			
			return put_user(val, (int *)arg);
			
		case SNDCTL_DSP_GETFMTS: //  Returns a mask 
			IOCTL_printk("SNDCTL_DSP_GETFMTS\n");
			return put_user(AFMT_U8|AFMT_S16_LE, (int *)arg);
			
		case SNDCTL_DSP_SETFMT:  //  Selects ONE fmt
			IOCTL_printk("SNDCTL_DSP_SETFMT\n");
			get_user_ret(val, (int *)arg, -EFAULT);
			
			if (file->f_mode & FMODE_READ) 
			{
				if (val == AFMT_S16_LE)
				{
					IOCTL_printk("RECORD - FORMAT IS 16 BIT\n");
					WaveFormat.wBitsPerSample=16;
				}
				else
				{
					IOCTL_printk("RECORD - FORMAT IS 8 BIT\n");
					WaveFormat.wBitsPerSample=8;
				}            
			}
			else
			{
				if (val == AFMT_S16_LE)
				{
					IOCTL_printk("PLAYBACK - FORMAT IS 16 BIT\n");
					WaveFormat.wBitsPerSample=16;
				}
				else
				{
					IOCTL_printk("PLAYBACK - FORMAT IS 8 BIT\n");
					WaveFormat.wBitsPerSample=8;
				}
			}
			
			return put_user(val, (int *)arg);
			
		case SNDCTL_DSP_POST:
			IOCTL_printk("SNDCTL_DSP_POST\n");
			return 0;
			
		case SNDCTL_DSP_GETTRIGGER:
			IOCTL_printk("SNDCTL_DSP_GETTRIGGER\n");
			
			val = 0;
			return put_user(val, (int *)arg);
			
		case SNDCTL_DSP_SETTRIGGER:
			IOCTL_printk("SNDCTL_DSP_SETTRIGGER\n");
			get_user_ret(val, (int *)arg, -EFAULT);
			if (file->f_mode & FMODE_READ) 
			{
				if (val & PCM_ENABLE_INPUT) 
				{
					if (WaveStarted)
						return 0;
				}
				else
				{
					IOCTL_printk("STOPPING ***************************************\n");
					if (WaveStarted)
						DURAUDIO_WaveStop();
				}
			}
			
			return 0;
			
		case SNDCTL_DSP_GETOSPACE:
			
			IOCTL_printk("SNDCTL_DSP_GETOSPACE: ");
			
			if (!(file->f_mode & FMODE_WRITE))
				return -EINVAL;
			
			CalculateBufferSize(0);	
			
			abinfo.fragsize   = FragSizeInBytes;
			abinfo.fragstotal = NUMBER_OF_FRAGS_BEFORE_WAIT;
			abinfo.bytes      = (NUMBER_OF_FRAGS_BEFORE_WAIT*FragSizeInBytes)-BytesToUse(); 									
			abinfo.fragments  = abinfo.bytes/FragSizeInBytes;    
			
			IOCTL_printk("fragsize:%d, bytes:%d, fragstotal:%d, fragments:%d\n", abinfo.fragsize, abinfo.bytes, abinfo.fragstotal, abinfo.fragments);
			
			return copy_to_user((void *)arg, &abinfo, sizeof(abinfo)) ? -EFAULT : 0;
			
		case SNDCTL_DSP_GETISPACE:
			
			IOCTL_printk("SNDCTL_DSP_GETISPACE\n");
			
			if (!(file->f_mode & FMODE_READ))
				return -EINVAL;
			
			abinfo.fragsize   = FragSizeInBytes;
			abinfo.fragstotal = NUMBER_OF_FRAGS_BEFORE_WAIT;
			abinfo.bytes      = (NUMBER_OF_FRAGS_BEFORE_WAIT*FragSizeInBytes)-BytesToUse(); 	
			abinfo.fragments  = abinfo.bytes/FragSizeInBytes;        
			
			IOCTL_printk("fragsize:%d, bytes:%d, fragstotal:%d, fragments:%d\n", abinfo.fragsize, abinfo.bytes, abinfo.fragstotal, abinfo.fragments);
			
			return copy_to_user((void *)arg, &abinfo, sizeof(abinfo)) ? -EFAULT : 0;
			
		case SNDCTL_DSP_NONBLOCK:
			IOCTL_printk("SNDCTL_DSP_NONBLOCK\n");
			file->f_flags |= O_NONBLOCK;
			return 0;
			
		case SNDCTL_DSP_GETODELAY:
			IOCTL_printk("SNDCTL_DSP_GETODELAY:  ");
			if (!(file->f_mode & FMODE_WRITE))
				return -EINVAL;
			
			//if (WaveStarted)
			val = BytesToUse();
			//else
			//val=0;
			
			IOCTL_printk("%d\n", val);
			return put_user(val, (int *)arg);
			
		case SNDCTL_DSP_GETIPTR:
			IOCTL_printk("SNDCTL_DSP_GETIPTR\n");
			
			if (!(file->f_mode & FMODE_READ))
				return -EINVAL;
			return -EFAULT;
			
		case SNDCTL_DSP_GETOPTR:
			IOCTL_printk("SNDCTL_DSP_GETOPTR\n");
			if (!(file->f_mode & FMODE_WRITE))
				return -EINVAL;
			return -EFAULT;
			
		case SNDCTL_DSP_GETBLKSIZE:
			IOCTL_printk("SNDCTL_DSP_GETBLKSIZE\n");
			IOCTL_printk("FragSizeInBytes: %d\n", FragSizeInBytes);
			return put_user(FragSizeInBytes, (int *)arg);
			
		case SNDCTL_DSP_SETFRAGMENT:
			IOCTL_printk("SNDCTL_DSP_SETFRAGMENT\n");
			get_user_ret(val, (int *)arg, -EFAULT);
			
			FragSizeInBytes=1;
			FragSizeInBytes<<=(val&0x0000FFFF);
			OSSFragSize=FragSizeInBytes;
			IOCTL_printk("FragSizeInBytes=%d *****************************************\n", FragSizeInBytes);
			return 0;
			
		case SNDCTL_DSP_SUBDIVIDE:
			IOCTL_printk("SNDCTL_DSP_SUBDIVIDE\n");
			return -EFAULT;
			
		case SOUND_PCM_READ_RATE:
			IOCTL_printk("SOUND_PCM_READ_RATE\n");
			return put_user(WaveFormat.nSamplesPerSec,(int *)arg);
			
		case SOUND_PCM_READ_CHANNELS:
			IOCTL_printk("SOUND_PCM_READ_CHANNELS\n");
			return put_user(WaveFormat.nChannels,(int *)arg);
			
		case SOUND_PCM_READ_BITS:
			IOCTL_printk("SOUND_PCM_READ_BITS\n");
			return put_user(WaveFormat.wBitsPerSample,(int *)arg);
			
		case SOUND_PCM_WRITE_FILTER:
		case SNDCTL_DSP_SETSYNCRO:
		case SOUND_PCM_READ_FILTER:
			IOCTL_printk("SOUND_PCM_READ_FILTER\n");
			
			return -EINVAL;
			
	}
	return -EINVAL;
}

PWAVEHDR AddNode(unsigned long size)
{
	PWAVEHDR   pwh;
	
	pwh=(PWAVEHDR) kmalloc(sizeof(WAVEHDR),GFP_KERNEL|GFP_DMA);
	pwh->lpData=(char *) kmalloc(size,GFP_KERNEL|GFP_DMA);
	
	if ((pwh==NULL) || (pwh->lpData==NULL))
	{
		printk("Error allocating memory!!!  BytesAllocated:%d \n", (int) BytesAllocated);
		return FALSE;
	}			
	
	pwh->dwBufferLength=(unsigned long) size;
	pwh->dwBytesRecorded=0;
	pwh->dwBytesRead=0;
	pwh->lpNext=NULL;
	
	if(pWaveHeader_TAIL==NULL)
	{
		M_printk("First time:  TAIL is null  pWaveHeader_HEAD:%08X,  pWaveHeader_TAIL:%08X\n", pWaveHeader_HEAD, pWaveHeader_TAIL);
		pWaveHeader_HEAD=pwh;
		pWaveHeader_TAIL=pwh;
	}
	else
	{
		pWaveHeader_TAIL->lpNext=pwh;
		pWaveHeader_TAIL=pwh;
	}
	
	BytesAllocated+=size;
	
	M_printk("ALLOCATED: %08X     Size: %d     BytesAllocated:%ld\n", (unsigned int) pwh, (unsigned int) size, BytesAllocated);
	
	return pwh;
}

static int cx5530_release(struct inode *inode, struct file *file)
{
	struct cx5530_state *s = (struct cx5530_state *)file->private_data;
	
	M_printk(KERN_WARNING "cx5530_release\n");
	
	if (WaveOpened)
	{
		if(WaveDirection==WAPI_OUT)
		{
			if(!WaveStarted)
				StartDMA();
			
			while(WaveOpened) 
				interruptible_sleep_on_timeout(&PLAY_wait, HZ);
		}
		else
		{
			while(WaveStarted) 
			{
				interruptible_sleep_on_timeout(&REC_wait, HZ);
			}
			DURAUDIO_WaveClose();
			WaveOpened=FALSE;
			EmptyList();
		}	
		
	}
	
	s->open_mode &= (~file->f_mode) & (FMODE_READ|FMODE_WRITE);
	
	wake_up(&s->open_wait);
	
	MOD_DEC_USE_COUNT;
	
	return 0;
}

#if LINUX_22
static struct file_operations cx5530_audio_fops = 
{
	&cx5530_llseek,
	&cx5530_read,
	&cx5530_write,
	NULL,  // readdir 
	&cx5530_poll,
	&cx5530_ioctl,
	&cx5530_mmap,
	&cx5530_open,
	NULL,   // flush 
	&cx5530_release,
	NULL,  // fsync 
	NULL,  // fasync 
	NULL,  // check_media_change 
	NULL,  // revalidate 
	NULL,  // lock 
};
#else
static struct file_operations cx5530_audio_fops = 
{
	owner:		THIS_MODULE,
	llseek:	cx5530_llseek,
	read:		cx5530_read,
	write:		cx5530_write,
	poll:		cx5530_poll,
	ioctl:		cx5530_ioctl,
	mmap:           cx5530_mmap,
	open:		cx5530_open,
	release:	cx5530_release,
};
#endif

static int 
cx5530_install(struct pci_dev *pcidev, int card_type)
{
	int iobase;
	struct cx5530_card *card;
	struct cx5530_state *s;
	
	FragSizeInBytes=OSSFragSize=DMA_BUFFER_SIZE;
	CalculateBufferSize(0);
	
	BytesAllocated=0;
	BytesRecorded=0;
	WaveStarted=0;
	WaveOpened=0;
	
	//
	//      Default format (for applications that don't set it before start recording/playing
	//
	WaveFormat.nSamplesPerSec=44100;
	WaveFormat.nChannels=2;
	WaveFormat.wBitsPerSample=16;
	
	STARTUP_printk(KERN_WARNING "cx5530_install - BEGIN\n");
	
	if(((pcidev->class >> 8) & 0xffff) != PCI_CLASS_MULTIMEDIA_AUDIO)
		return 0;
	
	iobase = SILLY_PCI_BASE_ADDRESS(pcidev); 
	
#ifdef LINUX_22
	if(check_region(iobase, 256))
	{
		printk(KERN_WARNING "cx5530: can't allocate 256 bytes I/O at 0x%4.4x\n", iobase);
		return 0;
	}
#endif
	
	// just to be sure 
	pci_set_master(pcidev);
	
	card = kmalloc(sizeof(struct cx5530_card), GFP_KERNEL);
	if(card == NULL)
	{
		printk(KERN_WARNING "cx5530: out of memory\n");
		return 0;
	}
	
	memset(card, 0, sizeof(*card));
	memcpy(&card->pcidev,pcidev,sizeof(card->pcidev));
	
#ifdef CONFIG_APM
	if (apm_register_callback(cx5530_apm_callback)) {
		printk(KERN_WARNING "cx5530: apm suspend might not work.\n");
	}
#endif
	
	card->iobase = iobase;
	card->card_type = TYPE_CX5530;
	card->irq = 5;
	spin_lock_init(&card->lock);
	devs = card;
	
	s=&card->state;
	s->card = card;
	
	spin_lock_init(&LockWave);
	
	init_waitqueue_head(&s->open_wait);
	
	spin_lock_init(&s->lock);
	
	s->dev_audio = register_sound_dsp(&cx5530_audio_fops, -1);
	
	if (!DURAUDIO_Initialize(card->irq, iobase, 1))
	{
		printk("cx5530: couldn't Initialize the Audio device!\n");
		return 0;
	}
	cx5530_ac97_init(card,iobase);
	
	if ((card->dev_mixer = register_sound_mixer(&cx5530_mixer_fops, -1)) < 0) 
	{
		printk("cx5530: couldn't register mixer!\n");
	} 
	else 
	{
		memcpy(card->mix.mixer_state,mixer_defaults,sizeof(card->mix.mixer_state));
		mixer_push_state(card);
	}
	
	if(request_irq(card->irq, cx5530_interrupt, SA_INTERRUPT, card_names[card_type], card))
	{
		printk(KERN_ERR "cx5530: unable to allocate irq %d,\n", card->irq);
		
		unregister_sound_mixer(card->dev_mixer);
		
		if(s->dev_audio != -1)
			unregister_sound_dsp(s->dev_audio);
		
#ifdef LINUX_22
		release_region(card->iobase, 256);      
#endif
		kfree(card);
		return 0;
	}
	
	STARTUP_printk(KERN_WARNING "cx5530_install - END\n");
	
	return 1; 
}

int init_module(void)
{
	struct pci_dev *pcidev = NULL;
	int foundone = 0;
	
	STARTUP_printk(KERN_WARNING "init_module - BEGIN\n");
	
	if (!pci_present())   //   No PCI bus in this machine! 
		return -ENODEV;
	STARTUP_printk(KERN_INFO "CX5530: version " DRIVER_VERSION " time " __TIME__ " " __DATE__ "\n");
	
	pcidev = NULL;
	
#ifdef CONFIG_APM
	init_waitqueue_head(&suspend_queue);
#endif
	init_waitqueue_head(&REC_wait);
	init_waitqueue_head(&PLAY_wait);
	
	//
	//           Looking for CX5530 Device
	//
	while((pcidev = pci_find_device(CYRIX_VENDOR_ID, CX5530_DEV_ID, pcidev))!=NULL) 
	{
		if (cx5530_install(pcidev, 0))
			foundone=1;
		else
			return -ENOMEM;      
	}
	
	//
	//          Looking for SC1200 device          
	//
	while((pcidev = pci_find_device(NATIONAL_VENDOR_ID, SC1200_DEV_ID, pcidev))!=NULL) 
	{
		if (cx5530_install(pcidev, 0))
			foundone=1;
		else
			return -ENOMEM;      
	}
	
	if( ! foundone ) 
	{
		printk("CX5530: no devices found.\n");
		return -ENODEV;
	}
	else
	{
		STARTUP_printk(KERN_ERR "cx5530: Found PCI device\n");  
		//STARTUP_printk(KERN_ERR "cx5530: IRQ=%d\n", pcidev->irq);		
	}
	
	STARTUP_printk(KERN_WARNING "init_module - END\n");
	
	return 0;
}

#ifdef MODULE
MODULE_AUTHOR("Mario Raposo - National Semiconductor");
MODULE_DESCRIPTION("CX5530 - Unicorn Audio Driver");
#ifdef M_DEBUG
MODULE_PARM(debug,"i");
#endif
MODULE_PARM(dsps_order,"i");

void cleanup_module(void)
{
	struct cx5530_state *cx5530;
	
	STARTUP_printk("Cleaning up...\n");
	
	DURAUDIO_Deinitialize();
	
	EmptyList();
	
#ifdef CONFIG_APM
	apm_unregister_callback(cx5530_apm_callback);
#endif
	free_irq(devs->irq, devs);
	unregister_sound_mixer(devs->dev_mixer);
	
	cx5530 = &devs->state;
	unregister_sound_dsp(cx5530->dev_audio);
	
#ifdef LINUX_22
	release_region(devs->iobase, 256);
#endif
	kfree(devs);
	
	STARTUP_printk("CX5530: After unloading\n");
}

#endif /* MODULE */


#ifdef CONFIG_APM

void
check_suspend(void)
{
   // printk("CX5530: check_suspend\n");
}

static int 
cx5530_suspend(void)
{
   //printk("CX5530: cx5530_suspend\n");
	return 0;
}
static int 
cx5530_resume(void)
{
   //printk("CX5530: cx5530_resume\n");
	return 0;
}

int 
cx5530_apm_callback(apm_event_t ae) 
{
   //printk("CX5530: cx5530_apm_callback\n");
	return 0;
}
#endif

