/*---------------------------------------------------------------------------
 * nsc_curs.c
 *
 * This file contains the routines to support the hardware cursor.  It makes
 * use of the Durango graphics software support package from National.
 *
 * Originally ported from the Cyrix driver developed by Annius Groenink.
 *
 * Copyright (c) 1999-2000 National Semicondictor.
 *---------------------------------------------------------------------------
 */

/* XFREE86 HEADER FILES */

#include "xf86.h"
#include "xf86Priv.h"
#include "vga.h"
#include "mipointer.h"
#include "cursorstr.h"
#include "servermd.h"

/* DURANGO HEADER FILE */

#include "gfx_rtns.h"

/* LOCAL ROUTINE DEFINITIONS */

static Bool GeodeRealizeCursor();
static Bool GeodeUnrealizeCursor();
static void GeodeSetCursor();
static void GeodeMoveCursor();
static void GeodeRecolorCursor();

/* FUNCTION POINTERS */

static miPointerSpriteFuncRec geodePointerSpriteFuncs =
{
   GeodeRealizeCursor,
   GeodeUnrealizeCursor,
   GeodeSetCursor,
   GeodeMoveCursor,
};

extern miPointerScreenFuncRec xf86PointerScreenFuncs;
extern xf86InfoRec xf86Info;

/* XAA PROVIDES AN ARRAY FOR BIT SWAPPING */

extern unsigned char byte_reversed[256];

/* LOCAL VARIABLES */

static int geodeCursGeneration = -1;
static int geodeXhotspot = 0;
static int geodeYhotspot = 0;
static CursorPtr geodeSavedCursor;

/* EXTERNAL DRIVER VARIABLES */

extern unsigned long geodeCursorOffset;

#define MAX_CURS 32

/*---------------------------------------------------------------------------
 * GeodeCursorInit
 *
 * This routine initializes the cursor function pointers.
 *---------------------------------------------------------------------------
 */
Bool GeodeCursorInit(pm, pScr)
     char *pm;
     ScreenPtr pScr;
{
	/* CHECK IF ALREADY DONE */

	if (geodeCursGeneration != serverGeneration) {
		if (!(miPointerInitialize(pScr, &geodePointerSpriteFuncs,
			&xf86PointerScreenFuncs, FALSE)))
               return FALSE;
		pScr->RecolorCursor = GeodeRecolorCursor;
		geodeCursGeneration = serverGeneration;
	}
	return TRUE;
}

/*---------------------------------------------------------------------------
 * GeodeShowCursor
 *
 * This routine enables the hardware cursor.
 *---------------------------------------------------------------------------
 */
void GeodeShowCursor()
{
	/* CALL DURANGO ROUTINE TO ENABLE CURSOR */

	gfx_set_cursor_enable(1);
}

/*---------------------------------------------------------------------------
 * GeodeHideCursor
 *
 * This routine disables the hardware cursor.
 *---------------------------------------------------------------------------
 */
void GeodeHideCursor()
{
	/* CALL DURANGO ROUTINE TO DISABLE CURSOR */

	gfx_set_cursor_enable(0);
}

/*---------------------------------------------------------------------------
 * GeodeRealizeCursor
 *
 * This routine stores a cursor pattern in allocated system memory.  It 
 * translates from the xfree86 bit data into the device dependent format.
 *---------------------------------------------------------------------------
 */
static Bool GeodeRealizeCursor(pScr, pCurs)
     ScreenPtr pScr;
     CursorPtr pCurs;

{
	int line;
	unsigned char *pServMsk, *pmask;
	unsigned char *pServSrc, *psource;
	int w, h;
	unsigned long *ram;
	unsigned long source, mask, blank_bits;
	CursorBitsPtr bits = pCurs->bits;
	int index = pScr->myNum;
	pointer *pPriv = &pCurs->bits->devPriv[index];

	/* CHECK IF ALREADY REALIZED */

	if (pCurs->bits->refcnt > 1)
      return TRUE;

	/* ALLOCATE SYSTEM MEMORY */

	ram = (unsigned long *)xalloc(MAX_CURS * MAX_CURS / 4);
	*pPriv = (pointer) ram;
	if (!ram) return FALSE;

	/* GET POINTERS TO SOURCE AND MASK */

	pServSrc = (unsigned char *)bits->source;
	pServMsk = (unsigned char *)bits->mask;

	h = bits->height;
	if (h > MAX_CURS) h = MAX_CURS;
	w = bits->width;
	if (w > MAX_CURS) w = MAX_CURS;
	blank_bits = (1L << (MAX_CURS - w)) - 1;

	for (line = 0; line < h; line++) 
	{
		/* CREATE 32-BIT AND MASK AND XOR MASK FOR EACH LINE */

		psource = pServSrc;
		pmask = pServMsk;
		source = ((unsigned long) byte_reversed[*psource++]) << 24;
		mask = ((unsigned long) byte_reversed[*pmask++]) << 24;
		if (w > 8)
		{
			source |= ((unsigned long) byte_reversed[*psource++]) << 16;
			mask |= ((unsigned long) byte_reversed[*pmask++]) << 16;
		}
		if (w > 16)
		{
			source |= ((unsigned long) byte_reversed[*psource++]) << 8;
			mask |= ((unsigned long) byte_reversed[*pmask++]) << 8;
		}
		if (w > 24)
		{
			source |= ((unsigned long) byte_reversed[*psource++]);
			mask |= ((unsigned long) byte_reversed[*pmask++]);
		}
		pServSrc += PixmapBytePad(bits->width, 1);
		pServMsk += PixmapBytePad(bits->width, 1);

		/* STORE 32-BIT AND MASK */

		ram[line] = (~mask) | blank_bits;

		/* STORE 32-BIT XOR MASK */
		/* Located in same memory after the AND mask. */

		ram[line+MAX_CURS] = (source & mask) & (~blank_bits);
	}

	/* EMPTY LINES AT THE END */

	while(line < MAX_CURS)
	{
		ram[line] = 0xFFFFFFFF;
		ram[line+MAX_CURS] = 0x00000000;   
		line++;
	}
	return TRUE;
}

/*---------------------------------------------------------------------------
 * GeodeUnrealizeCursor
 *
 * This routine frees the memory used to store the device dependent version
 * of the cursor data.
 *---------------------------------------------------------------------------
 */
static Bool GeodeUnrealizeCursor(pScr, pCurs)
     ScreenPtr pScr;
     CursorPtr pCurs;
{
   pointer priv;
   if (pCurs->bits->refcnt <= 1 &&
     (priv = pCurs->bits->devPriv[pScr->myNum])) {
         xfree(priv);
         pCurs->bits->devPriv[pScr->myNum] = 0x0;
   }
   return TRUE;
}

/*---------------------------------------------------------------------------
 * GeodeLoadCursor
 *
 * This routine loads a cursor pattern into graphics memory.
 *---------------------------------------------------------------------------
 */
static void GeodeLoadCursor(pScr, pCurs, x, y)
     ScreenPtr pScr;
     CursorPtr pCurs;
     int x, y;
{
	int index = pScr->myNum;
	unsigned long *pCursorData;

	/* IN SAMPLE DRIVER - NO IDEA WHAT FOR */

	if (!xf86VTSema) return;
	if (!pCurs) return;

	/* REMEMBER WHICH CURSOR IS LOADED */
   
	geodeSavedCursor = pCurs;
  
	/* DISABLE CURSOR, LOAD POSITION AND COLORS */
	/* Wait until vertical blank to avoid visual anomolies. */
	/* Call driver routines to adjust (x,y) for frame. */

	GeodeHideCursor();
	gfx_wait_vertical_blank();
	GeodeMoveCursor(0, x, y);
	GeodeRecolorCursor(pScr, pCurs, TRUE);
	
	/* COPY CURSOR PATTERN INTO MEMORY */
	/* Do after setting position to make sure they are loaded in time. */
	/* Position is latched at vsync, so set before then. */
	/* Stored all 32-bit per line AND data, followed by all XOR data. */

	pCursorData = (unsigned long *) pCurs->bits->devPriv[index];
	gfx_set_cursor_shape32(geodeCursorOffset, &pCursorData[0], 
		&pCursorData[MAX_CURS]);

	/* OK TO SHOW CURSOR */

	GeodeShowCursor();
}

static void GeodeSetCursor(pScr, pCurs, x, y, generateEvent)
     ScreenPtr pScr;
     CursorPtr pCurs;
     int   x, y;
     Bool  generateEvent;

{
   if (!pCurs) return;
   geodeXhotspot = pCurs->bits->xhot;
   geodeYhotspot = pCurs->bits->yhot;
   GeodeLoadCursor(pScr, pCurs, x, y);
}

/*---------------------------------------------------------------------------
 * GeodeRestoreCursor
 *
 * This routine resotores the shape of the previously loaded cursor.
 *---------------------------------------------------------------------------
 */
void GeodeRestoreCursor(pScr)
     ScreenPtr pScr;
{
   int x, y;
   miPointerPosition(&x, &y);
   GeodeLoadCursor(pScr, geodeSavedCursor, x, y);
}

/*---------------------------------------------------------------------------
 * GeodeMoveCursor
 *
 * This routine sets the position of the hardware cursor.
 *---------------------------------------------------------------------------
 */
static void GeodeMoveCursor(pScr, x, y)
     ScreenPtr pScr;
     int   x, y;
{
	/* IN SAMPLE DRIVER - NO IDEA WHAT FOR */

	if (!xf86VTSema) return;

	/* ADJUST FOR LOCATION OF SCREEN */

	x -= vga256InfoRec.frameX0;
	y -= vga256InfoRec.frameY0;

	/* CALL DURANGO ROUTINE TO SET POSITION */

	gfx_set_cursor_position(geodeCursorOffset, x, y, 
		geodeXhotspot, geodeYhotspot);
}

/*---------------------------------------------------------------------------
 * GeodeRecolorCursor
 *
 * This routine sets the colors of the hardware cursor.
 *---------------------------------------------------------------------------
 */
static void GeodeRecolorCursor(pScr, pCurs, displayed)
     ScreenPtr pScr;
     CursorPtr pCurs;
     Bool displayed;
{
	/* IN SAMPLE DRIVER - NO IDEA WHAT FOR */

   if (!xf86VTSema) return;
   if (!displayed) return;

   /* CALL DURANGO ROUTINE TO SET CURSOR COLORS */

   gfx_set_cursor_colors(((pCurs->backRed << 8) & 0x00FC0000) |
	                     ((pCurs->backGreen) & 0x00000FC00) |
	                     ((pCurs->backBlue >> 8) & 0x000000FC), 
						 ((pCurs->foreRed << 8) & 0x00FC0000) |
	                     ((pCurs->foreGreen) & 0x0000FC00) |
	                     ((pCurs->foreBlue >> 8) & 0x000000FC));
}

/*---------------------------------------------------------------------------
 * GeodeWarpCursor
 *
 * From sample driver - no idea what it does.
 *---------------------------------------------------------------------------
 */
void GeodeWarpCursor(pScr, x, y)
     ScreenPtr pScr;
     int x, y;
{
	miPointerWarpCursor(pScr, x, y);
	xf86Info.currentScreen = pScr;
}

/*---------------------------------------------------------------------------
 * GeodeQueryBestSize
 *
 * This routine reports the limits of the cursor size (32x32).
 *---------------------------------------------------------------------------
 */
void GeodeQueryBestSize(class, pwidth, pheight, pScreen)
     int class;
     unsigned short *pwidth;
     unsigned short *pheight;
     ScreenPtr pScreen;
{
	if (*pwidth > 0) {
		switch (class) {
		case CursorShape:
			if (*pwidth > 32) *pwidth = 32;
			if (*pheight > 32) *pheight = 32;
			break;
        default:
			mfbQueryBestSize(class, pwidth, pheight, pScreen);
			break;
      }
   }
}

/* END OF FILE */

