
/*
 * Rhine register definitions.
 */

#define FET_PAR0	0x00	/* node address 0 to 4 */
#define FET_PAR1	0x04	/* node address 2 to 6 */
#define FET_RXCFG	0x06	/* receiver config register */
#define FET_TXCFG	0x07	/* transmit config register */
#define FET_COMMAND	0x08	/* command register */
#define FET_ISR		0x0C	/* interrupt/status register */
#define FET_IMR		0x0E	/* interrupt mask register */
#define FET_MAR0	0x10	/* multicast hash 0 */
#define FET_MAR1	0x14	/* multicast hash 1 */
#define FET_RXADDR	0x18	/* rx descriptor list start addr */
#define FET_TXADDR	0x1C	/* tx descriptor list start addr */
#define FET_CURRXDESC0	0x20
#define FET_CURRXDESC1	0x24
#define FET_CURRXDESC2	0x28
#define FET_CURRXDESC3	0x2C
#define FET_NEXTRXDESC0	0x30
#define FET_NEXTRXDESC1	0x34
#define FET_NEXTRXDESC2	0x38
#define FET_NEXTRXDESC3	0x3C
#define FET_CURTXDESC0	0x40
#define FET_CURTXDESC1	0x44
#define FET_CURTXDESC2	0x48
#define FET_CURTXDESC3	0x4C
#define FET_NEXTTXDESC0	0x50
#define FET_NEXTTXDESC1	0x54
#define FET_NEXTTXDESC2	0x58
#define FET_NEXTTXDESC3	0x5C
#define FET_CURRXDMA	0x60	/* current RX DMA address */
#define FET_CURTXDMA	0x64	/* current TX DMA address */
#define FET_TALLYCNT	0x68	/* tally counter test register */
#define FET_PHYADDR	0x6C
#define FET_MIISR	0x6D
#define FET_BCR0	0x6E
#define FET_BCR1	0x6F
#define FET_MIICR	0x70
#define FET_MIIADDR	0x71
#define FET_MIIDATA	0x72
#define FET_EECSR	0x74
#define FET_MACREGCSR	0x75
#define FET_GPIO	0x76
#define FET_CFGA	0x78
#define FET_CFGB	0x79
#define FET_CFGC	0x7A
#define FET_CFGD	0x7B
#define FET_MPA_CNT	0x7C
#define FET_CRC_CNT	0x7E
#define FET_Micr0	0x80
#define FET_STICKHW	0x83
#define FET_CAMMSK	0x88
#define FET_CAMC	0x92
#define FET_CAMADD	0x93
#define FET_PHYANAR	0x95 /* for VT3106 */
#define FET_FlowCR0	0x98 /* for VT3106 */
#define FET_FlowCR1	0x99 /* for VT3106 */
#define FET_TxPauseTimer	0x9A
#define FET_WOLCR_CLR	0xA4
#define FET_WOLCG_CLR	0xA7
#define FET_PWRCSR_CLR	0xAC

/* 
 * registers for 3065
*/
#define FET_GFTEST	0x54
#define FET_RFTCMD	0x55
#define FET_TFTCMD	0x56
#define FET_GFSTATUS	0x57
#define FET_RXMISSED	0x7c
#define FET_TALLYCNTCRC	0x7d
#define FET_RXCRCERRS	0x7e
#define FET_MISC_CR0	0x80
#define FET_MISC_CR1	0x81
#define FET_STICKYHW	0x83
#define FET_WOLCRCLR	0xA4
#define FET_WOLCGCLR	0xA7
#define FET_PWRCSRCLR	0xAC

/*
 * RX config bits.
 */
#define FET_RXCFG_RX_ERRPKTS	0x01
#define FET_RXCFG_RX_RUNT	0x02
#define FET_RXCFG_RX_MULTI	0x04
#define FET_RXCFG_RX_BROAD	0x08
#define FET_RXCFG_RX_PROMISC	0x10
#define FET_RXCFG_RX_THRESH	0xE0

#define FET_RXTHRESH_32BYTES	0x00
#define FET_RXTHRESH_64BYTES	0x20
#define FET_RXTHRESH_128BYTES	0x40
#define FET_RXTHRESH_256BYTES	0x60
#define FET_RXTHRESH_512BYTES	0x80
#define FET_RXTHRESH_768BYTES	0xA0
#define FET_RXTHRESH_1024BYTES	0xC0
#define FET_RXTHRESH_STORENFWD	0xE0

/*
 * TX config bits.
 */
#define FET_TXCFG_RSVD0		0x01
#define FET_TXCFG_LB0		0x02
#define FET_TXCFG_LB1		0x04
#define FET_TXCFG_OFSET		0x08
#define FET_TXCFG_RSVD1		0x10
#define FET_TXCFG_TX_THRESH	0xE0

#define FET_TXTHRESH_64BYTES	0x00
#define FET_TXTHRESH_128BYTES	0x20
#define FET_TXTHRESH_256BYTES	0x40
#define FET_TXTHRESH_512BYTES	0x60
#define FET_TXTHRESH_STORENFWD	0x80

/*
 * Command register bits.
 */
#define FET_CMD_INIT		0x0001
#define FET_CMD_START		0x0002
#define FET_CMD_STOP		0x0004
#define FET_CMD_RX_ON		0x0008
#define FET_CMD_TX_ON		0x0010
#define FET_CMD_TDMD		0x0020
#define FET_CMD_RDMD		0x0040
#define FET_CMD_RSVD		0x0080
#define FET_CMD_RX_EARLY	0x0100
#define FET_CMD_TX_EARLY	0x0200
#define FET_CMD_FDX		0x0400
#define FET_CMD_TX_NOPOLL	0x0800
#define FET_CMD_KEYPAG		0x1000
#define FET_CMD_TDMD1		0x2000
#define FET_CMD_RDMD1		0x4000
#define FET_CMD_RESET		0x8000

/*
 * Interrupt status bits.
 */
#define FET_ISR_PRX		0x0001	/* packet rx ok */
#define FET_ISR_PTX		0x0002	/* packet tx ok */
#define FET_ISR_RXE		0x0004	/* packet rx with err */
#define FET_ISR_TXE		0x0008	/* packet tx with error */
#define FET_ISR_TU		0x0010	/* tx buffer underflow */
#define FET_ISR_RU		0x0020	/* no rx buffer available */
#define FET_ISR_BE		0x0040	/* PCI bus error */
#define FET_ISR_CNT		0x0080	/* stats counter oflow */
#define FET_ISR_ERI		0x0100	/* rx early */
#define FET_ISR_ETI		0x0200	/* tx early, for 3043/3071 */
#define FET_ISR_UDFI		0x0200	/* Tx fifo underflow event, for 3065 */
#define FET_ISR_OVFI		0x0400	/* rx FIFO overflow */
#define FET_ISR_PKTRACE		0x0800
#define FET_ISR_RX_NORBF	0x1000
#define FET_ISR_TX_ABTI		0x2000
#define FET_ISR_SRCI		0x4000
#define FET_ISR_KEYI		0x8000	/* for 3043/3071 */
#define FET_ISR_GENI		0x8000	/* for 3065 */

/*
 * Interrupt mask bits.
 */
#define FET_IMR_PRXM		0x0001	/* packet rx ok */
#define FET_IMR_PTXM		0x0002	/* packet tx ok */
#define FET_IMR_RXEM		0x0004	/* packet rx with err */
#define FET_IMR_TXEM		0x0008	/* tx aborted due to excess colls */
#define FET_IMR_TUM		0x0010	/* tx buffer underflow */
#define FET_IMR_RUM		0x0020	/* no rx buffer available */
#define FET_IMR_BEM		0x0040	/* PCI bus error */
#define FET_IMR_CNTM		0x0080	/* stats counter oflow */
#define FET_IMR_ERM		0x0100	/* rx early */
#define FET_IMR_ETM		0x0200	/* MII status change */
#define FET_ISR_UDFM		0x0200	/* Tx fifo underflow event, for 3065 */
#define FET_IMR_OVFM		0x0400	/* rx FIFO overflow */
#define FET_IMR_PRAIM		0x0800
#define FET_IMR_NBFM		0x1000
#define FET_IMR_ABTM		0x2000
#define FET_IMR_SRCM		0x4000
#define FET_IMR_KEYIM		0x8000
#define FET_ISR_GENIM		0x8000	/* for 3065 */

/* define imrshadow */

#define IMRShadow		0x7BFF

/*
 * MII status register.
 */

#define FET_MIISR_SPEED		0x01
#define FET_MIISR_LNKFL		0x02
#define FET_MIISR_MGTREADERR	0x04
#define FET_MIISR_MIIERR	0x08
#define FET_MIISR_PHYOPT	0x10
#define FET_MIISR_MDC_SPEED	0x20
#define FET_MIISR_RSVD		0x40
#define FET_MIISR_GPIO1POLL	0x80

/*
 * MII CSR offset (0x71) register bits.
 */
#define FET_MIIADDR_MIDLE	0x80
#define FET_MIIADDR_MSRCEN	0x40
#define FET_MIIADDR_MDONE	0x20
#define FET_MIIADDR_MAD4  	0x10
#define FET_MIIADDR_MAD3 	0x08
#define FET_MIIADDR_MAD2 	0x04
#define FET_MIIADDR_MAD1 	0x02
#define FET_MIIADDR_MAD0 	0x01

/*
 * MII command register bits.
 */
#define FET_MIICR_CLK		0x01
#define FET_MIICR_DATAOUT	0x02
#define FET_MIICR_DATAIN	0x04
#define FET_MIICR_DIR		0x08
#define FET_MIICR_DIRECTPGM	0x10
#define FET_MIICR_WCMD		0x20
#define FET_MIICR_RCMD		0x40
#define FET_MIICR_MAUTO		0x80

/*
 * EEPROM control bits.
 */
#define FET_EECSR_DATAIN	0x01	/* data out */
#define FET_EECSR_DATAOUT	0x02	/* data in */
#define FET_EECSR_CLK		0x04	/* clock */
#define FET_EECSR_CS		0x08	/* chip select */
#define FET_EECSR_DPM		0x10
#define FET_EECSR_LOAD		0x20
#define FET_EECSR_EMBP		0x40
#define FET_EECSR_EEPR		0x80

#define FET_EECMD_WRITE		0x140
#define FET_EECMD_READ		0x180
#define FET_EECMD_ERASE		0x1c0

/*
 * Bits in the BCR0 register
 */
#define FET_BCR0_MED2		0x80
#define FET_BCR0_LED100M	0x40
#define FET_BCR0_CRFT2		0x20
#define FET_BCR0_CRFT1		0x10
#define FET_BCR0_CRFT0		0x08
#define FET_BCR0_DMAL2		0x04
#define FET_BCR0_DMAL1		0x02
#define FET_BCR0_DMAL0		0x01

/*
 * Bits in the BCR1 register
 */
#define FET_BCR1_MED1		0x80
#define FET_BCR1_MED0		0x40
#define FET_BCR1_CTSF		0x20
#define FET_BCR1_CTFT1		0x10
#define FET_BCR1_CTFT0		0x08
#define FET_BCR1_POT2		0x04
#define FET_BCR1_POT1		0x02
#define FET_BCR1_POT0		0x01

/*
 * Bits in the CFGB register
 */
#define FET_CFGB_QPKTDIS	0x80
#define FET_CFGB_MRLDIS		0x20

/*
 * CFGC control bits.
 */ 
#define FET_CFGC_MEDEN		0x80
#define FET_CFGC_BROPT		0x40
#define FET_CFGC_DLYEN		0x20
#define FET_CFGC_DTSEL		0x10
#define FET_CFGC_BTSEL		0x08
#define FET_CFGC_BPS2		0x04
#define FET_CFGC_BPS1		0x02
#define FET_CFGC_BPS0		0x01

/*
 * CFGD control bits.
 */ 
#define FET_CFGD_GPIOEN		0x80
#define FET_CFGD_DIAG		0x40
#define FET_CFGD_MRDLEN		0x20
#define FET_CFGD_MAGIC		0x10
#define FET_CFGD_CRADOM		0x08
#define FET_CFGD_CAP		0x04
#define FET_CFGD_MBA		0x02
#define FET_CFGD_BAKOPT		0x01

/*
 * Test register bits.
 */
#define FET_TEST_TEST0		0x01
#define FET_TEST_TEST1		0x02
#define FET_TEST_TEST2		0x04
#define FET_TEST_TSTUD		0x08
#define FET_TEST_TSTOV		0x10
#define FET_TEST_BKOFF		0x20
#define FET_TEST_FCOL		0x40
#define FET_TEST_HBDES		0x80

/*
 * Config register bits.
 */
#define FET_CFG_GPIO2OUTENB	0x00000001
#define FET_CFG_GPIO2OUT	0x00000002	/* gen. purp. pin */
#define FET_CFG_GPIO2IN		0x00000004	/* gen. purp. pin */
#define FET_CFG_AUTOOPT		0x00000008	/* enable rx/tx autopoll */
#define FET_CFG_MIIOPT		0x00000010
#define FET_CFG_MMIENB		0x00000020	/* memory mapped mode enb */
#define FET_CFG_JUMPER		0x00000040	/* PHY and oper. mode select */
#define FET_CFG_EELOAD		0x00000080	/* enable EEPROM programming */
#define FET_CFG_LATMENB		0x00000100	/* larency timer effect enb. */
#define FET_CFG_MRREADWAIT	0x00000200
#define FET_CFG_MRWRITEWAIT	0x00000400
#define FET_CFG_RX_ARB		0x00000800
#define FET_CFG_TX_ARB		0x00001000
#define FET_CFG_READMULTI	0x00002000
#define FET_CFG_TX_PACE		0x00004000
#define FET_CFG_TX_QDIS		0x00008000
#define FET_CFG_ROMSEL0		0x00010000
#define FET_CFG_ROMSEL1		0x00020000
#define FET_CFG_ROMSEL2		0x00040000
#define FET_CFG_ROMTIMESEL	0x00080000
#define FET_CFG_RSVD0		0x00100000
#define FET_CFG_ROMDLY		0x00200000
#define FET_CFG_ROMOPT		0x00400000
#define FET_CFG_RSVD1		0x00800000
#define FET_CFG_BACKOFFOPT	0x01000000
#define FET_CFG_BACKOFFMOD	0x02000000
#define FET_CFG_CAPEFFECT	0x04000000
#define FET_CFG_BACKOFFRAND	0x08000000
#define FET_CFG_MAGICKPACKET	0x10000000
#define FET_CFG_PCIREADLINE	0x20000000
#define FET_CFG_DIAG		0x40000000
#define FET_CFG_GPIOEN		0x80000000

/* MISC.CR1 register bits */

#define FET_MISCCR1_FORSRST	0x40
#define FET_MISCCR1_VAUXJMP	0x20
#define FET_MISCCR1_PHYINT	0x02
#define FET_MISCCR1_TIMER1_EN	0x01


/* MISC definitions */
#define W_MAX_TIMEOUT		0x0FFFU
#define FET_TIMEOUT		1000
#define ETHER_ALIGN		2

/*
 * Rhine TX/RX list structure.
 */

struct fet_desc {
	u_int32_t		fet_status;
	u_int32_t		fet_ctl;
	u_int32_t		fet_ptr1;
	u_int32_t		fet_ptr2;
};

#define fet_data		fet_ptr1
#define fet_next		fet_ptr2


#define FET_RXSTAT_RXERR		0x00000001
#define FET_RXSTAT_CRCERR		0x00000002
#define FET_RXSTAT_FRAMEALIGNERR	0x00000004
#define FET_RXSTAT_FIFOOFLOW		0x00000008
#define FET_RXSTAT_GIANT		0x00000010
#define FET_RXSTAT_RUNT			0x00000020
#define FET_RXSTAT_BUSERR		0x00000040
#define FET_RXSTAT_BUFFERR		0x00000080
#define FET_RXSTAT_LASTFRAG		0x00000100
#define FET_RXSTAT_FIRSTFRAG		0x00000200
#define FET_RXSTAT_RLINK		0x00000400
#define FET_RXSTAT_RX_PHYS		0x00000800
#define FET_RXSTAT_RX_BROAD		0x00001000
#define FET_RXSTAT_RX_MULTI		0x00002000
#define FET_RXSTAT_RX_OK		0x00004000
#define FET_RXSTAT_RXLEN		0x07FF0000
#define FET_RXSTAT_RXLEN_EXT		0x78000000
#define FET_RXSTAT_OWN			0x80000000

#define FET_RXBYTES(x)		((x & FET_RXSTAT_RXLEN) >> 16)
#define FET_RXSTAT (FET_RXSTAT_OWN)

#define FET_RXCTL_BUFLEN	0x00000800
#define FET_RXCTL_BUFLEN_EXT	0x00007800
#define FET_RXCTL_CHAIN		0x00008000
#define FET_RXCTL_RX_INTR	0x00800000

#define FET_RXCTL (FET_RXCTL_BUFLEN)

#define FET_TXSTAT_DEFER	0x00000001
#define FET_TXSTAT_COLLCNT	0x0000000F
#define FET_TXSTAT_AQE		0x00000080
#define FET_TXSTAT_ABT		0x00000100
#define FET_TXSTAT_LATECOLL	0x00000200
#define FET_TXSTAT_CRS		0x00000400
#define FET_TXSTAT_UDF		0x00000800
#define FET_TXSTAT_TBUFF	0x00001000
#define FET_TXSTAT_SERR		0x00002000
#define FET_TXSTAT_JABTIMEO	0x00004000
#define FET_TXSTAT_ERRSUM	0x00008000
#define FET_TXSTAT_OWN		0x80000000

#define FET_TXCTL_BUFLEN	0x00000800
#define FET_TXCTL_BUFLEN_EXT	0x00007800
#define FET_TXCTL_TLINK		0x00008000
#define FET_TXCTL_FIRSTFRAG	0x00200000
#define FET_TXCTL_LASTFRAG	0x00400000
#define FET_TXCTL_FINT		0x00800000


#define FET_MAXFRAGS		16
#define FET_RX_LIST_CNT		64
#define FET_TX_LIST_CNT		128
#define FET_MIN_FRAMELEN	60
#define FET_FRAMELEN		1536
#define FET_RXLEN		1520
#define MAX_PACKET_LEN          1514
#define U_CRC_LEN               4
#define U_HEADER_LEN		14
#define FET_TXOWN(x)		x->fet_ptr->fet_status

struct fet_list_data {
	struct fet_desc		fet_rx_list[FET_RX_LIST_CNT];
	struct fet_desc		fet_tx_list[FET_TX_LIST_CNT];
};

struct fet_chain {
	struct fet_desc		*fet_ptr;
	struct mbuf		*fet_mbuf;
	struct fet_chain		*fet_nextdesc;
};

struct fet_chain_onefrag {
	struct fet_desc		*fet_ptr;
	struct mbuf		*fet_mbuf;
	struct fet_chain_onefrag	*fet_nextdesc;
};

struct fet_chain_data {
	struct fet_chain_onefrag	fet_rx_chain[FET_RX_LIST_CNT];
	struct fet_chain		fet_tx_chain[FET_TX_LIST_CNT];

	struct fet_chain_onefrag	*fet_rx_head;

	struct fet_chain		*fet_tx_head;
	struct fet_chain		*fet_tx_tail;
	struct fet_chain		*fet_tx_free;
        u_int16_t			fet_free_tx_count;
};

struct fet_type {
	u_int16_t		fet_vid;
	u_int16_t		fet_did;
	char			*fet_name;
};

struct fet_mii_frame {
	u_int8_t		mii_stdelim;
	u_int8_t		mii_opcode;
	u_int8_t		mii_phyaddr;
	u_int8_t		mii_regaddr;
	u_int8_t		mii_turnaround;
	u_int16_t		mii_data;
};

/*
 * MII constants
 */
#define FET_MII_STARTDELIM	0x01
#define FET_MII_READOP		0x02
#define FET_MII_WRITEOP		0x01
#define FET_MII_TURNAROUND	0x02

#define FET_FLAG_FORCEDELAY	1
#define FET_FLAG_SCHEDDELAY	2
#define FET_FLAG_DELAYTIMEO	3	

/* mode supported */
typedef enum {
    FET_AUTO,
    FET_100_FDX,
    FET_100_HDX,	
    FET_10_FDX,
    FET_10_HDX
} fet_modeinfo_t;    	

/* Supported media types. */
const int fet_media_standard[] = {
	IFM_ETHER|IFM_10_T|IFM_HDX,
	IFM_ETHER|IFM_10_T|IFM_FDX,
	IFM_ETHER|IFM_100_TX|IFM_HDX,
	IFM_ETHER|IFM_100_TX|IFM_FDX,
	IFM_ETHER|IFM_AUTO,
};

/*
 *	revision id
 */
#define	REV_ID_VT3043_E		0x04
#define REV_ID_VT3071_A		0x20
#define REV_ID_VT3071_B		0x21
#define REV_ID_VT3065_A		0x40
#define REV_ID_VT3065_B		0x41
#define REV_ID_VT3065_C		0x42

#if __FreeBSD_version >= 400000
struct fet_softc {
	struct arpcom		arpcom;		/* interface info */
	bus_space_handle_t	fet_bhandle;	/* bus space handle */
	bus_space_tag_t		fet_btag;	/* bus space tag */
	struct resource		*fet_res;
	struct resource		*fet_irq;
	void			*fet_intrhand;
	device_t		fet_miibus;
	struct fet_type		*fet_info;	/* Rhine adapter info */
	u_int8_t		fet_unit;	/* interface number */
	u_int8_t		fet_type;
	u_int8_t		fet_autoneg;
	u_int8_t		fet_full_duplex;
	u_int8_t 		fet_chip_revid;	/* revision of chip */ 	
	fet_modeinfo_t		fet_modeinfo;	/* type of mode */
	struct fet_list_data	*fet_ldata;
	struct fet_chain_data	fet_cdata;
	struct callout_handle	fet_stat_ch;
};

#elif __FreeBSD_version < 400000
struct fet_softc {
	struct arpcom		arpcom;		/* interface info */
	struct ifmedia		ifmedia;	/* media info */
	bus_space_handle_t	fet_bhandle;	/* bus space handle */
	bus_space_tag_t		fet_btag;	/* bus space tag */
	struct fet_type		*fet_info;	/* Rhine adapter info */
	struct fet_type		*fet_pinfo;	/* phy info */
	u_int8_t		fet_unit;	/* interface number */
	u_int8_t		fet_type;
	u_int8_t		fet_phy_addr;	/* PHY address */
	u_int8_t		fet_tx_pend;	/* TX pending */
	u_int8_t		fet_want_auto;
	u_int8_t		fet_autoneg;
	u_int8_t		fet_full_duplex;
	u_int8_t 		fet_chip_revid;	/* revision of chip */ 	
	fet_modeinfo_t		fet_modeinfo;	/* type of mode */
	caddr_t			fet_ldata_ptr;
	struct fet_list_data	*fet_ldata;
	struct fet_chain_data	fet_cdata;
};
#endif

/*
 * register space access macros
 */
#define CSR_WRITE_4(sc, reg, val)	\
	bus_space_write_4(sc->fet_btag, sc->fet_bhandle, reg, val)
#define CSR_WRITE_2(sc, reg, val)	\
	bus_space_write_2(sc->fet_btag, sc->fet_bhandle, reg, val)
#define CSR_WRITE_1(sc, reg, val)	\
	bus_space_write_1(sc->fet_btag, sc->fet_bhandle, reg, val)

#define CSR_READ_4(sc, reg)		\
	bus_space_read_4(sc->fet_btag, sc->fet_bhandle, reg)
#define CSR_READ_2(sc, reg)		\
	bus_space_read_2(sc->fet_btag, sc->fet_bhandle, reg)
#define CSR_READ_1(sc, reg)		\
	bus_space_read_1(sc->fet_btag, sc->fet_bhandle, reg)


/*
 * General constants that are fun to know.
 *
 * vendor ID
 */
#define	VENDORID		0x1106

/*
 * device IDs.
 */
#define	DEVICEID_3043	0x3043
#define DEVICEID_3065	0x3065

/*
 * PCI low memory base and low I/O base register, and
 * other PCI registers.
 */

#define FET_PCI_VENDOR_ID	0x00
#define FET_PCI_DEVICE_ID	0x02
#define FET_PCI_COMMAND		0x04
#define FET_PCI_STATUS		0x06
#define FET_PCI_REVID		0x08
#define FET_PCI_CLASSCODE	0x09
#define FET_PCI_LATENCY_TIMER	0x0D
#define FET_PCI_HEADER_TYPE	0x0E
#define FET_PCI_LOIO		0x10
#define FET_PCI_LOMEM		0x14
#define FET_PCI_BIOSROM		0x30
#define FET_PCI_INTLINE		0x3C
#define FET_PCI_INTPIN		0x3D
#define FET_PCI_MINGNT		0x3E
#define FET_PCI_MINLAT		0x0F
#define FET_PCI_RESETOPT	0x48
#define FET_PCI_EEPROM_DATA	0x4C
#define FET_PCI_MODE            0x50

#define PCI_VENDORID(x)     ((x) & 0xFFFF)
#define PCI_CHIPID(x)       (((x) >> 16) & 0xFFFF)

#define FET_MODE3_MIION         0x04
/* power management registers */
#define FET_PCI_CAPID		0xDC /* 8 bits */
#define FET_PCI_NEXTPTR		0xDD /* 8 bits */
#define FET_PCI_PWRMGMTCAP	0xDE /* 16 bits */
#define FET_PCI_PWRMGMTCTRL	0xE0 /* 16 bits */

#define FET_PSTATE_MASK		0x0003
#define FET_PSTATE_D0		0x0000
#define FET_PSTATE_D1		0x0002
#define FET_PSTATE_D2		0x0002
#define FET_PSTATE_D3		0x0003
#define FET_PME_EN		0x0010
#define FET_PME_STATUS		0x8000

#define PHY_UNKNOWN		6

#define FET_PHYADDR_MIN		0x00
#define FET_PHYADDR_MAX		0x1F

#define PHY_BMCR		0x00
#define PHY_BMSR		0x01
#define PHY_VENID		0x02
#define PHY_DEVID		0x03
#define PHY_ANAR		0x04
#define PHY_LPAR		0x05
#define PHY_ANEXP		0x06

#define PHY_ANAR_NEXTPAGE	0x8000
#define PHY_ANAR_RSVD0		0x4000
#define PHY_ANAR_TLRFLT		0x2000
#define PHY_ANAR_RSVD1		0x1000
#define PHY_ANAR_RSVD2		0x0800
#define PHY_ANAR_RSVD3		0x0400
#define PHY_ANAR_100BT4		0x0200
#define PHY_ANAR_100BTXFULL	0x0100
#define PHY_ANAR_100BTXHALF	0x0080
#define PHY_ANAR_10BTFULL	0x0040
#define PHY_ANAR_10BTHALF	0x0020
#define PHY_ANAR_PROTO4		0x0010
#define PHY_ANAR_PROTO3		0x0008
#define PHY_ANAR_PROTO2		0x0004
#define PHY_ANAR_PROTO1		0x0002
#define PHY_ANAR_PROTO0		0x0001

#define PHY_LPAR_NEXTPAGE	0x8000
#define PHY_LPAR_RSVD0		0x4000
#define PHY_LPAR_TLRFLT		0x2000
#define PHY_LPAR_RSVD1		0x1000
#define PHY_LPAR_RSVD2		0x0800
#define PHY_LPAR_RSVD3		0x0400
#define PHY_LPAR_100BT4		0x0200
#define PHY_LPAR_100BTXFULL	0x0100
#define PHY_LPAR_100BTXHALF	0x0080
#define PHY_LPAR_10BTFULL	0x0040
#define PHY_LPAR_10BTHALF	0x0020
#define PHY_LPAR_PROTO4		0x0010
#define PHY_LPAR_PROTO3		0x0008
#define PHY_LPAR_PROTO2		0x0004
#define PHY_LPAR_PROTO1		0x0002
#define PHY_LPAR_PROTO0		0x0001

#define PHY_ANEXP_LPAUTOABLE	0x0001

/*
 * These are the register definitions for the PHY (physical layer
 * interface chip).
 */
/*
 * PHY BMCR Basic Mode Control Register
 */
#define PHY_BMCR_RESET		0x8000
#define PHY_BMCR_LOOPBK		0x4000
#define PHY_BMCR_SPEEDSEL	0x2000
#define PHY_BMCR_AUTONEGENBL	0x1000
#define PHY_BMCR_RSVD0		0x0800	/* write as zero */
#define PHY_BMCR_ISOLATE	0x0400
#define PHY_BMCR_AUTONEGRSTR	0x0200
#define PHY_BMCR_DUPLEX		0x0100
#define PHY_BMCR_COLLTEST	0x0080
#define PHY_BMCR_RSVD1		0x0040	/* write as zero, don't care */
#define PHY_BMCR_RSVD2		0x0020	/* write as zero, don't care */
#define PHY_BMCR_RSVD3		0x0010	/* write as zero, don't care */
#define PHY_BMCR_RSVD4		0x0008	/* write as zero, don't care */
#define PHY_BMCR_RSVD5		0x0004	/* write as zero, don't care */
#define PHY_BMCR_RSVD6		0x0002	/* write as zero, don't care */
#define PHY_BMCR_RSVD7		0x0001	/* write as zero, don't care */

/* 
 * PHY BMCR register bit
 */
#define PHY_BMCR_RESET		0x8000  /* bit15 */
#define PHY_BMCR_LOOPBK		0x4000  /* bit14 */
#define PHY_BMCR_SPEEDSEL	0x2000  /* bit13 */
#define PHY_BMCR_AUTONEGENBL	0x1000  /* bit12 */
#define PHY_BMCR_RSVD0		0x0800  /* bit11 */
#define PHY_BMCR_ISOLATE	0x0400  /* bit10 */
#define PHY_BMCR_AUTONEGRSTR	0x0200  /* bit9 */
#define PHY_BMCR_DUPLEX		0x0100  /* bit8 */
#define PHY_BMCR_COLLTEST	0x0080  /* bit7 */
#define PHY_BMCR_RSVD1		0x0040  /* bit6 */
#define PHY_BMCR_RSVD2		0x0020  /* bit5 */
#define PHY_BMCR_RSVD3		0x0010  /* bit4 */
#define PHY_BMCR_RSVD4		0x0008  /* bit3 */
#define PHY_BMCR_RSVD5		0x0004  /* bit2 */
#define PHY_BMCR_RSVD6		0x0002  /* bit1 */
#define PHY_BMCR_RSVD7		0x0001  /* bit0 */

/*
 * RESET: 1 == software reset, 0 == normal operation
 * Resets status and control registers to default values.
 * Relatches all hardware config values.
 *
 * LOOPBK: 1 == loopback operation enabled, 0 == normal operation
 *
 * SPEEDSEL: 1 == 100Mb/s, 0 == 10Mb/s
 * Link speed is selected byt his bit or if auto-negotiation if bit
 * 12 (AUTONEGENBL) is set (in which case the value of this register
 * is ignored).
 *
 * AUTONEGENBL: 1 == Autonegotiation enabled, 0 == Autonegotiation disabled
 * Bits 8 and 13 are ignored when autoneg is set, otherwise bits 8 and 13
 * determine speed and mode. Should be cleared and then set if PHY configured
 * for no autoneg on startup.
 *
 * ISOLATE: 1 == isolate PHY from MII, 0 == normal operation
 *
 * AUTONEGRSTR: 1 == restart autonegotiation, 0 = normal operation
 *
 * DUPLEX: 1 == full duplex mode, 0 == half duplex mode
 *
 * COLLTEST: 1 == collision test enabled, 0 == normal operation
 */

/* 
 * PHY, BMSR Basic Mode Status Register 
 */   
#define PHY_BMSR_100BT4		0x8000
#define PHY_BMSR_100BTXFULL	0x4000
#define PHY_BMSR_100BTXHALF	0x2000
#define PHY_BMSR_10BTFULL	0x1000
#define PHY_BMSR_10BTHALF	0x0800
#define PHY_BMSR_RSVD1		0x0400	/* write as zero, don't care */
#define PHY_BMSR_RSVD2		0x0200	/* write as zero, don't care */
#define PHY_BMSR_RSVD3		0x0100	/* write as zero, don't care */
#define PHY_BMSR_RSVD4		0x0080	/* write as zero, don't care */
#define PHY_BMSR_MFPRESUP	0x0040
#define PHY_BMSR_AUTONEGCOMP	0x0020
#define PHY_BMSR_REMFAULT	0x0010
#define PHY_BMSR_CANAUTONEG	0x0008
#define PHY_BMSR_LINKSTAT	0x0004
#define PHY_BMSR_JABBER		0x0002
#define PHY_BMSR_EXTENDED	0x0001

#ifdef __alpha__
#undef vtophys
#define vtophys(va)		alpha_XXX_dmamap((vm_offset_t)va)
#endif
