/*
 *  bates_02      RGB LED strip driver for Sam Bates' creations (v 02)
 *
 *  This application provides PWM drive to GPIO output lines for control of LEDs
 *  to produce accent lighting for one of Sam's glass sculptures.  It is intended
 *  to run in a Teensy 3.1 or equivalent.  It was built with CodeSourcery Lite++
 *  (gcc) tool suite.
 *
 *  Updated from version 01 to provide nine channels of 16-bit PWM (three RGB groups).
 *
 *  Signal		FTM		Dev pin		Teensy pin
 *  ------		---		-------		----------
 *  RED.0		0:0		PTC1			22
 *  GRN.0		0:1		PTC2			23
 *  BLU.0		0:2		PTC3			 9

 *  RED.1		0:3		PTC4			10
 *  GRN.1		0:4		PTD4			 6
 *  BLU.1		0:5		PTD5			20
 *
 *  RED.2		0:6		PTD6			21
 *  GRN.2		0:7		PTD7			 5
 *  BLU.2		1:0		PTA12			 3
 *
 *  PWM is controlled by FTM compare operations.  The FTM channels are
 *  configured for active-low operation; the output is low during ON
 *  time, and switches to high on compare match.
 *
 *  The user assigns a brightness value in the range of 0 to 2000.
 *  This value is scaled to try and compensate for the eye's non-
 *  linearity, using a sqrt(2) function.
 */

#include  <stdio.h>
#include  <string.h>
#include  <stdint.h>
#include  "common.h"
#include  "arm_cm4.h"
#include  "uart.h"
#include  "termio.h"
#include  "pit.h"


#ifndef  FALSE
#define  FALSE   0
#define  TRUE    (!FALSE)
#endif


#define  PROG_NAME_STR		"bates_02"


#define  MAX_CMD_LEN	254
#define  MAX_TOKEN_LEN	32

#define  NUM_EEPROM_SCENES  10

#define	 FTM0_CHNL0		0
#define	 FTM0_CHNL1		1
#define	 FTM0_CHNL2		2
#define  FTM0_CHNL3		3
#define  FTM0_CHNL4		4
#define  FTM0_CHNL5		5


/*
 *  Values associated with the programmable interrupt timer (PIT) subsystem.
 */
#define  PITCHNL			0				/* PIT channel used for our timer */
#define  PITDELAY_USECS		1000			/* PIT delay in usecs (1 ms) */


#define  MAX_PWM_COUNT			4000		/* the PWM value for 100% brightness */
#define  MAX_USER_BRIGHTNESS	2000		/* the user value for 100% brighness */
#define  BRIGHTNESS_SCALER		((MAX_USER_BRIGHTNESS*MAX_USER_BRIGHTNESS)/MAX_PWM_COUNT)


#define  NUM_RGB			3				/* number of complete RGB LEDs supported */

typedef struct				LED
{
	volatile uint16_t		pwm;			// scaled ON-count value written to FTM channel for PWM
	uint32_t				user;			// user-requested brightness value (not scaled)
	int16_t					delta;			// increment used during fade; must be signed!
	uint16_t				target;			// fade target value
	uint8_t					fading;			// TRUE if fade not finished, FALSE if fade done
	uint32_t				*pchnl;			// CxV register address for this LED's PWM
}  LED;


LED							Red[NUM_RGB] =
{
	{ 0,  250,  11,  0,  FALSE,  (uint32_t *)&FTM0_C0V},
	{ 0,  600,   9,  0,  FALSE,  (uint32_t *)&FTM0_C3V},
	{ 0,  600,   9,  0,  FALSE,  (uint32_t *)&FTM0_C6V}
};


LED							Green[NUM_RGB] =
{
	{ 0,  500,  13,  0,  FALSE,  (uint32_t *)&FTM0_C1V},
	{ 0, 1200,  13,  0,  FALSE,  (uint32_t *)&FTM0_C4V},
	{ 0,  600,   9,  0,  FALSE,  (uint32_t *)&FTM0_C7V}
};

LED							Blue[NUM_RGB] =
{
	{ 0, 1000,   7,  0,  FALSE,  (uint32_t *)&FTM0_C2V},
	{ 0,  200,   7,  0,  FALSE,  (uint32_t *)&FTM0_C5V},
	{ 0, 1200,  13,  0,  FALSE,  (uint32_t *)&FTM1_C0V}
};



volatile char				cmd[MAX_CMD_LEN+1];
char						token[MAX_TOKEN_LEN+1];

volatile uint32_t			Timer;


uint16_t					Red0EE[NUM_EEPROM_SCENES];
uint16_t					Green0EE[NUM_EEPROM_SCENES];
uint16_t					Blue0EE[NUM_EEPROM_SCENES];
uint16_t					Red1EE[NUM_EEPROM_SCENES];
uint16_t					Green1EE[NUM_EEPROM_SCENES];
uint16_t					Blue1EE[NUM_EEPROM_SCENES];

uint8_t						ActiveScene;

int32_t						ind;			// global, holds index into command buffer

/*
 *  Local functions
 */
static void				gettoken(char  **ptr, char  *tok);
static void				ShowPrompt(void);
static void				GetCmd(void);
static void				ProcessCmd(void);
static void				SetPWM(LED  *pled);
static void				backgrnd(void);
static void				ShowEEScene(uint8_t  num);
static void				SaveEEScene(uint8_t  num);
static void				LoadEEScene(uint8_t  num);
static void				ShowCurrentRGB(void);
static void				DescribeActiveScene(void);
static void				ShowHelp(void);
static void				PWMInit(void);
static void				PITHndlr(char  chnl);
static uint32_t			ScaleBrightness(uint32_t  val);



int  main(void)
{
	UARTInit(TERM_UART_2, TERM_BAUD);		// set terminal UART for 115.2K baud, 8N1
 	xputs("\n\r" PROG_NAME_STR "\n\r");		// announce ourselves

	PWMInit();								// local function to set up PWM for all channels
	SetPWM(&Red[0]);						// start with all channels off
	SetPWM(&Red[1]);
	SetPWM(&Green[0]);
	SetPWM(&Green[1]);
	SetPWM(&Blue[0]);
	SetPWM(&Blue[1]);

	PITInit(PITCHNL, &PITHndlr, PITDELAY_USECS, 0, 3);
	PITStart(PITCHNL);

	ActiveScene = 0;
	LoadEEScene(ActiveScene);
	DescribeActiveScene();
	ShowHelp();

	EnableInterrupts;

	while (1)
	{
		ShowCurrentRGB();
		ShowPrompt();
		GetCmd();
		ProcessCmd();
	}

	return  0;
}



/*
 *  PWMInit      configure PWM channels
 *
 *  This routine sets up the PWM channels used for RGB control.  The
 *  PWM channels are all configured for legacy mode (none of the second
 *  set of FTM registers [starting with FTMn_MODE] are used).
 *
 *  FTMn_CNT sets the PWM initial value; this must be 0 for edge-aligned
 *  PWM mode.  This value will be the same for all PWM channels.
 *
 *  FTMn_MOD sets the PWM modulus (overflow) value.  This valuw will
 *  be the same for all PWM channels.
 *
 *  Each channel has its own output-compare match value, in register
 *  FTMn_CxV.  The value written to this register must be
 *  FTMn_CNT >= value < FTMn_MOD.  The value written to this register
 *  sets the duty cycle of the PWM waveform.
 */
static void  PWMInit(void)
{
	SIM_SCGC6 |= SIM_SCGC6_FTM0_MASK;		// enable the FTM0 subsystem clock
	SIM_SCGC6 |= SIM_SCGC6_FTM1_MASK;		// enable the FTM0 subsystem clock

	FTM0_SC = FTM_SC_CLKS(1) | FTM_SC_PS(2);	// use system clock, select divider
	FTM1_SC = FTM_SC_CLKS(1) | FTM_SC_PS(2);	// use system clock, select divider

/*
 *  Configure each PWM driver line for active-low operation.
 *  Use edge-aligned PWM; clear output when initial value is loaded, set output
 *  on match.
 *
 *  Configure each port line for FTM, high drive strength, totem-pole output.
 */
	PORTC_PCR1 = PORT_PCR_MUX(0x4);		// red.0 is on PTC1 (Teensy pin 22); FTM0 ch 0 (alt = 4)
	FTM0_C0SC = FTM_CnSC_MSB_MASK + FTM_CnSC_ELSA_MASK;	// Edge-aligned PWM, set output on match

	PORTC_PCR2 = PORT_PCR_MUX(0x4);			// green.0 is on PTC2 (Teensy pin 23); FTM0 ch 1 (alt = 4)
	FTM0_C1SC = FTM_CnSC_MSB_MASK + FTM_CnSC_ELSA_MASK;	// Edge-aligned PWM, set output on match

	PORTC_PCR3 = PORT_PCR_MUX(0x4);			// blue.0 is on PTC3 (Teensy pin 9); FTM0 ch 2 (alt = 4)
	FTM0_C2SC = FTM_CnSC_MSB_MASK + FTM_CnSC_ELSA_MASK;	// Edge-aligned PWM, set output on match

	PORTC_PCR4 = PORT_PCR_MUX(0x4);			// red.1 is on PTC4 (Teensy pin 10); FTM0 ch 3 (alt = 4)
	FTM0_C3SC = FTM_CnSC_MSB_MASK + FTM_CnSC_ELSA_MASK;	// Edge-aligned PWM, set output on match

	PORTD_PCR4 = PORT_PCR_MUX(0x4);			// green.1 is on PTD4 (Teensy pin 6); FTM0 ch 4 (alt = 4)
	FTM0_C4SC = FTM_CnSC_MSB_MASK + FTM_CnSC_ELSA_MASK;	// Edge-aligned PWM, set output on match

	PORTD_PCR5 = PORT_PCR_MUX(0x4);			// blue.1 is on PTD5 (Teensy pin 20); FTM0 ch 5 (alt = 4)
	FTM0_C5SC = FTM_CnSC_MSB_MASK + FTM_CnSC_ELSA_MASK;	// Edge-aligned PWM, set output on match

	PORTD_PCR6 = PORT_PCR_MUX(0x4);			// red.2 is on PTD6 (Teensy pin 21); FTM0 ch 6 (alt = 4)
	FTM0_C6SC = FTM_CnSC_MSB_MASK + FTM_CnSC_ELSA_MASK;	// Edge-aligned PWM, set output on match

	PORTD_PCR7 = PORT_PCR_MUX(0x4);			// green.2 is on PTD7 (Teensy pin 5); FTM0 ch 4 (alt = 4)
	FTM0_C7SC = FTM_CnSC_MSB_MASK + FTM_CnSC_ELSA_MASK;	// Edge-aligned PWM, set output on match

	PORTA_PCR12 = PORT_PCR_MUX(0x3);			// blue.2 is on PTA12 (Teensy pin 3); FTM1 ch 0 (alt = 3)
	FTM1_C0SC = FTM_CnSC_MSB_MASK + FTM_CnSC_ELSA_MASK;	// Edge-aligned PWM, set output on match

	FTM0_CNTIN = 0;							// initial value of PWM counter range is always 0
	FTM0_CNT = 0;							// any write to CNT copies CNTIN to the counter;
											// always init CNT before initing MOD!
	FTM0_MOD = MAX_PWM_COUNT;				// upper limit of PWM counter range is always max resolution

	FTM1_CNTIN = 0;							// initial value of PWM counter range is always 0
	FTM1_CNT = 0;							// any write to CNT copies CNTIN to the counter;
											// always init CNT before initing MOD!
	FTM1_MOD = MAX_PWM_COUNT;				// upper limit of PWM counter range is always max resolution


}


static void  ShowPrompt(void)
{
	xputs("\n\rCmd: ");
}


static void  GetCmd(void)
{
	ind = 0;				// always rewind the command buffer pointer
	while (get_line_r((char *)cmd, MAX_CMD_LEN, &ind) == 0)
	{
		backgrnd();
	}
	xputs("\n\r");
}


static void  ProcessCmd(void)
{
	char				*ptr;
	uint32_t			val;
	uint8_t				n;
	long int			tl;

	if (strlen((char *)cmd) == 0)  return;
	ptr = (char *)cmd;
	gettoken(&ptr, token);
	if (strlen(token) == 0)  return;
#if 0
	if (strcmp("r", token) == 0)			// r <val> changes red duty cycle
	{
		gettoken(&ptr, token);				// get the value, if any
		if (strlen(token) != 0)				// if value...
		{
			ptr = token;
			xatoi(&ptr, (long int *)&Red);
			SetPWM(RED_PWM, Red);
		}
	}
	else if (strcmp("g", token) == 0)		// g <val> changes green duty cycle
	{
		gettoken(&ptr, token);				// get the value, if any
		if (strlen(token) != 0)				// if value...
		{
			ptr = token;
			xatoi(&ptr, (long int *)&Green);
			SetPWM(GREEN_PWM, Green);
		}
	}
	else if (strcmp("b", token) == 0)		// b <val> changes blue duty cycle
	{
		gettoken(&ptr, token);				// get the value, if any
		if (strlen(token) != 0)				// if value...
		{
			ptr = token;
			xatoi(&ptr, (long int *)&Blue);
			SetPWM(BLUE_PWM, Blue);
		}
	}
	else if (strcmp("l", token) == 0)		// l  lists all saved RGB scenes
	{
		for (n=0; n<NUM_EEPROM_SCENES; n++)
		{
			ShowEEScene(n);
		}
		xputs("\n\r");
	}
	else if (strcmp("save", token) == 0)		// save n  saves current RGB as scene n
	{
		gettoken(&ptr, token);				// get the value (required)
		if (strlen(token) != 0)				// if value...
		{
			xatoi((char **)&token, &tl);
			val = tl;
			val = val & 0xff;
			SaveEEScene(val);
		}
		else
		{
			xputs("You must provide a scene number to save.\n\r");
		}
	}
	else if (strcmp("go", token) == 0)		// go n  load and show scene n
	{
		gettoken(&ptr, token);				// get the value (required)
		if (strlen(token) != 0)				// if value...
		{
			xatoi((char **)&token, &tl);
			val = tl;
			val = val & 0xff;
			if (val < NUM_EEPROM_SCENES)
			{
				LoadEEScene(val);
				DescribeActiveScene();
			}
			else
			{
				xprintf("You must go to a scene number from 0 to %d.\n\r", NUM_EEPROM_SCENES-1);
			}
		}
	}
	else if (strcmp("n", token) == 0)		// n  show next scene (wraps)
	{
		ActiveScene++;
		if (ActiveScene >= NUM_EEPROM_SCENES)  ActiveScene = 0;
		LoadEEScene(ActiveScene);
		DescribeActiveScene();
	}
	else if (strcmp("p", token) == 0)		// p  show previous scene (wraps)
	{
		ActiveScene--;
		if (ActiveScene >= NUM_EEPROM_SCENES)  ActiveScene = NUM_EEPROM_SCENES - 1;
		LoadEEScene(ActiveScene);
		DescribeActiveScene();
	}
	else if (strcmp("?", token) == 0)		// ?  show help screen
	{
		ShowHelp();
	}		
	else
	{
		xprintf("Unknown command: '%s'\n\r", token);
	}
#endif
}




static void  ShowHelp(void)
{
	xprintf("\n\r");
	xprintf("\n\r?       show this screen");
	xprintf("\n\rl       list all scenes in memory");
	xprintf("\n\rr val   set current red to val (0-%d)", MAX_PWM_COUNT);
	xprintf("\n\rg val   set current green to val (0-%d)", MAX_PWM_COUNT);
	xprintf("\n\rb val   set current blue to val (0-%d)", MAX_PWM_COUNT);
	xprintf("\n\rgo n    get current RGB values from scene n (0-%d)", NUM_EEPROM_SCENES-1);
	xprintf("\n\rsave n  save current RGB values to scene n");
	xprintf("\n\rn       get current RGB values from next scene");
	xprintf("\n\rp       get current RGB values from previous scene");
	xprintf("\n\r");
}



static void  SetPWM(LED  *pLED)
{
	if (pLED == 0)  return;

	pLED->pwm = ScaleBrightness(pLED->user);
	*(pLED->pchnl) = pLED->pwm;
}
			


static void  ShowCurrentRGB(void)
{
//	xprintf("Red=%d  Green=%d  Blue=%d\n\r", Red, Green, Blue);
}


static void  ShowEEScene(uint8_t  num)
{
	if (num < NUM_EEPROM_SCENES)
	{
//		xprintf("Scene %d: %3d  %3d  %3d\n\r", num, eeprom_read_byte(&RedEE[num]),
//				eeprom_read_byte(&GreenEE[num]), eeprom_read_byte(&BlueEE[num]));
	}
	else
	{
		xprintf("No such scene as %d; scenes are numbered 0 to %d.\n\r", NUM_EEPROM_SCENES);
	}
}
		


static void  SaveEEScene(uint8_t  num)
{
	if (num < NUM_EEPROM_SCENES)
	{
/*
		eeprom_write_byte(&RedEE[num], Red);
		eeprom_write_byte(&GreenEE[num], Green);
		eeprom_write_byte(&BlueEE[num], Blue);
*/
	}
}


static void  LoadEEScene(uint8_t  num)
{
	if (num < NUM_EEPROM_SCENES)
	{
/*
		Red = eeprom_read_byte(&RedEE[num]);
		Green = eeprom_read_byte(&GreenEE[num]);
		Blue = eeprom_read_byte(&BlueEE[num]);
*/
//		SetPWM(RED_PWM, Red);
//		SetPWM(GREEN_PWM, Green);
//		SetPWM(BLUE_PWM, Blue);
		ActiveScene = num;
	}
}


static void  DescribeActiveScene(void)
{
	xprintf("This is scene %d.\n\r", ActiveScene);
}


static void  gettoken(char  **ptr, char  *tok)
{
	uint8_t					n;

	while ((**ptr == ' ') || (**ptr == '\t'))  *ptr = *ptr + 1;	// step past whitespace
	n = 0;
	while ((**ptr != ' ') && (**ptr != '\t') && (**ptr))	// for all non-space chars...
	{
		*tok = **ptr;						// copy the char
		tok++;								// move to next open cell
		*ptr = *ptr + 1;					// move to next char in string
		n++;
		if (n == MAX_TOKEN_LEN)  break;		// don't run off the end of the buffer!
	}
	*tok = 0;								// be sure to terminate the string
}


static void  backgrnd(void)
{
	uint32_t			t;

	t = Timer;
	if (t != Timer)  t = Timer;

#if 0
	if (t == 0)
	{
		Red = Red + RedDelta;
		if ((Red > 251) || (Red < 4))  RedDelta = -RedDelta;
		SetPWM(RED_OCR, Red);

		Green = Green + GreenDelta;
		if ((Green > 251) || (Green < 4)) GreenDelta = -GreenDelta;
		SetPWM(GREEN_OCR, Green);

		Blue = Blue + BlueDelta;
		if ((Blue > 251) || (Blue < 4))  BlueDelta = -BlueDelta;
		SetPWM(BLUE_OCR, Blue);

		Timer = 3200;						// hard-coded delay (1 sec)
		if (Timer != 3200)  Timer = 3200;
	}
#endif

}


static uint32_t			ScaleBrightness(uint32_t  val)
{
	uint32_t			result;

	if (val == 0)  return  0;

	val = val * val;
	result = val / BRIGHTNESS_SCALER;
	if (result == 0)  result++;		// if val was not zero, use a minimum of one PWM count

	return  result;
}




/*
SIGNAL(TIMER2_OVF_vect)
{
	if (Timer)  Timer--;
}
*/


/*
 *  PITHndlr      ISR for handling all PIT interrupts
 *
 *  Control enters this routine when any of the PIT interrupts fires.
 *  The argument chnl contains the indentifier (0-3) of the PIT
 *  channel causing the interrupt.
 */
static void  PITHndlr(char  chnl)
{
	static uint32_t				timer = 0;
	uint32_t					tval;
	register uint32_t			n;

	if (timer)  timer--;

	if (timer == 0)
	{
		for (n=0; n<NUM_RGB; n++)
		{

			tval = Red[n].user + Red[n].delta;
			if (tval > MAX_USER_BRIGHTNESS)
			{
				if (Red[n].delta < 0)  tval = 0;
				else					tval = MAX_USER_BRIGHTNESS;
				Red[n].delta = -Red[n].delta;
			}
			Red[n].user = tval;
			SetPWM(&Red[n]);

			tval = Green[n].user + Green[n].delta;
			if (tval > MAX_USER_BRIGHTNESS)
			{
				if (Green[n].delta < 0)  tval = 0;
				else					tval = MAX_USER_BRIGHTNESS;
				Green[n].delta = -Green[n].delta;
			}
			Green[n].user = tval;
			SetPWM(&Green[n]);
	
			tval = Blue[n].user + Blue[n].delta;
			if (tval > MAX_USER_BRIGHTNESS)
			{
				if (Blue[n].delta < 0)  tval = 0;
				else					tval = MAX_USER_BRIGHTNESS;
				Blue[n].delta = -Blue[n].delta;
			}
			Blue[n].user = tval;
			SetPWM(&Blue[n]);
		}

		timer = 200;
	}
}
