/*
 * sysfuncs.c
 *
 *  Created on: 2 Nov 2010
 *      Author: nanoage.co.uk
 */
#include "LPC17xx.h"
#include <errno.h>
#include <sys/stat.h>
#include <sys/times.h>
#include <sys/unistd.h>
#include <string.h>
#include "uart.h"
#include "syscalls.h"


//static int					stdinfn;			// file number for stdin
//static int					stdoutfn;			// file number for stdout
//static int					stderrfn;			// file number for stderr




#undef errno
extern int				errno;



/*
 *  Define table entries for the various stream devices that can be activated
 *  by the calling program.
 */
DEVOPTAB			devoptab_uart0 = {"UART0", UART0Open, UART0Close,
							UART0Write, UART0Read, UART0Avail};

DEVOPTAB			devoptab_uart1 = {"UART1", UART1Open, UART1Close,
							UART1Write, UART1Read, UART1Avail};

DEVOPTAB			devoptab_uart2 = {"UART2", UART2Open, UART2Close,
							UART2Write, UART2Read, UART2Avail};

DEVOPTAB			devoptab_uart3 = {"UART3", UART3Open, UART3Close,
							UART3Write, UART3Read, UART3Avail};

/*
 *  Create the device operations table for handling the standard stream
 *  devices plus any others we might want to add at compile-time.
 *
 *  Define this as static so it can be modified by calls to set_stdio() below.
 */
static  DEVOPTAB			*devoptab_list[] =
{
   &devoptab_uart2,				// device 0 is always standard input
   &devoptab_uart2,				// device 1 is always standard output
   &devoptab_uart2,				// device 2 is always standard error
   &devoptab_uart3,				// declare device 3 and hook it to UART3
   0							// a null entry terminates the list
};



/*
 environ
 A pointer to a list of environment variables and their values.
 For a minimal environment, this empty list is adequate:
 */
char *__env[1] = { 0 };
char **environ = __env;


void _exit(int status) {
    _write(1, "exit", 4);
    while (1) {
        ;
    }
}



/*
 *  _close     close the selected file
 */
int  _close(int  fd)
{
	return  devoptab_list[fd]->close(fd);
}




/*
 execve
 Transfer control to a new process. Minimal implementation (for a system without processes):
 */
int _execve(char *name, char **argv, char **env) {
    errno = ENOMEM;
    return -1;
}
/*
 fork
 Create a new process. Minimal implementation (for a system without processes):
 */

int _fork() {
    errno = EAGAIN;
    return -1;
}
/*
 fstat
 Status of an open file. For consistency with other minimal implementations in these examples,
 all files are regarded as character special devices.
 The `sys/stat.h' header file required is distributed in the `include' subdirectory for this C library.
 */
int _fstat(int file, struct stat *st) {
    st->st_mode = S_IFCHR;
    return 0;
}

/*
 getpid
 Process-ID; this is sometimes used to generate strings unlikely to conflict with other processes. Minimal implementation, for a system without processes:
 */

int _getpid() {
    return 1;
}

/*
 isatty
 Query whether output stream is a terminal. For consistency with the other minimal implementations,
 */
int _isatty(int  file)
{
    switch (file)
	{
		case STDOUT_FILENO:
		case STDERR_FILENO:
		case STDIN_FILENO:
        return 1;
		
		default:
        //errno = ENOTTY;
        errno = EBADF;
        return 0;
    }
}


/*
 kill
 Send a signal. Minimal implementation:
 */
int _kill(int pid, int sig)
{
    errno = EINVAL;
    return (-1);
}

/*
 link
 Establish a new name for an existing file. Minimal implementation:
 */

int _link(char *old, char *new)
{
    errno = EMLINK;
    return -1;
}



/*
 *  _lseek      set position in a file
 *
 *  Minimal implementation
 */
int _lseek(int file, int ptr, int dir)
{
    return 0;
}


/*
 * _sbrk    reserve block of memory in heap
 *
 *  malloc and related functions depend on this
 */
caddr_t _sbrk(int incr)
{
    extern char 					*_end_bss; 		// defined by the linker
    static char						*heap_end;
    char							*prev_heap_end;
	char 							*stack;

    if (heap_end == 0)
	{
        heap_end = _end_bss;
    }
    prev_heap_end = heap_end;

	stack = (char*) __get_MSP();
	if (heap_end + incr >  stack)
    {
         errno = ENOMEM;
         return  (caddr_t) -1;
         //abort ();
	}

    heap_end += incr;
    return (caddr_t) prev_heap_end;
}





/*
 *  _read      read chars from a stream device
 *
 *  Returns -1 on error or blocks until the number of characters have been read.
 */
int _read(int file, char *ptr, int len)
{
	return  devoptab_list[file]->read(file, ptr, len);
}


#if 0
int _read(int file, char *ptr, int len)
{
    int							n;
    int							num;
	char						c;

	num = 0;								// nothing so far
    switch (file)
	{
    	case STDIN_FILENO:
        for (n=0; n<len; n++)
		{
			switch  (stdinfn)					// based on the UART on this stream...
			{
				case  0:						// if using UART0...
				c = UART0Read();
				break;

				case  1:
				c = UART1Read();
				break;

				case  2:
				c = UART2Read();
				break;

				case  3:
				c = UART3Read();
				break;

				default:						// illegal UART!
				errno = EBADF;
				return  -1;
			}
            *ptr++ = c;
            num++;
        }
        break;

   		default:
        errno = EBADF;
        return -1;
    }
    return num;
}
#endif



/*
 stat
 Status of a file (by name). Minimal implementation:
 int    _EXFUN(stat,( const char *__path, struct stat *__sbuf ));
 */

int _stat(const char *filepath, struct stat *st)
{
    st->st_mode = S_IFCHR;
    return 0;
}



/*
 *  _times
 *  Timing information for current process. Minimal implementation:
 */

clock_t _times(struct tms *buf)
{
    return -1;
}




/*
 *  _unlink
 *  Remove a file's directory entry. Minimal implementation:
 */
int _unlink(char *name) {
    errno = ENOENT;
    return -1;
}

/*
 wait
 Wait for a child process. Minimal implementation:
 */
int _wait(int *status)
{
    errno = ECHILD;
    return -1;
}



/*
 *  _open      open a stream connection based on a file/stream name
 */

int  _open(const char  *file, int  flags, int  mode)
{
   int						ntab;
   int						fd;

   ntab = 0;							// haven't found the device table entry yet
   fd = -1;								// start with an illegal file descriptor

/*
 *  Search for "file" in device table entries' .name field.
 */
	do
	{
		if (strcmp(devoptab_list[ntab]->name, file) == 0)	// if this is the device we want...
		{
			fd = ntab;					// the table index becomes the file descriptor
			break;
		}
	} while(devoptab_list[ntab++]);

	if (fd != -1)						// if we found a match in the table...
		devoptab_list[fd]->open(file, flags, mode);	// invoke the associated open() function
	else								// if we didn't find the name in the table...
		errno = ENODEV;					// show the device doesn't exist

	return  fd;							// file descriptor doubles as return code
}




#if 0
/*
 *  _write
 *  Write a character to a file. `libc' subroutines will use this system routine for
 *  output to all files, including stdout
 *
 *  Returns -1 on error or number of bytes sent
 */
int _write(int file, char *ptr, int len)
{
    int							n;

    switch (file)
	{
    	case STDOUT_FILENO:								// stdout
        for (n = 0; n < len; n++)
		{
			switch  (stdoutfn)							// based on the UART hooked to this stream...
			{
				case  0:								// if UART0...
				while ((LPC_UART0->LSR & LSR_THRE) == 0)  ;		// blocks until OK to send
				LPC_UART0->THR = (*ptr++ & (uint16_t)0xff);
				break;

				case  1:
				while ((LPC_UART1->LSR & LSR_THRE) == 0)  ;		// blocks until OK to send
				LPC_UART1->THR = (*ptr++ & (uint16_t)0xff);
				break;

				case  2:
				LPC_GPIO1->FIOSET = (1<<21);					// debug  turn on LED3
				while ((LPC_UART2->LSR & LSR_THRE) == 0)  ;		// blocks until OK to send
				LPC_UART2->THR = (*ptr++ & (uint16_t) 0xff);
				break;

				case  3:
				while ((LPC_UART3->LSR & LSR_THRE) == 0)  ;		// blocks until OK to send
				LPC_UART3->THR = (*ptr++ & (uint16_t)0xff);
				break;

				default:								// no such UART!
				errno = EBADF;
				return  -1;
			}
        }
		LPC_GPIO1->FIOCLR = (1<<21);						// debug  turn off LED3
        break;

	    case STDERR_FILENO:								// stderr
        for (n = 0; n < len; n++)
		{
			switch  (stderrfn)							// based on the UART hooked to this stream...
			{
				case  0:								// if UART0...
				while ((LPC_UART0->LSR & LSR_THRE) == 0)  ;		// blocks until OK to send
				LPC_UART0->THR = (*ptr++ & (uint16_t)0xff);
				break;

				case  1:
				while ((LPC_UART1->LSR & LSR_THRE) == 0)  ;		// blocks until OK to send
				LPC_UART1->THR = (*ptr++ & (uint16_t)0xff);
				break;

				case  2:
				LPC_GPIO1->FIOSET = (1<<21);					// debug  turn on LED3
				while ((LPC_UART2->LSR & LSR_THRE) == 0)  ;		// blocks until OK to send
				LPC_UART2->THR = (*ptr++ & (uint16_t) 0xff);
				break;

				case  3:
				while ((LPC_UART3->LSR & LSR_THRE) == 0)  ;		// blocks until OK to send
				LPC_UART3->THR = (*ptr++ & (uint16_t)0xff);
				break;

				default:								// no such UART!
				errno = EBADF;
				return  -1;
			}
        }
        break;

	    default:
        errno = EBADF;
        return -1;
    }
    return len;
}
#endif

int _write(int file, char *ptr, int len)
{
	return  devoptab_list[file]->write(file, ptr, len);
}






//  ===============================================================

/*
 *  _avail      test for available char from stdin
 *
 *  I added this function because some of my terminal I/O needs to check
 *  if a char is available from stdin, but not lock until one becomes
 *  available.
 *
 *  I'm sure there is a more standard way of doing this, but I'll use this
 *  for now.
 *
 *  Karl Lunt   26 Dec 2011
 */

int  _avail(int  file)
{
	return  devoptab_list[file]->avail(file);
}



#if 0
int  get_stdio(int  file)
{
	switch  (file)
	{
		case  STDIN_FILENO:
		return  stdinfn;

		case  STDOUT_FILENO:
		return  stdoutfn;

		case  STDERR_FILENO:
		return  stderrfn;

		default:
		errno = EBADF;
		return  -1;
	}
}
#endif


/*
 *  set_stdio      assign a UART by number to one of the stdio streams
 *
 *  Upon entry, argument file contains the file number for a stdio stream
 *  (STDIN_FILENO, STDOUT_FILENO, or STDERR_FILENO).  Argument uartn
 *  contains the identifier for one of the system UARTs (0-3).
 *
 *  Upon exit, this routine returns -1 if the file argument is invalid,
 *  or it returns 0.
 */
int  set_stdio(int  file, int  uartn)
{
	int					r;

	r = 0;							// assume this works
	switch  (file)
	{
		case  STDIN_FILENO:
		if		(uartn == 0) devoptab_list[STDIN_FILENO] = &devoptab_uart0; 
		else if (uartn == 1) devoptab_list[STDIN_FILENO] = &devoptab_uart1;
		else if (uartn == 2) devoptab_list[STDIN_FILENO] = &devoptab_uart2;
		else if (uartn == 3) devoptab_list[STDIN_FILENO] = &devoptab_uart3;
		else  r = -1;
		break;

		case  STDOUT_FILENO:
		if		(uartn == 0) devoptab_list[STDOUT_FILENO] = &devoptab_uart0; 
		else if (uartn == 1) devoptab_list[STDOUT_FILENO] = &devoptab_uart1;
		else if (uartn == 2) devoptab_list[STDOUT_FILENO] = &devoptab_uart2;
		else if (uartn == 3) devoptab_list[STDOUT_FILENO] = &devoptab_uart3;
		else  r = -1;
		break;

		case  STDERR_FILENO:
		if		(uartn == 0) devoptab_list[STDERR_FILENO] = &devoptab_uart0; 
		else if (uartn == 1) devoptab_list[STDERR_FILENO] = &devoptab_uart1;
		else if (uartn == 2) devoptab_list[STDERR_FILENO] = &devoptab_uart2;
		else if (uartn == 3) devoptab_list[STDERR_FILENO] = &devoptab_uart3;
		else  r = -1;
		break;

		default:						// oops, it didn't work
		errno = EBADF;					// record a global error
		r = -1;							// show an error result
	}
	return  r;
}

	