#include "chordinterpreter.h"

#include <QDateTime>
#include <QFile>
#include <QStringList>
#include <QProcess>
#include <QTextStream>
#include <QApplication>
#include <QScrollBar>
#include <QTextStream>

//Takes: list of chords
//During working takes characters.
//Executes chords according to list

chordInterpreter::chordInterpreter(QList<chord> database, QPlainTextEdit *debugLog)
{
    this->database=database;
    this->currentCombo="";
    this->debugLog=debugLog;
    this->debugLevel=0;
    this->trainingMode=0;
    this->recording=0;
    addDebug(0,"Initialized chord interpreter.");
}

void chordInterpreter::changeBase(QList<chord> database)
{
    this->database=database;
    this->currentCombo="";
    addDebug(2,"Reloaded chords list.");
}

void chordInterpreter::loadFromFile(QString name)
{
    this->fileName=name;
    addDebug(2,"Starting loading definitions from file: "+name);

    QFile keysFile(name);
    int items=0;
    if (!keysFile.exists())
    {
        addDebug(0,"File "+name+" does not exist!");
    }
    if (keysFile.open(QIODevice::ReadOnly|QIODevice::Text))
    {
        this->database.clear();
        QString line="";
        QTextStream in(&keysFile);

        while (!in.atEnd())
        {
            items++;
            chord x;
            line=in.readLine().trimmed();
            x.name=line;
            line=in.readLine().trimmed();
            x.combo=line;
            line=in.readLine().trimmed();
            x.command=line;
            if ((x.combo!="")&&(x.command!=""))
            {
                this->database.append(x);
            }
        }
    }
    else
    {
        addDebug(0,"Could not open the file!");
    }
    addDebug(2,"Loaded "+QString::number(items)+" chord definitions.");
}

void chordInterpreter::saveToFile()
{
    if (this->fileName=="")
    {
        return;
    }
    QFile keys(this->fileName);
    if (keys.open(QIODevice::WriteOnly))
    {
        QTextStream out(&keys);
        for (int i=0;i<this->database.count();i++)
        {
            out<<this->database.at(i).name<<"\n";
            out<<this->database.at(i).combo<<"\n";
            out<<this->database.at(i).command<<"\n";
        }
        out.flush();
        keys.close();
    }
}


void chordInterpreter::setDebugLevel(int level)
{
    //0 - normal
    //1 - commands
    //2 - everything
    this->debugLevel=level;
    addDebug(2, "Changed debug level to "+QString::number(level));
}

//insert debug message
void chordInterpreter::addDebug(int level, QString message)
{
    QString time=QDate::currentDate().toString("yyyy-MM-dd")+" "+QTime::currentTime().toString("HH:mm:ss");
    if (level<=this->debugLevel)
    {
        this->debugLog->appendPlainText(time+" "+message);
        this->debugLog->verticalScrollBar()->setValue(this->debugLog->verticalScrollBar()->maximum());
    }
}

//clean the command string of released keys
//TODO: Optimize this optimization.
QString optimize(QString input)
{
    QString output=input;
    for (int i=0;i<output.count();i++)
    {
        if (output.at(i)=='a')
        {
            for (int j=i;j<output.count();j++)
            {
                if (output.at(j)=='A')
                {
                    output.replace(i,1,'?');
                    output.replace(j,1,'?');
                }
            }
        }
        if (output.at(i)=='b')
        {
            for (int j=i;j<output.count();j++)
            {
                if (output.at(j)=='B')
                {
                    output.replace(i,1,'?');
                    output.replace(j,1,'?');
                }
            }
        }
        if (output.at(i)=='c')
        {
            for (int j=i;j<output.count();j++)
            {
                if (output.at(j)=='C')
                {
                    output.replace(i,1,'?');
                    output.replace(j,1,'?');
                }
            }
        }
        if (output.at(i)=='d')
        {
            for (int j=i;j<output.count();j++)
            {
                if (output.at(j)=='D')
                {
                    output.replace(i,1,'?');
                    output.replace(j,1,'?');
                }
            }
        }
        if (output.at(i)=='e')
        {
            for (int j=i;j<output.count();j++)
            {
                if (output.at(j)=='E')
                {
                    output.replace(i,1,'?');
                    output.replace(j,1,'?');
                }
            }
        }
    }
    output=output.replace("?","");
    return output;
}

//checks does the combination contain logical errors.
bool improper(QString input)
{
    for (int i=0;i<input.length();i++)
    {
        if (input.at(i).isLower())
        {
            QChar x=input.at(i);
            for (int j=i+1;j<input.length();j++)
            {
                if (input.at(j)==x)
                {
                    return true;
                }
                if (input.at(j)==x.toUpper())
                {
                    break;
                }
            }
        }
    }
    return false;
}

void chordInterpreter::stuffKey(QString key)
{
    addDebug(2,"Key received: "+key);
    if ((key.at(0).isUpper())&&(this->currentCombo.length()==0))
    {
        return;
    }
    this->currentCombo+=key;

    if (improper(this->currentCombo))
    {
        this->currentCombo="";
        return;
    }

    //if we are recording, we halt every activity except recording.
    if (this->recording)
    {
        this->recordKeystrokes(this->recorder);
        //we skip recording when we've exhausted combinations
        if ((this->currentCombo.count('a')==this->currentCombo.count('A'))&&
            (this->currentCombo.count('b')==this->currentCombo.count('B'))&&
            (this->currentCombo.count('c')==this->currentCombo.count('C'))&&
            (this->currentCombo.count('d')==this->currentCombo.count('D'))&&
            (this->currentCombo.count('e')==this->currentCombo.count('E')))
        {
            this->recording=0;
            this->currentCombo="";
        }
        return;
    }

    //evaluate combo vs base
    for (int i=0;i<this->database.length();i++)
    {
        if (this->currentCombo.startsWith(this->database.at(i).combo))
        {

            if (this->trainingMode)
            {
                addDebug(0," TEST MODE: Activating macro "+this->database.at(i).name);
            }
            else
            {
                addDebug(1,"Executing macro "+this->database.at(i).name);
                QApplication::processEvents();
                int code=-1024;
                //if found - execute
                QProcess *process = new QProcess();
                process->start(this->database.at(i).command,QIODevice::ReadWrite);
                if (!process->waitForStarted())
                {
                    addDebug(0, "Errror running command! - "+this->database.at(i).name);
                }
                process->waitForFinished(1000); //TODO - figure out timeouts
                code=process->exitCode();
                addDebug(1, "Exit code: "+QString::number(code));

            }

            this->currentCombo=optimize(this->currentCombo);
            //addDebug(2,this->currentCombo);
        }

    }
    //if not found - nothing

    //Now we need to remove the string but if and only if the buttons are balanced. It means that nothing
    //can be released anymore.
    //Another idea here is to use time as the indicator, this will allow two different combos like aAbB and aAcC.

    if ((this->currentCombo.count('a')==this->currentCombo.count('A'))&&
        (this->currentCombo.count('b')==this->currentCombo.count('B'))&&
        (this->currentCombo.count('c')==this->currentCombo.count('C'))&&
        (this->currentCombo.count('d')==this->currentCombo.count('D'))&&
        (this->currentCombo.count('e')==this->currentCombo.count('E')))
    {
        addDebug(2, "End of chord, clearing buffer");
        this->currentCombo="";
    }
    if (this->currentCombo.length()>0)
    {
        addDebug(2,"Buffer contents: "+this->currentCombo);
    }
}

void chordInterpreter::setTrainingMode(bool mode)
{
    this->trainingMode=mode;
    addDebug(0,"Training mode: "+QString::number(mode));
}

QList<chord> chordInterpreter::getBase()
{
    return this->database;
}

void chordInterpreter::recordKeystrokes(QLineEdit *le)
{
    this->recorder=le;
    this->recording=1;
    if (this->recording==1)
    {
        this->recorder->setText(this->currentCombo);
    }
}
