//Copyright 2014,2015 MCbx, All rights reserved.
//http://mcbx.netne.net/ictester
//This file is part of ICTester.
//ICTester is free software; you can redistribute it and/or modify
//it under the terms of the GNU General Public License as published by
//the Free Software Foundation; either version 2 of the License, or
//(at your option) any later version.
//ICTester is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//You should have received a copy of the GNU General Public License
//along with ICTester; if not, write to the Free Software Foundation,
//Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include <QtSerialPort/QSerialPort>
#include <QtSerialPort/QSerialPortInfo>
//#include <QMessageBox>
#include <qstring.h>
#include <QTime>
#include "devicedriver.h"
#ifdef Q_OS_WIN
#include <windows.h>
#include <QDateTime>
#endif

//Errors:
// -  0 - no error, operation completed.
// - -1 - port not opened
// - -2 - device not present
// - -3 - Improper response from device.
// - -4 - user data error. It's your fault.


DeviceDriver::DeviceDriver(QString portName, int baudRate, int timeout)
{
    serialConn = new QSerialPort();
    this->Timeout=timeout;
    this->openConn(portName,baudRate);
}

DeviceDriver::~DeviceDriver()
{
    if (serialConn->isOpen()) serialConn->close();
}

//PRIVATE CONNECTION FUNCTIONS
int DeviceDriver::openConn(QString portName, int baudRate=QSerialPort::Baud19200)
{
    serialConn->setPortName(portName);
    serialConn->setBaudRate(baudRate);
    serialConn->setDataBits(QSerialPort::Data8);
    serialConn->setStopBits(QSerialPort::OneStop);
    serialConn->setFlowControl(QSerialPort::NoFlowControl);
    serialConn->setParity(QSerialPort::NoParity);

    //open connection. Close it at destructor.
     if (!serialConn->open(QIODevice::ReadWrite))
     {
  //       QMessageBox::critical(NULL,"Error", serialConn->errorString());
         return -1;
     }
     return 0;
}

//waits for response of specified length. Then returns it. If timeout - returns empty array
QByteArray DeviceDriver::getResponse(int expLength)
{
    for (;;)
    {
#ifdef Q_OS_WIN         //Qt bug unfixed since 2010, Windows only. Writing this in 2015.
        bool ret;       //I have to rewrite waiting routine, the original sometimes doesn't work in Windows.
        qint64 start=QDateTime::currentDateTime().toMSecsSinceEpoch();
        while(QDateTime::currentDateTime().toMSecsSinceEpoch()<start+this->Timeout)
        {
            ret = serialConn->waitForReadyRead(this->Timeout); // wait at least one byte available
            if (ret)
            {
                qint64 bav = serialConn->bytesAvailable();
                if (bav < expLength) continue;

                QByteArray recvData = serialConn->read(expLength);
                serialConn->clear();
                return recvData;
            }
        }
        return QByteArray();
#else                //Linux version works flawlessly
        bool ret = serialConn->waitForReadyRead(this->Timeout); // wait at least one byte available
        if (ret)
        {
            qint64 bav = serialConn->bytesAvailable();
            if (bav < expLength) continue;

            QByteArray recvData = serialConn->read(expLength);
            serialConn->clear();
            return recvData;
        }
        else
        {
            return QByteArray();
        }

#endif
    }
}

/////////////////////////
////PUBLIC INTERFACES////
/////////////////////////

//sleep
void DeviceDriver::qSleep(int ms)
{
#ifdef Q_OS_WIN
    Sleep(uint(ms));
#else
    struct timespec ts = { ms / 1000, (ms % 1000) * 1000 * 1000 };
    nanosleep(&ts, NULL);
#endif
}

//change name and reopen connection
int DeviceDriver::changeName(QString portName)
{

    int rate=serialConn->baudRate();
    if (serialConn->isOpen()) serialConn->close();
    return openConn(portName,rate);
}

//change rate and reopen connection
int DeviceDriver::changeRate(int baudRate)
{

    QString name=serialConn->portName();
    if (serialConn->isOpen())  serialConn->close();
    return openConn(name,baudRate);
}

void DeviceDriver::changeTimeout(int timeout)
{
    this->Timeout=timeout;
}

int DeviceDriver::getTimeout()
{
    return this->Timeout;
}

///////////////////////
////DEVICE COMMANDS////
///////////////////////

//Reset and check is device present
int DeviceDriver::reset()
{
    if (!serialConn->isOpen())
    {
        return -1;
    }
    for (int i=0;i<3;i++) //it was in original source
    {
        serialConn->clear();
        serialConn->write(QByteArray(1,'0'));
        serialConn->flush();
    }

    QString res(getResponse(8));    //Ready.\r\n
    QString readysig="Ready.";
    readysig.append(0x0d);
    readysig.append(0x0a);
    if (res==readysig)
    {
        return 0;
    }
    if (res.length()==0)
    {
        return -2;  //no response
    }
    return -3;  //bullshit retrieved
}

//turn IC power ON
int DeviceDriver::powerON()
{
    if (!serialConn->isOpen())
    {
        return -1;
    }
    serialConn->clear();
    serialConn->write(QByteArray(1,'1'));
    serialConn->flush();
    QString res(getResponse(5));    //OK.\r\n
    QString oksig="OK.";
    oksig.append(0x0d);
    oksig.append(0x0a);
    if (res==oksig)
    {
        return 0;
    }
    if (res.length()==0)
    {
        return -2;  //no response
    }
    return -3;  //bullshit retrieved
}

//sets I/O config - 0 out, 1 in (out - to tested ic)
int DeviceDriver::setIO(QString pins)
{
  //  pins=pins.replace("X","1"); //This is as in original protocol in command 2 only.
    if (!serialConn->isOpen()) return -1;
    if (pins.length()>24) return -4; //error in inp data

    //This code prepares BIT PATTERN which is string of 24 chars. They can be
    //1 or 0, when 1 means PIC's input, 0 PIC's output.
    //pins is usually shorter, so it must "align" the chip into socket.
    QString sBitPattern;
    sBitPattern=sBitPattern.fill('1',12-(pins.length()/2)); //set unused pins IN - left side
    for (int F=1;F<=pins.length();F++)
    {
        if (pins[F-1]=='0')  //is output supplying signal INTO tested IC.
        {
            sBitPattern=sBitPattern+'0';
        }
        else  //is input taking signal FROM test IC or Vcc or GND or NC.
        {
            sBitPattern=sBitPattern+'1';
        }
    }
    sBitPattern=sBitPattern.leftJustified(24,'1'); //set unused pins IN - right side

    //QMessageBox::information(NULL,"pattern",sBitPattern);
    //Now we can prepare data packet for Device

    //The packet contains of 5 bytes, command (1 byte) and data.
    //Ports B,D,A,E - this order.
    //So you can see that not all pins are not in all ports - PIC has just some ports OC.
    //and they're unsuitable for tester.

    int nTotals[4];
    for (int i=0;i<4;i++) nTotals[i]=0;

    int nMul=128;
    for (int F=0;F<8;F++)
    {
        if (sBitPattern[F]=='1') //PORTB used entirely
        {
            nTotals[0]+=nMul;
        }
        if (sBitPattern[F+8]=='1') //PORTD used entirely
        {
            nTotals[1]+=nMul;
        }
        nMul=nMul/2;
    }

    //PORTA - we're not using PA4 here.
    //we're using PA0,1,2,3,5.
    nMul=32;
    for (int F=0;F<5;F++)
    {
       if (sBitPattern[F+16]=='1')
       {
           nTotals[2]+=nMul;
       }
       if (F==0)
       {
           nMul=nMul/2; //shift out first bit;
       }
       nMul=nMul/2;
    }

    //PORTE - 3 bits are used here.
    //PE0,1,2
    nMul=4;
    for (int F=0;F<3;F++)
    {
        if (sBitPattern[F+21]=='1')
        {
            nTotals[3]+=nMul;
        }
        nMul=nMul/2;
    }

    //send command and then bytes
    serialConn->clear();
    serialConn->write(QByteArray(1,'2'));

    //send 4 bytes to COM port
    for (int F=0;F<4;F++)
    {
        unsigned char arg=nTotals[F];
 //       QMessageBox::information(NULL,"Byte",QString::number(arg,16));
        serialConn->write(QByteArray(1,arg));
    }
    serialConn->flush();

    //wait for an answer
    QString res(getResponse(5));    //OK.\r\n
    QString oksig="OK.";
    oksig.append(0x0d);
    oksig.append(0x0a);
    if (res==oksig)
    {
        return 0;
    }
    if (res.length()==0)
    {
        return -2;  //no response
    }
    return -3;  //bullshit retrieved
}

int DeviceDriver::setData(QString pins)
{
   // pins=pins.replace("X","0");
    if (!serialConn->isOpen()) return -1;
    if (pins.length()>24) return -4; //error in inp data
    QString sBitPattern;
    sBitPattern=sBitPattern.fill('1',12-(pins.length()/2)); //set unused pins IN - left side
    sBitPattern+=pins;
   // sBitPattern=sBitPattern.leftJustified(24,'0'); //set unused pins IN - right side.
    //This mess is NOT in source, it's in protocol.
    //I think setting all to 1 helps PIC, as it has not to drain
    //current supplied by pull-ups, a bit more power goes to chip.
     sBitPattern=sBitPattern.leftJustified(24,'1');

    int nTotals[4];
    for (int i=0;i<4;i++) nTotals[i]=0;

    int nMul=128;
    for (int F=0;F<8;F++)
    {
        if (sBitPattern[F]=='1') //PORTB used entirely
        {
            nTotals[0]+=nMul;
        }
        if (sBitPattern[F+8]=='1') //PORTD used entirely
        {
            nTotals[1]+=nMul;
        }
        nMul=nMul/2;
    }

    //PORTA - we're not using PA4 here.
    //we're using PA0,1,2,3,5.
    nMul=32;
    for (int F=0;F<5;F++)
    {
       if (sBitPattern[F+16]=='1')
       {
           nTotals[2]+=nMul;
       }
       if (F==0)
       {
           nMul=nMul/2; //shift out first bit;
       }
       nMul=nMul/2;
    }

    //PORTE - 3 bits are used here.
    //PE0,1,2
    nMul=4;
    for (int F=0;F<3;F++)
    {
        if (sBitPattern[F+21]=='1')
        {
            nTotals[3]+=nMul;
        }
        nMul=nMul/2;
    }

    //send command and then bytes
    serialConn->clear();
    serialConn->write(QByteArray(1,'3'));

    //send 4 bytes to COM port
    for (int F=0;F<4;F++)
    {
        unsigned char arg=nTotals[F];
      //  QMessageBox::information(NULL,"Byte",QString::number(arg,16));
        serialConn->write(QByteArray(1,arg));
    }
    serialConn->flush();

    //wait for an answer
    QString res(getResponse(5));    //OK.\r\n
    QString oksig="OK.";
    oksig.append(0x0d);
    oksig.append(0x0a);
    if (res==oksig)
    {
        return 0;
    }
    if (res.length()==0)
    {
        return -2;  //no response
    }
    return -3;  //bullshit retrieved

}

QString DeviceDriver::getData()
{
    serialConn->clear();
    serialConn->write(QByteArray(1,'4'));
    serialConn->flush();
    QByteArray res = getResponse(9);
    if ((res[4]!='O')||(res[5]!='K'))
    {
        return "-3"; //bullshit detected
    }

    unsigned char k=res[0];
    QString result=QString::number(k,2).rightJustified(8,'0');
    k=res[1];
    result += QString::number(k,2).rightJustified(8,'0');
    k=res[2];
    QString sTemp= QString::number(k,2).rightJustified(8,'0').right(6);
    result += sTemp.left(1); //A5
    result += sTemp.mid(2);
    k=res[3];
    result += QString::number(k,2).rightJustified(8,'0').right(3);

    return result;
}

QString DeviceDriver::deviceVersion()
{
    serialConn->clear();
    serialConn->write(QByteArray(1,'6'));

    QTime dieTime= QTime::currentTime().addMSecs(200);
    while( QTime::currentTime() < dieTime )
    {
        serialConn->waitForReadyRead(200);
    }

    return QString(serialConn->readAll());

}
