/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  NetCentric Computing with Object Rexx                                   */
/*  Programming Example                                                     */
/*                                                                          */
/*    IBM Corporation 1998                                                  */
/*                                                                          */
/*  Clientx.cmd  -  A Generic Extended Session Client using TCP/IP Sockets  */
/*                  supporting local commands (agents)                      */
/*                                                                          */
/*    Client program for a generic server. Ask the user for a string,       */ 
/*    send the string to the server and display the result strings.         */
/*                                                                          */
/*    Parameters:                                                           */
/*      Server:   alias name of the server                                  */
/*      Alias:    alias name of the client (optional)                       */
/*      Password: authorization password (optional)                         */
/*      Port:     server port number (optional)                             */
/*                                                                          */
/*   Requires an Agency Server (agency.cmd)                                 */
/*                                                                          */
/*  Documentation: see agency.doc in directory docs                         */
/*                                                                          */
/*--------------------------------------------------------------------------*/

Parse Arg Server Alias Password Port
if Port = '' then Port = 1925          /* Default port of command server    */
                                       /* create client for server via port */
client = .Client~new(Server, Port, Alias, scramble(Password))

if client~connected then do            /* Is client connected to server?    */
  client~listenToServer                /* Start listening to the server     */ 
  say "Please enter a command" 
  say                                  /* Run session until entering 'quit' */ 
  
  do until keyWord = 'QUIT' /* | keyWord = 'SHUTDOWN' */
    rc = client~getInput               /* get input from user               */ 
    if rc < 0 then do                  /* if error quit client session      */
      client~stopListening             /* Stop listening of the client      */
      leave
    end
    command = client~Input 
                                       /* check for local processing        */ 
    if '#' = command~substr(1,1) then do 
                                       /* local command                     */
      say "---Begin of LOCAL transmission---" 
      call localCMD command~substr(2)  
      say "---End of LOCAL transmission---" 
    end
    else                               /* remote command; send it to server */
      if 0 > client~sendinput then leave

    keyWord = command~word(1)~translate
  end 
end

exit   
            
/****************************************************************************/

::REQUIRES "clients.frm"               /* Use the clients framework         */   

/*--------------------------------------------------------------------------*/
/* Client Class definition                                                  */
/*--------------------------------------------------------------------------*/
::CLASS Client SUBCLASS tcpClient

/*--------------------------------------------------------------------------*/
::METHOD init
  use arg server, port, client, option                 
                                       /* Let the superclass do the init    */
  self~init:super(server, port, client, option)
  self~Listening = .FALSE              /* Indicate not being listening yet  */ 
                                    
  if self~sysInfo = "Windows95" | self~sysInfo = "WindowsNT" then 
    if RxFuncQuery("InstMMFuncs") then do
      rc = RxFuncAdd("InstMMFuncs", "OODialog", "InstMMFuncs")
      rc = InstMMFuncs()               /* Initialize Windows MM functions   */ 
    end
   
/*--------------------------------------------------------------------------*/
::METHOD listenToServer UNGUARDED      /* Watch for any return data         */

  self~Listening = .TRUE               /* Indicate being listening now      */
  router = .router~new(self)           /* Provide a router for this client  */
  reply                                /* Return control and  continue      */
 
  do while self~Listening
    rc = self~receive                  /* receive the returned string       */
    if rc <= 0 then leave              /* if error terminate listening      */

    router~receive(self~Output)        /* process received protocol         */                                          
  end

  self~shutdown                        /* The client can be shutdown now    */

/*--------------------------------------------------------------------------*/
::METHOD Listening ATTRIBUTE           /* Set/Get the listening status      */

/*--------------------------------------------------------------------------*/
::METHOD stopListening                 /* Stop listening the conversation   */
  if self~Listening then               /* if still listening, force a quit  */
    self~sendInput('QUIT')             /* by sending the server 'quit'      */

/*--------------------------------------------------------------------------*/
::METHOD Beep                          /* Beep service                      */
  use arg tone, duration

  reply
  rc = beep(tone, duration)

/*--------------------------------------------------------------------------*/
::METHOD play                          /* MM play service                   */
  use arg playfile, playmode 

  system = self~sysInfo
  if system = 'AIX' | system = 'LINUX' then
    say "No MultiMedia function available"
  else do
    say "Playing...."       
    if playmode = '' then              /* if nothing else specified...      */
      reply                            /* play the file concurrently        */

    if system = 'OS/2' then            /* OS/2 play function (cmd file)     */ 
      call play 'file="'playfile'"'
    else do                            /* Windows play routine (OODialog)   */ 
      ext = ''
      pos = playfile~lastpos('.')      /* get the extension of playfile     */
      if pos > 0 then                  
        ext = playfile~substr(pos+1)~translate     

      select                           /* decide based on the extension ..  */ 
        when ext = 'WAV' then
          call playsoundfile playfile, 'NO'
        when ext = 'AVI' then
          if system = 'Windows95' then
            "mplayer /PLAY" playfile
          else 
            "mplay32 /PLAY" playfile
        when ext = 'MOV' then
          "play32" playfile
        otherwise
          say 'Unsupported Multimedia format'
      end 
    end 
  end 
  rc = sysFileDelete(playfile)         /* drop the playfile                 */

/*--------------------------------------------------------------------------*/
::METHOD edit                          /* edit service                      */
  use arg file, mode 

  say "Editing...."       
  system = self~sysInfo
  if system = 'AIX' | system = 'LINUX' then
    'vi' file
  else if system = 'OS/2' then         /* OS/2 play function (cmd file)     */ 
    'epm' file '/M'                    /* OS/2 editor                       */
  else                                
    'notepad' file                     /* Windows editor                    */
  return 0   

/*--------------------------------------------------------------------------*/
::METHOD shutdown UNGUARDED            /* shutdown the client               */

  self~shutdown:super                  /* shutdown the tcpClient            */
  rc = Beep(800, 500)                  /* with a long beep                  */
  rc = Beep(600, 200)                  /* and a short one                   */


/*--------------------------------------------------------------------------*/
::CLASS Router                         /* Client Router Class               */

::METHOD init                          /* router initialisation             */
  expose client
  use arg client                                   
      
/*--------------------------------------------------------------------------*/
::METHOD receive UNGUARDED             /* receive controls from server      */
  expose client
  use arg dataStream

  do until dataStream = ''         
    parse var dataStream head "#>>" control "<<#" tail
    rr = charout(, head)               /* show the result                   */
    if control = '' then leave     

    select                             /* look for special controls ...     */
      /*--------------------------------------------------------------------*/
      /* AGENT control                                                      */
      /*--------------------------------------------------------------------*/
      when control~abbrev("Agent ") then do
                                       /* get filename from control         */
        parse var control "Agent file=" cmdfile mode
                                       /* does file exist?                  */
        file = .stream~new(cmdfile)~query('EXISTS')
                                       /* was a file extension specified    */
        if file = '' & cmdfile~lastpos('.') = 0 then
          file = cmdfile'.cmd'         /* no, try it with .cmd extension    */  
        else            
          file = cmdfile               /* yes, use original filename        */
                                       /* send content of cmdfile to server */
        rc = client~sendFile(file) 
      end

      /*--------------------------------------------------------------------*/
      /* GET control                                                        */
      /*--------------------------------------------------------------------*/
      when control~abbrev("Get ") then do
                                       /* get filename from control         */
        parse var control "Get file=" filename option
        self~get(filename, option)     /* process get protocol              */
      end

      /*--------------------------------------------------------------------*/
      /* PUT control                                                        */
      /*--------------------------------------------------------------------*/
      when control~abbrev("Put ") then do
                                       /* get filename from control         */
        parse var control "Put file=" filename option
        self~put(filename, option)     /* process put portocol              */
      end

      /*--------------------------------------------------------------------*/
      /* PLAY control                                                       */
      /*--------------------------------------------------------------------*/
      when control~abbrev("Play ") then do
                                       /* get filename from control         */
        parse var control "Play file=" filename option
        self~play(filename, option)    /* process play protocol             */
      end

      /*--------------------------------------------------------------------*/
      /* BEEP control                                                       */
      /*--------------------------------------------------------------------*/
      when control~abbrev("Beep") then do
        parse var control "Beep" tone duration .
        client~Beep(tone, duration)     /* Invoke beep service              */
      end

      /*--------------------------------------------------------------------*/
      /* End of session control                                             */
      /*--------------------------------------------------------------------*/
      when control = "End_of_session" then do
        say "End of Session"
        say "---End_of_transmission---"
        client~Listening = .FALSE      /* Indicate not listening any more   */ 
      end

      /*--------------------------------------------------------------------*/
      /* TXTMODE control                                                    */
      /*--------------------------------------------------------------------*/
      when control = "BeginTxtMode" then do 
        dataStream = tail              /* process the text string           */
                                       /* output text data until EndTxtMode */
        do txtmode = 0 while client~Listening
          do while dataStream \= ''
                                       /* look for controls in text string  */
            parse var dataStream head "#>>" control "<<#" tail

            if control = '' then do    /* no control                        */
              rr = charout(, head)     /* put out head string               */
              leave 
            end                
                                       /* terminate text mode               */
            if control = 'EndTxtMode' then do
              rr = charout(, head)     /* show the result                   */
              leave txtmode
            end 
                                       /* put out everything but the tail   */
            rr = charout(, dataStream~substr(1, dataStream~length-tail~length))
            dataStream = tail
          end 

          rc = client~receive          /* receive more data into output     */
          if rc <= 0 then do 
            client~Listening = .FALSE  /* indicate not listening any more   */ 
            leave                      /* if error, terminate listening     */
          end 
          dataStream = client~Output   /* get the output received           */
        end
      end

      /*--------------------------------------------------------------------*/
      /* UNKNOWN  control                                                   */
      /*--------------------------------------------------------------------*/
      otherwise 
        self~tryAgent(control)         /* process play protocol             */
    end  
        
    dataStream = tail                  /* now process the tail string       */
  end 
      
/*--------------------------------------------------------------------------*/
::METHOD get UNGUARDED                 /* process the client get protocol   */
  expose client
  use arg filename, option
                                       /* remove the path from filename     */         
  fname = filespec('name', filename)
                                       /* replace option specified ?        */
  repl = 'REPLACE'~abbrev(option~translate, 3)
                                       /* check option validity             */ 
  if option \= '' & \repl then do
    say "Specified option unknown:" Option
    rc = client~send("#>>Error<<#")  
  end
  else if '' \= .stream~new(fname)~query('EXISTS') & \repl then do 
    say 'File' fname 'already exists; specify replace'
    rc = client~send("#>>Error<<#")  
  end
  else do                              /* ask server for file content       */
    rc = client~send("#>>Acknowledge<<#")  
                                       /* receive content in file now       */ 
    if client~receiveFile(fname) > 0 then 
      say 'File' fname 'successfully received'           
    else 
      say 'Error receiving file' fname 
  end 
      
/*--------------------------------------------------------------------------*/
::METHOD put UNGUARDED                 /* process the client put protocol   */
  expose client
  use arg filename, option

  rc = client~sendFile(filename)       /* send content of file to server    */
      
/*--------------------------------------------------------------------------*/
::METHOD play UNGUARDED                /* process the client play protocol  */
  expose client
  use arg filename, option
                                       /* remove the path from file name    */
  fname = filespec('name', filename)
  pos = fname~lastpos('.') 
  if pos > 0 then                      /* correct extension is required     */
    fname = 'temp????'fname~substr(pos)     
  else                                 /* assume wav sound file             */
    fname = 'temp????.wav'
                                       /* generate a temporary filename     */ 
  tempfile = sysTempFileName(fname)     
                                       /* tell server ready to receive      */
  rc = client~send("#>>Acknowledge<<#")  
  say 'Receiving' filename                                          
                                       /* write content of playfile to file */ 
  if client~receiveFile(tempfile) > 0 then  
                                       /* invoke MM play service            */
    client~play(tempfile, option)
  else
    say 'Error receiving playfile' filename 

/*--------------------------------------------------------------------------*/
::METHOD tryAgent UNGUARDED            /* process special client protocol   */
  expose client
  use arg control

  command = control~word(1) 
  if command~lastpos('.') = 0 then     /* was a file extension specified?   */
    agentfile = command'.orx'          /* no, try it with .orx extension    */  
  else 
    agentfile = command                /* use the original filename         */  
 
  fullname = .stream~new(agentfile)~query('EXISTS')
  if fullname = '' then do             /* Unknown command handling          */
    say "Local agent" command "unknown; control not recognized" 
  end  
  else do
    signal on syntax                   /* to intercept any problems         */  
                                       /* create a local agent              */ 
    agent = .residentAgent~new(agentfile)     
                                       /* process agent with opt. parameter */
    agent~dispatch(control~subword(2), client)  
  end 
  return
            
syntax:                                  
  except = condition('o')              /* report the problem                */ 
  say 'Error in local agent "'except~program'" line' except~position':'  
  do traceline over except~traceback 
    say traceline
  end 
  say sigl~right(6) '*-*' 'SOURCELINE'(sigl)
  say 'Error' except~rc except~errortext 
  say 'Error' except~code except~message


/*--------------------------------------------------------------------------*/
::ROUTINE scramble                     /* scramble the password             */
  use arg word

  return word~reverse                  /* simple scrambling function        */

/*--------------------------------------------------------------------------*/
::ROUTINE localCMD                     /* Process local command             */
  parse arg command parms

  select 
    when command = '?' then            /* simple command query              */
      say "Local commands: ?, ??" 

    when command = '??' then           /* command query with filter         */
      say "Local agents:" agentList(parms)

    otherwise do                       /* resident agent                    */
      if command~lastpos('.') = 0 then /* was a file extension specified?   */
        agentfile = command'.orx'      /* no, try it with .orx extension    */  
      else 
        agentfile = command            /* use the original filename         */  
      
      fullname = .stream~new(agentfile)~query('EXISTS')
      if fullname = '' then do         /* Unknown command handling          */
        say "Invalid local agent:" Command 
        say "Local agents:" agentList()
      end  
      else do
        signal on syntax               /* to intercept any problems         */  
                                       /* create a local agent              */ 
        agent = .residentAgent~new(agentfile)     
                                       /* process agent with opt. parameter */
        agent~dispatch(parms)  
      end 
    end 
  end 
  return
            
syntax:                                  
  except = condition('o')              /* report the problem                */ 
  say 'Error in local agent "'except~program'" line' except~position':'  
  do traceline over except~traceback 
    say traceline
  end 
  say sigl~right(6) '*-*' 'SOURCELINE'(sigl)
  say 'Error' except~rc except~errortext 
  say 'Error' except~code except~message


/*--------------------------------------------------------------------------*/
::CLASS residentAgent                  /* resident Agent Class              */

::METHOD init                          /* Agent initialisation              */
  use arg fileName                                 

  signal on syntax          
                                       /* Create method object from array   */  
  agentmethod = .method~newFile(fileName)
                                       /* method available with 'dispatch'  */
  self~setmethod('DISPATCH', agentmethod)  
  return

syntax:
  raise propagate  

/*--------------------------------------------------------------------------*/
::ROUTINE agentlist                    /* Generic fct providing agentlist   */

  if arg(1) = '' then 
    search = '*.orx'
  else 
    search = arg(1)'.orx'
  
  rc = sysfiletree(search, output.)    /* retrieve all .orx files           */ 
  if rc > 0 then return '<System ERROR: out of memory>'
   
  if output.0 = 0 then
    agents = '<none>'
  else do
    agents = ''
    do i = 1 to output.0               /* extract filename of each entry    */ 
      agent = filespec('name', output.i)
      pos = agent~lastpos('.')
      if pos > 0 then
        agent = agent~delstr(pos)
      agents = agents"," agent         /* include this command in the list  */
    end
    agents = agents~substr(3)
  end
  return agents                        /* return the list of local agents   */


