/********************************************************/
/*                                                      */
/* RCLIENT.CMD - IBM REXX Sample Program                */
/*                                                      */
/* Client program for the remote control server. The    */
/* client connects to the specified server and port     */
/* address. Several remote commands can be executed     */
/* until QUIT is entered.                               */
/*                                                      */
/* Parameters:                                          */
/*   Server: alias name of remote control server with   */
/*           optional port number (after a colon), e.g. */
/*           loopback:1234                              */
/*                                                      */
/* ---------------------------------------------------- */
/* (C) Copyright IBM Corp. 1996 - All Rights Reserved.  */
/*                                                      */
/* DISCLAIMER OF WARRANTIES.  The following [enclosed]  */
/* code is sample code created by IBM Corporation. This */
/* sample code is not part of any standard or IBM       */
/* product and is provided to you solely for the        */
/* purpose of assisting you in the development of your  */
/* applications.  The code is provided "AS IS", without */
/* warranty of any kind.  IBM shall not be liable for   */
/* any damages arising out of your use of the sample    */
/* code,  even if they have been advised of the         */
/* possibility of such damages.                         */
/*                                                      */
/********************************************************/
Parse Arg Server

/* check command line arguments, server is required     */
If Server = "" Then
 Do
   Say "Usage: RCLIENT Servername"
   Say "  Servername may contain a port number separated",
       "with a colon."
   Exit 1
 End

/* Load REXX Socket library if not already loaded       */
If RxFuncQuery("SockLoadFuncs") Then
 Do
   Call RxFuncAdd "SockLoadFuncs","RXSOCK","SockLoadFuncs"
   Call SockLoadFuncs
 End

/* Connect to remote control server                     */
Socket = Connect(Server)
If Socket = -1 Then
  Exit 1

/* loop until QUIT command was entered                  */
Do Until Command = "QUIT"
  Say "Please enter one of: 'DIR [path]', 'TYPE name'",
      " or 'QUIT'"
  Parse Pull CommandLine
  Parse Upper Var CommandLine Command Option
  If Length(Command) > 0 Then
    Call SendCommand Socket, CommandLine
End

/* Close connection to server                           */
Call Close Socket
Exit


/********************************************************/
/*                                                      */
/* Function:  Connect                                   */
/* Purpose:   Create a socket and connect it to server. */
/* Arguments: Server - server name, may contain port no.*/
/* Returns:   Socket number if successful, -1 otherwise */
/*                                                      */
/********************************************************/
Connect: Procedure
  Parse Arg Server

  /* if the servername has a port address specified     */
  /* then use this one, otherwise use the default port  */
  /* for the remote control server (1234)               */
  Parse Var Server Server ":" Port
  If Port = "" Then
    Port = 1234

  /* resolve server name alias to dotted IP address     */
  rc = SockGetHostByName(Server, "Host.!")
  If rc = 0 Then
   Do
     Say "Unable to resolve server:" Server
     Return -1
   End

  /* create a TCP socket                                */
  Socket = SockSocket("AF_INET", "SOCK_STREAM", "0")
  If Socket < 0 Then
   Do
     Say "Unable to create socket"
     Return -1
   End

  /* connect the new socket to the specified server     */
  Host.!family = "AF_INET"
  Host.!port = Port
  rc = SockConnect(Socket, "Host.!")
  If rc < 0 Then
   Do
     Say "Unable to connect to server:" Server
     Call Close Socket
     Return -1
   End

  Return Socket


/********************************************************/
/*                                                      */
/* Procedure: SendCommand                               */
/* Purpose:   Send a command via the specified socket   */
/*            and display the full response from server.*/
/* Arguments: Socket - active socket number             */
/*            Command - command string                  */
/* Returns:   nothing                                   */
/*                                                      */
/********************************************************/
SendCommand: Procedure
  Parse Arg Socket, Command

  /* send the command to the remote control server      */
  Call SockSend Socket, Command
  Do Forever
    BytesRcvd = SockRecv(Socket, "RcvData", 1024)

    /* error or end of response encountered             */
    If BytesRcvd <= 0 |,
       RcvData = ">>>End_of_transmission<<<" Then
      Leave

    /* display response and send acknowledge to server  */
    Say RcvData
    Call SockSend Socket, "OK!"
  End

  Say "----- end of output from command:" Command "-----"
  Return


/********************************************************/
/*                                                      */
/* Procedure: Close                                     */
/* Purpose:   Close the specified socket.               */
/* Arguments: Socket - active socket number             */
/* Returns:   nothing                                   */
/*                                                      */
/********************************************************/
Close: Procedure
  Parse Arg Socket

  Call SockShutDown Socket, 2
  Call SockClose Socket
  Return
