/*   Licensed Materials - Property of IBM               */
/*							*/
/*   5765-530            				*/
/*   (C) Copyright IBM Corp. 1996 All Rights Reserved.	*/
/*							*/

/******************************************************************************
*
*	File		: drivers/ing64/src/sql_connect.sc, drivers.unix, wm.v1
* 
*	Date		: 95/06/12
*
*	Language	: C
*
*	Operating System: AIX & SOLARIS & HPUX
*
*	Part of		: Ingres Database Driver
*
*	Functions	: Perform SQL functions, connect and fetch data
*
*
*****************************************************************************/

/******************************************************************************
*				Includes
*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <malloc.h>
#include <errno.h>
#include <string.h>
#include <iconv.h>

#include <files/generr.h>

#include "ipc.h"
#include "portname.h"
#include "msg.h"
#include "trace.h"
#include "iconv_defs.h"


EXEC SQL INCLUDE SQLDA;
EXEC SQL INCLUDE SQLCA;

/******************************************************************************
*				Defines
*****************************************************************************/

#define MAX_CURSORS	10	/* maximum number of concurrent cursors */
#define BIND_SIZE	10
#define IND_SIZE	30
#define MAX_STATEMENT	20480
#define PATH_TEMPLATE	"%s/ingres/utility:%s/ingres/bin:%s"

/******************************************************************************
*				Globals
*****************************************************************************/

EXEC SQL BEGIN DECLARE SECTION;
	char	idbname[15];
	char	iuser[20];
	char	iopts[20];
	char	statm[20480];
	char	errbuf[1024];
EXEC SQL END DECLARE SECTION;

char s[MAX_STATEMENT];

static IISQLDA *select_p0;	/* MAX_CURSORS pieces */ 
static IISQLDA *select_p1;
static IISQLDA *select_p2;
static IISQLDA *select_p3;
static IISQLDA *select_p4;
static IISQLDA *select_p5;
static IISQLDA *select_p6;
static IISQLDA *select_p7;
static IISQLDA *select_p8;
static IISQLDA *select_p9;

static IISQLDA 	*bind_dp[MAX_CURSORS];
static char 	da_inuse[MAX_CURSORS];
static int 	sd_size = SELECT_SIZE;
static int	connected = FALSE; /* TRUE if a connection with ingres exists */
char *dhu_portname = ING64D_PORTNAME;

extern IISQLDA 	*sqlda = (IISQLDA *)0;
extern int	curr_tracelevel;
extern char 	dbname[];

/******************************************************************************
*				Function Prototypes
*****************************************************************************/

void sql_fetch(IISQLDA *select_p);
void sql_get_desc(int index,  IISQLDA **select_p);
void do_close(int);
void desc_pl_list(IISQLDA *sdp, struct field_desc_t *list);

static void sql_convert_types(IISQLDA *select_p);
static char *get_type_name(int index);
static void sql_error( int sqlstat);

IISQLDA	*do_fetch( int index);


/*
 * Function  : sqlCommand5( char *, int *, struct field_desc_t *, 
 *			int *)
 * Arguments 
 *	arg1 :	SQL statement to execute
 *	arg2 :	Cursor for open statement 
 *	arg3 :	List of column descriptions 
 *	arg4 :	Number of columns in list
 * Return Value:
 *	0 on success, -1 on error
 * Comments  :
 *	Executes an SQL statement, returns the errorcode and message from
 *	database, and a list of column descriptions and a cursor if
 *	statement is select statement.
 * Author    : Marc Gathier
 * History 
 *	Created 	: Fri Sep 30 13:36:40 CET 1994
 *	Modifications	:
 *
 *
 */
int
sqlCommand5( char *s, int *descriptor, struct field_desc_t *list, 
	int *att_count)
{
	int	i, sqlstat,
		index;
	IISQLDA *bp;
	IISQLDA *temp_sp;
  
	TRACE_FUNC_ENTRY( 1, "sqlCommand5");
	TRACE_FUNC_DATA( 1, STRING, s);

        if( (int)strlen( s) > MAX_STATEMENT) {
                *att_count = 0;
                sqlstat = STATEMENT_TOO_LONG;
		append_message( "statement too long", sqlstat);
                TRACE_FUNC_EXIT( 1, "sqlCommand5", INT, (char *)&sqlstat);
                return( -1);
        }
	strcpy(statm, s);
  
	/* Find a free SQLDA for statement */
	index = get_free_desc();
	if (index < 0) {
		*att_count = 0;
		sqlstat = TOO_MANY_CONC_CURSORS;
		append_message( "too many concurrent cursors", sqlstat);
		TRACE_FUNC_EXIT( 1, "sqlCommand5", INT, (char *)&sqlstat);
    		return( -1);
	}
  
	/* Allocate memory for SQLDA */
  	bind_dp[index] = (IISQLDA *)calloc(1, 
			IISQDA_HEAD_SIZE + (SELECT_SIZE * IISQDA_VAR_SIZE));
	temp_sp = (IISQLDA *)calloc(1, 
			IISQDA_HEAD_SIZE + (SELECT_SIZE * IISQDA_VAR_SIZE));
  	if (bind_dp[index] == NULL || temp_sp == NULL) {
		*att_count = 0;
		sqlstat = CANNOT_ALLOCATE_MEMORY;
		append_message( "cannot allocate memory", sqlstat);
		TRACE_FUNC_EXIT( 1, "sqlCommand5", INT, (char *)&sqlstat);
    		return( -1);
  	}

	bind_dp[index]->sqln = SELECT_SIZE;
	temp_sp->sqln = SELECT_SIZE;

	EXEC SQL WHENEVER SQLERROR GOTO sql_err;

	switch (index) {
	case 0: 
		EXEC SQL PREPARE S0 FROM :statm; 
		break;
	case 1: 
		EXEC SQL PREPARE S1 FROM :statm; 
		break;
	case 2: 
		EXEC SQL PREPARE S2 FROM :statm; 
		break;
	case 3: 
		EXEC SQL PREPARE S3 FROM :statm; 
		break;
	case 4: 
		EXEC SQL PREPARE S4 FROM :statm; 
		break;
	case 5: 
		EXEC SQL PREPARE S5 FROM :statm; 
		break;
	case 6: 
		EXEC SQL PREPARE S6 FROM :statm; 
		break;
	case 7: 
		EXEC SQL PREPARE S7 FROM :statm; 
		break;
	case 8: 
		EXEC SQL PREPARE S8 FROM :statm; 
		break;
	case 9: 
		EXEC SQL PREPARE S9 FROM :statm; 
		break;
	default:
		fprintf(stderr,"index out of range %d in sqlCommand5\n", index);
	}

	bp = bind_dp[index];

	switch (index) {
	case 0: 
		EXEC SQL DESCRIBE S0 INTO bp; 
		break;
	case 1: 
		EXEC SQL DESCRIBE S1 INTO bp; 
		break;
	case 2: 
		EXEC SQL DESCRIBE S2 INTO bp; 
		break;
	case 3: 
		EXEC SQL DESCRIBE S3 INTO bp; 
		break;
	case 4: 
		EXEC SQL DESCRIBE S4 INTO bp; 
		break;
	case 5: 
		EXEC SQL DESCRIBE S5 INTO bp; 
		break;
	case 6: 
		EXEC SQL DESCRIBE S6 INTO bp; 
		break;
	case 7: 
		EXEC SQL DESCRIBE S7 INTO bp; 
		break;
	case 8: 
		EXEC SQL DESCRIBE S8 INTO bp; 
		break;
	case 9: 
		EXEC SQL DESCRIBE S9 INTO bp; 
		break;
	default:
		fprintf(stderr,"index out of range %d in sqlCommand5\n", index);
	}

	*att_count = bp->sqld;
  	if (bp->sqld != 0) {

		/* Statement is a SELECT */
  
		switch (index) {
		case 0: 
			EXEC SQL DECLARE C0 CURSOR FOR S0; 
			break;
		case 1: 
			EXEC SQL DECLARE C1 CURSOR FOR S1; 
			break;
		case 2: 
			EXEC SQL DECLARE C2 CURSOR FOR S2; 
			break;
		case 3: 
			EXEC SQL DECLARE C3 CURSOR FOR S3; 
			break;
		case 4: 
			EXEC SQL DECLARE C4 CURSOR FOR S4; 
			break;
		case 5: 
			EXEC SQL DECLARE C5 CURSOR FOR S5; 
			break;
		case 6: 
			EXEC SQL DECLARE C6 CURSOR FOR S6; 
			break;
		case 7: 
			EXEC SQL DECLARE C7 CURSOR FOR S7; 
			break;
		case 8: 
			EXEC SQL DECLARE C8 CURSOR FOR S8; 
			break;
		case 9: 
			EXEC SQL DECLARE C9 CURSOR FOR S9; 
			break;
		default:
			fprintf(stderr,"index out of range %d in sqlCommand5\n",
				index);
		}
  
		bind_dp[index]->sqln = bind_dp[index]->sqld;
		switch (index) {
		case 0: 
			EXEC SQL OPEN C0 ; /* USING DESCRIPTOR bp; */
			break;
		case 1: 
			EXEC SQL OPEN C1;
			break;
		case 2: 
			EXEC SQL OPEN C2;
			break;
		case 3: 
			EXEC SQL OPEN C3;
			break;
		case 4: 
			EXEC SQL OPEN C4;
			break;
		case 5: 
			EXEC SQL OPEN C5;
			break;
		case 6: 
			EXEC SQL OPEN C6;
			break;
		case 7: 
			EXEC SQL OPEN C7;
			break;
		case 8: 
			EXEC SQL OPEN C8;
			break;
		case 9: 
			EXEC SQL OPEN C9;
			break;
		default:
			fprintf(stderr,"index out of range %d in sqlCommand5\n",
				index);
		}
	}	/* Statement is SELECT */
	else	/* Other Statement */
		switch( index) {
		case 0: 
			EXEC SQL EXECUTE S0;
			break;
		case 1: 
			EXEC SQL EXECUTE S1;
			break;
		case 2: 
			EXEC SQL EXECUTE S2;
			break;
		case 3: 
			EXEC SQL EXECUTE S3;
			break;
		case 4: 
			EXEC SQL EXECUTE S4;
			break;
		case 5: 
			EXEC SQL EXECUTE S5;
			break;
		case 6: 
			EXEC SQL EXECUTE S6;
			break;
		case 7: 
			EXEC SQL EXECUTE S7;
			break;
		case 8: 
			EXEC SQL EXECUTE S8;
			break;
		case 9: 
			EXEC SQL EXECUTE S9;
			break;
		default:
			fprintf(stderr,"index out of range %d in sqlCommand5\n",
				index);
		}

	sql_get_desc(index, &temp_sp); 
  
	switch (index) {
	case 0: select_p0 = temp_sp; break;
	case 1: select_p1 = temp_sp; break;
	case 2: select_p2 = temp_sp; break;
	case 3: select_p3 = temp_sp; break;
	case 4: select_p4 = temp_sp; break;
	case 5: select_p5 = temp_sp; break;
	case 6: select_p6 = temp_sp; break;
	case 7: select_p7 = temp_sp; break;
	case 8: select_p8 = temp_sp; break;
	case 9: select_p9 = temp_sp; break;
	default:
		fprintf(stderr,"index out of range %d in sqlCommand5\n", index);
	}
  
	free_sqlda( bind_dp[index]);

	if (temp_sp->sqln > 0) {
		sql_convert_types(temp_sp);
		desc_pl_list(temp_sp, list);
		*descriptor = index;
	}
	else {
		free_sqlda( temp_sp);
		da_inuse[index] = FALSE;
		*descriptor = -1;
	}
  
	sqlstat = sqlca.sqlcode;
	sql_error( sqlstat);
  
	TRACE_FUNC_EXIT( 1, "sqlCommand5", INT, (char *)&sqlstat);
	return(0);

sql_err:
	EXEC SQL WHENEVER SQLERROR CONTINUE;
	
	free_sqlda(bind_dp[index]);
	free_sqlda(temp_sp);
	da_inuse[index] = FALSE;
	sqlstat = sqlca.sqlcode;
	sql_error( sqlstat);
	TRACE_FUNC_EXIT( 1, "sqlCommand5", INT, (char *)&sqlstat);
	return(0);
}

static void
sql_error( int sqlstat)
{
	EXEC SQL INQUIRE_SQL ( :errbuf = errortext);
	append_message( errbuf, sqlstat);
}

/*
 * Function  : sql_get_desc( int, IISQLDA **)
 * Arguments 
 *	arg1 :	Cursor for statement
 *	arg2 :	SQLDA pointer
 * Return Value:
 *	None
 * Comments  :
 *	Describes the columns for select in SQLDA structure.
 * Author    : Marc Gathier
 * History 
 *	Created 	: Fri Sep 30 14:03:55 CET 1994
 *	Modifications	:
 *
 *
 */
void
sql_get_desc(int index, IISQLDA **select_p)
{
	IISQLDA *p = *select_p;

	TRACE_FUNC_ENTRY( 2, "sql_get_desc");
	TRACE_FUNC_DATA( 2, INT, (char *)&index);
  
	switch (index) {
	case 0: 
		EXEC SQL DESCRIBE S0 INTO p; 
		break;
	case 1: 
		EXEC SQL DESCRIBE S1 INTO p; 
		break;
	case 2: 
		EXEC SQL DESCRIBE S2 INTO p; 
		break;
	case 3: 
		EXEC SQL DESCRIBE S3 INTO p; 
		break;
	case 4: 
		EXEC SQL DESCRIBE S4 INTO p; 
		break;
	case 5: 
		EXEC SQL DESCRIBE S5 INTO p; 
		break;
	case 6: 
		EXEC SQL DESCRIBE S6 INTO p; 
		break;
	case 7: 
		EXEC SQL DESCRIBE S7 INTO p; 
		break;
	case 8: 
		EXEC SQL DESCRIBE S8 INTO p; 
		break;
	case 9: 
		EXEC SQL DESCRIBE S9 INTO p; 
		break;
	default:
		fprintf(stderr,"index out of range %d in sql_get_desc\n", 
			index);
	}

	if (p->sqld < 0) {
		sd_size = -(p->sqld);
		free_sqlda(p);
		*select_p = (IISQLDA *)calloc(1, 
			IISQDA_HEAD_SIZE + (SELECT_SIZE * IISQDA_VAR_SIZE));
		p = *select_p;
		switch (index) {
		case 0: 
			EXEC SQL DESCRIBE S0 INTO p; 
			break;
		case 1: 
			EXEC SQL DESCRIBE S1 INTO p; 
			break;
		case 2: 
			EXEC SQL DESCRIBE S2 INTO p; 
			break;
		case 3: 
			EXEC SQL DESCRIBE S3 INTO p; 
			break;
		case 4:
			EXEC SQL DESCRIBE S4 INTO p; 
			break;
		case 5: 
			EXEC SQL DESCRIBE S5 INTO p; 
			break;
		case 6: 
			EXEC SQL DESCRIBE S6 INTO p; 
			break;
		case 7: 
			EXEC SQL DESCRIBE S7 INTO p; 
			break;
		case 8: 
			EXEC SQL DESCRIBE S8 INTO p; 
			break;
		case 9: 
			EXEC SQL DESCRIBE S9 INTO p; 
			break;
		default:
			fprintf(stderr,"index out of range %d in sql_get_desc\n"
				, index);
		}
	}
	p->sqln = p->sqld;
	TRACE_FUNC_EXIT( 2, "sql_get_desc", INT, (char *)&p->sqln);
}

/*
 * Function  : free_desc( IISQLDA *, int)
 * Arguments 
 *	arg1 :	pointer to the SQLDA
 *	arg2 :	index in the bind descriptor table
 * Return Value:
 *	None
 * Comments  :
 *	Frees the memory used for a SQLDA, after a statement is executed or
 *	cursor is closed
 * Author    : Marc Gathier
 * History 
 *	Created 	: Fri Sep 30 14:05:38 CET 1994
 *	Modifications	:
 *
 *
 */
static void 
free_sqlda_number(int index)
{
	int	i;
	IISQLDA	*select_p;

	TRACE_FUNC_ENTRY( 2, "free_sqlda_number");
	TRACE_FUNC_DATA( 2, INT, (char *)&index);
  
	switch (index) {
	case 0: select_p = select_p0 ; break;
	case 1: select_p = select_p1 ; break;
	case 2: select_p = select_p2 ; break;
	case 3: select_p = select_p3 ; break;
	case 4: select_p = select_p4 ; break;
	case 5: select_p = select_p5 ; break;
	case 6: select_p = select_p6 ; break;
	case 7: select_p = select_p7 ; break;
	case 8: select_p = select_p8 ; break;
	case 9: select_p = select_p9 ; break;
	default:
		fprintf(stderr,"index out of range %d in free_sqlda_number\n", 
			index);
	}
  
	free_sqlda(select_p);
	TRACE_FUNC_EXIT( 2, "free_sqlda_number", NONE, 0);
}

free_sqlda( IISQLDA *p)
{
	int i;

	if( ! p) return;

	for( i = 0 ; i < p->sqld ; i++) {
		if( p->sqlvar[i].sqldata)
			free( p->sqlvar[i].sqldata);
		if( p->sqlvar[i].sqlind)
			free( p->sqlvar[i].sqlind);
	}

	free( p);
}

int 
sqlFetchN(int sock, int index, int count)
{
	int 		i, sendall, records = count;
	short 		*indicator;
	struct field_t	field;
	IISQLDA 	*dp;

	TRACE_FUNC_ENTRY( 1, "sqlFetchN");
	TRACE_FUNC_DATA( 1, INT, (char *)&sock);
	TRACE_FUNC_DATA( 1, INT, (char *)&index);
	TRACE_FUNC_DATA( 1, INT, (char *)&count);

	/* Check whether a valid cursor is specified */
  
	if( count == -1) sendall = 1;
	else sendall = 0;

	if (!da_inuse[index]) {
		field.type = INVALID_CURSOR;
		Write( sock, (char *) &field, sizeof( field), "sqlFetchN", 
			FIELD_T);
		TRACE_FUNC_EXIT( 1, "sqlFetchN", INT, (char *)&field.type);
		return( 0);
	}

	while( sendall || count-- > 0) {
		/* do not place fetch in while condition or we lose values */
		if(!(dp = do_fetch( index)) || sqlca.sqlcode != GE_OK)
			break;
    
		field.indicator = 0;
		field.error = 0;
		field.value = s;
		field.type = TEXT;

  		for(i = dp->sqld-1; i >= 0; i--) {

      			indicator = (short *) dp->sqlvar[i].sqlind;
      			if ( dp->sqlvar[i].sqltype < 0 && *indicator < 0) {
				field.indicator = 1;
				field.error = 0;
				strcpy( s, "");
      			} else {
				field.indicator = 0;
				field.error = 0;
				get_field( dp->sqlvar[i].sqltype, 
					dp->sqlvar[i].sqllen, 
					dp->sqlvar[i].sqldata,
					s);
			}

    			field.len = strlen( s) + 1;
    			Write( sock, (char *) &field, sizeof( field), 
					"sqlFetchN", FIELD_T);

		}

  		field.type = NOMORE;
  		Write( sock, (char *) &field, sizeof( field), "sqlFetchN", 
						FIELD_T);
	}

	if( !(records == 1 && sqlca.sqlcode == GE_OK)) {
		if( sqlca.sqlcode == GE_NO_MORE_DATA
		||  sqlca.sqlcode == GE_OK) {
			field.type = NOT_FOUND;
			Write( sock, (char *) &field, sizeof( field), 
				"sqlFetchN", FIELD_T);
		} else {
			field.indicator = 0;
			field.type = TEXT;
			field.error = get_error_code();
			field.value = get_message();
			field.len = strlen( field.value) + 1;
			Write( sock, (char *) &field, sizeof( field),
				"sqlFetchN", FIELD_T);
		}
	}

	/* All records sent, Free memory for open statement */
	/* if all send or out of data */
	if( sqlca.sqlcode == GE_NO_MORE_DATA/*sendall || count > 0*/) {
		free_sqlda_number(index); 
		do_close(index);
		da_inuse[index] = FALSE;
	}


	TRACE_FUNC_EXIT( 1, "sqlFetchN", INT, (char *)&field.type);
	return( 0);
}


/*
 * Function  : sqlCloseCursor( int, int)
 * Arguments 
 *	arg1 :	Socket
 *	arg2 :	Cursor
 * Return Value:
 *	0
 * Comments  :
 *	Closes an open cursor
 * Author    : Marc Gathier
 * History 
 *	Created 	: Fri Sep 30 14:36:10 CET 1994
 *	Modifications	:
 *
 *
 */
int 
sqlCloseCursor(int index)
{
	TRACE_FUNC_ENTRY( 1, "sqlCloseCursor");
	TRACE_FUNC_DATA( 1, INT, (char *)&index);

	if( index < 0 || index >= MAX_CURSORS || !da_inuse[index] ) {
		TRACE_FUNC_EXIT( 1, "sqlCloseCursor", NONE, 0);
		return( INVALID_CURSOR);
	}


	free_sqlda_number(index); 
	do_close(index);
	da_inuse[index] = FALSE;

	TRACE_FUNC_EXIT( 1, "sqlCloseCursor", NONE, 0);
	return( 0);
}

/*
 * Function  : sqlConnectU( char *, char *, int *)
 * Arguments 
 *	arg1 :	Username
 *	arg2 :	Options
 *	arg3 :	Return Status
 * Return Value:
 *	0
 * Comments  :
 *	Changes the current Ingres user to specified user
 * Author    : Marc Gathier
 * History 
 *	Created 	: Fri Sep 30 14:36:10 CET 1994
 *	Modifications	:
 *
 *
 */
int
sqlConnectU( char *username, char *options)
{
	int sqlstat;
	TRACE_FUNC_ENTRY( 1, "sqlConnectU");
	TRACE_FUNC_DATA( 1, STRING, username);
	TRACE_FUNC_DATA( 1, STRING, options);

	EXEC SQL COMMIT ;
	EXEC SQL DISCONNECT ;

	EXEC SQL set lockmode session where readlock=nolock;

	strcpy( iuser, username);
	if( options != NULL && strcmp( options, "") ) {
		strcpy( iopts, options);
		EXEC SQL CONNECT :idbname identified by :iuser options = :iopts;
	}
	else {
		EXEC SQL CONNECT :idbname identified by :iuser;
	}

	sqlstat = sqlca.sqlcode;
	sql_error( sqlstat);

	if( sqlstat)
		EXEC SQL CONNECT :idbname ;

	TRACE_FUNC_EXIT( 1, "sqlConnectU", INT, (char *)&sqlstat);
	return( 0);
}

/*
 * Function  : sqlReconnect( )
 * Arguments 
 * Return Value:
 *	0
 * Comments  :
 *	Undoes the sqlConnectU and reconnects to original user
 * Author    : Marc Gathier
 * History 
 *	Created 	: Fri Sep 30 14:36:10 CET 1994
 *	Modifications	:
 *
 *
 */
int
sqlReconnect()
{
	int sqlstat;

	TRACE_FUNC_ENTRY( 1, "sqlReconnect");

	EXEC SQL COMMIT ;
	EXEC SQL DISCONNECT ;
	EXEC SQL set lockmode session where readlock=nolock;
	EXEC SQL CONNECT :idbname ;

	sqlstat = sqlca.sqlcode;
	sql_error( sqlstat);

	TRACE_FUNC_EXIT( 1, "sqlReconnect", INT, (char *)&sqlstat);
	return( 0);
}

/*
 * Function  : sqlConnect( struct sql_conn, int)
 * Arguments 
 *	arg1 :	Information for connect
 *	arg2 :	Connect type, DBMCONNECT or CONNECT
 *      arg3 :  dynamically allocated error message, space should be freed
 *              by the caller
 * Return Value:
 *	Return status of connect, 0 if Connected, < 0 if connect failed
 * Comments  :
 *	Changes the User ID of current process to specified user if password
 *	is correct and tries to connect to the database if type is CONNECT.
 * Author    : Marc Gathier
 * History 
 *	Created 	: Fri Sep 30 14:44:43 CET 1994
 *	Modifications	:
 *
 *
 */
int
sqlConnect( struct sql_conn conn, int options)
{
	char *path, *newpath;

	TRACE_FUNC_ENTRY( 1, "sqlConnect");
	TRACE_FUNC_DATA( 1, SQL_CONN, (char *)&conn);
	TRACE_FUNC_DATA( 1, INT, (char *)&options);

	if( ! SetUid( conn.name, conn.password) ) {
		append_message( "Unix Login Denied", -1);
		TRACE_FUNC_EXIT( 1, "sqlConnect", STRING, "Unix Login Denied");
		return( -1);
	}

	Setenv( "II_SYSTEM", conn.homedir);
  	strcpy( dbname, conn.dbname);

	path = getenv("PATH");
	newpath = malloc( strlen( path) + 
			  2 * strlen(conn.homedir) + 
			  strlen(PATH_TEMPLATE));
	sprintf( newpath, PATH_TEMPLATE, conn.homedir, conn.homedir, path);
	Setenv( "PATH", newpath);
	free( newpath);

	connected = FALSE;
  	if( options & OPTION_CONNECT) {
  		strcpy( idbname, conn.dbname);
		EXEC SQL set lockmode session where readlock=nolock;
  		EXEC SQL CONNECT :idbname;
  	}
	
  	TRACE_FUNC_EXIT( 1, "sqlConnect", INT, (char *)&(sqlca.sqlcode));

	sql_error( sqlca.sqlcode);

	if( !sqlca.sqlcode)
		connected = TRUE;

  	return( sqlca.sqlcode);
}

/*
 * Function  : sqlDisconnect()
 * Arguments 
 *	None
 * Return Value:
 *	None
 * Comments  :
 *	Commits work and disconnects from database, stops current process
 * Author    : Marc Gathier
 * History 
 *	Created 	: Fri Sep 30 14:46:46 CET 1994
 *	Modifications	:
 *
 *
 */
int 
sqlDisconnect()
{
	int i;

	TRACE_FUNC_ENTRY( 1, "sqlDisconnect");
  
	if( !connected)
		return 0;

	for (i = 0; i < MAX_CURSORS; i++) 
		if (da_inuse[i]) {
			free_sqlda_number(i); 
			do_close(i);
			da_inuse[i] = FALSE;
		}
  
	EXEC SQL WHENEVER SQLERROR GOTO sql_err;
	EXEC SQL COMMIT ;
	EXEC SQL DISCONNECT ;
	
	TRACE_FUNC_EXIT( 1, "sqlDisconnect", INT, (char *)&i);
	return( 0);

sql_err:
	EXEC SQL WHENEVER SQLERROR CONTINUE;
	sql_error( sqlca.sqlcode);

	TRACE_FUNC_EXIT( 1, "sqlDisconnect", INT, (char *)&i);
	return( 0);
}

int 
sqlCommit()
{
	TRACE_FUNC_ENTRY( 1, "sqlCommit");

	EXEC SQL WHENEVER SQLERROR GOTO sql_err;
	EXEC SQL COMMIT WORK;

	TRACE_FUNC_EXIT( 1, "sqlCommit", NONE, 0);
	return( 0);

sql_err:
	EXEC SQL WHENEVER SQLERROR CONTINUE;
	sql_error( sqlca.sqlcode);

	TRACE_FUNC_EXIT( 1, "sqlCommit", NONE, 0);
	return( 0);
}

int sqlRollback()
{
	TRACE_FUNC_ENTRY( 1, "sqlRollback");

	EXEC SQL WHENEVER SQLERROR GOTO sql_err;
	EXEC SQL ROLLBACK WORK;

	TRACE_FUNC_EXIT( 1, "sqlRollback", NONE, 0);
	return( 0);

sql_err:
	EXEC SQL WHENEVER SQLERROR CONTINUE;
	sql_error( sqlca.sqlcode);

	TRACE_FUNC_EXIT( 1, "sqlRollback", NONE, 0);
	return( 0);
}

/*
 * Function  : sql_convert_types( IISQLDA *)
 * Arguments 
 *	arg1 :	Pointer to a SQLDA
 * Return Value:
 *	None
 * Comments  :
 *	Prepares the SQLDA for fetching data by allocating memory for the
 *	columns and setting the output type
 * Author    : Marc Gathier
 * History 
 *	Created 	: Fri Sep 30 14:48:28 CET 1994
 *	Modifications	:
 *
 *
 */
static void 
sql_convert_types(IISQLDA *select_p)
{
	int i;
	TRACE_FUNC_ENTRY( 2, "sql_convert_types");
  
	for(i = 0; i < select_p->sqld; i++) {

		if( select_p->sqlvar[i].sqltype > 0)
			select_p->sqlvar[i].sqlind = (short *)0;
		else
			select_p->sqlvar[i].sqlind = 
					(short *) malloc(sizeof(short));

		switch (select_p->sqlvar[i].sqltype) {
		case 8: /* LONG */
			select_p->sqlvar[i].sqllen = MAXFIELD;
			break;
		case IISQ_DTE_TYPE: /* DATE */
			select_p->sqlvar[i].sqltype = IISQ_CHA_TYPE;
			select_p->sqlvar[i].sqllen = IISQ_DTE_LEN;
			break;
		case -IISQ_DTE_TYPE: /* DATE */
			select_p->sqlvar[i].sqltype = -IISQ_CHA_TYPE;
			select_p->sqlvar[i].sqllen = IISQ_DTE_LEN;
			break;
		case IISQ_MNY_TYPE: /* MONEY */
			select_p->sqlvar[i].sqltype = IISQ_FLT_TYPE;
			select_p->sqlvar[i].sqllen = 4;
			break;
		case -IISQ_MNY_TYPE: /* MONEY */
			select_p->sqlvar[i].sqltype = -IISQ_FLT_TYPE;
			select_p->sqlvar[i].sqllen = 4;
			break;
		case IISQ_CHA_TYPE:
		case -IISQ_CHA_TYPE:
			/* extra memory for trailing zero */
			select_p->sqlvar[i].sqllen++;
			break;
		case IISQ_VCH_TYPE:
		case -IISQ_VCH_TYPE:
			/* extra memory for the leading length field(short) */
			select_p->sqlvar[i].sqllen += sizeof(short);
			break;
			break;
		case 23: /* RAW */
			break;
		case 24: /* LONG RAW */
			select_p->sqlvar[i].sqllen = MAXFIELD;
			break;
		}

		select_p->sqlvar[i].sqldata = 
				malloc( select_p->sqlvar[i].sqllen);

	}
	TRACE_FUNC_EXIT( 2, "sql_convert_types", NONE, 0);
}

int 
get_free_desc()
{
	static int da_last = 0;
	int i = da_last;

	TRACE_FUNC_ENTRY( 2, "get_free_desc");
  
	while (da_inuse[i]) {
		i = (i + 1) % MAX_CURSORS;

		if (i == da_last) {
			TRACE_FUNC_EXIT( 2, "get_free_desc", INT, (char *)&i);
			return -1; /* no more free descriptors */
		}
	}
	da_inuse[i] = TRUE;
	da_last = i;
	TRACE_FUNC_EXIT( 2, "get_free_desc", INT, (char *)&i);
	return i;
}

void
do_close(int index)
{
	TRACE_FUNC_ENTRY( 2, "do_close");
	TRACE_FUNC_DATA( 2, INT, (char *)&index);

	EXEC SQL WHENEVER SQLERROR GOTO sql_err;

	switch (index) {
	case 0: 
		EXEC SQL CLOSE C0; 
		return;
	case 1: 
		EXEC SQL CLOSE C1; 
		return;
	case 2: 
		EXEC SQL CLOSE C2; 
		return;
	case 3: 
		EXEC SQL CLOSE C3; 
		return;
	case 4: 
		EXEC SQL CLOSE C4; 
		return;
	case 5: 
		EXEC SQL CLOSE C5; 
		return;
	case 6: 
		EXEC SQL CLOSE C6;
		return;
	case 7: 
		EXEC SQL CLOSE C7; 
		return;
	case 8: 
		EXEC SQL CLOSE C8; 
		return;
	case 9: 
		EXEC SQL CLOSE C9; 
		return;
	default:
		fprintf(stderr,"index out of range %d in do_close\n", index);
	}
	TRACE_FUNC_EXIT( 2, "do_close", NONE, 0);
	return;

sql_err:
	EXEC SQL WHENEVER SQLERROR CONTINUE;
	sql_error( sqlca.sqlcode);

	TRACE_FUNC_EXIT( 2, "do_close", NONE, 0);
	return;
}

void
desc_pl_list(IISQLDA *select_p, struct field_desc_t *list)
{
	int		i,
			j,
			length;
	struct field_desc_t *field_list;

	TRACE_FUNC_ENTRY( 2, "desc_pl_list");
  
	field_list = list;

	for(i = 0 ; i < select_p->sqld; i++) {

		length = MIN( select_p->sqlvar[i].sqlname.sqlnamel, NAMELEN-1); 
		switch( ABS(select_p->sqlvar[i].sqltype)) {
		case 30:	/* integer */
			switch(select_p->sqlvar[i].sqllen) {
			case 1:
				field_list->len = 4;
				break;
			case 2:
				field_list->len = 6;
				break;
			case 4:
				field_list->len = 11;
				break;
			default:
				field_list->len = select_p->sqlvar[i].sqllen;
			}
			break;
		default:
			field_list->len = select_p->sqlvar[i].sqllen;
		}

		for(j = 0; j < length ; j++)
			field_list->name[j] = 
				select_p->sqlvar[i].sqlname.sqlnamec[j];
		field_list->name[j] = '\0';

		strcpy( field_list->type, 
			get_type_name(select_p->sqlvar[i].sqltype));

		field_list++;
	}
	TRACE_FUNC_EXIT( 2, "desc_pl_list", NONE, 0);
}

static char *type_names[] = {
	"CHAR", 
	"VARCHAR", 
	"INTEGER", 
	"FLOAT", 
	"TABLE", 
	"DATE", 
	"MONEY", 
	"???"};

static char *
get_type_name(int index)
{
	switch (index) {
	case  20: 
	case  -20: 
		return type_names[0];
	case  21: 
	case  -21: 
		return type_names[1];
	case  30: 
	case  -30: 
		return type_names[2];
	case  31: 
	case  -31: 
		return type_names[3];
	case  52: 
		return type_names[4];
	case   3: 
	case   -3: 
		return type_names[5];
	case   5: 
	case   -5: 
		return type_names[6];
	default:  return type_names[7];
	}
}

/*
 * Function  : sqlFetchOne( int, char *)
 * Arguments 
 *	arg1 :	Socket
 *	arg2 :	Statement
 * Return Value:
 *	None
 * Comments  :
 *	Compiles sql-statement, fetches one record and closes cursor.
 *	Convinience routine.
 * Author    : Marc Gathier
 * History 
 *	Created 	: Fri Sep 30 15:14:01 CET 1994
 *	Modifications	:
 *
 *
 */
int 
sqlFetchOne(int sock, char *s)
{
	struct field_desc_t	list[SELECT_SIZE];
	int			crsr; 
	int			att_count;
  
	TRACE_FUNC_ENTRY( 1, "sqlFetchOne");
	TRACE_FUNC_DATA( 1, INT, (char *)&sock);
	TRACE_FUNC_DATA( 1, STRING, s);

	sqlCommand5( s, &crsr, list, &att_count) ;
	sqlFetchN( sock, crsr, 1);
	sqlCloseCursor( crsr);

	TRACE_FUNC_EXIT( 1, "sqlFetchOne", NONE, 0);
}

int
sqlTableAttr( char *tab, char *owner, int *crsr)
{
	struct field_desc_t	list[SELECT_SIZE];
	int			att_count;
	int			sqlstat;

	TRACE_FUNC_ENTRY( 1, "sqlTableAttr");
	TRACE_FUNC_DATA( 1, STRING, tab);
	TRACE_FUNC_DATA( 1, STRING, owner);

  	sprintf( s, "select \
			column_name, \
			column_datatype, \
			column_length, \
			column_scale, \
			column_nulls, \
			column_sequence \
		from \
			iicolumns \
		where \
			table_name = '%s' \
		and 	table_owner = '%s'", tab, owner);

	sqlCommand5( s, crsr, list, &att_count);
	sqlstat = get_error_code();
  	if( sqlstat < 0 ) *crsr = sqlstat;

	TRACE_FUNC_EXIT( 1, "sqlTableAttr", INT, (char *)&crsr);
}
  
int
sqlTableIndex( char *tab, char *owner, int *crsr)
{
	struct field_desc_t	list[SELECT_SIZE];
	int			att_count;
	int			sqlstat;

	TRACE_FUNC_ENTRY( 1, "sqlTableIndex");
	TRACE_FUNC_DATA( 1, STRING, tab);
	TRACE_FUNC_DATA( 1, STRING, owner);

	sprintf( s, 
		"select i.index_name, \
			unique_rule, \
			column_name, \
			key_sequence \
		 from 	iiindex_columns c, iiindexes i\
		 where 	i.base_name = '%s' and \
			i.base_owner = '%s' and \
			c.index_name = i.index_name and \
			c.index_owner = i.index_owner", tab, owner);

	sqlCommand5( s, crsr, list, &att_count);
	sqlstat = get_error_code();
	if( sqlstat < 0 ) *crsr = sqlstat;

	TRACE_FUNC_EXIT( 1, "sqlTableIndex", INT, (char *)&crsr);
}
  
/*
 * Function  : do_fetch( int)
 * Arguments 
 *	arg1 :	Cursor
 * Return Value:
 *	Pointer to the SQLDA for the cursor
 * Comments  :
 *	Fetches the next record for a statement, returns 0 if no more
 *	records found.
 * Author    : Marc Gathier
 * History 
 *	Created 	: Fri Sep 30 15:13:00 CET 1994
 *	Modifications	:
 *
 *
 */
IISQLDA	*
do_fetch( int index)
{
	IISQLDA *p;
	TRACE_FUNC_ENTRY( 2, "do_fetch");
	TRACE_FUNC_DATA( 2, INT, (char *)&index);

	EXEC SQL WHENEVER SQLERROR GOTO sql_err;

	switch (index) {
	case 0: 
		EXEC SQL FETCH C0 USING DESCRIPTOR select_p0;
		p = select_p0;
		break;
	case 1: 
		EXEC SQL FETCH C1 USING DESCRIPTOR select_p1;
		p = select_p1;
		break;
	case 2: 
		EXEC SQL FETCH C2 USING DESCRIPTOR select_p2;
		p = select_p2;
		break;
	case 3: 
		EXEC SQL FETCH C3 USING DESCRIPTOR select_p3;
		p = select_p3;
		break;
	case 4: 
		EXEC SQL FETCH C4 USING DESCRIPTOR select_p4;
		p = select_p4;
		break;
	case 5: 
		EXEC SQL FETCH C5 USING DESCRIPTOR select_p5;
		p = select_p5;
		break;
	case 6: 
		EXEC SQL FETCH C6 USING DESCRIPTOR select_p6;
		p = select_p6;
		break;
	case 7: 
		EXEC SQL FETCH C7 USING DESCRIPTOR select_p7;
		p = select_p7;
		break;
	case 8: 
		EXEC SQL FETCH C8 USING DESCRIPTOR select_p8;
		p = select_p8;
		break;
	case 9: 
		EXEC SQL FETCH C9 USING DESCRIPTOR select_p9;
		p = select_p9;
		break;
	default:
		fprintf(stderr,"index out of range %d in do_fetch\n", index);
	}


	TRACE_FUNC_EXIT( 2, "do_fetch", NONE, 0);
	return p;

sql_err:
	EXEC SQL WHENEVER SQLERROR CONTINUE;
	sql_error( sqlca.sqlcode);
	TRACE_FUNC_EXIT( 2, "do_fetch", INT, 
			(char *)&(sqlca.sqlcode));
	return (IISQLDA *) 0;
}


/*
 * Function  : get_field( int, int, char *, char *)
 * Arguments 
 *	arg1 :	Type of column
 *	arg2 :	Length of column
 *	arg3 :	Datafield of the SQLDA structure
 *	arg4 :	String containing the data
 * Return Value:
 *	None
 * Comments  :
 *	Converts a field value in the SQLDA structure to a character string
 * Author    : Marc Gathier
 * History 
 *	Created 	: Fri Sep 30 15:11:03 CET 1994
 *	Modifications	:
 *
 *
 */
int
get_field( int type, int len, char *data, char *field)
{
	TRACE_FUNC_ENTRY( 2, "get_field");
	TRACE_FUNC_DATA( 2, INT, (char *)&type);
	TRACE_FUNC_DATA( 2, INT, (char *)&len);
	TRACE_FUNC_DATA( 2, STRING, data);

	switch( type) {

	case IISQ_INT_TYPE:
	case -IISQ_INT_TYPE:
		switch( len) {
		case 1:
			sprintf( field, "%d", *(char *)data);
			break;
		case 2:
			sprintf( field, "%d", *(short *)data);
			break;
		default:
			sprintf( field, "%d", *(long *)data);
			break;
		}
		break;

	case IISQ_FLT_TYPE:
	case -IISQ_FLT_TYPE:
		switch( len) {
		case 4:
			sprintf( field, "%f", *(float *) data);
			break;
		default:
			sprintf( field, "%f", *(double *) data);
			break;
		}
		break;

	case IISQ_VCH_TYPE:
	case -IISQ_VCH_TYPE:
		strncpy( field, data + sizeof(short), *(short *)data);
		field[*(short *)data] = '\0';
		break;

	default:
		{
		int	length, j, k;

		length = MIN( (int)strlen( data), MAXFIELD);
	
		/* skip leading spaces*/
		for (k = 0; data[k] == ' ' && k < length; k ++)
	  		;

		for (j = k; j < length; j++) 
			field[j-k] = data[j];

		j = j-k;
       		while (j > 0 && s[j-1] == ' ')
	  		j--;
	
		field[j] = '\0';
		}
	}

	TRACE_FUNC_EXIT( 2, "get_field", STRING, field);
}
