-----------------------------------------------------------------------
--(c) Copyright IBM Corporation 2009  All rights reserved.           --
--                                                                   --
--This sample program is owned by International Business Machines    --
--Corporation or one of its subsidiaries ("IBM") and is copyrighted  --
--and licensed, not sold.                                            --
--BY ACCESSING, COPYING, OR USING THIS SAMPLE PROGRAM, YOU AGREE TO  --
--THE TERMS OF THE AGREEMENT TITLED "International License Agreement --
--for Non-Warranted db2perf Programs" LOCATED IN THE FILE NAMED      --
--"license.txt".                                                     --
--                                                                   --
-- db2perf_plandiff.db2                                              --
-- Steve Rees - srees@ca.ibm.com                                     --
--                                                                   --
-- Looks for differences between plans in the explain tables.        --
--                                                                   --
-----------------------------------------------------------------------

----------------------------------------------------------------------
----------------------------------------------------------------------
-- db2perf_planstring
-- 
-- Builds a string representation of a plan from the explain tables.
-- 
-- Basically, we start at the 'top' of the plan with the RETURN operator,
-- and work down from there, through the input and output streams to other
-- operators, and so on, until we reach the end of the plan.
----------------------------------------------------------------------
----------------------------------------------------------------------
CALL db2perf_quiet_drop('PROCEDURE db2perf_planstring')@

CREATE PROCEDURE db2perf_planstring( 
	IN explain_schema VARCHAR(128),
	IN requester VARCHAR(128),
	IN ts TIMESTAMP,
	IN schema VARCHAR(128),
	IN src_name VARCHAR(128),
	IN section INTEGER,
	INOUT plan_string VARCHAR(30000) )

LANGUAGE SQL
BEGIN
   DECLARE SQLSTATE CHAR(5);
   DECLARE statement VARCHAR(1024);

   DECLARE operator_id   INTEGER;
   DECLARE operator_type CHAR(6);

   DECLARE stream_id   INTEGER;
   DECLARE source_type CHAR(1);
   DECLARE source_id   INTEGER;
   DECLARE target_type CHAR(1);
   DECLARE target_id   INTEGER;
   DECLARE object_name VARCHAR(128);

   DECLARE at_end INT DEFAULT 0;

   DECLARE operator_cur CURSOR FOR operator_stmt;
   DECLARE input_stream_cur CURSOR FOR input_stream_stmt;
   DECLARE output_stream_cur CURSOR FOR output_stream_stmt;

   DECLARE CONTINUE HANDLER FOR NOT FOUND
     SET at_end = 1; 


   SET at_end = 0;
   SET plan_string = '';
   SET operator_id = 1;


   SET statement = 'SELECT operator_id, operator_type FROM '
	|| CASE WHEN explain_schema <> '' THEN explain_schema || '.' ELSE '' END
	|| 'explain_operator WHERE '
	|| 'explain_requester = ? ' 
	|| 'AND explain_time = ? ' 
	|| 'AND source_schema = ? ' 
	|| 'AND source_name = ? ' 
	|| 'AND sectno = ? ' 
	|| 'ORDER BY operator_id '
	|| 'FOR FETCH ONLY '
	|| 'WITH UR'
	;
   PREPARE operator_stmt from statement;


   SET statement = 'SELECT stream_id, source_type, source_id, target_type, target_id, object_name FROM '
	|| CASE WHEN explain_schema <> '' THEN explain_schema || '.' ELSE '' END
	|| 'explain_stream WHERE '
	|| 'explain_requester = ? '
	|| 'AND explain_time = ? '
	|| 'AND source_schema = ? '
	|| 'AND source_name = ? '
	|| 'AND sectno = ? '
	|| 'AND target_id = ? '
	|| 'ORDER BY stream_id '
	|| 'FOR FETCH ONLY '
	|| 'WITH UR'
	;
   PREPARE input_stream_stmt from statement;


   SET statement = 'SELECT stream_id, source_type, source_id, target_type, target_id, object_name FROM '
	|| CASE WHEN explain_schema <> '' THEN explain_schema || '.' ELSE '' END
	|| 'explain_stream WHERE '
	|| 'explain_requester = ? ' 
	|| 'AND explain_time = ? ' 
	|| 'AND source_schema = ? ' 
	|| 'AND source_name = ? ' 
	|| 'AND sectno = ? ' 
	|| 'AND source_id = ? '
	|| 'AND target_id = -1 '
	|| 'AND target_type = ''D'' '
	|| 'ORDER BY stream_id '
	|| 'FOR FETCH ONLY '
	|| 'WITH UR'
	;
   PREPARE output_stream_stmt from statement;


   ----------------------------------------------------------------------
   -- Open a cursor on EXPLAIN_OPERATOR, to pick up the operators in this
   -- plan, in order from RETURN and on down.
   ----------------------------------------------------------------------
   OPEN operator_cur USING requester, ts, schema, src_name, section;
   FETCH operator_cur INTO operator_id, operator_type;

   WHILE at_end = 0 DO
     
     ----------------------------------------------------------------------
     -- Append the current operator to the plan string - the operator name and then the number.

     SET plan_string = 
     	   plan_string 
	|| CASE operator_id WHEN 1 THEN '' ELSE ' ' END 
	|| rtrim(operator_type) 
	|| '(' 
	|| rtrim(char(operator_id)) 
	|| ')';
     
     ----------------------------------------------------------------------
     -- For this operator, get the input streams which supply data to this operator

     OPEN input_stream_cur USING 
	requester, ts, schema, src_name, section, operator_id;
     FETCH input_stream_cur INTO
        stream_id, source_type, source_id, target_type, target_id, object_name;

     WHILE at_end = 0 DO
        ----------------------------------------------------------------------
	-- Make sure we don't go too long

	IF length(plan_string) > 29500 THEN
	   SET plan_string = 'Too Long!';
	   SET at_end = 1;
        ELSE
           ----------------------------------------------------------------------
	   -- The source_type column tells us whether it's another operator supplying the data,
	   -- or an object

	   IF source_type = 'O' THEN
	     SET plan_string = plan_string || '<-Op(' || rtrim(char(source_id)) || ')';
	   ELSE
	     SET plan_string = plan_string || '<-Object(' || object_name || ')';
	   END IF;

	   FETCH input_stream_cur INTO
	      stream_id, source_type, source_id, target_type, target_id, object_name;
	END IF;

     END WHILE;
     CLOSE input_stream_cur;
     

     ----------------------------------------------------------------------
     -- Now that we've got the input streams for this operator, get the output streams, 
     -- if we're doing an insert, etc.

     SET at_end = 0;
     OPEN output_stream_cur USING 
	requester, ts, schema, src_name, section, operator_id;
     FETCH output_stream_cur INTO
        stream_id, source_type, source_id, target_type, target_id, object_name;

     WHILE at_end = 0 DO
	IF length(plan_string) > 29500 THEN
	   SET plan_string = 'Too Long!';
	   SET at_end = 1;
        ELSE
	   SET plan_string = plan_string || '<-Object(' || object_name || ')';

	   FETCH output_stream_cur INTO
	      stream_id, source_type, source_id, target_type, target_id, object_name;
	END IF;

     END WHILE;
     CLOSE output_stream_cur;

     
     IF plan_string <> 'Too Long!' THEN
        SET at_end = 0;
        FETCH operator_cur INTO operator_id, operator_type;
     END IF;

   END WHILE;
   CLOSE operator_cur;

END@


----------------------------------------------------------------------
----------------------------------------------------------------------
-- db2perf_compareplans
-- 
-- Gets the plan strings for the two plans identified by timestamps, sections, etc.
----------------------------------------------------------------------
----------------------------------------------------------------------


CALL db2perf_quiet_drop( 'PROCEDURE db2perf_compareplans( '
	|| 'VARCHAR(128), '
	|| 'VARCHAR(128), '
	|| 'TIMESTAMP, '
	|| 'VARCHAR(128), '
	|| 'VARCHAR(128), '
	|| 'INTEGER, '
	|| 'VARCHAR(128), '
	|| 'VARCHAR(128), '
	|| 'TIMESTAMP, '
	|| 'VARCHAR(128), '
	|| 'VARCHAR(128), '
	|| 'INTEGER, char(20))' )@

CREATE PROCEDURE db2perf_compareplans( 
	IN explain_schema1 VARCHAR(128),
	IN requester1 VARCHAR(128),
	IN ts1 TIMESTAMP,
	IN schema1 VARCHAR(128),
	IN src_name1 VARCHAR(128),
	IN section1 INTEGER,
	IN explain_schema2 VARCHAR(128),
	IN requester2 VARCHAR(128),
	IN ts2 TIMESTAMP,
	IN schema2 VARCHAR(128),
	IN src_name2 VARCHAR(128),
	IN section2 INTEGER,
	INOUT plans_differ char(20) )

LANGUAGE SQL
BEGIN
   DECLARE SQLSTATE CHAR(5);

   DECLARE plan_string1 VARCHAR(30000);
   DECLARE plan_string2 VARCHAR(30000);

   DECLARE at_end INT DEFAULT 0;

   call db2perf_planstring( explain_schema1,requester1,ts1,schema1,src_name1,section1, plan_string1 );
   call db2perf_planstring( explain_schema2,requester2,ts2,schema2,src_name2,section2, plan_string2 );

   CASE
        WHEN (plan_string1 = '') THEN
          SET plans_differ = 'plan 1 not found';
        WHEN (plan_string2 = '') THEN
          SET plans_differ = 'plan 2 not found';
        WHEN (plan_string1 = 'Too Long!') THEN
          SET plans_differ = 'plan too complex';
        WHEN (plan_string2 = 'Too Long!') THEN
          SET plans_differ = 'plan too complex';
        WHEN (plan_string1 = plan_string2) THEN
          SET plans_differ = 'No';
        ELSE
          SET plans_differ = 'Yes';
   END CASE;
END@


----------------------------------------------------------------------
----------------------------------------------------------------------
-- db2perf_plandiff_report
--
-- Reports the results of comparing two plans into our report table, db2perf_plandiff_report
----------------------------------------------------------------------
----------------------------------------------------------------------

CALL db2perf_quiet_drop('TABLE db2perf_plandiff_report')@

CREATE TABLE db2perf_plandiff_report( line INTEGER, message VARCHAR(128) )@


CALL db2perf_quiet_drop('PROCEDURE db2perf_plandiff_report')@

CREATE PROCEDURE db2perf_plandiff_report( 
	IN schema1 VARCHAR(128),
	IN src_name1 VARCHAR(128),
	IN section1 INTEGER,
	IN ts1 TIMESTAMP,
	IN total_cost1 DOUBLE,
	IN schema2 VARCHAR(128),
	IN src_name2 VARCHAR(128),
	IN section2 INTEGER,
	IN ts2 TIMESTAMP,
	IN total_cost2 DOUBLE,
	IN plans_differ char(20),
	IN text CLOB(2M),
	INOUT line INTEGER  )
LANGUAGE SQL
BEGIN
   DECLARE blanks         CHAR(50) DEFAULT ' ';
   DECLARE src_schema1    VARCHAR(300);
   DECLARE src_schema2    VARCHAR(300);
   DECLARE length_printed INTEGER DEFAULT 0;
   DECLARE stmt_buffer    VARCHAR(100);
   DECLARE start          INTEGER;
   DECLARE end            INTEGER;
   DECLARE done           INTEGER DEFAULT 0;


   ----------------------------------------------------------------------
   -- Put together the schema name & source name for the first plan.
   SET src_schema1 = rtrim(src_name1);
   IF length(src_schema1) > 32 THEN 
     SET src_schema1 = substr(src_schema1,1,32);
   ELSE
     SET src_schema1 = rtrim(schema1) || '.' || src_schema1;
     IF length(src_schema1) > 32 THEN 
       SET src_schema1 = substr(src_schema1,length(src_schema1)-32,32);
     ELSE
       SET src_schema1 = src_schema1 || CASE WHEN length(src_schema1) < 20 THEN substr(blanks,1,20-length(src_schema1)) ELSE '' END;
     END IF;
   END IF;


   ----------------------------------------------------------------------
   -- Put together the schema name & source name for the second plan.
   SET src_schema2 = rtrim(src_name2);
   IF length(src_schema2) > 32 THEN 
     SET src_schema2 = substr(src_schema2,1,32);
   ELSE
     SET src_schema2 = rtrim(schema2) || '.' || src_schema2;
     IF length(src_schema2) > 32 THEN 
       SET src_schema2 = substr(src_schema2,length(src_schema2)-32,32);
     ELSE
       SET src_schema2 = src_schema2 || CASE WHEN length(src_schema2) < 20 THEN substr(blanks,2,20-length(src_schema2)) ELSE '' END;
     END IF;
   END IF;


   ----------------------------------------------------------------------
   -- Print out the header.

   IF line = 3 THEN
     INSERT INTO db2perf_plandiff_report VALUES ( 
     	line,'' );
     SET line = line+1;
     INSERT INTO db2perf_plandiff_report VALUES ( 
     	line,'Plan    Package Name        Section Timestamp                    Cost        Statement Text' );
     SET line = line+1;
     INSERT INTO db2perf_plandiff_report VALUES ( 
     	line,'Change?                                                          (timerons)');
     SET line = line+1;
     INSERT INTO db2perf_plandiff_report VALUES ( 
     	line,'----------------------------------------------------------------------------------------------------------');
     SET line = line+1;
   END IF;


   ----------------------------------------------------------------------
   -- Print out the information about the plans (timestamps, sections, etc.
   -- If you wanted to just print out the changed plans, this would be the place.

   INSERT INTO db2perf_plandiff_report VALUES
	   ( line, 
	     cast(plans_differ AS CHAR(5))  || ' | ' 
	     || src_schema1    || ' '
	     || char(cast(section1 AS SMALLINT)) || ' ' 
	     || char(ts1) || '   '
	     || CASE WHEN total_cost1 < 1999999999 THEN char(cast(total_cost1 as INTEGER)) ELSE char(total_cost1) END
	   );
   SET line = line+1;

   INSERT INTO db2perf_plandiff_report VALUES
	   ( line, 
	     '      | ' 
	     || src_schema2    || ' '
	     || char(cast(section2 AS SMALLINT)) || ' '
	     || char(ts2) || '   '
	     || CASE WHEN total_cost2 < 1999999999 THEN char(cast(total_cost2 as INTEGER)) ELSE char(total_cost2) END
	   );
   SET line = line+1;


   ----------------------------------------------------------------------
   -- Loop through and print out the statement text a chunk at a time.

   SET start = 1;
   SET stmt_buffer = substr(text,start,50);
   
   WHILE done = 0 DO
     INSERT INTO db2perf_plandiff_report VALUES
	   ( line, 
	        '      |                                                                      ' 
	     || rtrim(ltrim(stmt_buffer)) );
     SET line = line+1;

     IF start+50 >= length(text) THEN
       SET done = 1;
     ELSE
       SET start = start + 50;
       SET stmt_buffer = substr(text,start,50);
     END IF;

   END WHILE;
       
   
END@



----------------------------------------------------------------------
----------------------------------------------------------------------
-- db2perf_plandiff
--
-- The main routine.  This is called with 'LIKE' patterns similar to db2exfmt, to match for requester,
-- schema, src_name (package) and section.
-- We use those patterns to filter candidate statements to compare, and if the statement texts match, then
-- we call the routines above to compare the plans.
----------------------------------------------------------------------
----------------------------------------------------------------------

CALL db2perf_quiet_drop( 
	'PROCEDURE db2perf_plandiff( '
	|| 'VARCHAR(128),'
	|| 'VARCHAR(128),'
	|| 'VARCHAR(128),'
	|| 'VARCHAR(128),'
	|| 'VARCHAR(128),'
	|| 'INTEGER ,'
	|| 'CHAR(5) )' )@

CREATE PROCEDURE db2perf_plandiff( 
	IN explain_schema1   VARCHAR(128),
	IN explain_schema2   VARCHAR(128),
	IN requester_pattern VARCHAR(128),
	IN schema_pattern    VARCHAR(128),
	IN src_name_pattern  VARCHAR(128),
	IN section_pattern   INTEGER,
	IN all_or_diffs      CHAR(5) )
DYNAMIC RESULT SETS 1
LANGUAGE SQL
BEGIN
   DECLARE db2perf_plandiff_version CHAR(10) DEFAULT '1.1.0';

   DECLARE SQLSTATE     CHAR(5);

   DECLARE status       VARCHAR(160);
   DECLARE requester1   VARCHAR(128);
   DECLARE ts1          TIMESTAMP;
   DECLARE src_schema1  VARCHAR(128);
   DECLARE src_name1    VARCHAR(128);
   DECLARE src_version1 VARCHAR(64);
   DECLARE stmtno1      INTEGER;
   DECLARE section1     INTEGER;
   DECLARE total_cost1  DOUBLE;
   DECLARE text1        VARCHAR(30000);

   DECLARE short_text   VARCHAR(3000);

   DECLARE requester2   VARCHAR(128);
   DECLARE ts2          TIMESTAMP;
   DECLARE src_schema2  VARCHAR(128);
   DECLARE src_name2    VARCHAR(128);
   DECLARE src_version2 VARCHAR(64);
   DECLARE stmtno2      INTEGER;
   DECLARE section2     INTEGER;
   DECLARE total_cost2  DOUBLE;
   DECLARE text2        VARCHAR(30000);

   DECLARE plans_compared    INTEGER DEFAULT 0;
   DECLARE plans_different   INTEGER DEFAULT 0;
   DECLARE plans_unchanged   INTEGER DEFAULT 0;
   DECLARE plans_too_complex INTEGER DEFAULT 0;

   DECLARE plans_differ CHAR(20);

   DECLARE line INT DEFAULT 0;
  
   DECLARE at_end INT DEFAULT 0;

   DECLARE statement VARCHAR(1024);

   DECLARE plan1_cur CURSOR FOR plan1_stmt;
   DECLARE plan2_cur CURSOR FOR plan2_stmt;
   DECLARE too_long_cur CURSOR FOR too_long_stmt;
   DECLARE total_cost1_cur CURSOR FOR total_cost1_stmt;
   DECLARE total_cost2_cur CURSOR FOR total_cost2_stmt;

   DECLARE report_cur CURSOR WITH RETURN TO CALLER FOR
     SELECT message
     FROM db2perf_plandiff_report
     ORDER BY line;

   DECLARE CONTINUE HANDLER FOR NOT FOUND
     SET at_end = 1; 


   ----------------------------------------------------------------------
   -- Clean out our report table.
   DELETE FROM db2perf_plandiff_report;
   INSERT INTO db2perf_plandiff_report VALUES ( line, 'db2perf_plandiff ' || db2perf_plandiff_version );
   SET line=line+1;


   IF all_or_diffs = '     ' THEN
     SET all_or_diffs = 'DIFFS';
   END IF;

   IF explain_schema2 = '' THEN
     SET explain_schema2 = explain_schema1;
   END IF;

   INSERT INTO db2perf_plandiff_report VALUES ( line, 'Explain table schema 1: ' || explain_schema1 );
   SET line=line+1;
   INSERT INTO db2perf_plandiff_report VALUES ( line, 'Explain table schema 2: ' || explain_schema2 );
   SET line=line+1;

   ----------------------------------------------------------------------
   -- Declare a cursor to select statements from EXPLAIN_STATEMENT which match the 
   -- parameters that were passed in.   We provide those when the cursor is opened, via parameter
   -- markers.

   SET statement = 'SELECT explain_requester,explain_time,source_schema,source_name,source_version,stmtno,sectno,statement_text '
   		|| 'FROM '
		|| CASE WHEN explain_schema1 <> '' THEN explain_schema1 || '.' ELSE '' END
   		|| 'explain_statement WHERE '
		|| '  length(statement_text) < 30000'  -- make sure we deal with statement sizes we can handle
		|| '  AND explain_level = ''O'' '      -- explain_level 'O' means the original statement (not optimized)
		|| '  AND explain_requester LIKE ? '
   		|| '  AND source_schema LIKE ? '
   		|| '  AND source_name LIKE ? '
   		|| CASE WHEN section_pattern <> 0 THEN '  AND sectno = ? ' ELSE '' END 
		;

   PREPARE plan1_stmt from statement;


   ----------------------------------------------------------------------
   -- Declare a similar cursor which picks up the statements which match the ones we got above - 
   -- using the same parameters, plus matching text.

   SET statement = 'SELECT explain_requester,explain_time,source_schema,source_name,source_version,stmtno,sectno,statement_text '
   		|| 'FROM '
		|| CASE WHEN explain_schema2 <> '' THEN explain_schema2 || '.' ELSE '' END
   		|| 'explain_statement WHERE '
		|| '  length(statement_text) < 30000'
		|| '  AND explain_level = ''O'' '
		|| '  AND explain_requester LIKE ? '
   		|| '  AND source_schema LIKE ? '
   		|| '  AND source_name LIKE ? '
   		|| CASE WHEN section_pattern <> 0 THEN '  AND sectno = ? ' ELSE '' END
   		|| '  AND cast(substr(statement_text,1,3000) as varchar(3000)) = ? ' -- match the first statement
		;

   IF explain_schema1 = explain_schema2 THEN
     -- the following predicates are intended to keep us 'going in one direction' and avoid infinite loops when we're fishing 
     -- around in a single set of explain tables.
     SET statement = statement 
   		|| ' AND NOT (explain_requester <= ? AND explain_time <= ? AND source_schema <= ? AND source_name <= ? AND sectno <= ? ) '
		;
   END IF;

   PREPARE plan2_stmt from statement;



   ----------------------------------------------------------------------
   -- we want to report statements that are too long or too complex to tackle.
   set statement = 'select count(*) from '
	|| case when explain_schema1 <> '' then explain_schema1 || '.' else '' end
   	|| 'explain_statement where explain_level = ''o'' and length(statement_text) >= 30000'
	;

   PREPARE too_long_stmt FROM statement;
   OPEN too_long_cur;
   FETCH too_long_cur INTO plans_too_complex;
   CLOSE too_long_cur;

   IF explain_schema1 <> explain_schema2 AND plans_too_complex = 0 THEN 
     SET statement = 'select count(*) from '
	  || CASE WHEN explain_schema2 <> '' THEN explain_schema2 || '.' else '' END
	  || 'explain_statement WHERE explain_level = ''O'' and length(statement_text) >= 30000'
	  ;

     PREPARE too_long_stmt FROM statement;
     OPEN too_long_cur;
     FETCH too_long_cur INTO plans_too_complex;
     CLOSE too_long_cur;
   END IF;



   ----------------------------------------------------------------------
   -- Open up the plan cursor with the patterns that were passed in.  We use '0' as a wildcard for sections.
   -- If it was passed in, then the section predicate is omitted completely.

   IF section_pattern <> 0 THEN
     OPEN plan1_cur USING requester_pattern, schema_pattern, src_name_pattern, section_pattern;
   ELSE
     OPEN plan1_cur USING requester_pattern, schema_pattern, src_name_pattern;
   END IF;



   ----------------------------------------------------------------------
   -- Declare cursors for the total_cost values - one each for the 'before' and 'after' tables

   SET statement = 'SELECT total_cost FROM '
	|| CASE WHEN explain_schema1 <> '' THEN explain_schema1 || '.' ELSE '' END
   	|| 'explain_statement '
	|| 'WHERE explain_requester = ? '
	|| 'AND explain_time = ? '
   	|| 'AND source_name = ? '
   	|| 'AND source_schema = ? '
   	|| 'AND source_version = ? '
	|| 'AND explain_level = ''P'' '
   	|| 'AND stmtno = ? '
   	|| 'AND sectno = ? '
	;
   PREPARE total_cost1_stmt FROM statement;
   


   SET statement = 'SELECT total_cost FROM '
	|| CASE WHEN explain_schema2 <> '' THEN explain_schema2 || '.' ELSE '' END
   	|| 'explain_statement '
	|| 'WHERE explain_requester = ? '
	|| 'AND explain_time = ? '
   	|| 'AND source_name = ? '
   	|| 'AND source_schema = ? '
   	|| 'AND source_version = ? '
	|| 'AND explain_level = ''P'' '
   	|| 'AND stmtno = ? '
   	|| 'AND sectno = ? '
	;
   PREPARE total_cost2_stmt FROM statement;

   SET at_end = 0;
   FETCH plan1_cur INTO requester1,ts1,src_schema1,src_name1,src_version1,stmtno1,section1,text1;

   WHILE at_end = 0 DO

     IF at_end = 0 THEN

       ----------------------------------------------------------------------
       -- We can handle up to 30k statements, but we grab the first 3k and we use that as a predicate
       -- value to find statements which PROBABLY match.  We then double check afterward.


       SET short_text = cast(substr(text1,1,3000) as varchar(3000));

    
       ----------------------------------------------------------------------
       -- We report the total cost, but we have to get it from the matching 'P'
       -- (optimized, rewritten) row from EXPLAIN_STATEMENT.  The 'O' row
       -- doesn't have the cost filled in.

       SET total_cost1 = 0;
       OPEN total_cost1_cur USING requester1, ts1, src_name1, src_schema1, src_version1, stmtno1, section1 ;
       FETCH total_cost1_cur INTO total_cost1;
       CLOSE total_cost1_cur;
       SET at_end = 0;


       ----------------------------------------------------------------------
       -- Open up the 2nd cursor to find statements we should compare with the 1st one.

       IF explain_schema1 = explain_schema2 THEN
	 IF section_pattern <> 0 THEN
	   OPEN plan2_cur USING requester_pattern, schema_pattern, src_name_pattern, section_pattern, short_text, 
			    requester1,ts1,src_schema1,src_name1,section1
	      ;
	 ELSE
	   OPEN plan2_cur USING requester_pattern, schema_pattern, src_name_pattern, short_text, 
			    requester1,ts1,src_schema1,src_name1,section1
	      ;
	 END IF;
       ELSE
	 IF section_pattern <> 0 THEN
	   OPEN plan2_cur USING requester_pattern, schema_pattern, src_name_pattern, section_pattern, short_text
	      ;
	 ELSE
	   OPEN plan2_cur USING requester_pattern, schema_pattern, src_name_pattern, short_text
	      ;
	 END IF;
       END IF;

       FETCH plan2_cur INTO requester2,ts2,src_schema2,src_name2,src_version2,stmtno2,section2,text2;


       WHILE at_end = 0 DO

         ----------------------------------------------------------------------
	 -- We used the first 3k of text in the cursor predicate, but we also check here to make
	 -- sure the statements really do match.

	 IF text1 = text2 THEN


           ----------------------------------------------------------------------
	   -- The statements match.  Get the cost of the 2nd one, and then compare their plans.

           SET total_cost2 = 0;
	   OPEN total_cost2_cur USING requester2, ts2, src_name2, src_schema2, src_version2, stmtno2, section2 ;
	   FETCH total_cost2_cur INTO total_cost2;
	   CLOSE total_cost2_cur;
	   SET at_end = 0;


           ----------------------------------------------------------------------
	   -- Compare the plans

	   CALL db2perf_compareplans( explain_schema1,requester1,ts1,src_schema1,src_name1,section1,
				      explain_schema2,requester2,ts2,src_schema2,src_name2,section2,
				      plans_differ );

           ----------------------------------------------------------------------
	   -- Report the results.
	   IF plans_differ <> 'plan too complex' THEN
	      IF plans_differ = 'Yes' OR translate(all_or_diffs) = 'ALL' THEN
		CALL db2perf_plandiff_report( src_schema1, src_name1, section1, ts1, total_cost1,
					      src_schema2, src_name2, section2, ts2, total_cost2,
					      plans_differ, text1,
					      line );
	      END IF;

	      SET plans_compared = plans_compared + 1;
	      IF plans_differ = 'Yes' THEN
		 SET plans_different = plans_different + 1;
	      ELSE
		 SET plans_unchanged = plans_unchanged + 1;
	      END IF;
	   ELSE
	      SET plans_too_complex = plans_too_complex + 1;
	   END IF;

	 END IF;

         FETCH plan2_cur INTO requester2,ts2,src_schema2,src_name2,src_version2,stmtno2,section2,text2;
       END WHILE;

       CLOSE plan2_cur;

       SET at_end = 0;
     END IF;

     FETCH plan1_cur INTO requester1,ts1,src_schema1,src_name1,src_version1,stmtno1,section1,text1;
   END WHILE;

   CLOSE plan1_cur;

   SET status = 
	'Compared ' || rtrim(char(plans_compared)) || ' plan pairs (' 
	|| rtrim(char(plans_different)) || ' look different, ' 
	|| rtrim(char(plans_unchanged)) || ' look unchanged). '
	|| 'Unable to compare ' || rtrim(char(plans_too_complex)) || ' plan(s) due to length or complexity.';

   INSERT INTO db2perf_plandiff_report VALUES ( line, ' ' );
   INSERT INTO db2perf_plandiff_report VALUES ( line+1, 'Report run at ' || char( CURRENT TIMESTAMP ) );
   INSERT INTO db2perf_plandiff_report VALUES ( line+2, status );

   OPEN report_cur;

END@



----------------------------------------------------------------------
----------------------------------------------------------------------
-- db2perf_plandiff
--
-- Version with the 4 v1.0 parameters.  Defaults explain-schema1 & explain-schema2.
----------------------------------------------------------------------
----------------------------------------------------------------------


CALL db2perf_quiet_drop( 
	'PROCEDURE db2perf_plandiff( '
	|| 'VARCHAR(128),'
	|| 'VARCHAR(128),'
	|| 'VARCHAR(128),'
	|| 'INTEGER )' )@

CREATE PROCEDURE db2perf_plandiff( 
	IN requester_pattern VARCHAR(128),
	IN schema_pattern    VARCHAR(128),
	IN src_name_pattern  VARCHAR(128),
	IN section_pattern   INTEGER )
DYNAMIC RESULT SETS 1
LANGUAGE SQL
BEGIN
   DECLARE report_cur CURSOR WITH RETURN TO CALLER FOR
     SELECT message
     FROM db2perf_plandiff_report
     ORDER BY line;

   CALL db2perf_plandiff( '','',requester_pattern, schema_pattern, src_name_pattern, section_pattern, 'ALL' );

   OPEN report_cur;
END@


----------------------------------------------------------------------
----------------------------------------------------------------------
-- db2perf_plandiff
--
-- Defaults all-or-diffs to diffs, if omitted
----------------------------------------------------------------------
----------------------------------------------------------------------

CALL db2perf_quiet_drop( 
	'PROCEDURE db2perf_plandiff( '
	|| 'VARCHAR(128),'
	|| 'VARCHAR(128),'
	|| 'VARCHAR(128),'
	|| 'VARCHAR(128),'
	|| 'VARCHAR(128),'
	|| 'INTEGER )' )@

CREATE PROCEDURE db2perf_plandiff( 
	IN explain_schema1   VARCHAR(128),
	IN explain_schema2   VARCHAR(128),
	IN requester_pattern VARCHAR(128),
	IN schema_pattern    VARCHAR(128),
	IN src_name_pattern  VARCHAR(128),
	IN section_pattern   INTEGER )
DYNAMIC RESULT SETS 1
LANGUAGE SQL
BEGIN
   DECLARE report_cur CURSOR WITH RETURN TO CALLER FOR
     SELECT message
     FROM db2perf_plandiff_report
     ORDER BY line;

   CALL db2perf_plandiff( explain_schema1,explain_schema2,requester_pattern, schema_pattern, src_name_pattern, section_pattern, 'DIFFS' );

   OPEN report_cur;
END@


----------------------------------------------------------------------
----------------------------------------------------------------------
-- db2perf_plandiff
--
-- Version with no parameters, to print out usage.
----------------------------------------------------------------------
----------------------------------------------------------------------

CALL db2perf_quiet_drop( 'PROCEDURE db2perf_plandiff()' )@

CREATE PROCEDURE db2perf_plandiff(  )
DYNAMIC RESULT SETS 1
LANGUAGE SQL
BEGIN

   DECLARE line INT DEFAULT 0;

   DECLARE report_cur CURSOR WITH RETURN TO CALLER FOR
     SELECT message
     FROM db2perf_plandiff_report
     ORDER BY line;

   DELETE from db2perf_plandiff_report;

   INSERT INTO db2perf_plandiff_report VALUES ( 
	line, 'Usage: ' );
   SET line=line+1;
   INSERT INTO db2perf_plandiff_report VALUES ( 
	line, '  db2perf_plandiff( explain-schema-1, explain-schema-2, requester-pattern, schema-pattern, src-name-pattern,' );
   SET line=line+1;
   INSERT INTO db2perf_plandiff_report VALUES ( 
	line, '                    section-pattern, all-or-diffs )' );
   SET line=line+1;
   INSERT INTO db2perf_plandiff_report VALUES ( 
	line, '		explain-schmea-1:  schema name of EXPLAIN tables (defaults to current schema)' );
   SET line=line+1;
   INSERT INTO db2perf_plandiff_report VALUES ( 
	line, '		explain-schmea-2:  schema name of 2nd set of EXPLAIN tables (defaults to explain-schema-1)' );
   SET line=line+1;
   INSERT INTO db2perf_plandiff_report VALUES ( 
	line, '		requester-pattern: LIKE-pattern to match explain_requester' );
   SET line=line+1;
   INSERT INTO db2perf_plandiff_report VALUES ( 
	line, '		schema-pattern:    LIKE-pattern to match explain_schema' );
   SET line=line+1;
   INSERT INTO db2perf_plandiff_report VALUES ( 
	line, '		src-name-pattern:  LIKE-pattern to match explain_src_name' );
   SET line=line+1;
   INSERT INTO db2perf_plandiff_report VALUES ( 
	line, '		section-pattern:   section # (0 for all)' );
   SET line=line+1;
   INSERT INTO db2perf_plandiff_report VALUES ( 
	line, '		all-or-diffs:      ALL = show comparisons whether or not they differ' );
   SET line=line+1;
   INSERT INTO db2perf_plandiff_report VALUES ( 
	line, '		                   DIFFS = show only statements with plan differences (default)' );
   SET line=line+1;

   OPEN report_cur;

END@
