/*
 * (c) Copyright 1996, KL GROUP INC.
 * ALL RIGHTS RESERVED
 *
 * This file is provided for demonstration and educational uses only.
 * Permission to use, copy, modify and distribute this file for
 * any purpose and without fee is hereby granted, provided that the
 * above copyright notice and this permission notice appear in all
 * copies, and that the name of KL Group not be used in advertising
 * or publicity pertaining to this material without the specific,
 * prior written permission of an authorized representative of
 * KL Group.
 *
 * KL GROUP MAKES NO REPRESENTATIONS AND EXTENDS NO WARRANTIES, EX-
 * PRESS OR IMPLIED, WITH RESPECT TO THE SOFTWARE, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR ANY PARTICULAR PURPOSE, AND THE WARRANTY AGAINST IN-
 * FRINGEMENT OF PATENTS OR OTHER INTELLECTUAL PROPERTY RIGHTS.  THE
 * SOFTWARE IS PROVIDED "AS IS", AND IN NO EVENT SHALL KL GROUP OR
 * ANY OF ITS AFFILIATES BE LIABLE FOR ANY DAMAGES, INCLUDING ANY
 * LOST PROFITS OR OTHER INCIDENTAL OR CONSEQUENTIAL DAMAGES
 * RELATING TO THE USE OF THIS SOFTWARE.
 */
//   RCSID -- $RCSfile: ContribCheckboxGroup.java $ $Revision: 2.0 $
//            $Date: 1997/06/09 18:21:39 $  $Locker: $  KL Group Inc.

package jclass.contrib;

import java.awt.CheckboxGroup;
import java.awt.Checkbox;
import java.util.Vector;

/**
 * The Java AWT CheckboxGroup class does not allow access to the
 * checkboxes it manages. This makes the task of manipulating
 * checkbox groups awkward. ContribCheckboxGroup stores the checkboxes
 * in a Vector.
 *
 * While ContribCheckboxGrop manages AWT Checkbox instances,
 * it works best with ContribCheckbox instances. The ContribCheckbox constructor will
 * automatically call the addCheckbox() member.
 *
 * Here is a typical use of the ContribCheckboxGroup class:
 <PRE>
   public myForm extends Panel {

   int colorValues      = { BLUE, GREEN, VIOLET };
   String colorStrings  = {"Blue", "Green", "Violet"};
   ContribCheckbox colorG;

   public myForm() {
     setLayout(new JCRowColLayout(-1, 1));
	 colorG = ContribCheckbox.makeGroup(this, colorStrings, colorValues);
   }

   private int getColor();
   private void setColor(int);

   public void getData() {
     colorG.set(colorValues, getColor());
   }

   public void setData() {
     setColor(colorG.getCurrentJC().value);
   }
   }
 </PRE>
 * If items change, simply change the colorValues and colorStrings arrays.
 * This makes mapping from the checkbox to the attribute to
 * set a relatively easy task.
 */

public class ContribCheckboxGroup extends CheckboxGroup {

	/** The Vector containing individual checkbox items. */
    private Vector objs = new Vector();

	/**
	 * Adds a specified checkbox to the list of managed
	 * checkbox. When addCheckbox() is called, it does not impose any
	 * layout restrictions; it adds the item to an array
	 * for retrieval by getCheckbox(). Checking for duplicate items is
     * not done.
	 *
	 *
	 * @param item this is the Checkbox instance to add
	 */
public void addCheckbox(Checkbox item) {
    if (!objs.contains(item)) {
        objs.addElement(item);
    }
}

	/**
	 * Returns the number of Checkbox instances that have been
	 * stored.
	 *
	 * @return        the number of items in the array of checkboxes
	 */

public int numCheckboxes() {
    return objs.size();
}


	/**
	 * Retrieves the checkbox at the specified index. An object
	 * is returned only if it is derived from the AWT checkbox class.
	 *
	 * @param i this is the index of the item
	 * @return        the Checkbox instance, or null if not found
	 */

public Checkbox getCheckbox(int i) {
    Checkbox rval = null;
    if (i >= 0 && i < objs.size() && objs.elementAt(i) instanceof Checkbox) {
        rval = (Checkbox)objs.elementAt(i);
    }
    return rval;
}

	/**
	 * Retrieves the currently-selected ContribCheckbox.  This function
	 * only works if the current selection is a ContribCheckbox.
	 *
	 * @return        the currently-selected ContribCheckbox (an object
	 *                representing the selected checkbox in the group)
	 */

public ContribCheckbox getCurrentJC() {
    Checkbox rval = super.getCurrent();
    if (rval instanceof ContribCheckbox) {
        return (ContribCheckbox)rval;
    }
    return null;
}

	/**
	 * Sets the currently-selected checkbox to that containing the
	 * value "val".  Typically, using ContribCheckbox and ContribCheckboxGroup
	 * involves creating an array representing values assigned to each
     * checkbox. This function takes the array and the desired value,
     * finds the checkbox item, and makes that checkbox the current item.
     * This makes it relatively easy to set checkbox values based on user
     * data.  For example:
	 * <PRE>
       int boxValues[] = { BLUE, GREEN, VIOLET };
       ContribCheckboxGroup boxG;

       public getColor() {
         boxG.set(boxValues, myObject.getColor());
       }
	 </PRE>
	 */


public void set(int[] values, int val) {
    int index = -1;
    for (int i = 0; i < values.length; i++) {
        if (values[i] == val) {
            index = i;
        }
    }
    if (index >= 0 && index < objs.size() &&
        objs.elementAt(index) instanceof Checkbox) {
        setCurrent((Checkbox)objs.elementAt(index));
    }
}

}




