
/** IMulticolumnListbox */

package COM.ibm.ivj.javabeans;

import java.awt.*;
import java.awt.event.*;
import java.util.*;

/** Listbox canvas draws data items and highlight status */
class IMulticolumnListboxCanvas extends Canvas
                             implements AdjustmentListener
{
   private static String copywrite = "IBM(R) VisualAge(TM) for Java(TM), Version 1 and others\n - Licensed Material - Program-Property of IBM\n(C) Copyright IBM Corp. 1997 - All Rights Reserved.\nUS Government Users Restricted Rights - Use, duplication or disclosure\nrestricted by GSA ADP Schedule Contract with IBM Corp.";
   /** The canvas is contained within a Listbox bean */
   private IMulticolumnListbox parent;
   private static int separatorWidth = 1;
   private boolean hasMouse = false;
   private Color selectedColor = new Color(0,0,128);

   public IMulticolumnListboxCanvas(IMulticolumnListbox listbox)
   {
      parent = listbox;

      // enable mouse events
      enableEvents( AWTEvent.MOUSE_EVENT_MASK );

      // These listeners are required because AWT calls paint before moving
      // the scroll area when it's clicked, but after when it's dragged.
      // The consequence is that, without listening and repainting when the
      // scrollbar is adjusted, clicking leaves part of the window unpainted.
      // This workaround causes a double repaint when the window is dragged
      // or resized.
      parent.listboxScrollPane.getVAdjustable().addAdjustmentListener(this);
      parent.listboxScrollPane.getHAdjustable().addAdjustmentListener(this);
   }

   // handle when font is null
   public Font getFont()
   {
      Font f = super.getFont();
      if ( f == null ) return parent.getFont();
      else return f;
   }

   protected int getRowHeight()
   {
      return getFontMetrics(getFont()).getHeight() + IMulticolumnListbox.rowPadding;
   }

   public Dimension getPreferredSize()
   {
      Dimension d = super.getPreferredSize();
      d.height = getRowHeight() * parent.getRowCount() + separatorWidth;
      return d;
   }

   protected void setWidth(int width)
   {
      Dimension d = getSize();
      d.width = width;
      setSize(d);
   }

   public void setSelectedColor( Color aColor ) { selectedColor = aColor; }
   public Color getSelectedColor() { return selectedColor; }

    /** paint: display the list content
     * @param g the graphics context
    */
   public void paint(Graphics g)
   {
      Dimension size = getSize();
      int[] columnWidths = parent.getColumnWidths();
      if ( columnWidths == null ) return;
      int height = getRowHeight();
      Color foregroundColor = getForeground();
      Color backgroundColor = getBackground();
      if ( foregroundColor.equals(selectedColor) )
         selectedColor = backgroundColor;

      int y = 0;
      int row = 0;

      // performance - skip the areas not visible in the scrollpane
      Point lastScrollPosition = parent.listboxScrollPane.getScrollPosition();
      int vStart = lastScrollPosition.y;
      int vEnd = parent.listboxScrollPane.getViewportSize().height + vStart;
      int hEnd = lastScrollPosition.x +
                    parent.listboxScrollPane.getViewportSize().width;

      // skip the area not visible
      if ( vStart > 0 )
      {
         row = vStart / height;
         y = height * row;
      }

      int separatorStartHeight = y;

      while (y < size.height)
      {
         if ( y > vEnd ) // rest of area not visible
            break;
         String values[] = parent.getRowData(row);
         if (values == null) // end of row data
         {
            // paint the end row separator
            g.setColor(foregroundColor);
            g.fillRect( 0
                      , y
                      , parent.getTotalWidth() - IMulticolumnListbox.columnPadding / 2 + separatorWidth
                      , separatorWidth );
            y += separatorWidth;
            // end of data, paint remainder of canvas area
            g.setColor(backgroundColor);
            g.fillRect(0, y, size.width, size.height - y);
            break;
         }
         else
         {
             // paint item, reverse colors if selected row
             int x = 0;
             Color fg,bg;
             if ( parent.isSelected(row) )
             {
               fg = backgroundColor;
               bg = selectedColor;
             }
             else
             {
               fg = foregroundColor;
               bg = backgroundColor;
             }

             // paint columns in overlapping fashion to simulate clipping
             for (int column = 0; column < columnWidths.length; column ++)
             {
                 g.setColor(bg);
                 g.fillRect( x, y, columnWidths[column],height);

                 g.setColor(fg);
                 if ( column < values.length && values[column] != null )
                    g.drawString(values[column], x, y + height - IMulticolumnListbox.rowPadding);
                 // clip the pad area
                 g.setColor(bg);
                 g.fillRect( x + columnWidths[column] - IMulticolumnListbox.columnPadding
                           , y
                           , IMulticolumnListbox.columnPadding
                           , height );
                 x += columnWidths[column];
                 if ( x > hEnd ) // rest of area not visible
                    break;
             }

             // clear to end of row to simulate clipping last column
             g.setColor(backgroundColor);
             int fillStart = x - IMulticolumnListbox.columnPadding / 2 + separatorWidth;
             g.fillRect( fillStart
                       , y
                       , size.width - fillStart
                       , height);
             if ( row == parent.lastClicked )
             {
                // draw the selected box
                g.setColor(foregroundColor);
                g.drawRect( 0
                          , y + 1
                          , fillStart - 2
                          , height - 3 );
             }
             y += height;
         }
         row ++;
      }
      // paint in the separators after, for performance
      g.setColor(foregroundColor);
      int separatorHeight;
      if ( y > vEnd )
         separatorHeight = vEnd - separatorStartHeight;
      else
         separatorHeight = y - separatorStartHeight;
      int x = 0;
      for (int column = 0; column < columnWidths.length; column ++)
      {
         g.fillRect( x + columnWidths[column] - IMulticolumnListbox.columnPadding / 2
                   , separatorStartHeight
                   , separatorWidth
                   , separatorHeight );
         x += columnWidths[column];
      }
   }

    /** paintRow: repaint a single row
     * @param row - the row to repaint
    */
   public void repaintRow(int row)
   {
      Graphics g = getGraphics();
      if ( g == null )
         return; // not currently visible
      Dimension size = getSize();
      int[] columnWidths = parent.getColumnWidths();
      if ( columnWidths == null ) return;
      int height = getRowHeight();
      Color foregroundColor = getForeground();
      Color backgroundColor = getBackground();
      if ( foregroundColor.equals(selectedColor) )
         selectedColor = backgroundColor;

      int y = 0;

      // performance - skip the areas not visible in the scrollpane
      int vStart = parent.listboxScrollPane.getScrollPosition().y;
      int vEnd = parent.listboxScrollPane.getViewportSize().height + vStart;
      int hEnd = parent.listboxScrollPane.getScrollPosition().x +
                    parent.listboxScrollPane.getViewportSize().width;

      y = height * row;
      if ( y <= vEnd )
      {
         String values[] = parent.getRowData(row);
         if ( values != null )
         {
            // paint item, reverse colors if selected row
            int x = 0;
            Color fg,bg;
            if ( parent.isSelected(row) )
            {
              fg = backgroundColor;
              bg = selectedColor;
            }
            else
            {
              fg = foregroundColor;
              bg = backgroundColor;
            }

            // paint columns in overlapping fashion to simulate clipping
            for (int column = 0; column < columnWidths.length; column ++)
            {
                g.setColor(bg);
                g.fillRect( x, y, columnWidths[column],height);

                g.setColor(fg);
                if ( column < values.length && values[column] != null )
                   g.drawString(values[column], x, y + height - IMulticolumnListbox.rowPadding);
                // clip the pad area
                g.setColor(bg);
                g.fillRect( x + columnWidths[column] - IMulticolumnListbox.columnPadding
                          , y
                          , IMulticolumnListbox.columnPadding
                          , height );
                // draw the separator
                g.setColor(foregroundColor);
                g.fillRect( x + columnWidths[column] - IMulticolumnListbox.columnPadding / 2
                          , y
                          , separatorWidth
                          , height );
                x += columnWidths[column];
                if ( x > hEnd ) // rest of area not visible
                   break;
            }
            // clear to end of row to simulate clipping last column
            g.setColor(backgroundColor);
            int fillStart = x - IMulticolumnListbox.columnPadding / 2 + separatorWidth;
            g.fillRect( fillStart
                      , y
                      , size.width - fillStart
                      , height);
            if ( row == parent.lastClicked )
            {
               // draw the selected box
               g.setColor(foregroundColor);
               g.drawRect( 0
                         , y + 1
                         , fillStart - 2
                         , height - 3 );
            }
         }
      }
   }

   public void adjustmentValueChanged( AdjustmentEvent e )
   {
      // make sure the visible area has been repainted, the repaint
      // request from the scrollPane comes before it's been adjusted.
      if ( e.getAdjustmentType() == AdjustmentEvent.TRACK )
         repaint();
   }
   // this methods are not synchronized on purpose:
   //   though the cursor might get out of sync in rare
   //   cases, this is preferable to the performance penalty
   protected void showBulkChangeMode()
   {
      setCursor(java.awt.Cursor.getPredefinedCursor(java.awt.Cursor.WAIT_CURSOR));
   }
   protected void resetBulkChangeMode()
   {
      setCursor(java.awt.Cursor.getDefaultCursor());
   }
   protected boolean hasCursor() { return hasMouse; }
   public int rowAt( Point p )
   {
      int height = getFontMetrics(getFont()).getHeight() + IMulticolumnListbox.rowPadding;
      return p.y / height;
   }

   protected void processMouseEvent(MouseEvent event)
   {
      if ( event.getID() == MouseEvent.MOUSE_ENTERED )
         _mouseEntered(event);
      else if ( event.getID() == MouseEvent.MOUSE_CLICKED )
         _mouseClicked(event);
      super.processMouseEvent(event);
   }

   private void _mouseClicked(MouseEvent event)
   {
      // determine row the click occured on
      int row = rowAt(event.getPoint());

      if ( row < parent.getRowCount() )
      {
         int oldLastClicked = parent.lastClicked;
         parent.lastClicked = row;
         if ( ! parent.isSelected(row) )
            parent.selectRow(row);
         else if ( parent.multipleMode )
            parent.unselectRow(row);
         if ( parent.multipleMode &&
              oldLastClicked != -1 &&
              oldLastClicked != row )
            repaintRow(oldLastClicked);
      }
   }
   private void _mouseEntered( MouseEvent e )
   {
      hasMouse = true;
      if ( parent.isBulkChangeMode() )
         showBulkChangeMode();
      else
      {
         java.awt.Cursor parentCursor = parent.getCursor();
         if ( parent.getCursor() != null )
            setCursor(parentCursor);
         else
            resetBulkChangeMode();
      }
   }

   // Override update to prevent the background from being cleared
   // Since we repaint all anyway, this helps reduce flicker.
   public void update( Graphics g )
   {
      paint(g);
   }
}

/** Listbox canvas draws headings and highlight status */
class IMulticolumnHeadingCanvas extends Canvas
                                implements AdjustmentListener
{
   private static String copywrite = "IBM(R) VisualAge(TM) for Java(TM), Version 1 and others\n - Licensed Material - Program-Property of IBM\n(C) Copyright IBM Corp. 1997 - All Rights Reserved.\nUS Government Users Restricted Rights - Use, duplication or disclosure\nrestricted by GSA ADP Schedule Contract with IBM Corp.";
   /** The canvas is contained within a Listbox bean */
   private IMulticolumnListbox parent;
   private static int separatorWidth = 1;
   private int width = 0;

    /** Constructor
     * @param listbox the container
    */
   public IMulticolumnHeadingCanvas( IMulticolumnListbox listbox )
   {
      parent = listbox;
      // enable mouse events of interest
      enableEvents( AWTEvent.MOUSE_EVENT_MASK + AWTEvent.MOUSE_MOTION_EVENT_MASK );
      // setup the heading to track the horizontal listboxPane changes
      parent.listboxScrollPane.getHAdjustable().addAdjustmentListener(this);
   }

   // handle when font is null
   public Font getFont()
   {
      Font f = super.getFont();
      if ( f == null ) return parent.getFont();
      else return f;
   }

   protected int getHeadingHeight()
   {
      int fontHeight;
      Font f = getFont();
      fontHeight = getFontMetrics(f).getHeight();
      return fontHeight +
             IMulticolumnListbox.rowPadding +
             2 * separatorWidth; // top and bottom separator
   }

   public Dimension getPreferredSize()
   {
      Dimension d = super.getPreferredSize();
      if ( width > 0 )
         d.width = width;
      return d;
   }

   protected void setWidth(int aWidth)
   {
      width = aWidth;
      Dimension d = new Dimension( aWidth, getHeadingHeight() );
      setSize(d);
   }

    /** paint: display the list content
     * @param g the graphics context
    */
   public void paint(Graphics g)
   {
      Dimension size = getSize();
      int[] columnWidths = parent.getColumnWidths();
      if ( columnWidths == null ) return;
      int rowHeight = getHeadingHeight();
      int height = rowHeight - IMulticolumnListbox.rowPadding + 2 * separatorWidth;
      String[] values = parent.getColumns();
      Color foregroundColor = getForeground();
      Color backgroundColor = getBackground();

      int y = 0;

      // paint columns in overlapping fashion to simulate clipping
      int x = 0;
      for (int column = 0; column < columnWidths.length && column < values.length; column ++)
       {
          g.setColor(backgroundColor);
          g.fillRect(x, y, columnWidths[column], rowHeight);
          g.setColor(foregroundColor);
          g.drawString(values[column], x, y + height );
          // clip the pad area
          g.setColor(backgroundColor);
          g.fillRect( x + columnWidths[column] - IMulticolumnListbox.columnPadding
                    , y
                    , IMulticolumnListbox.columnPadding
                    , rowHeight );
          // draw the separator
          g.setColor(foregroundColor);
          g.fillRect( x + columnWidths[column] - IMulticolumnListbox.columnPadding / 2
                    , y
                    , separatorWidth
                    , rowHeight );
          x += columnWidths[column];
       }

       // clear to end of row to simulate clipping last column
       g.setColor(backgroundColor);
       g.fillRect(x, y, size.width - (x - 1), rowHeight);

       // add the row height
       y += rowHeight;

   }

   private int columnBoundarySelected( int xOffset )
   {
      int column = 0;
      int[] columnWidths = parent.getColumnWidths();
      if ( columnWidths == null )
         return -1;
      int range = IMulticolumnListbox.columnPadding;
      while ( column < columnWidths.length &&
               xOffset > columnWidths[column] )
         xOffset -= columnWidths[column++];
      if ( column < columnWidths.length &&
              columnWidths[column] - xOffset < range )
         return column; // close to trailing edge
      else if ( xOffset < range && column > 0 &&
                  column - 1 < columnWidths.length )
         return --column; // close to leading edge
      else // no selection
         return -1;
   }

   protected void processMouseEvent(MouseEvent event)
   {
      if ( event.getID() == MouseEvent.MOUSE_ENTERED )
         columnResizing = false;
      else if ( event.getID() == MouseEvent.MOUSE_EXITED )
         columnResizing = false;
      else if ( event.getID() == MouseEvent.MOUSE_CLICKED )
         _mouseClicked(event);
      else if ( event.getID() == MouseEvent.MOUSE_PRESSED )
         _mousePressed(event);
      else if ( event.getID() == MouseEvent.MOUSE_RELEASED )
         _mouseReleased(event);
      super.processMouseEvent(event);
   }

    /** handleMouseClick: select item on mouse click
     * @param y the vertical location of the click
     */
   private void _mouseClicked(MouseEvent event)
   {
      if ( event.getClickCount() == 2 )
         parent.setToFitAvailableWidth();
      else if ( event.getClickCount() == 3 )
         parent.setToDataPreferredWidths();
      else if ( event.getClickCount() == 4 )
         parent.setToHeadingPreferredWidths();
   }
   private int xStart = 0;
   private int columnStart = -1;
   private boolean columnResizing = false;
   private void _mousePressed( MouseEvent e )
   {
      xStart = e.getX();
      columnStart = columnBoundarySelected(xStart);
      if ( columnStart >= 0  )
         columnResizing = true;
   }
   private void _mouseReleased( MouseEvent e )
   {
      if ( columnResizing && columnStart >= 0 )
      {
         int minSize = getFontMetrics(getFont()).charWidth(' ') + IMulticolumnListbox.columnPadding;
         int deltaSize = e.getX() - xStart;
         int newSize = parent.getColumnWidths()[columnStart] + deltaSize;
         if ( newSize < minSize ) newSize = minSize;
         parent.setColumnWidth(columnStart,newSize);
         // see if cursor should be changed back
         _mouseMoved(e);
      }
      columnResizing = false;
   }

   protected void processMouseMotionEvent(MouseEvent event)
   {
      if ( event.getID() == MouseEvent.MOUSE_MOVED )
         _mouseMoved(event);
      super.processMouseMotionEvent(event);
   }

   private void _mouseMoved( MouseEvent e )
   {
      if ( columnBoundarySelected(e.getX()) >= 0  )
         setCursor(java.awt.Cursor.getPredefinedCursor(java.awt.Cursor.E_RESIZE_CURSOR));
      else
         setCursor(java.awt.Cursor.getDefaultCursor());
   }

   // Override update to prevent the background from being cleared
   // Since we repaint all anyway, this helps reduce flicker.
   public void update( Graphics g )
   {
      paint(g);
   }

   // track changes to the listbox vertical position
   public void adjustmentValueChanged( AdjustmentEvent e )
   {
      // we're only listening to the horizontal scroll bar
      if ( e.getAdjustmentType() == AdjustmentEvent.TRACK &&
           e.getSource() == parent.listboxScrollPane.getHAdjustable() )
         parent.headingScrollPane.setScrollPosition(parent.listboxScrollPane.getScrollPosition());
   }


}

public class IMulticolumnListbox extends Panel
                                 implements ItemSelectable
                                          , java.io.Serializable
                                          , IVectorListener
{

   private static String copywrite = "IBM(R) VisualAge(TM) for Java(TM), Version 1 and others\n - Licensed Material - Program-Property of IBM\n(C) Copyright IBM Corp. 1997 - All Rights Reserved.\nUS Government Users Restricted Rights - Use, duplication or disclosure\nrestricted by GSA ADP Schedule Contract with IBM Corp.";
   private static ResourceBundle resources = null;
   static {
         try {
            resources = ResourceBundle.getBundle("COM.ibm.ivj.javabeans.IMulticolumnListboxResources");
         }
         catch (Exception exc) {
            resources = null;
         }
   }
   private transient boolean headingFontSet = false;
   private transient Vector rows = new Vector();
   private transient Vector keys = new Vector();
   private transient Vector selected = new Vector();
   private String[] columns;
   private int[] columnWidths = null;
   private transient IVector elements = null;
   protected boolean multipleMode = false;

   protected transient ScrollPane listboxScrollPane = new ScrollPane();
   protected transient ScrollPane headingScrollPane = new ScrollPane(ScrollPane.SCROLLBARS_NEVER);
   private transient IMulticolumnListboxCanvas listboxCanvas = new IMulticolumnListboxCanvas(this);
   private transient IMulticolumnHeadingCanvas headingCanvas = new IMulticolumnHeadingCanvas(this);
   protected transient boolean bulkChangeMode = false;

   private transient Vector itemListeners = null;

   protected static final int rowPadding = 4;
   protected static final int columnPadding = 8;

   protected transient int lastClicked = -1;

   public IMulticolumnListbox()
   {
      this(null);
   }

   public IMulticolumnListbox( boolean allowMultipleMode )
   {
      this();
      setMultipleMode(allowMultipleMode);
   }

   public IMulticolumnListbox( String[] columnHeadings )
   {
      super.setLayout(new GridBagLayout());

      setHeadingBackground(new Color(192,192,192));
      setHeadingForeground(Color.black);
      setDataBackground(new Color(160,160,164));
      setDataForeground(Color.black);

      listboxScrollPane.add(listboxCanvas);
      headingScrollPane.add(headingCanvas);

      GridBagConstraints c;

      c = new GridBagConstraints();
      c.fill = GridBagConstraints.HORIZONTAL;
      c.weightx = 1.0;
      c.weighty = 0.0;
      c.anchor = GridBagConstraints.NORTHWEST;
      c.gridwidth = GridBagConstraints.REMAINDER;
      add(headingScrollPane,c);

      c = new GridBagConstraints();
      c.fill = GridBagConstraints.BOTH;
      c.anchor = GridBagConstraints.NORTHWEST;
      c.weightx = 1.0;
      c.weighty = 1.0;
      c.gridwidth = GridBagConstraints.REMAINDER;
      add(listboxScrollPane,c);

      if ( columnHeadings != null )
         setColumns(columnHeadings);
   }

   public IMulticolumnListbox( String[] columnHeadings
                             , boolean allowMultipleMode )
   {
      this(columnHeadings);
      setMultipleMode(allowMultipleMode);
   }

   // handle when font is null
   public Font getFont()
   {
      Font f = super.getFont();
      if ( f == null )
         return new Font(getToolkit().getFontList()[0],Font.PLAIN,0);
      else
         return f;
   }

   public Dimension getPreferredSize()
   {
      // Scrollpane's don't consult their clients when returning the preferred
      // width.  To make the listbox startoff with a reasonable preferred size,
      // this method skips the scrollpanes and asks the display areas for their
      // preferred sizes.  The extra width helps workaround an awt scrollpane
      // problem; it gets confused if the scrollbars might be shown.  The size
      // is clipped at 600x400, which seems reasonable.
      Dimension d = super.getPreferredSize();
      Dimension l = listboxCanvas.getPreferredSize();
      Dimension h = headingCanvas.getPreferredSize();
      if ( d.width < h.width && h.width < 600 )
         d.width = h.width + listboxScrollPane.getVScrollbarWidth();
      if ( d.height < l.height + h.height && l.height + h.height < 400 )
         d.height = l.height + h.height;
      return d;
   }

   // redraw everything
   private void invalidateListbox()
   {
      if ( bulkChangeMode )
         return;
      headingCanvas.repaint();
      listboxCanvas.repaint();
      // make sure scroll bars are repainted
      listboxScrollPane.invalidate();
      validate();
   }

   // resource access
   private static String getString( String id )
   {
      if ( resources != null )
         return resources.getString(id);
      else
         return id;
   }

   private boolean isRowVisible(int index)
   {
      int vStart = listboxScrollPane.getScrollPosition().y;
      int vEnd = listboxScrollPane.getViewportSize().height + vStart;
      int rStart = vStart / listboxCanvas.getRowHeight();
      int rEnd = vEnd / listboxCanvas.getRowHeight();
      if ( index >= rStart && index <= rEnd )
         return true;
      else
         return false;
   }

   public synchronized String[] getRowData(int index)
   {
      return index < rows.size() ? (String[]) rows.elementAt(index) : null;
   }
   public synchronized String[] getRowData(Object row)
   {
      int index = rowIndexOf(row);
      if ( index > -1 )
         return (String[]) rows.elementAt(index);
      else
         return null;
   }

   public synchronized void setColumns( String[] columnHeadings )
   {
      unselectAll();
      rows.removeAllElements();
      keys.removeAllElements();
      selected.removeAllElements();
      columns = columnHeadings;
      if ( columns != null )
      {
         if ( columnWidths == null || columnWidths.length < columns.length )
         {
            int[] oldWidths = columnWidths;
            columnWidths = new int[columns.length];
            setToHeadingPreferredWidths();
            if ( oldWidths != null )
               for ( int i = 0; i < oldWidths.length && i < columnWidths.length; i++ )
                  columnWidths[i] = oldWidths[i];
         }
      }
      invalidateListbox();
   }
   public String[] getColumns() { return columns; }

   public int getRowCount() { return rows.size(); }

   public int getColumnCount()
   {
      return columns == null ? 0 : columns.length;
   }

   public Object getRowKey( int index )
   {
      if ( index < keys.size() )
         return keys.elementAt(index);
      else
         return null;
   }

   /* provide a version using == instead of .equals() */
   public synchronized int rowIndexOf( Object key )
   {
      int s = keys.size();
      int result = 0;
      for ( Enumeration e = keys.elements(); e.hasMoreElements(); result++ )
         if ( e.nextElement() == key )
            return result;
      return -1; // not found
   }

   // for performance - suppress screen changes while changing records
   public void setBulkChangeMode(boolean enable)
   {
      bulkChangeMode = enable;
      if ( bulkChangeMode )
      {
         if ( listboxCanvas.hasCursor() )
            listboxCanvas.showBulkChangeMode();
      }
      else
      {
         if ( listboxCanvas.hasCursor() )
            listboxCanvas.resetBulkChangeMode();
         invalidateListbox();
      }
   }
   public boolean isBulkChangeMode()
   {
      return bulkChangeMode;
   }

   // selection support

   public synchronized void addItemListener(ItemListener aListener)
   {
      Vector newListeners;
      if ( itemListeners == null )
         newListeners = new Vector();
      else
         newListeners = (Vector) itemListeners.clone();
      newListeners.addElement(aListener);
      itemListeners = newListeners;
   }
   public synchronized void removeItemListener(ItemListener aListener)
   {
      if ( itemListeners != null )
      {
         Vector newListeners = (Vector) itemListeners.clone();
         newListeners.removeElement(aListener);
         itemListeners = newListeners;
      }
   }

   private void fireItemStateChange(Object key, int selectState)
   {
      if ( itemListeners == null )
         return;
      Vector currentListeners = itemListeners;
      ItemEvent evt = new ItemEvent( this
                                 , ItemEvent.ITEM_STATE_CHANGED
                                 , key
                                 , selectState );
      for ( Enumeration e = currentListeners.elements(); e.hasMoreElements(); )
      {
         ItemListener l = (ItemListener) e.nextElement();
         l.itemStateChanged(evt);
      }
   }

   public int getSelectedIndex()
   {
      if ( selected.size() == 0 ||
           ( multipleMode && selected.size() > 1 ) )
         return -1;
      else
         return rowIndexOf(selected.firstElement());
   }

   public synchronized int[] getSelectedIndexes()
   {
      int s = selected.size();
      int[] results = new int[s];
      for ( int i = 0; i < s; i++ )
         results[i] = rowIndexOf(selected.elementAt(i));
      return results;
   }

   public synchronized Object[] getSelectedObjects()
   {
      int s = selected.size();
      Object[] results = new Object[s];
      selected.copyInto(results);
      return results;
   }

   public Object getSelectedObject()
   {
      if ( selected.size() == 0 ||
           ( multipleMode && selected.size() > 1 ) )
         return null;
      else
         return selected.firstElement();
   }

   public synchronized boolean isSelected(int index) { return isSelected(getRowKey(index)); }
   // provide our own implementation, using == instead of equals
   public synchronized boolean isSelected(Object key)
   {
      if ( key == null || selected.size() == 0 )
         return false;
      for( Enumeration e = selected.elements(); e.hasMoreElements(); )
         if ( e.nextElement() == key )
            return true;
      return false;
   }

   public synchronized void selectNext()
   {
      int index = getSelectedIndex() + 1;
      if ( index > 0 && index < getRowCount() )
         selectRow(index);
      else if ( ! multipleMode )
      {
         COM.ibm.ivj.javabeans.IMessageBox msg = new COM.ibm.ivj.javabeans.IMessageBox();
         msg.show(getString("bottomOfList"));
      }
   }
   public synchronized void selectLast()
   {
      if ( getRowCount() > 0 )
         selectRow(getRowCount() - 1);
   }
   public synchronized void selectFirst()
   {
      if ( getRowCount() > 0 )
         selectRow(0);
   }
   public synchronized void selectPrevious()
   {
      int index = getSelectedIndex();
      if ( index > 0 )
         selectRow(index-1);
      else if ( ! multipleMode )
      {
         COM.ibm.ivj.javabeans.IMessageBox msg = new COM.ibm.ivj.javabeans.IMessageBox();
         msg.show(getString("topOfList"));
      }
   }

   public synchronized void selectAll()
   {
      if ( selected.size() == keys.size() )
         return;
      for( Enumeration e = keys.elements(); e.hasMoreElements(); )
         selectRow(e.nextElement());
   }
   public synchronized void unselectAll()
   {
      if ( selected.size() == 0 )
         return;
      Vector v = (Vector) selected.clone();
      for( Enumeration e = v.elements(); e.hasMoreElements(); )
         unselectRow(e.nextElement());
   }

   public boolean isMultipleMode() { return multipleMode; }
   public synchronized void setMultipleMode(boolean enable)
   {
      multipleMode = enable;
      if ( enable == false &&
            selected.size() > 1 )
      {
         Object tempKey = selected.lastElement();
         unselectAll();
         selectRow(tempKey);
      }
   }

   public synchronized void selectRow( Object key )
   {
      if ( key != null && ! isSelected(key) )
      {
         if ( ! multipleMode && selected.size() > 0 )
            unselectRow(selected.firstElement());
         selected.addElement(key);
         fireItemStateChange(key,ItemEvent.SELECTED);
         int index = rowIndexOf(key);
         if ( isRowVisible(index) )
            listboxCanvas.repaintRow(index);
      }
   }

   public synchronized void unselectRow(Object key)
   {
      if ( key != null && isSelected(key) )
      {
         selected.removeElement(key);
         fireItemStateChange( key, ItemEvent.DESELECTED );
         int index = rowIndexOf(key);
         if ( isRowVisible(index) )
            listboxCanvas.repaintRow(index);
      }
   }

   public synchronized void selectRow( int index ) { selectRow(getRowKey(index)); }
   public synchronized void unselectRow( int index ) { unselectRow(getRowKey(index)); }


   // returns -1 if not selected
   private synchronized int _indexIfSelected(Object key)
   {
      if ( selected.size() == 0 )
         return -1;
      int result = 0;
      for( Enumeration e = selected.elements(); e.hasMoreElements(); ++result )
         if ( e.nextElement() == key )
            return result;
      return -1;
   }

   private synchronized void _replaceSelectedKey(Object oldKey, Object newKey)
   {
      int index = _indexIfSelected(oldKey);
      if ( index != -1 )
         selected.setElementAt(newKey,index);
   }

   // data modification

   public synchronized void updateRow( String[] values, int index )
   {
      rows.setElementAt(values,index);
      if ( isRowVisible(index) && ! bulkChangeMode )
         listboxCanvas.repaintRow(index);
   }

   public synchronized void updateRow( String[] values, Object key )
   {
      updateRow(values,rowIndexOf(key));
   }

   public synchronized void replaceRow( String[] values, Object key, int index )
   {
      if ( key == null ) // assign a default identifier
         key = values;
      Object oldKey = getRowKey(index);
      _replaceSelectedKey(oldKey,key);
      rows.setElementAt(values,index);
      keys.setElementAt(key,index);
      if ( isRowVisible(index) && ! bulkChangeMode )
         listboxCanvas.repaintRow(index);
   }

   public synchronized void replaceRow( String[] values, Object key, Object oldKey )
   {
      replaceRow( values, key, rowIndexOf(oldKey) );
   }

   public synchronized void addRow( String[] values, Object key, int index )
   {
      lastClicked = -1;
      if ( key == null ) // assign a default identifier
         key = values;
      if ( index > -1 &&
            index < getRowCount() )
      {
         rows.insertElementAt(values,index);
         keys.insertElementAt(key,index);
      }
      else
      {
         rows.addElement(values);
         keys.addElement(key);
      }
      if ( ! bulkChangeMode )
      {
         listboxCanvas.repaint();
         listboxScrollPane.invalidate();
         validate();
      }
   }

   public synchronized void addRow( String[] values, Object key )
   {
      addRow( values, key, -1 ); // add at end
   }

   public synchronized void removeRow( int index )
   {
      lastClicked = -1;
      unselectRow(index);
      rows.removeElementAt(index);
      keys.removeElementAt(index);
      if ( ! bulkChangeMode )
      {
         listboxCanvas.repaint();
         listboxScrollPane.invalidate();
         validate();
      }
   }

   public synchronized void removeRow( Object key ) { removeRow(rowIndexOf(key)); }

   // override panel removeAll, can cause some confusion...
   public void removeAll() { removeAllRows(); }
   public synchronized void removeAllRows()
   {
      lastClicked = -1;
      unselectAll();
      rows.removeAllElements();
      keys.removeAllElements();
      invalidateListbox();
   }

   // column width control

   protected synchronized int getTotalWidth()
   {
      if ( columnWidths == null )
         return 0;
      int totalWidth = 0;
      for ( int i = 0; i < columnWidths.length; i++ )
         totalWidth += columnWidths[i];
      return totalWidth;
   }

   protected synchronized int getPreferredDataWidth(int columnNumber)
   {
      Font f = listboxCanvas.getFont();
      FontMetrics fm = getFontMetrics(f);
      int result = getPreferredHeadingWidth(columnNumber);
      for ( Enumeration er = rows.elements(); er.hasMoreElements(); )
      {
         String[] values = (String[])er.nextElement();
         if ( values.length <= columnNumber || values[columnNumber] == null )
            continue;
         int entryWidth = fm.stringWidth(values[columnNumber]) + columnPadding;
         if ( result < entryWidth )
            result = entryWidth;
      }
      return result;
   }

   protected synchronized int getPreferredHeadingWidth(int columnNumber)
   {
      Font f = headingCanvas.getFont();
      FontMetrics fm = getFontMetrics(f);
      return columnNumber < columns.length ? fm.stringWidth(columns[columnNumber]) + columnPadding : 0;
   }

   // width is in pixels, not characters...
   protected synchronized int getColumnWidth(int columnNumber)
   {
      if ( columnWidths == null )
         return 0;
      return columnNumber < columnWidths.length ? columnWidths[columnNumber] : 0;
   }

   protected synchronized void setColumnWidth(int columnNumber, int width)
   {
      if ( columnWidths == null )
         return;
      if ( columnNumber >= columnWidths.length )
         return;
      columnWidths[columnNumber] = width;
      invalidateListbox();
   }

   // width is in pixels, not characters...
   public int[] getColumnWidths() { return columnWidths; }
   public synchronized void setColumnWidths( int[] widths )
   {
      if ( widths == null )
         setToHeadingPreferredWidths();
      else if ( columns == null || widths.length >= columns.length )
         columnWidths = widths;
      else
      {
         for ( int i = 0; i < widths.length && i < columnWidths.length; i++ )
            columnWidths[i] = widths[i];
      }
      invalidateListbox();
   }

   public synchronized void setToDataPreferredWidths()
   {
      if ( columns == null ) return;
      setCursor(java.awt.Cursor.getPredefinedCursor(java.awt.Cursor.WAIT_CURSOR));
      for ( int i = 0; i < columnWidths.length; i++ )
         columnWidths[i] = getPreferredDataWidth(i);
      setCursor(java.awt.Cursor.getDefaultCursor());
      invalidateListbox();
   }

   public synchronized void setToHeadingPreferredWidths()
   {
      if ( columns == null ) return;
      setCursor(java.awt.Cursor.getPredefinedCursor(java.awt.Cursor.WAIT_CURSOR));
      for ( int i = 0; i < columnWidths.length; i++ )
         columnWidths[i] = getPreferredHeadingWidth(i);
      setCursor(java.awt.Cursor.getDefaultCursor());
      invalidateListbox();
   }

   public synchronized void setToFitAvailableWidth()
   {
      if ( columns == null ) return;
      setCursor(java.awt.Cursor.getPredefinedCursor(java.awt.Cursor.WAIT_CURSOR));
      for ( int i = 0; i < columnWidths.length; i++ )
         columnWidths[i] = getPreferredDataWidth(i);
      int totalWidth = getTotalWidth();
      int availableWidth = listboxScrollPane.getViewportSize().width;
      if ( totalWidth == 0 )
         for ( int i = 0; i < columnWidths.length; i++ )
            columnWidths[i] = availableWidth / columnWidths.length;
      else
         for ( int i = 0; i < columnWidths.length; i++ )
            columnWidths[i] = columnWidths[i] * availableWidth / totalWidth;
      setCursor(java.awt.Cursor.getDefaultCursor());
      invalidateListbox();
   }

   /** doLayout: size the heading and listbox canvas */
   public void doLayout()
   {

      if ( ! headingFontSet )
      {
         // reset the heading font to be a bit bigger and bolder
         Font thisFont = getFont();
         if ( thisFont != null )
         {
            int newSize = thisFont.getSize();
            int newStyle = Font.PLAIN;
            if ( newSize == 0 )
               newStyle = Font.BOLD;
            else
               newSize += 3;
            setHeadingFont( new Font( thisFont.getName()
                                    , newStyle
                                    , newSize ) );
         }
      }
      // set the unit advance of the scrollbar to one row
      if ( listboxCanvas.getFont() != null )
         getVAdjustable().setUnitIncrement(listboxCanvas.getRowHeight());

      int width = getTotalWidth();
      // padding to allow for scroll bar in listbox window
      headingCanvas.setWidth(width + listboxScrollPane.getVScrollbarWidth());
      listboxCanvas.setWidth(width);

      // lay the scrollPanes on the panel
      Dimension d = getSize();

      // clip the headingCanvas to eliminate the ScrollPane boundary
      int headingHeight = headingCanvas.getHeadingHeight() + rowPadding;
      headingScrollPane.setBounds( 1, 1, d.width, headingHeight );
      listboxScrollPane.setBounds( 1, headingHeight + 1, d.width, d.height - headingHeight );

      super.doLayout();

   }

   // column heading appearance
   public void setHeadingFont(Font aFont)
   {
      headingCanvas.setFont(aFont); headingFontSet = true;
   }
   public Font getHeadingFont() { return headingCanvas.getFont(); }
   public void setHeadingForeground(Color aColor) { headingCanvas.setForeground(aColor); }
   public Color getHeadingForeground() { return headingCanvas.getForeground(); }
   public void setHeadingBackground(Color aColor) { headingCanvas.setBackground(aColor); }
   public Color getHeadingBackground() { return headingCanvas.getBackground(); }

   // allow the users to set the scroll bar behaviours.  Note that the headingScrollPane tracks
   // adjustments by the listboxScrollPane, so it doesn't need to be visible
   public Adjustable getHAdjustable() { return listboxScrollPane.getHAdjustable(); }
   public Adjustable getVAdjustable() { return listboxScrollPane.getVAdjustable(); }
   // convenience property to set the horizontal scroll units
   public int getHUnitIncrement() { return getHAdjustable().getUnitIncrement(); }
   public void setHUnitIncrement(int value) { getHAdjustable().setUnitIncrement(value); }

   // column data appearance
   public void setDataFont(Font aFont)
   {
      listboxCanvas.setFont(aFont);
      // set the unit advance of the scrollbar to one row
      if ( aFont != null )
         getVAdjustable().setUnitIncrement(listboxCanvas.getRowHeight());
   }
   public Font getDataFont() { return listboxCanvas.getFont(); }
   public void setDataForeground(Color aColor) { listboxCanvas.setForeground(aColor); }
   public Color getDataForeground() { return listboxCanvas.getForeground(); }
   public void setDataBackground(Color aColor) { listboxCanvas.setBackground(aColor); }
   public Color getDataBackground() { return listboxCanvas.getBackground(); }

   // selected data appearance
   public void setSelectedColor(Color aColor) { listboxCanvas.setSelectedColor(aColor); }
   public Color getSelectedColor() { return listboxCanvas.getSelectedColor(); }

   //
   // IVector integration
   //
   public void addRow(IMulticolumnListboxObject o
                     ,int index)
   {
     addRow( o.getAttributeStrings(), o, index );
   }
   public void addRow(IMulticolumnListboxObject o ) { addRow( o.getAttributeStrings(), o ); }
   public void updateRow(IMulticolumnListboxObject o ) { updateRow( o.getAttributeStrings(), o ); }
   public void replaceRow(IMulticolumnListboxObject newRow
                         ,IMulticolumnListboxObject oldRow )
   {
      replaceRow( newRow.getAttributeStrings(), newRow, oldRow );
   }
   public void replaceRow(IMulticolumnListboxObject newRow, int index )
   {
      replaceRow( newRow.getAttributeStrings(), newRow, index );
   }
   // for vector use
   private void _addElement( Object o, int index )
   {
      if ( o instanceof IMulticolumnListboxObject )
         addRow((IMulticolumnListboxObject)o,index);
      else // can't do much else!
         addRow(new String[] {o.toString()},o,index);
   }
   private void _updateElement( Object o, int index )
   {
      if ( o instanceof IMulticolumnListboxObject )
         replaceRow((IMulticolumnListboxObject)o,index);
      else // can't do much else!
         replaceRow(new String[] {o.toString()},o,index);
   }
   private void _refreshItems()
   {
      removeAllRows();
      if ( elements == null )
         return;
      boolean currentMode = isBulkChangeMode();
      setBulkChangeMode(true);
      for ( java.util.Enumeration e = elements.elements(); e.hasMoreElements(); )
         _addElement(e.nextElement(),-1);
      setBulkChangeMode(currentMode);
   }
   public void vectorElementAdded(IVectorEvent ev)
   {
      IVector v =  (IVector) ev.getSource ();
      int i = ev.getIndex ();

      if (i >= 0) // only one vector element is added
         _addElement(v.elementAt(i),i);
      else // refresh the whole list
         _refreshItems();
   }
   public void vectorElementChanged(IVectorEvent ev)
   {
      IVector v = (IVector) ev.getSource ();
      int i = ev.getIndex();
      _updateElement(v.elementAt(i),i);
   }
   public void vectorElementRemoved(IVectorEvent ev)
   {
      IVector v =  (IVector) ev.getSource ();
      int i = ev.getIndex ();

      if (i >= 0) // only one vector element is deleted
         removeRow(i);
      else // refresh the whole list
         _refreshItems();
   }
   public void vectorModified(IVectorEvent ev)
   {
      // refresh the whole list
      _refreshItems();
   }
   public IVector getElements() { return elements; }
   public synchronized void setElements( IVector aElements )
   {
      if ( elements != null )
         elements.removeIVectorListener(this);
      elements = aElements;
      if ( elements != null )
      {
         elements.addIVectorListener(this);
         _refreshItems();
      }
      else
         removeAllRows();
   }

   // Hide this method, to prevent the layout from being changed.
   // Also has the benifit of hiding the property in the VB
   public void setLayout( LayoutManager aManager ) {}

   // find the row given a point in the DataCanvas property, usually
   // returned by an AWTEvent (like mouseClicked)
   public int rowAt( Point p ) { return listboxCanvas.rowAt(p); }

   // these methods can be used to add various event listeners
   public Canvas getDataCanvas() { return listboxCanvas; }
   public Canvas getHeadingCanvas() { return listboxCanvas; }

}

/*// for testing... comment out the above } too
   public static void main( String[] args )
   {
      String[] titles = { "First Column"
                        , "Second Column"
                        , "third column"
                        , "forth column"
                        , "5th"
                        , "6"
                        , ""
                        , "eighth column"
                        , "Long Column"
                        , "Last Column" };
      IMulticolumnListbox l = new IMulticolumnListbox(titles,true);
      Frame f = new IMulticolumnListboxMainWindow(l);
      f.add("Center",l);
      f.pack();
//      f.setSize(f.getPreferredSize());
//      (new FillListbox(l)).start(); // threaded fill
      FillListbox.fillIt(l); // unthreaded fill
      f.setSize(new Dimension(800,400));
      f.show();
   }

}

class FillListbox extends Thread
{
   IMulticolumnListbox iOwner;
   public FillListbox( IMulticolumnListbox anOwner )
   {
      iOwner = anOwner;
   }
   public void run()
   {
      fillIt(iOwner);
   }
   public static void fillIt( IMulticolumnListbox owner )
   {
//      owner.setBulkChangeMode(true);
      for ( int i = 0; i < 50; i++ )
      {
         String[] newRow = { "Row " + i + " first column"
                           , "second column for " + i
                           , "3 for " + i
                           , "a big forth column for " + i
                           , "" + i
                           , "6-" + i
                           , "seventh column for " + i
                           , "eighth column for " + i
                           , "some really, really, really, really long column data for " + i
                           , "last column for " + i };
         owner.addRow(newRow,new Integer(i));
//         try
//         {
//            Thread.sleep(1000);
//         }
//         catch (InterruptedException e)
//         {
//         }
      }
//      owner.setToDataPreferredWidths();
//      owner.setBulkChangeMode(false);
   }
}

class IMulticolumnListboxMainWindow extends Frame
                  implements WindowListener
{
   private static String copywrite = "IBM(R) VisualAge(TM) for Java(TM), Version 1 and others\n - Licensed Material - Program-Property of IBM\n(C) Copyright IBM Corp. 1997 - All Rights Reserved.\nUS Government Users Restricted Rights - Use, duplication or disclosure\nrestricted by GSA ADP Schedule Contract with IBM Corp.";
   IMulticolumnListbox l;
   public IMulticolumnListboxMainWindow(IMulticolumnListbox anL)
   {
      super("IMulticolumnListbox Main Window");
      l = anL;
      addWindowListener(this);
   }
   public void windowClosing(WindowEvent e)
   {
      System.out.println("Selected: " + l.getSelectedObject());
      Object[] items = l.getSelectedObjects();
      System.out.println("Selected Objects: ");
      if ( items != null )
      {
         for ( int i = 0; i < items.length; i++ )
            System.out.println(items[i].toString());
      }
      Runtime.getRuntime().exit(0);
   }
   public void windowClosed(WindowEvent e)
   {
      Runtime.getRuntime().exit(0);
   }
   public void windowActivated(WindowEvent e) {}
   public void windowDeactivated(WindowEvent e) {}
   public void windowDeiconified(WindowEvent e) {}
   public void windowIconified(WindowEvent e) {}
   public void windowOpened(WindowEvent e) {}
}

*/

