package com.ibm.vap.Transactions;

/**
 * Transaction
 *
 * Licensed Material - Property of IBM 
 * IBM(R) VisualAge(TM) for Java(TM) Version 2.0 
 * (C) Copyright IBM Corp. 1997, 1998 - All Rights Reserved. 
 * US Government Users Restricted Rights - Use, duplication or disclosure 
 * restricted by GSA ADP Schedule Contract with IBM Corp. 
 *
 */

import java.util.Vector;
import java.util.Enumeration;
import java.rmi.RemoteException;
import com.ibm.vap.common.VapTransactionRequiredException;
import com.ibm.vap.common.VapTransactionFailureException;
import com.ibm.vap.common.VapMergeFailureException;
import com.ibm.vap.Persistence.ResourceManager;
import com.ibm.vap.Persistence.Resource;
import com.ibm.vap.Persistence.Session;
import com.ibm.vap.Persistence.DataStore;
import com.ibm.vap.Isolation.TransactionIsolationPolicy;

public class Transaction extends java.lang.Object implements VapDeferredMessageReceiver
{
	private static java.util.ResourceBundle resobjectExtender = java.util.ResourceBundle.getBundle("objectExtender");  //$NON-NLS-1$
	protected static Transaction current;
	protected static Transaction shared;
	protected Transaction parent = null;
	protected Vector children = new Vector();
	protected TransactionView view;
	protected ResourceManager resourceManager = null;
	protected TransactionIsolationPolicy isolationPolicy = null;
	protected TransactionState state;
	protected String name = null;
	protected TransactionSignaller transactionSignaller = new TransactionSignaller(this);
	private static final java.lang.String copyright  = "(c) Copyright International Business Machines Corporation, 1998";
/**
 * Transaction constructor (used only by the composition editor)
 */
public Transaction()
{
}
/**
 * Transaction constructor
 */
public Transaction (String aName)
{
	this.setName(aName);
	this.setState(this.createActiveState());
	this.setView(this.createView());
	this.primResume();
}
/**
 * Add a deferred message because we have been merged.
 * Have this transaction signal this event.
 */
public void abtMerged()
{
	this.addDeferredMessage(new VapDeferredMessage(this, "signalMergedIntoEvent"));
}
/**
 * Record the argument who wishes to be notified when we signal any events
 */
public void addActionListener(VapActionListener listener) {
	transactionSignaller.addActionListener(listener);
}
	
/**
 * Add a new child transaction
 */
protected void addChild(Transaction aChild)
{
	this.getChildren().addElement(aChild);
}
/**
 *	Add the aMsg argument as a deferred message.
 *	This is deferred to the TransactinSignaller object
 */
public void addDeferredMessage(VapDeferredMessage aMsg){
	transactionSignaller.addDeferredMessage(aMsg) ;
}
/**
 *	Add the version argument as a deferred signaller.
 *	This is deferred to the TransactinSignaller object
 */
public void addDeferredSignaller(Version aVersion){
	transactionSignaller.addDeferredSignaller(aVersion) ;
}
/**
 *	Add the argument as a deferred signaller.
 *	This is deferred to the TransactinSignaller object
 */
public void addLinkCollectionDeferredSignaller(CollectionChangesSignaller aCollectionChangesSignaller){
	transactionSignaller.addDeferredSignaller(aCollectionChangesSignaller) ;
}
/**
 * The addPropertyChangeListener method was generated to support the propertyChange field.
 */
public synchronized void addPropertyChangeListener(java.beans.PropertyChangeListener arg1) {
	transactionSignaller.addPropertyChangeListener(arg1);
}
/**
 * Begin an anonymous top-level transaction
 */
public static Transaction begin()
{
	return getShared().beginChild();
}
/**
 * Begin a named top-level transaction
 */
public static Transaction begin(String aName)
{
	return getShared().beginChild(aName);
}
/**
 * Begin an anonymous child transaction
 */
public Transaction beginChild()
{
	return this.beginChild(resobjectExtender.getString("VapUnknown")); //$NON-NLS-1$
}
/**
 * Begin a named child transaction
 */
public Transaction beginChild(String aName)
{
	Transaction child = this.createChildTransaction(aName);
	this.addChild(child);
	child.setParent(this);

	return child;
}
/**
 * Begin an anonymous read-only top-level transaction
 */
public static Transaction beginReadOnly()
{
	return getShared().beginReadOnlyChild();
}
/**
 * Begin a named read-only top-level transaction
 */
public static Transaction beginReadOnly(String aName)
{
	return getShared().beginReadOnlyChild(aName);
}
/**
 * Begin an anonymous read-only child transaction
 */
public Transaction beginReadOnlyChild()
{
	return this.beginReadOnlyChild(resobjectExtender.getString("VapUnknown")); //$NON-NLS-1$
}
/**
 * Begin a named read only child transaction
 */
public Transaction beginReadOnlyChild(String aName)
{
	Transaction child = this.createReadOnlyChildTransaction(aName);
	this.addChild(child);
	child.setParent(this);

	return child;
}
/**
 *	Call the deferred messages.  This is deferred to the transaction signaller
 */
public void callDeferredMessages(){
	transactionSignaller.callDeferredMessages();
}
/**
 *	Call the deferred signallers.  This is deferred to the transaction signaller
 */
public void callDeferredSignallers(){
	transactionSignaller.callDeferredSignallers();
}
/**
 * Request a transaction to commit
 */
public void commit() throws RemoteException, VapTransactionFailureException, VapTransactionRequiredException
{
	this.getState().commit(this);
}
/**
 * Commit child transactions
 */
protected void commitChildren() throws RemoteException, VapTransactionFailureException
{
	Enumeration theChildren;
	Transaction child;
	
	theChildren = this.getChildren().elements();
	while (theChildren.hasMoreElements())
	{
		child = (Transaction)theChildren.nextElement();
		child.secondaryCommit();
	}	
}
/**
 * Request a transaction to commit.
 * In case of failure rollback the transaction.
 */
public void commitOrRollback() throws RemoteException, VapTransactionRequiredException
{
	try {
		this.getState().commit(this);}
	catch (VapTransactionRequiredException aFailureException) {
		throw aFailureException;}
	catch (RemoteException aFailureException) {
		this.rollback();}		
}
/**
 * Create transaction's active state
 */
protected TransactionState createActiveState()
{
	return new ActiveTransactionState();
}
/**
 * Create a child transaction
 */
protected Transaction createChildTransaction(String aName)
{
	return new Transaction(aName);
}
/**
 * Create transaction's committed state
 */
protected TransactionState createCommittedState()
{
	return new CommittedTransactionState();
}
/**
 * Create transaction's isolation policy
 */
protected TransactionIsolationPolicy createDefaultIsolationPolicy()
{
	return TransactionIsolationPolicy.createRepeatableReadPolicy();
}
/**
 * Create transaction's mark to be rolled back state
 */
protected TransactionState createMarkedRollbackState()
{
	return new MarkedRollbackTransactionState();
}
/**
 * Create a read only child transaction
 */
protected Transaction createReadOnlyChildTransaction(String aName)
{
	return new ReadOnlyTransaction(aName);
}
/**
 * Create transaction's rolled back state
 */
protected TransactionState createRolledBackState()
{
	return new RolledBackTransactionState();
}
/**
 * Create transaction's view
 */
protected TransactionView createView()
{
	return new TransactionView(this);
}
/**
 * Answer an integer to represent our depth.  This is 0 if we have no parent
 */
public int depth()
{
	Transaction parentTx;

	parentTx = this;

	int i = 0;
	while (parentTx.getParent() != null)
	{
		i = i + 1;
		parentTx = parentTx.getParent();
	};

	return i;	
}
/**
 * Discard all the internal and external resources
 * needed by the transaction
 */
protected void discard()
{
	if (this.isTopLevel() || this.isParentShared())
		this.getParent().removeChild(this);

	this.suspend();
	this.discardChildren();
	this.getResourceManager().discard();
	this.getView().discard();
	this.setView(null);
	this.setParent(null);
	this.setIsolationPolicy (null);
}
/**
 * Discard the transaction's children
 */
protected void discardChildren()
{
	Enumeration theChildren;
	Transaction child;

	theChildren = this.getChildren().elements();
	while (theChildren.hasMoreElements())
	{
		child = (Transaction)theChildren.nextElement();
		child.discard();
	}
	
	this.setChildren(null);
}
/**
 * Answer child transactions
 */
protected Vector getChildren()
{
	return children;
}
/**
 * Aswer the current transaction
 */
public static synchronized Transaction getCurrent()
{
	Transaction currentTransaction;

	if ((currentTransaction = primGetCurrent()) == null)
	{
		currentTransaction = getShared();
		setCurrent(currentTransaction);
	}	

	return currentTransaction;
}
/**
 * Aswer the current thread
 */
public static ContextThread getCurrentThread()
{
	Thread currentThread;

	if ((currentThread = Thread.currentThread()) instanceof ContextThread)
		return (ContextThread)currentThread;
	else
		return null;
}
/**
 * Aswer the current transaction's view
 */
public static TransactionView getCurrentView()
{
	return getCurrent().getView();
}
/**
 * Aswer transaction's isolation policy
 */
public TransactionIsolationPolicy getIsolationPolicy()
{
	if (isolationPolicy == null)
		if (this.isParentReadOnly())
			isolationPolicy = this.createDefaultIsolationPolicy();
		else
			isolationPolicy = this.getParent().getIsolationPolicy().createCopy();

	return isolationPolicy;
}
/**
 * Answer the transaction's name
 */
public String getName()
{
	return name;
}
/**
 * Aswer child transaction's parent transaction
 */
public Transaction getParent()
{
	return parent;
}
/**
 * Answer transaction's resource manager
 */
public  ResourceManager getResourceManager()
{
	if (resourceManager == null)
		resourceManager = new ResourceManager(this);
	
	return resourceManager;
}
/**
 * Answer a read session for a data store
 */
public Session getSessionOn(DataStore aDataStore) throws RemoteException
{
	return this.getResourceManager().getReadSession(aDataStore);
}
/**
 * Aswer the shared transaction
 */
public static Transaction getShared()
{
	if (shared == null)
		shared = new SharedTransaction("Shared");

	return shared;
}
/**
 * Answer transaction's state
 */
protected TransactionState getState()
{
	return state;
}
/**
 * Aswer the top level transaction
 */
public Transaction getTopLevel()
{
	if (this.isTopLevel())
		return this;
	else
		return this.getParent().getTopLevel();
}
/**
 * Answer transaction's view
 */
public TransactionView getView()
{
	return view;
}
/**
 * Answer true if the transaction s active
 */
public boolean isActive( )
{
	if (this.getState() != null) 
		return this.getState().isActive();
	else
		return false;
}
/**
 * Test if the transaction is a child transaction
 */
public boolean isChild( )
{
	return true;
}
/**
 * Test if the transaction is descendant of another transaction
 */
public boolean isCommitted( )
{
	return this.getState().isCommitted();
}
/**
 * Test if the transaction is descendant of another transaction
 */
public boolean isDescendantOf(Transaction aTransaction)
{
	if (this == aTransaction)
		return true;
	else
		if (!this.isParentShared())
			return this.getParent().isDescendantOf(aTransaction);
		else
			return false;
}
/**
 * Test if the transaction is marked to be rolled back
 */
public boolean isMarkedRollback( )
{
	return this.getState().isMarkedRollback();
}
/**
 *
 */
public boolean isNotVapRemoved()
{
	return true;
	
}
/**
 * Test if the transaction's parent is kind of a read-only transaction
 */
public boolean isParentReadOnly( )
{
	return this.getParent().isShared() || this.getParent().isReadOnly();
}
/**
 * Test if the transaction's parent is a shared transaction
 */
public boolean isParentShared( )
{
	return this.getParent().isShared();
}
/**
 * Test if the transaction is a read-only transaction
 */
public boolean isReadOnly( )
{
	return false;
}
/**
 * Test if the transaction is rolled back
 */
public boolean isRolledBack( )
{
	return this.getState().isRolledBack();
}
/**
 * Test if the transaction is a shared transaction
 */
public boolean isShared( )
{
	return false;
}
/**
 * Test if the transaction is a top level transaction
 */
public boolean isTopLevel( )
{
	return false;
}
/**
 * Answer the levelId or index (child number) of the receiver local to its parent.
 * If no parent, answer 0.
 */
public int localLevelId( )
{
	Transaction parent = this.getParent();
	if (parent == null) 
		return 0;
	else
		return parent.getChildren().indexOf(this);
	

/**
	| aParent |
	aParent := self parent.
	^aParent == nil
		ifTrue: [0]
		ifFalse: [aParent children indexOf: self]
**/
}
/**
 * Request the transaction to be marked for rollback
 */
protected void markForRollback()
{
	this.getState().markForRollback(this);
}
/**
 * 
 */
public int maxChildDepth()
{
	return 150;

/**
	^150
**/
}
/**
 * 
 */
public int maxPrint()
{
	return 1000;

/**
	^1000
**/
}
/**
 * Merge the transaction's view to its parents view
 */
protected void mergeToParent()
{
	this.getView().mergeToParent();
}
/**
 * Commit the transaction
 */
protected void primCommit() throws RemoteException, VapTransactionFailureException
{
	Transaction holdParent;

	holdParent = this.getParent();

	try
	{
		this.commitChildren();
		this.primResume();
		this.testMergeToParent();
		if (!this.isReadOnly())
			this.synchronizeToResources();
		this.mergeToParent();

		if (this.isTopLevel() || this.isParentShared())
			this.discard();

		holdParent.primResume();
	}
	catch (RemoteException aFailureException)
	{
		this.markForRollback();
		throw aFailureException;
	}
}
/**
 * Aswer the current transaction
 */
public static Transaction primGetCurrent()
{
	ContextThread currentThread;

	if ((currentThread = getCurrentThread()) != null)
		return currentThread.getContext().getCurrentTransaction();
	else
		return current;
}
/**
 * 
 */
protected TransactionIsolationPolicy primGetIsolationPolicy()
{
	return isolationPolicy;
}
/**
 * Mark the transaction's children for rollback
 */
protected void primMarkForRollback()
{
	Enumeration theChildren;
	Transaction child;

	theChildren = this.getChildren().elements();
	while (theChildren.hasMoreElements())
	{
		child = (Transaction)theChildren.nextElement();
		child.markForRollback();
	}	
}
/**
 * Resume the transaction (make it the current one)
 */
protected void primResume()
{
	Transaction.resume(this);
}
/**
 * Roll back the transaction
 */
protected void primRollback()  throws RemoteException, VapTransactionRequiredException
{
	this.rollbackChildren();
	if (!this.isReadOnly())
		this.getResourceManager().rollback();
	this.getView().releaseLocks();
	this.getParent().primResume();
		
	if (this.isTopLevel() || this.isParentShared())
		this.discard();
}
/* Answer a string that represents us in a basic format */
public String primToString() {
	return this.getName() + ":" + this.getClass() ;
}
/**
 * Removethe argument from the list of objects who wish to be notified
 * of any events we signal
 */
public void removeActionListener(VapActionListener listener) {
	transactionSignaller.removeActionListener(listener);
}
	
/**
 * Remove a child transaction
 */
protected void removeChild(Transaction aChild)
{
	this.getChildren().removeElement(aChild);
}
/**
 * The removePropertyChangeListener method was generated to support the propertyChange field.
 */
public synchronized void removePropertyChangeListener(java.beans.PropertyChangeListener arg1) {
	transactionSignaller.removePropertyChangeListener(arg1);
}
/**
 * Discard all transactions
 */
public static synchronized void reset()
{
	if (shared != null)
		shared.discard();

	shared = null;
	current = null;
}
/**
 * Resume the transaction (make it the current one)
 */
public void resume() throws VapTransactionRequiredException
{
	this.getState().resume(this);
}
/**
 * Set the current transaction
 */
protected static synchronized void resume(Transaction aTransaction)
{
	setCurrent(aTransaction);
}
/**
 * Request a transaction to rollback
 */
public void rollback() throws RemoteException, VapTransactionRequiredException
{
	this.getState().rollback(this);
}
/**
 * Rollback child transactions
 */
protected void rollbackChildren() throws RemoteException, VapTransactionRequiredException
{
	Enumeration theChildren;
	Transaction child;
	
	theChildren = this.getChildren().elements();
	while (theChildren.hasMoreElements())
	{
		child = (Transaction)theChildren.nextElement();
		child.secondaryRollback();
	}	
}
/**
 * 
 */
public Transaction root()
{
	Transaction nextTx = this;
	while (nextTx.getParent() != null)
		{
		  nextTx = nextTx.getParent();
		};
	return nextTx;	

/**

	self doToRoot: [:each | 
		each parent isNil
			ifTrue: [^each]].


**/
}
/**
 * Request a transaction to commit
 */
public void secondaryCommit() throws RemoteException, VapTransactionFailureException, VapTransactionRequiredException
{
	this.getState().secondaryCommit(this);
}
/**
 * Request a transaction to rollback
 */
public void secondaryRollback() throws RemoteException, VapTransactionRequiredException
{
	this.getState().secondaryRollback(this);
}
/**
 * Set the child transactions
 */
protected void setChildren(Vector theChildren)
{
	children = theChildren;
}
/**
 * Set the current transaction
 */
public static void setCurrent(Transaction aTransaction)
{
	ContextThread currentThread;

	if ((currentThread = getCurrentThread()) != null)
		currentThread.getContext().setCurrentTransaction(aTransaction);
	else
		current = aTransaction;
}
/**
 * Set transaction's isolation policy
 */
public void setIsolationPolicy(TransactionIsolationPolicy aPolicy)
{
	isolationPolicy = aPolicy;
}
/**
 * Set the transaction's name
 */
public void setName(String aName)
{
	name = aName;
}
/**
 * Set child transaction's parent transaction
 */
protected void setParent(Transaction aParent)
{
	parent = aParent;
}
/**
 * Set transaction's resource manager
 */
public void setResourceManager(ResourceManager aManager)
{
	resourceManager = aManager;
}
/**
 * Set the shared transaction
 */
public static void setShared(Transaction theShared)
{
	shared = theShared;
}
/**
 * Set transaction's state
 */
protected void setState(TransactionState aState)
{
	state = aState;
}
/**
 * Set transaction's view
 */
protected void setView(TransactionView aView)
{
	view = aView;
}
/**
 * Signal that we have been committed.  Defer this to our signaller
 */
public void signalHasBeenCommitted()
{
	transactionSignaller.signalHasBeenCommitted();
}
/**
 * Signal that we have been rolledback.  Defer this to our signaller
 */
public void signalHasBeenRolledback()
{
	transactionSignaller.signalHasBeenRolledback();
}
/**
 * Signal that we have been merged.  Defer this to our signaller
 */
public void signalMergedIntoEvent()
{
	transactionSignaller.signalMergedIntoEvent();
}
/**
 * Signal that we have been committed.  We are not part of a cascaded commit,
 * although we might be committing children transactions that are children of us
 */
public void signalRootHasBeenCommitted()
{
	transactionSignaller.signalRootHasBeenCommitted();
}
/**
 * Signal that we have been rolledback.
 * We are not part of a cascaded rollback, although we might have children that are 
 */
public void signalRootHasBeenRolledback()
{
	transactionSignaller.signalRootHasBeenRolledback();
}
/**
 * 
 */
public void supportDirtyReads( )
{
	this.setIsolationPolicy(TransactionIsolationPolicy.createDirtyReadPolicy());
}
/**
 * 
 */
public void supportRepeatableReads( )
{
	this.setIsolationPolicy(TransactionIsolationPolicy.createRepeatableReadPolicy());
}
/**
 * 
 */
public void supportUnrepeatableReads( )
{
	this.setIsolationPolicy(TransactionIsolationPolicy.createUnrepeatableReadPolicy());
}
/**
 * Suspend the transaction (if current, reset the current)
 */
public void suspend()
{
	Transaction.suspend(this);
}
/**
 * Suspend the current transaction
 */
protected static synchronized void suspend(Transaction aTransaction)
{
	if (getCurrent() == aTransaction)
		setCurrent(null);
}
/**
 * Synchronize the transaction's view to
 * external resources
 */
protected void synchronizeToResources() throws java.rmi.RemoteException
{
	this.getResourceManager().synchronize(this.getView());
}
/**
 * Test if the transaction's view can be merged to its parents view
 */
protected void testMergeToParent() throws VapMergeFailureException
{
	this.getView().testMergeToParent();
}
/**
 * Return string representation of the object
 */
public String toString()
{
	return getState().toString(this) ;
}
/**
 * Answer the result of performing aMethodName on anObject within this
 * transaction. Resume the current transaction when finished.
 */
public Object transactionDo(String aMethodName, Object anObject) {

	return transactionDo(aMethodName, anObject, null);
}
/**
 * Answer the result of performing aMethodName on anObject within this
 * transaction. Resume the current transaction when finished.
 */
public Object transactionDo(String aMethodName, Object anObject, Object[] args) {

	return transactionDo(aMethodName, anObject, args, null);
}
/**
 * Answer the result of performing aMethodName on anObject within this
 * transaction. Resume the current transaction when finished.
 */
public Object transactionDo(String aMethodName, Object anObject, Object[] args, Class[] argTypes) {

	Object result;
	
	Transaction current = Transaction.getCurrent();
	try {
		this.resume();
	} catch (VapTransactionRequiredException e) {
		try {
			current.resume();
		} catch (VapTransactionRequiredException ex) {}
		return null;
	}
		
	result = com.ibm.vap.common.VapMethodInvocation.performMethod(aMethodName, anObject, args, argTypes);
	try {
			current.resume();
	} catch (VapTransactionRequiredException e) {}
	
	return result;
}
}