/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is TransforMiiX XSLT processor.
 *
 * The Initial Developer of the Original Code is
 * IBM Corporation.
 * Portions created by the Initial Developer are Copyright (C) 2002
 * IBM Corporation. All Rights Reserved.
 *
 * Contributor(s):
 *   IBM Corporation
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "Expr.h"
#include "NodeSet.h"
#include "txIXPathContext.h"
#include "txAXPathWalkCallback.h"
#include "nsCOMArray.h"
#include "XMLDOMUtils.h"
#include "txAtoms.h"

txLocalNameFilter::txLocalNameFilter(nsAutoPtr<Expr> aExpr, nsAutoPtr<Expr> aLocalName)
    : mExpr(aExpr),
      mLocalName(aLocalName)
{
}

nsresult
txLocalNameFilter::evaluate(txIEvalContext* aContext, txAExprResult** aResult)
{
    *aResult = nsnull;

    nsRefPtr<txAExprResult> exprRes;
    nsresult rv = mExpr->evaluate(aContext, getter_AddRefs(exprRes));
    NS_ENSURE_SUCCESS(rv, rv);

    NS_ENSURE_TRUE(exprRes->getResultType() == txAExprResult::NODESET,
                   NS_ERROR_XSLT_NODESET_EXPECTED);

    nsRefPtr<NodeSet> nodes =
        NS_STATIC_CAST(NodeSet*, NS_STATIC_CAST(txAExprResult*, exprRes));

    if (nodes->isEmpty()) {
        *aResult = nodes;
        NS_ADDREF(*aResult);

        return NS_OK;
    }

    rv = mLocalName->evaluate(aContext, getter_AddRefs(exprRes));
    NS_ENSURE_SUCCESS(rv, rv);

    nsRefPtr<NodeSet> resultSet;
    rv = aContext->recycler()->getNodeSet(getter_AddRefs(resultSet));
    NS_ENSURE_SUCCESS(rv, rv);

    NodeSet* localNameNodes;
    short resType = exprRes->getResultType();
    if (resType == txAExprResult::NODESET &&
        (localNameNodes =
            NS_STATIC_CAST(NodeSet*,
                           NS_STATIC_CAST(txAExprResult*,
                                          exprRes)))->size() > 1) {
        nsCOMArray<nsIAtom> localNames;
        int i;
        PRInt32 nameCount = localNameNodes->size();
        for (i = 0; i < nameCount; ++i) {
            nsAutoString localNameStr;
            XMLDOMUtils::getNodeValue(localNameNodes->get(i), localNameStr);
            nsCOMPtr<nsIAtom> localName = do_GetAtom(localNameStr);
            if (!localNames.AppendObject(localName)) {
                return NS_ERROR_OUT_OF_MEMORY;
            }
        }

        for (i = 0; i < nodes->size(); ++i) {
            nsCOMPtr<nsIAtom> nodeName;
            nodes->get(i)->getLocalName(getter_AddRefs(nodeName));
            int j;
            for (j = 0; j < nameCount; ++j) {
                if (localNames[j] == nodeName) {
                    rv = resultSet->append(nodes->get(i));
                    NS_ENSURE_SUCCESS(rv, rv);

                    break;
                }
            }
        }
    }
    else if (resType == txAExprResult::BOOLEAN) {
        // Ideally we shouldn't have created a txLocalNameFilter here, but our
        // ability to optimize variables is still too weak.
        // Also this is very unlikly to happen, comparing a localname to a bool
        // is pretty useless.
        
        PRBool res = exprRes->booleanValue();
        int i;
        for (i = 0; i < nodes->size(); ++i) {
            nsCOMPtr<nsIAtom> nodeName;
            nodes->get(i)->getLocalName(getter_AddRefs(nodeName));
            if (res && nodeName && nodeName != txXMLAtoms::_empty ||
                !res && (!nodeName || nodeName == txXMLAtoms::_empty)) {
                rv = resultSet->append(nodes->get(i));
                NS_ENSURE_SUCCESS(rv, rv);
            }
        }
    }
    else if (resType != txAExprResult::NUMBER) {
        // Having to compare to only one name is a common case so we'll have a
        // separate loop for that
        nsAutoString localNameStr;
        exprRes->stringValue(localNameStr);
        nsCOMPtr<nsIAtom> localName = do_GetAtom(localNameStr);

        int i;
        for (i = 0; i < nodes->size(); ++i) {
            nsCOMPtr<nsIAtom> nodeName;
            nodes->get(i)->getLocalName(getter_AddRefs(nodeName));
            if (localName == nodeName) {
                rv = resultSet->append(nodes->get(i));
                NS_ENSURE_SUCCESS(rv, rv);
            }
        }
    }

    *aResult = resultSet;
    NS_ADDREF(*aResult);

    return NS_OK;
}

void
txLocalNameFilter::toString(nsAString& aDest)
{
    mExpr->toString(aDest);
    aDest.Append(NS_LITERAL_STRING("[local-name() = "));
    mLocalName->toString(aDest);
    aDest.Append(PRUnichar(']'));
}

nsresult
txLocalNameFilter::iterateSubItems(txAXPathWalkCallback* aCallback)
{
    nsresult rv = aCallback->walkedExpr(mExpr, NUMBER_RESULT);
    NS_ENSURE_SUCCESS(rv, rv);

    return aCallback->walkedExpr(mLocalName, NUMBER_RESULT);
}

TX_IMPL_EXPR_STUBS(txLocalNameFilter, LOCALNAMEFILTER_EXPR, NODESET_RESULT)

Expr::ContextSensitivity
txLocalNameFilter::getContextSensitivity()
{
    return mExpr->getContextSensitivity() |
           mLocalName->getContextSensitivity();
}
