typedef unsigned char u_char;
typedef unsigned short u_short;
typedef unsigned long u_long;

/* Double is an 8-byte struct superimposed on a IEEE double */
typedef struct {u_char c1, c2, c3, c4, c5, c6, c7, c8;} Double;
typedef struct {u_char c5, c6, c7, c8;} Float;

/* Idouble is the internal format used for arithmetic.
 * M1 and M2 each contain 32 bits of the mantissa, M1 less significant.
 * The value is (-1)^sign * M2M1 * 2^expt.
 * Idouble2 is similar, but with the mantissa in four 16-bit parts.
 */
typedef struct {u_long M1, M2; short exp, sign;} Idouble;
typedef struct {u_short m[4]; short exp, sign;} Idouble2;

extern Double __add(), __subtract(), __multiply(), __divide();
extern Double __long2double(), __frexp(), __ldexp(), __modf();
extern Double __float2double();
extern Float __double2float();
extern long __double2long(), __compare();

/* static */ Double __pack_double();
/* static */ Idouble __unpack_double();
/* static */ long __Icompare();

Double __Zero          = {0, 0, 0, 0, 0, 0,    0,    0},
       __PlusInfinity  = {0, 0, 0, 0, 0, 0, 0xf0, 0x7f},
       __MinusInfinity = {0, 0, 0, 0, 0, 0, 0xf0, 0xff},
       __One           = {0, 0, 0, 0, 0, 0, 0xf0, 0x3f},
       __MinusOne      = {0, 0, 0, 0, 0, 0, 0xf0, 0xbf};

Float __SZero          = {0, 0,    0,    0},
      __SPlusInfinity  = {0, 0, 0x80, 0x7f},
      __SMinusInfinity = {0, 0, 0x80, 0xff};

#define IsZero(d) ((d).c8 == 0 && (d).c7 == 0)
#define IsInf(d)  (((d).c8 & 0x7f) == 0x7f && (d).c7 == 0xf0)
#define SIsInf(s) (((s).c8 & 0x7f) == 0x7f && (s).c7 == 0x80)
#define Sign(d)   (((d).c8 & 0x80) != 0)

Double __long2double(i)
long i;
{
	Idouble f;

	if(i == 0)
		return __Zero;

	f.exp = 0;
	f.sign = (i < 0);
	f.M1 = f.sign ? -i : i;		/* This works for -2^31 probably */
	f.M2 = 0;

	return __pack_double(f);
}

long __double2long(d)
Double d;
{
	Idouble f;
	long l;

	if(IsZero(d))
		return 0;
	if(IsInf(d))
		return Sign(d) ? 0x80000000 : 0x7fffffff;

	f = __unpack_double(d);

	/* The mantissa of f is now between 2^52 and 2^53-1.
	   Valid positive longs are between 0 and 2^31-1. */

	if(f.exp < -52)
		return 0;
	if(f.exp > -22)		/* this also covers -2^32 */
		return f.sign ? 0x80000000 : 0x7fffffff;

	l = ((f.M2 << 10) | (f.M1 >> 22));
	l >>= (-22 - f.exp);

	return f.sign ? -l : l;
}

/* static */ Double __pack_double(f)
Idouble f;
{
	Double d;
	int i;
	u_long t;

	if(f.M1 == 0 && f.M2 == 0)
		return __Zero;

	f.exp += 63;	/* move point to after most significant bit */

	/* We adjust f until its most significant bit is 1 */
	if(f.M2 == 0)
	{
		f.exp -= 32;
		f.M2 = f.M1;
		f.M1 = 0;
	}
	for(i=0; (f.M2 & 0x80000000) == 0; i++)
		f.M2 <<= 1;
	if(i > 0)
	{
		t = f.M1;
		t >>= (32 - i);		/* unsigned => logical shift */
		f.M1 <<= i;
		f.M2 |= t;
		f.exp -= i;
	}

	if(f.exp < -1022)
		return __Zero;		/* No denormals */
	if(f.exp > 1023)
		return f.sign ? __MinusInfinity : __PlusInfinity;
	f.exp += 1023;

	/* Truncate to 52 (+1 implicit) bits of mantissa */

	d.c8 = f.exp >> 4;
	if(f.sign)
		d.c8 |= 0x80;
	d.c7 =  (f.exp & 0x0f) << 4;
	d.c7 |= ((f.M2 & 0x78000000) >> 27);
	d.c6 =  ((f.M2 & 0x07f80000) >> 19);
	d.c5 =  ((f.M2 & 0x0007f800) >> 11);
	d.c4 =  ((f.M2 & 0x000007f8) >>  3);
	d.c3 =  ((f.M2 & 0x00000007) <<  5);
	d.c3 |= ((f.M1 & 0xf8000000) >> 27);
	d.c2 =  ((f.M1 & 0x07f80000) >> 19);
	d.c1 =  ((f.M1 & 0x0007f800) >> 11);

	return d;
}

/* Unpack_double must not be called with zero or infinity */

Idouble __unpack_double(d)
Double d;
{
	Idouble f;

	*(Double *)&f.M1 = d;
	f.M2 &= 0x0fffff;	/* Clear non-mantissa bits */
	f.M2 |= 0x100000;	/* Add the hidden bit */

	f.sign = (d.c8 & 0x80) != 0;

	f.exp = ((d.c8 & 0x7f) << 4) + (d.c7 >> 4);
	f.exp -= 1023;
	f.exp -= 52;		/* Move point to after least sig digit */

	return f;
}

/* static */ long __Icompare(a, b) /* Compare magnitude of internal-format doubles */
Idouble a, b;
{
	if(a.exp > b.exp)
		return 1;
	if(a.exp < b.exp)
		return -1;
	if(a.M2 > b.M2)
		return 1;
	if(a.M2 < b.M2)
		return -1;
	if(a.M1 > b.M1)
		return 1;
	if(a.M1 < b.M1)
		return -1;
	return 0;	
}

long __compare(a, b)
Double a, b;
{
	long c;
	
	if(Sign(a) != Sign(b))
		return Sign(a) ? -1 : 1;

	c = __Icompare(__unpack_double(a), __unpack_double(b));

	return Sign(a) ? -c : c;
}

Double __subtract(a, b)
Double a, b;
{
	/* Negate b and add. */
	((long *)&b)[1] ^= 0x80000000;
	return __add(a, b);
}

Double __add(a, b)
Double a, b;
{
	Idouble aa, bb, cc;
	int subtract, negate, shift;
	u_long t;

	if(IsZero(a))
		return b;
	if(IsZero(b))
		return a;

	if(IsInf(a))
		return a;	/* No NaNs so +inf + -inf = +inf for now */
	if(IsInf(b))
		return b;

	aa = __unpack_double(a);
	bb = __unpack_double(b);

	subtract = (aa.sign != bb.sign);

	if(__Icompare(aa, bb) == -1)
	{
		/* |a| < |b| so swap them */
		cc = aa;
		aa = bb;
		bb = cc;
	}
	
	negate = aa.sign;

	/* We now have |a| >= |b|.  Now shift b so that it lines up with a. */

	shift = aa.exp - bb.exp;

	if(shift >= 53)
		goto done;	/* b is much smaller - the result is a */

	if(shift >= 32)
	{
		bb.M1 = bb.M2;
		bb.M2 = 0;
		shift -= 32;
	}

	if(shift > 0)
	{
		bb.M1 >>= shift;
		bb.M1 |= bb.M2 << (32 - shift);
		bb.M2 >>= shift;
	}

	/* They're lined up.  Now add or subtract. */
	if(subtract)
	{
		if(aa.M1 < bb.M1)
			aa.M2--;
		aa.M1 -= bb.M1;
		aa.M2 -= bb.M2;
	}
	else
	{
		t = aa.M1;
		aa.M1 += bb.M1;
		if(t > aa.M1)		/* carry if the result is less */
			aa.M2++;	/* than either of the operands */
		aa.M2 += bb.M2;
	}		

    done:
	aa.sign = negate;
	return __pack_double(aa);
}

Double __multiply(a, b)
Double a, b;
{
	Idouble2 aa, bb;
	u_long t, c[8];
	register int i, j;

	if(IsZero(a) || IsZero(b))
		return __Zero;		/* No NaNs so 0 * inf = 0 */
	if(IsInf(a) || IsInf(b))
		return Sign(a) == Sign(b) ? __PlusInfinity : __MinusInfinity;

	*(Idouble *)&aa = __unpack_double(a);
	*(Idouble *)&bb = __unpack_double(b);

	for(i=0; i<8; i++)
		c[i] = 0;
	for(i=0; i<4; i++)
	    for(j=0; j<4; j++)
	    {
		t = (u_long)aa.m[i] * (u_long)bb.m[j];
		c[i+j] += (t & 0xffff);
		c[i+j+1] += (t >> 16);
	    }
	for(i=0; i<7; i++)
		c[i+1] += (c[i] >> 16);

	/* aa.m[3] and bb.m[3] were between 0x10 and 0x1f.  So
	 * c[6] is between 0x100 and 0x3ff.  So we don't lose
	 * any precision by using c[6]-c[3] (I think).      
	 */

	for(i=0; i<4; i++)
		aa.m[i] = (u_short)c[i+3];

	aa.exp += bb.exp;
	aa.exp += 48;		/* the zeroes from c[2]-c[0] */
	aa.sign ^= bb.sign;

	return __pack_double(aa);
}

Double __divide(a, b)
Double a, b;
{
	Idouble aa, bb;
	u_long x[4], y[4];
	int i, j;

	if(IsZero(a))		/* No NaNs so 0 / 0 = 0 */
		return __Zero;
	if(IsInf(a))		/* and inf / inf = inf */
		return Sign(a) == Sign(b) ? __PlusInfinity : __MinusInfinity;
	if(IsZero(b))
		return Sign(a) == Sign(b) ? __PlusInfinity : __MinusInfinity;

	aa = __unpack_double(a);
	bb = __unpack_double(b);

	x[0] = x[1] = y[0] = y[1] = 0;
	x[2] = aa.M1; x[3] = aa.M2;
	y[2] = bb.M1; y[3] = bb.M2;

	aa.M1 = aa.M2 = 0;

	for(i=0; i<53; i++)
	{
		int bit = 0;
/*
		printf("%08x %08x %08x %08x\n", x[3], x[2], x[1], x[0]);
		printf("%08x %08x %08x %08x\n", y[3], y[2], y[1], y[0]);
*/
		/* Can we subtract y from x? */
		for(j=3; j>=0; j--)
		{
			if(x[j] > y[j])
				break;		/* yes */
			if(x[j] < y[j])
				goto shift;	/* no */
		}

/*		printf("subtract ... ");		*/
		/* Yes, do it. */
		bit = 1;
		for(j=0; j<4; j++)
		{
			if(x[j] < y[j])
				x[j+1]--;
			x[j] -= y[j];
		}

	    shift:
/*		printf("shift\n"); */
		/* Shift the divisor down one */
		for(j=0; j<3; j++)
		{
			y[j] >>= 1;
			if(y[j+1] & 1)
				y[j] |= 0x80000000;
		}
		y[3] >>= 1;

		/* Shift the quotient up one */
		aa.M2 <<= 1;
		if(aa.M1 & 0x80000000)
			aa.M2 |= 1;
		aa.M1 <<= 1;
		aa.M1 |= bit;
	}

	aa.exp -= bb.exp;
	aa.exp -= 52;		/* point should be after first bit */
	aa.sign ^= bb.sign;

	return __pack_double(aa);
}

Double __frexp(value, eptr)
Double value;
int *eptr;
{
	Idouble f;

	if(IsZero(value) || IsInf(value))
	{
		*eptr = 0;
		return value;
	}

	f = __unpack_double(value);
	*eptr = f.exp + 52 + 1;
	f.exp = -53;
	return __pack_double(f);
}

Double __ldexp(value, exp)
Double value;
int exp;
{
	Idouble f;

	if(IsZero(value) || IsInf(value))
		return value;

	f = __unpack_double(value);
	f.exp += exp;
	return __pack_double(f);
}

Double __modf(value, iptr)
Double value, *iptr;
{
    Idouble f;
    Double ipart, fpart;
    int fdigits;

    if(IsZero(value) || IsInf(value))
    {
	*iptr = value;
	return __Zero;
    }

    f = __unpack_double(value);

    /* The mantissa of f is now between 2^52 and 2^53-1 */

    if(f.exp >= 0)
    {
	/* The number is too big to have a fractional part */
	*iptr = value;
	return __Zero;
    }

    if(f.exp <= -53)
    {
	/* The integer part is zero (or -1 if it's negative) */
	*iptr = f.sign ? __MinusOne : __Zero;
	return f.sign ? __add(value, __One) : value;
    }

    /* Zero out the (-f.exp) fractional digits (between 1 and 52 of them) */
    fdigits = -f.exp;
    if(fdigits >= 32)
    {
	f.M1 = 0;
	f.M2 &= ((unsigned long)0xffffffff << (fdigits - 32));
    }
    else
	f.M1 &= ((unsigned long)0xffffffff << fdigits);

    ipart = __pack_double(f);
    fpart = __subtract(value, ipart);

    if(f.sign && !IsZero(fpart))
    {				/* make the fractional part positive */
	ipart = __add(ipart, __MinusOne);
	fpart = __add(fpart, __One);
    }
	
    *iptr = ipart;
    return fpart;
}

Double __float2double(s)
Float s;
{
    int exp;
    Double f;

    if(SIsInf(s))
	return Sign(s) ? __MinusInfinity : __PlusInfinity;
    if(IsZero(s))
	return __Zero;

    exp = ((s.c8 & 0x7f) << 1) + (s.c7 >> 7);
    exp += (1023 - 127);

    f.c8 = (exp >> 4);
    if(Sign(s))
	f.c8 |= 0x80;
    
    f.c7 = ((exp & 0x0f) << 4) | ((s.c7 & 0x78) >> 3);
    f.c6 = ((s.c7 & 0x07) << 5) | (s.c6 >> 3);
    f.c5 = ((s.c6 & 0x07) << 5) | (s.c5 >> 3);
    f.c4 = ((s.c5 & 0x07) << 5);
    f.c3 = f.c2 = f.c1 = 0;
    
    return f;
}

Float __double2float(f)
Double f;
{
    int exp;
    Float s;

    if(IsInf(f))
	return Sign(f) ? __SMinusInfinity : __SPlusInfinity;
    if(IsZero(f))
	return __SZero;

    exp = ((f.c8 & 0x7f) << 4) + (f.c7 >> 4);
    exp += (127 - 1023);

    if(exp > 254 || exp < 1)	/* too big for a single float? */
	return Sign(f) ? __SMinusInfinity : __SPlusInfinity;

    s.c8 = (exp >> 1);
    if(Sign(s))
	s.c8 |= 0x80;
    
    s.c7 = ((exp & 0x01) << 7) | ((f.c7 & 0x0f) << 3) | (f.c6 >> 5);
    s.c6 = ((f.c6 & 0x1f) << 3) | (f.c5 >> 5);
    s.c5 = ((f.c5 & 0x1f) << 3) | (f.c4 >> 5);
    
    return s;
}

#ifdef test

void show(d)
Double d;
{
	printf("%s %d %08x_%08x", 
	       Sign(d) ? "-" : "+",
	       ((((u_long *)&d)[1] >> 20) & 2047),
	       (((u_long *)&d)[1] & 0xfffff), ((u_long *)&d)[0]);
}

Double atof(s)
char *s;
{
	Double f, ten, zero;
	int neg=0, dp=0;

	f = zero = long2double(0);
	ten = long2double(10);

	if(*s == '-')
		neg=1, s++;
	else if(*s == '+')
		s++;

	while(*s >= '0' && *s <= '9')
		f = add(long2double(*s++ - '0'), multiply(f, ten));
	if(*s++ != '.')
		return neg ? subtract(zero, f) : f;
	while(*s >= '0' && *s <= '9')
		f = add(long2double(*s++ - '0'), multiply(f, ten)), dp++;
	while(dp-- > 0)
		f = divide(f, ten);
	return neg ? subtract(zero, f) : f;
}

char *ftoa(f)
Double f;
{
	static char buf[30];
	int i;
	long l, m;

	m = l = double2long(add(multiply(f, long2double(1000)),
				divide(long2double(1), long2double(2))));
	l = l<0 ? -l : l;
	sprintf(buf, "%c%d.%03d", m<0 ? '-' : '+', l / 1000, l % 1000);

	return buf;
}

main(argc, argv)
int argc;
char **argv;
{
	Double d, e, f;

	d = atof(argv[1]);
	e = atof(argv[2]);

	printf("divide:\n");

	f = divide(d, e);
	
	printf("%s / ", ftoa(d));
	printf("%s = ", ftoa(e));
	printf("%s\n",  ftoa(f));

	f = add(d, e);
	
	printf("%s + ", ftoa(d));
	printf("%s = ", ftoa(e));
	printf("%s\n",  ftoa(f));
}

#endif
