	TITLE	'CP/M-85 BIOS LOADER (8085 CODE PORTION)     30 JUN 82'
;***
;
;	THIS IS THE BIOS LOADER FOR CP/M-85.
;	IT IS THE BIOS ON THE SYSTEM TRACK.
;	IT
;	  1. READS IN THE 8088 CODED PORTION OF THE REAL BIOS
;	  2. READS IN THE 8085 CODED PORTION OF THE REAL BIOS
;	  3. READS IN CHARACTER SET CHANGES/MAPPINGS
;	  4. JUMPS TO THE RELOCATOR WHICH RELOCATES THE 8085 CODED PORTION
;	     OF THE REAL BIOS
;	  5. THE RELOCATOR THEN TRANSFERS CONTROL TO THE 8085 CODED PORTION
;	     OF THE REAL BIOS AT THE COLD BOOT ENTRY POINT
;	ASSUMPTIONS
;	  1. INTERRUPTS ARE DISABLED
;	  2. BOOT DRIVE IS SELECTED AND MOTOR IS RUNNING
;

TRUE	EQU	1
FALSE	EQU	0

ASM86	EQU	FALSE
REL0	EQU	TRUE
WAIT	EQU	TRUE

;***	MACROS


;**	EXTRA PSEUDO OPS FOR USE WITH MAC
;

RB	MACRO	A
	DS	A
	ENDM

RW	MACRO	A
	DS	2*A
	ENDM


;***	EQUATES


;**	Z-MACHINE MEMORY MAP UTILIZATION EQUATES
;

BOOTORG	EQU	0400H			;BOOT CODE LOAD ADDRESS
BCODEL	EQU	256+32			;MAXIMUM BOOT CODE LENGTH

BLDR88	EQU	BOOTORG+BCODEL		;BIOS LOADER (8088 PORTION)
BLDR88L	EQU	64			;MAXIMUM BIOS LOADER (8088) LENGTH

B88STK	EQU	2*1024			;8088 STACK AREA

B88DAT	EQU	B88STK			;MONITOR ROM DATA AREA START ADDR

BIOS88	EQU	4*1024			;WHERE 8088 CODE OF BIOS WILL RESIDE

BLDRP0	EQU	16*1024			;BIOS LOADER PAGE ZERO BIAS VALUE

	IF	REL0
BLOADER	EQU	0000H			;WHERE BOOT CODE LOADS BIOS LOADER
	ENDIF
	IF	NOT REL0
BLOADER	EQU	0100H			;WHERE BOOT CODE LOADS BIOS LOADER
	ENDIF
BLDR85L	EQU	8*128-BCODEL-BLDR88L	;MAXIMUM BIOS LOADER (8085) LENGTH

BOOT85	EQU	0000H			;WHERE 8085 STARTS AFTER PROCESSOR SWAP

SWISWAP	EQU	0FEH			;SWI FOR SWAPPING PROCESSORS
SWAPVEC	EQU	SWISWAP*4		; VECTOR ADDRESS


BIAS	EQU	BLDRP0
;**	CP/M-85 SYSTEM EQUATES
;

;	OS VALUES

CCPL	EQU	0800H			;LENGTH OF CCP
BDOSL	EQU	0E00H			;LENGTH OF BDOS

;	RESERVED LOCATIONS IN PAGE ZERO

	ORG	BIAS
BOOT	RB	3			;JUMP TO WARM START ENTRY
IOBYTE	RB	1			;IOBYTE
DFTDRV	RB	1			;CURRENT EFAULT DRIVE
BDOS	RB	3			;JUMP TO BDOS
INT1	RB	8			;INTERRUPT LOCATION 1
INT2	RB	8			;INTERRUPT LOCATION 2
INT3	RB	8			;INTERRUPT LOCATION 3
INT4	RB	8			;INTERRUPT LOCATION 4
INT5	RB	8			;INTERRUPT LOCATION 5
INT6	RB	8			;INTERRUPT LOCATION 6 (RESERVED)
INT7	RB	8			;INTERRUPT LOCATION 7 (USED BY DDT)
BSCR	RB	16			;16 BYTE SCRATCH AREA FOR BIOS
	RB	12			;12 BYTES NOT USED BUT RESERVED
TFCB	RB	36			;TRANSIENT DEFAULT FCB
TFCB2	EQU	TFCB+16			;TRANSIENT 2ND FILENAME
TBUFF	RB	128			;DEFAULT DISK BUFFER
TPA	RB	0			;START OF TRANSIENT PROGRAM AREA

;	HEATH PAGE ZERO LOCATIONS

	IF	ASM86
	ORG	OFFSET INT1+3
	ENDIF
	IF	NOT ASM86
	ORG	INT1
	ENDIF
TICCNT	RW	2			;32 BIT TIMER TIC COUNTER
TIMEFLG	RB	1			;TIMER 2 FLAG (0=RUNNING , 1=TIME OUT)

	IF	ASM86
	ORG	OFFSET INT2
	ENDIF
	IF	NOT ASM86
	ORG	INT2
	ENDIF
COMREG	RB	0			;BIOS88 COMMUNICATION REGION
COMFUNC	RB	1			; BIOS FUNCTION VALUE = (#-CBOOT)/3
COMRA	RB	1			; (A)
COMRBC	RW	0			; (BC)
COMRC	RB	1			;  (C)
COMRB	RB	1			;  (B)
COMRDE	RW	0			; (DE)
COMRE	RB	1			;  (E)
COMRD	RB	1			;  (D)
COMRHL	RW	0			; (HL)
COMRL	RB	1			;  (L)
COMRH	RB	1			;  (H)
COMWHO	RB	1			;WHICH PROCESSOR IS RUNNING
					; ZPSPPS5 = 8085
					; ZPSPPS8 = 8088

BDMAP	EQU	BSCR+00H		;DRIVE MAP (MAX OF 8 ENTRIES 0-7)
	IF	ASM86
BBIOS	EQU	WORD PTR BSCR+0EH	;TRUE ADDRESS FOR START OF BIOS
	ENDIF
	IF	NOT ASM86
BBIOS	EQU	BSCR+0EH		;TRUE ADDRESS FOR START OF BIOS
	ENDIF

;	CP/M FUNCTIONS

RESET	EQU	0			;SYSTEM RESET
RDCON	EQU	1			;READ CONSOLE
WRCON	EQU	2			;WRITE CONSOLE
RDRDR	EQU	3			;READ READER
WRPUN	EQU	4			;WRITE PUNCH
WRLST	EQU	5			;WRITE LIST
DCONIO	EQU	6			;DIRECT CONSOLE I/O (2)
IOSTAT	EQU	7			;INTERROGATE I/O STATUS
ASTAT	EQU	8			;ALTER I/O STATUS
PRCONB	EQU	9			;PRINT CONSOLE BUFFER
RDCONB	EQU	10			;READ CONSOLE BUFFER
CCSTAT	EQU	11			;CHECK CONSOLE STATUS
LDH	EQU	12			;LIFT DISK HEAD
CPMVER	EQU	LDH			;GET CP/M VERSION (2)
RDS	EQU	13			;RESET DISK SYSTEM
SELDSK	EQU	14			;SELECT DISK
OPEN	EQU	15			;OPEN FILE
CLOSE	EQU	16			;CLOSE FILE
S1ST	EQU	17			;SEARCH FIRST
SNXT	EQU	18			;SEARCH NEXT
DELETE	EQU	19			;DELETE FILE
READ	EQU	20			;READ RECORD
WRITE	EQU	21			;WRITE RECORD
CREATE	EQU	22			;CREATE FILE
RENAME	EQU	23			;RENAME FILE
ILOG	EQU	24			;INTERROGATE LOGIN
IDSK	EQU	25			;INTERROGATE DISK
SETDMA	EQU	26			;SET DMA ADDRESS
IALLOC	EQU	27			;INTERROGATE ALLOCATION
WPD	EQU	28			;WRITE PROTECT DISK
GROV	EQU	29			;GET READ/ONLY VECTOR
SFA	EQU	30			;SET FILE ATTRIBUTES
GADPB	EQU	31			;GET ADDR OF DPB
SGUSR	EQU	32			;SET/GET USER CODE
READRR	EQU	33			;READ RANDOM RECORD
WRITERR	EQU	34			;WRITE RANDOM RECORD
CFSIZE	EQU	35			;COMPUTE FILE SIZE
SRR	EQU	36			;SET RANDOM RECORD
RESDRV	EQU	37			;RESET DRIVE
WRITERZ	EQU	40			;WRITE RANDOM WITH ZERO FILL

;	BIOS ENTRY POINTS.
;	  GIVEN AS DISPLACEMENTS FROM WARM BOOT ENTRY POINT WHOSE
;	  ADDRESS IS USUALLY AT LOCATION BOOT+1.
;	  TO COMPUTE DISPLACEMENTS FROM VALUE STORED AT 'BBIOS'
;	  SUBTRACT 'CBOOT'.  EG. DISPLACEMENT FOR WARM BOOT = WBOOT-CBOOT

CBOOT	EQU	-0003H			;COLD BOOT
WBOOT	EQU	0000H			;WARM BOOT
CONST	EQU	0003H			;CONSOLE STATUS
CONIN	EQU	0006H			;CONSOLE INPUT
CONOUT	EQU	0009H			;CONSOLE OUTPUT
LSTOUT	EQU	000CH			;LIST OUTPUT
PUNOUT	EQU	000FH			;PUNCH OUTPUT
RDRIN	EQU	0012H			;READER INPUT
HOME	EQU	0015H			;HOME DISK
SETDSK	EQU	0018H			;SET (SELECT) DISK DRIVE
SETTRK	EQU	001BH			;SET TRACK NUMBER
SETSEC	EQU	001EH			;SET SECTOR NUMBER
SDMA	EQU	0021H			;SET DMA ADDRESS
BREAD	EQU	0024H			;READ SELECTED SECTOR
BWRITE	EQU	0027H			;WRITE SELECTED SECTOR
BLSTST	EQU	002AH			;CHECK LIST DEVICE STATUS
BSECTRN	EQU	002DH			;SECTOR TRANSLATE ROUTINE
BFMT	EQU	0030H			;FORMAT
BRDTRK	EQU	0033H			;READ TRACK
BWRTRK	EQU	0036H			;WRITE TRACK
BWPC	EQU	0039H			;WRITE PROTECT CHECK

;	BIOS WRITE TYPES

BWRNOR	EQU	0			;NORMAL WRITE
BWRDIR	EQU	1			;WRITE TO A DIRECTORY SECTOR
BRWUA1	EQU	2			;WRITE TO 1ST SECTOR OF UNALLOC BLOCK

;	FILE CONTROL BLOCK

	ORG	0
FCBDN	RB	1			;DISK NAME
FCBFN	RB	8			;FILE NAME
FCBFNL	EQU	8			;FILE NAME LENGTH
FCBFT	RB	3			;FILE TYPE
FCBFTL	EQU	3			;FILE TYPE LENGTH
FCBRO	EQU	FCBFT+0			;R/O FLAG
FCBROF	EQU	10000000B		; R/O FLAG VALUE
FCBSYS	EQU	FCBFT+1			;SYS FLAG
FCBSYSF	EQU	10000000B		; SYS FLAG VALUE
FCBEX	RB	1			;EXTENT
	RB	1
FCBRWF	RB	1			;R/W FLAG
FCBRWFF EQU	10000000B		; R/W FLAG VALUE
FCBRC	RB	1			;RECORD COUNT
FCBDM	RB	16			;DISK ALLOCATION MAP
FCBNR	RB	1			;NEXT RECORD TO BE READ OR WRITTEN
FCBLS	EQU	33			;FCB LENGTH FOR SEQUENTIAL I/O
FCBRR	RB	3			;RANDOM RECORD POINTER
FCBLR	EQU	36			;FCB LENGTH FOR RANDOM I/O



;**	DISK PARAMETER TABLE EQUATES
;

;	DISK PARAMETER ENTRY DESCRIPTION

	ORG	0
DPEXLT	RW	1		;SECTOR TRANSLATE TABLE ADDRESS
	RW	3
DPEDIRB	RW	1		;DIRECTORY BUFFER ADDRESS
DPEDPB	RW	1		;DISK PARAMETER BLOCK ADDRESS
DPECSV	RW	1		;CHECKSUM VECTOR ADDRESS
DPEALV	RW	1		;ALLOCATION VECTOR ADDRESS
DPEHTH	RB	8		;HEATH EXTENSIONS
DPEL	EQU	24		;LENGTH OF DISK PARAMETER ENTRY

;	HEATH EXTENSIONS

DPEFLAG	EQU	DPEHTH+0	;FLAGS
DPETYPE	EQU	11100000B	;BIT 7-5 = DEVICE TYPE
DPENE	EQU	00000000B	; NON-EXISTENT
DPEZ207	EQU	00100000B	; Z207
DPEP7C	EQU	00010000B	;BIT 4 -- 0=BASE PORT 78H  1=PORT 7CH
DPE48RO	EQU	00010000B	;BIT 4 -- FOR H37
				;  48 TPI MEDIA IN 96 TPI DRIVE (R/O)
DPE96T	EQU	00001000B	;BIT 3 -- 0=48 TPI DRIVE  1=96 TPI DRIVE
DPEASGN	EQU	00000100B	;BIT 2 -- FOR H67 HARD DISK
				;         0=UNASSIGNED A PARTITION
				;         1=ASSIGNED A PARTITION
DPET0SD	EQU	00000100B	;BIT 2 -- 1=TRACK 0 IS SINGLE DENSITY
DPEDD	EQU	00000010B	;BIT 1 -- 0=SINGLE DENSITY  1=DOUBLE
DPE2S	EQU	00000001B	;BIT 0 -- 0=SINGLE SIDED  1=DOUBLE

DPEUNIT	EQU	DPEHTH+1	;UNIT SELECT VALUE
DPERPS	EQU	DPEHTH+2	;CP/M RECORDS PER PHYSICAL SECTOR
DPERPAB	EQU	DPEHTH+3	;CP/M RECORDS PER ALLOCATION BLOCK
DPETRK	EQU	DPEHTH+4	;TRACK COUNTER
DPEUNK	EQU	10000000B	; TRACK POSITION UNKNOWN
DPESEK	EQU	DPEHTH+5	;MOTOR SPEED AND SEEK SPEED
				;BIT 3-0 = SEEK SPEED VALUE
DPEFS	EQU	01000000B	;BIT 6 = FAST STEP FOR Z207
DPEMO	EQU	10000000B	;BIT 7 = MOTOR UP TO SPEED FLAG
				;  0=1 SEC  1=250 MSEC
DPEFLG2	EQU	DPEHTH+6	;2ND FLAG BYTE
DPEHLS	EQU	00000100B	;BIT 2  DRIVE HAS HEAD LOAD SELONOID
DPEIMG	EQU	00000010B	;BIT 1  IMAGINARY DRIVE
DPE96TM	EQU	00000001B	;BIT 0  0=48 TPI MEDIA  1=96 TPI MEDIA
DPELUN	EQU	DPEHTH+7	;LAST LOGICAL UNIT MOUNTED
DPELOG	EQU	11110000B	; CP/M LOGICAL DRIVE NAME FOR THIS ENTRY
DPEREAL	EQU	00001111B	; FOR IMAGINARY DRIVE, LOGICAL DRIVE NAME
				;  FOR CORRESPONDING REAL DRIVE
DPEMNT	EQU	00001111B	; FOR REAL DRIVE, LOGICAL DRIVE NAME FOR
				;  CURRENLY MOUNTED DISK

DPEHL	EQU	8		;LENGTH OF HEATH EXTENSION

;	DISK PARAMETER BLOCK

	ORG	0
DPBSPT	RW	1		;SECTORS PER TRACK
DPBBSH	RB	1		;BLOCK SHIFT FACTOR
DPBBLM	RB	1		;BLOCK MASK
DPBEXM	RB	1		;EXTENT MASK
DPBDSM	RW	1		;TOTAL # OF BLOCKS - 1
DPBDRM	RW	1		;# OF DIRECTORY ENTRIES - 1
DPBAL0	RB	1		;INITIAL AL0 VALUE
DPBAL1	RB	1		;INITIAL AL1 VALUE
DPBCKS	RW	1		;SIZE OF DIRECTORY CHECK VECTOR
DPBOFF	RW	1		;NUMBER OF SYSTEM TRACKS
DPBL	EQU	15		;LENGTH OF DISK PARAMETER BLOCK



;**	HEATH CP/M DISK LABEL DEFINITIONS
;
;  DISK LABELS ARE USED ON MOST DATA DISKS AND
;    ON ALL SYSGEN'ED (BOOTABLE) DISKS.
;  TO MAINTAIN COMPATIBILITY WITH OLDER RELEASES OF CP/M,
;    THE 5 1/4" HARD SECTOR AND 8" SOFT SECTOR FLOPPY DATA DISKS
;    DO NOT USE LABELS.
;
;  THE LABEL RESIDES ON THE 1ST SECTOR OF TRACK 0, SIDE 0.
;
;  AT THE END OF THE LABEL IS A CHECKSUM.
;    FOR VERSION # 0 OF THE LABEL, THE CHECKSUM IS CALCULATED
;      BY ADDING UP THE VALUES IN THE LABEL PRIOR TO THE CHECKSUM SLOT
;      A BYTE AT A TIME, THEN TAKING THE ONE'S COMPLEMENT OF THE SUM.
;

LABVER	EQU	0		;CURRENT FORM # FOR LABEL

LABBUF	EQU	0		;SLOT FOR JUMP INSTRUCTION AROUND LABEL
BDTYPE	EQU	LABBUF+3	;SLOT FOR DRIVE TYPE

LABEL	EQU	LABBUF+4
LABTYP	EQU	LABEL+0		;SLOT FOR LABEL TYPE
LABHTH	EQU	LABTYP+1	;SLOT FOR HEATH EXTENSIONS TO DPE
LABDPB	EQU	LABHTH+DPEHL	;SLOT FOR DISK PARAMETER BLOCK
LABCS	EQU	LABDPB+DPBL	;CHECKSUM

LABLEN	EQU	LABCS-LABEL+1	;LABEL LENGTH



;**	Z207 EQUATES
;

;	PORT ASSIGNMENTS

FDBASE	EQU	0B0H		;BASE PORT ADDRESS
FDCMD	EQU	FDBASE		;1797 COMMAND REGISTER
FDSTA	EQU	FDBASE		;     STATUS REGISTER
FDTRK	EQU	FDBASE+1	;     TRACK REGISTER
FDSEC	EQU	FDBASE+2	;     SECTOR REGISTER
FDDAT	EQU	FDBASE+3	;     DATA REGISTER
FDCON	EQU	FDBASE+4	;DISK CONTROL PORT
FDAS	EQU	FDBASE+5	;AUX STATUS PORT

;	COMMANDS

FDCRST	EQU	000H		;RESTORE
FDCSEK	EQU	010H		;SEEK
FDCSTP	EQU	020H		;STEP
FDCSTI	EQU	040H		;STEP IN
FDCSTO	EQU	060H		;STEP OUT
FDCRDS	EQU	080H		;READ SECTOR
FDCWRS	EQU	0A0H		;WRITE SECTOR
FDCRDA	EQU	0C0H		;READ ADDRESS
FDCRDT	EQU	0E0H		;READ TRACK
FDCWRT	EQU	0F0H		;WRITE TRACK
FDCFI	EQU	0D0H		;FORCE INTERRUPT

;	TYPE 1 COMMAND FLAGS

FDFUTR	EQU	00010000B	;UPDATE TRACK REGISTER
FDFHLB	EQU	00001000B	;HEAD LOAD AT BEGINNING
FDFVRF	EQU	00000100B	;VERIFY FLAGS

;	TYPE 1 COMMAND STEP RATE FLAGS

FDFSRM	EQU	00000011B	;STEP RATE MASK
FDFS6	EQU	00000000B	;STEP RATE  6(3) MS
FDFS12	EQU	00000001B	;          12(6)
FDFS20	EQU	00000010B	;          20(10)
FDFS30	EQU	00000011B	;          30(15)

;	TYPE 2&3 COMMAND FLAGS

FDFMRF	EQU	00010000B	;MULTIPLE RECORD FLAG
FDFSLF	EQU	00001000B	;SECTOR LENGTH FLAG
FDFDLF	EQU	00000100B	;30 MS DELAY
FDFSS1	EQU	00000010B	;SELECT SIDE 1
FDFDDM	EQU	00000001B	;DELETED DATA MARK

;	TYPE 4 COMMAND FLAGS

FDFINI	EQU	00000000B	;TERMINATE WITH NO INTERRUPT
FDFII0	EQU	00000001B	;NOT READY TO READY TRANSITION
FDFII1	EQU	00000010B	;READY TO NOT READY TRANSITION
FDFII2	EQU	00000100B	;INDEX PULSE
FDFII3	EQU	00001000B	;IMMEDIATE INTERRUPT

;	STATUS FLAGS

FDSNRD	EQU	10000000B	;NOT READY
FDSWPV	EQU	01000000B	;WRITE PROTECT VIOLATION
FDSHLD	EQU	00100000B	;HEAD IS LOADED
FDSRTE	EQU	00100000B	;RECORD TYPE
FDSWTF	EQU	00100000B	;WRITE FAULT
FDSSEK	EQU	00010000B	;SEEK ERROR
FDSRNF	EQU	00010000B	;RECORD NOT FOUND
FDSCRC	EQU	00001000B	;CRC ERROR
FDSTK0	EQU	00000100B	;FOUND TRACK 0
FDSLDT	EQU	00000100B	;LOST DATA
FDSIND	EQU	00000010B	;INDEX HOLE
FDSDRQ	EQU	00000010B	;DRQ
FDSBSY	EQU	00000001B	;BUSY

;	INFO RETURNED BY A READ ADDRESS COMMAND

FDRATRK	EQU	0		;TRACK
FDRASID	EQU	1		;SIDE
FDRASEC	EQU	2		;SECTOR
FDRASL	EQU	3		;SECTOR LENGTH
FDRACRC	EQU	4		;2 BYTE CRC
FDRAL	EQU	6		;LENGTH OF READ ADDRESS INFO

;	DISK HEADER SECTOR LENGTH VALUES

FDSL128	EQU	0		;SECTOR LENGTH 128
FDSL256	EQU	1		;SECTOR LENGTH 256
FDSL512	EQU	2		;SECTOR LENGTH 512
FDSL1K	EQU	3		;SECTOR LENGTH 1024

;	CONTROL REGISTER FLAGS

CONDS	EQU	00000011B	;DRIVE SELECT BITS
CONDS8	EQU	00000100B	;0=5 1/4" , 1=8"
CONDSEN	EQU	00001000B	;DRIVE SELECT ENABLE
CONPC	EQU	00010000B	;WRITE PRE-COMPENSATION
				; 5 1/4"  0=YES , 1=NO
				; 8"  0=ALL TRACKS , 1=TRACKS 44-76
CON5FS	EQU	00100000B	;5 1/4" FAST STEP
CONWE	EQU	01000000B	;ENABLE WAIT FOR DRQ OR IRQ
CONSD	EQU	10000000B	;ENABLE SINGLE DENSITY

;	AUXILARY STATUS REGISTER FLAGS

ASIRQ	EQU	00000001B	;1797 INTERRUPT REQUEST
ASMO	EQU	00000010B	;5 1/4" MOTOR ON
AS96T	EQU	00001000B	;5 1/4" DRIVES ARE 96TPI
AS5PC	EQU	00010000B	;5 1/4" DRIVES NEED WRITE PRE-COMPENSATION
AS2S	EQU	01000000B	;SELECTED 8" DRIVE CONTAINS 2 SIDED MEDIA
ASDRQ	EQU	10000000B	;1797 DRQ

;	MISCELLANEOUS VALUES

NTRKS37	EQU	40		;# TRACKS SINGLE DENSITY 5 1/4" (48 TPI)
NTRKD37	EQU	80		;# TRACKS DOUBLE DENSITY 5 1/4" (96 TPI)
NTRK837	EQU	77		;NUMBER OF TRACKS 8"
NSBT37	EQU	52		;NUMBER OF CP/M RECORDS TO BE LOADED AT BOOT



;**	Z-MACHINE PORT ASSIGNMENTS
;


ZVIDEO	EQU	0D8H		;VIDEO 68A21
ZCRTC	EQU	0DCH		;VIDEO 6845 CRTC
ZLPEN	EQU	0DEH		;LIGHT PEN COUNTER
ZGDP	EQU	0E0H		;GENERAL DATA PORT (68A21)
ZTIMER	EQU	0E4H		;8253 TIMER
ZSERA	EQU	0E8H		;SERIAL A
ZSERB	EQU	0ECH		;SERIAL B
Z8259AS	EQU	0F0H		;8259A SLAVE
Z8259AM	EQU	0F2H		;8259A MASTER
ZKEYBD	EQU	0F4H		;KEYBOARD
ZTSTAT	EQU	0FBH		;TIMER STATUS
ZMCL	EQU	0FCH		;MEMORY CONTROL LATCH
ZHAL	EQU	0FDH		;HI-ADDRESS LATCH
ZPSP	EQU	0FEH		;PROCESSOR SWAP PORT
ZDIPSW	EQU	0FFH		;DIP SWITCHES

;	MEMORY CONTROL LATCH

ZMCLMS	EQU	00000011B	;MAP SELECT MASK
ZMS0	EQU	000H		; MAP SELECT 0
ZMS1	EQU	001H		; MAP SELECT 1
ZMS2	EQU	002H		; MAP SELECT 2
ZMS3	EQU	003H		; MAP SELECT 3

ZMCLRM	EQU	00001100B	;MONITOR ROM MAPPING
ZRM0	EQU	000H		; POWER UP MODE - ROM EVERYWHERE ON READS
ZRM1	EQU	004H		; ROM AT THE TOP 8K OF EVERY 64K
ZRM2	EQU	008H		; ROM AT THE TOP 8K OF THE 8088'S ADDR SPACE
ZRM3	EQU	00CH		; DISABLE ROM

ZMCLP	EQU	00110000B	;PARITY HANDLING
ZMCLPZ	EQU	00010000B	; 0=SET PARITY TO THE ZERO STATE
ZMCLPK	EQU	00100000B	; 0=DISABLE PARITY CHECKING CIRCUITRY

;	HI-ADDRESS LATCH

ZHAL85	EQU	11111111B	;8085 MASK
ZHAL88	EQU	00001111B	;8088 MASK

;	PROCESSOR SWAP PORT

ZPSPPS	EQU	10000000B	;PROCESSOR SELECT (0=8085 , 1=8088)
ZPSPPS5	EQU	000H		; SELECT 8085
ZPSPPS8	EQU	080H		; SELECT 8088
ZPSPGIS	EQU	00000010B	;GENERATE INTERRUPT ON SWAPPING
ZPSPI88	EQU	00000001B	;8088 PROCESSES ALL INTERRUPTS




;**	ASCII CONTROL CHARACTER EQUATES
;

CTLC	EQU	003H			;CONTROL-C
CTLQ	EQU	011H			;CONTROL-Q
CTLS	EQU	013H			;CONTROL-S
CTLZ	EQU	01AH			;CONTROL-Z

ACK	EQU	006H			;<ACK>
BELL	EQU	007H			;BELL
CPMEOF	EQU	CTLZ			;ASCII FILE EOF MARK
CPMEOM	EQU	'$'			;CP/M END-OF-MESSAGE SYMBOL
CR	EQU	00DH			;CARRIAGE RETURN
ESC	EQU	01BH			;ESCAPE
ETX	EQU	CTLC			;<ETX>
DC1	EQU	CTLQ			;<DC1>
DC3	EQU	CTLS			;<DC3>
LF	EQU	00AH			;LINE FEED
NULL	EQU	000H			;NULL


;**	MISCELLANEOUS EQUATES

	IF	REL0
BIOSORG	EQU	CCPL+BDOSL		;ORIGIN FOR LOADER BIOS
	ELSE
BIOSORG	EQU	CCPL+BDOSL+0100H	;ORIGIN FOR LOADER BIOS
	ENDIF
BDOSORG	EQU	BIOSORG-BDOSL		;ORIGIN FOR BDOS
BDOSE	EQU	BDOSORG+6		;BDOS ENTRY POINT
HEATH	EQU	BOOTORG+LABHTH		;LOCATION OF HEATH EXTENSIONS TO DPE
COUNT	EQU	HEATH+DPETRK-DPEHTH	;# BYTES PER SECTOR
DPB	EQU	BOOTORG+LABDPB		;LOCATION OF DISK PARAMETER BLOCK
RELOC	EQU	TPA+080H		;LOCATION OF 8085 BIOS RELOCATOR

	PAGE
;**	BIOS LOADER JUMP VECTORS
;

	ORG	BIOSORG

	JMP	CBOOTE			;COLD BOOT ENTRY
	JMP	WBOOTE			;WARM BOOT
	JMP	UNUSED			;CONSOLE STATUS
	JMP	WBOOTE			;CONSOLE INPUT
					; (ONLY REASON IS DISK I/O ERROR)
	JMP	CONOUTE			;CONSOLE OUTPUT ENTRY
	JMP	UNUSED			;LIST OUTPUT
	JMP	UNUSED			;PUNCH OUTPUT
	JMP	UNUSED			;READER INPUT
	JMP	HOMEE			;HOME HEAD ENTRY
	JMP	SETDSKE			;SET DISK ENTRY
	JMP	SETTRKE			;SET TRACK ENTRY
	JMP	SETSECE			;SET SECTOR ENTRY
	JMP	SETDMAE			;SET DMA ENTRY
	JMP	READE			;READ CP/M SECTOR ENTRY
	JMP	UNUSED			;WRITE CP/M SECTOR ENTRY
	JMP	UNUSED			;LIST STATUS
	JMP	SECTRNE			;SECTOR TRANSLATE ENTRY

	DB	(BIOSEND-BIOSORG+255)/256	;BIOS LOADER SIZE IN PAGES

	PAGE
;**	MAIN PROGRAM
;

CBOOTE:
	LXI	SP,STACK		;INIT STACK POINTER

	LXI	H,BIOSORG		;PLACE START OF BIOS MEMORY AREA
	SHLD	BBIOS			; INTO PAGE ZERO

	IN	FDTRK			;GET CURRENT TRACK POSITION OF DRIVE
	STA	HSTTRK			; FROM CONTROLLER

	MVI	C,RDS			;RESET DISK SYSTEM
	CALL	BDOSE

	LXI	H,BIOS88		;READ 8088 CODED PORTION OF THE
	CALL	RDF			; THE REAL BIOS
	JZ	RDFE			;  BR IF FILE NOT FOUND

	MVI	A,'5'			;READ 8085 CODED PORTION OF THE
	STA	BIOSFCB+FCBFN+5
	LXI	H,TPA
	CALL	RDF			; THE REAL BIOS
	JZ	RDFE			;  BR IF FILE NOT FOUND

	MVI	B,11			;READ CHARACTER SET CHANGES/MAPPINGS
	LXI	D,ALTCHAR
	LXI	H,BIOSFCB+FCBFN
	CALL	MOVEIT
	LHLD	BUFPTR			;REMEMBER WHERE CHARACTER SET
	SHLD	BBIOS-2			; CHANGES/MAPPINGS ARE TO BE PLACED
	CALL	RDF			;DO READ
	JNZ	CBOOTE2			; BR IF FILE FOUND
	LXI	H,0			;INDICATE NO ALTERNATE CHARACTER SET
	SHLD	BBIOS-2

CBOOTE2:
	JMP	RELOC			;GO RELOCATE 8085 CODED PORTION
					; OF THE REAL BIOS AND DON'T RETURN
	PAGE
;**	CONSOLE OUTPUT ENTRY
;
;	ENTRY:	(C)=CHARACTER
;	EXIT:	NONE
;	USES:	ALL

CONOUTE:
	MOV	A,C			;MOVE CHARACTER TO COMMUNICATION
	STA	COMRC			; REGION
	MVI	A,(CONOUT-CBOOT)/3	;SET BIOS FUNCTION #
	STA	COMFUNC
	MVI	A,ZPSPPS8		;SWAP TO 8088 PROCESSOR
	OUT	ZPSP
	RET


;**	HOME HEAD ENTRY
;
;	ENTRY:	NONE
;	EXIT:	NONE
;	USES:	ALL

HOMEE:
	MVI	C,0			;TRACK REQUEST OF 0

	IF	$ NE SETTRKE
%:	'SETTRKE' MUST IMMEDIATELY FOLLOW 'HOMEE'
	ENDIF


;**	SET TRACK NUMBER ENTRY
;
;	ENTRY:	(BC)=TRACK NUMBER
;	EXIT:	NONE
;	USES:	ALL
;

SETTRKE:
	MOV	A,C			;SAVE TRACK NUMBER
	STA	REQTRK
	RET


;**	SET SECTOR NUMMBER ENTRY
;
;	ENTRY:	(BC)=SECTOR NUMBER
;	EXIT:	NONE
;	USES:	ALL
;

SETSECE:
	MOV	A,C			;SAVE SECTOR NUMBER
	STA	REQSEC
	RET


;**	SET DMA ADDRESS
;
;	ENTRY:	(BC)=ADDRESS
;	EXIT:	NONE
;	USES:	ALL
;

SETDMAE:
	MOV	H,B			;SAVE DMA ADDRESS
	MOV	L,C
	SHLD	DMAPTR
	RET


;**	SECTOR TRANSLATE ENTRY
;
;	ENTRY:	(BC)=SECTOR NUMBER
;		(DE)=POINTER TO TRANSLATE TABLE
;	EXIT:	(HL)=TRANSLATED SECTOR NUMBER
;	USES:	ALL
;

SECTRNE:
	XCHG				;(HL)=POINTER TO TRANSLATE TABLE
	MOV	A,H			;CHECK IF NULL TRANSLATE TABLE
	ORA	L
	DAD	B
	RZ				; RET ORIGINAL VALUE IF NULL TABLE
	MOV	L,M			;  OTHERWISE RET IN (HL)
	MVI	H,0			;  THE TRANSLATED VALUE
	RET


;**	SET DISK ENTRY
;
;	ENTRY:	NONE (DRIVE 'A' IS ASSUMED)
;	EXIT:	(HL)=POINTER TO 'DPE'
;	USES:	ALL
;

SETDSKE:
	LXI	H,0			;ASSUME NULL SECTOR TRANSLATE TABLE

	LDA	HEATH+DPEUNIT-DPEHTH
	ANI	CONDS8			;CHECK IF 8" DRIVE
	JZ	SDE1			; BR IF 5 1/4" DRIVE

	LDA	HEATH+DPERPS-DPEHTH
	CPI	1024/128		;CHECK IF EXTENDED DOUBLE DENSITY
	JZ	SDE1			; BR IF 5x1024

	LXI	H,XLATE1		;ASSUME SINGLE DENSITY TABLE

	LDA	HEATH+DPEFLAG-DPEHTH
	ANI	DPEDD			;CHECK IF DOUBLE DENSITY
	JZ	SDE1			; BR IF SINGLE DENSITY

	LXI	H,XLATE2		;USE DOUBLE DENSITY TABLE

SDE1:
	SHLD	DPE+DPEXLT		;SAVE SECTOR TRANSLATE TABLE ADDRESS

	LXI	H,DPE			;RETURN WITH (HL)=POINTER TO 'DPE'
	RET


;**	READ CP/M SECTOR ENTRY
;
;	ENTRY:	NONE
;	EXIT:	(A)=ERROR CODE
;		    0 = NO ERROR  , 0FFH = ERROR
;	USES:	ALL
;

READE:

;	DETERMINE PHYSICAL TRACK, SIDE, SECTOR

	LDA	REQTRK			;ASSUME PHYSICAL TRACK = REQ TRACK
	MOV	L,A
	MVI	H,0			;ASSUME SIDE 0
	LDA	HEATH+DPEFLAG-DPEHTH	;CHECK IF DOUBLE SIDED MEDIA
	RAR
	IF	DPE2S NE 1
%:	DPE2S NE 1
	ENDIF
	JNC	RDS1			; BR IF SINGLE SIDED MEDIA
	MOV	A,L			;PHYSICAL TRACK =
	ANA	A			; REQUESTED TRACK / 2
	RAR
	MOV	L,A
	JNC	RDS1			;EVEN PHYSICAL TRACK = SIDE 0
	MVI	H,FDFSS1		; ODD PHYSICAL TRACK = SIDE 1
RDS1:	SHLD	PHYTRK			;SAVE PHYSICAL TRACK & SIDE
	IF	(PHYTRK+1) NE PHYSID
%:	(PHYTRK+1) NE PHYSID
	ENDIF

	LDA	REQSEC			;PHYSICAL SECTOR =
	MOV	B,A			; REQUESTED CP/M RECORD 'SHR'
	LDA	HEATH+DPERPS-DPEHTH	;  LOG2(CP/M RECORDS PER SECTOR)
RDS1A:	ANA	A
	RAR
	MOV	C,A
	MOV	A,B
	JC	RDS1B
	RAR
	MOV	B,A
	MOV	A,C
	JMP	RDS1A
RDS1B:	STA	PHYSEC			;SAVE PHYSICAL SECTOR #

;	CHECK IF PHYSICAL SECTOR IS ALREADY IN HOST BUFFER

	LDA	PHYSID
	LXI	H,HSTSID
	CMP	M
	JNZ	RDS2			;BR IF NOT SAME SIDE
	LDA	PHYTRK
	LXI	H,HSTTRK
	CMP	M
	JNZ	RDS2			;BR IF NOT SAME TRACK
	LDA	PHYSEC
	LXI	H,HSTSEC
	CMP	M
	JZ	RDS5			;BR IF SAME SECTOR

;	NEED TO READ PHYSICAL SECTOR INTO HOST BUFFER

RDS2:
	LDA	HSTTRK
	LXI	H,PHYTRK
	CMP	M
	JZ	RDS3			;BR IF HEAD ALREADY AT DESIRED TRACK

	CALL	SEEK			;SEEK TO DESIRED TRACK

	LDA	HEATH+DPEFLAG-DPEHTH	;CHECK IF NEED TO DOUBLE STEP BECAUSE
	ANI	DPE48RO			; OF 48 TPI MEDIA ON 96 TPI DRIVE
	JZ	RDS2B			; BR IF NO NEED

	LDA	HSTTRK			;DO DOUBLE STEP BY FORCING CONTROLLER
	OUT	FDTRK			; TO SEEK BY SAME AMOUNT AS WAS
	CALL	SEEK			;  JUST DONE

RDS2B:
	MOV	A,M			;REMEMBER WHERE HEAD IS NOW
	STA	HSTTRK

RDS3:
	LDA	PHYSEC			;GET PHYSICAL SECTOR #
	STA	HSTSEC			;REMEMBER IT
	INR	A			;PUT INTO RANGE 1 TO SPT
	OUT	FDSEC			;TELL CONTROLLER

	IF	WAIT
	LDA	HEATH+DPEUNIT-DPEHTH	;ENABLE WAIT STATE I/O
	ORI	CONWE
	OUT	FDCON
	ENDIF

	LDA	PHYSID			;GET PHYSICAL SIDE #
	STA	HSTSID			;REMEMBER IT
	ORI	FDCRDS+FDFSLF+FDFDLF	;FORM READ COMMAND
	LHLD	COUNT			;GET # BYTES PER SECTOR
	LXI	D,HSTBUF		;HOST BUFFER POINTER
	OUT	FDCMD			;ISSUE COMMAND TO CONTROLLER

RDS4:
	IF	NOT WAIT
	IN	FDAS			;WAIT FOR DRQ OR IRQ
	ANI	ASDRQ+ASIRQ
	JZ	RDS4
	ENDIF
	IN	FDDAT			;GET DATA
	STAX	D			;STORE DATA INTO BUFFER
	INX	D			;INCREMENT BUFFER POINTER
	DCX	H			;DECREMENT BUFFER POINTER
	MOV	A,H
	ORA	L
	JNZ	RDS4			;LOOP FOR COUNT BYTES

	IF	WAIT
	LDA	HEATH+DPEUNIT-DPEHTH	;TURN OFF WAIT STATE I/O
	OUT	FDCON
	ENDIF

	CALL	WIO1			;WAIT FOR IRQ AND GET STATUS
	ANA	A
	JNZ	RDSE			; BR IF READ ERROR

;	MOVE CP/M RECORD FROM HOST BUFFER TO USER BUFFER

RDS5:
	LDA	REQSEC			;DETERMINE WHICH RECORD IN BUFFER
	MOV	B,A			; TO USE
	LDA	HEATH+DPERPS-DPEHTH	;  DISPLACEMENT =
	DCR	A			;   128 * ('REQSEC' MOD 'DPERPS')
	ANA	B
	LXI	H,HSTBUF-128
	LXI	D,128
RDS5A:	DAD	D
	DCR	A
	JP	RDS5A
	XCHG				;(DE)=CP/M RECORD
	LHLD	DMAPTR			;(HL)=ADDR OF USER BUFFER

	MVI	B,128			;MOVE DATA
	CALL	MOVEIT

;	RETURN INDICATING NO ERROR

UNUSED:
	XRA	A
	RET

;	READ ERROR -- INDICATE IT AND RETURN

RDSE:
	CPI	FDSRNF			;RECORD NOT FOUND
	JNZ	RDSE1			; BR IF NOT
	LXI	H,HEATH+DPEFLAG-DPEHTH	;Q. ALREADY DOUBLE STEPPING
	MOV	A,M
	ANI	DPE48RO
	JNZ	RDSE1			; BR IF YES

	MOV	A,M			;SET FOR DOUBLE STEPPING AND TRY AGAIN
	ORI	DPE48RO
	MOV	M,A

	MVI	A,FDCRST+FDFHLB+FDFS30	;RESTORE HEAD
	CALL	WAITIO
	XRA	A			;INDICATE TRACK 0
	STA	HSTTRK

	JMP	RDS2			;RETRY READ

RDSE1:
	MVI	A,0FFH
	RET


;*	SEEK TO DESIRED TRACK
;
;	ENTRY:	(HL)=POINTER TO DESIRED TRACK VALUE
;	EXIT:	VIA 'WAITIO'
;	USES:	A,F

SEEK:
	MOV	A,M			;SET DESIRED TRACK
	OUT	FDDAT
	MVI	A,FDCSEK+FDFHLB+FDFS30

	IF	$ NE WAITIO
%:	'WAITIO' DOES NOT IMMEDIATELY FOLLOW 'SEEK'
	ENDIF


;*	WAIT FOR I/O COMPLETION AND RETURN STATUS
;
;	ENTRY:	(A)=COMMAND
;	EXIT:	(A)=STATUS
;	USES:	A,F
;
;	ALTERNATE ENTRY POINT 'WIO1'
;	ENTRY:	NONE
;	EXIT:	(A)=STATUS
;	USES:	A,F
;

WAITIO:
	OUT	FDCMD			;ISSUE COMMAND

WIO1:
	IN	FDAS			;WAIT FOR IRQ
	RAR
	IF	ASIRQ NE 1
%:	ASIRQ NE 1
	ENDIF
	JNC	WIO1

	IN	FDSTA			;GET FINAL STATUS

	RET


;**	HALT DUE TO ERROR
;

RDFE:
	MVI	C,PRCONB		;PRINT ERROR MESSAGE
	LXI	D,LOADERR
	CALL	BDOSE

WBOOTE:
	XRA	A			;TURN OFF DRIVE
	OUT	FDCON

	HLT				;DIE

	PAGE
;*	MOVEIT - MOVE BLOCK OF MEMORY
;
;	ENTRY:	(B)=COUNT
;		(DE)=SOURCE ADDRESS
;		(HL)=DESTINATION ADDRESS
;	EXIT:	NONE
;	USES:	A,B,D,E,H,L
;

MOVEIT:
	LDAX	D
	MOV	M,A
	INX	H
	INX	D
	DCR	B
	JNZ	MOVEIT
	RET


;*	RDF - READ FILE INTO MEMORY
;
;	ENTRY:	(HL)=MEMORY STARTING ADDRESS
;	EXIT:	PSW/Z	0=NO ERROR , 1=ERROR
;	USES:	ALL
;

RDF:
	PUSH	H			;SAVE STARTING ADDR

	MVI	C,OPEN			;ATTEMPT TO OPEN FILE
	LXI	D,BIOSFCB
	CALL	BDOSE
	INR	A			;CHECK FOR ERROR

	POP	H			;RESTORE INTIAL BUFFER POINTER

	RZ				; RET IF ERROR

	XRA	A			;REWIND FILE
	STA	BIOSFCB+FCBNR

RDF1:
	SHLD	BUFPTR
	MVI	C,SETDMA		;TELL BDOS ABOUT BUFFER POINTER
	XCHG
	CALL	BDOSE

	MVI	C,READ			;READ NEXT CP/M SECTOR
	LXI	D,BIOSFCB
	CALL	BDOSE
	ORA	A			;CHECK FOR READ ERROR
	RNZ				; RET IF EOF

	LXI	D,128			;BUMP BUFFER POINTER
	LHLD	BUFPTR
	DAD	D

	JMP	RDF1			;CONTINUE READING FILE

	PAGE
;**	DATA STORAGE
;

BIOSFCB	DB	0,'BIOS88  SYS',0,0,0,0
	DB	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
	IF	($-BIOSFCB) NE FCBLS
%:	BIOS FCB IS NOT THE CORRECT LENGTH
	ENDIF

ALTCHAR	DB	'ALTCHAR SYS'

LOADERR	DB	'LOAD ERROR$'

XLATE1	DS	0		;8" SINGLE DENSITY SECTOR TRANSLATE TABLE
	DB	0,6,12,18,24
	DB	4,10,16,22
	DB	2,8,14,20
	DB	1,7,13,19,25
	DB	5,11,17,23
	DB	3,9,15,21

XLATE2	DS	0		;8" DOUBLE DENSITY SECTOR TRANSLATE TABLE
	DB	0,1,18,19,36,37
	DB	2,3,20,21,38,39
	DB	4,5,22,23,40,41
	DB	6,7,24,25,42,43
	DB	8,9,26,27,44,45
	DB	10,11,28,29,46,47
	DB	12,13,30,31,48,49
	DB	14,15,32,33,50,51
	DB	16,17,34,35

DPE	DS	0			;DISK PARAMETER ENTRY TABLE
	DW	0			; SECTOR TRANSLATE TABLE ADDRESS
	DW	0,0,0			; SCRATCH
	DW	DIRBUF			; DIRECTORY BUFFER
	DW	DPB			; DISK PARAMETER BLOCK POINTER
	DW	CSV			; CHECKSUM VECTOR POINTER
	DW	ALV			; ALLOCATION VECTOR POINTER

HSTSID	DB	-1			;CURRENT SIDE IN HOST BUFFER

	PAGE
;
;	*** NOTE ***
;
;	THERE ARE ONLY 52 CP/M SECTORS AVAILABLE FOR THE SYSTEM.
;	THE CCP AND BDOS OCCUPY 44 SECTORS, LEAVING 8 CP/M SECTORS
;	FOR THE BOOT CODE, THE 8088 CODED PORTION OF THE BIOS LOADER,
;	AND THE 8085 CODED PORTION OF THE BIOS LOADER.
;	THEREFORE, THE LOCATION COUNTER AT THIS POINT CANNOT BE GREATER THAN
;	BIOSORG+BLDR85L.
;

	IF	($-BIOSORG) GT BLDR85L
%:	8085 CODED PORTION OF THE BIOS LOADER IS TOO LARGE
	ENDIF

;

FCBPTR	DS	2			;FCB POINTER
BUFPTR	DS	2			;BUFFER POINTER
DIRBUF	DS	128			;DIRECTORY BUFFER
CSV	DS	64			;CHECKSUM VECTOR
ALV	DS	77			;ALLOCATION VECTOR
REQTRK	DS	1			;REQUESTED CP/M TRACK
REQSEC	DS	1			;REQUESTED CP/M SECTOR
DMAPTR	DS	2			;DMA POINTER
PHYTRK	DS	1			;REQUESTED PHYSICAL TRACK
PHYSID	DS	1			;REQUESTED PHYSICAL SIDE
PHYSEC	DS	1			;REQUESTED PHYSICAL SECTOR
HSTTRK	DS	1			;HOST TRACK
HSTSEC	DS	1			;CURRENT SECTOR IN HOST BUFFER

HSTBUF	DS	1024			;HOST DISK BUFFER

	DS	64			;STACK AREA
STACK	DS	0

BIOSEND	DS	0

	END
RRENT SECTOR IN HOST BUFFER

HSTBUF	DS	1024			;HOST DISK BUFFER

	DS	64			;STACK AREA
STACK	DS	0

