/*******************************************************************/
/*                                                                 */
/*       CXREF                         by: Wojciech BOK            */
/*                                                                 */
/*                                   date: May 23, 1988            */
/*                                                                 */
/*       Filename: XREF filename.C switches redirection            */
/*                                                                 */
/*       Purpose:   parse a file for legal function names, ignore  */
/*                  reserved words, and maintain a list of line    */
/*                  numbers where the function names occur         */
/*                                                                 */
/*       Note:  All global variables begin with a capital letter   */
/*                                                                 */
/*******************************************************************/

#include <stdio.h>                  /* definition of stderr */
#include <math.h>                   /* holds atoi */

#define     FALSE           0       /* not true */
#define     TRUE            1       /* true */
#define     NUL             0
#define     NAME_LENGTH     32      /* maximum length of a function name */
#define     NAME_BUFFER     NAME_LENGTH + 4 /* size of name buffer - 4 extra chars for special display */
#define     MAX_LINES       10      /* maximum number of lines per line link */
#define     BUF_SIZE        60      /* length of input buffer */
#define     EOL             '\n'    /* new line char */
#define     MOL             '\0'    /* middle of logical line char */

struct      S_library_functions
                {
                char     Name [ NAME_BUFFER ];           /* names of library functions */
                struct   S_library_functions *Following; /* address of next link */
                };
struct      S_library_functions Lib_head,               /* dummy starting point */
                                *Library,               /* current library function */
                                *Blank_lib;             /* blank space */

struct      S_line_numbers
                {
                int      Number [ MAX_LINES ];          /* line number lsit */
                int      Space_left;                    /* number of spaces left in this link */
                struct   S_line_numbers *Following;     /* address of next link */
                };
struct      S_line_numbers  *Blank_line;                /* next available space */
                
struct      S_function_name
                {
                char     Name [ NAME_BUFFER ];          /* function name */
                int      Too_many_lines;                /* line number overflow -- not enough memory */
                struct   S_line_numbers *First_line;    /* first link in line number list */
                struct   S_line_numbers *Curr_line;     /* current working number list */
                struct   S_function_name *Following;    /* address of next function */
                };
struct      S_function_name Head,                       /* dummy starting point */
                            *Function,                  /* current name in use */
                            *Blank_name;                /* next available space */

int         Line_count = 0;         /* current source line number */
int         Overflow = 0;           /* too many function names count and flag */
int         Eof = FALSE;            /* flag for end of file */

char        Line_buf[BUF_SIZE];     /* current line */
char        *Line_ptr;              /* input line pointer */
int         Input_file;             /* input file handle */
int         Output_file = FALSE;    /* handle of file for numbered listing */

int         XR_var = FALSE;         /* cross ref variables */
int         XR_func = FALSE;        /*           function names */
int         XR_lib = FALSE;         /*           library functions */
int         XR_resv = FALSE;        /*           reserved words */

#define     ONLY_LIB      !XR_var && !XR_func && !XR_resv && XR_lib  /* only the library switch was used */
#define     NO_XR_FLAGS   !XR_var && !XR_func && !XR_resv && !XR_lib /* no cross reference switches were used */
            
int         R_margin = 80;          /* width of listing */

/*******************************************************************/
/*                             IDENTIFICATION                      */
/*    Function name:  MAIN ( argc , argv )                         */
/*    Purpose      :  inputs the file name to be parsed and        */
/*                      controls the flow of the program           */
/*    Method       :  command line arguments                       */
/*                                                                 */
/*******************************************************************/
/*                             COMMUNICATION                       */
/*    Formal Paramters:                                            */
/*        NAME               TYPE                    USED FOR      */
/*     argc                  int                number of arguments*/
/*     argv                  char               arguments          */
/*                                                                 */
/*    Global Variables Needed:                                     */
/*        NAME               TYPE                    USED FOR      */
/*      TRUE                 int                                   */
/*      MAX_NAMES            int                limiter            */
/*      NAME_BUFFER          int                size of words      */
/*      MAX_LINES            int                limiter            */
/*      NO_XR_FLAGS          macro              test               */
/*      ONLY_LIB             macro              test               */
/*                                                                 */
/*      *Line_ptr            int                pointer            */
/*      Eof                  int                flag               */
/*      Line_count           int                counter            */
/*      R_margin             int                width of output    */
/*                                                                 */
/*      Output_file          int                file handle        */
/*      Input_file           int                file handle        */
/*      stderr               int                standard error     */
/*                                                handle           */
/*                                                                 */
/*      XR_resv              int                reserved words     */
/*      XR_lib               int                library functions  */
/*      XR_func              int                function names     */
/*      XR_var               int                variables          */
/*                                                                 */
/*      S_function_name      struct             variable type      */
/*                                                                 */
/*      Head                 struct             starting point for */
/*                                                linked list      */
/*        .Following         int                pointer            */
/*        .First_line        int                pointer            */
/*        .Curr_line         int                pointer            */
/*        .Too_many_lines    int                flag               */
/*        .Name              char               function name      */
/*                                                                 */
/*      Function             struct             linked list        */
/*                                                                 */
/*      Blank_name           struct             next available     */
/*                                                name location    */
/*        ->Following        int                pointer            */
/*        ->Too_many_lines   int                flag               */
/*                                                                 */
/*      Lib_head             struct             library list start */
/*        .Following         int                pointer            */
/*        .Name              char               function name      */
/*                                                                 */
/*      Library              struct             library list       */
/*                                                                 */
/*    Return Value:                                                */
/*                     none                                        */
/*******************************************************************/

main( argc, argv)
    char *argv[];
    {
    int number;                 /* number of functions found */
    char output_name[15];       /* name of output file if any */
    int  output_flag = FALSE;   /* flag for line numbered O/P */

    fputs("\n                   C Cross Reference Utility - version 1.10", stderr);
    fputs("\n                                 by: Soft Touch\n\n", stderr);

    if ( argc == 1 )            /* no filename entered */
        fatal_msg( 1, " ");

    if ( *argv[1] == '/' )      /* switch entered first - treat as /? */
        fatal_msg( 9, " ");

    if ( argc > 2 )
        {
        number = 2;

        while ( number < argc )
            {
            if ( *argv[number] == '/' )
                {
                argv[number]++;
                switch ( toupper( *argv[number] ))
                    {
                    case 'R':
                        XR_resv = TRUE;
                        break;
                    case 'L':
                        XR_lib = TRUE;
                        break;
                    case 'F':
                        XR_func = TRUE;
                        break;
                    case 'V':
                        XR_var = TRUE;
                        break;
                    case 'W':
                        {
                        argv[number]++;
                        if (*argv[number] != ':')
                            R_margin = 132;
                        else
                            if ( ( R_margin = atoi( ++argv[number] ) ) < 50 )
                                R_margin = 80;
                        break;
                        }
                    case 'N':
                        {
                        argv[number]++;
                        output_flag = TRUE;
                        if (*argv[number] != ':')
                            {
                            strcpy ( output_name, argv[1]);
                            Line_ptr = output_name;
                            while ( *Line_ptr && *Line_ptr != '.' )
                                Line_ptr++;

                            *Line_ptr = '.';
                            *(Line_ptr + 1) = 'L';
                            *(Line_ptr + 2) = '\0';
                            }
                        else
                            {
                            argv[number]++;
                            strcpy( output_name, argv[number] );
                            }
                        break;
                        }
                    case '?':
                        fatal_msg( 9, " " );    /* print help screen and die */
                    default:
                        break;
                    }
                }
            else
                argv[number]++;

            if ( *argv[number] == '\0' )
                ++number;
            }
        }

    if ( NO_XR_FLAGS )  /* default */
        {
        XR_func = TRUE;
        XR_var = TRUE;
        }

    Head.Following  = NUL;
    Head.First_line = NUL;
    Head.Curr_line  = NUL;
    Head.Too_many_lines = FALSE;
    for ( number = 0; number < NAME_BUFFER; number++)
        Head.Name [ number ] = ' ';

    Function = Head.Following;
    
    if ( Blank_name = malloc ( sizeof ( struct S_function_name )))
        {
        Blank_name->Following = NUL;
        Blank_name->Too_many_lines = FALSE;
        }

    new_blank_line();

    Lib_head.Following = NUL;
    for ( number = 0; number < NAME_BUFFER; number++)
        Lib_head.Name [ number ] = ' ';

    Library = Lib_head.Following;

    number = FALSE;                /* 'number' is used as a flag temporarily */

    if ( ( Input_file = fopen ( "C_LIB.XRF", "r" )))
        {
        fputs("Reading library file C_LIB.XRF\r", stderr );
        number = get_library();
        fclose ( Input_file );
        Line_count = 0;
        Eof = FALSE;
        fputs("                              \r", stderr );
        }

    if ( !number )
        if ( ONLY_LIB )
            fatal_msg( 4, " " );
        else
            {
            fputs("\n     C_LIB.XRF not found or no functions found in C_LIB.XRF", stderr);
            fputs("\n     Cannot tell library functions and user functions apart\n\n", stderr);
            XR_lib = FALSE;
            }

    if ( output_flag )
        if ( ( Output_file = creat ( output_name )) == -1 )
            fatal_msg( 3, output_name );
        else
            fprintf( stderr, "Creating numbered list named %s\n", strupr ( output_name ) );

    if (( Input_file = fopen ( argv[1], "r" )) == FALSE )
        fatal_msg( 2, argv[1]);

    fprintf( stderr, "Using %d column width\nParsing file %s for:\n", R_margin, strupr( argv[1] ) );

    if ( XR_resv )
        fputs(" Reserved Words;", stderr);

    if ( XR_lib )
        fputs(" Library Functions;", stderr);

    if ( XR_func )
        fputs(" Function Names;", stderr);

    if ( XR_var )
        fputs(" Variables;", stderr);

    fputs("\b.\n", stderr);

    number = parse_file();

    fclose ( Input_file );

    if ( Output_file )
        fclose ( Output_file );

    print_list( number, argv[1] );

    fputs("\nFinished\n", stderr);

    }

/*******************************************************************/
/*                             IDENTIFICATION                      */
/*    Function name: FATAL_MSG ( number , name )                   */
/*    Purpose      : A fatal error has occurred. Print message     */
/*                   and abort                                     */
/*    Method       : print to stderr ( cannot be redirected )      */
/*                   exit with error level 'number'                */
/*******************************************************************/
/*                             COMMUNICATION                       */
/*    Formal Paramters:                                            */
/*        NAME               TYPE                    USED FOR      */
/*      number                int                  message number  */
/*      name                  char                 special message */
/*                                                                 */
/*    Global Variables Needed:                                     */
/*        NAME               TYPE                    USED FOR      */
/*      stderr               int                standard error     */
/*                                                handle           */
/*    Return Value:                                                */
/*                    exit value                                   */
/*******************************************************************/

fatal_msg( number, name )
    int number;
    char *name;
    {
    
    switch ( number )
        {
        case 1:
            fputs("You must enter a filename", stderr);
            break;
        case 2:
            if ( *name != '/' )         /* check if switch is only thing entered */ 
                fprintf( stderr, "Cannot open source file %s", strupr ( name ) );
            break;
        case 3:
            fprintf( stderr, "Cannot open %s for numbered output", strupr ( name ) );
            break;
        case 4:
            fputs("C_LIB.XRF not found and only /L switch used", stderr);
            break;
        case 9:                  /* place holder -- currently no message */
            break;
        default:
            break;
        }

    fputs(  "\nUsage is:  CXREF source [/s ...] [ > list_file]", stderr);

    fputs("\n\n  source      the name of the 'C' file to be cross referenced ie: TEST.C", stderr);
    fputs(  "\n  list-file   an optional re-direction of the list to a file or printer PRN", stderr);

    fputs("\n\n   /R  reserved words       /F  functions  ( if /R, /L, /F, and /L are not  )", stderr);
    fputs(  "\n   /L  library functions    /V  variables  ( chosen then /F and /V are used )", stderr);
    fputs(  "\n      (needs C_LIB.XRF)", stderr);

    fputs("\n\n   /W[:#]    width of list  default # is 132 - must be more than 50 columns/page", stderr);
    fputs(  "\n   /N[:name] numbered source file - default is source with .L extension", stderr);
    fputs("\n\n   /?        this screen - terminates program\n\n", stderr);
    
    exit ( number );
    }

/*******************************************************************/
/*                             IDENTIFICATION                      */
/*    Function name: GET_LIBRARY                                   */
/*    Purpose      : read the 'C' library functions from disk      */
/*                                                                 */
/*    Method       : locate functions and add to linked list       */
/*                                                                 */
/*******************************************************************/
/*                             COMMUNICATION                       */
/*    Formal Paramters:                                            */
/*        NAME               TYPE                    USED FOR      */
/*               none                                              */
/*                                                                 */
/*    Global Variables Needed:                                     */
/*        NAME               TYPE                    USED FOR      */
/*      NAME_BUFFER           int          size of function names  */
/*      FALSE                 int                                  */
/*      TRUE                  int                                  */
/*      Eof                   int          flag                    */
/*                                                                 */
/*    Return Value:                                                */
/*                  TRUE if function(s) found else FALSE           */
/*******************************************************************/

get_library()
    {
    char word[NAME_BUFFER];
    int flag = FALSE;
    
    read_line(0);
    
    if (!Eof)
        parse_word ( word );
    
    while (!Eof)
        {
        if ( is_reserved ( word ) )
             ;
        else if ( is_function() )
            {
            add_to_library( word );
            flag = TRUE;
            }

        parse_word( word );
        }

    return ( flag );
    }

/*******************************************************************/
/*                             IDENTIFICATION                      */
/*    Function name: ADD_TO_LIBRARY ( word )                       */
/*    Purpose      : add a new function to the library list        */
/*                                                                 */
/*    Method       : linked list                                   */
/*                                                                 */
/*******************************************************************/
/*                             COMMUNICATION                       */
/*    Formal Paramters:                                            */
/*        NAME               TYPE                    USED FOR      */
/*       word                char            word to be added      */
/*                                                                 */
/*    Global Variables Needed:                                     */
/*        NAME               TYPE                    USED FOR      */
/*      TRUE                  int                                  */
/*                                                                 */
/*      S_library_functions   struc          variable type         */
/*                                                                 */
/*      Library               struct         library list          */
/*        ->Following         int            pointer               */
/*        ->Name              char           function name         */
/*                                                                 */
/*      Lib_head              struct         start of list         */
/*                                                                 */
/*      Blank_lib             struct         next memory location  */
/*        ->Following         int            pointer               */
/*                                                                 */
/*    Return Value:                                                */
/*                       none                                      */
/*******************************************************************/

add_to_library( word )
    char *word;
    {
    int test = TRUE;
    
    Library = &Lib_head;
    
    while ( Library->Following )
        {
        if ( ( test = strcmp ( Library->Following->Name, word)) >= 0 )
            break;
        
        Library = Library->Following;
        }
    
    if ( test )
        {
        if ( Blank_lib = malloc ( sizeof ( struct S_library_functions )))
            {
            Blank_lib->Following = Library->Following;
            Library->Following = Blank_lib;
            strcpy( Library->Following->Name, word );
            }
        }
    }

/*******************************************************************/
/*                             IDENTIFICATION                      */
/*    Function name:  PARSE_FILE                                   */
/*    Purpose      :  main control for locating function names in  */
/*                    input file                                   */
/*    Method       :  gets a function name, checks if is a reserved*/
/*                    word, locates it in current words, and adds  */
/*                    it to the list                               */
/*                                                                 */
/*******************************************************************/
/*                             COMMUNICATION                       */
/*    Formal Paramters:                                            */
/*        NAME               TYPE                    USED FOR      */
/*              none                                               */
/*                                                                 */
/*    Global Variables Needed:                                     */
/*        NAME               TYPE                    USED FOR      */
/*        Eof                int                  end of file flag */
/*        NAME_BUFFER        int                  length of        */
/*                                                  function name  */
/*        MAX_NAMES          int                  maximum number of*/
/*                                                functions allowed*/
/*        Overflow           int                  too many names   */
/*        stderr             int                  standard error   */
/*                                                  handle         */
/*        XR_resv            int                output reserved    */
/*                                                words            */
/*        XR_func            int                output function    */
/*                                                names            */
/*        XR_var             int                output variables   */
/*        XR_lib             int                library functions  */
/*                                                                 */
/*        Line_count         int                current line in    */
/*                                                input file       */
/*        TRUE               int                                   */
/*        FALSE              int                                   */
/*                                                                 */
/*    Return Value:                                                */
/*                  number of functions located                    */
/*******************************************************************/

parse_file()
    {
    char word[ NAME_BUFFER ];         /* holds found word */
    int  current_word = 0;            /* position of found word in list */
    int  number_words = 0;            /* number of words found */
    int  add_flag;                    /* add name to list or not */

    fputs("  Lines    Words\n", stderr);
    fprintf( stderr, "\r %6d     %3d\r", Line_count , number_words );

    read_line(1);

    if ( !Eof )
        parse_word( word );

    while ( !Eof )
        {
        add_flag = FALSE;

        if ( is_reserved ( word ) )    /* logic checks if word is to be added */
            {
            if ( XR_resv )
                {
                add_flag = strlen ( word );
                *(word + add_flag) = ' ';
                *(word + add_flag + 1) = '<';
                *(word + add_flag + 2) = 'R';
                *(word + add_flag + 3) = '>';
                *(word + add_flag + 4) = '\0';
                add_flag = TRUE;
                }
            }
        else if ( is_function () )
            {
            if ( is_library ( word ) )
                {
                if ( XR_lib )
                    {
                    add_flag = strlen( word );
                    *(word + add_flag) = '(';
                    *(word + add_flag + 1) = ' ';
                    *(word + add_flag + 2) = 'L';
                    *(word + add_flag + 3) = '\0';
                    add_flag = TRUE;
                    }
                }
            else if ( XR_func )
                {
                add_flag = strlen ( word );
                *(word + add_flag) = '(';
                *(word + add_flag + 1) = '\0';
                add_flag = TRUE;
                }
            }
        else if ( XR_var )
            add_flag = TRUE;
        
        if ( add_flag )
            number_words = add_to_list ( word );

        fprintf( stderr, " %6d      %3d\r", Line_count , number_words );

        parse_word( word );
        }

    return (number_words -1);
    }

/*******************************************************************/
/*                             IDENTIFICATION                      */
/*    Function name: PARSE_WORD ( word_ptr )                       */
/*    Purpose      : locates the next legal function name          */
/*                                                                 */
/*    Method       : successively checks the line buffer until     */
/*                    a) a legal function name char is found       */
/*                    b) an ignore char is found                   */
/*                                                                 */
/*                   copies the legal name into the word_ptr and   */
/*                     returns                                     */
/*                                                                 */
/*                   if the end of a line is reached, gets the next*/
/*                     line via a function call                    */
/*                                                                 */
/*******************************************************************/
/*                             COMMUNICATION                       */
/*    Formal Paramters:                                            */
/*        NAME               TYPE                    USED FOR      */
/*     word_ptr              int                   function name   */
/*                                                                 */
/*    Global Variables Needed:                                     */
/*        NAME               TYPE                    USED FOR      */
/*      *Line_ptr            int                parsing input line */
/*      Eof                  int                end of file flag   */
/*      EOL                  char               end of line        */
/*      MOL                  char               end of line buffer */
/*      NAME_BUFFER          int                limiter            */
/*      TRUE                 int                                   */
/*      FALSE                int                                   */
/*                                                                 */
/*    Return Value:                                                */
/*                  none                                           */
/*******************************************************************/

parse_word ( word_ptr )
    char *word_ptr;
    {
    int parse = TRUE;       /* remain in loop until false */
    int word = 0;           /* word found flag & length */

    while ( parse )
        {
        if ( isalnum( *Line_ptr ) || *Line_ptr == '_' )
            {
            if ( ( !word && isdigit( *Line_ptr ) ) || word >= NAME_LENGTH )  /* skip if first char is a number, or name is too long */
                ++Line_ptr;
            else
                {
                *word_ptr++ = *Line_ptr++;
                word++;
                }
            }
        else if ( *Line_ptr == MOL )       /* end of buffer */
            read_line(0);           /* rest of line - no increment of line counter */
        else if ( word )
            parse = FALSE;          /* end of function name */
        else if ( *Line_ptr == EOL )       /* end of physical line */
            read_line(1);           /* next line - increment line counter */
        else if ( ignore_char() )
            parse_ignore();         /* single / double quotes or comments */
        else
            Line_ptr++;             /* next char */

        if ( Eof )
            parse = FALSE;

        }
    *word_ptr = '\0';
    }
/*******************************************************************/
/*                             IDENTIFICATION                      */
/*    Function name: PARSE_IGNORE                                  */
/*    Purpose      : to skip any chars that are in single, double  */
/*                   quotes or in a comment                        */
/*                                                                 */
/*    Method       : look for a corresponding ignore char          */
/*                                                                 */
/*******************************************************************/
/*                             COMMUNICATION                       */
/*    Formal Paramters:                                            */
/*        NAME               TYPE                    USED FOR      */
/*              none                                               */
/*                                                                 */
/*    Global Variables Needed:                                     */
/*        NAME               TYPE                    USED FOR      */
/*     *Line_ptr             int                line parsing       */
/*     FALSE                 int                                   */
/*     Eof                   int                end of file flag   */
/*     EOL                   char               end of line        */
/*     MOL                   char               end of line buffer */
/*                                                                 */
/*    Return Value:                                                */
/*                  none                                           */
/*******************************************************************/

parse_ignore()
    {
    char ignore;                /* current ignore char */
    char back_slash = 0;        /* backslash counter */

    if ( *Line_ptr == '\'' || *Line_ptr == '"' )
        ignore = *Line_ptr;
    else
        {
        ignore = '*';
        Line_ptr++;
        }

    Line_ptr++;

    while ( ignore )
        {
        while ( *Line_ptr != ignore && !Eof )
            {
            switch ( *Line_ptr )
                {
                case MOL:
                    read_line(0);
                    break;
                case EOL:
                    read_line(1);
                    break;
                case '\\':
                    back_slash++;
                    Line_ptr++;
                    break;
                default:
                    back_slash = 0;
                    Line_ptr++;
                    break;
                }
            }
        
        switch ( ignore )
            {
            case '\'':
                {
                Line_ptr++;
                
                if ( *Line_ptr == MOL )
                    read_line(0);
                    
                if ( *Line_ptr == '\'' )
                    Line_ptr++;
                
                ignore = FALSE;
                break;
                }    
            case '"':
                {
                if ( !(back_slash % 2))         /* even number of backslashes immediately before " */
                    ignore = FALSE;
                else
                    Line_ptr++;
                
                break;
                }
            case '*':
                {
                Line_ptr++;
                
                if ( *Line_ptr == MOL )
                    read_line(0);
                
                if ( *Line_ptr == '/' )
                    ignore = FALSE;
                
                break;
                }
            default:
                break;
            }

        if ( Eof )
            ignore = FALSE;

        }

    Line_ptr++;             /* first char past ignore delimeter */
    }

/*******************************************************************/
/*                             IDENTIFICATION                      */
/*    Function name: IGNORE_CHAR                                   */
/*    Purpose      : to determine if current char is a comment,    */
/*                     single, or double quote                     */
/*    Method       : test                                          */
/*                                                                 */
/*******************************************************************/
/*                             COMMUNICATION                       */
/*    Formal Paramters:                                            */
/*        NAME               TYPE                    USED FOR      */
/*              none                                               */
/*                                                                 */
/*    Global Variables Needed:                                     */
/*        NAME               TYPE                    USED FOR      */
/*      TRUE                 int                                   */
/*      FALSE                int                                   */
/*      *Line_ptr            char             line pointer         */
/*      MOL                  char               end of line buffer */
/*                                                                 */
/*    Return Value:                                                */
/*             TRUE if ignore / FALSE if not                       */
/*******************************************************************/

ignore_char()
    {
    int flag = FALSE;
    
    switch ( *Line_ptr )
        {
        case '\'':              /* single or double quotes */
        case '"':
            flag = TRUE;
            break;
        case '/':
            if ( *(Line_ptr + 1) == '*' )   /* comment */
                flag = TRUE;
            else if ( *(Line_ptr + 1) == MOL )         /* end of logical line */
                {
                read_line(0);
                if ( *Line_ptr == '*' )     /* comment */
                    flag = TRUE;
                }
            break;
        default:
            break;
        }
    
    return flag;
    }

/*******************************************************************/
/*                             IDENTIFICATION                      */
/*    Function name:  READ_LINE ( increment )                      */
/*    Purpose      :  read a line from the input file and increment*/
/*                        the line counter                         */
/*                                                                 */
/*    Method       :  get chars from file up to buffer size        */
/*                                                                 */
/*******************************************************************/
/*                             COMMUNICATION                       */
/*    Formal Paramters:                                            */
/*        NAME               TYPE                    USED FOR      */
/*      increment            int           if the previous read did*/
/*                                         not get the entire line,*/
/*          does not increment the line counter, if the entire line*/
/*          was read, does increment the line counter              */
/*                                                                 */
/*    Global Variables Needed:                                     */
/*        NAME               TYPE                    USED FOR      */
/*     Line_ptr              int             buffer pointer        */
/*     Input_file            int             file handle           */
/*     Line_buf              char            line buffer           */
/*     Eof                   int             end of file flag      */
/*     BUF_SIZE              int             length of line buffer */
/*     Line_count            int             physycal line counter */
/*     TRUE                  int                                   */
/*     FALSE                 int                                   */
/*                                                                 */
/*    Return Value:                                                */
/*                  none                                           */
/*******************************************************************/

read_line ( increment )
    int increment;
    {

    Line_count += increment;

    if ( fgets( Line_buf, BUF_SIZE, Input_file ) )
        Eof = FALSE;
    else
        Eof = TRUE;

    Line_ptr = Line_buf;

    if ( Output_file )
        {
        if ( increment )
            fprintf ( Output_file, "%5d   %s", Line_count, Line_buf);
        else
            fprintf ( Output_file, "%s", Line_buf);
        }

    }

/*******************************************************************/
/*                             IDENTIFICATION                      */
/*    Function name: IS_RESERVED ( function )                      */
/*    Purpose      : find if function is a reserved word           */
/*                                                                 */
/*    Method       : sequentially check reserved word list         */
/*                                                                 */
/*    Note         : the reserved word list is taken from page 180 */
/*          ' The C Programming Language ' by  Brian W. Kernighan  */
/*                                             Dennis M. Ritchie   */
/*                                                                 */
/*******************************************************************/
/*                             COMMUNICATION                       */
/*    Formal Paramters:                                            */
/*        NAME               TYPE                    USED FOR      */
/*     function              char                name of function  */
/*                                                                 */
/*    Global Variables Needed:                                     */
/*        NAME               TYPE                    USED FOR      */
/*      TRUE                 int                                   */
/*      FALSE                int                                   */
/*                                                                 */
/*    Return Value:                                                */
/*                TRUE if found / FALSE if not found               */
/*******************************************************************/

is_reserved( function )
    char *function;
    {
    int max = 28;           /* number of reserved words */

    static char reserved_list[28][8] = "int",     "char",    "float",
      "double",  "struct",  "union",   "long",    "short",   "unsigned",
      "auto",    "extern",  "register","typedef", "static",  "goto",
      "return",  "sizeof",  "break",   "continue","if",      "else",
      "for",     "do",      "while",   "switch",  "case",    "default",
      "entry";

    char flag = FALSE;      /* return value */
    int search;             /* counter */

    for ( search = 0; search < max; search++)
        if ( !strcmp( reserved_list [ search ] , function) )
            {
            flag = TRUE;
            break;
            }

    return flag;
    }

/*******************************************************************/
/*                             IDENTIFICATION                      */
/*    Function name: IS_LIBRARY ( word )                           */
/*    Purpose      : find if word is a library function            */
/*                                                                 */
/*    Method       : sequentially search library linked list       */
/*                                                                 */
/*******************************************************************/
/*                             COMMUNICATION                       */
/*    Formal Paramters:                                            */
/*        NAME               TYPE                    USED FOR      */
/*       word                char           word to be tested      */
/*                                                                 */
/*    Global Variables Needed:                                     */
/*        NAME               TYPE                    USED FOR      */
/*      FALSE                 int                                  */
/*      TRUE                  int                                  */
/*      Library               struct        library linked list    */
/*        ->Following         int           pointer                */
/*        ->Name              char          library function name  */
/*      Lib_head              struct        start of library list  */
/*                                                                 */
/*    Return Value:                                                */
/*                  TRUE if word is library functionelse FALSE     */
/*******************************************************************/

is_library( word )
    char *word;
    {
    int flag = FALSE;

    Library = &Lib_head;

    while ( Library->Following )
        {
        if ( !strcmp ( Library->Following->Name, word) )
            {
            flag = TRUE;
            break;
            }

        Library = Library->Following;
        }

    return ( flag );
    }

/*******************************************************************/
/*                             IDENTIFICATION                      */
/*    Function name: IS_FUNCTION                                   */
/*    Purpose      : to check if function is a function            */
/*                                                                 */
/*    Method       : next non white space char must be a '('       */
/*                                                                 */
/*******************************************************************/
/*                             COMMUNICATION                       */
/*    Formal Paramters:                                            */
/*        NAME               TYPE                    USED FOR      */
/*                  none                                           */
/*    Global Variables Needed:                                     */
/*        NAME               TYPE                    USED FOR      */
/*     *Line_ptr              int                   line pointer   */
/*     EOL                    char                  end of logical */
/*                                                     line        */
/*     Eof                    int                   end of file    */
/*     MOL                    char              end of line buffer */
/*     TRUE                   int                                  */
/*     FALSE                  int                                  */
/*                                                                 */
/*    Return Value:                                                */
/*             TRUE if function name / FALSE if not                */
/*******************************************************************/

is_function()
    {
    int flag = TRUE;

    while ( flag )
        {
        switch ( *Line_ptr )
            {
            case ' ':
            case '\t':
                Line_ptr++;
                break;
            case EOL:
                read_line(1);
                break;
            case MOL:
                read_line(0);
                break;
            default:
                flag = FALSE;
                break;
            }

        if ( Eof )
            flag = FALSE;
        }

    if ( *Line_ptr == '(' )
        flag = TRUE;
    else
        flag = FALSE;

    return flag;
    }

/*******************************************************************/
/*                             IDENTIFICATION                      */
/*    Function name: ADD_TO_LIST ( function )                      */
/*    Purpose      : add a function to list, with current line no. */
/*                                                                 */
/*    Method       : copy function name, find last number and add  */
/*                   if needed add new link to list                */
/*                                                                 */
/*******************************************************************/
/*                             COMMUNICATION                       */
/*    Formal Paramters:                                            */
/*        NAME               TYPE                    USED FOR      */
/*     function              char              name of function    */
/*                                                                 */
/*    Global Variables Needed:                                     */
/*        NAME               TYPE                    USED FOR      */
/*      TRUE                  int                                  */
/*      FALSE                 int                                  */
/*      NUL                   int                                  */
/*      MAX_LINES             int             number of lines/link */
/*                                                                 */
/*      S_function_name       struct          variable type        */
/*                                                                 */
/*      Function              struct          function name list   */
/*        ->Following         int             pointer              */
/*        ->Name              char            function name        */
/*        ->Too_many_lines    int             flag                 */
/*        ->First_line        int             pointer              */
/*        ->Curr_line         int             pointer              */
/*          ->Space_left      int             flag and counter     */
/*          ->Number[]        int             line number          */
/*                                                                 */
/*      Head                  struct          start of list        */
/*                                                                 */
/*      Blank_name            struct          next blank spot      */
/*        ->Following         int             pointer              */
/*        ->Too_many_lines    int             flag                 */
/*                                                                 */
/*      Blank_line            struct          next blank line      */
/*      Overflow              int             flag and counter     */
/*      Line_count            int             current line number  */
/*                                                                 */
/*    Return Value:                                                */
/*                     number                                      */
/*******************************************************************/

add_to_list( name )
    char *name;
    {
    static int number = 0;      /* counter for number of names found */
    int test = TRUE;               /* flag for found name in list compare */
    
    Function = &Head;

    while ( Function->Following )
        {
        if ( ( test = strcmp( Function->Following->Name, name )) >= 0 )
            break;

        Function = Function->Following;
        }

    if ( test )
        {
        if ( Blank_name && Blank_line )
            {
            number++;

            Blank_name->Following = Function->Following;
            Function->Following = Blank_name;
            strcpy( Function->Following->Name, name );
            Function->Following->First_line = Function->Following->Curr_line = Blank_line;
            Function->Following->Curr_line->Number [ MAX_LINES - Function->Following->Curr_line->Space_left ] = Line_count;
            Function->Following->Curr_line->Space_left--;

            if ( Blank_name = malloc ( sizeof ( struct S_function_name )))
                {
                Blank_name->Following = NUL;
                Blank_name->Too_many_lines = FALSE;
                }

            new_blank_line();
            }
        else
            Overflow++;
        }
    else
        {
        if ( Function->Following->Curr_line->Space_left )
            {
            Function->Following->Curr_line->Number [ MAX_LINES - Function->Following->Curr_line->Space_left ] = Line_count;
            Function->Following->Curr_line->Space_left--;
            }
        else
            {
            if ( Blank_line )
                {
                Function->Following->Curr_line = Function->Following->Curr_line->Following = Blank_line;
                Function->Following->Curr_line->Number [ MAX_LINES - Function->Following->Curr_line->Space_left ] = Line_count;
                Function->Following->Curr_line->Space_left--;
                new_blank_line();
                }
            else
                Function->Following->Too_many_lines = TRUE;
            }
        }
    return number;
    }

/*******************************************************************/
/*                             IDENTIFICATION                      */
/*    Function name: NEW_BLANK_LINE                                */
/*    Purpose      : get a new blank line structure                */
/*                                                                 */
/*    Method       : obtain a block of memory from the system      */
/*                                                                 */
/*******************************************************************/
/*                             COMMUNICATION                       */
/*    Formal Paramters:                                            */
/*        NAME               TYPE                    USED FOR      */
/*                 none                                            */
/*                                                                 */
/*    Global Variables Needed:                                     */
/*        NAME               TYPE                    USED FOR      */
/*     S_line_numbers        struct          variable type         */
/*                                                                 */
/*     Blank_line            struct          address of block      */
/*      .Space_left            int           flag                  */
/*      .Following             int           pointer               */
/*      .Number[]              int           holds line numbers    */
/*                                                                 */
/*    Return Value:                                                */
/*                     none                                        */
/*******************************************************************/

new_blank_line()
    {
    int count;          /* general purpose counter */

    if ( Blank_line = malloc ( sizeof ( struct S_line_numbers ) ) )
        {
        Blank_line->Space_left = MAX_LINES ;
        Blank_line->Following = NUL;

        for ( count = 0; count < MAX_LINES; count++)
            Blank_line->Number [ count ] = 0;
        }
    }

/*******************************************************************/
/*                             IDENTIFICATION                      */
/*    Function name: PRINT_LIST ( number , name )                  */
/*    Purpose      : prints list of function names and line numbers*/
/*                   to the screen. can be redirected              */
/*                                                                 */
/*    Method       : O/P to screen                                 */
/*                                                                 */
/*******************************************************************/
/*                             COMMUNICATION                       */
/*    Formal Paramters:                                            */
/*        NAME               TYPE                    USED FOR      */
/*       name               char            name of input file     */
/*                                                                 */
/*    Global Variables Needed:                                     */
/*        NAME               TYPE                    USED FOR      */
/*    TRUE                  int                                    */
/*    FALSE                 int                                    */
/*    MAX_LINES             int                max lines per link  */
/*                                                                 */
/*    Function              struct             function name list  */
/*      ->Following         int                pointer             */
/*      ->Name              char               function name       */
/*      ->Too_many_lines    int                flag                */
/*      ->First_line        int                pointer             */
/*        ->Number[]        int                line number list    */
/*        ->Following       int                pointer             */
/*                                                                 */
/*    Head                  struct             start of functions  */
/*                                                                 */
/*    Overflow              int                  too many names    */
/*    R_margin              int                  width of screen   */
/*    XR_resv               int                output reserved     */
/*                                                words            */
/*    XR_func               int                output function     */
/*                                                names            */
/*    XR_var                int                output variables    */
/*    XR_lib                int                library functions   */
/*    stderr                int                standard error      */
/*                                                handle           */
/*    Return Value:                                                */
/*                   none                                          */
/*******************************************************************/

print_list( number, name )
    int number;
    char *name;
    {
    int line_flag;              /* flag */
    int col;                    /* counter */
    int count;                  /* counter */
    int margin;                 /* current margin */
    int max_length = 0;         /* longest name */

    fputs("\nPrinting\n" , stderr );

    Function = &Head;

    while ( Function->Following )
        {
        count = strlen ( Function->Following->Name );
        max_length = ( count > max_length ) ? count : max_length;
        Function = Function->Following;
        }

    max_length += 2;

    puts("\n\nC Cross Reference Utility - version 1.10\n  by Wojtek Bok\n\n");
    puts("Cross Reference for:\n");

    if ( XR_resv )
        puts("            Reserved Words\n");

    if ( XR_lib )
        puts("            Library Functions\n");

    if ( XR_func )
        puts("            Function Names\n");

    if ( XR_var )
        puts("            User Variables\n");

    printf("\nFilename  :%15s\nReferences:%15d\n\n", strupr(name) , number + 1 );

    puts("    Name");
    for ( col = max_length; col > 0; col--)
        puts(" ");
    puts("Line Number\n\n");

    Function = &Head;

    while ( Function->Following )
        {
        fprintf( stderr, "  %6d \r", number-- );

        puts( Function->Following->Name );
        count = max_length - strlen( Function->Following->Name ) - 1; 
        puts (" ");
        while ( count-- )
            puts(".");

        margin = max_length;
        col = 0;
        line_flag = TRUE;

        while ( line_flag )
            {
            if ( margin + 6 >= R_margin )
                {
                puts("\n");
                for ( count = max_length; count > 0; count--)
                    puts(" ");
                margin = max_length;
                }
            printf("%6d", Function->Following->First_line->Number [ col++ ] );
            margin += 6;

            if ( col >= MAX_LINES )
                {
                Function->Following->First_line = Function->Following->First_line->Following;
                col = 0;
                }

            if ( !Function->Following->First_line || Function->Following->First_line->Number [ col ] == 0 )
                line_flag = FALSE;

            }

        if ( Function->Following->Too_many_lines )
            {
            if ( margin + 6 >= R_margin )
                {
                puts("\n");
                for ( count = max_length; count > 0; count--)
                    puts(" ");
                }
            puts("<OVER>");         /* too many line numbers */
            }

        puts("\n");

        Function = Function->Following;

        }

    if ( Overflow )
        {
        puts("\n\n ************ Ran out of memory during run ***********\n");
        printf(   "************ %6d functions not found   ***********\n", Overflow );
        }

    co(12);
    }

/* end of CXREF */
