	TITLE	DSK - Disk I/O Code
	PAGE	,132
;	Copyright(C) 1984, Zenith Data Systems Corporation
;
;		RESTRICTED RIGHTS LEGEND
;		------------------------
;	
;	    "Use, duplication, or disclosure by the
;	Government is subject to restrictions as set forth
;	in paragraph (b) (3) (B) of the Rights in Technical
;	Data and Computer Software clause in DAR
;	7-104.9(a).  Contractor/manufacturer is Zenith
;	Data Systems Corporation of Hilltop Road, St.
;	Joseph, Michigan 49085.
;
;

.XLIST
	INCLUDE ASCII.DEF
	INCLUDE DRIVERS.DEF
	INCLUDE FIXED.DEF
	INCLUDE IOCONFIG.DEF
	INCLUDE LOADER.DEF
	INCLUDE MSDOS.DEF
	INCLUDE MACRO.ASM
	INCLUDE Z150BIOS.DEF
	INCLUDE Z150ROM.DEF
	INCLUDE DEFDSK.ASM
	INCLUDE DEFZ207.ASM
.LIST

BIOS	SEGMENT BYTE PUBLIC 'BIOS'
	ASSUME	CS:BIOS,DS:BIOS,SS:BIOS

	PUBLIC	DECODE, DISK_READ, DISK_WRITE, DISK_WRITE_VERIFY
	PUBLIC	DOP17, DOP18, DSK, TIM_RESET


	EXTRN	DSK_TPTR:WORD
	EXTRN	DOP_TYPE:BYTE
	EXTRN	DOP_VERIFY:BYTE
	EXTRN	PACKET:DWORD
	EXTRN	NUM_FLOPPY:BYTE
	EXTRN	SUCESS:NEAR
	EXTRN	SEC_CNT:WORD
	EXTRN	OLD_SP1:WORD
	EXTRN	DRIVE:BYTE
	EXTRN	FLOPPY_BPBS:BYTE
	EXTRN	FIXED_BPBS:BYTE
	EXTRN	FIXED_FLAGS:BYTE
	EXTRN	FIRST_SEC:WORD
	EXTRN	ONE_DRIVE:BYTE
	EXTRN	NEW_DISK:BYTE
	EXTRN	DISK_MSG:BYTE
	EXTRN	PMSG:NEAR
	EXTRN	CF:NEAR
	EXTRN	TIM_FLG:BYTE
	EXTRN	DISK_BUF:BYTE
	EXTRN	OLD_SP2:WORD
	EXTRN	RETRY_COUNT:BYTE
	EXTRN	L_ERR:ABS
	EXTRN	T_ERR:BYTE
	EXTRN	ERRORC:NEAR
	EXTRN	TIM_TOD:DWORD
	EXTRN	DATE:WORD
	EXTRN	TIM_PTR:WORD
	EXTRN	DSK_TIMOUT:BYTE
	EXTRN	NUM_DISK:BYTE
	EXTRN	DRIVE_MAP:NEAR
	EXTRN	ERROR:NEAR
	EXTRN	TIMESTAMP_FLG:BYTE
	EXTRN	BIOS_DATE:WORD
	EXTRN	BIOS_EXTRA:NEAR

;
;	Storage areas for temporary variables:
;
PIO_FLAG	DB	0		; Physical I/O flag
DSK_TBL_PTR	DW	-1  		; Pointer into disk table
DSKFUNC_FLAG	DB	0		; DSKFUNC return flag
LADDR_FLAG	DB	0		; Load address update flag
LADDR		DW	?		; User transfer address
MS_DOS_PACKET	DB	22 DUP (?)	; Ske;eton MSDOS packet for DSK
DSK_PACKET	DD	?		; Storage for Z100 packet address


;  DSK - This routine provides the necessary code to perform all disk I/O.
;
;        The particular function to perform is passed in the AL register.
;
;  Call with:
;
;    The register pair ES:BX points to a parameter block(for all functions
;    except GBIOSVEC and  MAPDSK) with the following fields:
;
;      DSKPR_DRIVE (byte):
;         Logical drive number (0-MAXDSK allowed)
;      DSKPR_SECTOR (word):
;         Logical sector number
;            (0-drive_max allowed)
;         (On read/write track function, side flag:
;            0 - side zero, 1 - side one)
;      DSKPR_COUNT (word):
;	transfer count
;            (1-?? (total bytes must fit within the same segment))
;      DSKPR_BUFF (double word):
;         Address of buffer. The first word is the offset, the second word is
;         the segment.
;
;  Register AL has the function to perform
;   (AL) = DSK_RESET - Reset disk(home head)
;   (AL) = DSK_STATUS - Get disk status(on success AH=aux status,AL=status)
;   (AL) = DSK_STEPIN - Step in head
;   (AL) = DSK_READ - Read the specified sectors to the buffer addr
;   (AL) = DSK_WRITE - Write the specified sectors from the buffer addr
;   (AL) = DSK_VERIFY - Verify the specified sectors. (Not implemented yet)
;   (AL) = DSK_FORMAT - Format a track.
;           The write track command is used to write the data at DSK_BUFF.
;   (AL) = DSK_READTRK - Perform the read track command.
;   (AL) = DSK_GBIOSVEC - Get addr of disk vector used by the BIOS in ES:BX.
;   (AL) = DSK_MAPDSK - Maps logical in AH to physical in AL
;   (AL) = DSK_SETFDC - Set force disk has been changed flag(DSK_FDC)
;   (AL) = DSK_PREAD - Same as DSK_READ but do not apply BSEC value
;   (AL) = DSK_ASSIGN - Set up BSEC and FATS values in partition
;
;  Returns:
;   AX = Status of operation
;         (See DEFDSK.ASM)
;   CY = 0 - Operation was a success
;   CY = 1 - Operation failed (see AX)
;   For DSK_READ and DSK_WRITE:
;     DSKPR_COUNT = number of sectors not read/written
;     DSKPR_BUFF = Addr at end of read/write
;   For DSK_GBIOSVEC:
;     ES:BX -> vector of disk table addresses
;
;
;  All register are used.
;  Segment registers are perserved.
;  (DS must be set up before call)

DSK	PROC	NEAR
	
	MOV	CS:DSK_TBL_PTR,-1  	; Flag as invalid pointer

; Check if the function is legal

	CMP	AL,DSK_FMAX	; Is function legal ?
	JNA 	D0A 		;   Yes, keep going
	STC			;   No, turn on CY to show error
	MOV	AX,DSKST_FNERR	; Show function error
	RET			;  and return

; Check for special functions

D0A:	
	CMP	AL,DSK_GBIOSVEC	; Is function Get Disk Vector ?
	JE	D0A1		;   Yes, skip to it
	CMP	AL,DSK_DRVNUM
	JE	D0A3		; Get number of drives
	CMP	AL,DSK_MAPDSK	; Is function Map Disk ?
	JNE	D0B		;   No, must be regular function

	PAGE

;
; Map device function
;
;     AH = current drive name
;     If AH = 0FFH
;         returns ES:BX pointer to map table
;     Else
;         returns AL mapped from AH
;

	CMP	AH,0FFH		; Is subfunction, Get Pointer to table?
	JNE	DSKFMAP1	;   No, map a particular drive

; Get pointer to the map table

	PUSH	CS		; Get segment
	POP	ES
	MOV	BX,OFFSET DSKMAP ; Get offset
	RET			;  and return

; Map drive in AH to one in AL

DSKFMAP1:
	MOV	AL,AH		; Make index into table
	XOR	AH,AH
	MOV	SI,AX		; Make index available
	MOV	AL,BYTE PTR DSKMAP[SI+1] ; Map disk
	RET			;  and return

; Disk map table
;
;     First entry is count of entries that follow
;     each following entry contains the physical
;     drive to access when referencing that logical
;     number.
;

DSKMAP	DB	12,0,1,2,3,4,5,6,7,8,9,10,11



;
; Get BIOS disk vector function
;

D0A1:
	MOV	BX,OFFSET DSK_TPTR ; Get offset of vector
	PUSH	CS		; Get segment of vector
	POP	ES	
	XOR	AX,AX		; Show success(clear CY)
	RET			;   and return

;
; DRVNUM - AL = number of 5 inch drives
;	   AH = number of 8 inch drives
;

D0A3:
	MOV	AL,NUM_FLOPPY	; Return number of 5.25" floppies
	MOV	AH,0		; No 8" disks supported
	CLC
	RET

	PAGE
;
; Regular disk function
; AL has function number

D0B:
	CBW	 		; Extend AL to AX
	SHL	AX,1		; Compute word offset
	MOV	DI,AX		; Make avail as index
	MOV	WORD PTR CS:DSK_PACKET, BX	; Save Z100 request packet
	MOV	WORD PTR CS:DSK_PACKET+2,ES


;	Check if drive number is legal
;	DI has function

	MOV	AL,ES:DSKPR_DRIVE[BX]	; Get drive number
	CMP	AL,0FFH		; Is it the special drive?
	JE D0D			; Yes, then OK
	CMP	AL,MAX_FIXED+MAX_FLOPPY	; Is drive number in range?
	JB	D0D		; Yes, skip
D0C:
	STC			; No, illegal drive number
	MOV	AX,DSKST_DNERR	; Get error code
	JMP	DSK_EXIT

D0D:
	CBW			; Make into a word pointer
	SHL	AX,1
	MOV	SI,AX		; Make available as index
	ADD	SI,OFFSET DSK_TPTR
	MOV	SI,[SI]		; Get address of disk parameter table
	CMP	SI,-1		; Does table exist?
	JE	D0C		; No, show error and return
	MOV	CS:DSK_TBL_PTR,SI	; Save index into table

;	Check if drive type valid

	CMP	BYTE PTR DSK_TYPE[SI],DSK_TZ217	; Fixed disk?
	JZ	DISPATCHW	; Yes
	
	CMP	BYTE PTR DSK_TYPE[SI],DSK_TZ2075 ; 5.25" floppy?
	JZ	DISPATCHF	; Yes

;	Invalid disk type

	STC			; Flag error
	MOV	AX,DSKST_DTERR	; Get error code
	RET

DISPATCHW:
	JMP	WORD PTR DSKW1[DI]	; Jump to function

DSKW1:
	DW      OFFSET DSKWFRS	; Reset function
  	DW	OFFSET DSKWFST	; Status function
	DW	OFFSET DSKWFRD	; Read function
	DW	OFFSET DSKWFWR	; Write function
	DW	OFFSET DSKWFVF	; Verify function
	DW	OFFSET DSKWFFT	; Format function
	DW	OFFSET DSKWFSI	; Step in function
	DW	OFFSET DSKBDO 	; Read track function
	DW	OFFSET DSKBDO	; GBIOSVEC
	DW	OFFSET DSKBDO	; MAPDSK
	DW	OFFSET DSKBDO	; SETFDC
	DW	OFFSET DSKWPRD	; PREAD
	DW	OFFSET DSKBDO 	; ASSIGN
	DW	OFFSET DSKWPWR	; PWRITE
	
DISPATCHF:
	JMP	WORD PTR DSKF1[DI]	; Jump to function

DSKF1:
	DW      OFFSET DSKFRS	; Restore function
  	DW	OFFSET DSKFST	; Status function
	DW	OFFSET DSKFRD	; Read function
	DW	OFFSET DSKFWR	; Write function
	DW	OFFSET DSKFVF	; Verify function
	DW	OFFSET DSKBDO	; Format function
	DW	OFFSET DSKFSI	; Step in function
	DW	OFFSET DSKBDO	; Read track function
	DW	OFFSET DSKBDO	; GBIOSVEC
	DW	OFFSET DSKBDO	; MAPDSK
	DW	OFFSET DSKFFD	; SETFDC
	DW	OFFSET DSKFRD	; PREAD
	DW	OFFSET DSKBDO	; ASSIGN
	DW	OFFSET DSKFWR	; PWRITE

DSKWFST:			; Winchester Status
DSKWFFT:			; Winchester format track
DSKWFSI:			; Winchester step in
DSKBDO:				; Bad operation
	MOV	AX,DSKST_NIERR	; Not implemented error
	STC			; Flag error
	JMP	DSK_EXIT	; Done

;
;  Reset(home) function for winchester
;
;      Clears errors and moves head to track zero.
;
;  Call with:
;      ES:BX -> parameter block addr
;      SI -> disk table
;
;  Returns:
;      CY clear - no errors
;      CY set - errors
;

; Entry for Force Disk Change on floppy also.

DSKFFD:
DSKWFRS:
	SUB	AX,AX		; No errors on RESET
	JMP	DSK_EXIT	;   and return

;
;  Status function
;
;    Clear errors and get current status
;      (ES:BX has parameter block addr,SI has offset of disk parameter table)
;
;	4/23/82 - Returns AUXSTAT in AH if no errors
;
	
DSKFST:
	MOV	AH,DIO_STATUS		; ROM Status function
	INT	DISK_IO_INTR		; Get Z150 status in AL

;	Translate into Z100 type status

DSKFSTA:
	CMP	AL,TIME_OUT		; Disk timeout error
	JNZ	DSKFST1			; No
	MOV	AL,FDSNRD		; Make it 'NOT READY'
	JMP	SHORT DSKFST_EXIT	; Done
DSKFST1:
	CMP	AL,BAD_SEEK		; Seek error?
	JNZ	DSKFST2			; No
	MOV	AL,FDSSEK		; Make it 'SEEK ERROR'
	JMP	SHORT DSKFST_EXIT	; Done
DSKFST2:
	CMP	AL,BAD_CRC		; CRC error?
	JNZ	DSKFST3			; No
	MOV	AL,FDSCRC		; Make it 'CRC ERROR'
	JMP	SHORT DSKFST_EXIT	; Done
DSKFST3:
	CMP	AL,RECORD_NOT_FOUND	; Couldn't find sector?
	JNZ	DSKFST4			; No
	MOV	AL,FDSRNF		; Make it 'RECORD NOT FOUND'
	JMP	SHORT DSKFST_EXIT	; Done
DSKFST4:
	CMP	AL,WRITE_PROTECT	; Write protect violation?
	JNZ	DSKFST5			; No
	MOV	AL,FDSWPV		; Make it 'WRITE PROTECT'
	JMP	SHORT DSKFST_EXIT	; Done

;	No translatable status was returned

DSKFST5:
	MOV	AL,0			; Show all OK
DSKFST_EXIT:
	MOV	AH,ASMO			; Show motor on in AUX STATUS
	CLC				; No error
	JMP	DSK_EXIT

;
;  Restore(home) disk function
;
;    Clear errors and move head to track zero.
;      (ES:BX has parameter block addr,SI has offset of disk parameter table)
;
DSKFRS:
	MOV	AL,ES:DSKPR_DRIVE[BX]	; Get drive number
	CALL	DRIVE_MAP		; Is it valid?
	MOV	AX,FDSNRD		; Show not ready error in case...
	JC	DSKFRS2			; Exit
	MOV	BYTE PTR DSK_LTRK[SI],0	; Show last track 0
	MOV	AH,DIO_RESET		; ROM reset command
	INT	DISK_IO_INTR		; Reset disk system
	JC	DSKFRS1			; Jump if error
	MOV	BYTE PTR DSK_LOPT[SI],DSK_ORS	; Show last op was reset
	MOV	AX,24H			; Show track 0 found
DSKFRS2:
	CLC				; No error
	JMP	DSK_EXIT		; Done
DSKFRS1:
	CALL	DSKFST			; Translate error status
	SUB	AH,AH			; Clear high status byte
	STC				; Set CY for error
	JMP	DSK_EXIT 

;
;  Step in function
;
;      (ES:BX has parameter block addr; SI has offset of disk paramter table)
;

DSKFSI:
	TEST	BYTE PTR DSK_LOPT[SI],DSK_OSI+DSK_OFT+DSK_ORS	; Was last
					; operation a step in, format or restore?
	JNZ	DSKFSI1			; Yes, all OK
	MOV	AX,DSKST_ORERR		; No, get error code
	STC				; Flag error
	JMP	DSK_EXIT

DSKFSI1:
	INC	WORD PTR DSK_LTRK[SI]	; Bump track number
	MOV	BYTE PTR DSK_LOPT[SI],DSK_OSI	; Show last op
	CLC				; No error
	SUB	AX,AX			; Clear status
	JMP	DSK_EXIT

;
;  DSKWFRD - Read function for winchester
;  DSKFRD  - Read function for floppy
;  DSKWFWR - Write function for winchester
;  DSKFWR  - Write function for floppy
;  DSKWFVF - Verify function for winchester
;  DSKFVF  - Verify function for floppy
;
;      Reads/writes/verifies one or more sectors from the disk
;
;  Call with:
;      ES:BX -> parameter block addr
;      SI -> disk table
;
;  Returns:
;      CY clear - no errors
;      CY set - errors
;

DSKWFVF:
DSKFVF:
	MOV	DOP_TYPE,DIO_VERIFY	; Verify function
	MOV	DOP_VERIFY,0		; Just verify once
	MOV	BYTE PTR DSK_LOPT[SI],DSK_ORD	; Flag as a read op
	JMP	SHORT DRW1
DSKWPWR:
	MOV	PIO_FLAG,0FFH		; Physical write
	JMP	SHORT DSKWFWR
DSKWPRD:
	MOV	PIO_FLAG,0FFH		; Physical read
;	JMP	SHORT DSKWFRD

DSKWFRD:
DSKFRD:
	MOV	DOP_TYPE,DIO_READ	; Read operation
	MOV	BYTE PTR DSK_LOPT[SI],DSK_ORD
	JMP	SHORT DRW1
DSKWFWR:
DSKFWR:
	MOV	DOP_VERIFY,0		; No verify
	MOV	DOP_TYPE,DIO_WRITE	; Write operation
	MOV	BYTE PTR DSK_LOPT[SI],DSK_OWR
;	JMP	SHORT DRW1

DRW1:
	CALL	BUILD_PACKET		; Build MS-DOS request packet
	MOV	SI,CS			; Get packet segment
	MOV	ES,SI
	MOV	BX,OFFSET MS_DOS_PACKET	; Get packet offset
	MOV	WORD PTR CS:PACKET,BX	; Save packet address
	MOV	WORD PTR CS:PACKET+2,ES
	MOV	CS:DSKFUNC_FLAG,0FFH	; Flag as a DSKFUNC call
	MOV	LADDR_FLAG,0FFH		; Flag invalid load address in BX
	JMP	DOP0A			; Jump to disk I/O code








;	All DSKFUNC functions return here

DSK_EXIT:
	MOV	CS:PIO_FLAG,0		; Clear physical I/O flag
	MOV	CS:DSKFUNC_FLAG,0	; Clear DSKFUNC return flag
	MOV	SI,CS:DSK_TBL_PTR	; Get pointer to disk table
	PUSHF
	CMP	SI,-1			; Is table pointer valid?
	JZ	DSK_EXIT2		; No, skip
	MOV	CS:DSK_STA[SI],AX	; Store status
DSK_EXIT2:
	POPF
	LES	BX,CS:DSK_PACKET	; Get pointer to Z100 packet
	JC	DSK_EXIT3		; Jump if there was an error
	MOV	WORD PTR ES:DSKPR_COUNT[BX],0	; Show no remaining sectors
	JMP	SHORT DSK_EXIT4		; Skip
DSK_EXIT3:
	MOV	ES:DSKPR_COUNT[BX],CX	; Store remaining sectors
DSK_EXIT4:
	PUSHF				; Save error flag
	CMP	LADDR_FLAG,0FFH		; Does LADDR contain updated load address?
	JZ	DSK_EXIT5		; No, skip
	MOV	DI,LADDR		; Get updated load address
	MOV	ES:DSKPR_BUFF[BX],DI	; Store updated transfer address	
DSK_EXIT5:
	POPF				; Restore error flag
	RET













;
;	BUILD_PACKET - This routine takes a Z100 type request packet
;		  	and builds a skeleton MS-DOS type request packet
;			at CS:MS_DOS_PACKET.
;
;	Entry:	ES:BX - pointer to Z100 request packet
;
BUILD_PACKET:
	PUSH	DI			; Save DI
	PUSH	AX			; Save AX
	MOV	DI,OFFSET MS_DOS_PACKET	; Load pointer to MS-DOS packet
	MOV	AL,ES:DSKPR_DRIVE[BX]	; Get unit number
	MOV	CS:SRH_UNIT[DI],AL	; Store unit number
	MOV	AX,ES:DSKPR_SECTOR[BX]	; Get sector
	MOV	CS:CRW_START[DI],AX	; Store it
	MOV	AX,ES:DSKPR_COUNT[BX]	; Get sector count
	MOV	CS:CRW_CNT[DI],AX	; Store it
	MOV	AX,WORD PTR ES:DSKPR_BUFF[BX]	; Get DTA offset
	MOV	WORD PTR CS:CRW_TADDR[DI],AX	; Store it
	MOV	AX,WORD PTR ES:DSKPR_BUFF[BX+2]	; Get DTA segment
	MOV	WORD PTR CS:CRW_TADDR[DI+2],AX	; Store it
	MOV	AL,NUM_FLOPPY			; Get number of floppies
	CMP	BYTE PTR CS:SRH_UNIT[DI],AL	; Is this a floppy?
	JB	BP3			; Jump if so
	MOV	BYTE PTR CS:CRW_MBYTE[DI],0F8H	; Store fixed disk media byte

BP2:
	POP	AX
	POP	DI
	RET

;	This is a floppy disk, calculate the media byte

BP3:
	MOV	AL,CS:SRH_UNIT[DI]	; Get unit number
	CBW				; Make into a word offset
	SHL	AX,1
	ADD	AX,OFFSET DSK_TPTR	; Make into a pointer into disk table
	PUSH	SI			; Save SI
	MOV	SI,AX
	MOV	SI,[SI]			; Get address of table entry
	MOV	AL,0FCH			; Assume 9X1 floppy
	CMP	BYTE PTR CS:DSK_SPT[SI],9	; Were we right?
	JE	BP3A			; Yes, skip
	MOV	AL,0FEH			; Make it an 8X1 floppy
BP3A:
	TEST	BYTE PTR CS:DSK_FLAG[SI],01H	; Is it double sided?
	JZ	BP3B			; No, skip
	INC	AL			; Bump media byte to show 2 sides
BP3B:
	MOV	CS:CRW_MBYTE[DI],AL	; Store media byte
	POP	SI			; Restore SI
	JMP	SHORT BP2		; Done



DSK	ENDP






;**	DISK I/O - This routine will handle all disk I/O calls passed
;		   through the MS-DOS driver interface.
;
;	ENTRY:	ES:BX - pointer to DOS packet 
;		DS = CS
;
;	EXIT:	None
;

DISK_READ LABEL NEAR

	MOV	DOP_TYPE,DIO_READ	; Read operation
	JMP	SHORT DOP0A

DISK_WRITE_VERIFY LABEL NEAR
	MOV	DOP_VERIFY,1		; Write with verify
	JMP	SHORT DOP0

DISK_WRITE LABEL NEAR
	MOV	DOP_VERIFY,0		; Write no verify
DOP0:
	MOV	DOP_TYPE,DIO_WRITE	; Write operation
DOP0A:
	MOV	CX,ES:CRW_CNT[BX]	; Count of sectors
	CMP	CX,0			; If count = 0 then done
	JNZ	DOP2
	CMP	CS:DSKFUNC_FLAG,0FFH	; Were we called from DSKFUNC?
	JZ	DOP1A   		; Yes, return there
	JMP	SUCESS
DOP1A:
	JMP	DSK_EXIT

;	Set up SI to point to disk BPB table

DOP2:
	MOV	SEC_CNT,CX		; Save sector count
	MOV	OLD_SP1,SP		; Save stack for error exit
	MOV	AL,ES:SRH_UNIT[BX]	; Get unit and save
	CALL	DRIVE_MAP		; Convert to Z150 drive number
	JNC	DOP2A			; Jump if no error
	JMP	DISK_ERROR		; Oops
DOP2A:
	CMP	AL,NUM_FLOPPY
	JAE	DOP3			; Not a floppy, set up fixed disk

	MOV	DRIVE,AL


;	Address disk parameter table

	MOV	AL,ES:CRW_MBYTE[BX]	; Get the media byte
	AND	AL,3H			; Get the 9 X 512 bit and the 1 side bit
	MOV	AH,TYPE BPB_STRUC
	MUL	AH
	ADD	AX,OFFSET FLOPPY_BPBS
	MOV	SI,AX			; Get pointer to BPB table
	MOV	AL,BYTE PTR BPB_SPT[SI]

;	Set the floppy disk parameter table with sectors per track

	MOV	BYTE PTR DS:BIOS_EXTRA[BIOS_DPT+4],AL

	PUSH	CS
	POP	DS

	ASSUME	DS:BIOS

	JMP	SHORT DOP5

;	Set up SI to point to Fixed disk BPB

DOP3:
	SUB	AL,NUM_FLOPPY		; Index from 0
	MOV	AH,TYPE BPB_STRUC	; Get offset of correct table
	MUL	AH
	ADD	AX,OFFSET FIXED_BPBS
	MOV	SI,AX			; Get address of BPB

	MOV	AL,BPB_UNIT[SI]		; Get and set unit of partition
	MOV	DRIVE,AL
	
DOP5:
	MOV	CX,SEC_CNT		; Set up sector count in case of error

;	Check if last requested sector lies on this disk

	MOV	DX,ES:CRW_START[BX]

	IF	DT_STAMP
	MOV	CS:FIRST_SEC,DX		; Save first sector for later use
	ENDIF

	MOV	DI,DX			; Start sector
	ADD	DI,CX			; # of sectors
	JC	DOP7			; Bad parms, no such sector

	CMP	PIO_FLAG,0FFH		; Are we doing physical I/O?
	JZ	DOP8A			; Jump if so

DOP6:
	CMP	DI,BPB_SECS[SI]	; compare to last sector on disk
	JBE	DOP8			; Request OK

DOP7:
	MOV	AL,SRHS_ESNF		; Sector not found error
	CMP	CS:DSKFUNC_FLAG,0FFH	; DSKFUNC call?
	JNZ	DOP7A			; Skip if not
	MOV	AL,FDSRNF		; Flag as record not found
DOP7A:
	JMP	DISK_ERROR1

DOP8:
	ADD	DX,BPB_HIDDEN[SI]	; Add in hidden sectors
DOP8A:

;	Check for a one drive system and map if necessary

	MOV	AL,DRIVE		; Check for a fixed disk
	TEST	AL,80H
	JNZ	DOP10			; Not a floppy, no logical drives

	CMP	ONE_DRIVE,1
	JNE	DOP10			; Not a one drive system
	SUB	DI,DI


	MOV	AH,AL			; Save drive
	XCHG	DS:BYTE PTR BIOS_EXTRA[LAST_DRIVE],AH	; Fetch and update last accessed drive
	PUSH	CS			; Restore BIOS data
	POP	DS

	ASSUME	DS:BIOS

	CMP	AL,AH			; Check if accessing same drive
	JZ	DOP9			; Same drive, no mapping

;	Map the imaginary drive and get a disk from the user

	ADD	AL,'A'          	; Map to a drive letter
	MOV	NEW_DISK,AL		; Put in message

	PUSH	DX			; Ask user for a disk
	MOV	DX,OFFSET DISK_MSG
	CALL	PMSG
	POP	DX

	CALL	CF			; Flush input buffer
	SUB	AH,AH			; Wait for a character to be typed
	INT	KEYBOARD_IO_INTR
DOP9:
	MOV	DRIVE,0			; Physical drive 0

;	Calculate track, sector number and head

DOP10:
	CALL	DECODE			; Set necessary regs for call to DOP18
	MOV	DL,DRIVE		; Move drive to DL for ROM

;	Check if transfer lies within this 64K boundry for DMA

	LES	BX,ES:CRW_TADDR[BX]	; Get transfer address
DOP11:
	MOV	LADDR_FLAG,0		; Show valid load address
	MOV	AX,ES			; Get 16 bits of transfer address
	PUSH	CX
	MOV	CL,4
	SHL	AX,CL
	MOV	CL,9			; Prepare for next shift operation
	ADD	AX,BX
	NEG	AX			; Find remainder of this 64K
	JNZ	DOP12
	MOV	AH,80H			; Special case, full 64K available
	DEC	CL			; Correct shift count
DOP12:
	SHR	AX,CL			;  this 64K
	POP	CX
	JZ	DOP14			; If none left in this 64K, get 1 sector
	CMP	AX,SEC_CNT		; Can request be honored?
					;  AX <  SEC_CNT Can't service
					;  AX >= SEC_CNT Can service
	JB	DOP13
	MOV	AX,SEC_CNT

;	R/W the requested number of sectors

DOP13:
	CALL	DOP18			; R/W the sectors
	MOV	LADDR,BX		; Save load address
	JC	DISK_ERRORL
	CMP	SEC_CNT,0		; Update sector count
	JNZ	DOP14			; R/W across 64kb boundry

DISK_EXIT:
	IF	DT_STAMP

;	Since the date/time stamp routine uses DISK_BUF, we can't
;	do stamping until BIOS initialization has been completed,
;	and TIM_FLG has been set to 0.

	CMP	CS:TIM_FLG,0		; Are stamps allowed?
	JNZ	TIM_RESET		; No
	CALL	UPDATE			; Update date/time stamp on disk

	ENDIF

TIM_RESET:
	IF	D_2SEC

	MOV	DI,CS:WORD PTR TIM_PTR	; Get offset into timout table
	MOV	CS:WORD PTR TIM_PTR,0FFFH	; Make pointer invalid
	MOV	AL,NUM_FLOPPY		; Get number of floppies
	CBW
	SHL	AX,1
	CMP	DI,AX              	; Is it a floppy?
	JGE	DISK_EXIT1		; No
	MOV	CS:WORD PTR DSK_TIMOUT[DI],200	; Reset to 2 secs

DISK_EXIT1:
	ENDIF
	CMP	CS:DSKFUNC_FLAG,0FFH	; Were we called from DSKFUNC?
	JZ	DISK_EXIT2		; Yes, return there
	JMP	SUCESS			; All done!
DISK_EXIT2:
	JMP	DSK_EXIT		; Return to DSKFUNC routine exit

;	Need to R/W 1 sector to cross 64Kb boundry

DOP14:
	CLD				; Clear direction for later use
	PUSH	ES			; Save user buffer area
	PUSH	BX

	CMP	DOP_TYPE,DIO_READ
	JZ	DOP15			; Read one sector

;	Copy user data to internal buffer and write it

	PUSH	ES			; Set up segments for source and dest.
	PUSH	CS
	POP	ES
	POP	DS
	PUSH	SI			; Save disk table pointer
	MOV	SI,BX			; Set up source offset
	MOV	DI,OFFSET DISK_BUF	; Set up dest. offset
	MOV	BX,DI			; Set up X-fer offset for disk write

	PUSH	CX			; Save CX
	MOV	CX,SEC_SIZE/2		; Move 1 sector (512b) of data
	REP	MOVSW
	POP	CX			; Restore CX
	POP	SI			; Restore disk table pointer
	PUSH	CS			; Restore data segment
	POP	DS

	CALL	DOP17			; Write 1 sector
	MOV	LADDR,BX		; Save load address
	JC	DISK_ERRORL
	POP	BX			; Restore user buffer
	POP	ES
	ADD	BH,SEC_SIZE SHR 8
	JMP	SHORT DOP16

DISK_ERRORL:
	JMP	DISK_ERROR

;	Read 1 sector, transfer it to the users area and continue

DOP15:
	PUSH	CS			; Set up BIOS transfer address
	POP	ES
	MOV	BX,OFFSET DISK_BUF
	CALL	DOP17
	JC	DISK_ERRORL
	POP	DI			; restore user transfer address
	POP	ES

	PUSH	SI			; Save disk pointer table
	MOV	SI,OFFSET DISK_BUF
	PUSH	CX			; Save CX
	MOV	CX,SEC_SIZE/2		; Move 1 sector of data to user
	REP	MOVSW
	POP	CX			; restore CX
	POP	SI			; Restore disk table pointer

	MOV	BX,DI			; Update user address
	MOV	LADDR,BX		; Save load address
DOP16:
	CMP	SEC_CNT,0		; All done?
	JZ	DOP16A   
	JMP	DOP11
DOP16A:
	JMP	DISK_EXIT

;*	Read/Write sectors
;
;	Entry:
;		AX - count of sectors to R/W (1 if entering at DOP17)
;		CH - Track number
;		CL - Sector number
;		DL - Drive
;		DH - Head number
;		ES:BX - transfer address
;		DS:SI - BPB of disk
;		DOP_TYPE - Code to pass the ROM for read or write
;
;	Exit:
;		'C' set, a disk error has occured
;		'C' clear, operation sucessful
;		AX,
;		BX (updated load address),
;		CX (updated track and sector),
;		DH (updated head number),
;		DI.
;
DOP17:
	MOV	AX,1			; R/W 1 Sector
DOP18:
	MOV	OLD_SP2,SP		; Save old stack for quick error exit
	MOV	RETRY_COUNT,NUM_RETRY
	PUSH	AX			; Save count
	MOV	AL,BYTE PTR BPB_SPT[SI]	; Get sectors per track
	PUSH	CX
	AND	CL,3FH
	SUB	AL,CL			; How many sectors left on this track?
	POP	CX
	INC	AL
	CBW				; Clear high byte
	POP	DI			; Restore count
	CMP	DI,AX			; See if request will fit in this track	
	JAE	DOP19			; Yes, fulfill request
	MOV	AX,DI			; No, get remainder of this track

;	Do the actual disk read/write

DOP19:
	PUSH	AX			; Save count
	MOV	AH,DOP_TYPE		; Pass operation type
	INT	DISK_IO_INTR		; Call ROM
	JC	RETRY			; Retry on error
	POP	AX			; Restore count

;	Check for verify

	CMP	WORD PTR DOP_TYPE,1 SHL 8 + DIO_WRITE ; Check for verify-write
	JNZ	DOP20
	PUSH	AX
	MOV	AH,DOP_VERIFY		; Verify sectors
	INT	DISK_IO_INTR
	JC	RETRY
	POP	AX

DOP20:
	SUB	SEC_CNT,AX		; Update sector count

	MOV	AH,AL			; Copy sector count to AH
	SHL	AH,1			; Turn into 512 byte sectors
	ADD	BH,AH			; Update load address
	SUB	AH,AH			; Fix up sector count

	SUB	DI,AX			; Update count

;	Update head, track and sector

DOP21:
	PUSH	BX			; Free up a register
	MOV	BH,CL			; Get Cylinder in BX
	ROL	BH,1
	ROL	BH,1
	AND	BH,3
	MOV	BL,CH

	AND	CL,3FH			; Isolate sector bits
	ADD	CL,AL			; Get new sector to start at
	CMP	CL,BYTE PTR BPB_SPT[SI]	; Check if beyond end of track
	JBE	DOP22			; No, skip
	MOV	CL,1			; Past end, now at sector 1
	INC	DH
	CMP	DH,BYTE PTR BPB_HEADS[SI]
	JB	DOP22
	SUB	DH,DH			; Back to head zero
	INC	BX			; Next track
DOP22:
	ROR	BH,1			; Set up track and sector registers
	ROR	BH,1
	OR	CL,BH
	MOV	CH,BL
	POP	BX			; Restore transfer address

	CMP	DI,0			; Operation done?
	JNZ	DOP23
	CLC
	RET
DOP23:
	MOV	AX,DI			; Update remaining sector count
	JMP	DOP18			; Continue

RETRY:
	PUSH	AX			; Save return code
	MOV	AH,DIO_RESET		; Reset disk system
	INT	DISK_IO_INTR
	POP	AX			; Restore return
	CMP	AH,TIME_OUT		; Test for disk time out
	JZ	RETRY1
	DEC	RETRY_COUNT
	JZ	RETRY1
	POP	AX			; Restore count and
	JMP	DOP19			;  try again
RETRY1:
	MOV	SP,OLD_SP2		; Restore stack to entry conditions
	STC				; Show error
	RET

;	Retry error has occured. Get the ROM error code and map to
;	an MS-DOS error code.

DISK_ERROR:
	PUSH	CS			; Set up addressing
	POP	ES

	CMP	CS:DSKFUNC_FLAG,0FFH	; Is this a DSKFUNC call?
	JNZ	DISK_ERROR0		; No, skip

;	Decode error for Z100 status

	MOV	AL,AH			; Error code in AL
	CALL	DSKFSTA			; Translate to Z100 status
	MOV	CS:DSKFUNC_FLAG,0FFH	; This is still a DSKFUNC call
	PUSH	SI
	MOV	SI,DSK_TBL_PTR		; Get pointer to disk table
	MOV	AH,0			; Clear high status byte
	OR	AL,80H			; Show drive not ready
	MOV	WORD PTR CS:DSK_STA[SI],AX	; Store error status
	MOV	DSK_TBL_PTR,-1  	; Flag that status has already been saved
	POP	SI
	JMP	DISK_ERROR1		; Continue error processing

DISK_ERROR0:
	MOV	AL,AH			; Code in AL for scan
	MOV	CX,L_ERR		; Length of table
	MOV	DI,OFFSET T_ERR
	CLD
	REPNZ	SCASB

;	Note: the following construct is due to problems with MASM

	DEC	DI
	MOV	AL,[DI+L_ERR]	; Get the mapped error code
	INC	DI

;*	Disk error
;
;	Entry: AL - Error code
;	       SEC_CNT - Count of sectors not yet done
;
DISK_ERROR1:
	MOV	CX,SEC_CNT		; Return the number of sectors left to transfer
	MOV	SP,OLD_SP1		; Pop the stack to entry conditions
	CMP	CS:DSKFUNC_FLAG,0FFH	; Did we get here from DSKFUNC?
	JZ	DSKERR2			; If so, jump there
	JMP	ERRORC
DSKERR2:
	STC				; Show error
	JMP	DSK_EXIT		; Return to DSKFUNC exit routine


	IF	DT_STAMP
;	Routine to update date/time stamp on a disk.
;	If no update has been made for at least 2 min,
;	the current date/time is written to the boot loader
;	of the disk that was just accessed.
;
;	Entry:	DS:SI - Pointer to BPB for disk
;		DRIVE - Drive number for ROM
;		FIRST_SEC - First sector of previous disk operation
;
;	Exit:	None
;

UPDATE	PROC	NEAR
	CMP	CS:FIRST_SEC,MIN_SEC	; Is sector in range?
	JNC	UPDT_EXIT		; No
	CMP	TIMESTAMP_FLG,0		; Time to update?
	JZ 	NO_UPDT			; Time is not up yet
UPDT:
	MOV	WORD PTR TIM_TOD,12000	; Reset timer for 2 mins
	MOV	TIMESTAMP_FLG,0		; Reset flag
	MOV	CS:DOP_TYPE,DIO_READ	; Prepare to read loader sector
	CALL	RWSEC			; Read it
	JC	NO_UPDT			; Ignore if error
	CMP	CS:WORD PTR (DISK_BUF+3),'EZ'  ; Check loader version
	JNZ	NO_UPDT			; Wrong version
	CMP	CS:BYTE PTR (DISK_BUF+5),'N'   ; More version check
	JNZ	NO_UPDT			; Wrong version
	MOV	DI,OFFSET DISK_BUF	; Point to disk buffer
	ADD	DI,LOADER_DATE_OFFSET	; Point to date area
	PUSH	SI			; Save SI
	MOV	SI,OFFSET BIOS_DATE	; Point to date and time
	MOV	CX,7			; 7 bytes to transfer
	PUSH	ES
	PUSH	CS			; Set ES = CS
	POP	ES
	REPNZ	MOVSB			; Move the date
	POP	ES			; Restore ES
	POP	SI
	MOV	CS:DOP_TYPE,DIO_WRITE	; Prepare to re-write sector
	CALL	RWSEC			; Write it
UPDT_EXIT:
NO_UPDT:
	RET
UPDATE	ENDP


;	Routine to read or write sector 0 (boot loader) on the last 
;	drive accessed.
;
;	Called from UPDATE

RWSEC 	PROC	NEAR
	PUSH	ES			; Save pointer to DOS packet
	PUSH	BX
	PUSH	CS			; Load ES with code segment pointer
	POP	ES
	SUB	DX,DX			; Sector 0
	ADD	DX,BPB_HIDDEN[SI]	; Add hidden sectors
	CALL	DECODE			; Get head, sector and cylinder
	MOV	DL,CS:DRIVE		; Drive number for ROM
	MOV	BX,OFFSET DISK_BUF	; Set transfer address
	CALL	DOP17			; Perform the I/O
	POP	BX			; Restore pointer to DOS packet
	POP	ES
	RET
RWSEC	ENDP

	ENDIF


;	DECODE - This routine will take a MS-DOS logical sector number
;		and decode it into head sector and cylinder for the
;		given disk.
;
;	ENTRY:	DX - Logical sector number
;		DS:SI - Pointer to BPB for this disk
;
;	EXIT:	CH - Cylinder number
;		CL - Sector number
;		DH - Head
;
;	USES:	AX
;
DECODE	PROC	NEAR

	PUSH	BX
	MOV	BX,DX
	SUB	AX,AX
	XCHG	AX,DX			; AX = Start sector
	DIV	WORD PTR BPB_SPT[SI]	; divide by sectors per track
	INC	DL			; DL = Sector #, AX = track
	MOV	CL,DL			; Move sector into CL for ROM
	SUB	DX,DX
	DIV	WORD PTR BPB_HEADS[SI]
	MOV	DH,DL			; Put head into DH for ROM
	AND	AH,3			; Ensure only two high bits for cyl.
	ROR	AH,1			; Move high cyl. bits to left side of byte
	ROR	AH,1
	OR	CL,AH			; Set sector and cyl. high bits
	MOV	CH,AL			; Low part of cyl. for ROM
	MOV	DL,BL			; Restore DL
	POP	BX
	RET

DECODE	ENDP

BIOS	ENDS
	END
