#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "DZasm.h"
#include "avltree.h"

extern FILE		*infile;
extern DZarea		*gExtern;			/* list	of extern areas	*/
extern DZarea		*gAreas;			/* list	of areas currently examined */
extern avltree		*gLabelRef;			/* Binary tree of program labels */
extern avltree		*gDataRef;			/* Binary tree of address references */
extern char		*gAreaTypes[];			/* text	identifiers for	area type enumerations */
extern char		Z80codeflnm[64];		/* name	of Z80 code file */
extern char		*gIncludeFiles[];		/* string array	of include filename mnemonics */
extern enum truefalse	gIncludeList[];			/* array of 'touched' (true) include file references */

extern long		Org, Codesize, gEndOfCode;


enum symbols		{       space, strconq, dquote, squote, semicolon, comma, fullstop, lparen, lcurly, rcurly, rparen,
				plus, minus, multiply, divi, mod, power, assign, bin_and, bin_or, bin_xor, less,
				greater, log_not, constexpr, newline, lessequal, greatequal, notequal, name, number,
				decmconst, hexconst, binconst, charconst, negated, nil
			};

enum symbols		sym;
enum symbols		ssym[] = {
				space, strconq,	dquote,	squote,	semicolon, comma, fullstop,
				lparen,	lcurly,	rcurly,	rparen,	plus, minus, multiply, divi, mod, power,
				assign,	bin_and, bin_or, bin_xor, less,	greater, log_not, constexpr
			 };

char			separators[] = " &\"\';,.({})+-*/%^=~|:<>!#";
char			ident[255];
short			counter;
FILE			*collect = NULL;

void			GenCollectFile(void);
void			ExtAddrItem(LabelRef  *itemptr);
void			LocAddrItem(LabelRef  *itemptr);
void			ExtDataItem(DataRef  *itemptr);
void			LocDataItem(DataRef  *itemptr);
void			ListAreas(DZarea  *currarea);
void			ReadCollectFile(void);
void			GetAreas(DZarea	 **arealist);
void			GetLabelRef(avltree  **avlptr, enum truefalse  scope);
void			GetDataRef(avltree  **avlptr, enum truefalse  scope);
void			CreateDataRef(avltree  **avlptr, long  labeladdr, enum truefalse  scope);
void			CreateLabelRef(avltree	**avlptr, long	labeladdr, enum	truefalse  scope);
void			ListIncludeFiles(void);
void			GetIncludeFiles(void);
enum atype		GetAreaType(char  *ident);
enum symbols		GetSym(void);
long			GetConstant(void);
int			InsertArea(struct area **arealist, long	 startrange, long  endrange, enum atype	 t);
int			CmpAddrRef(LabelRef *key, LabelRef *node);
int			CmpDataRef(DataRef *key, DataRef *node);
LabelRef		*InitLabelRef(unsigned short  label);
DataRef			*InitDataRef(unsigned short  label);


/* Create 'collect' file which contains:
 *	1. External label references
 *	2. Local label references
 *	3. External data references
 *	4. Local Data references
 *	5. Local areas
 *	6. External areas
 */
void	GenCollectFile(void)
{
	collect	= fopen("collect","w");
	if (collect == NULL) {
		puts("Couldn't create collect file.");
		return;
	}

	puts("Creating Collect file...");

	fprintf(collect, "; File:\n{%s}\n\n", Z80codeflnm);

	fputs("; Local Areas:\n", collect);
	ListAreas(gAreas);
	fputs("\n", collect);

	fputs("; External Areas:\n", collect);
	ListAreas(gExtern);
	fputs("\n", collect);

	fputs("; Local Label References:\n{", collect);
	counter	= 0;
	if (gLabelRef != NULL) inorder(gLabelRef, (void	 (*)())	LocAddrItem);
	fputs("\n}\n\n", collect);

	fputs("; External Label References:\n{", collect);
	counter	= 0;
	if (gLabelRef != NULL) inorder(gLabelRef, (void	 (*)())	ExtAddrItem);
	fputs("\n}\n\n", collect);

	fputs("; Local Data References:\n{", collect);
	counter	= 0;
	if (gDataRef !=	NULL) inorder(gDataRef,	(void  (*)()) LocDataItem);
	fputs("\n}\n\n", collect);

	fputs("; External Data References:\n{",	collect);
	counter	= 0;
	if (gDataRef !=	NULL) inorder(gDataRef,	(void  (*)()) ExtDataItem);
	fputs("\n}\n\n", collect);

	fputs("; Include files referenced in code:\n", collect);
	ListIncludeFiles();

	puts("Collect file created.");
	fclose(collect);
}


void	ExtAddrItem(LabelRef	*itemptr)
{
	if (itemptr->local == false) {
		if (counter++ %	8 == 0)
			fprintf(collect, "\n$%04X", itemptr->addr);
		else
			fprintf(collect, ", $%04X", itemptr->addr);
	}
}


void	LocAddrItem(LabelRef	*itemptr)
{
	if (itemptr->local == true) {
		if (counter++ %	8 == 0)
			fprintf(collect, "\n$%04X", itemptr->addr);
		else
			fprintf(collect, ", $%04X", itemptr->addr);
	}
}


void	ExtDataItem(DataRef    *itemptr)
{
	if (itemptr->local == false && itemptr->referenced == true) {
		if (counter++ %	8 == 0)
			fprintf(collect, "\n$%04X", itemptr->addr);
		else
			fprintf(collect, ", $%04X", itemptr->addr);
	}
}


void	LocDataItem(DataRef    *itemptr)
{
	if (itemptr->local == true && itemptr->referenced == true) {
		if (counter++ %	8 == 0)
			fprintf(collect, "\n$%04X", itemptr->addr);
		else
			fprintf(collect, ", $%04X", itemptr->addr);
	}
}


void	ListAreas(DZarea  *currarea)
{
	short	 items = 0;

	fputs("{", collect);

	while(currarea != NULL)	{
		if (items++ % 4	== 0)
			fprintf(collect, "\n($%04X, $%04X %s)",
				currarea->start, currarea->end,	gAreaTypes[currarea->areatype]);
		else
			fprintf(collect, ", ($%04X, $%04X %s)",
				currarea->start, currarea->end,	gAreaTypes[currarea->areatype]);

		currarea = currarea->nextarea;
	}

	fputs("\n}\n", collect);
}


/* Read	data from collect file to establish previous parsing of	code */
void	ReadCollectFile(void)
{
	char		*cptr;
	int		c;

	collect	= fopen("collect", "r");
	if (collect == NULL) {
		puts("Collect file not found.");
		return;
	}
	else
		puts("Reading collect file...");

	cptr = ident;
	while(GetSym() != lcurly);		/* Get to start	of name	*/
	while((c=fgetc(collect)) != '}') {      /* read filename of collect file */
		*cptr++	= c;
	}
	*cptr =	'\0';
	printf("Filename: %s\n", ident);
	strcpy(Z80codeflnm, ident);		/* store to global file	name */

	infile = fopen(Z80codeflnm,"rb");	/* get Z80 code	*/
	if (infile == NULL) {
		printf("'%s' not found.\n", Z80codeflnm);
		fclose(collect);
		return;
	}

	puts("Reading local areas...");
	GetAreas(&gAreas);

	Org = gAreas->start;
	fseek(infile, 0L, SEEK_END);	/* file	pointer	to end of file */
	Codesize = ftell(infile);
	gEndOfCode = Org+Codesize-1;

	puts("Reading extern areas...");
	GetAreas(&gExtern);

	puts("Reading local label references...");
	GetLabelRef(&gLabelRef,	true);

	puts("Reading extern label references...");
	GetLabelRef(&gLabelRef,	false);

	puts("Reading local data references...");
	GetDataRef(&gDataRef, true);

	puts("Reading extern data references...");
	GetDataRef(&gDataRef, false);

	puts("Reading Include file references...");
	GetIncludeFiles();

	fclose(collect);
}


void	GetLabelRef(avltree  **avlptr, enum truefalse  scope)
{
	long		labeladdr;

	while(GetSym() != lcurly);	/* Get to start	of label references */

	GetSym();			/* get first label reference */
	while(sym != rcurly) {
		if (sym==hexconst || sym==decmconst) {
			labeladdr = GetConstant();
			CreateLabelRef(avlptr, labeladdr, scope);
			if (GetSym() ==	comma) GetSym();       /* Read past comma */
		}
		else {
			puts("Label reference address not identified.");
			break;
		}
	}
}


void	GetDataRef(avltree  **avlptr, enum truefalse  scope)
{
	long		dataaddr;

	while(GetSym() != lcurly);	/* Get to start	of data	references */

	GetSym();			/* get first data reference */
	while(sym != rcurly) {
		if (sym==hexconst || sym==decmconst) {
			dataaddr = GetConstant();
			CreateDataRef(avlptr, dataaddr,	scope);
			if (GetSym() ==	comma) GetSym();       /* Read past comma */
		}
		else {
			puts("Data reference address not identified.");
			break;
		}
	}
}


void	CreateLabelRef(avltree	**avlptr, long	labeladdr, enum	truefalse  scope)
{
	LabelRef	*newref;

	newref = InitLabelRef(labeladdr);
	if (newref == NULL) {
		puts("No room");
		return;
	}

	newref->local =	scope;
	newref->referenced = true;
	insert(avlptr, newref, (int (*)()) CmpAddrRef);
}


void	ListIncludeFiles(void)
{
	int	i;

	fputs("{\n", collect);

	for (i = 1; i<=20; i++)	{
		if (gIncludeList[i] == true)
			fprintf(collect, "\t1 ; '%s.def'\n", gIncludeFiles[i]);
		else
			fprintf(collect, "\t0 ; '%s.def'\n", gIncludeFiles[i]);
	}

	fputs("}\n\n", collect);
}


void	GetIncludeFiles(void)
{
	int	i;

	while(GetSym() != lcurly);	/* Get to start	of Include file	references */

	for (i = 1; i<=20; i++)	{
		if (GetSym() ==	rcurly)
			break;
		else {
			if ((sym == decmconst) || (sym == hexconst)) {
				if (GetConstant() == 1)
					gIncludeList[i]	= true;
				else
					gIncludeList[i]	= false;
			}
			else {
				puts("Unknown parameter.");
				break;
			}
		}
	}
}


void	CreateDataRef(avltree  **avlptr, long  labeladdr, enum truefalse  scope)
{
	DataRef	       *newref;

	newref = InitDataRef(labeladdr);
	if (newref == NULL) {
		puts("No room");
		return;
	}

	newref->local =	scope;
	newref->referenced = true;
	insert(avlptr, newref, (int (*)()) CmpDataRef);
}


void	GetAreas(DZarea	 **arealist)
{
	long		start, end;
	enum atype	areaid;

	while(GetSym() != lcurly);		/* Get to start	of local areas */

	GetSym();				/* Read	( */
	while(sym != rcurly) {
		if (sym	== lparen) {
			GetSym();
			if (sym==hexconst || sym==decmconst) {
				start =	GetConstant();
				if (GetSym() ==	comma) GetSym();	/* read	past ; */
				if (sym==hexconst || sym==decmconst) {
					end = GetConstant();
					if (GetSym() ==	name) {
						areaid = GetAreaType(ident);
						if (areaid != notfound)
							InsertArea(arealist, start, end, areaid);
						else {
							puts("Area name not recognized.");
							break;
						}
					}
					else {
						puts("Missing area name.");
						break;
					}

					if (GetSym() ==	rparen)	{
						if (GetSym() ==	comma) GetSym();       /* Read past comma */
					}
					else {
						puts("Missing ).");
						break;
					}
				}
			}
			else {
				puts("Illegal Constant.");
				break;
			}
		}
		else {
			puts("Area start not identified.");
			break;
		}
	}
}


enum atype	GetAreaType(char  *ident)
{
	enum atype	a;

	for(a =	vacuum;	a<notfound; a++)
		if (stricmp(ident, gAreaTypes[a]) == 0)	return a;

	return notfound;
}


void	Skipline(void)
{
	int	c;

	while ((c=fgetc(collect)) != '\n' && c!=EOF); /* get to	beginning of next line... */
}


enum symbols		GetSym(void)
{
	char		       *instr;
	int			c, chcount = 0;

	ident[0] = '\0';

	for (;;) {              /* Ignore leading white spaces, if any... */
		c = fgetc(collect);
		if ((c == EOF) || (c ==	'\x1A')) {
			sym = newline;
			return newline;
		}
		else {
			if (!isspace(c)) break;
		}
	}

	instr =	strchr(separators, c);
	if (instr != NULL) {
		sym = ssym[instr - separators];	/* index of found char in separators[] */
		switch(sym) {
			case	newline:	GetSym();
						return sym;

			case	semicolon:	while(sym == semicolon)	{
							Skipline();	/* ignore comment line,	prepare	for next line */
							GetSym();
						}
						return sym;

			default:		return sym;
		}
		
	}

	ident[chcount++] = (char) toupper(c);
	switch (c) {
		case '$':
			sym = hexconst;
			break;

		case '@':
			sym = binconst;
			break;

		case '#':
			sym = name;
			break;

		default:
			if (isdigit(c))
				sym = decmconst;	/* a decimal number found */
			else {
				if (isalpha(c))
					sym = name;	/* an identifier found */
				else
					sym = nil;	/* rubbish ... */
			}
			break;
	}

	/* Read	identifier until space or legal	separator is found */
	if (sym	== name) {
		for (;;) {
			c = fgetc(collect);
			if ((!iscntrl(c)) && (strchr(separators, c) == NULL)) {
				if (!isalnum(c)) {
					if (c != '_') {
						sym = nil;
						break;
					} else
						ident[chcount++] = '_';	/* underscore in identifier */
				} else
					ident[chcount++] = (char) toupper(c);
			} else {
				ungetc(c, collect);  /*	puch character back into stream	for next read */
				break;
			}
		}
	} else
		for (;;) {
			c = fgetc(collect);
			if (!iscntrl(c)	&& (strchr(separators, c) == NULL))
				ident[chcount++] = c;
			else {
				ungetc(c, collect);  /*	puch character back into stream	for next read */
				break;
			}
		}

	ident[chcount] = '\0';
	return sym;
}


long	GetConstant(void)
{
	long			size, l, intresult = 0;
	unsigned short		bitvalue = 1;

	if ((sym != hexconst) && (sym != binconst) && (sym != decmconst)) {
		return -1;		/* syntax error	- illegal constant definition */
	}
	size = strlen(ident);
	if (sym	!= decmconst)
		if ((--size) ==	0) {
			return -1;	/* syntax error	- no constant specified	*/
		}
	switch (ident[0]) {
		case '@':
			if (size > 8) {
				return -1;     /* max 8	bit */
			}
			for (l = 1; l <= size; l++)
				if (strchr("01", ident[l]) == NULL) {
					return -1;
				}
			/* convert ASCII binary	to integer */
			for (l = size; l >= 1; l--) {
				if (ident[l] ==	'1')
					intresult += bitvalue;
				bitvalue <<= 1;	       /* logical shift	left & 16 bit 'adder' */
			}
			return (intresult);

		case '$':
			for (l = 1; l <= size; l++)
				if (isxdigit(ident[l]) == 0) {
					return -1;
				}
			sscanf((char *)	(ident + 1), "%lx", &intresult);
			return (intresult);

		default:
			for (l = 0; l <= (size - 1); l++)
				if (isdigit(ident[l]) == 0) {
					return -1;
				}
			return (atol(ident));
	}
}
