#include <stdio.h>
#include "DZasm.h"
#include "avltree.h"

FILE			*asmfile = NULL;

extern long		Org, Codesize;
extern char		*gIncludeFiles[];
extern enum truefalse	gIncludeList[];
extern DZarea		*gExtern;		/* list	of extern areas	*/
extern DZarea		*gAreas;		/* list	of areas currently examined */
extern avltree		*gLabelRef;		/* Binary tree of program labels */
extern avltree		*gDataRef;		/* Binary tree of address references */

static int		CmpDataRef(unsigned short *key,	DataRef	*node);
static int		CmpAddrRef(unsigned short *key,	LabelRef *node);

unsigned char		getbyte(unsigned short pc);
unsigned short		Disassemble(char *str, unsigned	short pc, enum truefalse  dispaddr);
void			DZpass2(void);
void			Z80Asm(unsigned	short  pc, unsigned short  endarea);
void			VoidOutput(long	 pc, unsigned short  endarea);
void			DefwOutput(long	 pc, unsigned short  endarea);
void			DefmOutput(long	 pc, unsigned short  endarea);
void			DeclExtProg(void);
void			DeclExtData(void);
void			VoidAsciiOutput(unsigned short	pc);
void			XrefAddrItem(LabelRef  *itemptr);
void			XrefDataItem(DataRef  *itemptr);


/* Create assembler source (with labels) from program and data areas */
void	DZpass2(void)
{
	DZarea		*currarea;
	enum files	ifile;

	puts("Generating assembler source...");

	asmfile	= fopen("out.asm", "w");
	if (asmfile == NULL) {
		puts("Couldn't open assembler output file.");
		return;
	}
	
	fputs("MODULE out\n\n",	asmfile);

	DeclExtProg();			/* XREF	for all	external program references */
	DeclExtData();			/* XREF	for all	external data references */

	fprintf(asmfile, "\nORG $%04X\n\n", Org);

	for (ifile = stdio; ifile <= intrrupt;	ifile++) {
		if (gIncludeList[ifile]	== true)
			fprintf(asmfile, "\tINCLUDE \"#%s_def\"\n", gIncludeFiles[ifile]);
	}
	fputc('\n', asmfile);

	currarea = gAreas;		/* point at first list */
	while(currarea != NULL)	{
		switch(currarea->areatype) {
			case	program:
					Z80Asm(currarea->start,	currarea->end);
					break;

			case	vacuum:
					VoidOutput(currarea->start, currarea->end);
					break;

			case	addrtable:
					DefwOutput(currarea->start, currarea->end);
					break;

			case	string:
					DefmOutput(currarea->start, currarea->end);
					break;
		}

		currarea = currarea->nextarea;
	}

	fclose(asmfile);
	puts("Assembler Source Generation completed.");
}


void	VoidOutput(long	 pc, unsigned short  endarea)
{
	unsigned short		column,	offset;
	enum truefalse		exitdump;
	DataRef		       *foundlabel;

	exitdump = false;
	fprintf(asmfile, "; %04Xh\n", pc);

	do {
		for(column=0; column<16; column++) {
			if (pc+column >	endarea) {
				if (column != 0) VoidAsciiOutput(pc);
				exitdump = true;
				break;
			}

			offset = pc+column;
			if ((foundlabel	= find(gDataRef, &offset, (int (*)()) CmpDataRef)) != NULL) {
				if (column != 0) VoidAsciiOutput(pc);
				fprintf(asmfile, ".L_%04X\t; Data reference\n",	foundlabel->addr);
				pc = foundlabel->addr;
				column=0;
			}

			if (column == 0)
				fprintf(asmfile, "\t\t\t\tDEFB $%02X", getbyte(pc));
			else
				fprintf(asmfile, ",$%02X", getbyte(pc +	column));
		}
		if (exitdump ==	false) {
			VoidAsciiOutput(pc);
			pc += 16;
		}
	}
	while(exitdump == false);

	fputc('\n', asmfile);
}


void	VoidAsciiOutput(unsigned short	pc)
{
	unsigned short	column,	b;

	fprintf(asmfile, "\t; ");
	for(column=0; column<16; column++) {
		b = getbyte(pc + column);
		fprintf(asmfile, "%c", (b>=32 && b<=127) ? b : '.' );
	}
	fputc('\n', asmfile);
}


static int	CmpDataRef(unsigned short *key,	DataRef	*node)
{
	return *key - node->addr;
}


static int	CmpAddrRef(unsigned short *key,	LabelRef *node)
{
	return *key - node->addr;
}


void	DefwOutput(long	 pc, unsigned short  endarea)
{
	unsigned short		pointer;
	DataRef			*foundlabel;

	while(pc < endarea) {
		pointer	= pc;
		foundlabel = find(gDataRef, &pointer, (int (*)()) CmpDataRef);
		if (foundlabel != NULL)	fprintf(asmfile, ".L_%04X ; Data reference\n", foundlabel->addr);
		pointer	= getbyte(pc++);
		pointer	+= getbyte(pc++) * 256;
		fprintf(asmfile, "\t\t\t\tDEFW L_%04X\n", pointer);
	}
}


void	DefmOutput(long	 pc, unsigned short  endarea)
{
	enum truefalse		asciistring, startline;
	unsigned short		pointer, byte;
	DataRef			*foundlabel;

	asciistring = false;
	startline = true;

	do {
		pointer	= pc;
		foundlabel = find(gDataRef, &pointer, (int (*)()) CmpDataRef);
		if (foundlabel != NULL)	{
			startline = true;
			if (asciistring	== true) {
				asciistring = false;
				fprintf(asmfile, "\"\n.L_%04X\n\t\t\t\tDEFM ", foundlabel->addr);
			}
			else {
				fprintf(asmfile, "\n.L_%04X\n\t\t\t\tDEFM ", foundlabel->addr);
			}
		}
		byte = getbyte(pc++);
		if (byte >= 32 && byte < 127) {
			if (asciistring	== true)
				fputc(byte, asmfile);
			else {
				asciistring = true;
				if (startline == true) {
					startline = false;
					fprintf(asmfile, "\"%c", byte);
				}
				else
					fprintf(asmfile, " & \"%c", byte);
			}
		}
		else {
			if (asciistring	== true) {
				asciistring = false;
				fprintf(asmfile, "\" & $%02X", byte);
			}
			else {
				if (startline == true) {
					startline = false;
					fprintf(asmfile, "$%02X", byte);
				}
				else
					fprintf(asmfile, " & $%02X", byte);
			}
		}
	}
	while(pc <= endarea);

	if (asciistring	== true)
		fprintf(asmfile, "\"\n\n");
	else
		fputs("\n\n", asmfile);
}


/* XREF	for all	external program references */
void	DeclExtProg(void)
{
	fprintf(asmfile, "\n; External Call References:\n");
	if (gLabelRef == NULL)
		fputs("; None.\n", asmfile);
	else
		inorder(gLabelRef, (void  (*)()) XrefAddrItem);

	fputs("\n\n",asmfile);
}


/* XREF	for all	external data references */
void	DeclExtData(void)
{
	fprintf(asmfile, "\n; External Data References:\n");
	if (gDataRef ==	NULL)
		fputs("; None.\n", asmfile);
	else
		inorder(gDataRef, (void	 (*)())	XrefDataItem);

	fputs("\n\n",asmfile);
}


void	XrefAddrItem(LabelRef  *itemptr)
{
	static short	items =	0;

	if (itemptr->local != true) {
		if (items++ % 8	== 0)
			fprintf(asmfile, "\nXREF L_%04X", itemptr->addr);
		else
			fprintf(asmfile, ", L_%04X", itemptr->addr);
	}
}


void	XrefDataItem(DataRef  *itemptr)
{
	static short	items =	0;

	if (itemptr->local != true) {
		if (items++ % 8	== 0)
			fprintf(asmfile, "\nXREF L_%04X", itemptr->addr);
		else
			fprintf(asmfile, ", L_%04X", itemptr->addr);
	}
}


/* generate Z80	assembler code,	based on parsed	areas and label	references */
void	Z80Asm(unsigned	short  pc, unsigned short  endarea)
{
	enum truefalse		linefeed;
	LabelRef		*foundlabel;
	DataRef			*founddata;
	char			mnemonic[64];

	while(pc <= endarea) {
		foundlabel = find(gLabelRef, &pc, (int (*)()) CmpAddrRef);
		if (foundlabel != NULL)
			fprintf(asmfile, ".L_%04X\n", foundlabel->addr);
		else {
			/* program label not found, try	data reference... */
			founddata = find(gDataRef, &pc,	(int (*)()) CmpDataRef);
			if (founddata != NULL) fprintf(asmfile,	".L_%04X ; Data reference\n", founddata->addr);
		}
		pc = Disassemble(mnemonic, pc, false);
		fprintf(asmfile,"\t\t\t\t%s\n",	mnemonic);
	}

	fputc('\n', asmfile);
}
