#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "DZasm.h"
#include "table.h"
#include "avltree.h"

/*
 * Development history:
 *
 * V0.1: January '96:
 * Algorithms for inserting/adjusting areas
 * Parser stack	implemented for	recursive traversing of	code
 * Parsing code	using Disassembly table	lookup
 * Source code generation level	1 (Z80 mnemonics, CALL_OZ, void	areas as DEFB)
 *
 * V0.2, 26.02.96:
 * Source code generation level	2 (Z88 OZ include files, undocumented instructions)
 * trapping of non-defined instructions	during parsing using separate lookup tables
 * Sample Disassembly for evaluation
 * Sample memory View
 * Search memory for opcodes (hex string)
 * JP vector table code	parsing
 * Address lookup (DEFW) code parsing
 * Collect file	read/write implemented.
 *
 * V0.21, 27.02.96
 * Define address (data) table area
 * Define string area
 * Define program area
 * Include file	reference information added to 'collect' file.
 * ';' replaced	by ',' in local, external areas.
 * PC port adjustments.
 * GetSym() optimized.
 *
 * V0.22, 28.02.96
 * Source code generation 3 (better differentiation of constants, IX/IY	offset written properly)
 * Data	reference labels now written during Z80	mnemonic output	(if found at PC)
 * DDCB	instruction parsing table implemented to trap unknown instructions
 */

#ifdef QDOS
#include <qdos.h>
void			consetup_title();
void			(*_consetup) ()	= consetup_title;
int			(*_readkbd) (chanid_t, timeout_t, char *) = readkbd_move;
struct WINDOWDEF	_condetails = {2, 1, 0, 7, 484, 256, 0, 0};
#endif

char			_prog_name[] = "Z80/Z88 Source Generator";
char			_vers[]	= "0.22";
char			_copyright[] = "\x7f InterLogic 1996";

enum truefalse		debug =	false;

FILE			*infile	= NULL,	*logfile = NULL;

char			Z80codeflnm[64];	/* Filename of Z80 code	file */
long			Org, Codesize, gEndOfCode;

DZarea			*gExtern;		/* list	of extern areas	*/
DZarea			*gAreas;		/* list	of areas currently examined */
struct PrsAddrStack	*gParseAddress;		/* stack of addresses to be disassembled */
avltree			*gLabelRef;		/* Binary tree of program labels */
avltree			*gDataRef;		/* Binary tree of address references */

struct PrsAddrStack	*AllocStackItem(void);
unsigned char		getbyte(unsigned short pc);
unsigned short		Disassemble(char *str, unsigned	short pc, enum truefalse  dispaddr);
unsigned short		ParseArea(long pc);
unsigned short		PopItem(struct PrsAddrStack **stackpointer);
enum atype		SearchArea(DZarea  *currarea, unsigned short  pc);
enum truefalse		LocalLabel(unsigned short  pc);
enum truefalse		CmpString(long saddr, unsigned char  *sptr, unsigned char  l);
DZarea			*InitArea(long	startaddr, long	 endaddr, enum atype  t);
int			InsertArea(struct area **arealist, long	 startrange, long  endrange, enum atype	 t);
float			ResolvedAreas(void);
void			JoinAreas(DZarea  *currarea);
void			PushItem(unsigned short	addr, struct PrsAddrStack **stackpointer);
void			InitDZ(void);
void			DZpass1(void);
void			DZpass2(void);
void			DispAreas(FILE *out, DZarea *arealist, enum truefalse  resolved);
void			ParseDZ(void);
void			SampleDZ(void);
void			SampleMemory(unsigned short  pc);
void			DefineMemory(void);
void			DispVoidAreas(FILE  *out, DZarea *arealist);
void			StoreAddrRef(unsigned short  label);
void			GenCollectFile(void);
void			ReadCollectFile(void);
void			DispParsedArea(unsigned	short start, unsigned short end);
void			FindCode(void);
void			ClearDataStructs(void);
void			StoreDataRef(unsigned short  label);


/* Read	a byte at the current Disassembler PC */
unsigned char	getbyte(unsigned short pc)
{
	pc -= Org;
	fseek(infile, pc, SEEK_SET);
	return fgetc(infile);
}


/* Entry of Intelligent	Disassembler */
main(void)
{
	unsigned short	pc;
	char		key;

	gParseAddress =	NULL;				/* init	address	parsing	stack */
	gLabelRef = NULL;				/* init	label references avltree */
	gDataRef = NULL;				/* init	data reference avltree */
	gAreas = NULL;					/* init	local areas */
	gExtern	= NULL;					/* init	extern areas */

	printf("%s, %s\n", _prog_name, _vers);

	if ((infile = fopen("collect", "r")) !=	NULL) {
		fclose(infile);
		ReadCollectFile();			/* read	previous parsing info from 'collect' file */
	}
	else {
		printf("Enter filename of Z80 code: ");	scanf("%s", Z80codeflnm);
		infile = fopen(Z80codeflnm,"rb");		   /* get Z80 code */
		if (infile == NULL) {
			puts("Code file not found.");
			exit(-1);
		}

		InitDZ();	/* Input origin	and calculate code boundaries */
	}

	if (debug == true) logfile = fopen("log.txt", "w");

	do {
		printf("\n%-3.2f%% resolved.\n", ResolvedAreas());
		puts("S)ample DZ   P)arse DZ   A)reas   V)oid Areas   M)emory View  D)efine Area");
		puts("F)ind code  G)enerate Source   C)reate Collect file   R)ead Collect file");
		puts("Q)uit");

		key=toupper(getch());
		switch(key) {
			case	'S':	SampleDZ();
					break;

			case	'P':	ParseDZ();
					break;

			case	'M':	printf("View Memory Address (hex): "); scanf("%hx", &pc);
					puts("Press <ESC> to abort, <Q>, <A> to move one page up, down.\n");
					SampleMemory(pc);
					break;

			case	'F':	FindCode();
					break;

			case	'D':	DefineMemory();
					break;

			case	'A':	DispAreas(stdout, gAreas, true);
					break;

			case	'V':	DispVoidAreas(stdout, gAreas);
					break;

			case	'G':	DZpass2();
					break;

			case	'C':	GenCollectFile();
					break;

			case	'R':	ClearDataStructs();
					fclose(infile);
					ReadCollectFile();
					break;

			case	'Q':	break;
		}
	}
	while(key != 'Q');

	ClearDataStructs();			/* remove parsing information from memory */
	if (infile != NULL) fclose(infile);	/* free	code file handle */

	puts("DZasm ended.");
	return 0;		/* finished */
}


void	ParseDZ(void)
{
	long		startrange, endrange, pc, pointer;
	char		key;

	printf("\nParse: P)rogram  J)P vector table  A)ddress table\n");
	key=toupper(getch());
	switch(key) {
		case	'P':
			printf("\nEnter DZ parse address: "); scanf("%lx", &pc);
			if ((pc	>= Org)	&& (pc <= gEndOfCode)) {
				PushItem(pc, &gParseAddress);		/* first address to parse */
				StoreAddrRef(pc);			/* define entry	also as	a label	*/
				DZpass1();				/* Parse areas from pc onwards */
			}
			else
				puts("DZ address out of program range.");
			break;

		case	'J':
			printf("\nEnter address of JP vector table: ");	scanf("%lx", &pc);
			if ((pc	>= Org)	&& (pc <= gEndOfCode)) {
				startrange = pc;
				while(getbyte(pc) == JP_opcode)	{
					PushItem(pc, &gParseAddress);		/* first address to parse */
					StoreAddrRef(pc);			/* define JP vector as a label */
					DZpass1();				/* Parse areas from pc onwards */
					pc += 3;				/* point at next JP instruction	*/
				}
			}
			else
				puts("JP vector table out of program range.");
			break;

		case	'A':
			printf("Enter start address of lookup table: "); scanf("%lx", &pc);
			printf("Enter end address of lookup table: "); scanf("%lx", &endrange);
			startrange = pc;
			StoreDataRef(pc);			/* Define beginning of table as	label... */
			while(pc < endrange) {
				pointer	= getbyte(pc++);
				pointer	+= getbyte(pc++) * 256;
				PushItem(pointer, &gParseAddress);	/* first address to parse */
				StoreAddrRef(pointer);			/* define pointer as a label */
				DZpass1();				/* Parse areas from pc onwards */
			}
			InsertArea(&gAreas, startrange,	endrange, addrtable);
			break;
	}
}


void	SampleDZ(void)
{
	unsigned short	pc, lines;
	char		mnemonic[64], key;

	printf("\nEnter DZ address to evaluate (hex): "); scanf("%hx", &pc);
	puts("Press <ESC> to stop");
	if ((pc	>= Org)	&& (pc <= Org+Codesize-1)) {
		key = '.';
		while(key != 27) {
			for(lines=0; lines<16; lines++)	{
				pc = Disassemble(mnemonic, pc, true);
				puts(mnemonic);
			}
			key = getch();
		}
	}
	else
		puts("DZ address out of program range.");
}


void	SampleMemory(unsigned short  pc)
{
	unsigned short	rows, columns, b;
	char		key;

	while(1) {
		putchar('\n');
		for(rows=0; rows<8; rows++) {
			printf("%04X ",	pc+rows*16);
			for(columns=0; columns<16; columns++)
				printf("%02X ",	getbyte(pc + rows*16 + columns));
			for(columns=0; columns<16; columns++) {
				b = getbyte(pc + rows*16 + columns);
				printf("%c", (b>=32 && b<=127) ? b : '.' );
			}
			putchar('\n');
		}
		while( (key=toupper(getch()))!=27 && key!='Q' && key!='A');
		switch(key) {
			case	27:	putchar('\n');
					return;

			case	'Q':	pc -= 16*8;
					break;

			case	'A':	pc += 16*8;
					break;
		}
	}
}


void	FindCode(void)
{
	unsigned char	inp[32];
	unsigned char	s[16];
	unsigned char	*searchptr, i, c, length;
	long		saddr;

	searchptr = s;

	printf("Input hex codes (joined):"); scanf("%s", inp);
	length = strlen((char *)inp);
	if (length%2 !=	0) {
		puts("Illegal hex string!");
		return;
	}

	for(i=0; i<length; i +=	2) {
		inp[i] = toupper(inp[i]); inp[i+1] = toupper(inp[i+1]);
		*searchptr = 16	* ((inp[i]<='9') ? inp[i]-48 : inp[i]-55);
		*searchptr += (inp[i+1]<='9') ?	inp[i+1]-48 : inp[i+1]-55;
		++searchptr;
	}
	length = length	/ 2;	/* actual length of search chars */

	printf("Search address: "); scanf("%lx", &saddr);

	i = 0; c=0;
	printf("Searching from %04X", saddr);
	while(saddr < gEndOfCode) {
		if (!c++) putchar('.');
		if (CmpString(saddr, s,	length)	== true) {
			++i;
			putchar('\n');
			SampleMemory(saddr);
			printf("Searching");
		}

		++saddr;
	}

	printf("\n%d matches were found.\n", i);
}


enum truefalse	CmpString(long saddr, unsigned char  *sptr, unsigned char  l)
{
	while(l--)
		if (*sptr++ != getbyte(saddr++)) return	false;

	return true;
}


void	DefineMemory(void)
{
	char		key;
	unsigned short	pointer;
	long		start, end, table;

	printf("\nDefine:  P)rogram  A)ddress table  S)tring \n");
	key=toupper(getch());
	switch(key) {
		case	'P':
			printf("\nEnter program area start: ");	scanf("%lx", &start);
			printf("Enter program area end: "); scanf("%lx", &end);
			if ((start < Org) || (start > gEndOfCode) || (end < Org) || (end > gEndOfCode))	{
				puts("Area out of code range.");
				break;
			}
			if (start > end) {
				puts("Illegal range.");
				break;
			}
			StoreAddrRef(start);	/* define entry	also as	a label	*/
			if (!InsertArea(&gAreas, start,	end, program)) puts("No room");
			break;

		case	'A':
			printf("\nEnter start address of data lookup table: ");	scanf("%lx", &start);
			printf("Enter end address of data lookup table: "); scanf("%lx", &end);
			if ((start < Org) || (start > gEndOfCode) || (end < Org) || (end > gEndOfCode))	{
				puts("Area out of code range.");
				break;
			}
			if (start > end) {
				puts("Illegal range.");
				break;
			}
			if ((end - start) % 2 != 0) {
				puts("Address table cannot fit last pointer.");
				break;
			}

			StoreDataRef(start);		     /*	Define beginning of table as label... */
			for(table = start; table<end; table += 2) {
				pointer	= getbyte(table);
				pointer	= 256 *	getbyte(table+1);
				StoreDataRef(pointer);			/* The pointer is label... */
			}
			if (!InsertArea(&gAreas, start,	end, addrtable)) puts("No room");
			break;

		case	'S':
			printf("\nEnter start address of DEFM string: "); scanf("%lx", &start);
			printf("Enter end address of DEFM string: "); scanf("%lx", &end);
			if ((start < Org) || (start > gEndOfCode) || (end < Org) || (end > gEndOfCode))	{
				puts("String out of code range.");
				break;
			}
			if (start > end) {
				puts("Illegal range.");
				break;
			}
			StoreDataRef(start);		     /*	Define beginning of string as label... */
			if (!InsertArea(&gAreas, start,	end, string)) puts("No room");
			break;
	}
}


/* Parse code and create program and data areas	*/
void	DZpass1(void)
{
	long		pc, endarea;
	enum atype	foundarea;

	while(gParseAddress!=NULL) {
		pc = PopItem(&gParseAddress);
		if (LocalLabel(pc) == true) {
			foundarea = SearchArea(gAreas, pc);
			switch(foundarea) {
				case	data:
						puts("Warning: parsing code inside data area!");

				case	notfound:
				case	vacuum:
						endarea	= ParseArea(pc);
						DispParsedArea(pc, endarea);
						if (debug == true) fprintf(logfile, "<%04Xh-%04Xh>\t", pc, endarea);
						if (!InsertArea(&gAreas, pc, endarea, program))	{
							puts("No room");	/* No more room	for data structures */
							return;
						}
			}
		}
	}

	JoinAreas(gAreas);	/* scan	list and join equal type areas */
}


void	DispParsedArea(unsigned	short start, unsigned short end)
{
	static short	i = 0;

	if (i++	% 4 == 0)
		printf("\n[%04Xh-%04Xh]", start, end);
	else
		printf("\t[%04Xh-%04Xh]", start, end);
}


/* User	input of code origin and program calculation of	implicit code size,
 * local and extern code areas.
 */
void	InitDZ(void)
{
	fseek(infile, 0L, SEEK_END);	/* file	pointer	to end of file */
	Codesize = ftell(infile);

	printf("Origin of code (hex): "); scanf("%lx", &Org);
	gEndOfCode = Org+Codesize-1;
	InsertArea(&gAreas, Org, gEndOfCode, vacuum); /* define	local area to be parsed	*/

	if (Org	> 0) InsertArea(&gExtern, 0, Org-1, vacuum);				/* extern area,	before code */
	if (Org+Codesize < 65535) InsertArea(&gExtern, (Org+Codesize), 65535, vacuum);	/* extern area,	after code */

	printf("Extern areas: "); DispVoidAreas(stdout,	gExtern); putchar('\n');
	printf("Code area to be parsed is %lXh-%lXh\n",	Org, gEndOfCode);
}


void	DeleteAreas(DZarea  *currarea)
{
	DZarea		*nextarea;

	while(currarea != NULL)	{
		nextarea = currarea->nextarea;
		free(currarea);
		currarea = nextarea;
	}
}


void	DelLabel(LabelRef  *labelptr)
{
	free(labelptr);
}

void	DelAddr(DataRef	 *labelptr)
{
	free(labelptr);
}


void	ClearDataStructs(void)
{
	puts("Releasing allocated memory...");

	while(gParseAddress != NULL) PopItem(&gParseAddress);	/* Delete parsing stack	*/

	DeleteAreas(gAreas);
	gAreas = NULL;
	DeleteAreas(gExtern);
	gExtern	= NULL;

	deleteall(&gLabelRef, (void (*)()) DelLabel);
	deleteall(&gDataRef, (void (*)()) DelAddr);
}
