/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <inttypes.h>

#include <string.h>
#include <glib-object.h>
#include <glib.h>
#include <libgnome/gnome-i18n.h>
#include <gsf/gsf.h>
#include <gsf-gnome/gsf-input-gnomevfs.h>
#include <gsf/gsf-input-stdio.h>

#include "application.h"
#include "debug.h"
#include "marshallers.h"
#include "presentation.h"
#include "presentation-parser.h"
#include "theme.h"

struct _CriaPresentationPrivate {
	GnomeVFSURI	* uri;
	GList		* slides;
	GHashTable	* themes;
};

enum {
	PROP_0,
	PROP_TITLE,
	PROP_URI,
};

enum {
	INSERTED_SLIDE_SIGNAL,
	N_SIGNALS
};

static void		  cria_presentation_get_property       (GObject		* object,
								guint		  prop_id,
								GValue		* value,
								GParamSpec	* param_spec);
static CriaPresentation*  cria_presentation_new_from_uri       (GnomeVFSURI	* uri,
								GError		**error);
static void		  cria_presentation_init	       (CriaPresentation* self);
static void		  cria_presentation_set_property       (GObject		* object,
								guint		  prop_id,
								const	GValue	* value,
								GParamSpec	* param_spec);

static guint		  cria_presentation_signals[N_SIGNALS] = { 0 };
static void		  cria_presentation_inserted_slide     (CriaPresentation* self,
								gint		  new_position,
								CriaSlide	* new_slide);

/**
 * cria_presentation_add_theme:
 * @self: a CriaPresentation
 * @theme: a CriaTheme
 *
 * Adds the @theme to presentation references by @self.
 */
void
cria_presentation_add_theme(CriaPresentation* self, CriaTheme* theme) {
	g_return_if_fail(CRIA_IS_PRESENTATION (self));
	g_return_if_fail(CRIA_IS_THEME (theme));
	g_return_if_fail(g_hash_table_lookup(self->priv->themes, cria_theme_get_name (theme)) == NULL);

	g_hash_table_insert(self->priv->themes,
			    g_strdup(cria_theme_get_name(theme)),
			    theme);
}

/**
 * cria_presentation_append_slide:
 * @self: a #CriaPresentation
 * @slide: a #CriaSlide
 *
 * Appends @slide to @presentation.
 */
void
cria_presentation_append_slide(CriaPresentation* self, CriaSlide* slide) {
	g_return_if_fail(CRIA_IS_PRESENTATION(self));
	g_return_if_fail(CRIA_IS_SLIDE(slide));

	g_debug ("Presentation::appendSlide(): appending slide \"%s\" with index %i (dev-count)", cria_slide_get_title (slide), g_list_length (self->priv->slides));
	
	cria_presentation_insert_slide(self, slide, -1);
}

static void
cria_presentation_class_init(CriaPresentationClass* cria_presentation_class) {
	GObjectClass	* g_object_class;

	/* setting up the presentation class */
	cria_presentation_class->inserted_slide = cria_presentation_inserted_slide;

	cria_presentation_signals[INSERTED_SLIDE_SIGNAL] = g_signal_new("inserted-slide",
									CRIA_TYPE_PRESENTATION,
									G_SIGNAL_RUN_LAST,
									G_STRUCT_OFFSET(CriaPresentationClass,
											inserted_slide),
									NULL,
									NULL,
									cria_marshal_VOID__INT_OBJECT,
									G_TYPE_NONE,
									2,
									G_TYPE_INT,
									CRIA_TYPE_SLIDE);

	/* setting up the gobject class */
	g_object_class = G_OBJECT_CLASS(cria_presentation_class);

	g_object_class->set_property = cria_presentation_set_property;
	g_object_class->get_property = cria_presentation_get_property;

	g_object_class_install_property(g_object_class,
					PROP_TITLE,
					g_param_spec_string("title",
							    "Title",
							    "The title of the presentation",
							    _("untitled"),
							    G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
	g_object_class_install_property(g_object_class,
					PROP_URI,
					g_param_spec_pointer("uri",
							     "GnomeVFSURI for this file",
							     "The URI this file can be located at, NULL if newly created",
							     G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
}

const gchar*
cria_presentation_get_filename(CriaPresentation* self) {
	g_assert(self != NULL && CRIA_IS_PRESENTATION(self));
	g_assert(self->priv != NULL);

	if(self->priv->uri == NULL) {
		return _("Unsaved Presentation");
	} else {
		GnomeVFSFileInfo* info = gnome_vfs_file_info_new();
		gnome_vfs_get_file_info_uri(self->priv->uri, info, GNOME_VFS_FILE_INFO_DEFAULT);
		return info->name;
	}
}

static void
cria_presentation_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* param_spec) {
	CriaPresentation	* self;

	self = CRIA_PRESENTATION(object);

	switch(prop_id) {
	case PROP_TITLE:
		g_value_set_string(value, self->title);
		break;
	case PROP_URI:
		g_value_set_pointer(value, self->priv->uri);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object,
						  prop_id,
						  param_spec);
		break;
	}
}

CriaSlide*
cria_presentation_get_slide (
		CriaPresentation* self,
		guint		  slide)
{
	CriaSlide	* retval;
	
	g_return_val_if_fail (CRIA_IS_PRESENTATION (self), NULL);
	g_return_val_if_fail (slide < cria_presentation_n_slides (self), NULL);
	
	retval = CRIA_SLIDE (g_list_nth (self->priv->slides, slide)->data);

	return retval;
}

/**
 * cria_presentation_get_slide_index:
 * @self: The presentation to search in
 * @slide: The slide to be found
 *
 * Get the index of a slide in a presentation.
 *
 * Returns the index of the slide if found, -1 otherwise
 */
gint
cria_presentation_get_slide_index(CriaPresentation* self, CriaSlide const* slide) {
	g_assert(self != NULL && CRIA_IS_PRESENTATION(self));
	g_assert(slide != NULL && CRIA_IS_SLIDE(slide));
	g_assert(self->priv != NULL);
	
	return g_list_index(self->priv->slides, slide);
}

/**
 * cria_presentation_get_slide_size:
 * @self: The presentation to query
 *
 * Get the slide size for this presentation (in Master Coordinates, 576dpi).
 * Don't free it, it's internal data from the presentation.
 *
 * Returns the size of a slide
 */
const GoPoint*
cria_presentation_get_slide_size(CriaPresentation* self) {
#warning "Presentation::getSlideSize(): FIXME make this configurable and return the value of the presentation instead of a predfined one"
	static GoPoint* size = NULL;
	
	if(!size) {
		size = g_new0(GoPoint,1);
		size->x = 5760;
		size->y = 4320;
	};

	g_debug("Presentation:getSlideSlize(): %llix%lli", size->x, size->y);

	return size;
}

CriaTheme*
cria_presentation_get_theme (
		CriaPresentation* self,
		const	char	* theme)
{
	g_return_val_if_fail (CRIA_IS_PRESENTATION (self), NULL);

	return CRIA_THEME (g_hash_table_lookup (self->priv->themes, theme));
}

/**
 * cria_presentation_get_title:
 * @self: The presentation to get the title from
 *
 * Get the title of the presentation.
 *
 * Returns the title of the presentation, of one is set, %NULL otherwise
 */
const char*
cria_presentation_get_title(CriaPresentation* self) {
	g_assert(self != NULL && CRIA_IS_PRESENTATION(self));

	g_debug("Presentation::getTitle(): %s", self->title);
	
	return self->title;
}

GType
cria_presentation_get_type (void)
{
	static GType	type = 0;

	if (!type)
	{
		const GTypeInfo info = {
			sizeof (CriaPresentationClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_presentation_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof (CriaPresentation),
			0,
			(GInstanceInitFunc)cria_presentation_init,
			0
		};

		type = g_type_register_static (
				G_TYPE_OBJECT,
				"CriaPresentation",
				&info,
				0);
	}

	return type;
}

/**
 * cria_presentation_get_uri:
 * @self: the presentation to get the uri from
 *
 * Get information about the path of a presentation.
 *
 * Returns a GnomeVFSURI defining the path and filename of a presentation
 */
const GnomeVFSURI*
cria_presentation_get_uri(CriaPresentation* self) {
	g_assert(self != NULL && CRIA_IS_PRESENTATION(self));
	g_assert(self->priv != NULL);
#warning "Presentation::getURI(): FIXME: ensure some valid URI"
	g_assert(self->priv->uri != NULL);
	
	return self->priv->uri;
}


static void
cria_presentation_init (CriaPresentation *self)
{
	self->priv = g_new0 (CriaPresentationPrivate, 1);

	self->priv->themes = g_hash_table_new_full (
			g_str_hash,
			g_str_equal,
			g_free,
			g_object_unref);
}

/**
 * cria_presentation_insert_slide:
 * @self: a #CriaPresentation
 * @slide: a #CriaSlide
 * @pos: a position
 *
 * Insert @slide at @pos in @self. If @pos is smaller than 0 or bigger than the
 * list's length, the slide will be appended.
 */
void
cria_presentation_insert_slide(CriaPresentation* self, CriaSlide* slide, gint pos) {
	g_return_if_fail(CRIA_IS_PRESENTATION(self));
	g_return_if_fail(CRIA_IS_SLIDE(slide));
	g_return_if_fail(cria_presentation_get_slide_index(self, slide) == -1);
	
	self->priv->slides = g_list_insert(self->priv->slides, slide, pos);

	pos = cria_presentation_get_slide_index(self, slide);
	g_debug("Presentation::insertSlide(): new index = %i", pos);
	g_signal_emit(self, cria_presentation_signals[INSERTED_SLIDE_SIGNAL], 0, pos, slide);
}

static void
cria_presentation_inserted_slide(CriaPresentation* self, gint new_position, CriaSlide* slide) {
#warning "Presentation::insertedSlide(): set or unset the save state here"
}

/**
 * cria_presentation_new_default:
 *
 * Creates a simple untitled presentation containing one empty slide.
 *
 * Returns a new presentation
 */
CriaPresentation*
cria_presentation_new_default(void) {
	CriaPresentation* self  = CRIA_PRESENTATION(g_object_new(CRIA_TYPE_PRESENTATION, NULL));
	CriaTheme	* theme = cria_theme_new(_("Default Theme"));
	CriaSlide	* master_slide = cria_slide_new(NULL),
			* slide = cria_slide_new(self);

#warning "Presentation::newDefault(): set a GnomeVFSURI"
#warning "Presentation::newDefault(): create a pool of unsaved presentations to be able handle 'Unsaved 1', 'Unsaved 2', etc."
	cria_theme_add_master_slide(theme, master_slide);
	cria_presentation_add_theme(self, theme);
	cria_slide_set_theme(slide, theme);
	cria_slide_set_master_slide(slide, master_slide);
	
	return self;
}

/**
 * cria_presentation_new_from_file:
 * @filename: the name of the file to open
 * @error: a location to return a GError, if this is 
 *
 * Creates a new presentation by parsing a file. @filename needs to be given
 * relative to the working directory. This function is used only for parsing
 * command line arguments.
 *
 * Returns the new presentation or NULL if an error occured
 */
CriaPresentation*
cria_presentation_new_from_file(const gchar* filename, GError** error) {
	CriaPresentation* presentation;
	gchar		* uri	= NULL;

	g_assert(filename != NULL && strlen(filename));
	g_assert(*error == NULL);
	
	g_debug("Presentation::newFromFile(): start");

	uri = gnome_vfs_make_uri_from_shell_arg(filename);
	presentation = cria_presentation_new_from_text_uri(uri, error);
	
	if(*error) {
		if(presentation) {
			g_object_unref(presentation);
		}
		
		presentation = NULL;
	} else {
		cria_application_add_to_recent(uri);
		g_debug("Presentation::newFromFile(): added uri '%s'", uri);
	}

	g_debug("Presentation::newFromFile(): end");
	
	g_free(uri);
	uri = NULL;
	return presentation;
}

/**
 * cria_presentation_new_from_text_uri:
 * @text_uri: the text representation of a URI, e.g. ftp://user@host/path/to/file.ext
 * @error: a pointer to return a GError
 *
 * Create a new presentation by parsing the file given by the text_uri.
 *
 * Returns a new presentation
 */
CriaPresentation*
cria_presentation_new_from_text_uri(const gchar* text_uri, GError**errloc) {
	CriaPresentation* presentation;
	GError		* error	= NULL;
	GnomeVFSURI	* uri	= NULL;
	GnomeVFSFileInfo* info  = NULL;

	g_assert(text_uri != NULL && strlen(text_uri));
	g_assert(*errloc == NULL);

	uri = gnome_vfs_uri_new(text_uri);
	info = gnome_vfs_file_info_new();
	gnome_vfs_get_file_info_uri(uri, info, GNOME_VFS_FILE_INFO_DEFAULT | GNOME_VFS_FILE_INFO_GET_MIME_TYPE);

	presentation = cria_presentation_new_from_uri(uri, &error);
	gnome_vfs_uri_unref(uri);

	if(error) {
		*errloc = error;
		return NULL;
	}
	
	return presentation;
}

/*
 * cria_presentation_new_from_uri:
 * @uri: ...
 * errloc: ...
 *
 * ...
 *
 * Returns ...
 */
static CriaPresentation*
cria_presentation_new_from_uri(GnomeVFSURI* uri, GError** errloc) {
	GsfInput	* input = NULL;
	GError		* error = NULL;
	CriaPresentation* self = NULL;

	g_assert(uri != NULL);
	g_assert(*errloc == NULL);

	input = gsf_input_gnomevfs_new_uri(uri, &error);

	if (error) {
		*errloc = error;
		return NULL;
	}

	/* TODO add mime type detection:
	 * this one works basically by getting a GnomeVFSFileInfo from the URI
	 * and let GnomeVFS do the detection */
	/* TODO extract the file reading into a plugin (refer to gnumeric's code
	 * for examples) */
	
	self = g_object_new(CRIA_TYPE_PRESENTATION, NULL);
	cria_presentation_set_uri(self, uri);
	cria_presentation_populate_from_xml_input(self, input, &error);

	if (error) {
		*errloc = error;
		return NULL;
	}
	
	return self;
}

guint
cria_presentation_n_slides(CriaPresentation* self) {
	g_return_val_if_fail (CRIA_IS_PRESENTATION (self), 0);
	
	return g_list_length (self->priv->slides);
}

static void
cria_presentation_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* param_spec) {
	CriaPresentation	* self;
	
	self = CRIA_PRESENTATION (object);
	
	switch (prop_id) {
	case PROP_TITLE:
		cria_presentation_set_title(self, g_value_get_string(value));
		break;
	case PROP_URI:
		cria_presentation_set_uri(self, g_value_get_pointer(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object,
						  prop_id,
						  param_spec);
		break;
	}
}

/**
 * cria_presentation_set_title:
 * @self: the presentation to set the title for
 * @title: the new title for the presentation
 *
 * Sets a new title for the presentation. The title is only changed if
 * both titles are different, so we prevent recursive loops with title
 * entry elements.
 */
void
cria_presentation_set_title(CriaPresentation* self, const gchar* title) {
	g_assert(self != NULL && CRIA_IS_PRESENTATION(self));

	g_debug("Presentation::setTitle(): setting to %s", title);

	if((title == NULL && self->title == NULL) ||
	   (title != NULL && self->title != NULL && !strcmp(title, self->title))) {
		/* the text's are equal */
		return;
	}

	if(self->title != NULL) {
		g_free(self->title);
	}

	if(title) {
		self->title = g_strdup(title);
	}

	g_object_notify(G_OBJECT(self), "title");
}

void
cria_presentation_set_uri(CriaPresentation* self, GnomeVFSURI* uri) {
	g_assert(self != NULL && CRIA_IS_PRESENTATION(self));
	g_assert(self->priv != NULL);
	/* unfortunately we don't have GType information for GnomeVFSURI */
	/* g_assert(self->priv->uri == NULL || GNOME_IS_VFS_URI(self->priv->uri)); */
	/* g_assert(uri == NULL || GNOME_IS_VFS_URI(uri)); */

	if(self->priv->uri) {
		gnome_vfs_uri_unref(self->priv->uri);
		self->priv->uri = NULL;
	}

	self->priv->uri = uri;
	if(uri) {
		gnome_vfs_uri_ref(uri);
	}

	g_object_notify(G_OBJECT(self), "uri");
}

