/*
 * File: dw_ext_iterator.c
 *
 * Copyright (C) 2002 Sebastian Geerken <S.Geerken@ping.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

/*
 * DwExtIterator is an extension of DwIterator, which stores
 * DwIterator's in a stack, for cases where recursions are not
 * possible.
 */

#include "dw_ext_iterator.h"
#include "list.h"

/*
 * Create a new DwExtIterator from an existing DwIterator.
 * The content of the return value will be the content of "it".
 * NOTES:
 * (i)  If you want to continue using "it", pass a_Dw_iterator_clone (it).
 * (ii) The mask of "it" must include DW_CONTENT_WIDGET, but 
 *      a_Dw_ext_iterator_next will never return widgets.
 */
DwExtIterator* a_Dw_ext_iterator_new (DwIterator *it)
{
   DwExtIterator *eit = g_new (DwExtIterator, 1);
   DwWidget *w;
   int sp;

   eit->stack_top = 0;
   /* If this widget has parents, we must construct appropiate
    * iterators. TODO: There should be a faster way instead of
    * iterating through the parent widgets. */
   /* NOTE: This has not yet been tested fully! */
   for (w = it->widget; w->parent != NULL; w = w->parent)
      eit->stack_top++;
   eit->stack_max = 4;
   while (eit->stack_top >= eit->stack_max)
      eit->stack_max <<= 1;

   /* Construct the iterators. */
   for (w = it->widget, sp = eit->stack_top - 1;
       w->parent != NULL;
       w = w->parent, sp--) {
      eit->stack[sp] = a_Dw_widget_iterator (w->parent, it->mask);
      while (TRUE) {
         while (!a_Dw_iterator_next(eit->stack[sp])) {
            g_warning ("BUG in DwExtIterator!");
            return NULL;
         }
         if (eit->stack[sp]->content.type == DW_CONTENT_WIDGET &&
             eit->stack[sp]->content.data.widget == w)
            break;
      }
   }

   eit->stack = g_new (DwIterator*, eit->stack_max);
   eit->stack[eit->stack_top] = it;
   eit->content = it->content;
   return eit;
}

/*
 * Move iterator forward and store content it. Returns TRUE on
 * success.
 */
gboolean a_Dw_ext_iterator_next (DwExtIterator *eit)
{
   DwIterator *it = eit->stack[eit->stack_top];

   if (a_Dw_iterator_next(it)) {
      if (it->content.type == DW_CONTENT_WIDGET) {
         /* Widget: new iterator on stack, to search in this widget. */
         eit->stack_top++;
         a_List_add (eit->stack, eit->stack_top, eit->stack_max);
         eit->stack[eit->stack_top] =
            a_Dw_widget_iterator (it->content.data.widget, it->mask);
         return a_Dw_ext_iterator_next (eit);
      } else {
         /* Simply return the content of the iterartor. */
         eit->content = it->content;
         return TRUE;
      }
   } else {
      /* No more data in the top-most widget. */
      if (eit->stack_top > 0) {
         /* Pop iterator from stack, and move to next item in the old one. */
         a_Dw_iterator_free (it);
         eit->stack_top--;
         return a_Dw_ext_iterator_next (eit);
      } else
         /* Stack is empty. */
         return FALSE;
   }         
}

/*
 * Create an exact copy of the iterator, which then can be used
 * independantly of the original one.
 */
DwExtIterator* a_Dw_ext_iterator_clone (DwExtIterator *eit)
{
   int i;
   DwExtIterator *eit2 = g_new (DwExtIterator, 1);
   eit2->stack_top = eit->stack_top;
   eit2->stack_max = eit->stack_max;
   eit2->content = eit->content;

   eit2->stack = g_new (DwIterator*, eit2->stack_max);
   for (i = 0; i < eit2->stack_top; i++)
      eit2->stack[i] = a_Dw_iterator_clone (eit->stack[i]);

   return eit2;
}

/*
 * Free memory of iterator.
 */
void a_Dw_ext_iterator_free (DwExtIterator *eit)
{
   int i;
   for (i = 0; i < eit->stack_top; i++)
      a_Dw_iterator_clone (eit->stack[i]);
   g_free (eit->stack);
   g_free (eit);
}
