/************************************************************

Copyright (c) 1996-2002 the kicker authors. See file AUTHORS.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "zone.h"

#include <kcolorbutton.h>
#include <kconfig.h>
#include <kdebug.h>
#include <kstandarddirs.h>
#include <kstringhandler.h>
#include <klocale.h>

#include <qfile.h>
#include <qtooltip.h>
#include <klistview.h>

#include <stdlib.h>
#include <time.h>

static void setCurrentTimeZone(const QString& zone)
{
  // set TZ variable for this program
  // Ever occurring error:
  // TZ="" means UTC
  // TZ unset means localtime
  // When changing code remember that.
  if (!zone.isEmpty()) {
    ::putenv(strdup(QString("TZ=" + zone).latin1()));
  } else {
    ::unsetenv("TZ");
  }
  tzset();
}

Zone::Zone(KConfig* conf):
    config(conf),
    _zoneIndex(0)
{
    _defaultTZ = ::getenv("TZ");
    tzset();

    config->setGroup("General");

    /* default displayable timezones */
    QString tzList = config->readEntry("RemoteZones",
                                        "America/Los_Angeles,America/New_York,"
                                        "Australia/Sydney,Asia/Tokyo,"
                                        "Europe/Berlin,Europe/London,Europe/Moscow");
    _remotezonelist = QStringList::split(",", tzList);
    setZone(config->readNumEntry("Initial_TZ", 0));
}

Zone::~Zone()
{
    writeSettings();
}

void Zone::setZone(int z)
{
    if (_zoneIndex > _remotezonelist.count())
        z = 0;

    _zoneIndex = z;
}

QString Zone::zone(int z) const
{
    return (z == 0 ? _defaultTZ : _remotezonelist[z-1]);
}

int Zone::calc_TZ_offset(const QString& zone, bool reset)
{
  if (reset)
     setCurrentTimeZone("DUMMY"); // Make sure to reread local timezone info
  setCurrentTimeZone(zone);
  QDateTime t1( QDateTime::currentDateTime() );
  setCurrentTimeZone(_defaultTZ);
  return QDateTime::currentDateTime().secsTo(t1);
}

void Zone::readZoneList(KListView *listView )
{
  QFile f("/usr/share/zoneinfo/zone.tab");

  if (!f.open(IO_ReadOnly))
  {
      kdDebug() << "Can't open zone.tab" << endl;
      listView->setEnabled(false);
      QToolTip::add(listView, i18n("Cannot generate time-zone list"));
      return;
  }

  QTextStream str(&f);
  QRegExp rx("[ \t]");
  QMap<QString, QListViewItem*> KontinentMap;
  QListViewItem *Kontinent;

  listView->setRootIsDecorated(true);

  while (!str.atEnd())
  {
    QString line = str.readLine();
    if (line.isEmpty() || '#' == line[0])
	continue;

    QStringList tokens = KStringHandler::perlSplit(rx, line, 4);
    if (tokens.count() < 3)
	continue;

    QString tzName = tokens[2];
    if (!tzName.isEmpty())
    {
       tokens[2] = i18n(tokens[2].utf8());
       tokens[2].replace("_", " ");
    }
    
    QString comment = tokens[3];
    if (!comment.isEmpty())
       comment = i18n(comment.utf8());

    QStringList KontCity = QStringList::split("/", tokens[2]);
    Kontinent = KontinentMap[KontCity[0]];
    if (!Kontinent) {
	KontinentMap[KontCity[0]] = new QListViewItem(listView, KontCity[0]);
	Kontinent = KontinentMap[KontCity[0]];
	Kontinent->setExpandable(true);
    }

    QCheckListItem *li = new QCheckListItem(Kontinent, KontCity[1], QCheckListItem::CheckBox);
    li->setText(1, comment);
    li->setText(2, tzName); /* store complete path in ListView */

    if (_remotezonelist.findIndex(tzName) != -1)
       li->setOn(true);

    // locate the flag from /l10n/%1/flag.png
    // if not available select default "C" flag
    QString flag = locate( "locale", QString("l10n/%1/flag.png").arg(tokens[0].lower()) );
    if (!QFile::exists(flag))
       flag = locate( "locale", "l10n/C/flag.png" );
    if (QFile::exists(flag))
       li->setPixmap(0, QPixmap(flag));
  }
}

void Zone::getSelectedZonelist(KListView *listView)
{
     _remotezonelist.clear();

     /* loop through all entries */
     QListViewItem *root = listView->firstChild();
     while (root) {
	if (root->firstChild()) {
		root = root->firstChild();
		continue;
	}

	QCheckListItem *cl = (QCheckListItem*) root;
	if (cl->isOn()) {
		_remotezonelist.append(cl->text(2));
	}

	if (root->nextSibling()) {
		root = root->nextSibling();
		continue;
	}
	root = root->parent();
	if (root)
		root = root->nextSibling();
     }
}

void Zone::nextZone()
{
    if (++_zoneIndex > _remotezonelist.count())
	_zoneIndex = 0;
}

void Zone::prevZone()
{
    if (_zoneIndex == 0)
	_zoneIndex = _remotezonelist.count();
    else
	--_zoneIndex;
}

void Zone::writeSettings()
{
    config->setGroup("General");
    config->writeEntry("RemoteZones", _remotezonelist.join(","));
    config->writeEntry("Initial_TZ",_zoneIndex);
    config->sync();
}
