/*
    This file is part of libkcal.
    Copyright (c) 2001 Cornelius Schumacher <schumacher@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#include <kglobal.h>
#include <klocale.h>
#include <kdebug.h>

#include "event.h"

using namespace KCal;

Event::Event() :
  mHasEndDate( false ), mTransparency( Opaque )
{
}

Event::Event(const Event &e) : Incidence(e)
{
  mDtEnd = e.mDtEnd;
  mHasEndDate = e.mHasEndDate;
  mTransparency = e.mTransparency;
}

Event::~Event()
{
}

Incidence *Event::clone()
{
  return new Event(*this);
}

bool KCal::operator==( const Event& e1, const Event& e2 )
{
    return operator==( (const Incidence&)e1, (const Incidence&)e2 ) &&
        e1.dtEnd() == e2.dtEnd() &&
        e1.hasEndDate() == e2.hasEndDate() &&
        e1.transparency() == e2.transparency();
}


bool Event::contains ( Event* from )
{

    if ( !from->summary().isEmpty() )
        if (  !summary().startsWith(  from->summary() ))
            return false; 
    if ( from->dtStart().isValid() )
        if (dtStart() != from->dtStart() )
            return false;
    if ( from->dtEnd().isValid() )
        if ( dtEnd() !=  from->dtEnd() )
            return false;
    if ( !from->location().isEmpty() )
        if (  !location().startsWith( from->location() ) )
            return false;
    if ( !from->description().isEmpty() )
        if (  !description().startsWith(  from->description() ))
            return false;
    if ( from->alarms().count() ) {
        Alarm *a = from->alarms().first();
        if (  a->enabled()  ){
            if ( !alarms().count() )
                return false;
            Alarm *b = alarms().first();
            if( ! b->enabled() )
                return false;
            if ( ! (a->offset() == b->offset() ))
                return false;
        }
    }
    QStringList cat = categories();
    QStringList catFrom = from->categories();
    QString nCat;
    unsigned int iii;
    for ( iii = 0; iii < catFrom.count();++iii ) {
        nCat = catFrom[iii];
        if ( !nCat.isEmpty() )
            if ( !cat.contains( nCat )) {
                return false;
            }
    }
    if ( from->doesRecur() ) 
        if ( from->doesRecur() != doesRecur() && ! (from->doesRecur()== Recurrence::rYearlyMonth && doesRecur()== Recurrence::rYearlyDay) )
            return false;
    return true;
}

void Event::setDtEnd(const QDateTime &dtEnd)
{
  if (mReadOnly) return;

  mDtEnd = getEvenTime( dtEnd );

  setHasEndDate(true);
  setHasDuration(false);

  updated();
}

QDateTime Event::dtEnd() const
{
  if (hasEndDate()) return mDtEnd;
  if (hasDuration()) return dtStart().addSecs(duration());

  return dtStart();
}

QString Event::dtEndTimeStr() const
{
  return KGlobal::locale()->formatTime(mDtEnd.time());
}

QString Event::dtEndDateStr(bool shortfmt) const
{
  return KGlobal::locale()->formatDate(mDtEnd.date(),shortfmt);
}

QString Event::dtEndStr(bool shortfmt) const
{
  return KGlobal::locale()->formatDateTime(mDtEnd, shortfmt);
}

void Event::setHasEndDate(bool b)
{
  mHasEndDate = b;
}

bool Event::hasEndDate() const
{
  return mHasEndDate;
}

bool Event::isMultiDay() const
{
  bool multi = !(dtStart().date() == dtEnd().date());
  return multi;
}

void Event::setTransparency(Event::Transparency transparency)
{
  if (mReadOnly) return;
  mTransparency = transparency;
  updated();
}

Event::Transparency Event::transparency() const
{
  return mTransparency;
}

void Event::setDuration(int seconds)
{
  setHasEndDate(false);
  Incidence::setDuration(seconds);
}
QDateTime Event::getNextAlarmDateTime( bool * ok, int * offset, QDateTime start_dt ) const
{
  
    bool yes;
     QDateTime incidenceStart = getNextOccurence( start_dt, &yes );
     if ( ! yes || cancelled() ) {
         *ok = false;
         return QDateTime ();
     }
  
     bool enabled = false;
     Alarm* alarm;
     int off = 0;
     QDateTime alarmStart = QDateTime::currentDateTime().addDays( 3650 );;
    //  if ( QDateTime::currentDateTime() > incidenceStart ){
//          *ok = false;
//          return incidenceStart;
//      }
     for (QPtrListIterator<Alarm> it(mAlarms); (alarm = it.current()) != 0; ++it) {
         if (alarm->enabled()) {
             if ( alarm->hasTime () ) {
                 if ( alarm->time() < alarmStart ) {
                     alarmStart = alarm->time();
                     enabled = true;
                     off = alarmStart.secsTo( incidenceStart );
                 }

             } else {
                 int secs = alarm->startOffset().asSeconds();
                 if ( incidenceStart.addSecs( secs ) < alarmStart ) {
                     alarmStart = incidenceStart.addSecs( secs );
                     enabled = true;
                     off = -secs;
                 }
             }
         }
     }
     if ( enabled ) {
         if ( alarmStart >  start_dt  ) {
             *ok = true;
             * offset = off;
             return alarmStart;
         }
     }
     *ok = false;
     return QDateTime ();

}
