/*
 *	$Id: maplemouse.c,v 1.2 2002/01/23 06:56:48 mrbrown Exp $
 * 	SEGA Dreamcast mouse driver
 *	Based on drivers/usb/usbmouse.c
 */

#include <linux/kernel.h>
#include <linux/malloc.h>
#include <linux/input.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/timer.h>
#include <linux/maple.h>

MODULE_AUTHOR("YAEGASHI Takeshi <t@keshi.org>");
MODULE_DESCRIPTION("SEGA Dreamcast mouse driver");

struct dc_mouse {
	struct input_dev dev;
	int open;
};


static void dc_mouse_callback(struct maple_driver_data *data)
{
	int buttons, relx, rely, relz;
	struct mapleq *mq=& data->mq;
	struct dc_mouse *mouse = data->private_data;
	struct input_dev *dev = &mouse->dev;
	unsigned char *res = mq->recvbuf;

	buttons = ~res[8];
	relx=*(unsigned short *)(res+12)-512;
	rely=*(unsigned short *)(res+14)-512;
	relz=*(unsigned short *)(res+16)-512;

	input_report_key(dev, BTN_LEFT,   buttons&4);
	input_report_key(dev, BTN_MIDDLE, buttons&9);
	input_report_key(dev, BTN_RIGHT,  buttons&2);
	input_report_rel(dev, REL_X,      relx);
	input_report_rel(dev, REL_Y,      rely);
	input_report_rel(dev, REL_WHEEL,  relz);
}


static int dc_mouse_open(struct input_dev *dev)
{
	struct dc_mouse *mouse = dev->private;
	mouse->open++;
	return 0;
}


static void dc_mouse_close(struct input_dev *dev)
{
	struct dc_mouse *mouse = dev->private;
	mouse->open--;
}


static int dc_mouse_connect(struct maple_driver_data *d)
{
	unsigned long data = d->function_data;
	struct dc_mouse *mouse;

	if (!(mouse = kmalloc(sizeof(struct dc_mouse), GFP_KERNEL)))
		return -1;
	memset(mouse, 0, sizeof(struct dc_mouse));

	d->private_data = mouse;

	mouse->dev.evbit[0] = BIT(EV_KEY) | BIT(EV_REL);
	mouse->dev.keybit[LONG(BTN_MOUSE)] = BIT(BTN_LEFT) | BIT(BTN_RIGHT) | BIT(BTN_MIDDLE);
	mouse->dev.relbit[0] = BIT(REL_X) | BIT(REL_Y) | BIT(REL_WHEEL);

	mouse->dev.private = mouse;
	mouse->dev.open = dc_mouse_open;
	mouse->dev.close = dc_mouse_close;
	mouse->dev.event = NULL;

	mouse->dev.name = d->dev->product_name;
	mouse->dev.idbus = BUS_MAPLE;
	
	input_register_device(&mouse->dev);

	printk(KERN_INFO "input%d: mouse(0x%lx): %s\n",
	       mouse->dev.number, data, mouse->dev.name);

	MOD_INC_USE_COUNT;

	return 0;
}


static void dc_mouse_disconnect(struct maple_driver_data *d)
{
	struct dc_mouse *mouse = d->private_data;

	input_unregister_device(&mouse->dev);

	kfree(mouse);

	MOD_DEC_USE_COUNT;
}


static struct maple_driver dc_mouse_driver = {
	function:	MAPLE_FUNC_MOUSE,
	name:		"Dreamcast mouse",
	connect:	dc_mouse_connect,
	disconnect:	dc_mouse_disconnect,
	reply:		dc_mouse_callback,
	vblank:		maple_getcond_vblank_callback,
};


static int __init dc_mouse_init(void)
{
	maple_register_driver(&dc_mouse_driver);
	return 0;
}


static void __exit dc_mouse_exit(void)
{
	maple_unregister_driver(&dc_mouse_driver);
}


module_init(dc_mouse_init);
module_exit(dc_mouse_exit);

/*
 * Local variables:
 * c-basic-offset: 8
 * End:
 */
