/*+M*************************************************************************
 * Symbol Technologies Trilogy IEEE PCMCIA device driver for Linux.
 *
 * Copyright (c) 2000 Symbol Technologies Inc. -- http://www.symbol.com
 * All rights reserved.
 *
 * Developed for Symbol Technologies Inc. by TriplePoint, Inc.
 *   http://www.triplepoint.com
 *
 *---------------------------------------------------------------------------
 * This driver supports the following features:
 *   - Hot plug/unplug
 *   - Access Point and Ad-Hoc (peer-to-peer) communication
 *   - Card power management
 *
 *   Refer to the manual page for additional configuration, feature, and
 *   support information.
 *---------------------------------------------------------------------------
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * ALTERNATIVELY, this driver may be distributed under the terms of
 * the following license, in which case the provisions of this license
 * are required INSTEAD OF the GNU General Public License. (This clause
 * is necessary due to a potential bad interaction between the GPL and
 * the restrictions contained in a BSD-style copyright.)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, and the entire permission notice in its entirety,
 *    including the disclaimer of warranties.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *-M*************************************************************************/
#include "Spectrum24tApi.h"
#include "Spectrum24t.h"

// #define TEST_FULL_SCAN 1
#define MAX_IOVEC 8

typedef struct {
    __u16   Rid;
    __u32   ParamOffset;
    char    *Name;
} S24tGenricRidSetupType, *PS24tGenricRidSetupType;

DECLARE_WAIT_QUEUE_HEAD(ScanResultsWaitQueue);
DECLARE_WAIT_QUEUE_HEAD(GroupOrdWaitQueue);

#if DBG
static drv_info_t *DbgInfo = &S24tInfo;
STATIC char *S24tRidDebug(__u16 Rid);
#endif

STATIC void S24tReads(struct S24T_private *lp,__u8 *dest,__u16 offset,__u32 len);
STATIC void S24tWrites(struct S24T_private *lp,__u8 *src,__u16 offset,__u32 len);
STATIC int S24tReadMacAddress(struct S24T_private *lp);
STATIC void S24tParseInfoRec(struct S24T_private *lp);
STATIC int S24tSetMUEncryptionAlgorithm(struct S24T_private *lp);
STATIC int S24tGenericInit(struct S24T_private *lp);
STATIC int S24tWriteRidSetupRec(struct S24T_private *lp,PS24tGenricRidSetupType p);
STATIC int S24tSetMacAddress(struct S24T_private *lp);
STATIC int S24tGetFirmwareVersion(struct S24T_private *lp);
STATIC int S24tSetAuxPort(struct S24T_private *lp, __u32 Enable);
STATIC void S24tInitBpdu(struct S24T_private *lp,PENCAPSULATED_WLAP_BPDU_TYPE pBpdu);
STATIC int S24tSetBinaryRec(struct S24T_private *lp,__u32 Rid,__u8 *Buff,__u32 Len);
STATIC int S24tGetBinaryRec(struct S24T_private *lp,__u32 Rid,__u8 *Buff,__u32 *Len);
STATIC int S24tGetOrdinalsAddress(struct S24T_private *lp);

STATIC int S24tWaitForCommandNotBusy(struct S24T_private *lp,__u16 *CmdStatus);
#if TRILOGY3
STATIC int S24tStartGroupOrd(struct S24T_private *lp);
#endif

__inline__ __u32
S24tIoAddr(
    struct S24T_private *lp,
    __u32               offset
    )
{
    DBG_FUNC("S24tIoAddr")
    DBG_ASSERT(offset <= NIC_CCSR);
    return(lp->dev->base_addr + offset);
}

__inline__ __u8
S24tInb(
    struct S24T_private *lp,
    __u32               offset
    )
{
    __u8 Val = inb( S24tIoAddr(lp,offset) );

    DBG_IO(DbgInfo,"%04x %04x <-- %02x\n",S24tIoAddr(lp,0),(uint)offset,(uint)Val);

    return(Val);
}

__inline__ __u16
S24tInw(
    struct S24T_private *lp,
    __u32               offset
    )
{
    __u16 Val = inw( S24tIoAddr(lp,offset) );

    DBG_IO(DbgInfo,"%04x %04x <-- %04x\n",S24tIoAddr(lp,0),(uint)offset,(uint)Val);

    return(Val);
}

__inline__ void
S24tInsb(
    struct S24T_private *lp,
    __u32               offset,
    __u8                *dest,
    __u32               len
    )
{
    insb( S24tIoAddr(lp,offset), dest, len );

    DBG_IO(DbgInfo,"%04x %04x <-- len %u\n",S24tIoAddr(lp,0),(uint)offset,len);
}

__inline__ void
S24tInsw(
    struct S24T_private *lp,
    __u32               offset,
    __u8                *dest,
    __u32               wlen
    )
{
    insw( S24tIoAddr(lp,offset), dest, wlen );

    DBG_IO(DbgInfo,"%04x %04x <-- wlen %u\n",S24tIoAddr(lp,0),(uint)offset,wlen);
}

__inline void
S24tOutb(
    struct S24T_private *lp,
    __u32               offset,
    __u8                val
    )
{
    outb( val, S24tIoAddr(lp,offset) );

    DBG_IO(DbgInfo,"%04x %04x --> %02x\n",S24tIoAddr(lp,0),(uint)offset,(uint)val);
}

__inline void
S24tOutw(
    struct S24T_private *lp,
    __u32               offset,
    __u16               val
    )
{
    outw( val, S24tIoAddr(lp,offset) );

    DBG_IO(DbgInfo,"%04x %04x --> %04x\n",S24tIoAddr(lp,0),(uint)offset,(uint)val);
}

__inline__ void
S24tOutsb(
    struct S24T_private *lp,
    __u32               offset,
    __u8                *src,
    __u32               len
    )
{
    outsb( S24tIoAddr(lp,offset), src, len );

    DBG_IO(DbgInfo,"%04x %04x --> len %u\n",S24tIoAddr(lp,0),(uint)offset,len);
}

__inline__ void
S24tOutsw(
    struct S24T_private *lp,
    __u32               offset,
    __u8                *src,
    __u32               wlen
    )
{
    outsw( S24tIoAddr(lp,offset), src, wlen );

    DBG_IO(DbgInfo,"%04x %04x --> wlen %u\n",S24tIoAddr(lp,0),(uint)offset,wlen);
}

__u8
S24tReadCor(
    struct S24T_private *lp
    )
{
    DBG_FUNC("S24tReadCor")
    __u8 Cor=0;
    if (lp->Cor)
    {
        Cor = lp->Cor[OFFSET_COR];
    }
#if TRILOGY3
    else if (lp->CorIo)
    {
        Cor = (__u8)S24tInb(lp,lp->CorIo);
    }
#endif
    else
    {
        DBG_ASSERT(0);
    }
    return(Cor);
}

void
S24tWriteCor(
    struct S24T_private *lp,
    __u8                Cor
    )
{
    DBG_FUNC("S24tWriteCor")

    if (lp->Cor)
    {
        lp->Cor[OFFSET_COR] = Cor;
    }
#if TRILOGY3
    else if (lp->CorIo)
    {
        S24tOutb(lp,lp->CorIo,(__u16)Cor);
    }
#endif
    else
    {
        DBG_ASSERT(0);
    }
}

int
S24tCardInserted(
    struct S24T_private *lp
    )
{

    if (lp->CorIo)
    {
        return(1);
    }
    else
    {
        __u8 Cor = S24tReadCor(lp);
    
        if (Cor == COR_IO_MODE)
        {
            return(1);
        }
        else
        {
            DBG_PRINT("Spectrum24t not inserted %02x\n",(uint)Cor);
            return(0);
        }
    }
}

STATIC DECLARE_WAIT_QUEUE_HEAD(S24tPauseWaitQueue);
STATIC void
S24tPause(__u32 msec)
{
    __u32 njiffies = (msec*HZ)/1000;
    interruptible_sleep_on_timeout(&S24tPauseWaitQueue,njiffies);
}

int
S24tInit(
    struct S24T_private *lp
    )
{
    DBG_FUNC("S24tInit")
    int Status;
    int i;
    __u8 Mfg[CS_CIS_MFG_NAME_LEN];

    DBG_ENTER(DbgInfo);
#if TRILOGY3
    DBG_ASSERT(lp->Cor || lp->CorIo);
#else
    DBG_ASSERT(lp->Cor);
#endif
#if DBG
    printk(KERN_ERR "sizeof(Trilogy_TX_CTL) %d\n",sizeof(Trilogy_TX_CTL));
#if S24_PROC_SYS
    printk(KERN_ERR "sizeof(TRILOGY_XMIT_HEADER) %d\n",sizeof(TRILOGY_XMIT_HEADER));
    printk(KERN_ERR "sizeof(TRILOGY_HEADER) %d\n",sizeof(TRILOGY_HEADER));
    DBG_ASSERT(sizeof(TRILOGY_XMIT_HEADER) == 8);
    DBG_ASSERT(sizeof(TRILOGY_HEADER) == (sizeof(TRILOGY_MAC_HDR)+sizeof(TRILOGY_XMIT_HEADER)));
#endif
    printk(KERN_ERR "sizeof(TRILOGY_MAC_HDR) %d\n",sizeof(TRILOGY_MAC_HDR));
#endif
    DBG_ASSERT(sizeof(Trilogy_TX_CTL) == 14);
    DBG_ASSERT(sizeof(TRILOGY_MAC_HDR) == 60);
    DBG_ASSERT(sizeof(TRILOGY_MAC_HDR) == (sizeof(Trilogy_TX_CTL)+46));
    DBG_ASSERT(ETHERNET_HEADER_SIZE == 14);

    /*
     * Get the data queues initialized.
     */
    skb_queue_head_init(&lp->Tx802_3Queue);
    skb_queue_head_init(&lp->ToBeFreed);

#if TRILOGY3
    if (lp->CorIo)
    {
        lp->ulIoPAddr = lp->dev->base_addr;
        lp->uiEsquare = 1;
        Status = (-1);
        for (i=0; Status && (i<1); i++)
        {
            if (TrilogyDownLoadFirmwareToT3Adapter(lp) == NDIS_STATUS_SUCCESS)
            {
                Status = 0;
                break;
            }
        }
        if (Status)
        {
            DBG_PRINT("Could not download spectrum24 firmware.\n");
            return(Status);
        }
    }
#endif

    if (!S24tCardInserted(lp))
    {
        DBG_LEAVE(DbgInfo);
        return(-ENODEV);
    }

    Mfg[CS_CIS_MFG_NAME_LEN-1] = '\0';

    if (lp->Cor)
    {
        for (i=0; i<(CS_CIS_MFG_NAME_LEN-1); i++)
        {
            Mfg[i] = lp->Cor[(CS_CIS_MFG_NAME_OFFSET+i)*2];
        }

        DBG_NOTICE(DbgInfo, "MfgName '%s'\n",Mfg);
    }

    if ((Status = S24tCommand(lp,CR_INITIALIZE,NULL,NULL,NULL,NULL)))
    {
        DBG_PRINT("Failed to initialize.\n");
        DBG_LEAVE(DbgInfo);
        return(Status);
    }
    mdelay(200);
    S24tWrite16(lp,NIC_EVACK,0xffff);

    /*
     * Set the MAC address from the input options.
     */
    if (lp->MACAddress[0] | lp->MACAddress[1] | lp->MACAddress[2])
    {
        if ((Status = S24tSetMacAddress(lp)))
        {
            DBG_PRINT("Failed to set MAC address.\n");
            DBG_LEAVE(DbgInfo);
            return(Status);
        }
    }

    if ((Status=S24tReadMacAddress(lp)))
    {
        DBG_PRINT("Failed to read MAC address.\n");
        DBG_LEAVE(DbgInfo);
        return(Status);
    }

    if ((Status=S24tGetOrdinalsAddress(lp)))
    {
        DBG_PRINT("Failed to read ordinals table addresses.\n");
        DBG_LEAVE(DbgInfo);
        return(Status);
    }

    if ((Status = S24tSetMUEncryptionAlgorithm(lp)))
    {
        DBG_PRINT("Failed to set encryption algorithm.\n");
        DBG_LEAVE(DbgInfo);
        return(Status);
    }

    if ((Status=S24tGenericInit(lp)))
    {
        DBG_PRINT("Failed generic init.\n");
        DBG_LEAVE(DbgInfo);
        return(Status);
    }

    if ((Status = S24tSetEssId(lp,NULL)))
    {
        DBG_PRINT("Failed to set ESS.\n");
        DBG_LEAVE(DbgInfo);
        return(Status);
    }

    if ((Status = S24tSetStationName(lp,lp->StationName)))
    {
        DBG_PRINT("Failed to set station name.\n");
        DBG_LEAVE(DbgInfo);
        return(Status);
    }

    if ((Status = S24tGetFirmwareVersion(lp)))
    {
        DBG_PRINT("Failed to get firmware version.\n");
        DBG_LEAVE(DbgInfo);
        return(Status);
    }
    DBG_LEAVE(DbgInfo);
    return(0);
}

int
S24tEnable(
    struct S24T_private *lp,
    __u32 ApMode
    )
{
    DBG_FUNC("S24tEnable")
    int Status;
    __u16   Cmd;
    __u16   P0;

    DBG_ENTER(DbgInfo);

    /*
     * Remember if in AP mode.
     */
    lp->ApMode = ApMode;

    Cmd = (ApMode ? CR_ENABLE_AP : CR_ENABLE);

    if ((Status = S24tCommand(lp,Cmd,NULL,NULL,NULL,NULL)))
    {
        DBG_PRINT("Failed to enable.\n");
        DBG_LEAVE(DbgInfo);
        return(Status);
    }
    mdelay(5);

    P0 = 1585;
    if ((Status = S24tCommand(lp,CR_ALLOCATE,NULL,&P0,NULL,NULL)))
    {
        DBG_PRINT("S24tEnable - failed to start CR_ALLOCATE.\n");
        DBG_LEAVE(DbgInfo);
        return(Status);
    }

    Status = S24tIrqEnable(lp);

    /*
     * Pause for a 100 msec while the adapter gets its act together.
     */
    S24tPause(100);

    DBG_LEAVE(DbgInfo);
    return(Status);
}

int
S24tDisable(
    struct S24T_private *lp,
    __u32 ApMode
    )
{
    DBG_FUNC("S24tDisable")
    int Status;

    DBG_ENTER(DbgInfo);

    /*
     * Remember if in AP mode.
     */
    lp->ApMode = ApMode;

    S24tIrqDisable(lp);
    if ((Status = S24tCommand(lp,CR_DISABLE,NULL,NULL,NULL,NULL)))
    {
        DBG_PRINT("S24tDisable - failed to disable.\n");
        DBG_LEAVE(DbgInfo);
        return(Status);
    }

    /*
     * The transmit FID becomes invalid.
     */
    lp->txFid = 0;

    DBG_LEAVE(DbgInfo);
    return(Status);
}

int
S24tIrqEnable(
    struct S24T_private *lp
    )
{
    DBG_FUNC("S24tIrqEnable")

    DBG_ENTER(DbgInfo);

    if (!S24tCardInserted(lp))
    {
        DBG_LEAVE(DbgInfo);
        return(-ENXIO);
    }
    S24tWrite16(lp,NIC_INTEN,NIC_INTEN_MASK);

    DBG_ENTER(DbgInfo);
    return(0);
}

int
S24tIrqDisable(
    struct S24T_private *lp
    )
{
    DBG_FUNC("S24tIrqDisable")

    DBG_ENTER(DbgInfo);

    if (!S24tCardInserted(lp))
    {
        DBG_LEAVE(DbgInfo);
        return(-ENXIO);
    }
    S24tWrite16(lp,NIC_INTEN,0);

    DBG_LEAVE(DbgInfo);
    return(0);
}

int
S24tEvStat(
    struct S24T_private *lp,
    __u16               *EvStat
    )
{
    if (!S24tCardInserted(lp))
    {
        return(-ENXIO);
    }
    *EvStat = S24tRead16(lp,NIC_EVSTAT);

    return(0);
}

int
S24tEvAck(
    struct S24T_private *lp,
    __u16               EvAck
    )
{
    DBG_FUNC("S24tEvAck")

    DBG_ENTER(DbgInfo);

    if (!S24tCardInserted(lp))
    {
        DBG_LEAVE(DbgInfo);
        return(-ENXIO);
    }
    S24tWrite16(lp,NIC_EVACK,EvAck);

    DBG_LEAVE(DbgInfo);
    return(0);
}

int
S24tGetTxFid(
    struct S24T_private *lp
    )
{
    DBG_FUNC("S24tGetTxFid")

    DBG_ENTER(DbgInfo);

    if (!S24tCardInserted(lp))
    {
        DBG_LEAVE(DbgInfo);
        return(-ENXIO);
    }

    DBG_ASSERT(!lp->txFid);

    lp->txFid = S24tRead16(lp,NIC_ALLOC_FID);

    DBG_ASSERT(lp->txFid);

    DBG_TX(DbgInfo, "TxFid %04x\n",lp->txFid);

    DBG_LEAVE(DbgInfo);
    return(0);
}

STATIC int
S24tCopyDownVec(
    struct S24T_private *lp,
    __u16               Fid,
    __u16               *P1,
    __u16               *P2,
    __u32               Command,
    struct iovec        *vec,
    __u32               nvec
    )
{
    DBG_FUNC("S24tCopyDownVec")
    int Status = 0;
    __u16 Offset;
    __u32 i;
    __u32 elen;

    DBG_ENTER(DbgInfo);

    Offset = S24tRead16(lp,NIC_OFFSET0);
    if (Offset & OFFSET_REGISTER_BUSY)
    {
        DBG_PRINT("S24tCopyDown offset reg busy.\n");
        DBG_LEAVE(DbgInfo);
        return(-EBUSY);
    }

    S24tWrite16(lp,NIC_SELECT0,Fid);
    S24tWrite16(lp,NIC_OFFSET0,0);

    do {
        Offset = S24tRead16(lp,NIC_OFFSET0);
    } while ((S24tCardInserted(lp)) && (Offset & OFFSET_REGISTER_BUSY));

    DBG_ASSERT(nvec);
    DBG_ASSERT(vec);

    /*
     * Send out all but the last vector.
     */
    for (i=0; i<(nvec-1); i++)
    {
        DBG_ASSERT(!(vec[i].iov_len & 1));
        S24tWrites(lp,vec[i].iov_base,NIC_DATA0,vec[i].iov_len);
    }

    /*
     * Only the last vector can be odd length.
     */
    if ((elen=vec[nvec-1].iov_len) & 1)
    {
        elen++;
    }
    S24tWrites(lp,vec[nvec-1].iov_base,NIC_DATA0,elen);

    if (Command)
    {
        Status = S24tCommand(lp,Command,NULL,&Fid,P1,P2);
        /*
         * txFid is busy until the firmware reallocates it.
         */
        if ((Command & (CR_TRANSMIT|CR_RECLAIM)) == (CR_TRANSMIT|CR_RECLAIM))
        {
            lp->txFid = 0;
        }
    }
    DBG_LEAVE(DbgInfo);
    return(Status);
}

STATIC int
S24tCopyDown(
    struct S24T_private *lp,
    __u16               Fid,
    __u16               *P1,
    __u16               *P2,
    __u8                *Buff1,
    __u32               Len1,
    __u8                *Buff2,
    __u32               Len2,
    __u8                *Buff3,
    __u32               Len3,
    __u32               Command
    )
{
    DBG_FUNC("S24tCopyDown")

    int Status;
    struct iovec        vec[3];
    __u32               nvec=0;

    DBG_ENTER(DbgInfo);

    DBG_ASSERT(Len1);
    DBG_ASSERT(Buff1);

    vec[nvec].iov_base = Buff1;
    vec[nvec].iov_len = Len1;
    nvec++;

    if (Buff2 && Len2)
    {
        vec[nvec].iov_base = Buff2;
        vec[nvec].iov_len = Len2;
        nvec++;

        if (Buff3 && Len3)
        {
            vec[nvec].iov_base = Buff3;
            vec[nvec].iov_len = Len3;
            nvec++;
        }
    }

    Status = S24tCopyDownVec(lp,Fid,P1,P2,Command,vec,nvec);
    DBG_LEAVE(DbgInfo);
    return(Status);
}

STATIC int
S24tCopyUp(
    struct S24T_private *lp,
    __u16               Rid,
    __u8                *Buff,
    __u32               Len,
    __u16               Command
    )
{
    DBG_FUNC("S24tCopyUp")

    int Status;
    __u16 OffsetReg;
    __u16 tRid;

    DBG_ENTER(DbgInfo);

    DBG_ASSERT(!(Len & 1));

    if ((!Buff) || (!Len))
    {
        DBG_LEAVE(DbgInfo);
        return(0);
    }

    if (Command)
    {
        tRid = Rid;
        Status =
        S24tCommand(
            lp,
            Command,
            NULL,
            &tRid,
            NULL,
            NULL
        );
        if (Status)
        {
            DBG_PRINT("S24tCopyUp - CR_ACCESS failed.\n");
            DBG_LEAVE(DbgInfo);
            return(Status);
        }
    }

    OffsetReg = S24tRead16(lp,NIC_OFFSET1);
    if (OffsetReg & OFFSET_REGISTER_BUSY)
    {
        DBG_PRINT("S24tCopyUp - Offset reg busy.\n");
        DBG_LEAVE(DbgInfo);
        return(-EBUSY);
    }

    S24tWrite16(lp,NIC_SELECT1,Rid);
    S24tWrite16(lp,NIC_OFFSET1,0);

    do {
        OffsetReg = S24tRead16(lp,NIC_OFFSET1);
        if (!(OffsetReg & OFFSET_REGISTER_BUSY))
        {
            break;
        }
        udelay(10);
    } while (S24tCardInserted(lp));

    S24tReads(lp,Buff,NIC_DATA1,Len);

    DBG_LEAVE(DbgInfo);
    return(0);
}

STATIC void
S24tCopyUpSecond(
    struct S24T_private *lp,
    __u8                *Buff,
    __u32               Len
    )
{
    DBG_FUNC("S24tCopyUpSecond")
    DBG_ENTER(DbgInfo);

    S24tReads(lp,Buff,NIC_DATA1,Len);

    DBG_LEAVE(DbgInfo);
}

#if DBG
#define MAX_PKT_SIZ (sizeof(TRILOGY_MAC_HDR)+MAX_ETHERNET_BODY_SIZE+RFC1042_SIZE)
__u8 PktBuff[MAX_PKT_SIZ];

void S24tPrintPacket(
    struct iovec        *vec,
    __u32               nvec,
    char                *Prefix
    )
{
    __u32   ivec=0;
    __u32   tlen=0;
    __u32   i=0;
    __u8    *d;

    PHDR_802_3 Hdr = (PHDR_802_3)PktBuff;

    if (!strncmp(Prefix,"Rx",2))
    {
        if (!(DBG_FLAGS(DbgInfo) & DBG_BR_RX_ON))
        {
            return;
        }
    }
    if (!strncmp(Prefix,"Tx",2))
    {
        if (!(DBG_FLAGS(DbgInfo) & DBG_BR_TX_ON))
        {
            return;
        }
    }

    for (ivec=0; ivec<nvec; ivec++)
    {
        d = vec[ivec].iov_base;
        for (tlen=0; tlen<vec[ivec].iov_len; tlen++)
        {
            PktBuff[i++] = d[tlen];
        }
    }

    if (Prefix)
    {
        printk(KERN_INFO "%s",Prefix);
    }
    printk("%02x:%02x:%02x:%02x:%02x:%02x ",
        Hdr->ucDestAddr[0],Hdr->ucDestAddr[1],Hdr->ucDestAddr[2],Hdr->ucDestAddr[3],Hdr->ucDestAddr[4],Hdr->ucDestAddr[5]
    );
    printk("%02x:%02x:%02x:%02x:%02x:%02x ",
        Hdr->ucSrcAddr[0],Hdr->ucSrcAddr[1],Hdr->ucSrcAddr[2],Hdr->ucSrcAddr[3],Hdr->ucSrcAddr[4],Hdr->ucSrcAddr[5]
    );
    printk("%04x ",Hdr->usLength);

    d = &PktBuff[ETHERNET_HEADER_SIZE];
    for (tlen=ETHERNET_HEADER_SIZE; tlen<i; tlen++)
    {
        printk("%02x ",PktBuff[tlen]);
    }
    printk("\n");
}

void S24tPrintPacketWithTrilogyHeader(
    struct iovec        *vec,
    __u32               nvec,
    __u32               Tx
    )
{
    __u32   ivec=0;
    __u32   tlen=0;
    __u32   i=0;

    PTRILOGY_MAC_HDR T;

    if (Tx)
    {
        if (!(DBG_FLAGS(DbgInfo) & DBG_BR_TX_ON))
        {
            return;
        }
    }
    else
    {
        if (!(DBG_FLAGS(DbgInfo) & DBG_BR_RX_ON))
        {
            return;
        }
    }

    for (ivec=0; ivec<nvec; ivec++)
    {
        __u8 *d = vec[ivec].iov_base;
        for (tlen=0; tlen<vec[ivec].iov_len; tlen++)
        {
            PktBuff[i++] = d[tlen];
        }
    }

    T = (PTRILOGY_MAC_HDR)PktBuff;
    if (Tx)
    {
        printk(KERN_INFO "Tx: CTRL:%04x ",T->u.tx.usTxControl);
    }
    else
    {
        printk(KERN_INFO "Rx: STAT:%04x ",T->u.rx.usStatus);
    }
    printk("FC:%04x ",T->usFrameControl);
    printk("A1-%02x:%02x:%02x:%02x:%02x:%02x ",
        T->Dst[0],T->Dst[1],T->Dst[2],T->Dst[3],T->Dst[4],T->Dst[5]
    );
    printk("A2-%02x:%02x:%02x:%02x:%02x:%02x ",
        T->Src[0],T->Src[1],T->Src[2],T->Src[3],T->Src[4],T->Src[5]
    );
    printk("A3-%02x:%02x:%02x:%02x:%02x:%02x ",
        T->BssId[0],T->BssId[1],T->BssId[2],T->BssId[3],T->BssId[4],T->BssId[5]
    );
    printk("A4-%02x:%02x:%02x:%02x:%02x:%02x ",
        T->ucAddress4[0],T->ucAddress4[1],T->ucAddress4[2],T->ucAddress4[3],T->ucAddress4[4],T->ucAddress4[5]
    );
    printk("802.3-%02x:%02x:%02x:%02x:%02x:%02x ",
        T->Dot3Dest[0],T->Dot3Dest[1],T->Dot3Dest[2],T->Dot3Dest[3],T->Dot3Dest[4],T->Dot3Dest[5]
    );
    printk("%02x:%02x:%02x:%02x:%02x:%02x ",
        T->Dot3Src[0],T->Dot3Src[1],T->Dot3Src[2],T->Dot3Src[3],T->Dot3Src[4],T->Dot3Src[5]
    );
    printk("%04x ",T->Dot3Len);

    for (tlen=sizeof(TRILOGY_MAC_HDR); tlen<i; tlen++)
    {
        printk("%02x ",PktBuff[tlen]);
    }
    printk("\n");

}

#endif

#if S24_PROC_SYS
static int
S24TxApVec(
    struct S24T_private *lp,
    struct sk_buff      *skb
    )
{
    DBG_FUNC("S24tTxAp")

    int                     Status;
    PSCORCH_RW              Scorch;
    PTRILOGY_HEADER         Trilogy;
    TRILOGY_XMIT_HEADER     *Xmt;
    __u16                   TransmitParameters;
    FRAME_CONTROL           *Fc;

#   define                  MAX_TVEC MAX_IOVEC
    struct iovec            tvec[MAX_TVEC];
    __u32                   ntvec;

    DBG_ENTER(DbgInfo);

    DBG_ASSERT(lp->txFid);
    DBG_ASSERT(skb->len >= (sizeof(SCORCH_DEVICE_HEADER)+sizeof(TRILOGY_HEADER)));

    Scorch = (PSCORCH_RW)skb->data;
    Trilogy = &Scorch->Data.Trilogy;
    Xmt = &Trilogy->u.Xmt;
    Fc = (FRAME_CONTROL *)&Trilogy->Hdr.usFrameControl;

    /*
     * Since this packet originated from the AP application, assume the frame control
     * is correct except for the more bit.
     */
    Fc->more_data = S24tIsPktWithMoreBit(skb) ? 1 : 0;

    /*
     * Only the Trilogy MAC header on down gets transmitted.
     */
    tvec[0].iov_base = &Trilogy->Hdr;
    tvec[0].iov_len = skb->len - (sizeof(SCORCH_DEVICE_HEADER)+sizeof(TRILOGY_XMIT_HEADER));
    ntvec = 1;

    /*
     * Tell the CW10 what rate to transmit this packet.
     */
    TransmitParameters = (__u16)(Xmt->TransmitRate / TX_RATE_UNITS);
    TransmitParameters |= ((__u16)(Xmt->AttemptsThreshHold<<8));

    /*
     * Tell the firmware to interrupt on completion or error.
     */
    Trilogy->Hdr.u.tx.usTxControl = (TX_CTL_TX_INT|TX_CTL_TX_EX|TX_CTL_802_11);

    Status =
    S24tCopyDownVec(
        lp,
        lp->txFid,
        &TransmitParameters,
        NULL,
        CR_TRANSMIT|CR_RECLAIM,
        tvec,
        ntvec
    );
    if (Status)
    {
        DBG_PRINT("%s transmit failed.\n",__FUNC__);
    }

    DBG_LEAVE(DbgInfo);
    return(Status);
}
#endif

#define H802_2_SNAP_LEN 6
#define H802_11_SNAP_LEN (H802_2_SNAP_LEN+2)
STATIC __u8 Snap802_2[H802_2_SNAP_LEN] =       { 0xAA, 0xAA, 0x03, 0x00, 0x00, 0x00 };

static int
S24tMakeTxSNAP(
    struct S24T_private *lp,
    TRILOGY_MAC_HDR     *T,
    PHDR_802_3          pHdr,
    struct iovec        *Vec,
    __u8                *Buff,
    __u32               Len
    )
{
    DBG_FUNC("S24tMakeTxSNAP")

    int         nvec=0;
    PHDR_802_3  Hdr = (PHDR_802_3)Buff;
    __u16       DataFrameLen;

    DBG_ENTER(DbgInfo);

    Vec[nvec].iov_base = (void *)T;
    Vec[nvec].iov_len = sizeof(TRILOGY_MAC_HDR) - ETHERNET_HEADER_SIZE;
    nvec++;

    memset(T,0,sizeof(TRILOGY_MAC_HDR));

    T->u.tx.usTxControl = __cpu_to_le16(TX_CTL_802_3|TX_CTL_TX_EX|TX_CTL_TX_INT);

    /*
     * Check for SNAP tunnel needed.
     */
    if (__be16_to_cpu(Hdr->usLength) > (MAX_ETHERNET_BODY_SIZE+H802_11_SNAP_LEN))
    {
        /*
         * Insert 2 bytes of length and 6 bytes of SNAP just prior to the original
         * ethernet DIX type field.
         */
        DataFrameLen = (Len - ETHERNET_HEADER_SIZE) + H802_11_SNAP_LEN;

        Vec[nvec].iov_base = (void *)pHdr;
        Vec[nvec].iov_len = ETHERNET_HEADER_SIZE;
        nvec++;

        /*
         * Create a new 802.3 header with a length field.
         */
        memcpy(pHdr,Hdr,2*ETH_ALEN);
        pHdr->usLength = __cpu_to_be16(DataFrameLen);

        /*
         * Insert the SNAP wrapper.
         */
        Vec[nvec].iov_base = (void *)Snap802_2;
        Vec[nvec].iov_len = H802_2_SNAP_LEN;
        nvec++;

        /*
         * Output the rest of the original data frame (and DIX type).
         */
        Vec[nvec].iov_base = (void *)&Hdr->usLength;
        Vec[nvec].iov_len = Len - (2*ETH_ALEN);
        nvec++;
    }
    /*
     * If the length field is not a type, then this packet is probably already
     * SNAP encapsulated.
     */
    else
    {
#if DBG
        memcpy(pHdr,Hdr,ETHERNET_HEADER_SIZE);
#endif
        DataFrameLen = Len - ETHERNET_HEADER_SIZE;

        Vec[nvec].iov_base = (void *)Hdr;
        Vec[nvec].iov_len = Len;
        nvec++;
    }

    T->usDataLen = __cpu_to_le16(DataFrameLen);

    DBG_LEAVE(DbgInfo);

    return(nvec);
}

/*
 * Translate a Linux BPDU announcement into a WLAP BPDU announcement. The Linux BPDU is 
 * an original SNAP packet. A WLAP BPDU is a tunnelled DIX packet with DIX type 8287H.
 */
static int
S24tMakeBPDUAnnounce(
    struct S24T_private *lp,
    TRILOGY_MAC_HDR     *T,
    PHDR_802_3          pHdr,
    struct iovec        *Vec,
    __u8                *Buff,
    __u32               Len
    )
{
    DBG_FUNC("S24tMakeBPDUAnnounce")

    int         nvec;
    __u16       *SnapTyp;
    PHDR_802_3  Hdr = (PHDR_802_3)Buff;

    DBG_ENTER(DbgInfo);

    SnapTyp = &Hdr->usLength;
    SnapTyp++;

    if (
        (!lp->Param[PromiscuousMode])
        ||
        (__be16_to_cpu(Hdr->usLength) > MAX_ETHERNET_BODY_SIZE)
        ||
        (*SnapTyp != __cpu_to_be16(SNAP_BPDU))
       )
    {
        DBG_LEAVE(DbgInfo);
        return(0);
    }

    if (
        lp->UseHelloTime
        &&
        ((lp->LastHelloTime + __be16_to_cpu(lp->Bpdu.Bpdu.HelloTime)) > (jiffies/HZ))
       )
    {
        DBG_LEAVE(DbgInfo);
        return(-1);
    }
    lp->LastHelloTime = jiffies/HZ;

    nvec = S24tMakeTxSNAP(lp,T,pHdr,Vec,(__u8 *)&lp->Bpdu,SIZEOF_ENCAPSULATED_WLAP_BPDU_TYPE);

    T->u.tx.usTxControl = __cpu_to_le16(TX_CTL_802_11|TX_CTL_TX_EX|TX_CTL_TX_INT);
    T->usFrameControl = __cpu_to_le16(FCTL_DATA|FCTL_TO_DS|FCTL_FROM_DS);

    memcpy(T->Dst,           lp->BssId,       ETH_ALEN);
    memcpy(T->Src,           lp->MACAddress,  ETH_ALEN);
    memcpy(T->BssId,         Hdr->ucDestAddr, ETH_ALEN);
    memcpy(T->ucAddress4,    Hdr->ucSrcAddr,  ETH_ALEN);

    DBG_LEAVE(DbgInfo);
    return(nvec);
}

/*
 * Make a bridged packet.
 */
static int
S24tMakeBPDU(
    struct S24T_private *lp,
    TRILOGY_MAC_HDR     *T,
    PHDR_802_3          pHdr,
    struct iovec        *Vec,
    __u8                *Buff,
    __u32               Len
    )
{
    DBG_FUNC("S24tMakeBPDU")
    int         nvec;
    PHDR_802_3  Hdr = (PHDR_802_3)Buff;

    DBG_ENTER(DbgInfo);

    /*
     * Don't create a bridged packet if not in promiscuous mode.
     */
    if (
        (!lp->Param[PromiscuousMode])
       )
    {
        DBG_LEAVE(DbgInfo);
        return(0);
    }

    nvec = S24tMakeTxSNAP(lp,T,pHdr,Vec,Buff,Len);

    T->u.tx.usTxControl = __cpu_to_le16(TX_CTL_802_11|TX_CTL_TX_EX|TX_CTL_TX_INT);
    T->usFrameControl = __cpu_to_le16(FCTL_DATA|FCTL_TO_DS|FCTL_FROM_DS);

    memcpy(T->Dst,           lp->BssId,       ETH_ALEN);
    memcpy(T->Src,           lp->MACAddress,  ETH_ALEN);
    memcpy(T->BssId,         Hdr->ucDestAddr, ETH_ALEN);

    if (!is_valid_ether_addr(Hdr->ucSrcAddr))
    {
        DBG_PRINT("Invalid source address %02x:%02x:%02x:%02x:%02x:%02x\n",
            Hdr->ucSrcAddr[0],
            Hdr->ucSrcAddr[1],
            Hdr->ucSrcAddr[2],
            Hdr->ucSrcAddr[3],
            Hdr->ucSrcAddr[4],
            Hdr->ucSrcAddr[5]
        );
    }

    memcpy(T->ucAddress4,    Hdr->ucSrcAddr,  ETH_ALEN);

    DBG_LEAVE(DbgInfo);
    return(nvec);
}

#if S24_PROC_SYS
static int
S24tTxAp802_3(
    struct S24T_private *lp,
    struct sk_buff      *skb,
    TRILOGY_MAC_HDR     *T,
    __u16               *pTransmitParameters
    )
{
    DBG_FUNC("S24tTxAp802_3")

    PHDR_802_3      Hdr = (PHDR_802_3)skb->data;
    FRAME_CONTROL   *Fc;

    DBG_ASSERT(lp->ApMode);
    DBG_ASSERT(lp->HashTab);
    DBG_ASSERT(skb->len >= sizeof(HDR_802_3));

    /*
     * Initialize frame and transmit control.
     */
    Fc              = (FRAME_CONTROL *)&T->usFrameControl;
    Fc->type        = TYPE_DATA;
    Fc->subtype     = SUBTYPE_DATA;
    Fc->from_ds     = 1;
    Fc->to_ds       = 0;
    Fc->more_data   = S24tIsPktWithMoreBit(skb) ? 1 : 0;

    T->u.tx.usTxControl = __cpu_to_le16(TX_CTL_802_11|TX_CTL_TX_EX|TX_CTL_TX_INT);

    /*
     * All directed addresses must exist in the MU hash table.
     */
    if (!(Hdr->ucDestAddr[0] & 1))
    {
        PMuHashTableElemType Elem;

        if (!(Elem = MuHashFind(lp->HashTab,Hdr->ucDestAddr,1)))
        {
            DBG_BR_TX(DbgInfo,"Non-existant Tx destination %02x:%02x:%02x:%02x:%02x:%02x\n",
                Hdr->ucDestAddr[0],Hdr->ucDestAddr[1],Hdr->ucDestAddr[2],Hdr->ucDestAddr[3],Hdr->ucDestAddr[4],Hdr->ucDestAddr[5]
            );
            return(-1);
        }

        /*
         * Setup the transmit rate and attempts thresh hold values.
         */
        *pTransmitParameters = (__u16)(Elem->TransmitRate / TX_RATE_UNITS);
        *pTransmitParameters |= (__u16)(Elem->AttemptsThreshHold << 8);

        /*
         * If the destination address is not the same as the MU, then make this a
         * bridge packet.
         */
        if (!MuHashIsMu(Elem))
        {
            DBG_BR_TX(DbgInfo,"Bridge to %02x:%02x:%02x:%02x:%02x:%02x via %02x:%02x:%02x:%02x:%02x:%02x\n",
                Hdr->ucDestAddr[0],Hdr->ucDestAddr[1],Hdr->ucDestAddr[2],Hdr->ucDestAddr[3],Hdr->ucDestAddr[4],Hdr->ucDestAddr[5],
                Elem->MuAddress[0],Elem->MuAddress[1],Elem->MuAddress[2],Elem->MuAddress[3],Elem->MuAddress[4],Elem->MuAddress[5]
            );

            Fc->to_ds = 1;
            memcpy(T->Dst,Elem->MuAddress,ETH_ALEN);
            memcpy(T->Src,lp->MACAddress,ETH_ALEN);
            memcpy(T->BssId, Hdr->ucDestAddr,ETH_ALEN);
            memcpy(T->ucAddress4, Hdr->ucSrcAddr, ETH_ALEN);
        }
    }
    else
    {
        *pTransmitParameters = (__u16)(lp->BcastRate / TX_RATE_UNITS);
    }

    /*
     * It is a simple FROM_DS packet.
     */
    if (!Fc->to_ds)
    {
        memcpy(T->Dst,Hdr->ucDestAddr,ETH_ALEN);
        memcpy(T->Src,lp->MACAddress,ETH_ALEN);
        memcpy(T->BssId, Hdr->ucSrcAddr,ETH_ALEN);
    }

    return(0);
}
#endif

static int
S24tTx802_3(
    struct S24T_private *lp,
    struct sk_buff      *skb
    )
{
    DBG_FUNC("S24tTx802_3")

    HDR_802_3       Hdr;
    struct iovec    Vec[MAX_IOVEC];
    int             Nvec=0;
    TRILOGY_MAC_HDR TrilogyHeader;
    int             status=(-1);
    __u16           *pTransmitParameters=NULL;
#if S24_PROC_SYS
    __u16           TransmitParameters;
#endif

    DBG_ENTER(DbgInfo);
    DBG_ASSERT(lp->txFid);

    if (skb->len <= ETHERNET_HEADER_SIZE)
    {
        DBG_PRINT("%s - Bogus packet length, only %d bytes.\n",__FUNC__,skb->len);
        DBG_LEAVE(DbgInfo);
        return(-EIO);
    }

    if (!lp->Param[PromiscuousMode])
    {
        Nvec = S24tMakeTxSNAP(lp,&TrilogyHeader,&Hdr,Vec,skb->data,skb->len);
    }
    /*
     * Returns (-1) if the transmit is to be ignored.
     */
    else if (!(Nvec = S24tMakeBPDUAnnounce(lp,&TrilogyHeader,&Hdr,Vec,skb->data,skb->len)))
    {
        if (!(Nvec=S24tMakeBPDU(lp,&TrilogyHeader,&Hdr,Vec,skb->data,skb->len)))
        {
            Nvec = S24tMakeTxSNAP(lp,&TrilogyHeader,&Hdr,Vec,skb->data,skb->len);
        }
    }

#if S24_PROC_SYS
    if (lp->ApMode && (Nvec > 0))
    {
#if DBG
        FRAME_CONTROL   *Fc;
        __u8 *Dst;
        __u8 *Src;
#endif
        pTransmitParameters = &TransmitParameters;
        if ((status=S24tTxAp802_3(lp,skb,&TrilogyHeader,pTransmitParameters)) < 0)
        {
            DBG_LEAVE(DbgInfo);
            return(status);
        }
#if DBG
        Fc = (FRAME_CONTROL *)&TrilogyHeader.usFrameControl;

        if (Fc->to_ds && Fc->from_ds)
        {
            Dst = TrilogyHeader.BssId;
            Src = TrilogyHeader.ucAddress4;
        }
        else
        {
            DBG_ASSERT(Fc->from_ds);
            Dst = TrilogyHeader.Dst;
            Src = TrilogyHeader.BssId;
        }
#endif
        DBG_BR_TX(DbgInfo,"Tx FC:%04x %02x:%02x:%02x:%02x:%02x:%02x %02x:%02x:%02x:%02x:%02x:%02x %04x\n",
            TrilogyHeader.usFrameControl,
            Dst[0],Dst[1],Dst[2],Dst[3],Dst[4],Dst[5],
            Src[0],Src[1],Src[2],Src[3],Src[4],Src[5],
            TrilogyHeader.usDataLen
        );
    }
#endif

    if (
        (Nvec > 0)
        &&
        ((status = S24tCopyDownVec(lp,lp->txFid,pTransmitParameters,NULL,CR_TRANSMIT|CR_RECLAIM,Vec,(__u32)Nvec)) < 0)
       )
    {
        DBG_PRINT("%s transmit failed.\n",__FUNC__);
    }

    DBG_LEAVE(DbgInfo);
    return(status);
}

void
S24tStartTx(
    struct S24T_private *lp
    )
{
    DBG_FUNC("S24tStartTx")

    int             status;
    struct sk_buff  *skb=NULL;
    unsigned long   flags;
    __u32           skb_dequeued=1;

    DBG_ENTER(DbgInfo);

    local_irq_save(flags);

    if (!S24tCardInserted(lp))
    {
        local_irq_restore(flags);
        DBG_LEAVE(DbgInfo);
        return;
    }

    while (skb_dequeued && lp->txFid)
    {
        skb_dequeued = 0;

#if S24_PROC_SYS
        if ((skb = __skb_dequeue(&lp->ApTxQueue)))
        {
            skb_dequeued = 1;
            status = S24TxApVec(lp,skb);
        }
        else
#endif
        if ((skb = __skb_dequeue(&lp->Tx802_3Queue)))
        {
            skb_dequeued = 1;
            status = S24tTx802_3(lp,skb);
        }

        /*
         * Schedule the SKB to be freed.
         */
        if (skb)
        {
            /*
             * Free the SKB sometime later.
             */
            __skb_queue_tail(&lp->ToBeFreed,skb);
        }
    }

    local_irq_restore(flags);

    DBG_LEAVE(DbgInfo);
}

/*
 * Enqueue and attempt to transmit 802.3 frames.
 */
void
S24tTx(
    struct S24T_private *lp,
    struct sk_buff      *skb
    )
{
    DBG_FUNC("S24tTx")

    unsigned long flags;

    DBG_ENTER(DbgInfo);

    local_irq_save(flags);

#if S24_PROC_SYS
    /*
     * Init out of band info in the SKB.
     */
    S24tOobInit(skb);
    S24tApMgmtPkt(skb,0);

    skb = S24tApFilterTx(lp,skb);
#endif

    if (skb)
    {
        __skb_queue_tail(&lp->Tx802_3Queue,skb);
    }

    S24tStartTx(lp);

    /*
     * Flush transmitted SKBs.
     */
    __skb_queue_purge(&lp->ToBeFreed);

    local_irq_restore(flags);

    DBG_LEAVE(DbgInfo);

}

#if S24_PROC_SYS
struct sk_buff *
S24tRxApData(
    struct S24T_private *lp,
    PTRILOGY_MAC_HDR     Mac
    )
{
    DBG_FUNC("S24tRxApData")

    struct sk_buff          *skb=NULL;
    __u32                   skbLen;
    PMuHashTableElemType    Mu;
    __u32                   AddressedToMe;
    __u16                   Fc = le16_to_cpu(Mac->usFrameControl);

    /*
     * Make sure the sender is known and that I'm the destination.
     */
    if (
        (Fc & FCTL_TO_DS)
        &&
        (Fc & FCTL_FROM_DS)
       )
    {
        Mu = MuHashFind(lp->HashTab,Mac->Src,0);
        AddressedToMe = (!memcmp(Mac->Dst,lp->MACAddress,ETH_ALEN));
    }
    else if (Fc & FCTL_TO_DS)
    {
        if (!(Mu = MuHashFind(lp->HashTab,Mac->Src,0)))
        {
            Mu = MuHashFind(lp->HashTab,Mac->Dot3Src,0);
        }
        AddressedToMe = (!memcmp(Mac->Dst,lp->MACAddress,ETH_ALEN));
    }
    else if (Fc & FCTL_FROM_DS)
    {
        Mu = MuHashFind(lp->HashTab,Mac->BssId,0);
        AddressedToMe = (!memcmp(Mac->Src,lp->MACAddress,ETH_ALEN));
    }
    else
    {
        Mu = MuHashFind(lp->HashTab,Mac->Src,0);
        AddressedToMe = (!memcmp(Mac->BssId,lp->MACAddress,ETH_ALEN));
    }

    if ((!Mu) || (!AddressedToMe))
    {
        DBG_BR_RX(DbgInfo,"Not addressed to me - Mu %08x AddressedToMe %u FC:%04x %02x:%02x:%02x:%02x:%02x:%02x %02x:%02x:%02x:%02x:%02x:%02x %02x:%02x:%02x:%02x:%02x:%02x %02x:%02x:%02x:%02x:%02x:%02x %02x:%02x:%02x:%02x:%02x:%02x\n",
            (__u32)Mu,AddressedToMe,
            Mac->usFrameControl,
            Mac->Dst[0],Mac->Dst[1],Mac->Dst[2],Mac->Dst[3],Mac->Dst[4],Mac->Dst[5],
            Mac->Src[0],Mac->Src[1],Mac->Src[2],Mac->Src[3],Mac->Src[4],Mac->Src[5],
            Mac->BssId[0],Mac->BssId[1],Mac->BssId[2],Mac->BssId[3],Mac->BssId[4],Mac->BssId[5],
            Mac->ucAddress4[0],Mac->ucAddress4[1],Mac->ucAddress4[2],Mac->ucAddress4[3],Mac->ucAddress4[4],Mac->ucAddress4[5],
            Mac->Dot3Src[0],Mac->Dot3Src[1],Mac->Dot3Src[2],Mac->Dot3Src[3],Mac->Dot3Src[4],Mac->Dot3Src[5]
        );
        return(NULL);
    }

    if (__cpu_to_le16(Mac->usDataLen) > (MAX_ETHERNET_BODY_SIZE+H802_11_SNAP_LEN))
    {
        DBG_PRINT("Received AP data packet too long %u\n",(__u32)__cpu_to_le16(Mac->usDataLen));
        return(NULL);
    }

    skbLen = __cpu_to_le16(Mac->usDataLen) + ETHERNET_HEADER_SIZE;

    skb = dev_alloc_skb(skbLen+2);
    if (!skb)
    {
        DBG_PRINT("Could not allocate RX data packet len %u\n",skbLen);
        return(skb);
    }
    skb_reserve(skb, 2);

    /*
     * Note that this is a data packet SKB.
     */
    S24tOobInit(skb);
    S24tApMgmtPkt(skb,0);

    /*
     * Get the rest of the packet. Perform SNAP translation on the fly.
     */
    memcpy(skb->data,Mac->Dot3Dest,ETHERNET_HEADER_SIZE);

    if (__le16_to_cpu(Mac->u.rx.usStatus) & RX_TUNNELED)
    {
        DBG_ASSERT(skbLen >= H802_11_SNAP_LEN);

        /*
         * Strip off the tunnel SNAP header.
         */
        S24tCopyUpSecond(lp,skb->data+(2*ETH_ALEN),H802_2_SNAP_LEN);
        skbLen -= H802_2_SNAP_LEN;

        DBG_ASSERT(skbLen);
        S24tCopyUpSecond(lp,skb->data+(2*ETH_ALEN),skbLen - ETHERNET_HEADER_SIZE);
    }
    else if (skbLen > ETHERNET_HEADER_SIZE)
    {
        S24tCopyUpSecond(lp,skb->data+ETHERNET_HEADER_SIZE,skbLen-ETHERNET_HEADER_SIZE);
    }
    else
    {
        DBG_PRINT("%s:%d skbLen %d\n",__FUNC__,__LINE__,skbLen);
        __skb_queue_tail(&lp->ToBeFreed,skb);
        return(NULL);
    }


    DBG_BR_RX(DbgInfo, "Rx: Stat:%04x FC:%04x %02x:%02x:%02x:%02x:%02x:%02x %02x:%02x:%02x:%02x:%02x:%02x %04x\n",
        __le16_to_cpu(Mac->u.rx.usStatus),
        __le16_to_cpu(Mac->usFrameControl),
        skb->data[0],skb->data[1],skb->data[2],skb->data[3],skb->data[4],skb->data[5],
        skb->data[6],skb->data[7],skb->data[7],skb->data[9],skb->data[10],skb->data[11],
        *(__u16 *)(skb->data+12)
    );

    /*
     * Check for bridged packets. Add the 'bridge to' address if neccessary.
     */
    if (
        (Fc & FCTL_TO_DS)
        &&
        (Fc & FCTL_FROM_DS)
       )
    {
        PMuHashTableElemType Elem = MuHashFind(lp->HashTab,Mac->ucAddress4,1);
        if (!Elem)
        {
            MuHashAdd(lp->HashTab,Mac->ucAddress4,Mu->MuAddress,0,0,0,0);
        }
    }
    if (skbLen > (MAX_ETHERNET_BODY_SIZE+ETHERNET_HEADER_SIZE))
    {
#if FIXME
        printk(KERN_INFO "%s: pkt too long %d, Rx status %04x, data len %u\n",
            __FUNC__,
            skbLen,
            __le16_to_cpu(Mac->u.rx.usStatus),
            (__u32)__cpu_to_le16(Mac->usDataLen)
        );
        printk(KERN_INFO "%02x:%02x:%02x:%02x:%02x:%02x-%02x:%02x:%02x:%02x:%02x:%02x %04x\n",
            skb->data[0],skb->data[1],skb->data[2],skb->data[3],skb->data[4],skb->data[5],
            skb->data[6],skb->data[7],skb->data[8],skb->data[9],skb->data[10],skb->data[11],
            *(__u16 *)&skb->data[12]
        );
#endif
        skbLen = (MAX_ETHERNET_BODY_SIZE+ETHERNET_HEADER_SIZE);
    }

    skb_put(skb,skbLen);
    return(skb);
}

struct sk_buff *
S24tRxAp(
    struct S24T_private *lp
    )
{
    DBG_FUNC("S24tRxAp")
    struct sk_buff *Skb=NULL;
    int Status;
    TRILOGY_MAC_HDR Mac;
    PTRILOGY_HEADER Rcv;
    PSCORCH_RW      Rw;
    __u32           Len;
    __u32           FrameLen;

    DBG_ENTER(DbgInfo);

    lp->rxFid = S24tRead16(lp,NIC_RX_FID);

    /*
     * Get a copy of the Trilogy receive header.
     */
    Status =
    S24tCopyUp(
        lp,
        lp->rxFid,
        (__u8 *)&Mac,
        sizeof(TRILOGY_MAC_HDR),
        0
    );
    if (Status)
    {
        DBG_PRINT("S24tRxAp - header read error.\n");
        goto S24tRxApExit;
    }

    /*
     * All error fields in the status word should be 0.
     */
    if (__le16_to_cpu(Mac.u.rx.usStatus) & (RX_STATUS_UNDECRYPT_ERR|RX_STATUS_FCS_ERR))
    {
        DBG_PRINT("S24tRx - receive status error %04x.\n",Mac.u.rx.usStatus);
        goto S24tRxApExit;
    }

    FrameLen = (__u32)__le16_to_cpu(Mac.usDataLen);
    if (FrameLen > MAX_802_11_PKT_LEN)
    {
        DBG_PRINT("S24tRx - receive length too long %04x.\n",FrameLen);
        goto S24tRxApExit;
    }

    if (__le16_to_cpu(Mac.usFrameControl) & FCTL_DATA)
    {
        Skb = S24tRxApData(lp,&Mac);
        goto S24tRxApExit;
    }

    Len = sizeof(SCORCH_DEVICE_HEADER) + FrameLen + sizeof(TRILOGY_HEADER);
    Skb = dev_alloc_skb(Len);
    if (!Skb)
    {
        DBG_PRINT("Could not allocate RX Packet len %u\n",Len);
        goto S24tRxApExit;
    }
    skb_put(Skb,Len);

    /*
     * Note that this is a management packet.
     */
    S24tOobInit(Skb);
    S24tApMgmtPkt(Skb,1);

    Rw = (PSCORCH_RW)Skb->data;
    Rcv = (PTRILOGY_HEADER)Rw->Data.Ap;

    /*
     * Zero out unused fields.
     */
    memset(&Rw->Hdr,0,sizeof(SCORCH_DEVICE_HEADER));
    memset(&Rcv->u.Rcv,0,sizeof(TRILOGY_RCV_HEADER));

    /*
     * Note that this is an AP receive.
     */
    Rw->Hdr.Ap = AP_RECV;
    Rw->Hdr.Len = FrameLen + sizeof(TRILOGY_HEADER);

    /*
     * Copy the MAC header.
     */
    Rcv->Hdr = Mac;

    /*
     * Copy up the frame body.
     */
    S24tCopyUpSecond(lp,((__u8 *)Rcv)+sizeof(TRILOGY_HEADER),FrameLen);

S24tRxApExit:
    DBG_LEAVE(DbgInfo);
    return(Skb);
}
#endif

struct sk_buff *
S24tRx(
    struct S24T_private *lp
    )
{
    DBG_FUNC("S24tRx")

    int Status;
    TRILOGY_MAC_HDR TrilogyHeader;
    struct sk_buff *Skb=NULL;
    __u32           SkbLen=0;
#if DBG
    struct iovec    Vec[MAX_IOVEC];
    __u32           Nvec=0;
    __u8            buff[H802_11_SNAP_LEN];
#endif
    DBG_ENTER(DbgInfo);

#if S24_PROC_SYS
    if (lp->ApMode)
    {
        Skb = S24tRxAp(lp);
        DBG_LEAVE(DbgInfo);
        return(Skb);
    }
#endif

    lp->rxFid = S24tRead16(lp,NIC_RX_FID);

    Status =
    S24tCopyUp(
        lp,
        lp->rxFid,
        (__u8 *)&TrilogyHeader,
        sizeof(TRILOGY_MAC_HDR)-ETHERNET_HEADER_SIZE,
        0
    );
    if (Status)
    {
        DBG_PRINT("S24tRx - header read error.\n");
        goto S24tRxExit;
    }

    /*
     * Make sure the packet lengths are sane.
     */
    if (TrilogyHeader.usDataLen > (MAX_ETHERNET_BODY_SIZE+H802_11_SNAP_LEN))
    {
        DBG_PRINT("Rx %u too large\n",(uint)TrilogyHeader.usDataLen);
        goto S24tRxExit;
    }
    if (!TrilogyHeader.usDataLen)
    {
        //DBG_PRINT("Rx runt, len %u\n",(uint)TrilogyHeader.usDataLen);
        goto S24tRxExit;
    }
    if (
        (TrilogyHeader.u.rx.usStatus & RX_TUNNELED)
        &&
        (TrilogyHeader.usDataLen <= H802_11_SNAP_LEN)
       )
    {
        //DBG_PRINT("Rx tunneled runt, len %u\n",(uint)TrilogyHeader.usDataLen);
        goto S24tRxExit;
    }

    SkbLen = TrilogyHeader.usDataLen + ETHERNET_HEADER_SIZE;

    if (!(Skb = dev_alloc_skb(SkbLen+2)))
    {
        DBG_PRINT("Could not allocate RX SKB len %u\n",(uint)SkbLen+2);
        goto S24tRxExit;
    }
    skb_reserve(Skb, 2);

#if DBG
    Vec[0].iov_base = &TrilogyHeader;
    Vec[0].iov_len = sizeof(TRILOGY_MAC_HDR) - ETHERNET_HEADER_SIZE;
#endif

    /*
     * The receive packet has a tunnel SNAP header.
     */
    if (TrilogyHeader.u.rx.usStatus & RX_TUNNELED)
    {
        S24tCopyUpSecond(lp,Skb->data,2*ETH_ALEN);
#if DBG
        S24tCopyUpSecond(lp,buff,H802_11_SNAP_LEN);
#else
        S24tCopyUpSecond(lp,Skb->data+(2*ETH_ALEN),H802_11_SNAP_LEN);
#endif
        S24tCopyUpSecond(lp,Skb->data+(2*ETH_ALEN),SkbLen - ((2*ETH_ALEN)+H802_11_SNAP_LEN) );

        SkbLen -= H802_11_SNAP_LEN;
#if DBG
        Vec[1].iov_base = Skb->data;
        Vec[1].iov_len = 2*ETH_ALEN;
        Vec[2].iov_base = buff;
        Vec[2].iov_len = 8;
        Vec[3].iov_base = Skb->data+(2*ETH_ALEN);
        Vec[3].iov_len = SkbLen - (2*ETH_ALEN);
        Nvec = 4;
#endif
    }
    else
    {
        S24tCopyUpSecond(lp,Skb->data,SkbLen);
#if DBG
        Vec[1].iov_base = Skb->data;
        Vec[1].iov_len = SkbLen;
        Nvec = 2;
#endif
    }

    if (lp->Param[PromiscuousMode])
    {
        PENCAPSULATED_WLAP_BPDU_TYPE Wlap;
#if DBG
        S24tPrintPacketWithTrilogyHeader(Vec,Nvec,0);

        Vec[0].iov_base = Skb->data;
        Vec[0].iov_len = SkbLen;
        S24tPrintPacket(Vec,1,"Rx: ");
#endif
        /*
         * Check for a WLAP config BPDU.
         */
        if (SkbLen >= SIZEOF_ENCAPSULATED_WLAP_BPDU_TYPE)
        {
            Wlap = (PENCAPSULATED_WLAP_BPDU_TYPE)Skb->data;
            if (Wlap->Dix == __cpu_to_be16(SNAP_WLAP_BPDU))
            {
                S24tInitBpdu(lp,Wlap);
                DEV_KFREE_SKB(Skb);
                Skb = NULL;
            }
        }
    }

S24tRxExit:

    if (Skb)
    {
        skb_put(Skb,SkbLen);
    }

    DBG_LEAVE(DbgInfo);
    return(Skb);
}

STATIC int
S24tGetBinaryRec(
    struct S24T_private *lp,
    __u32 Rid,
    __u8 *Buff,
    __u32 *Len
    )
{
    DBG_FUNC("S24tGetBinaryRec")

    int     Status;
    __u32   rLen;
    int flags;

    DBG_ENTER(DbgInfo);

    local_irq_save(flags);

    if (*Len & 1)
    {
        DBG_PRINT("%s odd len %u\n",__FUNC__,(uint)*Len);
        Status = (-EIO);
        goto S24tGetBinaryRecExit;
    }

    Status =
    S24tCopyUp(
        lp,
        Rid,
        (__u8 *)&lp->InfoRec.Rec,
        sizeof(S24tRidType),
        CR_ACCESS
    );
    if (Status)
    {
        DBG_PRINT("%s failed to get record header\n",__FUNC__);
        goto S24tGetBinaryRecExit;
    }

    if (lp->InfoRec.Rec.Rid != Rid)
    {
        DBG_PRINT("%s Bogus Rid %04x, expected %04x\n",__FUNC__,lp->InfoRec.Rec.Rid,Rid);
        Status = (-EIO);
        goto S24tGetBinaryRecExit;
    }

    if (!lp->InfoRec.Rec.WordLen)
    {
        DBG_PRINT("%s Bogus info record length %u, type %04x, expected %u\n",
            __FUNC__,
            (uint)((lp->InfoRec.Rec.WordLen-1) * 2),
            (uint)lp->InfoRec.Rec.Rid,
            (uint)*Len
        );
        Status = (-ENXIO);
        goto S24tGetBinaryRecExit;
    }

    rLen = (lp->InfoRec.Rec.WordLen-1) * 2;
    if (rLen > (*Len))
    {
        DBG_PRINT("%s clipping rid %04x rLen %u to %u\n",__FUNC__,Rid,(uint)rLen,(uint)*Len);
        rLen = (*Len);
    }

    if (((__u32)Buff) & 1)
    {
        DBG_PRINT("%s Warning - unaligned buffer on Rid %04x.\n",__FUNC__,Rid);
        S24tCopyUpSecond(
            lp,
            lp->InfoRec.Data,
            rLen
        );
        memcpy(Buff,lp->InfoRec.Data,rLen);
    }
    else
    {
        S24tCopyUpSecond(
            lp,
            Buff,
            rLen
        );
    }

    *Len = rLen;

#if DBG
    if (DBG_FLAGS(DbgInfo) & DBG_TRACE_ON)
    {
        __u32 i;
        __u8 *p;
        DBG_PRINT("Rid %04x Wlen %04x ",Rid,(*Len));
        for (i=0,p=Buff; i<(*Len); i++,p++)
        {
            printk("%02x ",*p);
        }
        printk("\n");
    }
#endif

S24tGetBinaryRecExit:

    local_irq_restore(flags);
    
    DBG_LEAVE(DbgInfo);
    return(Status);
}

STATIC int
S24tSetBinaryRec(
    struct S24T_private *lp,
    __u32               Rid,
    __u8                *Buff,
    __u32               Len
    )
{
    DBG_FUNC("S24tSetBinaryRec")

    int Status;
    S24tRidType Rec;

    DBG_ENTER(DbgInfo);

    memset(&Rec,0,sizeof(S24tRidType));

    Rec.WordLen = 1 + ((Len+1) / 2);
    Rec.Rid = Rid;

#if DBG
    if (DBG_FLAGS(DbgInfo) & DBG_TRACE_ON)
    {
        __u32 i;
        __u8 *p;
        DBG_PRINT("Rid %04x Wlen %04x ",Rec.Rid,Rec.WordLen);
        for (i=0,p=Buff; i<Len; i++,p++)
        {
            printk("%02x ",*p);
        }
        printk("\n");
    }
#endif

    Status =
    S24tCopyDown(
        lp,
        Rid,
        NULL,
        NULL,
        (__u8 *)&Rec,
        sizeof(S24tRidType),
        Buff,
        Len,
        NULL,
        0,
        CR_ACCESS_WRITE
    );

    DBG_LEAVE(DbgInfo);
    return(Status);
}

int
S24tReadCw10Reg(
    struct S24T_private *lp,
    __u8                RegNum,
    __u8                *RegVal
    )
{
    DBG_FUNC("S24tReadCw10Reg")
    int Status;
    __u16   P0 = (__u16)RegNum;
    int     flags;

    if (RegNum >= MAX_CW10_REGISTER)
    {
        DBG_PRINT("%s Bogus CW10 register index %d\n",__FUNC__,RegNum);
        return(-EINVAL);
    }

    local_irq_save(flags);

    /*
     * The register index is  in the upper byte.
     */
    P0 <<= 8;

    Status =
    S24tCommand(
        lp,
        CR_RADIO_REG,
        NULL,
        &P0,
        NULL,
        NULL
    );

    local_irq_restore(flags);

    if (!Status)
    {
        *RegVal = (__u8)P0;
    }
    return(Status);
}

/*
 * Array of CW10 regisiter read/write capabilities.
 */
__u8 Cw10Rw[MAX_CW10_REGISTER] = 
{
    0, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0
};

int
S24tWriteCw10Reg(
    struct S24T_private *lp,
    __u8                RegNum,
    __u8                RegVal
    )
{
    DBG_FUNC("S24tWriteCw10Reg")
    int     Status;
    __u16   P0;
    int     flags;

    if (RegNum >= MAX_CW10_REGISTER)
    {
        DBG_PRINT("%s Bogus CW10 register index %d\n",__FUNC__,RegNum);
        return(-EINVAL);
    }

    /*
     * Silently return if the register is not writable.
     */
    if (!Cw10Rw[RegNum])
    {
        return(0);
    }

    local_irq_save(flags);

    /*
     * The register index is  in the upper byte. If the MSB of the index is set, then
     * its a write.
     */
    P0 = (__u16)RegNum;
    P0 |= 0x80;
    P0 <<= 8;
    P0 |= (__u16)RegVal;

    Status =
    S24tCommand(
        lp,
        CR_RADIO_REG,
        NULL,
        &P0,
        NULL,
        NULL
    );

    local_irq_restore(flags);

    DBG_PRINT("%s reg %d val %02x\n",__FUNC__,RegNum,RegVal);

    return(Status);
}

int
S24tSetMulticast(
    struct S24T_private *lp,
    __u32               McBytes,
    __u8                *Mc
    )
{
    DBG_FUNC("S24tSetMulticast")

    int Status;
    int     flags;


    DBG_ENTER(DbgInfo);

    local_irq_save(flags);

    Status = S24tSetBinaryRec(lp,RID_GROUP_ADDR,Mc,McBytes);

    local_irq_restore(flags);

    DBG_LEAVE(DbgInfo);
    return(Status);
}

STATIC int
S24tSetMacAddress(
    struct S24T_private *lp
    )
{
    DBG_FUNC("S24tSetMacAddress")

    int Status;

    DBG_ENTER(DbgInfo);

    Status = S24tSetBinaryRec(lp,RID_CFG_MAC_ADDRESS,lp->MACAddress,ETH_ALEN);

    DBG_LEAVE(DbgInfo);
    return(Status);
}

int
S24tPromiscuousMode(
    struct S24T_private *lp,
    __u16               Enabled
    )
{
    DBG_FUNC("S24tPromiscuousMode")

    int Status=0;

    DBG_ENTER(DbgInfo);

    if (!lp->Param[PromiscuousMode] && Enabled)
    {
        S24tInitBpdu(lp,NULL);
    }

    lp->Param[PromiscuousMode] = Enabled;
#if FIXME
    Status = S24tSetBinaryRec(lp,RID_PROMISCUOUS_MODE,(__u8 *)&Enabled,sizeof(Enabled));
#endif
    DBG_LEAVE(DbgInfo);
    return(Status);
}

STATIC void
S24tGetScanResults(
    struct S24T_private *lp,
    S24tRidType         *Rec
    )
{
    DBG_FUNC("S24tGetScanResults")
    __u16 WordLen;

    DBG_ENTER(DbgInfo);

    WordLen = le16_to_cpu(Rec->WordLen);

    DBG_ASSERT(WordLen);
    DBG_ASSERT(!((WordLen-1) % (sizeof(DS_SCAN_RESULT)/2)));

    lp->ScanResults.num_scan_results = 
        ((WordLen-1)*sizeof(__u16)) / sizeof(DS_SCAN_RESULT);

    if (lp->ScanResults.num_scan_results > MAX_SCAN_RESULTS)
    {
        DBG_PRINT("%s Clipping scan results from %d to %d records.\n",
            __FUNC__,
            lp->ScanResults.num_scan_results,
            MAX_SCAN_RESULTS
        );
        lp->ScanResults.num_scan_results = MAX_SCAN_RESULTS;
    }

    if (lp->ScanResults.num_scan_results)
    {
        S24tCopyUpSecond(
            lp,
            (__u8 *)lp->ScanResults.scan_results,
            lp->ScanResults.num_scan_results * sizeof(DS_SCAN_RESULT)
        );
    }

    wake_up_interruptible(&ScanResultsWaitQueue);
    DBG_TRACE(DbgInfo,"Scan Results len %d recs, WordLen %04x.\n",lp->ScanResults.num_scan_results,lp->InfoRec.Rec.WordLen);

    /*
     * Get the group ord stuff as I'm sure it will be needed shortly.
     */
    if (lp->ScanAuto)
    {
        S24tStartGroupOrd(lp);
    }

    DBG_LEAVE(DbgInfo);
}

int
S24tGetInfoRec(
    struct S24T_private *lp
    )
{
    DBG_FUNC("S24tGetInfoRec")

    __u16 Rid;
    __u16 WordLen;
    int Status;

    DBG_ENTER(DbgInfo);

    Rid = S24tRead16(lp,NIC_INFO_FID);

    Status =
    S24tCopyUp(
        lp,
        Rid,
        (__u8 *)&lp->InfoRec.Rec,
        sizeof(S24tRidType),
        0
    );
    if (Status)
    {
        DBG_PRINT("S24tGetInfoRec failed to get record header\n");
        DBG_LEAVE(DbgInfo);
        return(Status);
    }

    WordLen = le16_to_cpu(lp->InfoRec.Rec.WordLen);

    /*
     * Treat host scan results special because they can potentially be
     * quite large.
     */
    if (le16_to_cpu(lp->InfoRec.Rec.Rid) == RID_INFO_HOST_SCAN_RESULTS)
    {
        S24tGetScanResults(lp,&lp->InfoRec.Rec);
        DBG_LEAVE(DbgInfo);
        return(0);
    }

    if (
        (!WordLen)
        ||
        ((WordLen-1) > (MAX_INFO_REC_DATA/2))
       )
    {
        DBG_PRINT("Bogus info record length %u, type %04x\n",
            (uint)WordLen,
            (uint)le16_to_cpu(lp->InfoRec.Rec.Rid)
        );
        DBG_LEAVE(DbgInfo);
        return(-ENXIO);
    }

    S24tCopyUpSecond(
        lp,
        lp->InfoRec.Data,
        (WordLen-1)*2
    );

    S24tParseInfoRec(lp);

    DBG_LEAVE(DbgInfo);
    return(0);
}

STATIC int
S24tSetString(
    struct S24T_private *lp,
    __u16 Rid,
    __u16 RidLen,
    char *Str
    )
{
    DBG_FUNC("S24tSetString")

    int Status;
    PS24tByteStringType S;

    memset(lp->InfoRec.Data,0,MAX_INFO_REC_DATA);
    lp->InfoRec.Rec.WordLen = RidLen;
    lp->InfoRec.Rec.Rid = Rid;

    S = (PS24tByteStringType)lp->InfoRec.Data;

    S->Len = strlen(Str);

    if (S->Len)
    {
        strcpy(S->String,Str);
    }

    Status =
    S24tCopyDown(
        lp,
        lp->InfoRec.Rec.Rid,
        NULL,
        NULL,
        (__u8 *)&lp->InfoRec,
        (lp->InfoRec.Rec.WordLen+1)*2,
        NULL,
        0,
        NULL,
        0,
        CR_ACCESS_WRITE
    );
    if (Status)
    {
        DBG_PRINT("Could not set %s to %s.\n",S24tRidDebug(Rid),Str);
        return(Status);
    }
    else
    {
        DBG_NOTICE(DbgInfo,"Set %s to %s\n",S24tRidDebug(Rid),Str);
    }

    if (!Status)
    {
        memset(lp->InfoRec.Data,0,MAX_INFO_REC_DATA);

        Status =
        S24tCopyUp(
            lp,
            Rid,
            (__u8 *)&lp->InfoRec,
            (RidLen+1)*2,
            CR_ACCESS
        );
        if (Status)
        {
            DBG_PRINT("Could not read %s\n",S24tRidDebug(Rid));
        }
        else
        {
            DBG_NOTICE(DbgInfo,"%s %u '%s'\n",S24tRidDebug(Rid),(uint)S->Len,S->String);
        }
    }

    return(Status);
}

int
S24tSetStationName(
    struct S24T_private *lp,
    char                *StationName
    )
{
    DBG_FUNC("S24tSetStationName")
    int Status;
    int flags;
    DBG_ENTER(DbgInfo);
    local_irq_save(flags);
    Status = S24tSetString(lp,RID_CFG_NAME,18,StationName);
    if ((!Status) && (StationName != lp->StationName))
    {
        strcpy(lp->StationName,StationName);
    }
    local_irq_restore(flags);
    DBG_LEAVE(DbgInfo);
    return(Status);
}

int
S24tSetEssId(
    struct S24T_private *lp,
    char                *essid
    )
{
    DBG_FUNC("S24tSetEssId")
    int Status;
    int flags;
    DBG_ENTER(DbgInfo);

    local_irq_save(flags);

    if (essid)
    {
        memset(lp->NetworkName,0,S24T_MAX_NAME_LEN+1);
        strncpy(lp->NetworkName,essid,S24T_MAX_NAME_LEN);
    }
    Status = S24tSetString(lp,RID_CFG_SSID,18,lp->NetworkName);

    local_irq_restore(flags);

    DBG_LEAVE(DbgInfo);
    return(Status);
}

STATIC S24tGenricRidSetupType RidSetup[] = {
    { RID_DIVERSITY,            AntennaDiversity,           "Antenna Diversity" },
    { RID_CFG_PORT_TYPE,        PortType,                   "Port Type" },
    { RID_CFG_CHANNEL,          Channel,                    "Channel" },
    //{ RID_CFG_SYSTEM_SCALE,     APDensity,                "AP Density" },
    { RID_CFG_MAX_DATA_LENGTH,  MaxDataLength,              "MaxDataLength" },
    { RID_RTS_THRESHOLD,        RtsThreshHold,              "RtsThreshHold" },
    { RID_TX_RATE,              TxRateControl,              "TxRateControl" },
    { RID_PROMISCUOUS_MODE,     PromiscuousMode,            "PromiscuousMode" },
    { RID_TICK_TIME,            TickTime,                   "TickTime" },
    { RID_FRAG_THRESHOLD,       FragmentationThreshHold,    "FragmentationThreshHold" },
    { RID_CFG_PM_ENABLE,        PowerSaveMode,              "PowerSaveMode" },
    { RID_CFG_MULTICAST_RCV,    ReceiveAllMCast,            "ReceiveAllMCast" },
    { RID_CFG_MAX_SLEEP_TIME,   MaxSleepDuration,           "MaxSleepDuration" },
    { RID_ACK_TIMEOUT,          AckTimeout,                 "AP Distance" },
    { 0, 0, "" }
};

STATIC int
S24tWriteRidSetupRec(
    struct S24T_private     *lp,
    PS24tGenricRidSetupType p
    )
{
    DBG_FUNC("S24tWriteRidSetupRec")

    int Status=(-1);

    DBG_ENTER(DbgInfo);

    while (p && p->Rid)
    {
        DBG_ASSERT(p->ParamOffset < MaxParam);

        Status = S24tSetBinaryRec(lp,p->Rid,(__u8 *)&lp->Param[p->ParamOffset],sizeof(__u16));

        if (Status == NIC_STATUS_RESULT_CMD_ERR)
        {
            DBG_PRINT("Could not set %s.\n",p->Name);
            DBG_LEAVE(DbgInfo);
            return(Status);
        }
        else
        {
            DBG_NOTICE(DbgInfo,"Set %s to %u\n",p->Name,(uint)*(__u16 *)lp->InfoRec.Data);
        }
        p++;
    }
    DBG_LEAVE(DbgInfo);
    return(Status);
}

STATIC int
S24tGenericInit(
    struct S24T_private *lp
    )
{
    DBG_FUNC("S24tGenericInit")
    int Status=0;

    DBG_ENTER(DbgInfo);
    Status = S24tWriteRidSetupRec(lp,RidSetup);
    DBG_LEAVE(DbgInfo);
    return(Status);
}

int
S24tSetChannel(
    struct S24T_private *lp,
    __u16               chan
    )
{
    int flags;
    int status;

    local_irq_save(flags);

    status = S24tSetBinaryRec(lp,RID_CFG_CHANNEL,(__u8 *)&chan,sizeof(__u16));
    if (!status)
    {
        lp->Param[Channel] = chan;
    }
    local_irq_restore(flags);
    return(status);
}

int
S24tSetReceiveAllMulticasts(
    struct S24T_private *lp,
    __u16               enabled
    )
{
    int flags;
    int status;

    local_irq_save(flags);

    status = S24tSetBinaryRec(lp,RID_CFG_MULTICAST_RCV,(__u8 *)&enabled,sizeof(__u16));
    if (!status)
    {
        lp->Param[ReceiveAllMCast] = enabled;
    }
    local_irq_restore(flags);
    return(status);
}

int
S24tSetAntennaDiversity(
    struct S24T_private *lp,
    __u16               diversity
    )
{
    int flags;
    int status;

    local_irq_save(flags);

    status = S24tSetBinaryRec(lp,RID_DIVERSITY,(__u8 *)&diversity,sizeof(__u16));
    if (!status)
    {
        lp->Param[AntennaDiversity] = diversity;
    }
    local_irq_restore(flags);
    return(status);
}


int
S24tGetChannel(
    struct S24T_private *lp,
    __u16               *chan
    )
{
    int status;
    __u32 Len = sizeof(__u16);

    status = S24tGetBinaryRec(lp,RID_CURRENT_CHANNEL,(__u8 *)chan,&Len);
    return(status);
}

int
S24tSetApDensity(
    struct S24T_private *lp,
    __u32               Density
    )
{
    int flags;
    int status;

    if (Density > 3)
    {
        return(-EFAULT);
    }

    local_irq_save(flags);
#if FIXME
    status = S24tSetBinaryRec(lp,RID_CFG_SYSTEM_SCALE,(__u8 *)&Density,sizeof(__u16));
#else
    status = 0;
#endif
    if (!status)
    {
        lp->Param[APDensity] = (__u16)Density;
    }
    local_irq_restore(flags);
    return(status);
}

/*
 * ACK timeout in usec is roughly 324+(11*miles). The valid range of timeouts are 324-404.
 */
__u32 
S24tMilesToAckTime(
    struct S24T_private *lp,
    __u32               miles
    )
{
    __u32 usec;

    usec = S24T_MIN_ACK_TIMEOUT + (S24T_ACK_TIMEOUT_MILES_FACTOR * miles);
    if (usec < S24T_MIN_ACK_TIMEOUT)
    {
        usec = S24T_MIN_ACK_TIMEOUT;
    }
    else if (usec > S24T_MAX_ACK_TIMEOUT)
    {
        usec = S24T_MAX_ACK_TIMEOUT;
    }
    return(usec);
}

STATIC S24tGenricRidSetupType EncryptionEnabledSetup[] = {
    { RID_CFG_ENCRYPT_INDEX,    EncryptionKeyId,            "EncryptionKeyId" },
    // KeyLength is a read only field designed to inform the driver which keylengths are supported
    // Trying to set it will generate an error
    //   { RID_CFG_KEY_LENGTH,       EncryptionKeyLen,           "EncryptionKeyLen" },
    //
    { RID_CFG_ENCRYPT_FLAG,     EncryptionEnabled,          "EncryptionEnabled" },
    /*
     * Authentication must come after all other encryption setting.
     */
    { RID_CFG_AUTH_TYPE,        EncryptionAuthentication,   "EncryptionAuthentication" },
    { 0, 0, "" }
};

STATIC S24tGenricRidSetupType EncryptionDisabledSetup[] = {
    { RID_CFG_ENCRYPT_FLAG,     EncryptionEnabled,          "EncryptionEnabled" },
    { 0, 0, "" }
};

STATIC int
S24tSetMUEncryptionAlgorithm(
    struct S24T_private *lp
    )
{
    DBG_FUNC("S24tSetMUEncryptionAlgorithm")

    int Status=0;
    __u32 i;

    DBG_ENTER(DbgInfo);

    if (lp->Param[EncryptionEnabled] == ENCRYPT_FLAG_DISABLE)
    {
        Status = S24tWriteRidSetupRec(lp,EncryptionDisabledSetup);
        DBG_LEAVE(DbgInfo);
        return(Status);
    }

    Status = S24tWriteRidSetupRec(lp,EncryptionEnabledSetup);
    if (Status)
    {
        DBG_LEAVE(DbgInfo);
        return(Status);
    }

    for (i=0; i<MAX_NUM_KEY; i++)
    {
        Status = S24tSetBinaryRec(lp,RID_CFG_KEY1+i,lp->EncryptionKey[i],lp->Param[EncryptionKeyLen]);
        if (Status)
        {
            DBG_PRINT("Could not set encryption key %u.\n",i);
        }
        else
        {
            DBG_NOTICE(DbgInfo,"Set encryption key %u to %s\n",i,lp->EncryptionKey[i]);
        }
    }


    DBG_LEAVE(DbgInfo);
    return(Status);
}

static int
S24tGetCurrentBssId(
    struct S24T_private *lp
    )
{
    int Status;
    __u32 Len;

    memset(lp->BssId,0,ETH_ALEN);

    Len = ETH_ALEN;
    Status = S24tGetBinaryRec(lp,RID_CURRENT_BSSID,lp->BssId,&Len);
    return(Status);
}

int
S24tGeneralInfo(
    struct S24T_private *lp
    )
{
    DBG_FUNC("S24tGeneralInfo")

    int Status;
    __u32 Len;
    PS24tByteStringType S;

    S = (PS24tByteStringType)lp->InfoRec.Data;

    memset(&lp->InfoRec,0,sizeof(S24tInfoRecType));

    Len = sizeof(__u16);
    Status = S24tGetBinaryRec(lp,RID_PORT_STATUS,lp->InfoRec.Data,&Len);
    if (!Status)
    {
        DBG_NOTICE(DbgInfo,"Port Status %u\n", *(__u16 *)lp->InfoRec.Data);
    }
    else
    {
        return(Status);
    }

    memset(&lp->InfoRec,0,sizeof(S24tInfoRecType));
    Len = sizeof(__u16) * 18;
    Status = S24tGetBinaryRec(lp,RID_CURRENT_SSID,lp->InfoRec.Data,&Len);
    if (!Status)
    {
        DBG_NOTICE(DbgInfo,"Current SSID len %u '%s'\n",(uint)S->Len,S->String);
    }
    else
    {
        return(Status);
    }

    Status = S24tGetCurrentBssId(lp);
    if (!Status)
    {
        DBG_NOTICE(DbgInfo,"Current BSSID %02x:%02x:%02x:%02x:%02x:%02x\n",
            lp->BssId[0],
            lp->BssId[1],
            lp->BssId[2],
            lp->BssId[3],
            lp->BssId[4],
            lp->BssId[5]
        );
    }
    else
    {
        return(Status);
    }

    if ((Status = S24tGetComms(lp)))
    {
        return(Status);
    }

    memset(&lp->InfoRec,0,sizeof(S24tInfoRecType));
    Len = sizeof(__u16) * 1;
    Status = S24tGetBinaryRec(lp,RID_CURR_TX_RATE,lp->InfoRec.Data,&Len);
    if (!Status)
    {
        DBG_NOTICE(DbgInfo,"Current Tx Rate %u\n",
            (uint) *(__u16 *)&lp->InfoRec.Data[0]
        );
    }
    else
    {
        return(Status);
    }

    memset(&lp->InfoRec,0,sizeof(S24tInfoRecType));
    Len = sizeof(__u16) * 1;
    Status = S24tGetBinaryRec(lp,RID_PHY_TYPE,lp->InfoRec.Data,&Len);
    if (!Status)
    {
        DBG_NOTICE(DbgInfo,"Phy type %u\n",
            (uint) *(__u16 *)&lp->InfoRec.Data[0]
        );
    }
    else
    {
        return(Status);
    }

    memset(&lp->InfoRec,0,sizeof(S24tInfoRecType));
    Len = sizeof(__u16) * 1;
    Status = S24tGetBinaryRec(lp,RID_CURRENT_CHANNEL,lp->InfoRec.Data,&Len);
    if (!Status)
    {
        DBG_NOTICE(DbgInfo,"Current channel %u\n",
            (uint) *(__u16 *)&lp->InfoRec.Data[0]
        );
    }
    else
    {
        return(Status);
    }

    return(Status);
}

int
S24tGetComms(
    struct S24T_private *lp
    )
{
    DBG_FUNC("S24tGetComms")
    int Status;
    __u32 Len;
    int flags;

    local_irq_save(flags);

    memset(&lp->InfoRec,0,sizeof(S24tInfoRecType));
    Len = sizeof(__u16) * 3;
    Status = S24tGetBinaryRec(lp,RID_COMMS_QUALITY,lp->InfoRec.Data,&Len);
    if (!Status)
    {
        lp->Cq = *(__u16 *)&lp->InfoRec.Data[0];
        lp->Asl = *(__u16 *)&lp->InfoRec.Data[2];
        lp->Anl = *(__u16 *)&lp->InfoRec.Data[4];

        DBG_NOTICE(DbgInfo,"Comms Quality CQ %u ASL %u ANL %u\n",
            (uint) *(__u16 *)&lp->InfoRec.Data[0],
            (uint) *(__u16 *)&lp->InfoRec.Data[2],
            (uint) *(__u16 *)&lp->InfoRec.Data[4]
        );
    }
    local_irq_restore(flags);
    return(Status);
}

STATIC int S24tGetFirmwareVersion(
    struct S24T_private *lp
    )
{
    int Status;
    __u32 Len;

    Len = S24_FW_VERSION_LEN;
    Status = S24tGetBinaryRec(lp,RID_SECONDARY_VERSION,lp->FwVersion,&Len);
    if (!Status)
    {
        lp->FwVersion[S24_FW_VERSION_LEN] = '\0';
    }
    else
    {
        DBG_PRINT("GetFirmwareVersion - failed to get firmware version.\n");
        return(Status);
    }

    Len = S24_FW_VERSION_LEN;
    Status = S24tGetBinaryRec(lp,RID_SECONDARY_DATE,lp->FwDate,&Len);
    if (!Status)
    {
        lp->FwDate[S24_FW_VERSION_LEN] = '\0';
    }
    else
    {
        DBG_PRINT("GetFirmwareVersion - failed to get firmware date.\n");
        return(Status);
    }

    return(Status);
}

int S24tReadRid(
    struct S24T_private *lp,
    __u16               rid,
    __u8                *buff,
    __u32               data_len
    )
{
    return(S24tGetBinaryRec(lp,rid,buff,&data_len));
}

int S24tWriteRid(
    struct S24T_private *lp,
    __u16               rid,
    __u8                *buff,
    __u32               data_len
    )
{
    return(S24tSetBinaryRec(lp,(__u32)HTOCW10S(rid),buff,data_len));
}

STATIC void S24tSetAuxPortAddr(
    struct S24T_private *lp,
    __u32               addr
    )
{
    S24tWrite16(lp,NIC_AUX_BASE,(__u16)(addr >> 7));
    S24tWrite16(lp,NIC_AUX_OFFSET,(__u16)(addr & 0x7F));
}

//	AdapAuxPortGet
//
//		Retrieves the AUX port data.
//
//	Arguments:
//
//		pAdapter		- pointer to the adapter block
//		pDestination	- pointer to destination buffer
//		ulAddress		- address on controller to start reading from
//		usSize			- size in WORDs to read

int S24tReadAuxPort(
    struct S24T_private *lp,
    __u8                *dest,
    __u32               addr,
    __u32               len
    )
{
    DBG_FUNC("S24tReadAuxPort")
	__u16 usTemp;

    if (!S24tCardInserted(lp))
    {
        DBG_LEAVE(DbgInfo);
        return(-ENXIO);
    }

    DBG_IO(DbgInfo,"addr %08x len %u\n",addr,len);

	if ((addr & 1) && len)
	{
        S24tSetAuxPortAddr(lp,addr-1);

        usTemp = S24tRead16(lp,NIC_AUX_DATA);
        *dest = (__u8)(usTemp >> 8);

        addr++;
        dest++;
        len--;
	}

    if (len > 1)
    {
        if (((__u32)dest) & 1)
        {
            DBG_PRINT("%s unaligned buffer %08x\n",__FUNC__,(__u32)dest);
            DBG_LEAVE(DbgInfo);
            return(-ENXIO);
        }

        S24tSetAuxPortAddr(lp,addr);
        S24tReads(lp,dest,NIC_AUX_DATA,len & (~1));
        addr += (len & (~1));
        dest += (len & (~1));
    }

    if (len & 1)
    {
        S24tSetAuxPortAddr(lp,addr);
        usTemp = S24tRead16(lp,NIC_AUX_DATA);

        *dest = (__u8)usTemp;
    }

    return(0);
}

int S24tWriteAuxPort(
    struct S24T_private *lp,
    __u8                *src,
    __u32               addr,
    __u32               len
    )
{
    DBG_FUNC("S24tWriteAuxPort")

	__u16 usTemp;

    if (!S24tCardInserted(lp))
    {
        DBG_LEAVE(DbgInfo);
        return(-ENXIO);
    }

    DBG_IO(DbgInfo,"src %02x %02x ... addr %08x len %u\n",src[0],src[1],addr,len);

    /*
     * Evenize the address.
     */
	if ((addr & 1) && len)
	{
        S24tSetAuxPortAddr(lp,addr-1);

        usTemp = S24tRead16(lp,NIC_AUX_DATA);
        usTemp &= 0x00ff;
        usTemp |= (((__u16)*src) << 8);

        S24tSetAuxPortAddr(lp,addr-1);
        S24tWrite16(lp,NIC_AUX_DATA,usTemp);

        addr++;
        src++;
        len--;
	}
	
    if (len > 1)
    {
        S24tSetAuxPortAddr(lp,addr);
        S24tWrites(lp,src,NIC_AUX_DATA,len & (~1));
        addr += (len & (~1));
        src += (len & (~1));
    }

    if (len & 1)
    {
        S24tSetAuxPortAddr(lp,addr);
        usTemp = S24tRead16(lp,NIC_AUX_DATA);
        usTemp &= 0xff00;
        usTemp |= ((__u16)*src);

        S24tSetAuxPortAddr(lp,addr);
        S24tWrite16(lp,NIC_AUX_DATA,usTemp);
    }

    return(0);
}

STATIC int
S24tWaitForCommandNotBusy(
    struct S24T_private *lp,
    __u16               *CmdStatus
    )
{
    __u32 MsMaxDelay;

    MsMaxDelay = 50000;

#if FIXME
    // Make sure the adapter has settled before reading.
    // LKA -- 02/04/01
    udelay(10);
#endif

    while (
           MsMaxDelay
           &&
           (((*CmdStatus) = S24tRead16(lp,NIC_COMMAND)) & COMMAND_REGISTER_BUSY)
          )
    {
        udelay(100);
        if ((--MsMaxDelay) == 0)
        {
            DBG_PRINT("CmdStatus timeout, last value %04x.\n",(*CmdStatus));
            DBG_PRINT("EvStat looks like %04x.\n", S24tRead16(lp,NIC_EVSTAT));
            return(-ENXIO);
        }
    }

    return(0);
}

int
S24tCommand(
    struct S24T_private *lp,
    __u16               Command,
    __u16               *Result,
    __u16               *P0,
    __u16               *P1,
    __u16               *P2
    )
{
    DBG_FUNC("S24tCommand")
    int Status;
    __u16 CmdStatus;
    __u16 EvStatus;
    __u32 MsMaxDelay;

    DBG_ENTER(DbgInfo);

    if (!S24tCardInserted(lp))
    {
        DBG_LEAVE(DbgInfo);
        return(-ENXIO);
    }

    if ((Status=S24tWaitForCommandNotBusy(lp,&CmdStatus)))
    {
        DBG_LEAVE(DbgInfo);
        return(Status);
    }

    if (P0) S24tWrite16(lp,NIC_PARAM0,*P0);
    if (P1) S24tWrite16(lp,NIC_PARAM1,*P1);
    if (P2) S24tWrite16(lp,NIC_PARAM2,*P2);


    // The windows driver checks the command register here as well, and I figure that it can't hurt
    // LKA -- 02/04/01
    if ((Status = S24tWaitForCommandNotBusy(lp, &CmdStatus)))
    {
        DBG_LEAVE(DbgInfo);
        return(Status);
    }

    S24tWrite16(lp,NIC_COMMAND,Command);
    udelay(10);

    MsMaxDelay = 100000;
    while (!(Status = S24tEvStat(lp,&EvStatus)))
    {
        if (EvStatus & EVS_CMD)
        {
            break;
        }

        udelay(10);
        if ((--MsMaxDelay) == 0)
        {
            if (Command == CR_INITIALIZE)
            {
                break;
            }
            DBG_PRINT("EvStatus command timeout, last value %04x.\n",EvStatus);
            DBG_ASSERT(0);
            DBG_LEAVE(DbgInfo);
            return(-ENXIO);
        }
    }

    if (Status)
    {
        DBG_PRINT("EvStat read error.\n");
        DBG_LEAVE(DbgInfo);
        return(Status);
    }

    if (P0) *P0 = S24tRead16(lp,NIC_RESP0);
    if (P1) *P1 = S24tRead16(lp,NIC_RESP1);
    if (P2) *P2 = S24tRead16(lp,NIC_RESP2);

    CmdStatus = S24tRead16(lp,NIC_STATUS);

    if ((CmdStatus & NIC_STATUS_CMD_MASK) != (Command & NIC_STATUS_CMD_MASK))
    {
        DBG_PRINT("S24tCommand Status %04x != Command %04x, Stat %04x P0 %04x P1 %04x P2 %04x\n",
            (CmdStatus & NIC_STATUS_CMD_MASK),(Command & NIC_STATUS_CMD_MASK),
            CmdStatus,
            P0 ? *P0 : 0,
            P1 ? *P1 : 0,
            P2 ? *P2 : 0
        );
    }

    CmdStatus &= NIC_STATUS_RESULT_MASK;

    if (CmdStatus == NIC_STATUS_RESULT_CMD_ERR)
    {
        DBG_PRINT("CmdStatus %04x Command %04x P0 %04x P1 %04x P2 %04x\n",CmdStatus,Command,(P0 ? *P0 : 0),(P1 ? *P1 : 0),(P2 ? *P2 : 0));
        return(-EFAULT);
    }

    if (Result) *Result = CmdStatus;

    S24tWrite16(lp,NIC_EVACK,ACK_CMD);

    DBG_LEAVE(DbgInfo);
    return(0);
}

__u16
S24tRead16(
    struct S24T_private *lp,
    __u16               offset
    )
{
    if (!S24tCardInserted(lp))
    {
        return(S24T_EJECTED);
    }

    return(S24tInw(lp,offset));
}

__u8
S24tRead8(
    struct S24T_private *lp,
    __u16               offset
    )
{
    if (!S24tCardInserted(lp))
    {
        return((__u8)S24T_EJECTED);
    }

    return(S24tInb(lp,offset));
}

STATIC void
S24tReads(
    struct S24T_private *lp,
    __u8                *dest,
    __u16               offset,
    __u32               len
    )
{
    __u32 wlen = len/2;
    __u32 blen = len & 1;

    if (!S24tCardInserted(lp))
    {
        memset(dest,(char)S24T_EJECTED,len);
        return;
    }

    if (wlen)
    {
        S24tInsw(lp,offset,dest,wlen);
        dest += (wlen * 2);
    }

    if (blen)
    {
        S24tInsb(lp,offset,dest,blen);
    }
}

void
S24tWrite16(
    struct S24T_private *lp,
    __u16               offset,
    __u16               val
    )
{
    if (!S24tCardInserted(lp))
    {
        DBG_PRINT("Write attempt while card not inserted.\n");
        return;
    }

    S24tOutw(lp,offset,val);
}

void
S24tWrite8(
    struct S24T_private *lp,
    __u16               offset,
    __u8                val
    )
{
    if (!S24tCardInserted(lp))
    {
        DBG_PRINT("Write attempt while card not inserted.\n");
        return;
    }

    S24tOutb(lp,offset,val);
}

STATIC void
S24tWrites(
    struct S24T_private *lp,
    __u8                *src,
    __u16               offset,
    __u32               len
    )
{
    __u32 wlen = len/2;
    __u32 blen = len & 1;

    if (!S24tCardInserted(lp))
    {
        DBG_PRINT("Write string attempt while card not inserted.\n");
        return;
    }

    if(len == 0) return;

    // do a byte write if the src address is not word aligned 
    if(((ptrdiff_t)src) & 1) 
    {
	S24tOutsb(lp, offset, src, 1);
	len --;
	src ++;
	wlen = len / 2;
	blen = len & 1;
    }

    if(len == 0) return;

    if (wlen)
    {
        S24tOutsw(lp,offset,src,wlen);
        src += (wlen * 2);
    }

    if (blen)
    {
        S24tOutsb(lp,offset,src,blen);
    }
}

void
S24tHalt(
    struct S24T_private *lp
    )
{
    DBG_FUNC("S24tHalt")

    DBG_ENTER(DbgInfo);

    S24tWriteCor(lp,COR_RESET);
    mdelay(1);
    DBG_PRINT("lp->Cor[OFFSET_COR] = %x\n",S24tReadCor(lp));

    DBG_LEAVE(DbgInfo);
}

#if FIXME
STATIC void
S24ClearHalt(
    struct S24T_private *lp
    )
{
    DBG_FUNC("S24ClearHalt")

    DBG_ENTER(DbgInfo);

    S24tWriteCor(lp,0);
    mdelay(1);
    DBG_PRINT("lp->Cor[OFFSET_COR] = %x\n",S24tReadCor(lp));

    DBG_LEAVE(DbgInfo);
}

STATIC void
S24SetIoMode(
    struct S24T_private *lp
    )
{
    DBG_FUNC("S24SetIoMode")
    DBG_ENTER(DbgInfo);

    S24tWriteCor(lp,COR_IO_MODE);
    mdelay(1);
    DBG_PRINT("lp->Cor[OFFSET_COR] = %x\n",S24tReadCor(lp));

    DBG_LEAVE(DbgInfo);
}

STATIC int
S24tResetCOR(
    struct S24T_private *lp
    )
{
    DBG_FUNC("S24tResetCOR")
    int Status;

    DBG_ENTER(DbgInfo);

    S24tHalt(lp);

    S24ClearHalt(lp);
    S24ClearHalt(lp);

    S24SetIoMode(lp);
    S24SetIoMode(lp);

    mdelay(10);

    Status = S24tCardInserted(lp);

    DBG_LEAVE(DbgInfo);
    return(Status ? 0 : (-ENODEV));
}
#endif

STATIC int S24tSetAuxPort(struct S24T_private *lp, __u32 Enable)
{
    DBG_FUNC("S24tSetAuxPort")
    int i;
    __u16 Ctrl;

    DBG_ENTER(DbgInfo);

    if (Enable)
    {
        S24tWrite16(lp,NIC_PARAM0,AUX_PARAM0_KEY);
        S24tWrite16(lp,NIC_PARAM1,AUX_PARAM1_KEY);
        S24tWrite16(lp,NIC_PARAM2,AUX_PARAM2_KEY);
    }

    Ctrl = Enable ? CTL_AUX_EN : CTL_AUX_DISABLE;

    S24tWrite16(lp,NIC_CONTROL,Ctrl);
    mdelay(10);

    for (i=100; i; i--)
    {
        Ctrl = S24tRead16(lp,NIC_CONTROL);

        if ((!Ctrl) || (Ctrl==CTL_AUX_OPEN))
        {
            break;
        }
        mdelay(10);
    }

    if (!i)
    {
        DBG_TRACE(DbgInfo,"Ctrl %04x\n",Ctrl);
        DBG_LEAVE(DbgInfo);
        return(-ENXIO);
    }

    if ((!Ctrl) && (!Enable))
    {
        DBG_LEAVE(DbgInfo);
        return(0);
    }
    else if ((Ctrl==CTL_AUX_OPEN) && Enable)
    {
        DBG_LEAVE(DbgInfo);
        return(0);
    }

    DBG_LEAVE(DbgInfo);
    return(-ENXIO);
}

STATIC int
S24tReadMacAddress(
    struct S24T_private *lp
    )
{
    DBG_FUNC("S24tReadMacAddress")
    int Status;
    __u32 Len;
    __u8 MACAddress[ETH_ALEN];

    DBG_ENTER(DbgInfo);

    Len = ETH_ALEN;
    Status = S24tGetBinaryRec(lp,RID_CFG_MAC_ADDRESS,MACAddress,&Len);
    if (Status || (Len != ETH_ALEN))
    {
        DBG_PRINT("S24tReadMacAddress failed\n");
    }
    DBG_NOTICE(DbgInfo,"Mac Addr %02x:%02x:%02x:%02x:%02x:%02x\n",
        MACAddress[0],
        MACAddress[1],
        MACAddress[2],
        MACAddress[3],
        MACAddress[4],
        MACAddress[5]
    );

    /*
     * Don't over ride the MAC address if it is a command line parameter.
     */
    if (!(lp->MACAddress[0] | lp->MACAddress[1] | lp->MACAddress[2]))
    {
        memcpy(lp->MACAddress, MACAddress, ETH_ALEN);
    }
    DBG_LEAVE(DbgInfo);
    return(0);
}

#if TRILOGY3
int
S24tStartScan(
    struct S24T_private *lp,
    __u32               Enable,
    __u32               OneTime
    )
{
    DBG_FUNC("S24tStartScan")

    int Status = 0;
    if (lp->CorIo)
    {
        int flags;
        __u16   options;

        lp->ScanAuto = 0;

        if (Enable)
        {
            if (!OneTime)
            {
                lp->ScanAuto = 1;
                options = RID_HOST_SCAN_OPTIONS_5_SEC |
                          RID_HOST_SCAN_OPTIONS_PASSIVE |
                          RID_HOST_SCAN_OPTIONS_BCAST;
            }
            else
            {
                options = RID_HOST_SCAN_OPTIONS_ONE_TIME|RID_HOST_SCAN_OPTIONS_BCAST;
            }
        }
        else
        {
            options = 0;
        }

        local_irq_save(flags);
        Status = S24tSetBinaryRec(lp,RID_BDCST_SCAN,(__u8 *)&options,sizeof(__u16));
        if (Status)
        {
            DBG_PRINT("%s: scan could not be started.\n",__FUNC__);
        }
        local_irq_restore(flags);
    }
    return(Status);
}

STATIC int
S24tStartGroupOrd(
    struct S24T_private *lp
    )
{
    DBG_FUNC("S24tStartGroupOrd")

    int Status = 0;
    if (lp->CorIo)
    {
        int flags;
        __u16 Rid;
        local_irq_save(flags);

        Rid = RID_INFO_GROUP_ORD;
        Status = S24tCommand(lp,CR_INQUIRE,NULL,&Rid,NULL,NULL);
        if (!Status)
        {
            lp->GroupOrdValid = 0;
        }
        else
        {
            DBG_PRINT("%s: group ord could not be started.\n",__FUNC__);
        }
        local_irq_restore(flags);
    }
    return(Status);
}
#endif

int
S24tReadScanResults(
    struct S24T_private *lp,
    DS_SCAN_RESULTS     *results
    )
{
#if TRILOGY3
    DBG_FUNC("S24tReadScanResults")

    int     flags;
    int     status;
    __u32   i;

    if (!lp->CorIo)
    {
        return(-EIO);
    }

    local_irq_save(flags);

    if (!lp->ScanResults.num_scan_results)
    {
        if ((!lp->ScanAuto) && (!(status = S24tStartScan(lp,1,1))))
        {
            interruptible_sleep_on_timeout(&ScanResultsWaitQueue,6*HZ);
            cli();
        }
    }

    if (!lp->ScanResults.num_scan_results)
    {
        if (!lp->ScanAuto)
        {
            DBG_PRINT("%s: scan timed out\n",__FUNC__);
            status = (-EIO);
        }
        else
        {
            status = 0;
            results->num_scan_results = 0;
        }
    }
    else
    {
        status = sizeof(DS_SCAN_RESULTS);

        results->num_scan_results = lp->ScanResults.num_scan_results;
        for (i=0; i<results->num_scan_results; i++)
        {
            results->scan_results[i] = lp->ScanResults.scan_results[i];
            if (results->scan_results[i].service_set_id_len > SCAN_MAX_SSID_LEN)
            {
                DBG_PRINT("%s Clipping service_set_id_len from %d to %d\n",
                    __FUNC__,
                    results->scan_results[i].service_set_id_len,
                    SCAN_MAX_SSID_LEN
                );
                results->scan_results[i].service_set_id_len = SCAN_MAX_SSID_LEN;
            }
            else if (results->scan_results[i].service_set_id_len == SCAN_MAX_SSID_LEN)
            {
                DBG_PRINT("%s Warning - service_set_id_len == %d\n",
                    __FUNC__,
                    SCAN_MAX_SSID_LEN
                );
            }
            /*
             * Cleanup the SSID string. The firmware always seems to terminate it with a period.
             */
            if (results->scan_results[i].service_set_id_len < SCAN_MAX_SSID_LEN)
            {
                memset(results->scan_results[i].service_set_id,0,SCAN_MAX_SSID_LEN);
                memcpy(
                    results->scan_results[i].service_set_id,
                    lp->ScanResults.scan_results[i].service_set_id,
                    results->scan_results[i].service_set_id_len
                );
            }
        }

#if TEST_FULL_SCAN
        for (; i<MAX_SCAN_RESULTS; i++)
        {
            int j;
            results->scan_results[i] = lp->ScanResults.scan_results[0];
            for (j=0; j<6; j++)
            {
                results->scan_results[i].mac_address[j] = (__u8)(i+j);
            }
        }
        results->num_scan_results = MAX_SCAN_RESULTS;
#endif

        /*
         * Invalidate the current scan results once they have been read.
         */
        lp->ScanResults.num_scan_results = 0;
    }

    local_irq_restore(flags);
    return(status);
#else
    return(-EIO);
#endif
}

int
S24tWriteApDistance(
    struct S24T_private *lp,
    __u32               miles
    )
{
    __u32 usec = S24tMilesToAckTime(lp,miles);
    int Status;
    int flags;

    local_irq_save(flags);

    if ((Status=S24tDisable(lp,lp->ApMode)) >= 0)
    {
        if ((Status = S24tSetBinaryRec(lp,RID_ACK_TIMEOUT,(__u8 *)&usec,sizeof(__u16))) >= 0)
        {
            lp->Param[AckTimeout] = (__u16)usec;
            lp->ApDistance = miles;
        }
        S24tEnable(lp,lp->ApMode);
    }

    local_irq_restore(flags);
    return(Status);
}

int
S24tWriteTxRate(
    struct S24T_private *lp,
    __u32               rate
    )
{
    int Status;
    int flags;

    if ((!rate) || (rate > 15))
    {
        return(-EIO);
    }

    local_irq_save(flags);

    if ((Status=S24tDisable(lp,lp->ApMode)) >= 0)
    {
        lp->Param[TxRateControl] = (__u16)rate;
        Status = S24tSetBinaryRec(lp,RID_TX_RATE,(__u8 *)&lp->Param[TxRateControl],sizeof(__u16));
        S24tEnable(lp,lp->ApMode);
    }

    local_irq_restore(flags);
    return(Status);
}

int
S24tGetCurrTxRate(
    struct S24T_private *lp,
    __u16               *rate
    )
{
    int Status;
    __u32 Len = sizeof(__u16);
    Status = S24tGetBinaryRec(lp,RID_CURR_TX_RATE,(__u8 *)rate,&Len);
    return(Status);
}

int
S24tWriteRts(
    struct S24T_private *lp,
    __u32                rts
    )
{
    int Status;
    int flags;

    local_irq_save(flags);

    if ((Status=S24tDisable(lp,lp->ApMode)) >= 0)
    {
        lp->Param[RtsThreshHold] = (__u16)rts;
        Status = S24tSetBinaryRec(lp,RID_RTS_THRESHOLD,(__u8 *)&lp->Param[RtsThreshHold],sizeof(__u16));
        S24tEnable(lp,lp->ApMode);
    }

    local_irq_restore(flags);
    return(Status);
}

int
S24tWriteFragThreshold(
    struct S24T_private *lp,
    __u32                frag
    )
{
    int Status;
    int flags;

    local_irq_save(flags);

    if ((Status=S24tDisable(lp,lp->ApMode)) >= 0)
    {
        lp->Param[FragmentationThreshHold] = (__u16)frag;
        Status = S24tSetBinaryRec(lp,RID_FRAG_THRESHOLD,(__u8 *)&lp->Param[FragmentationThreshHold],sizeof(__u16));
        S24tEnable(lp,lp->ApMode);
    }

    local_irq_restore(flags);
    return(Status);
}

int
S24tReadGroupOrd(
    struct S24T_private *lp,
    GROUP_ORD           *grp
    )
{
#if TRILOGY3
    DBG_FUNC("S24tReadScanResults")

    int     flags;
    int     status;

    if (!lp->CorIo)
    {
        return(-EIO);
    }

    local_irq_save(flags);

    if (!lp->GroupOrdValid)
    {
        if (!lp->ScanAuto)
        {
            if (!(status = S24tStartGroupOrd(lp)))
            {
                interruptible_sleep_on_timeout(&GroupOrdWaitQueue,5*HZ);
                cli();
            }
        }
    }

    if (!lp->GroupOrdValid)
    {
        if (!lp->ScanAuto)
        {
            DBG_PRINT("%s: group ord timed out\n",__FUNC__);
            status = (-EIO);
        }
        else
        {
            /*
             * If channel is 0, then non of the rest are valid.
             */
            grp->channel = 0;
            status = 0;
        }
    }
    else
    {
        lp->GroupOrdValid = 0;
        memcpy(grp,&lp->GroupOrd,sizeof(GROUP_ORD));
        status = sizeof(GROUP_ORD);
    }

    local_irq_restore(flags);
    return(status);
#else
    return(-EIO);
#endif
}

STATIC int
S24tGetOrdinalsAddress(
    struct S24T_private *lp
    )
{
    DBG_FUNC("S24tGetOrdinalsAddress")

    int Status;
    int i;
    __u32 Len;

    if (AdapEnableAux(lp) != TRUE)
    {
        DBG_PRINT("Could not enable Aux port.\n");
        return(-EIO);
    }

    Len = NUM_ORDINAL_TABLES*sizeof(__u16);
    Status = S24tGetBinaryRec(lp,RID_ORDINALS,(__u8 *)lp->OrdinalTablesStartAddr,&Len);
    if (Status)
    {
        return(Status);
    }
    if (Len != (NUM_ORDINAL_TABLES*sizeof(__u16)))
    {
        DBG_PRINT("%s strange length returned %d\n",__FUNC__,Len);
    }
    for (i=0; i<NUM_ORDINAL_TABLES; i++)
    {
        DBG_PRINT("Ordinals Table %d addr %04x\n",i,lp->OrdinalTablesStartAddr[i]);
        if ((lp->OrdinalTablesStartAddr[i] < 0x1000) | (lp->OrdinalTablesStartAddr[i] > 0x3000))
        {
            DBG_PRINT("%s warning table %d address wierd? %04x\n",__FUNC__,i,lp->OrdinalTablesStartAddr[i]);
        }
        AdapAuxPortGet(lp,&lp->OrdinalTablesLen[i],lp->OrdinalTablesStartAddr[i],sizeof(__u16)/2);
        DBG_PRINT("Ordinals Table %d len %04x\n",i,lp->OrdinalTablesLen[i]);
    }
    return(Status);
}

int
S24tGetTableOneOrdinal(
    struct S24T_private *lp,
    __u32               Ord,
    __u32               *Val
    )
{
    int Status;
    int flags;
    __u32 OrdAddr;

    local_irq_save(flags);

    if (lp->OrdinalTablesStartAddr[0] && lp->OrdinalTablesLen[0])
    {
        /*
         * Get the address of the ordinal from table one.
         */
        Status =
        S24tReadAuxPort(
            lp,
            (__u8 *)&OrdAddr,
            (__u32)(lp->OrdinalTablesStartAddr[0] + (sizeof(__u16)*Ord)),
            sizeof(__u16)
        );

        OrdAddr &= 0x0000ffff;

        if (!Status)
        {
            Status = (-EIO);
            if (OrdAddr != 0xffff)
            {
                /*
                 * Get the value of the ordinal.
                 */
                Status = S24tReadAuxPort(lp,(__u8 *)Val,(OrdAddr & 0x00007fff),sizeof(__u32));
                if (!Status)
                {
                    /*
                     * If it is a 16 bit value, then clear the upper bits.
                     */
                    if (OrdAddr & 0x00008000)
                    {
                        *Val &= 0x0000ffff;
                    }
                }
            }
        }
    }
    else
    {
        Status = (-EIO);
    }

    local_irq_restore(flags);

    return(Status);
}

STATIC void
S24tInitBpdu(
    struct S24T_private         *lp,
    PENCAPSULATED_WLAP_BPDU_TYPE pBpdu
    )
{
    __u16 t;

    if (!pBpdu)
    {
#if DBG
        printk(KERN_INFO "sizeof(ENCAPSULATED_WLAP_BPDU_TYPE) %d SIZEOF_ENCAPSULATED_WLAP_BPDU_TYPE %d\n",
            sizeof(ENCAPSULATED_WLAP_BPDU_TYPE),
            SIZEOF_ENCAPSULATED_WLAP_BPDU_TYPE
        );
#endif

        memset(&lp->Bpdu,0,sizeof(ENCAPSULATED_WLAP_BPDU_TYPE));

        /*
         * These values are changed by the config BPDU from the AP.
         */
        memcpy(lp->Bpdu.Bpdu.RootAddr, lp->MACAddress, ETH_ALEN);
        lp->Bpdu.Bpdu.RootPriority  = __cpu_to_be16(0xffff);
        lp->Bpdu.Bpdu.RootPathCost  = __cpu_to_be16(0);
        lp->Bpdu.Bpdu.MessageAge    = __cpu_to_be16(0);
        lp->Bpdu.Bpdu.MaxAge        = __cpu_to_be16(0x64);
        lp->Bpdu.Bpdu.HelloTime     = __cpu_to_be16(0x14);
        lp->Bpdu.Bpdu.ForwardDelay  = __cpu_to_be16(5);

        /*
         * Send the config BPDU until you get a response. Then send it out at
         * HelloTime rates.
         */
        lp->UseHelloTime = 0;
    }
    else
    {
        lp->Bpdu.Bpdu.RootPriority = pBpdu->Bpdu.RootPriority;

        memcpy(lp->Bpdu.Bpdu.RootAddr, pBpdu->Bpdu.RootAddr, ETH_ALEN);

        t = __be16_to_cpu(pBpdu->Bpdu.RootPathCost);
        lp->Bpdu.Bpdu.RootPathCost = __cpu_to_be16(t+1);

        t = __be16_to_cpu(pBpdu->Bpdu.MessageAge);
        lp->Bpdu.Bpdu.MessageAge = __cpu_to_be16(t+1);

        lp->Bpdu.Bpdu.MaxAge        = pBpdu->Bpdu.MaxAge;
        lp->Bpdu.Bpdu.HelloTime     = pBpdu->Bpdu.HelloTime;
        lp->Bpdu.Bpdu.ForwardDelay  = pBpdu->Bpdu.ForwardDelay;

        lp->UseHelloTime = 1;

#if DBG
        if (DBG_FLAGS(DbgInfo) & DBG_BR_RX_ON)
        {
            printk(KERN_INFO "Updating BPDU RootPriority %04x ",__be16_to_cpu(lp->Bpdu.Bpdu.RootPriority));
            printk("RootAddr %02x:%02x:%02x:%02x:%02x:%02x ",
                lp->Bpdu.Bpdu.RootAddr[0],lp->Bpdu.Bpdu.RootAddr[1],lp->Bpdu.Bpdu.RootAddr[2],lp->Bpdu.Bpdu.RootAddr[3],lp->Bpdu.Bpdu.RootAddr[4],lp->Bpdu.Bpdu.RootAddr[5]
            );
            printk("RootPathCost %04x ",__be16_to_cpu(lp->Bpdu.Bpdu.RootPathCost));
            printk("MessageAge %04x ",__be16_to_cpu(lp->Bpdu.Bpdu.MessageAge));
            printk("MaxAge %04x ",__be16_to_cpu(lp->Bpdu.Bpdu.MaxAge));
            printk("HelloTime %04x ",__be16_to_cpu(lp->Bpdu.Bpdu.HelloTime));
            printk("ForwardDelay %04x\n",__be16_to_cpu(lp->Bpdu.Bpdu.ForwardDelay));
        }
#endif
    }

    /*
     * BssId may change if association changes.
     */
    lp->Bpdu.Dix = __cpu_to_be16(SNAP_WLAP_BPDU);
    memcpy(lp->Bpdu.Dest,   lp->BssId,      ETH_ALEN);
    memcpy(lp->Bpdu.Src,    lp->MACAddress, ETH_ALEN);
    memcpy(lp->Bpdu.Bpdu.WlapAddr, lp->MACAddress, ETH_ALEN);
    lp->Bpdu.Bpdu.InterfaceId = __cpu_to_be16(0x8001);
    lp->Bpdu.Bpdu.WlapPriority = __cpu_to_be16(0xffff);
}

STATIC void
S24tParseInfoRec(
    struct S24T_private *lp
    )
{
    DBG_FUNC("S24tParseInfoRec")

    int status;
    __u16 WordLen = le16_to_cpu(lp->InfoRec.Rec.WordLen);

    DBG_ENTER(DbgInfo);

    DBG_TRACE(DbgInfo,"Info Rec %s wlen %u data %04x\n",
        S24tRidDebug(lp->InfoRec.Rec.Rid),
        WordLen,
        *(__u16 *)lp->InfoRec.Data
    );

    switch (le16_to_cpu(lp->InfoRec.Rec.Rid))
    {
        case RID_INFO_LINK_STATUS:
        {
            switch (*(__u16 *)lp->InfoRec.Data)
            {
                case 1:
                    lp->LinkStatus = 1;
                    printk(KERN_NOTICE "%s: Link connect status connected\n",lp->dev->name);
                    break;
                case 3:
                    lp->LinkStatus = 1;
                    printk(KERN_NOTICE "%s: Link connect status connected with AP change\n",lp->dev->name);
                    break;
                case 4:
                {
                    __u16   tRid = cpu_to_le16(RID_INFO_HOST_SCAN_RESULTS);
                    status =
                    S24tCommand(
                        lp,
                        CR_INQUIRE,
                        NULL,
                        &tRid,
                        NULL,
                        NULL
                    );

                    if (status)
                    {
                        DBG_PRINT("%s: RID_INFO_HOST_SCAN_RESULTS failed.\n",__FUNC__);
                    }
                    break;
                }
                default:
                    lp->LinkStatus = 0;
                    printk(KERN_NOTICE "%s: Link connect status disconnected %d\n",
                        lp->dev->name,
                        (int)*(__u16 *)lp->InfoRec.Data
                    );
                    break;
            }
            break;
        }
        case RID_INFO_ASSOCIATION_STATUS:
        {
            switch (*(__u16 *)lp->InfoRec.Data)
            {
            case 1:
                lp->LinkStatus = 1;
                printk(KERN_NOTICE "%s: Link association status associated to %02x:%02x:%02x:%02x:%02x:%02x\n",
                    lp->dev->name,
                    lp->InfoRec.Data[2],
                    lp->InfoRec.Data[3],
                    lp->InfoRec.Data[4],
                    lp->InfoRec.Data[5],
                    lp->InfoRec.Data[6],
                    lp->InfoRec.Data[7]
                );
                break;
            case 2:
                lp->LinkStatus = 1;
                printk(KERN_NOTICE "%s: Link association status reassociated to %02x:%02x:%02x:%02x:%02x:%02x from %02x:%02x:%02x:%02x:%02x:%02x\n",
                    lp->dev->name,
                    lp->InfoRec.Data[2],
                    lp->InfoRec.Data[3],
                    lp->InfoRec.Data[4],
                    lp->InfoRec.Data[5],
                    lp->InfoRec.Data[6],
                    lp->InfoRec.Data[7],

                    lp->InfoRec.Data[8],
                    lp->InfoRec.Data[9],
                    lp->InfoRec.Data[10],
                    lp->InfoRec.Data[11],
                    lp->InfoRec.Data[12],
                    lp->InfoRec.Data[13]
                );
                break;
            default:
                lp->LinkStatus = 0;
                printk(KERN_NOTICE "%s: Link association status disassociated\n",lp->dev->name);
                break;
            }
            break;
        }

        case RID_INFO_HOST_SCAN_RESULTS:
        {
            DBG_ASSERT(0);
            break;
        }
        case RID_INFO_GROUP_ORD:
        {
            DBG_ASSERT(WordLen <= (sizeof(GROUP_ORD)+sizeof(__u16)));
            memcpy(&lp->GroupOrd,lp->InfoRec.Data,(WordLen-1)*sizeof(__u16));
            lp->GroupOrdValid = 1;
            wake_up_interruptible(&GroupOrdWaitQueue);
            DBG_TRACE(DbgInfo,"Group Ord WordLen %u.\n",WordLen);
            break;
        }
        default:
            DBG_PRINT("%s: Unknown info RID %04x\n",__FUNC__,lp->InfoRec.Rec.Rid);
            break;
    }

    /*
     * Update the connected AP MAC address.
     */
    S24tGetCurrentBssId(lp);
    if (lp->LinkStatus)
    {
#if FIXME
        printk(KERN_INFO "spectrum24t: Connected to AP %02x:%02x:%02x:%02x:%02x:%02x\n",
            lp->BssId[0],
            lp->BssId[1],
            lp->BssId[2],
            lp->BssId[3],
            lp->BssId[4],
            lp->BssId[5]
        );
#endif
    }
    DBG_LEAVE(DbgInfo);
}

#if DBG
typedef struct {
    __u16   Rid;
    char    *Name;
} RidDebugType;

RidDebugType RidDebug[] = {
    {RID_INFO_HANDOVER_ADDR,        "RID_INFO_HANDOVER_ADDR"},
    {RID_INFO_COMM_TALLIES,         "RID_INFO_COMM_TALLIES"},
    {RID_INFO_SCAN_RESULTS,         "RID_INFO_SCAN_RESULTS"},
    {RID_INFO_LINK_STATUS,          "RID_INFO_LINK_STATUS"},
    {RID_INFO_ASSOCIATION_STATUS,   "RID_INFO_ASSOCIATION_STATUS"},
    {RID_INFO_HOST_SCAN_RESULTS,    "RID_INFO_HOST_SCAN_RESULTS"},
    {RID_CFG_PORT_TYPE,             "RID_CFG_PORT_TYPE"},
    {RID_CFG_MAC_ADDRESS,           "RID_CFG_MAC_ADDRESS"},
    {RID_CFG_SSID,                  "RID_CFG_SSID"},
    {RID_CFG_CHANNEL,               "RID_CFG_CHANNEL"},
    {RID_CFG_OWN_SSID,              "RID_CFG_OWN_SSID"},
    {RID_CFG_OWN_ATIM,              "RID_CFG_OWN_ATIM"},
    {RID_CFG_SYSTEM_SCALE,          "RID_CFG_SYSTEM_SCALE"},
    {RID_CFG_MAX_DATA_LENGTH,       "RID_CFG_MAX_DATA_LENGTH"},
    {RID_CFG_WDS_ADDRESS,           "RID_CFG_WDS_ADDRESS"},
    {RID_CFG_PM_ENABLE,             "RID_CFG_PM_ENABLE"},
    {RID_CFG_PM_EPS,                "RID_CFG_PM_EPS"},
    {RID_CFG_MULTICAST_RCV,         "RID_CFG_MULTICAST_RCV"},
    {RID_CFG_MAX_SLEEP_TIME,        "RID_CFG_MAX_SLEEP_TIME"},
    {RID_CFG_PM_HOLDOVER_TIME,      "RID_CFG_PM_HOLDOVER_TIME"},
    {RID_CFG_NAME,                  "RID_CFG_NAME"},
    {RID_CFG_MKK_FLAG,              "RID_CFG_MKK_FLAG"},
    {RID_CFG_PREFERRED_BSS_ID,      "RID_CFG_PREFERRED_BSS_ID"},
    {RID_CFG_MANDATORY_BSS_ID,      "RID_CFG_MANDATORY_BSS_ID"},
    {RID_CFG_AUTHENTICATION,        "RID_CFG_AUTHENTICATION"},
    {RID_CFG_ENCRYPT_INDEX,         "RID_CFG_ENCRYPT_INDEX"},
    {RID_CFG_KEY1,                  "RID_CFG_KEY1"},
    {RID_CFG_KEY2,                  "RID_CFG_KEY2"},
    {RID_CFG_KEY3,                  "RID_CFG_KEY3"},
    {RID_CFG_KEY4,                  "RID_CFG_KEY4"},
    {RID_CFG_ENCRYPT_FLAG,          "RID_CFG_ENCRYPT_FLAG"},
    {RID_CFG_AUTH_TYPE,             "RID_CFG_AUTH_TYPE"},
    {RID_CFG_KEY_LENGTH,            "RID_CFG_KEY_LENGTH"},
    {RID_GROUP_ADDR,                "RID_GROUP_ADDR"},
    {RID_CREATE_IBSS,               "RID_CREATE_IBSS"},
    {RID_FRAG_THRESHOLD,            "RID_FRAG_THRESHOLD"},
    {RID_RTS_THRESHOLD,             "RID_RTS_THRESHOLD"},
    {RID_TX_RATE,                   "RID_TX_RATE"},
    {RID_PROMISCUOUS_MODE,          "RID_PROMISCUOUS_MODE"},
    {RID_DIVERSITY,                 "RID_DIVERSITY"},
    {RID_OPTIONS,                   "RID_OPTIONS"},
    {RID_IF_FREQUENCY,              "RID_IF_FREQUENCY"},
    {RID_BASIC_RATES,               "RID_BASIC_RATES"},
    {RID_TX_INHIBIT,                "RID_TX_INHIBIT"},
    {RID_CLEAR_STATS,               "RID_CLEAR_STATS"},
    {RID_TICK_TIME,                 "RID_TICK_TIME"},
    {RID_MAX_DOWNLOAD_TIME,         "RID_MAX_DOWNLOAD_TIME"},
    {RID_DOWNLOAD_BUFFER,           "RID_DOWNLOAD_BUFFER"},
    {RID_PRI_IDENTITY,              "RID_PRI_IDENTITY"},
    {RID_PRI_SUP_RANGE,             "RID_PRI_SUP_RANGE"},
    {RID_PRIMARY_VERSION,           "RID_PRIMARY_VERSION"},
    {RID_PRIMARY_DATE,              "RID_PRIMARY_DATE"},
    {RID_NIC_SERIAL,                "RID_NIC_SERIAL"},
    {RID_NIC_FW_ID,                 "RID_NIC_FW_ID"},
    {RID_CHANNEL_LIST,              "RID_CHANNEL_LIST"},
    {RID_REGULATORY,                "RID_REGULATORY"},
    {RID_TEMP_TYPE,                 "RID_TEMP_TYPE"},
    {RID_CIS,                       "RID_CIS"},
    {RID_COUNTRY_CODE,              "RID_COUNTRY_CODE"},
    {RID_COUNTRY_NAME,              "RID_COUNTRY_NAME"},
    {RID_MKK_CALL_SIGN,             "RID_MKK_CALL_SIGN"},
    {RID_MKK_RSSI,                  "RID_MKK_RSSI"},
    {RID_ORDINALS,                  "RID_ORDINALS"},
    {RID_STA_IDENTITY,              "RID_STA_IDENTITY"},
    {RID_STA_SUP_RANGE,             "RID_STA_SUP_RANGE"},
    {RID_MFI_ACT_RANGE,             "RID_MFI_ACT_RANGE"},
    {RID_CFI_ACT_RANGE,             "RID_CFI_ACT_RANGE"},
    {RID_SECONDARY_VERSION,         "RID_SECONDARY_VERSION"},
    {RID_SECONDARY_DATE,            "RID_SECONDARY_DATE"},
    {RID_PORT_STATUS,               "RID_PORT_STATUS"},
    {RID_CURRENT_SSID,              "RID_CURRENT_SSID"},
    {RID_CURRENT_BSSID,             "RID_CURRENT_BSSID"},
    {RID_COMMS_QUALITY,             "RID_COMMS_QUALITY"},
    {RID_CURR_TX_RATE,              "RID_CURR_TX_RATE"},
    {RID_SHORT_RETRY,               "RID_SHORT_RETRY"},
    {RID_LONG_RETRY,                "RID_LONG_RETRY"},
    {RID_PRIVACY,                   "RID_PRIVACY"},
    {RID_PHY_TYPE,                  "RID_PHY_TYPE"},
    {RID_CURRENT_CHANNEL,           "RID_CURRENT_CHANNEL"},
    {RID_POWER_MODE,                "RID_POWER_MODE"},
    {RID_CCA_MODE,                  "RID_CCA_MODE"},
    {RID_ACK_TIMEOUT,               "RID_ACK_TIMEOUT"},
    {0,                             "Unknown"}
};

STATIC char *
S24tRidDebug(
    __u16 Rid
    )
{
    RidDebugType *p = RidDebug;
    static char buff[16];
    while (p->Rid)
    {
        if (p->Rid == Rid)
        {
            return(p->Name);
        }
        p++;
    }

    sprintf(buff,"%04x",Rid);
    return(buff);
}
#endif

