/* Start of file */

/* {{{ [fold] Comments  */

/*
 * qc-usb, Logitech Quickcam video driver with V4L support
 * Derived from qce-ga, linux V4L driver for the Quickcam Express and Dexxa Quickcam
 *
 * qc-driver.c - main driver part
 *
 * Copyright (C) 2001  Jean-Fredric Clere, Nikolas Zimmermann, Georg Acher
 * Mark Cave-Ayland, Carlo E Prelz, Dick Streefland
 * Copyright (C) 2002  Tuukka Toivonen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/* Cam variations of Logitech Quickcam:
   P/N 861037:      Sensor HDCS1000        ASIC STV0600
   P/N 861050-0010: Sensor HDCS1000        ASIC STV0600
   P/N 861050-0020: Sensor Photobit PB100  ASIC STV0600-1 ("Quickcam Express")
   P/N 861055:      Sensor ST VV6410       ASIC STV0610 ("LEGO cam")
   P/N 861075-0040: Sensor HDCS1000        ASIC
   P/N 961179-0700: Sensor ST VV6410       ASIC STV0602 (Dexxa WebCam USB)
   P/N 861040-0000: Sensor ST VV6410       ASIC STV0610 ("Quickcam Web")

   For any questions ask 
   	qce-ga-devel@lists.sourceforge.net	- about code
   	qce-ga-discussion@lists.sourceforge.net	- about usage
*/
/* }}} */
/* {{{ [fold] Includes  */
#ifdef NOKERNEL
#include "quickcam.h"
#else
#include <linux/quickcam.h>
#endif
#include <linux/module.h>
#include "qc-memory.h"

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
#include <linux/slab.h>
#else
#include <linux/malloc.h>
#endif
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/list.h>
#include <linux/mm.h>
#include <linux/proc_fs.h>
#include <linux/smp_lock.h>
#include <linux/vmalloc.h>
#include <linux/wrapper.h>
#include <linux/spinlock.h>
#include <asm/io.h>
#include <asm/uaccess.h>
/* }}} */
/* {{{ [fold] Module parameters  */
MODULE_PARM_DESC(debug, "Sets the debug output (bitfield)");
MODULE_PARM(debug, "i");
int debug = DEBUGLEVEL;

MODULE_PARM_DESC(keepsettings, "Keep picture settings across one open to another (0-1)");
MODULE_PARM(keepsettings, "i");
static int keepsettings = 0;

MODULE_PARM_DESC(settle, "Let picture settle at device open (0-1)");
MODULE_PARM(settle, "i");
static int settle = 0;

/* Subsampling is used to allow higher scan rate with smaller images. */
MODULE_PARM_DESC(subsample, "Sets subsampling (0-1)");
MODULE_PARM(subsample, "i");
static int subsample = 0;	/* normal or sub-sample (sub-sample to increase the speed) */

MODULE_PARM_DESC(compress, "Enable compressed mode (0-1)");
MODULE_PARM(compress, "i");
static int compress = 0;	/* Enable compressed mode if available (higher framerate) */

MODULE_PARM_DESC(frameskip, "How frequently capture frames (0-10)");
MODULE_PARM(frameskip, "i");
static int frameskip = 0;

MODULE_PARM_DESC(quality, "Sets the picture quality (0-3)");
MODULE_PARM(quality, "i");
static int quality = 3;	/* 3 = bilinear interpolation */

MODULE_PARM_DESC(adaptive, "Automatic adaptive brightness control (0-1)");
MODULE_PARM(adaptive, "i");
static int adaptive = 1;

MODULE_PARM_DESC(equalize, "Equalize image (0-1)");
MODULE_PARM(equalize, "i");
static int equalize = 0;	/* Disabled by default */

MODULE_PARM_DESC(retryerrors, "Retry if image capture fails, otherwise return error code (0-1)");
MODULE_PARM(retryerrors, "i");
static int retryerrors = 1;	/* Enabled by default */

/* Bug in Xvideo(?): if the width is not divisible by 8 and Xvideo is used, the frame is shown wrongly */
MODULE_PARM_DESC(compatible, "Enable workaround for bugs in application programs (bitfield)");
MODULE_PARM(compatible, "i");
static int compatible = 0;	/* Disabled by default */
#define COMPAT_8X	BIT(0)
#define COMPAT_DBLBUF	BIT(1)
#define COMPAT_TORGB	BIT(2)	/* Video4Linux API is buggy and doesn't specify byte order for RGB images */

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,5)
MODULE_PARM_DESC(video_nr, "Set videodevice number (/dev/videoX)");
MODULE_PARM(video_nr,"i");
/* video_nr option allows to specify a certain /dev/videoX device */
/* (like /dev/video0 or /dev/video1 ...)                          */
/* for autodetect first available use video_nr=-1 (defaultvalue)  */
static int video_nr = -1;
#endif
/* }}} */
/* {{{ [fold] Miscellaneous data  */
#ifndef MODULE_LICENSE		/* Appeared in 2.4.10 */
#ifdef MODULE
#define MODULE_LICENSE(license) \
static const char __module_license[] __attribute__((section(".modinfo"))) = \
	"license=" license
#else
#define MODULE_LICENSE(license)
#endif
#endif

MODULE_SUPPORTED_DEVICE("video");
MODULE_DESCRIPTION("Logitech Quickcam USB driver");
MODULE_AUTHOR("See README");
MODULE_LICENSE("GPL");
EXPORT_NO_SYMBOLS;

static const int min_framewidth  = 32;	/* Minimum image size we allow delivering to user application */
static const int min_frameheight = 32;

static const char qc_name[] = "quickcam";

static __devinitdata struct usb_device_id qc_device_table[] = {
	{ USB_DEVICE(0x046D, 0x0840) },		/* Quickcam Express */
	{ USB_DEVICE(0x046D, 0x0850) },		/* LEGO cam / Quickcam Web */
	{ USB_DEVICE(0x046D, 0x0870) },		/* Dexxa webcam USB - not yet supported */
	{ }
};
MODULE_DEVICE_TABLE(usb, qc_device_table);

extern const struct qc_sensor qc_sensor_pb0100;
extern const struct qc_sensor qc_sensor_hdcs1000;
extern const struct qc_sensor qc_sensor_hdcs1020;
extern const struct qc_sensor qc_sensor_vv6410;

static const struct qc_sensor *sensors[] = {
	&qc_sensor_hdcs1000,
	&qc_sensor_hdcs1020,
	&qc_sensor_pb0100,
	&qc_sensor_vv6410,
};

// FIXME: should this list be protected by a lock/semaphore?
static LIST_HEAD(quickcam_list);		/* Linked list containing all Quickcams */

static void qc_usb_exit(struct quickcam *qc);
static int qc_capt_init(struct quickcam *qc);
static void qc_capt_exit(struct quickcam *qc);
static int qc_isoc_init(struct quickcam *qc);
static void qc_isoc_exit(struct quickcam *qc);
/* }}} */

/* {{{ [fold] **** Miscellaneous functions ************************************** */

/* {{{ [fold] int qc_get_i2c(struct quickcam *qc, const struct qc_sensor *sensor, int reg) */
/* Read a sensor byte or word wide register value via STV0600 I2C bus
 * qc_i2c_init() must be called first!
 */
int qc_get_i2c(struct quickcam *qc, const struct qc_sensor *sensor, int reg)
{
	struct usb_device *dev = qc->dev;
	u8 tmpbuff[35];		/* USB transfer buffer. Must *not* be static! FIXME:is stack ok or should it be kmalloc'ed? */
	u8 buf[2];
	int ret;

	if (debug&DEBUGLOGIC || debug&DEBUGCAMERA) PDEBUG("qc_get_i2c(qc=%p,sensor=%p,reg=0x%04X)",qc,sensor,reg);
	if (PARANOID && dev==NULL) { PDEBUG("dev==NULL!"); return -EINVAL; }
	if (PARANOID && buf==NULL) { PDEBUG("buf==NULL!"); return -EINVAL; }

	/* We need here extra write to the STV register before reading the I2C register */
	/* Also wait until there are no pending control URB requests */
	if ((ret = qc_stv_set(qc, STV_REG23, sensor->reg23))<0) goto fail;

	memset(tmpbuff, 0, sizeof(tmpbuff));
	tmpbuff[0]    = reg;
	tmpbuff[0x20] = sensor->i2c_addr;
	tmpbuff[0x21] = 0;			/* 0+1 = 1 value, one byte or word wide register */
	tmpbuff[0x22] = 3;			/* Read I2C register */
	ret = usb_control_msg(dev, usb_sndctrlpipe(dev, 0),
		0x04,
		0x40,
		0x1400, 0,			/* Write I2C register address */
		tmpbuff, 0x23, HZ);
	if (ret < 0) goto fail;
	ret = usb_control_msg(dev, usb_rcvctrlpipe(dev, 0),
		0x04,
		0xC0,
		0x1410, 0, 			/* Read register contents from I2C */
		buf, sensor->length_id, HZ);
	if (ret < 0) goto fail;
	ret = buf[0];
	if (sensor->length_id>1) ret |= buf[1]<<8;	/* Assume LSB is always first from data received via USB */
	if (debug&DEBUGCAMERA) PDEBUG("qc_get_i2c(reg=0x%04X) = %04X", reg, ret);
	return ret;

fail:	PDEBUG("qc_get_i2c failed, code=%i",ret);
	return ret;
}
/* }}} */
/* {{{ [fold] int qc_stv_set(struct quickcam *qc, unsigned short reg, unsigned char val) */
/*
 * Set one byte register in the STV-chip. qc_i2c_init() must be called first!
 */
int qc_stv_set(struct quickcam *qc, unsigned short reg, unsigned char val)
{
	int ret;
	char buff[1];			/* USB transfer buffer (FIXME:should this be kmallocd?) */
	if (debug&DEBUGLOGIC || debug&DEBUGCAMERA) PDEBUG("qc_stv_set(qc=%p,reg=0x%04X,val=%u)",qc,(int)reg,(int)val);
	if (PARANOID && qc==NULL) { PDEBUG("qc==NULL!"); return -EINVAL; }
	qc_i2c_wait(qc);		/* Wait until no pending commands from qc_i2c_* */
	buff[0] = val;
	ret = usb_control_msg(qc->dev, usb_sndctrlpipe(qc->dev, 0),
		0x04,			/* Request */
		0x40,			/* RequestType */
		reg, 0,			/* Value, Index */
		buff, 1, HZ);
	if ((debug&DEBUGERRORS || debug&DEBUGLOGIC) && ret<0) PDEBUG("Failed qc_stv_set()=%i", ret);
	if (ret<0) return ret;
	return 0;
}
/* }}} */
/* {{{ [fold] int qc_stv_get(struct quickcam *qc, unsigned short reg) */
/*
 * Read one byte register in the STV-chip. qc_i2c_init() must be called first!
 * Return the unsigned register value or negative error code on error.
 */
int qc_stv_get(struct quickcam *qc, unsigned short reg)
{
	u8 buff[1];			/* USB transfer buffer (FIXME:should this be kmallocd?) */
	int ret;
	if (debug&DEBUGLOGIC || debug&DEBUGCAMERA) PDEBUG("qc_stv_get(qc=%p,reg=0x%04X)",qc,(int)reg);
	if (PARANOID && qc==NULL) { PDEBUG("qc==NULL!"); return -EINVAL; }
	qc_i2c_wait(qc);		/* Wait until no pending commands from qc_i2c_* */
	ret = usb_control_msg(qc->dev, usb_rcvctrlpipe(qc->dev, 0),
		0x04,			/* Request */
		0xC0,			/* RequestType */
		reg, 0,			/* Value, Index */
		buff, 1, HZ);
	if ((debug&DEBUGERRORS || debug&DEBUGLOGIC) && ret<0) PDEBUG("Failed qc_stv_get()=%i", ret);
	if (ret<0) return ret;
	if (debug&DEBUGCAMERA) PDEBUG("qc_stv_get(reg=0x%04X)=%02X", reg, buff[0]);
	return buff[0];
}
/* }}} */
/* {{{ [fold] int qc_stv_setw(struct quickcam *qc, unsigned short reg, unsigned short val) */
/*
 * Set two byte register in the STV-chip. qc_i2c_init() must be called first!
 * "w" means either "word" or "wide", depending on your religion ;)
 */
int qc_stv_setw(struct quickcam *qc, unsigned short reg, unsigned short val)
{
	char buff[2];		/* USB transfer buffer (FIXME:should this be kmalloc'd?) */
	int ret;
	if (debug&DEBUGLOGIC || debug&DEBUGCAMERA) PDEBUG("qc_stv_setw(qc=%p,reg=0x%04X,val=%i)",qc,(int)reg,(int)val);
	if (PARANOID && qc==NULL) { PDEBUG("qc==NULL!"); return -EINVAL; }
	qc_i2c_wait(qc);
	buff[0] = val & 0xFF;
	buff[1] = (val >> 8) & 0xFF;
	ret = usb_control_msg(qc->dev, usb_sndctrlpipe(qc->dev, 0),
		0x04,
		0x40,
		reg, 0,
		buff, 2, HZ);
	if ((debug&DEBUGERRORS || debug&DEBUGLOGIC) && ret<0) PDEBUG("Failed qc_stv_setw()=%i", ret);
	if (ret<0) return ret;
	return 0;
}
/* }}} */
/* {{{ [fold] void qc_hsv2rgb(s16 hue, u16 sat, u16 val, int *red, int *green, int *blue) */
/* Convert HSI (hue, saturation, intensity) to RGB (red, green, blue).
 * All input and output values are 0..65535.
 * Hue is actually signed, so it is -32768..32767, but this is equivalent
 * since it is the angle around full circle (0=Red, 21845=Green, 43690=Blue).
 * Based on libgimp, converted to 16.16 fixed point by tuukkat.
 */
void qc_hsv2rgb(s16 hue, u16 sat, u16 val, int *red, int *green, int *blue)
{
	unsigned int segment, valsat;
	signed int   h = (u16)hue;
	unsigned int s = (sat<32768) ? 0 : (sat-32768)*2;	/* 32768 or less = no saturation */
	unsigned int v = val;					/* value = intensity */
	unsigned int p;

#if 1	/* Make common case more efficient */
	if (s == 0) {
		*red   = v;
		*green = v;
		*blue  = v;
		return;
	}
#endif
	segment = (h + 10923) & 0xFFFF;		
	segment = segment*3 >> 16;		/* 0..2: 0=R, 1=G, 2=B */
	hue -= segment * 21845;			/* -10923..10923 */
	h = hue;
	h *= 3;
	valsat = v*s >> 16;			/* 0..65534 */
	p = v - valsat;
	if (h>=0) {
		unsigned int t = v - (valsat * (32769 - h) >> 15);
		switch (segment) {
		default:
			PDEBUG("hsi2rgb: this can never happen!");
		case 0:	/* R-> */
			*red   = v;
			*green = t;
			*blue  = p;
			break;
		case 1:	/* G-> */
			*red   = p;
			*green = v;
			*blue  = t;
			break;
		case 2:	/* B-> */
			*red   = t;
			*green = p;
			*blue  = v;
			break;
		}
	} else {
		unsigned int q = v - (valsat * (32769 + h) >> 15);
		switch (segment) {
		default:
			PDEBUG("hsi2rgb: this can never happen!");
		case 0:	/* ->R */
			*red   = v;
			*green = p;
			*blue  = q;
			break;
		case 1:	/* ->G */
			*red   = q;
			*green = v;
			*blue  = p;
			break;
		case 2:	/* ->B */
			*red   = p;
			*green = q;
			*blue  = v;
			break;
		}
	}
	//PDEBUG("hue=%i sat=%i val=%i  segment=%i h=%i  r=%i g=%i b=%i\n",hue,sat,val, segment,h, *red,*green,*blue);
}

/* }}} */

/* }}} */
/* {{{ [fold] **** qc_i2c:    I2C URB messaging routines (qc_i2c_*) ************* */

/* We have here a quite typical producer-consumer scheme:
 * URB interrupt handler routine consumes i2c data, while
 * kernel mode processes create more of it.
 * Ref: Linux Device Drivers, Alessandro Rubini et al, 2nd edition, pg. 279
 * "Using Circular Buffers"
 */

static const int qc_i2c_maxbufsize = 0x23;

/* {{{ [fold] (private) qc_i2c_nextpacket(struct quickcam *qc) */
/* Fill URB and submit it, if there are more data to send 
 * Consume data from "commands" array. May be called from interrupt context.
 * Return standard error code.
 */
static int qc_i2c_nextpacket(struct quickcam *qc)
{
	struct qc_i2c_data *id = &qc->i2c_data;
	struct urb *urb = id->urb;
	u8 *tb = urb->transfer_buffer, flags;
	struct usb_ctrlrequest *cr = (struct usb_ctrlrequest *)urb->setup_packet;
	unsigned int newtail, length, regnum, i, j;
	signed int r;

	if (debug&DEBUGLOGIC) PDEBUG("qc_i2c_nextpacket(quickcam=%p), tail=%i, head=%i, interrupt=%i",qc,id->tail,id->head,in_interrupt());
	IDEBUG_TEST(*id);
	newtail = id->tail;					/* First data to fetch */
	if (id->packets<=1 && newtail==id->head) {	/* packets==0 or 1: no extra URB need to be scheduled */
		if (debug&DEBUGCONTROLURBS) PDEBUG("No more control URBs to send");
		r = 0;
		goto nourbs;
	}
	if (id->packets<=1) {
		/* Collect data from circular buffer to URB transfer buffer */
		/* Now id->tail!=id->head: there's at least one packet to send */
		if (PARANOID && newtail==id->head) {PRINTK(KERN_CRIT,"no urbs to send"); return -1;}
		id->packets = 1;
		if (qc->dev->descriptor.idProduct==0x0850) id->packets = 2;
		regnum = 0x0400;
		length = qc_i2c_maxbufsize;

		i = 0;							/* Transfer buffer position */
		if (!(id->commands[newtail].flags & I2C_FLAG_WORD)) {
			/* Normal byte-wide register write */
			if (debug&DEBUGCONTROLURBS) PDEBUG("Setting byte-wide registers");
			do {
				tb[i]      = id->commands[newtail].regnum;
				tb[i+0x10] = id->commands[newtail].loval;
				flags      = id->commands[newtail].flags;
				i++;
				newtail    = (newtail + 1) % I2C_MAXCOMMANDS;	/* Next data to fetch */
				if (flags & I2C_FLAG_BREAK) break;		/* Start new packet */
				if (newtail == id->head) break;		/* No more data? */
				if (i > 0x0F) break;			/* Transfer buffer full? */
				if (id->commands[newtail].flags & I2C_FLAG_WORD) break;
			} while (TRUE);
/*if (flags&I2C_FLAG_BREAK) PDEBUG("breaking!!!!!!!!!!");
{
int mm;
for(mm=0;mm<i;mm++) printk("%02X=%02X ",tb[mm],tb[mm+0x10]);
printk("\n");
}*/
			for (j=i; j<0x10; j++) tb[j+0x10] = 0;	/* Zero out unused register values just to be sure */
		} else {
			/* Two-byte-wide register write (used in Photobit) */
			if (debug&DEBUGCONTROLURBS) PDEBUG("Setting word-wide registers");
			do {
				tb[i]        = id->commands[newtail].regnum;
				tb[i*2+0x10] = id->commands[newtail].loval;
				tb[i*2+0x11] = id->commands[newtail].hival;
				flags        = id->commands[newtail].flags;
				i++;
				newtail = (newtail + 1) % I2C_MAXCOMMANDS;	/* Next data to fetch */
				if (flags & I2C_FLAG_BREAK) break;		/* Start new packet */
				if (newtail == id->head) break;		/* No more data? */
				if (i > 0x07) break;			/* Transfer buffer full? */
				if (!(id->commands[newtail].flags & I2C_FLAG_WORD)) break;
			} while (TRUE);
			for (j=i*2; j<0x10; j++) tb[j+0x10] = 0;	/* Zero out unused register values just to be sure */
		}
		for (j=i; j<0x10; j++) tb[j] = 0;	/* Zero out unused register addresses just to be sure */
		tb[0x20] = qc->sensor_data.sensor->i2c_addr;
		tb[0x21] = i-1;			/* Number of commands to send - 1 */
		tb[0x22] = 1;			/* Write cmd, 03 would be read. */
		id->tail = newtail;
		if (debug&DEBUGCONTROLURBS) PDEBUG("sending i2c packet, cmds=%i, reg0=%02X, val0=%02X",tb[0x21]+1,tb[0],tb[0x10]);
	} else {
		/* id->packets==2: send extra packet for Quickcam Web */
		if (debug&DEBUGCONTROLURBS) PDEBUG("sending finalization packet");
		id->packets = 1;
		regnum = 0x1704;
		length = 1;
		tb[0] = 1;
	}
	urb->dev    = qc->dev;		/* 2.4.x zeroes urb->dev after submission */
	urb->pipe   = usb_sndctrlpipe(qc->dev, 0);
	urb->transfer_buffer_length = length;
	cr->wValue  = cpu_to_le16(regnum);
	cr->wLength = cpu_to_le16(length);
	r = usb_submit_urb(urb);
	if (r<0) {
		PRINTK(KERN_ERR,"Failed qc_i2c_nextpacket()=%i", r);
		goto nourbs;
	}
	return 0;

nourbs:	id->packets = 0;	/* No more URBs are scheduled */
	wake_up(&id->wq);
	return r;
}
/* }}} */
/* {{{ [fold] (private) qc_i2c_handler(struct urb *urb) */
/* URB completion handler, called in interrupt context */
static void qc_i2c_handler(struct urb *urb)
{
	struct quickcam *qc = urb->context;

	if (debug&DEBUGINTERRUPTS) PDEBUG("[INTR] qc_i2c_handler(urb=%p)",urb);
	if (PARANOID && urb==NULL) { PRINTK(KERN_CRIT,"urb==NULL"); return; }
	if (PARANOID && qc==NULL) { PRINTK(KERN_CRIT,"qc==NULL"); return; }
	IDEBUG_TEST(qc->i2c_data);
	if (urb->status<0) {
		switch (urb->status) {
		default:
			/* Seen here: ECONNABORTED    103     Software caused connection abort */
			PRINTK(KERN_ERR,"Unhandled control URB error %i",urb->status);
		case -EPROTO:		/* Bitstuff error or unknown USB error */
		case -EILSEQ:		/* CRC mismatch */
		case -ETIMEDOUT:	/* Transfer timed out */
		case -EREMOTEIO:	/* Short packet detected */
		case -EPIPE:		/* Babble detect or endpoint stalled */
			/* We could try resubmitting the URB here */
		case -ENOENT:		/* URB was unlinked */
		case -ENODEV:		/* Device was removed */
			PRINTK(KERN_ERR,"Control URB error %i",urb->status);
			qc->i2c_data.packets = 0;	/* Don't schedule more URBs */
			wake_up(&qc->i2c_data.wq);
			return;
		}
	}
	qc_i2c_nextpacket(qc);
}
/* }}} */
/* {{{ [fold] qc_i2c_flush(struct quickcam *qc) */
/* Allow all register settings set earlier to be scheduled and sent to camera */
static int qc_i2c_flush(struct quickcam *qc)
{
	struct qc_i2c_data *id = &qc->i2c_data;
	int r = 0;

	if (debug&DEBUGLOGIC) PDEBUG("qc_i2c_flush(quickcam=%p,regs=%i)",qc,
			(id->newhead+I2C_MAXCOMMANDS-id->head)%I2C_MAXCOMMANDS);
	IDEBUG_TEST(*id);
	id->head = id->newhead;
	if (id->packets==0)	/* Schedule URB if there aren't any in progress */
		r = qc_i2c_nextpacket(qc);
	return r;
}
/* }}} */
/* {{{ [fold] qc_i2c_wait(struct quickcam *qc) */
/* Wait until all previosly set registers are set or abort all transmissions
 * and return error code (but afterwards no more URBs must be scheduled)
 */
int qc_i2c_wait(struct quickcam *qc)
{
	struct qc_i2c_data *id = &qc->i2c_data;
	int r = -ENODEV, r2;

	if (debug&DEBUGLOGIC) PDEBUG("qc_i2c_wait(quickcam=%p)",qc);
	if (PARANOID && in_interrupt()) { PDEBUG("qc_i2c_wait() in interrupt!!"); return -EINVAL; }
	if (PARANOID && qc==NULL) { PRINTK(KERN_CRIT,"qc==NULL"); return -EINVAL; }
	IDEBUG_TEST(*id);

	if (!qc->dev) goto cancel;
	r = qc_i2c_flush(qc);
	if (r>=0) r = wait_event_interruptible(id->wq, id->packets==0);
	if (r<0) goto cancel;
	return 0;

cancel:	if (debug&DEBUGLOGIC) PDEBUG("Canceling all pending URBs");
	if ((r2=usb_unlink_urb(id->urb))<0) PRINTK(KERN_ERR,"usb_unlink_urb()=%i failed!", r2);
	/* If unlink failed, pray that there won't be more USB interrupts! */
	id->packets = 0;
	return r;
}
/* }}} */
/* {{{ [fold] (private) qc_i2c_set0(struct quickcam *qc, unsigned char regnum, unsigned char loval, unsigned char hival, int flags) */
/* Called from qc_i2c_set and qc_i2c_setw, should not be called elsewhere */
static int qc_i2c_set0(struct quickcam *qc, unsigned char regnum, unsigned char loval, unsigned char hival, int flags)
{
	struct qc_i2c_data *id = &qc->i2c_data;
	unsigned int newhead;
	signed int r;

	if (debug&DEBUGLOGIC) PDEBUG("qc_i2c_set0(quickcam=%p,reg=%02X,val=%02X%02X)",qc,regnum,hival,loval);
	if (PARANOID && qc==NULL) { PRINTK(KERN_CRIT,"qc==NULL"); return -EINVAL; }
	IDEBUG_TEST(*id);
	newhead = id->newhead;
	id->commands[newhead].loval  = loval;
	id->commands[newhead].hival  = hival;
	id->commands[newhead].regnum = regnum;
	id->commands[newhead].flags  = flags;
	newhead = (newhead + 1) % I2C_MAXCOMMANDS;
	if (newhead == id->tail) {		/* If buffer is full, wait until it's empty */
		if (debug&DEBUGCONTROLURBS) PDEBUG("i2c buffer is full, waiting");
		r = qc_i2c_wait(qc);
		if (r<0) return r;
	}
	if (PARANOID && newhead==id->tail) {
		PRINTK(KERN_CRIT,"no i2c buffer space but nothing to send!!!");
		return -ENOSPC;
	}
	id->newhead = newhead;
	return 0;
}
/* }}} */
/* {{{ [fold] qc_i2c_set(struct quickcam *qc, unsigned char reg, unsigned char val) */
/* Set an I2C register to desired value */
/* (queue setting to be sent later when qc_i2c_flush() is called) */
inline int qc_i2c_set(struct quickcam *qc, unsigned char reg, unsigned char val)
{
	return qc_i2c_set0(qc, reg, val, 0, 0);
}
/* }}} */
/* {{{ [fold] qc_i2c_setw(struct quickcam *qc, unsigned char reg, unsigned short val) */
/* Set a two-byte (word length) I2C register to desired value (queue setting to be sent later) */
/* (queue setting to be sent later when qc_i2c_flush() is called) */
inline int qc_i2c_setw(struct quickcam *qc, unsigned char reg, unsigned short val)
{
	return qc_i2c_set0(qc, reg, val & 0xFF, (val >> 8) & 0xFF, I2C_FLAG_WORD);
}
/* }}} */
/* {{{ [fold] qc_i2c_break(struct quickcam *qc)  */
/* The next register written will be sent in another packet */
int qc_i2c_break(struct quickcam *qc)
{
	struct qc_i2c_data *id = &qc->i2c_data;
	unsigned int prevhead;
	if (debug&DEBUGLOGIC) PDEBUG("qc_i2c_break(quickcam=%p)",qc);
	if (PARANOID && qc==NULL) { PRINTK(KERN_CRIT,"qc==NULL"); return -EINVAL; }
	IDEBUG_TEST(*id);
	/* We access an entry that may be already submitted and even finished */
	/* But it should not harm */
	prevhead = (id->newhead + I2C_MAXCOMMANDS - 1) % I2C_MAXCOMMANDS;
	id->commands[prevhead].flags |= I2C_FLAG_BREAK;
	barrier();
	return qc_i2c_flush(qc);
}
/* }}} */
/* {{{ [fold] qc_i2c_init(struct quickcam *qc) */
/* Initialize structures and hardware for I2C communication */
static int qc_i2c_init(struct quickcam *qc)
{
	struct qc_i2c_data *id = &qc->i2c_data;
	struct urb *urb;
	struct usb_ctrlrequest *cr;
	int r = -ENOMEM;

	if (debug&DEBUGLOGIC || debug&DEBUGINIT) PDEBUG("qc_i2c_init(quickcam=%p)",qc);
	if (PARANOID && qc==NULL) { PDEBUG("qc==NULL!"); return -EINVAL; }

	id->tail = id->head = id->newhead = 0;	/* Next position to be filled and sent is 0 */
	id->packets = 0;
	init_waitqueue_head(&id->wq);

	/* Allocate an URB and associated buffers and fill them */
	urb = id->urb = usb_alloc_urb(0);
	if (!urb) goto fail1;
	urb->setup_packet = (unsigned char *)cr = kmalloc(sizeof(struct usb_ctrlrequest), GFP_KERNEL);
	if (!cr) goto fail2;
	urb->transfer_buffer = kmalloc(qc_i2c_maxbufsize*sizeof(u8), GFP_KERNEL);	/* Allocate maximum ever needed */
	if (!urb->transfer_buffer) goto fail3;
	spin_lock_init(&urb->lock);
	urb->complete = (usb_complete_t)qc_i2c_handler;
	urb->context  = qc;
	urb->timeout  = HZ;			/* 1 s */
	urb->next     = NULL;
	cr->bRequestType = 0x40;
	cr->bRequest     = 0x04;
	cr->wIndex       = 0;
	IDEBUG_INIT(*id);
	return 0;

fail3:	kfree(cr);
fail2:	usb_free_urb(urb);
	if (PARANOID) id->urb = NULL;
fail1:	return r;
}
/* }}} */
/* {{{ [fold] qc_i2c_exit(struct quickcam *qc) */
/* Close messaging, free up memory, stop messaging */
static void qc_i2c_exit(struct quickcam *qc)
{
	struct qc_i2c_data *id = &qc->i2c_data;

	if (debug&DEBUGLOGIC || debug&DEBUGINIT) PDEBUG("qc_i2c_exit(qc=%p)",qc);
	if (PARANOID && qc==NULL) { PDEBUG("qc==NULL!"); return; }
	qc_i2c_wait(qc);
	kfree(id->urb->setup_packet);
	kfree(id->urb->transfer_buffer);
	usb_free_urb(id->urb);
	if (PARANOID) id->urb = NULL;
	IDEBUG_EXIT(*id);
}
/* }}} */

/* }}} */
/* {{{ [fold] **** qc_proc:   /proc interface *********************************** */
#if HAVE_PROCFS

static struct proc_dir_entry *qc_proc_entry = NULL;
extern struct proc_dir_entry *video_proc_entry;

/* {{{ [fold] qc_proc_read(char *page, char **start, off_t off, int count, int *eof, void *data) */
static inline const char *qc_proc_yesno(Bool b)
{
	return b ? "Yes" : "No";
}

static int qc_proc_read(char *page, char **start, off_t off, int count, int *eof, void *data)
{
	struct quickcam *qc = data;
	char *out = page;
	int len;

	out += sprintf(out, "\tGeneral driver status\n");
	out += sprintf(out, "Driver version   : %s\n", VERSION);
	out += sprintf(out, "Users            : %i\n", qc->users);
	out += sprintf(out, "Disconnected     : %s\n", qc_proc_yesno(qc->dev==NULL));

	out += sprintf(out, "\n\tPicture settings set by user\n");
	out += sprintf(out, "Brightness       : %d\n", (int)qc->vpic.brightness);
	out += sprintf(out, "Hue              : %d\n", (int)qc->vpic.hue);
	out += sprintf(out, "Color            : %d\n", (int)qc->vpic.colour);
	out += sprintf(out, "Contrast         : %d\n", (int)qc->vpic.contrast);
	out += sprintf(out, "Whiteness        : %d\n", (int)qc->vpic.whiteness);
	out += sprintf(out, "Depth            : %d\n", (int)qc->vpic.depth);
	out += sprintf(out, "Palette          : %s\n", qc_fmt_getname(qc->vpic.palette));

	out += sprintf(out, "\n\tOutput window\n");
	out += sprintf(out, "Width            : %d\n", (int)qc->vwin.width);
	out += sprintf(out, "Height           : %d\n", (int)qc->vwin.height);

	out += sprintf(out, "\n\tSensor\n");
	out += sprintf(out, "Type             : %s\n", qc->sensor_data.sensor->name);
	out += sprintf(out, "Manufacturer     : %s\n", qc->sensor_data.sensor->manufacturer);
	out += sprintf(out, "Maximum width    : %d\n", qc->sensor_data.maxwidth);
	out += sprintf(out, "Maximum height   : %d\n", qc->sensor_data.maxheight);
	out += sprintf(out, "Current width    : %d\n", qc->sensor_data.width);
	out += sprintf(out, "Current height   : %d\n", qc->sensor_data.height);

	out += sprintf(out, "\n\tI2C command stream\n");
	out += sprintf(out, "Scheduled packets: %d\n", qc->i2c_data.packets);
	out += sprintf(out, "Packets on queue : %d\n", (I2C_MAXCOMMANDS + qc->i2c_data.head - qc->i2c_data.tail) % I2C_MAXCOMMANDS);

	out += sprintf(out, "\n\tIsochronous data stream\n");
	out += sprintf(out, "Stream enabled   : %s\n", qc_proc_yesno(qc->isoc_data.streaming));
	out += sprintf(out, "Transfer errors  : %d\n", qc->isoc_data.errorcount);

	out += sprintf(out, "\n\tFrame buffering\n");
	out += sprintf(out, "Frames on queue  : %d\n", (FRAME_BUFFERS + qc->frame_data.head - qc->frame_data.tail) % FRAME_BUFFERS);
	out += sprintf(out, "Capturing        : %s\n", qc_proc_yesno(qc->stream_data.capturing));
	out += sprintf(out, "Waiting processes: %d\n", qc->frame_data.waiting);

	out += sprintf(out, "\n\tAutomatic exposure control\n");
	out += sprintf(out, "Picture intensity: %d\n", qc->adapt_data.oldmidvalue);
	out += sprintf(out, "Exposure setting : %d\n", qc->adapt_data.exposure);
	out += sprintf(out, "Gain setting     : %d\n", qc->adapt_data.gain);
	out += sprintf(out, "Delta value      : %d\n", qc->adapt_data.olddelta);
	out += sprintf(out, "Control algorithm: ");
	switch (qc->adapt_data.controlalg) {
		case EXPCONTROL_SATURATED: out += sprintf(out, "Saturated\n"); break;
		case EXPCONTROL_NEWTON:    out += sprintf(out, "Newton\n"); break;
		case EXPCONTROL_FLOAT:     out += sprintf(out, "Float\n"); break;
		default: out += sprintf(out, "?\n"); break;
	}

	out += sprintf(out, "\n\tDefault settings\n");
	out += sprintf(out, "Debug            : 0x%02X\n", debug);
	out += sprintf(out, "Keep settings    : %s\n", qc_proc_yesno(qc->settings.keepsettings));
	out += sprintf(out, "Settle image     : %s\n", qc_proc_yesno(qc->settings.settle));
	out += sprintf(out, "Subsampling      : %s\n", qc_proc_yesno(qc->settings.subsample));
	out += sprintf(out, "Compress         : %s\n", qc_proc_yesno(qc->settings.compress));
	out += sprintf(out, "Frame skipping   : %i\n", qc->settings.frameskip);
	out += sprintf(out, "Image quality    : %i\n", qc->settings.quality);
	out += sprintf(out, "Adaptive         : %s\n", qc_proc_yesno(qc->settings.adaptive));
	out += sprintf(out, "Equalize         : %s\n", qc_proc_yesno(qc->settings.equalize));
	out += sprintf(out, "Retryerrors      : %s\n", qc_proc_yesno(qc->settings.retryerrors));
	out += sprintf(out, "Compatible 8x    : %s\n", qc_proc_yesno(qc->settings.compat_8x));
	out += sprintf(out, "Compatible DblBuf: %s\n", qc_proc_yesno(qc->settings.compat_dblbuf));
	out += sprintf(out, "Compatible ToRgb : %s\n", qc_proc_yesno(qc->settings.compat_torgb));

	len = out - page;
	len -= off;
	if (len < count) {
		*eof = 1;
		if (len <= 0) return 0;
	} else
		len = count;
	*start = page + off;
	return len;
}
/* }}} */
/* {{{ [fold] qc_proc_write(struct file *file, const char *buffer, unsigned long count, void *data) */
static int qc_proc_write(struct file *file, const char *buffer, unsigned long count, void *data)
{
	/* we don't support this....yet? Might replace qcset some day */
	return -EINVAL;
}
/* }}} */
/* {{{ [fold] qc_proc_create(struct quickcam *qc) */
/* Called for each camera plugged in, create file containing information of the camera */
static int qc_proc_create(struct quickcam *qc)
{
	char name[9];
	struct proc_dir_entry *entry;
	if (debug&DEBUGLOGIC || debug&DEBUGINIT) PDEBUG("qc_proc_create(quickcam=%p)",qc);
	if (PARANOID && (!qc || !qc_proc_entry)) {
		PDEBUG("Can not create proc entry, qc=%p, qc_proc_entry=%p!!!",qc,qc_proc_entry);
		return -ENXIO;
	}
	sprintf(name, "video%d", qc->vdev.minor);
	entry = create_proc_entry(name, S_IFREG | S_IRUGO | S_IWUSR, qc_proc_entry);
	if (!entry) return -ENOMEM;
	entry->data = qc;
	entry->read_proc = qc_proc_read;
	entry->write_proc = qc_proc_write;
	qc->proc_entry = entry;
	return 0;
}
/* }}} */
/* {{{ [fold] qc_proc_destroy(struct quickcam *qc) */
static void qc_proc_destroy(struct quickcam *qc)
{
	char name[9];
	if (debug&DEBUGLOGIC || debug&DEBUGINIT) PDEBUG("qc_proc_destroy(quickcam=%p)",qc);
	if (PARANOID && (!qc || !qc->proc_entry)) {
		PDEBUG("Can not destroy proc entry!!!");
		return;
	}
	sprintf(name, "video%d", qc->vdev.minor);
	remove_proc_entry(name, qc_proc_entry);
	if (PARANOID) qc->proc_entry = NULL;
	if (debug&DEBUGLOGIC) PDEBUG("qc_proc_destroy() done");
}
/* }}} */
/* {{{ [fold] qc_proc_init(void) */
/* Called when the driver is initially loaded, creates "quickcam" subdirectory */
static int qc_proc_init(void)
{
	if (debug&DEBUGLOGIC || debug&DEBUGINIT) PDEBUG("qc_proc_init()");
	if (!video_proc_entry) return -ENXIO;
	qc_proc_entry = create_proc_entry(qc_name, S_IFDIR, video_proc_entry);
	if (!qc_proc_entry) return -ENOMEM;
	qc_proc_entry->owner = THIS_MODULE;
	return 0;
}
/* }}} */
/* {{{ [fold] qc_proc_exit(void) */
static void qc_proc_exit(void)
{
	if (debug&DEBUGLOGIC || debug&DEBUGINIT) PDEBUG("qc_proc_exit()");
	if (PARANOID && !qc_proc_entry) {
		PDEBUG("Missing proc dir entry, cannot remove");
		return;
	}
	remove_proc_entry(qc_name, video_proc_entry);
	if (PARANOID) qc_proc_entry = NULL;
}
/* }}} */

#else
static inline int qc_proc_create(struct quickcam *qc) { return 0; }
static inline void qc_proc_destroy(struct quickcam *qc) { }
static inline int qc_proc_init(void) { return 0; }
static inline void qc_proc_exit(void) { }
#endif /* HAVE_PROCFS */
/* }}} */
/* {{{ [fold] **** qc_adapt:  Automatic exposure control ************************ */

/* {{{ [fold] qc_adapt_init(struct quickcam *qc) */
/* Initialize automatic exposure control structure. */
static int qc_adapt_init(struct quickcam *qc)
{
	struct qc_adapt_data *ctrl = &qc->adapt_data;
	ctrl->gain         = 32767;
	ctrl->olddelta     = 4*256;			/* best guess */
	ctrl->exposure     = 32767;
	ctrl->oldexposure  = ctrl->exposure + 1;	/* Slightly different for _issettled() */
	ctrl->midvaluesum  = ctrl->oldmidvalue = 0;
	ctrl->framecounter = 0;
	ctrl->controlalg   = EXPCONTROL_SATURATED;
	IDEBUG_INIT(*ctrl);
	return 0;
}
/* }}} */
/* {{{ [fold] qc_adapt_exit(struct quickcam *qc) */

static inline void qc_adapt_exit(struct quickcam *qc)
{
#ifndef NDEBUG
	struct qc_adapt_data *ctrl = &qc->adapt_data;
	if (debug&DEBUGINIT) PDEBUG("qc_adapt_exit(ctrl=%p)",ctrl);
	IDEBUG_EXIT(*ctrl);
#endif
}

/* }}} */
/* {{{ [fold] qc_adapt_reset(struct quickcam *qc) */
/* Must be called each time just before starting video adaptation */
static inline void qc_adapt_reset(struct quickcam *qc)
{
	IDEBUG_TEST(qc->adapt_data);
	if (!qc->settings.keepsettings) {
		IDEBUG_EXIT(qc->adapt_data);
		qc_adapt_init(qc);
	}
}
/* }}} */
/* {{{ [fold] qc_adapt_hassettled(struct quickcam *qc) */
/* Return TRUE if the image brightness has settled */
static Bool qc_adapt_hassettled(struct quickcam *qc)
{
	struct qc_adapt_data *ctrl = &qc->adapt_data;
	IDEBUG_TEST(*ctrl);
	if (ctrl->framecounter != 0) return FALSE;
//PDEBUG("control=%i  oldexp=%i  exp=%i",ctrl->controlalg,ctrl->oldexposure,ctrl->exposure);
	return ctrl->controlalg==EXPCONTROL_FLOAT || ctrl->oldexposure==ctrl->exposure;
}
/* }}} */
/* {{{ [fold] qc_adapt(struct quickcam *qc, int midvalue, int target, int *ret_exposure, int *ret_gain) */

/* Set image exposure and gain so that computed midvalue approaches the target value */
static void qc_adapt(struct quickcam *qc, int midvalue, int target, int *ret_exposure, int *ret_gain)
{
	struct qc_adapt_data *ctrl = &qc->adapt_data;
#if !MEASURE_ADAPT_DELAY
	/* Here are some constant for controlling the adaptation algorithm. You may play with them. */
	static const int saturation_min = 32;			/* (0-127) If midvalue is out of this range, image is */
	static const int saturation_max = 256 - 8;		/* (128-255) considered saturated and no Newton is used */

	static const int adaptation_min = 3;			/* (0-128) For small variations, do not change exposure */

	static const int delta_min      = 256/2;		/* (2-16*256) Minimum and maximum value for delta */
	static const int delta_max      = 256*256;		/* (4*256-1024*256) */
	
	static const int dmidvalue_min  = 4;			/* (1-128) Minimum differences, under which delta estimation */
	static const int dexposure_min  = 4;			/* (1-32000) will not be done due to inaccuracies */
	
	static const int delta_speed    = 256;			/* (0-256) How fast or slowly delta can change */
	static const int small_adapt    = 8;			/* (0-1024) When very near optimal, exposure change size */
	static const int underestimate  = 16;			/* (0-250) Underestimation, may prevent oscillation */
	static const int bestguess      = 256/2;		/* (2-1024*256) If delta can not be computed, guess this */
	static const int midvalueaccum  = 2;			/* (1-100) How many frames to use for midvalue averaging */
	static const int framedelay     = 4;			/* (0-8) How many frames there are before a new exposure setting in effect */
								/* With Quickcam Web: if set at frame #0, it will be in effect at frame #4; skip 3 frames #1,#2,#3 */
								/* -> should be 3 with Quickcam Web, but it oscillates, FIXME:why? Setting to 4 fixes this */
	static const int gainstep       = 256;			/* (0-32768) Amount to change gain at one step */
	static const int gainneeded     = 10;			/* (0-255) How eagerly to change brightness with gain */
	/* End of tunable constants */

	int newexposure, delta=0;
	int dexposure=0, dmidvalue=0;
	int deviation=0;			/* Deviation of actual brightness from target brightness */
	int smoothdelta=0;			/* Final, smoothed, value of delta */

	if (PARANOID && (ctrl==NULL || ret_gain==NULL || ret_exposure==NULL)) { PRINTK(KERN_CRIT,"ctrl||ret==NULL"); return; }
	IDEBUG_TEST(*ctrl);

	/* When setting a new exposure, it takes 4 frames to have effect (at 7.5 fps) */
	if (ctrl->framecounter >= framedelay)
		ctrl->midvaluesum += midvalue;
	ctrl->framecounter++;
	if (ctrl->framecounter < framedelay+midvalueaccum) {
		*ret_exposure = ctrl->exposure;
		*ret_gain     = ctrl->gain;
		return;
	}

	midvalue = ctrl->midvaluesum / midvalueaccum;
	ctrl->framecounter = 0;
	ctrl->midvaluesum  = 0;

	if (ctrl->exposure >= qc->sensor_data.sensor->adapt_gainhigh && 
	    ctrl->oldexposure >= qc->sensor_data.sensor->adapt_gainhigh &&
	    target - ctrl->oldmidvalue > gainneeded &&
	    target - midvalue > gainneeded)
	{
		/* Exposure is at maximum, but image is still too dark. Increase gain.*/
		ctrl->gain = ctrl->gain + ctrl->gain/2 + gainstep;
		if (ctrl->gain > 65535) ctrl->gain = 65535;
		if (debug&DEBUGADAPTATION) PDEBUG("increasing gain to %i", ctrl->gain);
	} else 
	if (ctrl->exposure <= qc->sensor_data.sensor->adapt_gainlow &&
	    ctrl->oldexposure <= qc->sensor_data.sensor->adapt_gainlow &&
	    target - ctrl->oldmidvalue <= gainneeded &&
	    target - midvalue <= gainneeded)
	{
		/* Decrease gain if unnecessarily high */
		ctrl->gain = ctrl->gain - ctrl->gain/2 - gainstep;
		if (ctrl->gain < 0) ctrl->gain = 0;
		if (debug&DEBUGADAPTATION) PDEBUG("decreasing gain to %i", ctrl->gain);
	}
	
	if (ctrl->oldmidvalue<saturation_min || midvalue<saturation_min) {
		/* Image was undersaturated, Newton method would give inaccurate results */
		if (debug&DEBUGADAPTATION) PDEBUG("Increasing exposure");
		ctrl->controlalg = EXPCONTROL_SATURATED;
		newexposure = ctrl->exposure * 2;
	} else
	if (ctrl->oldmidvalue>=saturation_max || midvalue>=saturation_max) {
		/* Image is oversaturated, Newton method would give inaccurate results */
		if (debug&DEBUGADAPTATION) PDEBUG("Decreasing exposure");
		ctrl->controlalg = EXPCONTROL_SATURATED;
		newexposure = ctrl->exposure / 2;
	} else {
		deviation = target - midvalue;
		if (ABS(deviation) < adaptation_min) {
			/* For small variations, adapt linearly */
			if (debug&DEBUGADAPTATION) PDEBUG("Small deviation %i",deviation);
			ctrl->controlalg = EXPCONTROL_FLOAT;
			newexposure = small_adapt * SGN(deviation) + ctrl->exposure;
		} else {
			/* Try using Newton method for estimating correct exposure value */
			ctrl->controlalg = EXPCONTROL_NEWTON;
			dmidvalue = midvalue       - ctrl->oldmidvalue;
			dexposure = ctrl->exposure - ctrl->oldexposure;
			if (ABS(dmidvalue) <  dmidvalue_min || 
			    ABS(dexposure) <  dexposure_min ||
			    SGN(dmidvalue) != SGN(dexposure))
			{
				/* Can not estimate delta with Newton method, just guess */
				if (ctrl->olddelta < 2) {
					if (debug&DEBUGADAPTATION) PDEBUG("Best guessing");
					smoothdelta = bestguess;
				} else {
					smoothdelta = ctrl->olddelta * 3 / 2;
					if (debug&DEBUGADAPTATION) PDEBUG("Change more exposure, smoothdelta=%i",smoothdelta);
				}			
			} else {
				/* Everything is well, use here actual Newton method */
				delta       = (256 - underestimate) * dexposure / dmidvalue;
				smoothdelta = (delta_speed*delta + (256-delta_speed)*ctrl->olddelta) / 256;
				if (debug&DEBUGADAPTATION) PDEBUG("Using Newton, delta=%i",delta);
			}
		}
		/* Compute new exposure based on guessed/computed delta */
		smoothdelta = CLIP(smoothdelta, delta_min,delta_max);
		dexposure = deviation * smoothdelta / 256;
		/* Newton works linearly, but exposure/brightness are not linearly related */
		/* The following test fixes the worst deficiencies due to that (I hope) */
		if (-dexposure > ctrl->exposure/2)
			dexposure = -ctrl->exposure/2;
		newexposure = dexposure + ctrl->exposure;
		ctrl->olddelta = smoothdelta;
	}

	newexposure       = CLIP(newexposure, 2,65535);

	if (debug&DEBUGADAPTATION) 
		PDEBUG("midval=%i dev=%i dmidv=%i dexp=%i smdelta=%i olddelta=%i newexp=%i gain=%i",
		midvalue,deviation,dmidvalue,dexposure,smoothdelta,ctrl->olddelta,newexposure,ctrl->gain);

	ctrl->oldexposure = ctrl->exposure;
	ctrl->exposure    = newexposure;
	ctrl->oldmidvalue = midvalue;
	*ret_exposure     = newexposure;
	*ret_gain         = ctrl->gain;
#else
	/* This code is for measuring the delay between an exposure settings and until
	 * it becomes in effect. Only useful for developing the adaptation algorithm. */
	static const int changedel = 20;
	static Bool black = FALSE;
	static int framenum = 0;
	PRINTK(KERN_CRIT,"Measuring: framenum=%i, midvalue=%i",framenum,midvalue);
	if ((framenum % changedel)==0) {
		PRINTK(KERN_CRIT,"Measuring: set to white at frame %i",framenum);
		black = FALSE;
	}
	if ((framenum % changedel)==changedel/2) {
		PRINTK(KERN_CRIT,"Measuring: set to black at frame %i",framenum);
		black = TRUE;
	}
	if (black) {
		*ret_exposure = 0;
		*ret_gain = 0;
	} else {
		*ret_exposure = 65535;
		*ret_gain = 65535;
	}
	framenum++;
#endif
}

/* }}} */

/* }}} */
/* {{{ [fold] **** qc_frame:  Frame capturing functions ************************* */

/* From /usr/src/linux/Documentation/smp.tex:
 * + Kernel mode process (e.g. system calls):
 *   - No other kernel mode processes may run simultaneously/pre-empt
 *     (kernel mode processes are atomic with respect to each other)
 *   - Exception is voluntary sleeping, in which case re-entry is allowed
 *   - Interrupts may pre-empt (but return to same process)
 *     (interrupts can be disabled if necessary)
 * + Interrupt mode execution
 *   - Kernel mode process may not pre-empt/execute simultaneously
 *   - Other interrupts may pre-empt, however same interrupt is not nested
 */

/* We have here a quite typical producer-consumer scheme:
 * Interrupt routine produces more frame data, while
 * kernel mode processes consume it
 * Read: Linux Device Drivers, Alessandro Rubini et al, 2nd edition, pg. 279
 * "Using Circular Buffers"
 */

/* Initialization and cleanup routines, called from kernel mode processes */
/* {{{ [fold] qc_frame_init(struct quickcam *qc) */
static int qc_frame_init(struct quickcam *qc)
{
	struct qc_frame_data *fd = &qc->frame_data;
	int n;

	if (debug&DEBUGLOGIC || debug&DEBUGINIT) PDEBUG("qc_frame_init(quickcam=%p)",qc);
	if (PARANOID && (qc==NULL || fd==NULL)) { PRINTK(KERN_CRIT,"qc||fd==NULL"); return -EINVAL; }
	if (PARANOID && in_interrupt()) { PDEBUG("qc_frame_init() in interrupt!!"); return -EBUSY; }
	fd->rawdatabuf = vmalloc(FRAME_DATASIZE * FRAME_BUFFERS);
	if (!fd->rawdatabuf) return -ENOMEM;
	memset(fd->rawdatabuf, 0, FRAME_DATASIZE * FRAME_BUFFERS);	/* Never let user access random kernel data */
	fd->head       = 0;		/* First buffer to fill */
	fd->tail       = 0;		/* First buffer to get */
	init_waitqueue_head(&fd->wq);
	fd->waiting    = 0;
	fd->exiting    = FALSE;
	for (n=0; n<FRAME_BUFFERS; n++) fd->buffers[n].rawdatalen = 0;
	IDEBUG_INIT(*fd);
	return 0;
}
/* }}} */
/* {{{ [fold] qc_frame_exit(struct quickcam *qc) */
static void qc_frame_exit(struct quickcam *qc)
{
	struct qc_frame_data *fd = &qc->frame_data;
#if PARANOID
	unsigned long startjiffy = jiffies;
#endif
	if (debug&DEBUGLOGIC || debug&DEBUGINIT) PDEBUG("qc_frame_exit(quickcam=%p)",qc);
	if (PARANOID && in_interrupt()) { PDEBUG("qc_frame_exit() in interrupt!!"); return; }
	if (PARANOID && (qc==NULL || fd==NULL)) { PRINTK(KERN_CRIT,"qc||fd==NULL"); return; }
	fd->exiting = TRUE;
	fd->maxrawdatalen = 0;		/* Hopefully stops all ongoing captures, might need locking though */
	wake_up(&fd->wq);
	if (debug&DEBUGLOGIC) PDEBUG("waiting=%i",fd->waiting);
	if (debug&DEBUGMUTEX) PDEBUG("up() in qc_frame_exit()");
	up(&qc->lock);			/* The lock was down when entering this function */
	while (fd->waiting > 0) {
		schedule();
#if PARANOID
		if (jiffies-startjiffy > 4*HZ) {
			PRINTK(KERN_CRIT,"Wait queue never completing!! (waiting=%i)",fd->waiting);
			break;
		}
#endif
	}
	if (debug&DEBUGMUTEX) PDEBUG("down() again in qc_frame_exit()");
	down(&qc->lock);
	vfree(fd->rawdatabuf);
	if (PARANOID) fd->rawdatabuf = NULL;
	IDEBUG_EXIT(*fd);
}
/* }}} */

/* Consumer routines, called from kernel mode processes */
/* {{{ [fold] qc_frame_get(struct quickcam *qc, unsigned char **buf) */
/* Wait until next frame is ready and return the frame length
 * and set buf to point to the frame. If error happens,
 * return standard Linux negative error number.
 * qc_frame_free() must be called after the frame is not needed anymore.
 */
static int qc_frame_get(struct quickcam *qc, unsigned char **buf)
{
	struct qc_frame_data *fd = &qc->frame_data;
	int ret;

	if (PARANOID && (qc==NULL || fd==NULL)) { PRINTK(KERN_CRIT,"qc||fd==NULL"); return -EINVAL; }
	if (PARANOID && in_interrupt()) { PDEBUG("qc_frame_get() in interrupt!!"); return -EBUSY; }
	IDEBUG_TEST(*fd);

	/* Wait until the next frame is available */
	if (debug&DEBUGLOGIC) PDEBUG("qc_frame_get(quickcam=%p,tail=%i)",qc,fd->tail);
	fd->waiting++;
	if (debug&DEBUGMUTEX) PDEBUG("up() in qc_frame_exit()");
	up(&qc->lock);					/* Release lock while waiting */
	ret = wait_event_interruptible(fd->wq, fd->head!=fd->tail || fd->exiting);	//FIXME:What if we get -ERESTARTSYS?
	if (debug&DEBUGMUTEX) PDEBUG("down() again in qc_frame_get()");
	down(&qc->lock);
	if (!ret) {
		if (!fd->exiting) {
			*buf = fd->rawdatabuf + fd->tail*FRAME_DATASIZE;
			ret  = fd->buffers[fd->tail].rawdatalen;
		} else {
			ret = -ENODATA;
		}
	}
	fd->waiting--;
	if (ret<0 && (debug&(DEBUGERRORS|DEBUGLOGIC))) PDEBUG("failed qc_frame_get()=%i",ret);
	return ret;
}
/* }}} */
/* {{{ [fold] qc_frame_free(struct quickcam *qc) */
/* Free up the last frame returned from qc_frame_get() (it must be called first) */
static inline void qc_frame_free(struct quickcam *qc)
{
	struct qc_frame_data *fd = &qc->frame_data;
	if (PARANOID && (qc==NULL || fd==NULL)) { PRINTK(KERN_CRIT,"qc||fd==NULL"); return; }
	if (PARANOID && in_interrupt()) { PDEBUG("qc_frame_get() in interrupt!!"); return; }
	IDEBUG_TEST(*fd);
	/* Free up previous frame and advance to next */
	fd->tail = (fd->tail + 1) % FRAME_BUFFERS;
	if (debug&DEBUGLOGIC) PDEBUG("qc_frame_free()");
}
/* }}} */
/* {{{ [fold] qc_frame_test(struct quickcam *qc) */
/* Return TRUE if next frame is immediately available, FALSE otherwise. */
static inline Bool qc_frame_test(struct quickcam *qc)
{
	struct qc_frame_data *fd = &qc->frame_data;
	IDEBUG_TEST(*fd);
	return fd->head != fd->tail;
}
/* }}} */

/* Producer routines, called from interrupt context */
/* {{{ [fold] qc_frame_begin(struct quickcam *qc) */
/* Begin capturing next frame from camera. If buffer is full, the frame will be lost */
static void qc_frame_begin(struct quickcam *qc)
{
	struct qc_frame_data *fd = &qc->frame_data;
	int framesize, n;
	if (debug&DEBUGLOGIC) PDEBUG("qc_frame_begin(quickcam=%p)",qc);
	if (PARANOID && (qc==NULL || fd==NULL)) { PRINTK(KERN_CRIT,"qc||fd==NULL"); return; }
	IDEBUG_TEST(*fd);
	if (fd->exiting) return;
	if (PARANOID && !fd->rawdatabuf) {
		PRINTK(KERN_CRIT,"rawdatabuf==NULL (1)!");
		return;
	}

	n = fd->head;
	fd->buffers[n].rawdatalen = 0;

	/* Use sensor information to get the framesize (i.e. how much we expect to receive bytes per image) */
	framesize = qc->sensor_data.width * qc->sensor_data.height;
	fd->maxrawdatalen = MIN(framesize, FRAME_DATASIZE);
}
/* }}} */
/* {{{ [fold] qc_frame_add(struct quickcam *qc, unsigned char *data, int datalen) */
/* Store more data for a frame, return nonzero if too much data or other error */
static int qc_frame_add(struct quickcam *qc, unsigned char *data, int datalen)
{
	struct qc_frame_data *fd = &qc->frame_data;
	int n = fd->head;
	int bytes;

	if (PARANOID && (qc==NULL || fd==NULL)) { PRINTK(KERN_CRIT,"qc||fd==NULL"); return -EINVAL; }
	IDEBUG_TEST(*fd);
	if (PARANOID && !fd->rawdatabuf) {
		PRINTK(KERN_CRIT,"rawdatabuf==NULL (2)!");
		return -EINVAL;
	}
	if (fd->maxrawdatalen <= fd->buffers[n].rawdatalen) {
		if (debug&DEBUGERRORS) PDEBUG("maxrawdatalen=%i   rawdatalen=%i  datalen=%i",fd->maxrawdatalen,fd->buffers[n].rawdatalen, datalen);
		return -EBUSY;
	}
	bytes = MIN(datalen, fd->maxrawdatalen - fd->buffers[n].rawdatalen);
	memcpy(fd->rawdatabuf + n*FRAME_DATASIZE + fd->buffers[n].rawdatalen, data, bytes);
	fd->buffers[n].rawdatalen += bytes;
	if (bytes < datalen) {
		if (debug&DEBUGERRORS) PRINTK(KERN_ERR,"out of buffer space by %i", datalen-bytes);
		return -ENOSPC;
	}
	return 0;
}
/* }}} */
/* {{{ [fold] qc_frame_end(struct quickcam *qc) */
/* Finished capturing most recent frame from camera */
/* (may be premature end, in which case some data is missing) */
static void qc_frame_end(struct quickcam *qc)
{
	static const int minrawdatalen = 256;	/* If frame length is less than this many bytes, discard it */
	struct qc_frame_data *fd = &qc->frame_data;
	unsigned int n = fd->head;
	if (PARANOID && (qc==NULL || fd==NULL)) { PRINTK(KERN_CRIT,"qc||fd==NULL"); return; }
	if (debug&DEBUGLOGIC) PDEBUG("qc_frame_end(quickcam=%p,n=%i), got %i bytes",qc,n,fd->buffers[n].rawdatalen);
	IDEBUG_TEST(*fd);
	fd->maxrawdatalen = 0;			/* Stop frame data capturing */
	if (fd->buffers[n].rawdatalen < minrawdatalen) {
		/* No enough data in buffer, don't advance index */
		if (debug&DEBUGERRORS) PDEBUG("Discarding frame with only %u bytes", fd->buffers[n].rawdatalen);
		return;
	}
	n = (n + 1) % FRAME_BUFFERS;		/* Select next frame buffer to fill */
	if (fd->tail != n) {
		fd->head = n;
	} else {
		PRINTK(KERN_NOTICE,"frame lost");
	}
	wake_up(&fd->wq);
}
/* }}} */
/* {{{ [fold] qc_frame_flush(struct quickcam *qc)  */
/* Reject the current data already captured into buffer and end frame */
void qc_frame_flush(struct quickcam *qc)
{
	struct qc_frame_data *fd = &qc->frame_data;
	unsigned int n = fd->head;
	if (PARANOID && (qc==NULL || fd==NULL)) { PRINTK(KERN_CRIT,"qc||fd==NULL"); return; }
	if (debug&DEBUGLOGIC) PDEBUG("qc_frame_flush(quickcam=%p), flush %i bytes",qc,fd->buffers[n].rawdatalen);
	IDEBUG_TEST(*fd);
	fd->buffers[n].rawdatalen = 0;		/* Empty buffer */
	fd->maxrawdatalen = 0;			/* Stop frame data capturing */
}
/* }}} */

/* }}} */
/* {{{ [fold] **** qc_stream: USB datastream processing functions *************** */

/* {{{ [fold] qc_stream_init(struct quickcam *qc) */
/* Initialize datastream processing */
static int qc_stream_init(struct quickcam *qc)
{
	if (debug&DEBUGLOGIC || debug&DEBUGINIT) PDEBUG("qc_stream_init(quickcam=%p)",qc);
	qc->stream_data.capturing = FALSE;
	qc->stream_data.frameskip = frameskip;
	IDEBUG_INIT(qc->stream_data);
	return 0;
}
/* }}} */
/* {{{ [fold] qc_stream_exit(struct quickcam *qc) */
/* Stop datastream processing, after this qc_stream_add should not be called */
static void qc_stream_exit(struct quickcam *qc)
{
	if (debug&DEBUGLOGIC || debug&DEBUGINIT) PDEBUG("qc_stream_exit(quickcam=%p)",qc);
	if (qc->stream_data.capturing)
		qc_frame_end(qc);
	IDEBUG_EXIT(qc->stream_data);
}
/* }}} */
/* {{{ [fold] qc_stream_error(struct quickcam *qc) */
/* This is called when there are data lost due to errors in the stream */
static void qc_stream_error(struct quickcam *qc)
{
	/* Skip rest of data for this frame */
	if (debug&DEBUGERRORS) PDEBUG("qc_stream_error(qc=%p)", qc);
	if (qc->stream_data.capturing)
		qc_frame_end(qc);
	IDEBUG_EXIT(qc->stream_data);
	qc_stream_init(qc);
}
/* }}} */
/* {{{ [fold] qc_stream_add(struct quickcam *qc, unsigned char *data, int datalen) */
/*
 * Analyse an USB packet of the data stream and store it appropriately.
 * Each packet contains an integral number of chunks. Each chunk has
 * 2-bytes identification, followed by 2-bytes that describe the chunk
 * length. Known/guessed chunk identifications are:
 * 8001/8005/C001/C005 - Begin new frame
 * 8002/8006/C002/C006 - End frame
 * 0200/4200           - Contains actual image data, bayer or compressed
 * 0005                - 11 bytes of unknown data
 * 0100                - 2 bytes of unknown data
 * The 0005 and 0100 chunks seem to appear only in compressed stream.
 * Return the amount of image data received or negative value on error.
 */
static int qc_stream_add(struct quickcam *qc, unsigned char *data, int datalen)
{
	struct qc_stream_data *sd = &qc->stream_data;
	int id, len, error, totaldata = 0;
	
	IDEBUG_TEST(*sd);
	while (datalen) {
		if (datalen < 4) {
			if (debug&DEBUGBITSTREAM) PRINTK(KERN_ERR,"missing chunk header");
			break;
		}
		id  = (data[0]<<8) | data[1];
		len = (data[2]<<8) | data[3];
		data    += 4;
		datalen -= 4;
		if (datalen < len) {
			if (debug&DEBUGBITSTREAM) PRINTK(KERN_ERR,"missing chunk contents");
			break;
		}
		switch (id) {
		case 0x8001:
		case 0x8005:
		case 0xC001:
		case 0xC005:
			/* Begin new frame, len should be zero */
			if (PARANOID && len!=0) PDEBUG("New frame: len!=0");
			if (sd->capturing) {
				if (debug&DEBUGBITSTREAM) PDEBUG("Missing frame end mark in stream");
				qc_frame_end(qc);
			}
			sd->capturing = TRUE;
			if (--sd->frameskip < 0) sd->frameskip = qc->settings.frameskip;
			if (sd->frameskip==0) qc_frame_begin(qc);
			break;
		case 0x8002:
		case 0x8006:
		case 0xC002:
		case 0xC006:
			/* End frame, len should be zero */
			if (PARANOID && len!=0) PDEBUG("End frame: len!=0");
			if (sd->capturing) {
				if (sd->frameskip==0) qc_frame_end(qc);
			} else {
				if (debug&DEBUGBITSTREAM) PDEBUG("Missing frame begin mark in stream");
			}
			sd->capturing = FALSE;
			break;
		case 0x0200:
		case 0x4200:
			/* Image data */
			if (!sd->capturing && (debug&DEBUGBITSTREAM)) PDEBUG("Chunk of data outside frames!");
			if (sd->capturing && sd->frameskip==0) {
				error = qc_frame_add(qc, data, len);
			} else {
				error = 0;
			}
			if (error) {
				/* If qc_frame_add returns error, there is more data than the frame may have,
				 * in which case we assume stream is corrupted and skip rest packet */
				if (debug&DEBUGERRORS) PDEBUG("qc_frame_add error %i",error);
			} else {
				totaldata += len;
			}
			break;
		default:
			/* Unknown chunk */
			#ifndef NDEBUG
			if (debug&DEBUGBITSTREAM) {
				static char dump[4*1024];
				char *dump_p = dump;
				int i;
				for (i=0; i<len && (3*i+9)<sizeof(dump); i++) dump_p+=sprintf(dump_p, "%02X ", data[i]);
				PDEBUG("Unknown chunk %04X: %s", id, dump);
			}
			#endif
		}
		data    += len;
		datalen -= len;
	}
	return totaldata;
}
/* }}} */

/* }}} */
/* {{{ [fold] **** qc_isoc:   Isochronous USB transfer related routines ********* */

/*
 * On my system (Acer Travelmate 332T, usb-ohci) there happens frequently
 * errors. Most common are:
 * -18	EXDEV	(even inside individual frames)
 * -84	EILSEQ
 * -71	EPROTO
 * -110 ETIMEDOUT
 * -75  EOVERFLOW ??
 */
/* {{{ [fold] qc_isoc_handler(struct urb *urb) */

/* This is URB completion handler and is called in interrupt context */
static void qc_isoc_handler(struct urb *urb)
{
	struct quickcam *qc;
	int payload = 0;	/* Amount of all data camera sent */
	int totaldata = 0;	/* Amount of image data camera sent */
	int i;

	if (debug&DEBUGINTERRUPTS) PDEBUG("[INTR] qc_isoc_handler(urb=%p)",urb);
	if (PARANOID && !urb) {
		PRINTK(KERN_CRIT,"urb=NULL in qc_isoc_handler!!!");
		return;
	}
	qc = urb->context;

	/* These should never happen */
	if (PARANOID && !qc) {
		PRINTK(KERN_CRIT,"qc=NULL in qc_isoc_handler!!!");
		return;
	}
	IDEBUG_TEST(qc->isoc_data);
	if (PARANOID && !qc->isoc_data.streaming) {
		PDEBUG("aieee! interrupt when it should not happen");
		return;
	}

	if (!qc->dev) return;	/* Camera was disconnected */

	if (urb->status) {
		qc->isoc_data.errorcount++;
		switch (urb->status) {
		case -EXDEV:		/* Partially completed, look at individual frame status */
			break;
		default:
			if (debug&DEBUGERRORS) PRINTK(KERN_ERR,"Unhandled isoc URB error %i",urb->status);
			/* Seen here: -EOVERFLOW (75): Value too large for defined data type */
		case -EPROTO:		/* Bitstuff error or unknown USB error */
		case -EILSEQ:		/* CRC mismatch */
		case -ETIMEDOUT:	/* Transfer timed out */
		case -EREMOTEIO:	/* Short packet detected */
		case -EPIPE:		/* Babble detect or endpoint stalled */
			goto resubmit;
		case -ENOENT:		/* URB was unlinked */
		case -ENODEV:		/* Device was removed */
			return;
		}
	}
	
	for (i=0; i<urb->number_of_packets; i++) {
		if (urb->iso_frame_desc[i].status<0) {
			if (debug&DEBUGERRORS) PDEBUG("USB transfer error %i", urb->iso_frame_desc[i].status);
			qc->isoc_data.errorcount++;
			qc_stream_error(qc);
			continue;
		}
		qc->isoc_data.errorcount = 0;
		payload += urb->iso_frame_desc[i].actual_length;
		totaldata += qc_stream_add(qc, urb->transfer_buffer + urb->iso_frame_desc[i].offset,
			urb->iso_frame_desc[i].actual_length);
	}
	if (debug&DEBUGBITSTREAM) PDEBUG("payload=%i  totaldata=%i",payload,totaldata);
	if (debug&(DEBUGBITSTREAM|DEBUGERRORS)) if (payload==0) PDEBUG("USB interrupt, but no data received!");
resubmit:
	/* Resubmit URB */
	if (qc->isoc_data.errorcount < ISOC_PACKETS*ISOC_URBS*2) {
		urb->dev = qc->dev;			/* Required for 2.4.x */
		i = usb_submit_urb(urb);
		if (i) PDEBUG("failed to resubmit URB, code=%i",i);
	} else {
		PDEBUG("Too many errors, giving up");
	}
}

/* }}} */
/* {{{ [fold] qc_isoc_start(struct quickcam *qc) */
/*
 * Start USB isochronous image transfer from camera to computer
 * (Set USB camera interface and allocate URBs and submit them)
 * Sensor must be initialized beforehand (qc_init_sensor)
 */
static int qc_isoc_start(struct quickcam *qc)
{
	struct qc_isoc_data *id = &qc->isoc_data;
	int ret = -ENOMEM;		/* Return value on error */
	struct urb *urb;
	int i, b;

	if (debug&DEBUGLOGIC || debug&DEBUGINIT) PDEBUG("qc_isoc_start(qc=%p)",qc);
	if (PARANOID && (qc==NULL || id==NULL)) { PRINTK(KERN_CRIT,"qc||id==NULL"); return -EINVAL; }
	IDEBUG_TEST(*id);

	if (id->streaming) return 0;		/* Already started */
	id->streaming = TRUE;
	id->errorcount = 0;

	/* Allocate transfer buffer */
	id->buffer = kmalloc(ISOC_URBS * ISOC_PACKETS * ISOC_PACKET_SIZE, GFP_KERNEL);
	if (!id->buffer) {
		PDEBUG("Out of memory allocating id->buffer");
		goto fail1;
	}

	/* Allocate URBs, fill them, and put them in the URB array */
	for (b=0; b<ISOC_URBS; b++) {
		urb = id->urbs[b] = usb_alloc_urb(ISOC_PACKETS);	/* Zeroes the allocated data up to iso_frame_desc[], *not* including the last! */
		if (!urb) {
			PDEBUG("Out of memory allocating urbs");
			goto fail2;
		}
		urb->dev                    = qc->dev;
		urb->context                = qc;
		urb->pipe                   = usb_rcvisocpipe(qc->dev, QUICKCAM_ISOPIPE);
		urb->transfer_flags         = USB_ISO_ASAP;
		urb->complete               = qc_isoc_handler;
		urb->number_of_packets      = ISOC_PACKETS;
		urb->transfer_buffer        = id->buffer;
		urb->transfer_buffer_length = ISOC_URBS * ISOC_PACKETS * ISOC_PACKET_SIZE;
		for (i=0; i<ISOC_PACKETS; i++) {
			urb->iso_frame_desc[i].offset = b*ISOC_PACKETS*ISOC_PACKET_SIZE + i*ISOC_PACKET_SIZE;
			urb->iso_frame_desc[i].length = ISOC_PACKET_SIZE;
		}
	}

	/* Alternate interface 3 is the biggest frame size */
	/* JFC use 1: but do not know why */
	/* Quickcam Web: Interface 0, alternate 1, endpoint 0x81 -tuukkat */
	qc_i2c_wait(qc);			/* There must not be control URBs going when calling set_interface() */
	ret = usb_set_interface(qc->dev, qc->iface, 1);
	if (ret<0) {
		PDEBUG("set_interface failed");
		goto fail3;
	}

	/* Submit URBs */
	for (b=0; b<ISOC_URBS; b++) {
		ret = usb_submit_urb(id->urbs[b]);
		if (ret<0) {
			PDEBUG("submit urbs failed");
			goto fail4;
		}
	}

	/* Tell camera to start sending data */
	ret = qc->sensor_data.sensor->start(qc);	/* Start current frame */
	if (ret<0) {
		PDEBUG("sensor_data.start failed");
		goto fail5;
	}
	ret = qc_stv_set(qc, STV_ISO_ENABLE, 1);	/* Start isochronous streaming */
	if (ret<0) {
		PDEBUG("qc_stv_set() failed");
		goto fail6;
	}
	return 0;

	/* Cleanup and return error code on failure */
fail6:	qc->sensor_data.sensor->stop(qc);		/* stop current frame. */
fail5:	b = ISOC_URBS;
fail4:	while (--b >= 0) usb_unlink_urb(id->urbs[b]);
	usb_set_interface(qc->dev, qc->iface, 0);	/* Set packet size to 0 (Interface 0, alternate 0, endpoint 0x81 -tuukkat) */
fail3:	b = ISOC_URBS;
fail2:	while (--b >= 0) usb_free_urb(id->urbs[b]);
	kfree(id->buffer);
fail1:	if (debug&DEBUGLOGIC || debug&DEBUGERRORS) PDEBUG("failed qc_isoc_init()=%i",ret);
	return ret;
}
/* }}} */
/* {{{ [fold] qc_isoc_stop(struct quickcam *qc) */
/*
 * Stop USB isochronous image transfer from camera to computer
 * (Tell camera to stop sending images, set idle USB interface and free URBs)
 */
static void qc_isoc_stop(struct quickcam *qc)
{
	struct qc_isoc_data *id = &qc->isoc_data;
	int b, r;

	if (debug&DEBUGLOGIC || debug&DEBUGINIT) PDEBUG("qc_isoc_stop(quickcam=%p)",qc);
	if (PARANOID && (qc==NULL || id==NULL)) { PRINTK(KERN_CRIT,"qc||id==NULL"); return; }
	IDEBUG_TEST(*id);

	if (!id->streaming) return;					/* Already stopped */
	if (qc->dev) {
		if ((r=qc_stv_set(qc, STV_ISO_ENABLE, 0))<0)		/* stop ISO-streaming. */
			PRINTK(KERN_ERR,"qc_stv_set error %i",r);
		if ((r=qc->sensor_data.sensor->stop(qc))<0)		/* stop current frame. */
			PRINTK(KERN_ERR,"sensor_data.stop error %i",r);
		qc_i2c_wait(qc);					/* When calling set_interface(), there must not be control URBs on way */
		if (usb_set_interface(qc->dev, qc->iface, 0) < 0)	/* Set packet size to 0 (Interface 0, alternate 0, endpoint 0x81 -tuukkat) */
			PRINTK(KERN_ERR,"usb_set_interface error");
	}
	for (b=0; b<ISOC_URBS; b++)					/* Unschedule all of the iso td's */
		usb_unlink_urb(id->urbs[b]);
	id->streaming = FALSE;
	for (b=0; b<ISOC_URBS; b++)
		usb_free_urb(id->urbs[b]);

	kfree(id->buffer);
	if (PARANOID) id->buffer=NULL;
	return;
}
/* }}} */
/* {{{ [fold] qc_isoc_init(struct quickcam *qc) */
/*
 * Initialize isochronous streaming functions
 */
static int qc_isoc_init(struct quickcam *qc)
{
	struct qc_isoc_data *id = &qc->isoc_data;
	if (debug&DEBUGLOGIC || debug&DEBUGINIT) PDEBUG("qc_isoc_init(quickcam=%p)",qc);
	if (PARANOID && (qc==NULL || id==NULL)) { PRINTK(KERN_CRIT,"qc||id==NULL"); return -1; }
	IDEBUG_INIT(*id);
	id->streaming = FALSE;
	return 0;
}
/* }}} */
/* {{{ [fold] qc_isoc_exit(struct quickcam *qc) */
/*
 * Uninitialize isochronous streaming functions
 */
static inline void qc_isoc_exit(struct quickcam *qc)
{
	qc_isoc_stop(qc);
	IDEBUG_EXIT(qc->isoc_data);
}
/* }}} */

/* }}} */
/* {{{ [fold] **** qc_sensor: Common routines for all sensors ******************* */

/* {{{ [fold] qc_sensor_setsize(struct quickcam *qc, unsigned int width, unsigned int height) */
/* Called when the application requests a specific image size. Should set the
 * actual delivered image size to as close to the requested as possible.
 * The image size, as delivered from the camera, can be also set to reduce
 * required bandwidth, if possible, but it is not necessary.
 */
static int qc_sensor_setsize(struct quickcam *qc, unsigned int width, unsigned int height)
{
	int r;
	if (debug&DEBUGLOGIC) PDEBUG("qc_sensor_setsize(qc=%p,width=%i,height=%i)",qc,width,height);
	if (PARANOID && qc==NULL) { PDEBUG("qc==NULL!"); return -EINVAL; }

	if (width < min_framewidth) {
		width = min_framewidth;
	} else if (width > qc->sensor_data.maxwidth) {
		width = qc->sensor_data.maxwidth;
	}
	if (height < min_frameheight) {
		height = min_frameheight;
	} else if (height > qc->sensor_data.maxheight) {
		height = qc->sensor_data.maxheight;
	}
	/* Applications require, when using Xvideo extension, that the
	 * frame size is multiple of 8. This is a bug in apps or Xvideo. -tuukkat */
	if (qc->settings.compat_8x) {
		width  = (width /8)*8;
		height = (height/8)*8;
	}
#if COMPRESS
	/* Currently we can't crop image after decompression, so size is fixed */
	if (qc->sensor_data.compress) {
		width  = 320;
		height = 240;
	}
#endif
	/* Set the size only if changed */
	if (qc->vwin.width==width && qc->vwin.height==height) return 0;

	qc->sensor_data.width = width;		/* The sensor-specific code may modify these if not suitable */
	qc->sensor_data.height = height;
	if ((r = qc->sensor_data.sensor->set_size(qc, width, height))<0) {
		PDEBUG("set_size sensor failed");
		return r;
	}

	/* Set the closest size we can actually deliver to application */
	qc->vwin.width  = width;
	qc->vwin.height = height;
	if ((r = qc_i2c_wait(qc))<0) return r;
	qc_frame_flush(qc);	
	return 0;
}
/* }}} */
/* {{{ [fold] qc_sensor_init(struct quickcam *qc) */
/*
 * Initialise sensor. Initializes all data in qc->sensor which is common to all
 * types of sensors and calls the sensor-specific initialization routine.
 * The Photobit starts the pixel integration immediately after the reset.
 * Note: must call qc_i2c_init() and qc_frame_init() before this function!
 */
static int qc_sensor_init(struct quickcam *qc)
{
	int r;

	if (debug&DEBUGLOGIC || debug&DEBUGINIT) PDEBUG("qc_init_sensor(qc=%p)",qc);
	if (PARANOID && qc==NULL) { PDEBUG("qc==NULL!"); return -EINVAL; }

	qc->sensor_data.width     = -1;
	qc->sensor_data.height    = -1;
	qc->sensor_data.exposure  = -1;
	qc->sensor_data.rgain     = -1;
	qc->sensor_data.ggain     = -1;
	qc->sensor_data.bgain     = -1;
	qc->sensor_data.subsample = qc->settings.subsample;
	qc->sensor_data.compress  = qc->settings.compress;

	if ((r = qc->sensor_data.sensor->init(qc))<0) goto fail;
	if ((r = qc_stv_set(qc, STV_ISO_ENABLE, 0))<0) goto fail;		/* Stop isochronous streaming */
	if ((r = qc->sensor_data.sensor->stop(qc))<0) goto fail;		/* Stop current frame */

	/* Set capture size */
	qc->vwin.width  = 0;			/* Set to illegal value (ensures resetting) */
	qc->vwin.height = 0;
	if ((r = qc_sensor_setsize(qc, qc->sensor_data.maxwidth, qc->sensor_data.maxheight))<0) goto fail;

	/* Set brightness settings */
	if ((r = qc->sensor_data.sensor->set_levels(qc, qc->vpic.brightness, qc->vpic.contrast, qc->vpic.hue, qc->vpic.colour))<0) goto fail;
	if (qc->sensor_data.sensor->set_target!=NULL)
		if ((r = qc->sensor_data.sensor->set_target(qc, qc->vpic.brightness))<0) goto fail;
	return 0;

fail:	PRINTK(KERN_ERR,"sensor initialization failed: %i",r);
	return r;
}
/* }}} */

/* }}} */
/* {{{ [fold] **** qc_capt:   User image capturing functions ******************** */

/* {{{ [fold] qc_capt_get(struct quickcam *qc, unsigned char **frame) */
/* Wait until next image is ready and return the image length in bytes
 * and set "frame" to point to the image. If error happens,
 * return standard Linux negative error number. The image will be in
 * palette and size requested by the user (quickcam->vpic,vwin).
 */
static int qc_capt_get(struct quickcam *qc, unsigned char **frame)
{
	struct qc_capt_data *cd = &qc->capt_data;
	unsigned char *rawdata;		/* Raw data from camera */
	int rawdatalen;
	int retrycount = qc->settings.retryerrors ? 10 : 0;
	int r;

	if (debug&DEBUGLOGIC) PDEBUG("qc_capt_get(quickcam=%p)",qc);
	IDEBUG_TEST(*cd);
	if ((r = qc_isoc_start(qc))<0) goto fail;		/* Start receiving data */

	do {
		r = qc_frame_get(qc, &rawdata);
		if (r >= 0) {
			int midvalue;
			rawdatalen = r;
			r = qc_fmt_convert(qc, rawdata, rawdatalen, cd->frame, MAX_FRAME_SIZE, &midvalue);
			if (qc->settings.adaptive && !qc->sensor_data.sensor->autoexposure && r>=0 && midvalue>=0) {
				int ex, gn;
				qc_adapt(qc, midvalue, qc->vpic.brightness>>8, &ex, &gn);
				qc->sensor_data.sensor->set_levels(qc, ex, gn, qc->vpic.hue, qc->vpic.colour);
			}
			qc_frame_free(qc);
		}
//PDEBUG("settled: %i\n",qc_adapt_hassettled(qc));
		qc_i2c_flush(qc);	/* Send all pending I2C transfers */
		retrycount--;
		if (r>=0) retrycount = 0;
		if (!cd->settled) cd->settled = qc_adapt_hassettled(qc);
		if (retrycount<=0 && cd->settled) break;
		schedule();
	} while (1);
	if (r<0) goto fail;
	if (frame) *frame = cd->frame;
	return r;

fail:	if (debug&(DEBUGERRORS|DEBUGLOGIC)) PDEBUG("failed qc_capt_get()=%i", r);
	return r;
}
/* }}} */
/* {{{ [fold] qc_capt_frameaddr(struct quickcam *qc, unsigned char **frame) */
/* Return size and address of the capture buffer that is suitable for mmapping,
 * Standard Linux errno on error */
static inline int qc_capt_frameaddr(struct quickcam *qc, unsigned char **frame)
{
	IDEBUG_TEST(qc->capt_data);
	if (frame!=NULL) *frame = qc->capt_data.frame;
	return MAX_FRAME_SIZE;
}
/* }}} */
/* {{{ [fold] qc_capt_test(struct quickcam *qc) */
/* Return TRUE if next image is immediately available, FALSE otherwise. */
static inline Bool qc_capt_test(struct quickcam *qc)
{
	IDEBUG_TEST(qc->capt_data);
	return qc_frame_test(qc);
}
/* }}} */
/* {{{ [fold] qc_capt_init(struct quickcam *qc) */
static int qc_capt_init(struct quickcam *qc)
{
	struct qc_capt_data *cd = &qc->capt_data;
	int r;

	if (debug&DEBUGLOGIC || debug&DEBUGINIT) PDEBUG("qc_capt_init(quickcam=%p)",qc);

	cd->settled = !(qc->settings.settle && qc->settings.adaptive);

	/* Allocate memory for the (mmappable) capture buffer */
	cd->frame = rvmalloc(MAX_FRAME_SIZE);
	if (!cd->frame) {
		PRINTK(KERN_ERR, "unable to allocate frame");
		r = -ENOMEM;
		goto fail1;
	}

	/* Initialize submodules */
	if ((r=qc_frame_init(qc))<0) goto fail2;	/* Must be before sensor_init() */
	if ((r=qc_sensor_init(qc))<0) goto fail3;	/* Start the sensor (must be after qc_i2c_init but before qc_adapt_init) */
	if ((r=qc_stream_init(qc))<0) goto fail3;
	if ((r=qc_fmt_init(qc))<0) goto fail4;
	if ((r=qc_isoc_init(qc))<0) goto fail5;
	IDEBUG_INIT(*cd);
	return 0;

fail5:	qc_fmt_exit(qc);
fail4:	qc_stream_exit(qc);
fail3:	qc_frame_exit(qc);
fail2:	rvfree(cd->frame, MAX_FRAME_SIZE);
fail1:	PDEBUG("failed qc_capt_init()=%i",r);
	return r;
}
/* }}} */
/* {{{ [fold] qc_capt_exit(struct quickcam *qc) */
static void qc_capt_exit(struct quickcam *qc)
{
	struct qc_capt_data *cd = &qc->capt_data;
	if (debug&DEBUGLOGIC || debug&DEBUGINIT) PDEBUG("qc_capt_exit(quickcam=%p)",qc);
	qc_isoc_exit(qc);
	qc_fmt_exit(qc);
	qc_stream_exit(qc);
	qc_frame_exit(qc);
	rvfree(cd->frame, MAX_FRAME_SIZE);
	if (PARANOID) cd->frame = NULL;
	IDEBUG_EXIT(*cd);
}
/* }}} */

/* }}} */
/* {{{ [fold] **** qc_v4l:    Start of Video 4 Linux API ************************ */

/* {{{ [fold] qc_v4l_poll(struct video_device *dev, struct file *file, poll_table *wait) */
static unsigned int qc_v4l_poll(struct video_device *dev, struct file *file, poll_table *wait)
{
	if (debug&DEBUGUSER) PDEBUG("qc_v4l_poll(dev=%p,file=%p,wait=%p)",dev,file,wait);
	return -EINVAL;
}
/* }}} */
/* {{{ [fold] qc_v4l_init(struct quickcam *qc) */
/* Called when the device is opened */
static int qc_v4l_init(struct quickcam *qc)
{
	int r, fps;

	if (!qc->settings.keepsettings) {
		/* Reset brightness settings */
		qc->vpic.brightness = 32767;
		qc->vpic.hue        = 32767;
		qc->vpic.colour     = 32767;
		qc->vpic.contrast   = 32767;
		qc_adapt_reset(qc);				/* qc_adapt_init() is called from qc_usb_init() */
	}
	qc->vpic.palette = VIDEO_PALETTE_RGB24;
	qc->vpic.depth   = qc_fmt_getdepth(qc->vpic.palette);

	fps = qc->settings.subsample ? 30 : 8;	/* May actually vary depending on image size */
	fps = qc->settings.compress ? 15 : fps;	/* Actually 7.5 fps, but we must round it */
	qc->vwin.flags = fps << 16;		/* Bits 22..16 contain framerate in Philips driver. We do the same. */

	if ((r = qc_capt_init(qc))<0) goto fail;
	return 0;

fail:	if (debug&DEBUGLOGIC || debug&DEBUGERRORS) PDEBUG("failed qc_v4l_init()=%i",r);
	return r;
}
/* }}} */
/* {{{ [fold] qc_v4l_open(struct video_device *dev, int flags) */
static int qc_v4l_open(struct video_device *dev, int flags)
{
	struct quickcam *qc = (struct quickcam *)dev->priv;
	int r;

	if (debug&DEBUGLOGIC || debug&DEBUGUSER) PDEBUG("qc_v4l_open(qc=%p), users=%i",qc,qc->users);
	if (PARANOID && qc==NULL) { PRINTK(KERN_CRIT,"qc==NULL"); return -EINVAL; }
	if (down_interruptible(&qc->lock)) return -EINTR;
	if (qc->dev == NULL) {
		r = -ENODEV;
		goto fail1;
	}
	qc->users++;
	if (qc->users == 1) {
		if (debug&DEBUGLOGIC) PDEBUG("First user, initializing");
		if ((r = qc_v4l_init(qc))<0) goto fail2;
	}
	MOD_INC_USE_COUNT;
	up(&qc->lock);
	return 0;

fail2:	qc->users--;
fail1:	up(&qc->lock);
	if (debug&DEBUGLOGIC || debug&DEBUGERRORS) PDEBUG("failed qc_v4l_open()=%i",r);
	return r;
}
/* }}} */
/* {{{ [fold] qc_v4l_exit(struct quickcam *qc) */
/* Release all resources allocated at qc_v4l_init() */
static inline void qc_v4l_exit(struct quickcam *qc)
{
	if (debug&DEBUGLOGIC || debug&DEBUGINIT) PDEBUG("qc_v4l_cleanup(%p)", qc);
	qc_capt_exit(qc);
}
/* }}} */
/* {{{ [fold] qc_v4l_close(struct video_device *dev) */
static void qc_v4l_close(struct video_device *dev)
{
	struct quickcam *qc = (struct quickcam *)dev->priv;
	if (debug&DEBUGLOGIC || debug&DEBUGUSER) PDEBUG("qc_v4l_close(dev=%p,qc=%p)",dev,qc);
	if (PARANOID && qc==NULL) { PRINTK(KERN_CRIT,"qc==NULL"); return; }
	if (debug&DEBUGMUTEX) PDEBUG("down() in qc_v4l_close()");
	down(&qc->lock);		/* Can not interrupt, we must success */
	qc->users--;
	if (qc->users == 0) {
		/* No more users, device is deallocated */
		qc_v4l_exit(qc);
		if (qc->dev == NULL) {
			/* Camera was unplugged and freeing was postponed: free resources now here */
			if (debug&DEBUGLOGIC) PDEBUG("Performing postponed free");
			qc_usb_exit(qc);
			qc = NULL;
		}
	}
	if (qc) {
		if (debug&DEBUGMUTEX) PDEBUG("up() in qc_v4l_close()");
		up(&qc->lock);
	}
	MOD_DEC_USE_COUNT;
	if (debug&DEBUGLOGIC) PDEBUG("v4l_close() ok");
}
/* }}} */
/* {{{ [fold] qc_v4l_read(struct video_device *dev, char *buf, unsigned long count, int noblock) */
static long qc_v4l_read(struct video_device *dev, char *buf, unsigned long count, int noblock)
{
	struct quickcam *qc = (struct quickcam *)dev->priv;
	int frame_len;
	unsigned char *frame;
	long r = 0;

	if (debug&DEBUGLOGIC || debug&DEBUGUSER)
		PDEBUG("qc_v4l_read(dev,%p,buf=%p,count=%li,noblock=%i,qc=%p)",dev,buf,count,noblock,qc);
	if (!qc || !buf) {
		PDEBUG("qc_read: no video_device available or no buffer attached :( EFAULT");
		return -EFAULT;
	}
	if (down_interruptible(&qc->lock)) return -EINTR;
	if (!qc->dev) {
		r = -ENODEV;
		goto fail;
	}
	if (noblock && !qc_capt_test(qc)) {
		r = -EAGAIN;
		goto fail;
	}
	frame_len = qc_capt_get(qc, &frame);
	if (frame_len < 0) {
		r = frame_len;
		goto fail;
	}
	if (count > frame_len) count = frame_len;
	if (copy_to_user(buf, frame, count)) {
		r = -EFAULT;
		goto fail;
	}
	r = count;

fail:	up(&qc->lock);
	if (debug&DEBUGLOGIC || debug&DEBUGERRORS) if (r<0) PDEBUG("failed qc_v4l_read()=%i", (int)r);
	return r;
}
/* }}} */
/* {{{ [fold] qc_v4l_mmap(struct vm_area_struct *vma, struct video_device *dev, const char *adr, unsigned long size) */
#if HAVE_VMA
static int qc_v4l_mmap(struct vm_area_struct *vma, struct video_device *dev, const char *adr, unsigned long size)
#else
static int qc_v4l_mmap(struct video_device *dev, const char *adr, unsigned long size)
#endif
{
	struct quickcam *qc = (struct quickcam *)dev->priv;
	unsigned long start = (unsigned long)adr;
	unsigned long page, pos;
	unsigned char *frame;
	int frame_size, ret = 0;

	if (debug&DEBUGLOGIC || debug&DEBUGUSER) PDEBUG("qc_v4l_mmap(dev=%p,adr=%p,size=%li,qc=%p)",dev,adr,size,qc);
	if (down_interruptible(&qc->lock)) return -EINTR;
	if (!qc->dev) { ret = -ENODEV; goto fail; }
	frame_size = qc_capt_frameaddr(qc, &frame);
	if (frame_size<0) { ret = frame_size; goto fail; }		/* Should never happen */
	if (size > ((frame_size + PAGE_SIZE - 1) & ~(PAGE_SIZE - 1))) return -EINVAL;

	pos = (unsigned long)frame;
	while (size > 0) {
		page = kvirt_to_pa(pos);
		if (qc_remap_page_range(vma, start, page, PAGE_SIZE, PAGE_SHARED)<0) {
			ret = -EAGAIN;
			goto fail;
		}
		start += PAGE_SIZE;
		pos += PAGE_SIZE;
		if (size > PAGE_SIZE)
			size -= PAGE_SIZE;
		else
			size = 0;
	}

fail:	up(&qc->lock);
	if (ret<0) if (debug&DEBUGLOGIC || debug&DEBUGERRORS) PDEBUG("failed qc_v4l_mmap()=%i",ret);
	return ret;
}
/* }}} */
/* {{{ [fold] qc_v4l_ioctl(struct video_device *dev, unsigned int cmd, void *arg) */
static int qc_v4l_ioctl(struct video_device *dev, unsigned int cmd, void *arg)
{
	struct quickcam *qc = (struct quickcam *)dev->priv;
	int i, retval = 0;

	if (debug&DEBUGLOGIC || debug&DEBUGUSER) PDEBUG("qc_v4l_ioctl(dev=%p,cmd=%u,arg=%p,qc=%p)",dev,cmd,arg,qc);
	if (!qc->dev) return -ENODEV;
	if (down_interruptible(&qc->lock)) return -EINTR;
	switch (cmd) {
/* {{{ [fold] VIDIOCGCAP:     Capability query */
		case VIDIOCGCAP:	/* Capability query */
		{
			struct video_capability b;
			if (debug&DEBUGUSER) PDEBUG("VIDIOCGCAP");
			memset(&b, 0, sizeof(b));
			strcpy(b.name, "Logitech USB Camera");
			b.type      = qc->vdev.type;
			b.channels  = 1;
			b.audios    = 0;
			b.maxwidth  = qc->sensor_data.maxwidth;
			b.maxheight = qc->sensor_data.maxheight;
			if (qc->settings.compat_8x) {
				b.maxwidth  = (b.maxwidth /8)*8;
				b.maxheight = (b.maxheight/8)*8;
			}
			b.minwidth  = min_framewidth;
			b.minheight = min_frameheight;
			if (copy_to_user(arg, &b, sizeof(b))) retval = -EFAULT;
			break;
		}
/* }}} */
/* {{{ [fold] VIDIOCGCHAN:    Get properties of the specified channel */
		case VIDIOCGCHAN:	/* Get properties of the specified channel */
		{
			struct video_channel v;
			if (copy_from_user(&v, arg, sizeof(v))) {
				retval = -EFAULT;
				break;
			}
			if (debug&DEBUGUSER) PDEBUG("VIDIOCGCHAN channel:%i",v.channel);
			if (v.channel != 0) {
				retval = -EINVAL;
				break;
			}
			v.flags = 0;
			v.tuners = 0;
			v.type = VIDEO_TYPE_CAMERA;
			strcpy(v.name, "Camera");
			if (copy_to_user(arg, &v, sizeof(v))) retval = -EFAULT;
			break;
		}
/* }}} */
/* {{{ [fold] VIDIOCSCHAN:    Select channel to capture */
		case VIDIOCSCHAN:	/* Select channel to capture */
		{
			if (copy_from_user(&i, arg, sizeof(i))) {
				retval = -EFAULT;
				break;
			}
			if (debug&DEBUGUSER) PDEBUG("VIDIOCSCHAN channel:%i",i);
			if (i != 0) retval = -EINVAL;
			break;
		}
/* }}} */
/* {{{ [fold] VIDIOCGPICT:    Get image properties */
		case VIDIOCGPICT:	/* Get image properties */
		{
			if (debug&DEBUGUSER) PDEBUG("VIDIOCGPICT");
			if (copy_to_user(arg, &qc->vpic, sizeof(qc->vpic))) retval = -EFAULT;
			break;
		}
/* }}} */
/* {{{ [fold] VIDIOCSPICT:    Set image properties */
		case VIDIOCSPICT:	/* Set image properties */
		{
			struct video_picture p;
			if (copy_from_user(&p, arg, sizeof(p))) {
				retval = -EFAULT;
				break;
			}
			if (debug&DEBUGUSER) PDEBUG("VIDIOCSPICT depth:%d palette:%s bright=%i",p.depth,qc_fmt_getname(p.palette),p.brightness);

			if (p.palette != 0) {		/* 0 = do not change palette */
				retval = qc_fmt_issupported(p.palette);
				if (retval) break;
				qc->vpic.palette = p.palette;
				qc->vpic.depth   = qc_fmt_getdepth(p.palette);
				if (qc->vpic.depth != p.depth) PDEBUG("warning: palette depth mismatch");
			}

			qc->vpic.brightness = p.brightness;
			qc->vpic.hue        = p.hue;
			qc->vpic.colour     = p.colour;
			qc->vpic.contrast   = p.contrast;

			if (!qc->settings.adaptive) {
				/* Set new values now */
				if ((retval = qc->sensor_data.sensor->set_levels(qc, qc->vpic.brightness, qc->vpic.contrast, qc->vpic.hue, qc->vpic.colour))<0) break;
			} else {
				if (qc->sensor_data.sensor->set_target!=NULL)
					if ((retval = qc->sensor_data.sensor->set_target(qc, qc->vpic.brightness))<0) break;
			}
			break;
		}
/* }}} */
/* {{{ [fold] VIDIOCSWIN:     Set capture area width and height */
		case VIDIOCSWIN:	/* Set capture area width and height */
		{
			struct video_window vw;
			int fps;
			if (copy_from_user(&vw, arg, sizeof(vw))) {
				retval = -EFAULT;
				break;
			}
			fps = (vw.flags>>16) & 0x3F;	//FIXME:are there 6 or 7 bits for framerate?
			if (fps && ((qc->vwin.flags>>16)&0x3F)!=fps) {
				PDEBUG("Application tries to change framerate");
			}
			if (debug&DEBUGUSER) PDEBUG("VIDIOCSWIN width:%i height:%i flags:%d clipcount:%d",vw.width,vw.height,vw.flags,vw.clipcount);
			retval = qc_sensor_setsize(qc, vw.width, vw.height);
			break;
		}
/* }}} */
/* {{{ [fold] VIDIOCGWIN:     Get current capture area */
		case VIDIOCGWIN:	/* Get current capture area */
		{
			if (debug&DEBUGUSER) PDEBUG("VIDIOCGWIN");
			if (copy_to_user(arg, &qc->vwin, sizeof(qc->vwin))) retval = -EFAULT;
			break;
		}
/* }}} */
/* {{{ [fold] VIDIOCGMBUF:    Get mmap buffer size and frame offsets */
		case VIDIOCGMBUF:	/* Get mmap buffer size and frame offsets */
		{
			struct video_mbuf vm;
			if (debug&DEBUGUSER) PDEBUG("VIDIOCGMBUF");
			memset(&vm, 0, sizeof(vm));
			vm.size = qc_capt_frameaddr(qc, NULL);
			if (vm.size<0) {	/* Negative value denotes error */
				retval = vm.size;
				break;
			}
			vm.frames = 1;
			vm.offsets[0] = 0;
			if (qc->settings.compat_dblbuf) {
				/* Really many applications are broken and don't work with a single buffer */
				vm.frames = 2;
				vm.offsets[1] = 0;
			}
			if (copy_to_user(arg, &vm, sizeof(vm))) retval = -EFAULT;
			break;
		}
/* }}} */
/* {{{ [fold] VIDIOCMCAPTURE: Start capturing specified frame in the mmap buffer with specified size */
		case VIDIOCMCAPTURE:	/* Start capturing specified frame in the mmap buffer with specified size */
		{
			struct video_mmap vm;
			if (copy_from_user(&vm, arg, sizeof(vm))) {
				retval = -EFAULT;
				break;
			}
			/* Bug in V4L: sometimes it's called palette, sometimes format. We'll stick with palette */
			if (debug&DEBUGUSER) PDEBUG("VIDIOCMCAPTURE frame:%d size:%dx%d palette:%s", vm.frame, vm.width, vm.height, qc_fmt_getname(vm.format));
			if (vm.frame!=0 && !(qc->settings.compat_dblbuf)) {
				retval = -EINVAL;
				break;
			}
			if (vm.format!=0 && qc->vpic.palette!=vm.format) {	/* 0 = do not change palette */
				retval = qc_fmt_issupported(vm.format);
				if (retval) break;
				qc->vpic.palette = vm.format;
				qc->vpic.depth   = qc_fmt_getdepth(vm.format);
			}
			retval = qc_sensor_setsize(qc, vm.width, vm.height);
			break;
		}
/* }}} */
/* {{{ [fold] VIDIOCSYNC:     Wait until specified frame in the mmap buffer has been captured */
		case VIDIOCSYNC:	/* Wait until specified frame in the mmap buffer has been captured */
		{
			if (debug&DEBUGUSER) PDEBUG("VIDIOCSYNC");
			retval = qc_capt_get(qc, NULL);
			if (retval>0) retval = 0;
			break;
		}
/* }}} */
/* {{{ [fold] VIDIOCGFBUF:    Get currently used frame buffer parameters */
		case VIDIOCGFBUF:	/* Get currently used frame buffer parameters */
		{
			struct video_buffer vb;
			if (debug&DEBUGUSER) PDEBUG("VIDIOCGFBUF");
			memset(&vb, 0, sizeof(vb));
			if (copy_to_user(arg, &vb, sizeof(vb))) retval = -EFAULT;
			break;
		}
/* }}} */
/* {{{ [fold] VIDIOCKEY:      Undocumented? */
		case VIDIOCKEY:		/* Undocumented? */
			if (debug&DEBUGUSER) PDEBUG("VIDIOCKEY");
			retval = -EINVAL;
			break;
/* }}} */
/* {{{ [fold] VIDIOCCAPTURE:  Activate overlay capturing directly to framebuffer */
		case VIDIOCCAPTURE:	/* Activate overlay capturing directly to framebuffer */
			if (debug&DEBUGUSER) PDEBUG("VIDIOCCAPTURE");
			retval = -EINVAL;
			break;
/* }}} */
/* {{{ [fold] VIDIOCSFBUF:    Set frame buffer parameters for the capture card */
		case VIDIOCSFBUF:	/* Set frame buffer parameters for the capture card */
			if (debug&DEBUGUSER) PDEBUG("VIDIOCSFBUF");
			retval = -EINVAL;
			break;
/* }}} */
/* {{{ [fold] VIDIOCxTUNER:   Get properties of the specified tuner / Select tuner to use */
		case VIDIOCGTUNER:	/* Get properties of the specified tuner */
		case VIDIOCSTUNER:	/* Select tuner to use */
			if (debug&DEBUGUSER) PDEBUG("VIDIOCxTUNER");
			retval = -EINVAL;
			break;
/* }}} */
/* {{{ [fold] VIDIOCxFREQ:    Get current tuner frequency / Set tuner frequency */
		case VIDIOCGFREQ:	/* Get current tuner frequency */
		case VIDIOCSFREQ:	/* Set tuner frequency */
			if (debug&DEBUGUSER) PDEBUG("VIDIOCxFREQ");
			retval = -EINVAL;
			break;
/* }}} */
/* {{{ [fold] VIDIOCxAUDIO:   Get/Set audio properties */
		case VIDIOCGAUDIO:	/* Get audio properties */
		case VIDIOCSAUDIO:	/* Set audio properties */
			if (debug&DEBUGUSER) PDEBUG("VIDIOCxAUDIO");
			retval = -EINVAL;
			break;
/* }}} */
		/********** Private IOCTLs ***********/
/* {{{ [fold] VIDIOCQCxDEBUG:        Sets/gets the debug output (1,2,4,8,16,32) */
		case VIDIOCQCSDEBUG:		/* Sets the debug output (1,2,4,8,16,32) */
			if (get_user(debug, (int *)arg)) { retval=-EFAULT; break; }
		case VIDIOCQCGDEBUG:		/* Gets the debug output (1,2,4,8,16,32) */
			if (put_user(debug, (int *)arg)) { retval=-EFAULT; break; }
			if (debug&DEBUGUSER) PDEBUG("VIDIOCxDEBUG");
			break;
/* }}} */
/* {{{ [fold] VIDIOCQCxKEEPSETTINGS: Set/get keep gain settings across one open to another (0-1) */
		case VIDIOCQCSKEEPSETTINGS:	/* Set keep gain settings across one open to another (0-1) */
			if (get_user(i, (int *)arg)) { retval=-EFAULT; break; }
			qc->settings.keepsettings = i;
		case VIDIOCQCGKEEPSETTINGS:	/* Get keep gain settings across one open to another (0-1) */
			i = qc->settings.keepsettings;
			if (put_user(i, (int *)arg)) { retval=-EFAULT; break; }
			if (debug&DEBUGUSER) PDEBUG("VIDIOCxKEEPSETTINGS");
			break;
/* }}} */
/* {{{ [fold] VIDIOCQCxSETTLE:       Set/get if we let image brightness to settle (0-1) */
		case VIDIOCQCSSETTLE:		/* Set if we let image brightness to settle (0-1) */
			if (get_user(i, (int *)arg)) { retval=-EFAULT; break; }
			qc->settings.settle = i;
		case VIDIOCQCGSETTLE:		/* Get if we let image brightness to settle (0-1) */
			i = qc->settings.settle;
			if (put_user(i, (int *)arg)) { retval=-EFAULT; break; }
			if (debug&DEBUGUSER) PDEBUG("VIDIOCxSETTLE");
			break;
/* }}} */
/* {{{ [fold] VIDIOCQCxSUBSAMPLE:    Sets/gets the speed (0-1) */
		case VIDIOCQCSSUBSAMPLE:	/* Sets the speed (0-1) */
			if (get_user(i, (int *)arg)) { retval=-EFAULT; break; }
			qc->settings.subsample = i;
		case VIDIOCQCGSUBSAMPLE:	/* Gets the speed (0-1) */
			i = qc->settings.subsample;
			if (put_user(i, (int *)arg)) { retval=-EFAULT; break; }
			if (debug&DEBUGUSER) PDEBUG("VIDIOCxSUBSAMPLE");
			break;
/* }}} */
/* {{{ [fold] VIDIOCQCxCOMPRESS:     Sets/gets the compression mode (0-1) */
		case VIDIOCQCSCOMPRESS:		/* Sets the compression mode (0-1) */
			if (get_user(i, (int *)arg)) { retval=-EFAULT; break; }
			qc->settings.compress = i;
		case VIDIOCQCGCOMPRESS:		/* Gets the compression mode (0-1) */
			i = qc->settings.compress;
			if (put_user(i, (int *)arg)) { retval=-EFAULT; break; }
			if (debug&DEBUGUSER) PDEBUG("VIDIOCxCOMPRESS");
			break;
/* }}} */
/* {{{ [fold] VIDIOCQCxFRAMESKIP:    Set/get frame capture frequency (0-10) */
		case VIDIOCQCSFRAMESKIP:	/* Set frame capture frequency (0-10) */
			if (get_user(i, (int *)arg)) { retval=-EFAULT; break; }
			qc->settings.frameskip = i;
		case VIDIOCQCGFRAMESKIP:	/* Get frame capture frequency (0-10) */
			i = qc->settings.frameskip;
			if (put_user(i, (int *)arg)) { retval=-EFAULT; break; }
			if (debug&DEBUGUSER) PDEBUG("VIDIOCxFRAMESKIP");
			break;
/* }}} */
/* {{{ [fold] VIDIOCQCxQUALITY:      Sets/gets the interpolation mode (0-2) */
		case VIDIOCQCSQUALITY:		/* Sets the interpolation mode (0-2) */
			if (get_user(i, (int *)arg)) { retval=-EFAULT; break; }
			qc->settings.quality = i;
		case VIDIOCQCGQUALITY:		/* Gets the interpolation mode (0-2) */
			i = qc->settings.quality;
			if (put_user(i, (int *)arg)) { retval=-EFAULT; break; }
			if (debug&DEBUGUSER) PDEBUG("VIDIOCxQUALITY");
			break;
/* }}} */
/* {{{ [fold] VIDIOCQCxADAPTIVE:     Set/get automatic adaptive brightness control (0-1) */
		case VIDIOCQCSADAPTIVE:		/* Set automatic adaptive brightness control (0-1) */
			if (get_user(i, (int *)arg)) { retval=-EFAULT; break; }
			qc->settings.adaptive = i;
		case VIDIOCQCGADAPTIVE:		/* Get automatic adaptive brightness control (0-1) */
			i = qc->settings.adaptive;
			if (put_user(i, (int *)arg)) { retval=-EFAULT; break; }
			if (debug&DEBUGUSER) PDEBUG("VIDIOCxADAPTIVE");
			break;
/* }}} */
/* {{{ [fold] VIDIOCQCxEQUALIZE:     Set/get equalize image (0-1) */
		case VIDIOCQCSEQUALIZE:		/* Set equalize image (0-1) */
			if (get_user(i, (int *)arg)) { retval=-EFAULT; break; }
			qc->settings.equalize = i;
		case VIDIOCQCGEQUALIZE:		/* Get equalize image (0-1) */
			i = qc->settings.equalize;
			if (put_user(i, (int *)arg)) { retval=-EFAULT; break; }
			if (debug&DEBUGUSER) PDEBUG("VIDIOCxEQUALIZE");
			break;
/* }}} */
/* {{{ [fold] VIDIOCQCxRETRYERRORS:  Set/get if we retry when error happen in capture (0-1) */
		case VIDIOCQCSRETRYERRORS:	/* Set if we retry when error happen in capture (0-1) */
			if (get_user(i, (int *)arg)) { retval=-EFAULT; break; }
			qc->settings.retryerrors = i;
		case VIDIOCQCGRETRYERRORS:	/* Get if we retry when error happen in capture (0-1) */
			i = qc->settings.retryerrors;
			if (put_user(i, (int *)arg)) { retval=-EFAULT; break; }
			if (debug&DEBUGUSER) PDEBUG("VIDIOCxRETRYERRORS");
			break;
/* }}} */
/* {{{ [fold] VIDIOCQCxCOMPATIBLE:   Set enable workaround for Xawtv/Motv bugs (0-1) */
		case VIDIOCQCSCOMPATIBLE:	/* Set enable workaround for Xawtv/Motv bugs (0-1) */
			if (get_user(i, (int *)arg)) { retval=-EFAULT; break; }
			qc->settings.compat_8x     = (i & COMPAT_8X)     != 0;
			qc->settings.compat_dblbuf = (i & COMPAT_DBLBUF) != 0;
			qc->settings.compat_torgb  = (i & COMPAT_TORGB)  != 0;
		case VIDIOCQCGCOMPATIBLE:	/* Get enable workaround for Xawtv/Motv bugs (0-1) */
			i  = ~(qc->settings.compat_8x    -1) & COMPAT_8X;
			i |= ~(qc->settings.compat_dblbuf-1) & COMPAT_DBLBUF;
			i |= ~(qc->settings.compat_torgb -1) & COMPAT_TORGB;
			if (put_user(i, (int *)arg)) { retval=-EFAULT; break; }
			if (debug&DEBUGUSER) PDEBUG("VIDIOCxCOMPATIBLE");
			break;
/* }}} */
/* {{{ [fold] VIDIOCQCxVIDEONR:      Set videodevice number (/dev/videoX) */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,5)
		case VIDIOCQCSVIDEONR:		/* Set videodevice number (/dev/videoX) */
			if (debug&DEBUGUSER) PDEBUG("VIDIOCSVIDEONR");
			retval = -EINVAL;	/* Can not set after the module is loaded */
			break;
		case VIDIOCQCGVIDEONR:		/* Get videodevice number (/dev/videoX) */
			if (debug&DEBUGUSER) PDEBUG("VIDIOCGVIDEONR");
			if (put_user(video_nr, (int *)arg)) { retval=-EFAULT; break; }
			break;
#endif
/* }}} */
/* {{{ [fold] VIDIOCQCxSTV:          Read/write STV chip register value */
		/* Encoding: bits 31..16 of the int argument contain register value, 15..0 the reg number */
		case VIDIOCQCGSTV:		/* Read STV chip register value */
		{
			int reg, val;
			if (debug&DEBUGUSER) PDEBUG("VIDIOCQCGSTV");
			if (get_user(reg, (int *)arg)) { retval=-EFAULT; break; }
			reg &= 0xFFFF;
			val = qc_stv_get(qc, reg);
			if (val<0) { retval=val; break; }
			val = (val<<16) | reg;
			if (put_user(val, (int *)arg)) { retval=-EFAULT; break; }
			break;
		}
		case VIDIOCQCSSTV:		/* Write STV chip register value */
		{
			int regval;
			if (debug&DEBUGUSER) PDEBUG("VIDIOCQCSSTV");
			if (get_user(regval, (int *)arg)) { retval=-EFAULT; break; }
			retval = qc_stv_set(qc, regval & 0xFFFF, regval >> 16);
			break;
		}
/* }}} */
/* {{{ [fold] VIDIOCQCxI2C:          Read/write sensor chip register value via I2C */
		case VIDIOCQCGI2C:		/* Read sensor chip register value via I2C */
		{
			int reg, val;
			if (debug&DEBUGUSER) PDEBUG("VIDIOCQCGI2C");
			if (get_user(reg, (int *)arg)) { retval=-EFAULT; break; }
			reg &= 0xFFFF;
			val = qc_get_i2c(qc, qc->sensor_data.sensor, reg);
			if (val<0) { retval=val; break; }
			val = (val<<16) | reg;
			if (put_user(val, (int *)arg)) { retval=-EFAULT; break; }
			break;
		}
		case VIDIOCQCSI2C:		/* Write sensor chip register value via I2C */
		{
			int regval;
			if (debug&DEBUGUSER) PDEBUG("VIDIOCQCSI2C");
			if (get_user(regval, (int *)arg)) { retval=-EFAULT; break; }
			retval = qc_i2c_set(qc, regval & 0xFFFF, regval >> 16);
			if (retval<0) break;
			retval = qc_i2c_wait(qc);
			break;
		}
/* }}} */
		default:
			if (debug&DEBUGUSER) PDEBUG("Unknown IOCTL %08X",cmd);
			retval = -ENOIOCTLCMD;
			break;
	}
	up(&qc->lock);
	if (retval<0) if (debug&(DEBUGLOGIC|DEBUGUSER|DEBUGERRORS)) PDEBUG("failed qc_v4l_ioctl()=%i",retval);
	return retval;
}
/* }}} */
/* {{{ [fold] qc_v4l_write(struct video_device *dev, const char *buf, unsigned long count, int noblock) */
static long qc_v4l_write(struct video_device *dev, const char *buf, unsigned long count, int noblock)
{
	if (debug&DEBUGLOGIC || debug&DEBUGUSER) PDEBUG("qc_v4l_write()");
	return -EINVAL;
}
/* }}} */

static struct video_device qc_v4l_template = {
	name:		"QuickCam USB",
	type:		VID_TYPE_CAPTURE | VID_TYPE_SUBCAPTURE,
	hardware:	VID_HARDWARE_QCAM_USB,
	initialize:     NULL,
	open:		qc_v4l_open,
	close:		qc_v4l_close,
	read:		qc_v4l_read,
	write:		qc_v4l_write,
	ioctl:		qc_v4l_ioctl,
	mmap:		qc_v4l_mmap,
	poll:		qc_v4l_poll,
	minor:		0
};
/* }}} */
/* {{{ [fold] **** qc_usb:    Start of USB API ********************************** */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
static void *qc_usb_probe(struct usb_device *dev, unsigned int iface, const struct usb_device_id *id);
#else
static void *qc_usb_probe(struct usb_device *dev, unsigned int iface);
#endif
static void qc_usb_disconnect(struct usb_device *dev, void *ptr);

static struct usb_driver qc_usb_driver = {
	name:		qc_name,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
	id_table:	qc_device_table,
#endif
	probe:		qc_usb_probe,
	disconnect:	qc_usb_disconnect,
};

/* {{{ [fold] qc_usb_init(struct usb_device *dev, unsigned int iface) */
/* Detect sensor, initialize the quickcam structure, register V4L device, create /proc entry */
/* Return pointer to the allocated quickcam structure or NULL on error */
/* If there is some camera already open but disconnected, reuse the quickcam structure */
static struct quickcam *qc_usb_init(struct usb_device *dev, unsigned int iface)
{
	struct quickcam *qc;
	Bool reuse_qc;
	int i, r = 0;

	if (debug&DEBUGLOGIC) PDEBUG("qc_usb_init(dev=%p)", dev);
	if (PARANOID && dev==NULL) { PRINTK(KERN_CRIT,"dev==NULL"); return NULL; }

	/* Check if there is already a suitable quickcam struct that can be reused */
	reuse_qc = FALSE;
	list_for_each_entry(qc, &quickcam_list, list) {
		if (qc->dev==NULL) {
			/* Found unplugged camera */
			reuse_qc = TRUE;
			break;
		}
	}

	if (reuse_qc) {
		/* Reuse existing quickcam (which is already opened) */
		if (debug&DEBUGLOGIC) PDEBUG("Reusing existing quickcam");
		if (PARANOID && qc->users<=0) PRINTK(KERN_CRIT, "Unplugged unused camera detected!\n");
		if (debug&DEBUGMUTEX) PDEBUG("down() in qc_usb_init(), reusing");
		down(&qc->lock);			/* Can not interrupt, we must success */
		qc_isoc_stop(qc);
		qc_i2c_wait(qc);
		qc_frame_flush(qc);
	} else {
		/* Allocate and initialize some members of the new qc */
		if (debug&DEBUGLOGIC) PDEBUG("Allocating new quickcam");
		qc = kmalloc(sizeof(*qc), GFP_KERNEL);
		if (qc == NULL) {
			PRINTK(KERN_ERR,"couldn't kmalloc quickcam struct");
			goto fail1;
		}
		if (PARANOID) memset(qc, 0, sizeof(*qc));
		init_MUTEX(&qc->lock);
		if (debug&DEBUGMUTEX) PDEBUG("down() in qc_usb_init(), not reusing");
		down(&qc->lock);			/* Can not interrupt, we must success */
		qc->users = 0;
		if ((r=qc_i2c_init(qc))<0) goto fail2;
	}
	qc->dev   = dev;
	qc->iface = iface;

	/* Probe for the sensor type */
	qc_i2c_wait(qc);						/* Necessary before set_interface() */
	if ((r=usb_set_interface(dev, qc->iface, 0))<0) goto fail3;	/* Set altsetting 0 */
	if ((r=qc_stv_set(qc, STV_ISO_ENABLE, 0))<0) goto fail3;	/* Disable isochronous stream */
	for (i=0; i<SIZE(sensors); i++) {
		if ((r = qc_get_i2c(qc, sensors[i], sensors[i]->id_reg))<0) goto fail3;
		r = (r >> (sensors[i]->length_id-1) * 8) & 0xFF;	/* Get MSB of received value */
		if (debug&DEBUGCAMERA) PDEBUG("Probing %s: expecting %02X, got %02X", sensors[i]->name, sensors[i]->id, r);
		if (r == sensors[i]->id) break;
	}
	if (i>=SIZE(sensors)) {
		PRINTK(KERN_INFO,"unsupported sensor");
		goto fail3;
	}
	qc->sensor_data.sensor = sensors[i];
	PRINTK(KERN_INFO,"Sensor %s detected", sensors[i]->name);
	
	if ((r=qc_stv_set(qc, STV_ISO_ENABLE, 0))<0) goto fail3;	/* Disable isochronous streaming */
	if ((r=qc_stv_set(qc, STV_REG23, 1))<0) goto fail3;

	if (!reuse_qc) {
		/* Set default settings */
		qc->vpic.brightness = 32767;
		qc->vpic.hue        = 32767;
		qc->vpic.colour     = 32767;
		qc->vpic.contrast   = 32767;
		qc->vpic.whiteness  = 0;		/* Not used */
		qc->settings.keepsettings  = keepsettings;
		qc->settings.settle        = settle;
		qc->settings.subsample     = subsample;
		qc->settings.compress      = compress;
		qc->settings.frameskip     = frameskip;
		qc->settings.quality       = quality;
		qc->settings.adaptive      = adaptive;
		qc->settings.equalize      = equalize;
		qc->settings.retryerrors   = retryerrors;
		qc->settings.compat_8x     = compatible & COMPAT_8X ? 1 : 0;
		qc->settings.compat_dblbuf = compatible & COMPAT_DBLBUF ? 1 : 0;
		qc->settings.compat_torgb  = compatible & COMPAT_TORGB ? 1 : 0;

		/* Register V4L video device */
		memcpy(&qc->vdev, &qc_v4l_template, sizeof(qc_v4l_template));
		qc->vdev.priv = qc;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,5)
		r = video_register_device(&qc->vdev, VFL_TYPE_GRABBER, video_nr);
#else
		r = video_register_device(&qc->vdev, VFL_TYPE_GRABBER);
#endif
		if (r<0) goto fail3;
		if ((r=qc_adapt_init(qc))<0) goto fail4;
		if ((r=qc_proc_create(qc))<0) goto fail5;		/* Create /proc entry */
		list_add(&qc->list, &quickcam_list);
	}

	if (reuse_qc && qc->frame_data.waiting>0) {
		/* Restart capturing */
		int width = qc->vwin.width;
		int height = qc->vwin.height;
		qc_sensor_init(qc);
		qc_sensor_setsize(qc, width, height);
		qc_isoc_start(qc);
	}
	if (debug&DEBUGMUTEX) PDEBUG("up() in qc_usb_init()");
	up(&qc->lock);
	return qc;

fail5:	qc_adapt_exit(qc);
fail4:	video_unregister_device(&qc->vdev);
fail3:	if (!reuse_qc) qc_i2c_exit(qc);
	qc->dev = NULL;
	up(&qc->lock);
fail2:	if (!reuse_qc) kfree(qc);
fail1:	if (debug&DEBUGLOGIC || debug&DEBUGERRORS) PDEBUG("failed qc_usb_init()=%i",r);
	return NULL;
}
/* }}} */
/* FIXME: can usb_disconnect and usb_probe pre-empt other kernel mode processes? Assume no */
/* {{{ [fold] qc_usb_probe(struct usb_device *dev, unsigned int iface, const struct usb_device_id *id) */
/* Called when any USB device is connected, check if it is a supported camera */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
static void *qc_usb_probe(struct usb_device *dev, unsigned int iface, const struct usb_device_id *id)
#else
static void *qc_usb_probe(struct usb_device *dev, unsigned int iface)
#endif
{
	struct usb_interface_descriptor *interface;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
	/* Check if the device has a product number that we support */
	struct usb_device_id *i;
        for (i=qc_device_table; i->idVendor; i++) {
		if (dev->descriptor.idVendor == i->idVendor &&
		    dev->descriptor.idProduct == i->idProduct) break;
	}
	if (!i->idVendor) return NULL;
#endif
	if (debug&DEBUGLOGIC || debug&DEBUGCAMERA) PDEBUG("qc_usb_probe(dev=%p,iface=%i)", dev, iface);
	if (PARANOID && dev==NULL) { PRINTK(KERN_CRIT,"dev==NULL"); return NULL; }
	
	/* We don't handle multi-config cameras */
	if (dev->descriptor.bNumConfigurations != 1) return NULL;

	/*
         * Checking vendor/product is not enough
         * In case on Quickcam Web the audio is at class 1 and subclass 1/2.
         * one /dev/dsp and one /dev/mixer
         */
	interface = &dev->actconfig->interface[iface].altsetting[0];
	if (PARANOID && interface->bInterfaceNumber!=iface)
		PRINTK(KERN_CRIT,"bInterfaceNumber(%i)!=iface(%i)!!",interface->bInterfaceNumber,iface);
	if (interface->bInterfaceClass != 0xFF) return NULL;
	if (interface->bInterfaceSubClass != 0xFF) return NULL;

	/* We found a Quickcam */
	PRINTK(KERN_INFO,"QuickCam USB camera found (driver version %s)", VERSION);

	if (debug&DEBUGCAMERA) PDEBUG("USB Quickcam Class %02X SubClass %02X idVendor %04X idProduct %04X",
		interface->bInterfaceClass, interface->bInterfaceSubClass,
		dev->descriptor.idVendor, dev->descriptor.idProduct);

	/* The interface is claimed (bound) automatically to us when we return non-NULL from this function */
	return qc_usb_init(dev, iface);
}
/* }}} */
/* {{{ [fold] qc_usb_exit(struct quickcam *qc) */
/* Free up resources allocated in qc_usb_init() when not needed anymore */
static void qc_usb_exit(struct quickcam *qc)
{
	if (debug&DEBUGLOGIC || debug&DEBUGINIT) PDEBUG("qc_usb_exit(qc=%p)",qc);
	if (PARANOID && qc==NULL) { PRINTK(KERN_CRIT,"qc==NULL"); return; }

	if (qc->dev != NULL) {
		/* FIXME: Is this necessary? */
		usb_driver_release_interface(&qc_usb_driver, &qc->dev->actconfig->interface[qc->iface]);
	}
	qc_proc_destroy(qc);
	qc_adapt_exit(qc);
	if (debug&DEBUGLOGIC) PDEBUG("video_unregister_device(%p)", &qc->vdev);
	video_unregister_device(&qc->vdev);
	qc_i2c_exit(qc);
	list_del(&qc->list);
	if (PARANOID) memset(qc, 0, sizeof(*qc));
	kfree(qc);
	if (debug&DEBUGLOGIC) PDEBUG("qc_usb_exit() done");
}
/* }}} */
/* {{{ [fold] qc_usb_disconnect(struct usb_device *dev, void *ptr) */
/* Called when the camera is disconnected. We may not free struct quickcam here,
 * because the camera might be in use (open() called). In that case, the freeing is
 * postponed to the last close() call. However, all submitted URBs must be unlinked.
 */
static void qc_usb_disconnect(struct usb_device *dev, void *ptr)
{
	struct quickcam *qc = (struct quickcam *)ptr;

	if (debug&DEBUGLOGIC || debug&DEBUGCAMERA) PDEBUG("qc_usb_disconnect(qc=%p)",qc);
	if (PARANOID && !qc) { PDEBUG("qc==NULL in qc_usb_disconnect!"); return; }
	if (PARANOID && qc->dev==NULL) { PDEBUG("disconnecting disconnected device!!");return;}
	if (PARANOID && dev!=qc->dev) { PDEBUG("disconnecting not our device!!"); return; }

	if (debug&DEBUGMUTEX) PDEBUG("down() in qc_usb_disconnect()");
	down(&qc->lock);		/* Can not interrupt, we must success */
	/*
	 * When the camera is unplugged (maybe even when it is capturing), quickcam->dev is set to NULL.
	 * All functions called from user mode and all _exit functions must check for this.
	 */
	if (qc->users <= 0) {
		/* Free resources */
		qc_usb_exit(qc);
	} else {
		/* Can not free resources if device is open: postpone to when it is closed */
		if (debug&DEBUGLOGIC) PDEBUG("Disconnect while device open: postponing cleanup");
		qc->dev = NULL;
		qc_isoc_stop(qc);		/* Unlink and free isochronous URBs */
		qc_i2c_wait(qc);		/* Wait until there are no more I2C packets on way */
		if (debug&DEBUGMUTEX) PDEBUG("up() again in qc_usb_disconnect()");
		up(&qc->lock);
	}
}
/* }}} */

/* }}} */
/* {{{ [fold] **** qc:        Start of module API ******************************* */

/* {{{ [fold] qc_init(void) */
static int __init qc_init(void)
{
	int r;
	if (debug) PDEBUG("----------LOADING QUICKCAM MODULE------------");
	if (debug) PDEBUG("struct quickcam size: %i", sizeof(struct quickcam));
	if (debug&DEBUGLOGIC || debug&DEBUGINIT) PDEBUG("qc_init()");
	r = qc_proc_init();
	if (r>=0) {
		r = usb_register(&qc_usb_driver);
		if (r<0) qc_proc_exit();
	}
	if (r<0) if (debug&DEBUGLOGIC || debug&DEBUGERRORS) PDEBUG("failed qc_init()=%i",r);
	return r;
}
/* }}} */
/* {{{ [fold] qc_exit(void) */
static void __exit qc_exit(void)
{
	if (debug&DEBUGLOGIC || debug&DEBUGINIT) PDEBUG("qc_exit()");
	usb_deregister(&qc_usb_driver);		/* Will also call qc_usb_disconnect() if necessary */
	qc_proc_exit();
}
/* }}} */

module_init(qc_init);
module_exit(qc_exit);
/* }}} */

/* End of file */
