/*
 *  linux/drivers/char/serial_mx1ads.c
 *
 *  Driver for Samsung S3C2410's internal serial ports (UART0 & UART1)
 *
 *  Copyright (C) 2002 Steve Hein, SGI Inc.  (ssh@sgi.com)
 *  Cho, Hyun-ho <firebird@legend.co.kr> corrected baudrate config.
 *
 *  Adapted from:
 *
 *  Driver for MX1s' dual serial ports.
 *
 *  Based on drivers/serial/serial_amba.c
 *   
 *
 *  Copyright 1999 ARM Limited
 *  Copyright (C) 2000 Deep Blue Solutions Ltd.
 *  Copyright (C) 2002 Shane Nay (shane@minirl.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *
 * Based on the AMBA driver, this is a driver for the S3C2410 boards
 * UARTs.  
 */
#include <linux/config.h>
#include <linux/module.h>
#include <linux/errno.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/major.h>
#include <linux/string.h>
#include <linux/fcntl.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/circ_buf.h>
#include <linux/serial.h>
#include <linux/console.h>
#include <linux/sysrq.h>

#include <asm/system.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/uaccess.h>
#include <asm/bitops.h>

#if defined(CONFIG_SERIAL_S3C2410_CONSOLE) && defined(CONFIG_MAGIC_SYSRQ)
#define SUPPORT_SYSRQ
#endif

#include <linux/serial_core.h>

#include <asm/hardware/serial_s3c2410.h>

#define UART_NR		2

#define MINOR_START		64

#define SERIAL_S3C2410_NAME	"ttyS"
#define SERIAL_S3C2410_MAJOR	TTY_MAJOR
#define SERIAL_S3C2410_MINOR	MINOR_START
#define SERIAL_S3C2410_NR	UART_NR

#define CALLOUT_S3C2410_NAME	"cuas"
#define CALLOUT_S3C2410_MAJOR	205
#define CALLOUT_S3C2410_MINOR	MINOR_START
#define CALLOUT_S3C2410_NR	UART_NR


static struct tty_driver normal, callout;
static struct tty_struct *s3c2410_table[UART_NR];
static struct termios *s3c2410_termios[UART_NR], *s3c2410_termios_locked[UART_NR];
#ifdef SUPPORT_SYSRQ
static struct console s3c2410_console;
#endif

#define S3C2410_ISR_PASS_LIMIT	256

/*
 * Access macros for the S3C2410 UARTs
 */

#define UART_GET_CHAR(p)	__raw_readb((p)->membase + S3C2410_UARTRXH0_OFF)
#define UART_PUT_CHAR(p,c)	__raw_writeb((c), (p)->membase + S3C2410_UARTTXH0_OFF)

#define UART_GET_ULCON(p)      	__raw_readl((p)->membase + S3C2410_UARTLCON_OFF)
#define UART_GET_UCON(p)       	__raw_readl((p)->membase + S3C2410_UARTCON_OFF)
#define UART_GET_UFCON(p)      	__raw_readl((p)->membase + S3C2410_UARTFCON_OFF)
#define UART_GET_UMCON(p)      	__raw_readl((p)->membase + S3C2410_UARTMCON_OFF)
#define UART_GET_UBRDIV(p)     	__raw_readl((p)->membase + S3C2410_UARTBRDIV_OFF)

#define UART_GET_UTRSTAT(p)   	__raw_readl((p)->membase + S3C2410_UARTTRSTAT_OFF)
#define UART_GET_UERSTAT(p)    	__raw_readl((p)->membase + S3C2410_UARTERSTAT_OFF)
#define UART_GET_UFSTAT(p)     	__raw_readl((p)->membase + S3C2410_UARTFSTAT_OFF)
#define UART_GET_UMSTAT(p)     	__raw_readl((p)->membase + S3C2410_UARTMSTAT_OFF)

#define UART_PUT_ULCON(p,c)    	__raw_writel(c, (p)->membase + S3C2410_UARTLCON_OFF)
#define UART_PUT_UCON(p,c)     	__raw_writel(c, (p)->membase + S3C2410_UARTCON_OFF)
#define UART_PUT_UFCON(p,c)    	__raw_writel(c, (p)->membase + S3C2410_UARTFCON_OFF)
#define UART_PUT_UMCON(p,c)    	__raw_writel(c, (p)->membase + S3C2410_UARTMCON_OFF)
#define UART_PUT_UBRDIV(p,c)   	__raw_writel(c, (p)->membase + S3C2410_UARTBRDIV_OFF)


/* When using the integer divisor for the UART,
 * we must set the IR to 0xf prior to programming
 * the divisor
 */


#define UART_RX_DATA(s)		(((s) & S3C2410_UTRSTAT_RXDR) == S3C2410_UTRSTAT_RXDR)
#define UART_TX_READY(s)	(((s) & S3C2410_UTRSTAT_TXFE) == S3C2410_UTRSTAT_TXFE)
#define TX_FIFOCOUNT(port)      (((UART_GET_UFSTAT(port))>>4)&0xf)

#define TX_IRQ(port)            (port->irq + 1)
#define RX_IRQ(port)            (port->irq)

#define UART_DUMMY_RSR_RX	256
#define UART_PORT_SIZE		64

/*
 * Our private driver data mappings.
 */

#define drv_old_status	driver_priv

static void
s3c2410uart_stop_tx(struct uart_port *port, u_int from_tty)
{
	disable_irq(TX_IRQ(port));
}

static void
s3c2410uart_start_tx(struct uart_port *port, u_int nonempty, u_int from_tty)
{
	if (nonempty) {
		enable_irq(TX_IRQ(port));
	}
}

static void
s3c2410uart_stop_rx(struct uart_port *port)
{
	disable_irq(RX_IRQ(port));
}

static void
s3c2410uart_enable_ms(struct uart_port *port)
{
	/* What is MS? */
}

static void
#ifdef SUPPORT_SYSRQ
s3c2410uart_rx_chars(struct uart_info *info, struct pt_regs *regs)
#else
s3c2410uart_rx_chars(struct uart_info *info)
#endif
{
	struct tty_struct *tty = info->tty;
	unsigned int status, ch, max_count = 256;
	struct uart_port *port = info->port;
	

	status = UART_GET_UTRSTAT(port);
	while (UART_RX_DATA(status) && max_count--) {
		if (tty->flip.count >= TTY_FLIPBUF_SIZE) {
			tty->flip.tqueue.routine((void *) tty);
			if (tty->flip.count >= TTY_FLIPBUF_SIZE) {
				printk(KERN_WARNING "TTY_DONT_FLIP set\n");
				return;
			}
		}

		ch = UART_GET_CHAR(port);

		*tty->flip.char_buf_ptr = ch;
		*tty->flip.flag_buf_ptr = TTY_NORMAL;
		port->icount.rx++;
		tty->flip.flag_buf_ptr++;
		tty->flip.char_buf_ptr++;
		tty->flip.count++;
		/* No error handling just yet.
		 * On the MX1 these are seperate
		 * IRQs, so we need to deal with
		 * the sanity of 5 IRQs for one
		 * serial port before we deal
		 * with the error path properly.
		 */
		status = UART_GET_UTRSTAT(port);
	}
	tty_flip_buffer_push(tty);
	return;
}

static void
s3c2410uart_tx_chars(struct uart_info *info)
{
	struct uart_port *port = info->port;
	int count;

	if (port->x_char) {
		UART_PUT_CHAR(port, port->x_char);
		port->icount.tx++;
		port->x_char = 0;
		return;
	}
	if (info->xmit.head == info->xmit.tail
	    || info->tty->stopped || info->tty->hw_stopped) {
		s3c2410uart_stop_tx(port, 0);
		return;
	}

	count = port->fifosize - TX_FIFOCOUNT(port);
	do {
		UART_PUT_CHAR(port, info->xmit.buf[info->xmit.tail]);
		info->xmit.tail = (info->xmit.tail + 1) & (UART_XMIT_SIZE - 1);
		port->icount.tx++;
		if (info->xmit.head == info->xmit.tail)
			break;
	} while (--count > 0);

	if (CIRC_CNT(info->xmit.head, info->xmit.tail, UART_XMIT_SIZE) <
	    WAKEUP_CHARS)
		uart_event(info, EVT_WRITE_WAKEUP);

	if (info->xmit.head == info->xmit.tail)
		s3c2410uart_stop_tx(info->port, 0);
}

static void
s3c2410uart_modem_status(struct uart_info *info)
{
	/*
	 * Not currently supported.
	 *
	 */
}

static void
s3c2410uart_int(int irq, void *dev_id, struct pt_regs *regs)
{
	struct uart_info *info = dev_id;
	unsigned int status;
	status = UART_GET_UTRSTAT(info->port);
	if(irq==(TX_IRQ(info->port))) {
		if (UART_TX_READY(status))
			s3c2410uart_tx_chars(info);
	}
	
	if (UART_RX_DATA(status)) {
#ifdef SUPPORT_SYSRQ
		s3c2410uart_rx_chars(info, regs);
#else
		s3c2410uart_rx_chars(info);
#endif
	}
}

static u_int
s3c2410uart_tx_empty(struct uart_port *port)
{
	return UART_GET_UTRSTAT(port) & S3C2410_UTRSTAT_TXFE ? TIOCSER_TEMT : 0;
}

static u_int
s3c2410uart_get_mctrl(struct uart_port *port)
{
	/* Not currently supported */
	return 0;
}

static void
s3c2410uart_set_mctrl(struct uart_port *port, u_int mctrl)
{
	/* Not currently supported */
}

static void
s3c2410uart_break_ctl(struct uart_port *port, int break_state)
{
	unsigned long ucon=UART_GET_UCON(port);
	if (break_state==-1)
	    ucon |= S3C2410_UCON_SBREAK;
	else
	    ucon &= ~S3C2410_UCON_SBREAK;
	UART_PUT_UCON(port,ucon);
}

static int
s3c2410uart_startup(struct uart_port *port, struct uart_info *info)
{

	int retval;
	/*
	 * Allocate the IRQ
	 */
	retval = request_irq(RX_IRQ(port), s3c2410uart_int, 0, "s3c2410uarx", info);
	if (retval)
		return retval;
	retval = request_irq(TX_IRQ(port), s3c2410uart_int, 0, "s3c2410uatx", info);
	if (retval)
		return retval;
	
	UART_PUT_UCON(port, S3C2410_UCON_DEFAULT);
	UART_PUT_UFCON(port, S3C2410_UFCON_DEFAULT);
	UART_PUT_UMCON(port, 0);
       
     /* printk("PORT 0x%08X ULCON 0x%08X UCON 0x%08X UFCON 0x%08X UMCON 0x%08X \n",
	port->membase, UART_GET_ULCON(port), UART_GET_UCON(port),
	UART_GET_UFCON(port), UART_GET_UMCON(port)); 
     */

//	UART_PUT_UBRDIV(port,(int)(PCLK/CURRENT_BAUD_RATE*16))-1);
	return 0;

}

static void
s3c2410uart_shutdown(struct uart_port *port, struct uart_info *info)
{
	/*
	 * Free the interrupt
	 */

	free_irq(RX_IRQ(port), info);
	free_irq(TX_IRQ(port), info);
}

static void
s3c2410uart_change_speed(struct uart_port *port, u_int cflag, u_int iflag,
			u_int quot)
{
	unsigned long flags, lcon;
#if DEBUG
	printk("s3c2410uart_set_cflag(0x%x) called\n", cflag);
#endif
	/* first, disable everything */
	save_flags(flags);
	cli();
	lcon=UART_GET_ULCON(port);
	/* Clear the bits we're going to configure */
	lcon &= ~(S3C2410_LCON_CFGMASK);
	
	switch (cflag & CSIZE) {
	case CS5:
		lcon |= S3C2410_LCON_CS5;
		break;
	case CS6:
		lcon |= S3C2410_LCON_CS6;
		break;
	case CS7:
		lcon |= S3C2410_LCON_CS7;
		break;
	default:
		lcon |= S3C2410_LCON_CS8;
		break;
	}
	if(cflag & CSTOPB) {
	    lcon |= S3C2410_LCON_STOP;
	}
	if(cflag & PARENB) {
		lcon |= (cflag & PARODD)? S3C2410_LCON_PODD:S3C2410_LCON_PEVEN;
	} else {
		lcon |= S3C2410_LCON_PNONE;
	}

	UART_PUT_ULCON(port, lcon);
	if(iflag & BRKINT) {
		unsigned long ucon=UART_GET_UCON(port);
		ucon |= S3C2410_UCON_SBREAK;
		UART_PUT_UCON(port, ucon);
	}

	UART_PUT_UBRDIV(port, quot);
	restore_flags(flags);

}

static const char *
s3c2410uart_type(struct uart_port *port)
{
	return port->type == PORT_S3C2410 ? "S3C2410" : NULL;
}

/*
 * Release the memory region(s) being used by 'port'.
 */
static void
s3c2410uart_release_port(struct uart_port *port)
{
	return;
}

/*
 * Request the memory region(s) being used by 'port'.
 */
static int
s3c2410uart_request_port(struct uart_port *port)
{
	return 0;
}

/*
 * Configure/autoconfigure the port.
 */
static void
s3c2410uart_config_port(struct uart_port *port, int flags)
{
	if (flags & UART_CONFIG_TYPE && s3c2410uart_request_port(port) == 0)
		port->type = PORT_S3C2410;
}

static int
s3c2410uart_verify_port(struct uart_port *port, struct serial_struct *ser)
{
	return 0;
}

static struct uart_ops s3c2410_pops = {
	tx_empty:	s3c2410uart_tx_empty,
	set_mctrl:	s3c2410uart_set_mctrl,
	get_mctrl:	s3c2410uart_get_mctrl,
	stop_tx:	s3c2410uart_stop_tx,
	start_tx:	s3c2410uart_start_tx,
	stop_rx:	s3c2410uart_stop_rx,
	enable_ms:	s3c2410uart_enable_ms,
	break_ctl:	s3c2410uart_break_ctl,
	startup:	s3c2410uart_startup,
	shutdown:	s3c2410uart_shutdown,
	change_speed:	s3c2410uart_change_speed,
	type:		s3c2410uart_type,
	release_port:	s3c2410uart_release_port,
	request_port:	s3c2410uart_request_port,
	config_port:	s3c2410uart_config_port,
	verify_port:	s3c2410uart_verify_port,
};

static struct uart_port s3c2410_ports[UART_NR] = {
	{
	      membase:	(void *) VA_UART_BASE,
	      mapbase:	VA_UART_BASE,
	      iotype:	SERIAL_IO_MEM,
	      irq:	IRQ_RXD0,
	      uartclk:	PCLK,
	      fifosize:	16,
	      unused:	{4, 5},	
	      ops:	&s3c2410_pops,
	      type:	PORT_S3C2410,
	      flags:	ASYNC_BOOT_AUTOCONF,
	 },
	{
	      membase:	(void *) (VA_UART_BASE + S3C2410_UART1_OFF),
	      mapbase:	(VA_UART_BASE + S3C2410_UART1_OFF),
	      iotype:	SERIAL_IO_MEM,
	      irq:	IRQ_RXD1,
	      uartclk:	PCLK,
	      fifosize:	16,
	      unused:	{6, 7},
	      ops:	&s3c2410_pops,
	      type:	PORT_S3C2410,
	      flags:	ASYNC_BOOT_AUTOCONF,
	 }
};

#ifdef CONFIG_SERIAL_S3C2410_CONSOLE

#ifdef used_and_not_const_char_pointer
static int
s3c2410uart_console_read(struct uart_port *port, char *s, u_int count)
{
	unsigned int status;
	int c;

#if DEBUG
	printk("s3c2410uart_console_read() called\n");
#endif

	c = 0;
	while (c < count) {
		status = UART_GET_USR2(port);
		if (UART_RX_DATA(status)) {
			*s++ = UART_GET_CHAR(port);
			c++;
		} else {
			// nothing more to get, return
			return c;
		}
	}
	// return the count
	return c;
}
#endif


static void
s3c2410uart_console_write(struct console *co, const char *s, u_int count)
{
	struct uart_port *port = s3c2410_ports + co->index;
	unsigned int status;
	int i;
	/*
	 *      Now, do each character
	 */
	for (i = 0; i < count; i++) {
		do {
			status = UART_GET_UTRSTAT(port);
		} while (!UART_TX_READY(status));
		UART_PUT_CHAR(port, (s[i]) & 0xff);
		if (s[i] == '\n') {
			do {
				status = UART_GET_UTRSTAT(port);
			} while (!UART_TX_READY(status));
			UART_PUT_CHAR(port, '\r');
		}
	}
}

static kdev_t
s3c2410uart_console_device(struct console *co)
{

	return MKDEV(SERIAL_S3C2410_MAJOR, SERIAL_S3C2410_MINOR + co->index);
}

static int
s3c2410uart_console_wait_key(struct console *co)
{
	struct uart_port *port = s3c2410_ports + co->index;
	unsigned int status;

	do {
		status = UART_GET_UTRSTAT(port);
	} while (!UART_RX_DATA(status));
	return UART_GET_CHAR(port);
}

static void __init
s3c2410uart_console_get_options(struct uart_port *port, int *baud, int *parity,
			       int *bits)
{
#if 0
	*baud = (7833600)/(UART_GET_LCR(port) +1);
#else
	*baud = CURRENT_BAUD_RATE;
	
#endif
	*bits = 8;
	*parity = 'n';
}

static int __init
s3c2410uart_console_setup(struct console *co, char *options)
{
	struct uart_port *port;
	int baud = CURRENT_BAUD_RATE;
	int bits = 8;
	int parity = 'n';
	int flow = 'n';

	/*
	 * Check whether an invalid uart number has been specified, and
	 * if so, search for the first available port that does have
	 * console support.
	 */
	port = uart_get_console(s3c2410_ports, UART_NR, co);

	if (options)
		uart_parse_options(options, &baud, &parity, &bits, &flow);
	else
		s3c2410uart_console_get_options(port, &baud, &parity, &bits);

	return uart_set_options(port, co, baud, parity, bits, flow);
}

static struct console s3c2410_console = {
	name:		SERIAL_S3C2410_NAME,
	write:		s3c2410uart_console_write,
#ifdef used_and_not_const_char_pointer
	read:		s3c2410uart_console_read,
#endif
	device:		s3c2410uart_console_device,
	wait_key:	s3c2410uart_console_wait_key,
	setup:		s3c2410uart_console_setup,
	flags:		CON_PRINTBUFFER,
	index:		-1,
};

void __init
s3c2410uart_console_init(void)
{
	register_console(&s3c2410_console);
}

#define S3C2410_CONSOLE	&s3c2410_console
#else
#define S3C2410_CONSOLE	NULL
#endif

static struct uart_driver s3c2410_reg = {
	owner:		THIS_MODULE,
	normal_major:	SERIAL_S3C2410_MAJOR,
#ifdef CONFIG_DEVFS_FS
	normal_name:	SERIAL_S3C2410_NAME "%d",
	callout_name:	CALLOUT_S3C2410_NAME "%d",
#else
	normal_name:	SERIAL_S3C2410_NAME,
	callout_name:	CALLOUT_S3C2410_NAME,
#endif
	normal_driver	:&normal,
	callout_major:	CALLOUT_S3C2410_MAJOR,
	callout_driver:	&callout,
	table:		s3c2410_table,
	termios:	s3c2410_termios,
	termios_locked:	s3c2410_termios_locked,
	minor:		SERIAL_S3C2410_MINOR,
	nr:		UART_NR,
	port:		s3c2410_ports,
	cons:		S3C2410_CONSOLE,
};

static int __init
s3c2410uart_init(void)
{
	return uart_register_driver(&s3c2410_reg);
}

static void __exit
s3c2410uart_exit(void)
{
	uart_unregister_driver(&s3c2410_reg);
}

module_init(s3c2410uart_init);
module_exit(s3c2410uart_exit);

EXPORT_NO_SYMBOLS;

MODULE_AUTHOR("Steve Hein <ssh@sgi.com>");
MODULE_DESCRIPTION("Samsung S3C2410X01 serial port driver");
MODULE_LICENSE("GPL");
