/*
 * i2c-omap1510.c
 *
 * Simple bit algo I2C test driver for the TI OMAP1510.
 *
 * Copyright (C) 2002 MontaVista Software, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/ioport.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/stddef.h>
#include <linux/parport.h>
#include <linux/rtc.h>

#include <linux/i2c.h>
#include <linux/i2c-algo-bit.h>

#include <asm/hardware.h>
#include <asm/io.h>
#include <asm/system.h>
#include "i2c-omap1510.h"

#undef	DEBUG
#if	defined(DEBUG)
#	define	DPRINTK(fmt, args...)	printk(KERN_DEBUG "%s: " fmt, __FUNCTION__ , ## args)
#else
#	define	DPRINTK(fmt, args...)
#endif


static int iic_acquired;
static int iic_data_out;
static int iic_clock_out;


/*
 * Grab control of the I2C pins
 */
static inline void iic_acquirepins(void)
{
	outw(inw(I2C_SYSTEST) | I2C_SYSTEST_SCL_O, I2C_SYSTEST);
	outw(inw(I2C_SYSTEST) & I2C_SYSTEST_SDA_O, I2C_SYSTEST);
	iic_acquired = 1;
	iic_clock_out = 1;
	iic_data_out = 0;
}

/*
 * Release control of the I2C pins
 */
static inline void iic_releasepins(void)
{
	outw(inw(I2C_SYSTEST) | ( I2C_SYSTEST_SCL_O | I2C_SYSTEST_SDA_O ), I2C_SYSTEST);
	iic_acquired = 0;
}

/*
 * Initialize the interface
 */
static void iic_init(void)
{
	outw(I2C_SYSTEST_ST_EN | I2C_SYSTEST_FREE | (3 << I2C_SYSTEST_TMODE_SHIFT), I2C_SYSTEST);
	iic_releasepins();
}


/* --------- */

static void bit_omap1510_setscl(void *data, int state)
{
	if (state) {
		outw(inw(I2C_SYSTEST) | I2C_SYSTEST_SCL_O, I2C_SYSTEST);
	} else {
		outw(inw(I2C_SYSTEST)  & ~I2C_SYSTEST_SCL_O, I2C_SYSTEST);
	}
}

static void bit_omap1510_setsda(void *data, int state)
{
	if (state) {
		outw(inw(I2C_SYSTEST) | I2C_SYSTEST_SDA_O, I2C_SYSTEST);
	} else {
		outw(inw(I2C_SYSTEST) & ~I2C_SYSTEST_SDA_O, I2C_SYSTEST);
	}
} 

static int bit_omap1510_getscl(void *data)
{
	return (inw(I2C_SYSTEST) & I2C_SYSTEST_SCL_I) ? 1 : 0;
}

static int bit_omap1510_getsda(void *data)
{
	return (inw(I2C_SYSTEST) & I2C_SYSTEST_SDA_I) ? 1 : 0;
}

/* ---------- */

static int bit_omap1510_reg(struct i2c_client *client)
{
	return 0;
}

static int bit_omap1510_unreg(struct i2c_client *client)
{
	return 0;
}

static void bit_omap1510_inc_use(struct i2c_adapter *adap)
{
#ifdef MODULE
	MOD_INC_USE_COUNT;
#endif
}

static void bit_omap1510_dec_use(struct i2c_adapter *adap)
{
#ifdef MODULE
	MOD_DEC_USE_COUNT;
#endif
}

static struct i2c_algo_bit_data bit_omap1510_data = {
	data:		NULL,
	setsda:		bit_omap1510_setsda,
	setscl:		bit_omap1510_setscl,
	getsda:		bit_omap1510_getsda,
	getscl:		bit_omap1510_getscl,
	udelay:		4,
	mdelay:		10,
	timeout:	100,
};

static struct i2c_adapter bit_omap1510_ops = {
	name:			"OMAP1510",
	id:			I2C_HW_B_IOC,
	algo:			NULL,
	algo_data:		&bit_omap1510_data,
	inc_use:		bit_omap1510_inc_use,
	dec_use:		bit_omap1510_dec_use,
	client_register:	bit_omap1510_reg,
	client_unregister:	bit_omap1510_unreg,
};

int __init i2c_omap1510_init(void)
{
	printk("i2c-omap1510.o: i2c OMAP1510 module\n");
	/* reset hardware to sane state */
	iic_init();
	if (i2c_bit_add_bus(&bit_omap1510_ops) < 0) {
		printk("i2c-omap1510: Unable to register with I2C\n");
		return -ENODEV;
	}
	return 0;
}

void __exit i2c_omap1510_exit(void)
{
	i2c_bit_del_bus(&bit_omap1510_ops);
}


#ifdef MODULE
MODULE_AUTHOR("George G. Davis <davis_g@mvista.com>");
MODULE_DESCRIPTION("I2C-Bus bit algo test driver for TI OMAP1510");
#ifdef MODULE_LICENSE
MODULE_LICENSE("GPL");
#endif

int init_module(void)
{
	return i2c_omap1510_init();
}

void cleanup_module(void)
{
        i2c_omap1510_exit();
}
#endif
