/*
 *  linux/arch/arm/mach-omap/av500_irr_fiq.c
 *
 * BRIEF MODULE DESCRIPTION
 *   Decoder for AV500 Infrared Remote Receiver
 *   WARNING: runs from FIQ
 *
 * Copyright (C) 2004 Archos SA.
 * Author:
 *         Matthias Welwarsky <welwarsky@archos.com>
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <linux/config.h>
#include <linux/module.h>
#include <linux/types.h>

#include <asm/io.h>
#include <asm/fiq.h>
#include <asm/irq.h>
#include <asm/system.h>
#include <asm/hardware.h>
#include <asm/arch/system.h>
#include <asm/arch/gpio.h>

// **************************************************************************
// Modifiers
#define IRR_MOD_NORMAL	0
#define IRR_MOD_REPEAT	1

typedef enum { StateIdle=0, StateHeader=1, StateCode=2 } IRRStateValues;
static IRRStateValues IRR_State __sramdata;
static u16 IRR_Nbits __sramdata;
static u32 IRR_IncWord __sramdata;

static u32 IRR_TimeLastBit __sramdata;
static u32 IRR_TimeCurrentBit __sramdata;

#define IRR_CTIMEOUT	170	// 170  = 85.000 ms
#define IRR_RTIMEOUT	1000	// 1000 = 500.000 ms
static u32 IRR_CTimeOut __sramdata;	// Timeout for the reception of a frame
static u32 IRR_RTimeOut __sramdata;	// Timeout for the validity of the last code when receiving a repeat code

#define IRR_GPIO	4

static u32 irr_pin_state __sramdata;

#define VA_DM270_BASE		0xE2000000UL
#define DM270_IRAM_BASE		(VA_DM270_BASE+0x8000)
#define IRRemoteLock		*(volatile unsigned short*)(DM270_IRAM_BASE +256 +528 +126 +2 +8 +20)

static int __sram TimedOut(u32 t, u32 to)
{
	return (s32)(to-t) < 0;
}

u32 av500_irr_irrcode;
u32 av500_irr_keymode;
int av500_irr_key_vld;
static void __sram IRR_ReportKey(int mode)
{
	av500_irr_keymode = mode;
	if (mode != IRR_MOD_REPEAT)
		av500_irr_irrcode = IRR_IncWord;
	av500_irr_key_vld = 1;
}

static void __sram IRR_decode(void)
{
	u32 TimeDiff;

	TimeDiff = IRR_TimeCurrentBit - IRR_TimeLastBit;
	
	if ( IRR_State == StateIdle ) {
		// The first edge of the frame has been received - just remind the time it is now
		IRR_CTimeOut = IRR_TimeCurrentBit + IRR_CTIMEOUT;
		IRR_State = StateHeader;
		IRRemoteLock = 1;
	}
	else if ( IRR_State == StateHeader ) {
		// This may be the second edge i.e. end of the header
		if ( TimeDiff < 20 ) {
			// 20 = 10.0 ms
			// Something was wrong, this is nothing we understand
			IRR_State = StateIdle;
			IRRemoteLock = 0;
		} else
		if ( TimeDiff < 24 ) {
			// 24 = 12.0 ms
			// This was a repeat header
			IRR_State = StateIdle;
			IRR_ReportKey(IRR_MOD_REPEAT);
			IRRemoteLock = 0;
		} else
		if ( TimeDiff < 29 ) {
			// 29 = 14.5 ms
			// This was a normal header
			IRR_State = StateCode;
			IRR_Nbits = 0;
			IRR_IncWord = 0;
		} else {
			// It is a too long pulse. we assume it is a new frame -> restart the operation
		}
	}
	else if ( IRR_State == StateCode ) {
		if ( TimeDiff <= 3 ) {
			// 3 = 1.5 ms
			// We received a 0
			IRR_Nbits ++;
			IRR_IncWord = (IRR_IncWord << 1);
		} else if ( TimeDiff <= 5 ) {
			// 5 = 2.5 ms
			// We received a 1
			IRR_Nbits ++;
			IRR_IncWord = (IRR_IncWord << 1) + 1;
		} else {
			// We received sth incorrect - handle it as a 0
			IRR_Nbits ++;
			IRR_IncWord = (IRR_IncWord << 1);
		}

		if ( TimedOut( IRR_TimeCurrentBit, IRR_CTimeOut ) ) {
			// We reached the timeout without receiving the whole frame -> cancel the sequence
			IRR_State = StateIdle;
			IRRemoteLock = 0;
		}

		if ( IRR_Nbits >= 32 ) {
			// We received a complete frame - let us try to get the key code from the frame
			IRR_ReportKey(IRR_MOD_NORMAL);
			IRR_State = StateIdle;
			IRRemoteLock = 0;
		}
	}

	IRR_TimeLastBit = IRR_TimeCurrentBit;
}

void __sram av500_init_irr_decoder(void)
{
	irr_pin_state = 1;
}

#ifdef CONFIG_AV500_KEEPALIVE
#define MPUIO_OUTPUT_REG     ((volatile unsigned short*)0xfffb5004)
#endif
void __sram av500_fiq_handler(void)
{	
	u32 pin_state = omap_mpuio_get(IRR_GPIO);
#ifdef CONFIG_AV500_KEEPALIVE
	*MPUIO_OUTPUT_REG ^= (1<<7); 
#endif
	/* increment clock ticks so that we can decode the protocol */
	IRR_TimeCurrentBit++;	
	/* sample MPUIO4 pin and return if nothing has changed */
	if ( pin_state == irr_pin_state ) {
		// if nothing changes for a certain time, reset to StateIdle
		if ( IRR_State != StateIdle && TimedOut(IRR_TimeCurrentBit, IRR_CTimeOut) ) {
			IRR_CTimeOut = IRR_TimeCurrentBit + IRR_CTIMEOUT;
			IRR_State = StateIdle;
			IRRemoteLock = 0;
		}
		return;
	
	}
	
	/* save pin state */
	irr_pin_state = pin_state;
	/* if state has changed to LOW, call decoder */
	if (!pin_state)
		IRR_decode();
}

EXPORT_SYMBOL(av500_fiq_handler);
EXPORT_SYMBOL(av500_irr_irrcode);
EXPORT_SYMBOL(av500_irr_keymode);
EXPORT_SYMBOL(av500_irr_key_vld);
EXPORT_SYMBOL(av500_init_irr_decoder);
