/**************************************************************
 * PCMCIA Driver for the SDG Systems MobiliScan
 * Bard Code Reader
 *
 * July 28, 2002
 *  Copyright 2002, 2003 SDG Systems, LLC
 *
 *   This file is part of sdgmhsgpl_cs.
 *
 *   sdgmhsgpl_cs is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   sdgmhsgpl_cs is distributed in the hope that it will be useful, but
 *   WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with sdgmhsgpl_cs; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *   USA
 *
 * This version of the driver is distributed under the terms of the GNU
 * Public License and comes with no support. A full-featured, fully
 * supported version of this driver is distributed under the terms of a
 * different license and provides accompanying software including GUI
 * configuration, audio support for good-scan notification, and optional
 * any-key device triggering. See our website for details:
 * http://sdsystems.com/
 *
 **************************************************************/

/*
 * DATAFLOW:
 *
 *   +-----+   serial  /////////////////////////
 *   | ISR |<----------//   In-hand Scanner   //
 *   +--+--+    rx     /////////////////////////
 *      |                                 ^
 *      v                                 |
 *   +---------------------+              |  serial tx
 *   | enqueue_rx_char_irq |              +--------------------------------+
 *   +--+------------------+                                               |
 *      |(schedule tasklet          ksimtimer  +-----------------------+   |
 *      |  at EOP)             +-------------->| simulate_keyboard_irq |   |
 *      |                      |               +-----------------------+   |
 *      v                      |                                           |
 *   +-------------------------+----+ (N)ACK  +-----------------------+    |
 *   | process_receive_tasklet_func +-------->| enqueue_tx_return_irq +--+ |
 *   +--+---------------------------+  PARAM  +-----------------------+  | |
 *      | DECODE                       REVISION                          | |
 *      |           +----------------------------+                       | |
 *      +---------%>| enqueue_txacknack_irq      +%-----+                | |
 *                  | serial_request_to_send_irq |      |                | |
 *                  +----------------------------+      |                | |
 *   +----------------------------+                     v                | |
 *   | enqueue_transmit_irq       | txtimer    +---------------------+   | |
 *   | serial_request_to_send_irq +----------->| serial_transmit_irq +-----+
 *   +----------------------------+            +---------------------+   |
 *                       ^                                               v
 *   IRQ MODE            |                                            |-----|
 * ######################|######### need spinlocks at this boundary ##|-----|##
 *   TASK MODE           |                                            |-----|
 *                       | (not in irq mode)                          +-----+
 *   +-------+        +--+-----------------------+                       |
 *   | ioctl +<------>+ do_transmit_wait_for_ack |                       |
 *   +-------+        +------------+-------------+                       |
 *       ^               ^         |         ^                           |
 *       |               | timeout |         |                           |
 *   +---+---+           +---------+         +---------------------------+
 *   | user  |
 *   | space |
 *   +-------+
 *
 * A note on naming conventions: A function name that ends in _irq means that
 * that function *can* run in interrupt mode (via a hw interrupt, timer, or
 * tasklet).  It doesn't necessarily mean that it is always called from the
 * interrupt mode.  These functions must take special precautions to *never*
 * sleep or schedule.  If they do, they need to make sure they are not
 * currently in the irq mode.
 *
 * Note the boundary where we need spinlocks.  This is because on an SMP
 * system, we can have one processor (and only one) running the *_irq
 * functions and another processor handling an ioctl, for example.  For
 * non-SMP systems, we only need to halt IRQ while enqueueing and dequeueing,
 * but this is handled automatically by the spinlock macros.
 */

/*
 * Version x.y is encoded in SDGMHSGPL_DRIVER_VERSION as x * 0x100 + y.
 */
#define SDGMHSGPL_DRIVER_VERSION 0x100

/* Linux Includes  (= */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/poll.h>
#include <linux/serial.h>
#include <linux/major.h>
#include <linux/ioport.h>
#include <linux/ioctl.h>
#include <linux/timer.h>
#include <linux/vt_kern.h>
#include <linux/kbd_ll.h>
#include <linux/kbd_kern.h>
#include <linux/sched.h>
#include <linux/spinlock.h>
/* =) */

/* Assembly Include (= */
#include <asm/io.h>
/* =) */

/* PCMCIA Includes (= */
#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/cisreg.h>
#include <pcmcia/cistpl.h>
#include <pcmcia/ciscode.h>
#include <asm/io.h>
#include <pcmcia/ds.h>
#include <pcmcia/cisreg.h>
/* =) */

#include "sdgmhsgpl_cs.h"

#define MSG_DEBUG(X) /* printk X */

/* Parameters (= */
#define DRIVER_NAME "sdgmhsgpl_cs"
#define MAX_DEVICES 2
#define BANNER "SDG Systems GPL MobiliScan HS Driver\n" \
               "Copyright 2002-2003, SDG Systems, LLC.\n"
/* =) */

/* Module stuff (= */
MODULE_AUTHOR("SDG Systems, LLC.");
MODULE_DESCRIPTION("Driver for the SDG Systems MobiliScan HS");
/* =) */


#define BUFFER_SIZE 260
/* Internal Structures (= */
typedef struct packet_t { /* (= */
    u_int8_t buffer[BUFFER_SIZE];
    int length;
    int idx;
    enum {
        OK,
        OK_REVISION,
        OK_PARAM,
        RETRANSMIT,
        BAD_CONTEXT,
        DENIED,
        HOST_ACK,
        UNKNOWN
    } status;
    struct packet_t *next;
} packet_t; /* =) */
/* This is the private structure used to keep whatever information is
 * necessary about each instance of the driver. */
typedef struct my_card_info_t {
    dev_link_t             dev_link;
    int                    got_io:1;
    int                    got_irq:1;
    int                    keyboard_sim:1;
    int                    poll_scan_done:1;
    int                    initialized:1;
    int                    opened;
    unsigned int           port;
    char                  *pre_suf_leftoff;
    volatile packet_t     *txpackets;
    volatile packet_t     *txtail;
    spinlock_t             txpacket_sl;
    spinlock_t             txrts_sl;

    volatile packet_t     *txacknack;
    volatile packet_t     *txacknacktail;
    spinlock_t             txacknack_sl;

    volatile packet_t     *rxpackets;
    volatile packet_t     *rxtail;
    unsigned long          rx_last_receive;
    spinlock_t             rxpacket_sl;
    packet_t              *rxcurrent;
    u_int16_t              rx_cksum;
    enum {
        SOP,
        MOP,
        DROP,
        DROP_MOP
    }                      rx_state;
    int                    decode_state;
    int                    rx_drop_length;
    enum {
        IDLE,
        MOP_ACKNACK,
        MOP_NORMAL
    }                      tx_state;


    volatile packet_t     *waiting_for_ack_queue;
    volatile packet_t     *waiting_for_ack_tail;
    spinlock_t             waiting_for_ack_sl;

    volatile packet_t     *txreturn_queue;
    volatile packet_t     *txreturn_tail;
    spinlock_t             txreturn_sl;

    volatile packet_t     *decode_packet_queue;
    volatile packet_t     *decode_packet_tail;
    packet_t              *last_decode;
    spinlock_t             decode_packet_sl;

    char                   prefix[32];
    char                   suffix[32];
    struct timer_list      txtimer;
    struct timer_list      ksimtimer;
    struct timer_list      scandonetimer;
    u_int32_t              baud;
    u_int32_t              baud_base;
    wait_queue_head_t      read_wait;
    wait_queue_head_t      poll_wait;
    wait_queue_head_t      txreturn_wait;
    wait_queue_head_t      misc_sleep;
    sdgmhs_versions_t      versions;
    atomic_t               screeching_halt;
    unsigned int           mcr;      /* Used to keep track of the MCR register */

    packet_t              *freehead, *freetail;
    packet_t              *allocarray[4];
    int                    next_alloc;
    struct tasklet_struct  process_rx_tasklet;

    /* Robustness variables */
    int                    out_of_sync;
    struct timer_list      out_of_sync_timer;
    struct my_card_info_t *prev, *next;
} my_card_info_t;
/* =) */

#define OUT_OF_SYNC_DELAY HZ

/* Function prototypes (= */
static void      serial_setup(my_card_info_t *mcip);
static void      serial_clear(my_card_info_t *mcip);
static void      serial_transmit_irq(unsigned long data);
static void      serial_request_to_send_irq(unsigned long data);
static void      process_receive_tasklet_func(unsigned long data);
static void      simulate_keyboard_irq(unsigned long data);
static void      my_irq_handler(int irq, void *data, struct pt_regs *pt);
static void      wake_up_the_neighborhood(unsigned long data);
static void      done_with_out_of_sync(unsigned long data);
static void      reset_everything(my_card_info_t *mcip);
static void      enqueue_transmit_irq(my_card_info_t *mcip, packet_t *p);
static packet_t *dequeue_transmit_irq(my_card_info_t *mcip);
static void      enqueue_txacknack_irq(my_card_info_t *mcip, packet_t *p);
static packet_t *dequeue_txacknack_irq(my_card_info_t *mcip);
static u_char    get_next_character(my_card_info_t *mcip, int *done);
static void      change_baud(my_card_info_t *mcip, int baud);
static packet_t *do_transmit_wait_for_ack(my_card_info_t *mcip, packet_t *p);
static void      enqueue_packets_for_ack_irq(my_card_info_t *mcip, packet_t *p);
static packet_t *dequeue_packets_for_ack(my_card_info_t *mcip);
static void      enqueue_decode_packet(my_card_info_t *mcip, packet_t *p);
static packet_t *dequeue_decode_packet(my_card_info_t *mcip);

/* File operations functions */
static ssize_t mhs_read(struct file *f, char *c, size_t size, loff_t *off);
static ssize_t mhs_write(struct file *f, const char *c, size_t size,
    loff_t *off);
static unsigned int mhs_poll(struct file *f, struct poll_table_struct *ptsp);
static int mhs_ioctl(struct inode *node, struct file *f, unsigned int ioctl,
    unsigned long generic);
static int mhs_open(struct inode *node, struct file *f);
static int mhs_close(struct inode *node, struct file *f);

/* Packet memory management */
static packet_t *allocate_packet(my_card_info_t *mcip);
static packet_t *allocate_packet_sleep(my_card_info_t *mcip, int tries);
static void      free_packet(my_card_info_t *mcip, packet_t *p);
static void      deallocate_all_packets(my_card_info_t *mcip);
static packet_t *make_ack_packet(my_card_info_t *mcip);
static packet_t *make_nack_packet(my_card_info_t *mcip);
/* =) */

/* Static Variables (= */
static my_card_info_t *card_head;  /* Linked list head */
static my_card_info_t *card_tail;  /* Linked list tail */
static dev_info_t      dev_info = DRIVER_NAME;        /* A string describing the driver */
static my_card_info_t *devices[MAX_DEVICES];
static int             attached;
static int             my_major;

static struct file_operations fops = {
    owner:   THIS_MODULE,
    read:    mhs_read,
    write:   mhs_write,
    poll:    mhs_poll,
    ioctl:   mhs_ioctl,
    open:    mhs_open,
    release: mhs_close,
};

/* =) */

static ssize_t
mhs_read(struct file *f, char *c, size_t size, loff_t *off) /* (= */
{
    ssize_t retval;
    my_card_info_t *mcip = f->private_data;

    if(!attached) return -ENODEV;

    if(atomic_read(&mcip->screeching_halt) != 0) {
        return -ENODEV;
    }

    /* See if we have any data here */

    retval = 0;

    if(mcip->keyboard_sim) {
        *c = '!';
        retval = 1;
    } else {
        int done;
        u_char nextc;
        DECLARE_WAITQUEUE(wait, current);

        if(mcip->decode_packet_queue == 0) {
            /* The Queue is Empty */
            if(f->f_flags & O_NONBLOCK) {
                return -EAGAIN;
            }

            add_wait_queue(&mcip->read_wait, &wait);
            set_current_state(TASK_INTERRUPTIBLE);
            if(mcip->decode_packet_queue == 0) {
                schedule();
            }
            set_current_state(TASK_RUNNING);
            remove_wait_queue(&mcip->read_wait, &wait);

            if(atomic_read(&mcip->screeching_halt) != 0) {
                return -EIO;
            }
            if(signal_pending(current)) {
                return -ERESTARTSYS;
            }
        }

        done = 0;
        while(!done && size) {
            nextc = get_next_character(mcip, &done);
            if(!done) {
                *c = nextc;
                c++, size--, retval++;
            } else {
                free_packet(mcip, dequeue_decode_packet(mcip));
            }
        }
    }

    return retval;
} /* =) */

static ssize_t
mhs_write(struct file *f, const char *c, size_t size, loff_t *off) /* (= */
{
    ssize_t retval;
    my_card_info_t *mcip = f->private_data;

    if(!attached) return -ENODEV;

    if(atomic_read(&mcip->screeching_halt) != 0) {
        return -ENODEV;
    }

    retval = 0;
    return retval;
} /* =) */

static unsigned int
mhs_poll(struct file *f, struct poll_table_struct *ptsp) /* (= */
{
    my_card_info_t *mcip;
    int retval = 0;

    if(!attached) return POLLERR;

    mcip = f->private_data;
    if(!mcip) return POLLERR;

    if(atomic_read(&mcip->screeching_halt) != 0) {
        return POLLERR;
    }

    poll_wait(f, &mcip->poll_wait, ptsp);

    if(atomic_read(&mcip->screeching_halt) != 0) {
        return POLLERR;
    }
    if(mcip->poll_scan_done) {
        mcip->poll_scan_done = 0;
        retval |=  POLLIN | POLLRDNORM;
    }
    return retval;
} /* =) */

static int
mhs_ioctl(struct inode *node, struct file *f, unsigned int ioctl, /* (= */
    unsigned long generic)
{
    my_card_info_t *mcip;
    int genint;

    if(!attached) return -ENODEV;

    mcip = f->private_data;
    if(atomic_read(&mcip->screeching_halt) != 0) {
        return -ENODEV;
    }

    switch(ioctl) {
        case SDGMHS_START_DECODE: /* (= */
            {
                packet_t *p;
                u_int8_t *b;
                u_int16_t cksum;

                if(mcip->out_of_sync) {
                    return -EAGAIN;
                }

                p = allocate_packet_sleep(mcip, 5);
                if(!p) {
                    return -ENOMEM;
                }
                b = p->buffer;

                *(b++) = 0x04; /* Length */
                *(b++) = 0xe4; /* Opcode */
                *(b++) = 0x04; /* Source */
                *(b++) = 0x00; /* Status */
                cksum = 0;
                cksum -= 0x04 + 0xe4 + 0x04 + 0x00;
                *(b++) = (cksum >> 8) & 0xff; /* Checksum High */
                *(b++) = cksum & 0xff; /* Checksum Low */
                p->length = 6;

                p = do_transmit_wait_for_ack(mcip, p);
                if(!p) {
                    printk(KERN_ERR DRIVER_NAME ": ioctl: Start Decode failed\n");
                    return -EAGAIN;
                }
                free_packet(mcip, p);
            }
            break; /* =) */
        case SDGMHS_SET_BAUD: /* (= */
            MSG_DEBUG((KERN_DEBUG "Set baud to %ld\n", generic));
            if(atomic_read(&mcip->screeching_halt) == 0) {
                change_baud(mcip, generic);
            }
            break; /* =) */
        case SDGMHS_GET_BAUD: /* (= */
            copy_to_user((void *)generic, &mcip->baud, sizeof(mcip->baud));
            break; /* =) */
        case SDGMHS_SET_PRE: /* (= */
            /* We don't want to update the prefix while we're reading from it
             * somewhere else. */
            copy_from_user(mcip->prefix, (void *)generic, sizeof(mcip->prefix));
            break; /* =) */
        case SDGMHS_GET_PRE: /* (= */
            copy_to_user((void *)generic, mcip->prefix, sizeof(mcip->prefix));
            break; /* =) */
        case SDGMHS_SET_SUFF: /* (= */
            /* We don't want to update the suffix while we're reading from it
             * somewhere else. */
            copy_from_user(mcip->suffix, (void *)generic, sizeof(mcip->suffix));
            break; /* =) */
        case SDGMHS_GET_SUFF: /* (= */
            copy_to_user((void *)generic, mcip->suffix, sizeof(mcip->suffix));
            break; /* =) */
        case SDGMHS_SET_KBDSIM: /* (= */
            mcip->keyboard_sim = generic?1:0;
            break; /* =) */
        case SDGMHS_GET_KBDSIM: /* (= */
            genint = mcip->keyboard_sim;
            copy_to_user((void *)generic, &genint, sizeof(genint));
            break; /* =) */
        case SDGMHS_SET_BAUDBASE: /* (= */
            mcip->baud_base = generic;
            break; /* =) */
        case SDGMHS_GET_BAUDBASE: /* (= */
            copy_to_user((void *)generic, &mcip->baud_base,
                sizeof(mcip->baud_base));
            break; /* =) */
        case SDGMHS_GET_VERSIONS: /* (= */
            copy_to_user((void *)generic, &mcip->versions,
                sizeof(mcip->versions));
            break; /* =) */
        case SDGMHS_ISALIVE: /* (= */
            break; /* =) */
        default: /* (= */
            /* printk(KERN_ERR DRIVER_NAME "Unknown ioctl: %08x\n", ioctl); */
            return -EINVAL;
            break; /* =) */
    }
    return 0;
} /* =) */

static int
mhs_open(struct inode *node, struct file *f) /* (= */
{
    u_int minor;
    my_card_info_t *mcip;
    unsigned long flags;

    if(!attached) return -ENODEV;

    minor = MINOR(node->i_rdev);

    if(minor >= MAX_DEVICES) {
        printk(KERN_ERR DRIVER_NAME ": Request to open minor %d\n", minor);
        return -ENODEV;
    }
    /* Fill in the private data for the file */
    mcip = devices[minor];
    f->private_data = mcip;
    if(mcip->opened == 0) {
        save_flags(flags);
        cli();
        mcip->opened = 1;
        restore_flags(flags);
        if(atomic_read(&mcip->screeching_halt) != 0) {
            mcip->opened = 0;
            mcip->initialized = 0;
            return -ENODEV;
        }
        mcip->rx_state = SOP;
        mcip->decode_state = 0;
        mcip->initialized = 1;
        serial_setup(mcip);
    } else {
        mcip->opened++;
        if(atomic_read(&mcip->screeching_halt) != 0) {
            return -ENODEV;
        }
    }
    return 0;
} /* =) */

static int
mhs_close(struct inode *node, struct file *f) /* (= */
{
    u_int minor;
    my_card_info_t *mcip;

    if(!attached) return 0;

    minor = MINOR(node->i_rdev);

    if(minor >= MAX_DEVICES) {
        printk(KERN_ERR DRIVER_NAME ": Request to open minor %d\n", minor);
        return -ENODEV;
    }

    mcip = devices[minor];

    mcip->opened--;
    if(!mcip->opened) {
        /* This is the last to close */
        if(atomic_read(&mcip->screeching_halt) == 0) {
            /* Only touch the hardware if the card has NOT been removed */
            serial_clear(mcip);
        }
    }
    f->private_data = 0;
    return 0;
} /* =) */

static int
get_tuple(int func, client_handle_t handle, tuple_t *tuple, /* (= */
    cisparse_t *parse)
{
    int retval;
    if(CS_SUCCESS != CardServices(func, handle, tuple)) {
        printk(KERN_ERR DRIVER_NAME ": get_tuple Failed\n");
        return -1;
    }
    if(CS_SUCCESS != CardServices(GetTupleData, handle, tuple)) {
        printk(KERN_ERR DRIVER_NAME ": GetTupleData Failed\n");
        return -2;
    }
    retval = CardServices(ParseTuple, handle, tuple, parse);
    if(retval != CS_SUCCESS) {
        printk(KERN_ERR DRIVER_NAME ": ParseTuple Failed: %d\n", retval);
        return -3;
    }
    return 0;
}
/* =) */

static int
set_up_serial(my_card_info_t *mcip) /* (= */
{
    client_handle_t handle = mcip->dev_link.handle;
    tuple_t tuple;
    cisparse_t parse;
    u_short tuplebuf[64];
    int i;
    int retval;
    cisinfo_t cisinfo;
    config_info_t ci;
    int find_vcc = 0;


    if(CS_SUCCESS != CardServices(ValidateCIS, handle, &cisinfo)) {
        printk(KERN_ERR DRIVER_NAME ": Invalid CIS\n");
        return -1;
    }

    /* Get Device Version Info (= */
    memset(&tuple, 0, sizeof(tuple));
    tuple.DesiredTuple = CISTPL_VERS_1;
    tuple.TupleData = (cisdata_t *)tuplebuf;
    tuple.TupleDataMax = (cisdata_t) sizeof(tuplebuf);
    tuple.TupleOffset = 0;
    if(0 == (retval = get_tuple(GetFirstTuple, handle, &tuple, &parse))) {
            mcip->versions.hw_version =
                parse.version_1.major * 0x100 + parse.version_1.minor;
            strncpy(mcip->versions.hw_string1, parse.version_1.str,
                sizeof(mcip->versions.hw_string1));
            if(parse.version_1.ofs[1]) {
                strncpy(mcip->versions.hw_string2,
                    &parse.version_1.str[parse.version_1.ofs[1]],
                    sizeof(mcip->versions.hw_string2));
            }
    } else {
        printk(KERN_ERR DRIVER_NAME ": ParseTuple for CISTPL_VERS_1 Failed: %d\n", retval);
    }
    /* =) */

    /* Get Manufacturer ID (= */
    memset(&tuple, 0, sizeof(tuple));
    tuple.DesiredTuple = CISTPL_MANFID;
    tuple.TupleData = (cisdata_t *)tuplebuf;
    tuple.TupleDataMax = (cisdata_t) sizeof(tuplebuf);
    tuple.TupleOffset = 0;
    if(0 == get_tuple(GetFirstTuple, handle, &tuple, &parse)) {
        if(parse.manfid.manf != 0x104 || parse.manfid.card != 0x9f) {
            printk(KERN_ERR DRIVER_NAME ": Unknown Manfid: %08x:%08x\n", parse.manfid.manf,
                parse.manfid.card);
            return -1;
        }
    }
    /* =) */

    /* Get Function ID (= */
    memset(&tuple, 0, sizeof(tuple));
    tuple.DesiredTuple = CISTPL_FUNCID;
    tuple.TupleData = (cisdata_t *)tuplebuf;
    tuple.TupleDataMax = (cisdata_t) sizeof(tuplebuf);
    tuple.TupleOffset = 0;
    if(0 == get_tuple(GetFirstTuple, handle, &tuple, &parse)) {
        if(parse.funcid.func != 2) {
            printk(KERN_ERR DRIVER_NAME ": Unexpected FunctionID: %08x:%08x\n",
                parse.funcid.func, parse.funcid.sysinit);
        }
    }
    /* =) */


    /* Now for the important stuff */
    /* Get CONFIG (= */
    memset(&tuple, 0, sizeof(tuple));
    tuple.DesiredTuple = CISTPL_CONFIG;
    tuple.TupleData = (cisdata_t *)tuplebuf;
    tuple.TupleDataMax = (cisdata_t) sizeof(tuplebuf);
    tuple.TupleOffset = 0;
    memset(&parse, 0, sizeof(parse));
    retval = get_tuple(GetFirstTuple, handle, &tuple, &parse);
    if(0 == retval) {
        /* Config retrieval successful */
        mcip->dev_link.conf.ConfigBase = parse.config.base;
        mcip->dev_link.conf.Present = parse.config.rmask[0];
    } else {
        printk(KERN_ERR DRIVER_NAME ": Could not get CONFIG\n");
        goto clean_up;
    }
    /* =) */
    /* Get ConfigurationInfo for Vcc (= */
    retval = CardServices(GetConfigurationInfo, handle, &ci);
    if(retval != CS_SUCCESS) {
        printk(KERN_ERR DRIVER_NAME ": Unable to GetConfigurationInfo\n");
        goto clean_up;
    }
    /* =) */

    /* Get CF Table Entries (= */
    memset(&tuple, 0, sizeof(tuple));
    tuple.DesiredTuple = CISTPL_CFTABLE_ENTRY;
    tuple.TupleData = (cisdata_t *)tuplebuf;
    tuple.TupleDataMax = (cisdata_t) sizeof(tuplebuf);
    tuple.TupleOffset = 0;
    memset(&parse, 0, sizeof(parse));
    retval = get_tuple(GetFirstTuple, handle, &tuple, &parse);
    while(retval == 0) {
        if(parse.cftable_entry.flags & CISTPL_CFTABLE_DEFAULT) {
            MSG_DEBUG((KERN_DEBUG "Info %d: flags=0x%02x, iface=%d, "
                "io windows=%d\n",
                parse.cftable_entry.index,
                parse.cftable_entry.flags,
                parse.cftable_entry.interface, parse.cftable_entry.io.nwin));
            MSG_DEBUG((KERN_DEBUG "    IRQ: %08x, %08x\n",
                    parse.cftable_entry.irq.IRQInfo1,
                    parse.cftable_entry.irq.IRQInfo2));
            for(i = 0; i < parse.cftable_entry.io.nwin && i < CISTPL_IO_MAX_WIN;
                    i++) {
                /* Clear structures (= */
                memset(&mcip->dev_link.io, 0, sizeof(mcip->dev_link.io));
                memset(&mcip->dev_link.irq, 0, sizeof(mcip->dev_link.irq));
                /* =) */

                mcip->dev_link.io.BasePort1 =
                    parse.cftable_entry.io.win[i].base;
                mcip->dev_link.io.NumPorts1 =
                    parse.cftable_entry.io.win[i].len;
                mcip->dev_link.io.IOAddrLines = 16;
                retval = CardServices(RequestIO, handle, &mcip->dev_link.io);
                if(retval == CS_SUCCESS) {
                    MSG_DEBUG((KERN_DEBUG "    IO: base 0x%04x, range: %u\n",
                        parse.cftable_entry.io.win[i].base,
                        parse.cftable_entry.io.win[i].len
                        ));
                    mcip->got_io = 1;
                    mcip->dev_link.irq.Attributes =
                        IRQ_TYPE_EXCLUSIVE | IRQ_HANDLE_PRESENT;
                    mcip->dev_link.irq.IRQInfo1 =
                        parse.cftable_entry.irq.IRQInfo1;
                    mcip->dev_link.irq.IRQInfo2 =
                        parse.cftable_entry.irq.IRQInfo2;
                    mcip->dev_link.irq.Handler = &my_irq_handler;
                    mcip->dev_link.irq.Instance = mcip;
                    retval = CardServices(RequestIRQ, handle,
                        &mcip->dev_link.irq);
                    if(retval == CS_SUCCESS) {
                        MSG_DEBUG((KERN_DEBUG "   Assigned IRQ: %d\n",
                            mcip->dev_link.irq.AssignedIRQ));
                        mcip->got_irq = 1;
                        /* Vcc  (= */
                        if(parse.cftable_entry.vcc.present
                                & (1 << CISTPL_POWER_VNOM)) {
                            /* Need to convert from units of 10 microvolts to
                             * units of 1/10 volt.  */
                            mcip->dev_link.conf.Vcc =
                                parse.cftable_entry.vcc.
                                param[CISTPL_POWER_VNOM]/10000;
                            MSG_DEBUG((KERN_DEBUG "Vcc is %d.%d volts\n",
                                mcip->dev_link.conf.Vcc / 10,
                                mcip->dev_link.conf.Vcc % 10));
                            if(mcip->dev_link.conf.Vcc != ci.Vcc) {
                                MSG_DEBUG((KERN_DEBUG
                                    "Default Vcc is not acceptable\n"));
                                find_vcc = parse.cftable_entry.index;
                            }
                        } else {
                            printk(KERN_ERR DRIVER_NAME
                                ": No VNOM in the power entry for Vcc\n");
                            goto clean_up;
                        }
                        /* =) */
                        /* Vppx  (= */
                        if(parse.cftable_entry.vpp1.present
                                & (1 << CISTPL_POWER_VNOM)) {
                            /* Need to convert from units of 10 microvolts to
                             * units of 1/10 volt.
                             */
                            mcip->dev_link.conf.Vpp1 =
                                parse.cftable_entry.vpp1.
                                param[CISTPL_POWER_VNOM]/10000;
                            mcip->dev_link.conf.Vpp2 = mcip->dev_link.conf.Vpp1;
                        } else {
                            if(parse.cftable_entry.vpp2.present
                                    & (1 << CISTPL_POWER_VNOM)) {
                                /* Need to convert from units of 10 microvolts
                                 * to units of 1/10 volt.  */
                                mcip->dev_link.conf.Vpp1 =
                                    parse.cftable_entry.vpp2.
                                    param[CISTPL_POWER_VNOM]/10000;
                                mcip->dev_link.conf.Vpp2 =
                                    mcip->dev_link.conf.Vpp1;
                            } else {
                                MSG_DEBUG((KERN_DEBUG
                                    "No VNOM in the power entry for Vppx\n"));
                                /* This is not an error */
                            }
                        }
                        /* =) */
                        /* PRESENT_OPTION (= */
                        if(mcip->dev_link.conf.Present & PRESENT_OPTION) {
                            mcip->dev_link.conf.ConfigIndex =
                                parse.cftable_entry.index;
                            MSG_DEBUG((KERN_DEBUG
                                "Programming ConfigIndex to %d\n",
                                mcip->dev_link.conf.ConfigIndex));
                        }
                        /* =) */
                        /* PRESENT_STATUS (= */
                        if(mcip->dev_link.conf.Present & PRESENT_STATUS) {
                            mcip->dev_link.conf.Status = 0;
                        }
                        /* =) */
                        /* PRESENT_PIN_REPLACE (= */
                        if(mcip->dev_link.conf.Present & PRESENT_PIN_REPLACE) {
                            mcip->dev_link.conf.Pin = 0;
                        }
                        /* =) */
                        /* PRESENT_COPY (= */
                        if(mcip->dev_link.conf.Present & PRESENT_COPY) {
                            mcip->dev_link.conf.Copy = 0;
                        }
                        /* =) */
                        /* PRESENT_EXT_STATUS (= */
                        if(mcip->dev_link.conf.Present & PRESENT_EXT_STATUS) {
                            mcip->dev_link.conf.ExtStatus = 0;
                        }
                        /* =) */
                        mcip->dev_link.conf.IntType = INT_MEMORY_AND_IO;
                        mcip->dev_link.conf.Attributes = CONF_ENABLE_IRQ;
                        goto io_irq_allocated;
                    } else {
                        MSG_DEBUG((KERN_DEBUG "Could not get IRQ\n"));
                        /* Free the IO */
                        if(CS_SUCCESS !=
                              CardServices(ReleaseIO, handle,
                              &mcip->dev_link.io)) {
                            printk(KERN_ERR DRIVER_NAME ": Could not release IO\n");
                        } else {
                            MSG_DEBUG((KERN_DEBUG "IO Released\n"));
                        }
                    }
                } else {
                    MSG_DEBUG((KERN_DEBUG "Did not get IO\n"));
                }
                mcip->got_io = 0;
                mcip->got_irq = 0;
            }
        }
        retval = get_tuple(GetNextTuple, handle, &tuple, &parse);
    }

    printk(KERN_ERR DRIVER_NAME ": Unable to parse I/O and IRQ tuples\n");
    goto clean_up;
    /* =) */

io_irq_allocated:

    if(find_vcc) {
        /* Look for an acceptable Vcc */
        while(CS_NO_MORE_ITEMS !=
                (retval = get_tuple(GetNextTuple, handle, &tuple, &parse))) {
            if(retval == CS_SUCCESS) {
                if(parse.cftable_entry.index == find_vcc) {
                    if(parse.cftable_entry.vcc.present &
                            (1 << CISTPL_POWER_VNOM)) {
                        mcip->dev_link.conf.Vcc =
                            parse.cftable_entry.vcc.
                            param[CISTPL_POWER_VNOM]/10000;
                        MSG_DEBUG((KERN_DEBUG "Vcc is %d.%d volts\n",
                                mcip->dev_link.conf.Vcc / 10,
                            mcip->dev_link.conf.Vcc % 10));
                        if(mcip->dev_link.conf.Vcc == ci.Vcc) {
                            break;
                        }
                    }
                }
            } else {
                printk(KERN_ERR DRIVER_NAME ": Could not find compatible Vcc\n");
                goto clean_up;
            }
        }
    }

    mcip->port = mcip->dev_link.io.BasePort1;

    if(CS_SUCCESS !=
            (retval = CardServices(RequestConfiguration, handle,
                                   &mcip->dev_link.conf))) {
        printk(KERN_ERR DRIVER_NAME ": Could not RequestConfiguration: %d\n", retval);
        goto clean_up;
    }

    mcip->dev_link.state &= ~DEV_CONFIG_PENDING;
    mcip->dev_link.state |= DEV_CONFIG;

    return 0;

clean_up:
    if(mcip->got_io) {
        if(CS_SUCCESS != CardServices(ReleaseIO, handle, &mcip->dev_link.io)) {
            printk(KERN_ERR DRIVER_NAME ": Could not release IO\n");
        } else {
            MSG_DEBUG((KERN_DEBUG "IO Released\n"));
        }
    }
    if(mcip->got_irq) {
        if(CS_SUCCESS !=
                CardServices(ReleaseIRQ, handle, &mcip->dev_link.irq)) {
            printk(KERN_ERR DRIVER_NAME ": Could not release IRQ\n");
        } else {
            MSG_DEBUG((KERN_DEBUG "IRQ Released\n"));
        }
        mcip->got_irq = 0;
    }
    return -1;
}
/* =) */

static int
mhs_event(event_t event, int priority, event_callback_args_t *args) /* (= */
{
    my_card_info_t *mcip;
    dev_link_t *link;

    MSG_DEBUG((KERN_DEBUG "Event: %d\n", event));

    mcip = (my_card_info_t *)args->client_data;
    link = &mcip->dev_link;

    switch(event) {
        case CS_EVENT_WRITE_PROTECT:
            break;
        case CS_EVENT_CARD_LOCK:
            break;
        case CS_EVENT_CARD_INSERTION:
            atomic_set(&mcip->screeching_halt, 0);
            mcip->dev_link.state = DEV_PRESENT | DEV_CONFIG_PENDING;
            if(!set_up_serial(mcip)) {
                return -1;
            }
            break;
        case CS_EVENT_EJECTION_REQUEST:
            atomic_set(&mcip->screeching_halt, 1);
            /* Wake up the neighborhood so they can return errors */
            wake_up_interruptible(&mcip->read_wait);
            wake_up_interruptible(&mcip->poll_wait);
            if(link->state & DEV_CONFIG) {
                CardServices(ReleaseConfiguration, link->handle);
            }
            link->state &= ~DEV_CONFIG;
            link->state &= ~DEV_PRESENT;
            break;
        case CS_EVENT_CARD_REMOVAL:
            if(atomic_read(&mcip->screeching_halt) == 0) {
                atomic_set(&mcip->screeching_halt, 1);
                /* Wake up the neighborhood so they can return errors */
                wake_up_interruptible(&mcip->read_wait);
                wake_up_interruptible(&mcip->poll_wait);
            }
            if(link->state & DEV_CONFIG) {
                CardServices(ReleaseConfiguration, link->handle);
            }
            link->state &= ~DEV_CONFIG;
            link->state &= ~DEV_PRESENT;
            break;
        case CS_EVENT_BATTERY_DEAD:
            break;
        case CS_EVENT_BATTERY_LOW:
            break;
        case CS_EVENT_READY_CHANGE:
            break;
        case CS_EVENT_CARD_DETECT:
            break;
        case CS_EVENT_RESET_REQUEST:
            if(mcip->opened) {
                return -EBUSY;
            }
            break;
        case CS_EVENT_RESET_PHYSICAL:
            serial_clear(mcip);
            if(link->state & DEV_CONFIG) {
                CardServices(ReleaseConfiguration, link->handle);
            }
            break;
        case CS_EVENT_CARD_RESET: /* Misleading. This is called when we are to
                                     un-reset the device. */
            if(link->state & DEV_CONFIG) {
                if(CS_SUCCESS != CardServices(RequestConfiguration,
                        link->handle, &link->conf)) {
                    printk(KERN_ERR DRIVER_NAME ": RequestConfiguration in "
                        "CS_EVENT_CARD_RESET failed\n");
                } else {
                    serial_setup(mcip);
                }
            }
            break;
        case CS_EVENT_REGISTRATION_COMPLETE:
            break;
        case CS_EVENT_RESET_COMPLETE:
            break;
        case CS_EVENT_PM_SUSPEND:
            link->state |= DEV_SUSPEND;
            if(link->state & DEV_CONFIG) {
                CardServices(ReleaseConfiguration, link->handle);
            }
            break;
        case CS_EVENT_PM_RESUME:
            link->state &= ~DEV_SUSPEND;
            if(link->state & DEV_CONFIG) {
                if(CS_SUCCESS != CardServices(RequestConfiguration,
                        link->handle, &link->conf)) {
                    printk(KERN_ERR DRIVER_NAME ": RequestConfiguration in "
                        "CS_EVENT_PM_RESUME failed\n");
                } else {
                    serial_setup(mcip);
                }
            }
            break;
        case CS_EVENT_INSERTION_REQUEST:
            break;
        case CS_EVENT_MTD_REQUEST:
            break;
        case CS_EVENT_ERASE_COMPLETE:
            break;
        case CS_EVENT_REQUEST_ATTENTION:
            break;
        case CS_EVENT_CB_DETECT:
            break;
        case CS_EVENT_3VCARD:
            break;
        case CS_EVENT_XVCARD:
            break;
        default:
            printk(KERN_ERR DRIVER_NAME ": Unknown event: %d\n", event);
    }
    return 0;
} /* =) */

static void
change_baud(my_card_info_t *mcip, int baud) /* (= */
{
    mcip->baud = baud;
    serial_setup(mcip);
}
/* =) */

/*
 * mhs_attach:
 *      Called when a device is bound to a socket. I am still unclear at this
 *      point about what causes this attach.
 */
static dev_link_t *
mhs_attach(void) /* (= */
{
    my_card_info_t *mcip;
    client_reg_t client_reg;
    int minor = -1;

    {
        /* Find a free device */
        int i;
        for(i = 0; i < MAX_DEVICES; i++) {
            if(devices[i] == 0) {
                minor = i;
                break;
            }
        }
        if(i == MAX_DEVICES) {
            printk(KERN_ERR DRIVER_NAME ": Only support %d devices\n", MAX_DEVICES);
            return 0;
        }
    }

    mcip = kmalloc(sizeof(*mcip), GFP_KERNEL);
    if(!mcip) {
        printk(KERN_ERR DRIVER_NAME ": Unable to allocate kernel memory (%d bytes)\n",
            sizeof(*mcip));
        return 0;
    }
    memset(mcip, 0, sizeof(*mcip));
    init_timer(&mcip->txtimer);
    mcip->txtimer.function = serial_transmit_irq;
    mcip->txtimer.data = (unsigned long)mcip;
    init_timer(&mcip->ksimtimer);
    mcip->ksimtimer.function = simulate_keyboard_irq;
    mcip->ksimtimer.data = (unsigned long)mcip;
    init_timer(&mcip->scandonetimer);
    mcip->scandonetimer.function = wake_up_the_neighborhood;
    mcip->scandonetimer.data = (unsigned long )mcip;
    init_timer(&mcip->out_of_sync_timer);
    mcip->out_of_sync_timer.function = done_with_out_of_sync;
    mcip->out_of_sync_timer.data = (unsigned long )mcip;
    mcip->baud = 9600;
    init_waitqueue_head(&mcip->read_wait);
    init_waitqueue_head(&mcip->poll_wait);
    init_waitqueue_head(&mcip->txreturn_wait);
    init_waitqueue_head(&mcip->misc_sleep);
    strcpy(mcip->prefix, "");
    strcpy(mcip->suffix, "\r\n");
    mcip->baud_base = 460800;
    mcip->versions.driver_version = SDGMHSGPL_DRIVER_VERSION;
    mcip->versions.hw_version = -1;
    mcip->rx_state = SOP;
    mcip->tx_state = IDLE;
    mcip->decode_state = 0;
    mcip->keyboard_sim = 1;
    tasklet_init(&mcip->process_rx_tasklet, process_receive_tasklet_func,
        (unsigned long)mcip);
    spin_lock_init(&mcip->rxpacket_sl);
    spin_lock_init(&mcip->txreturn_sl);
    spin_lock_init(&mcip->waiting_for_ack_sl);
    spin_lock_init(&mcip->txpacket_sl);
    spin_lock_init(&mcip->txrts_sl);
    spin_lock_init(&mcip->txacknack_sl);
    spin_lock_init(&mcip->decode_packet_sl);
#ifdef SHARP_KBDFILTER
    INIT_TQUEUE(&mcip->tq_scan, start_decode, mcip);
#endif

    mcip->dev_link.dev = kmalloc(sizeof(*mcip->dev_link.dev), GFP_KERNEL);
    if(!mcip->dev_link.dev) {
        printk(KERN_ERR DRIVER_NAME
            ": Unable to allocate kernel memory (%d bytes)\n",
            sizeof(*mcip->dev_link.dev));
        return 0;
    }
    memset(mcip->dev_link.dev, 0, sizeof(*mcip->dev_link.dev));
    mcip->dev_link.dev->major = my_major;
    mcip->dev_link.dev->minor = minor;
    devices[minor] = mcip;
    /* snprintf(mcip->dev_link.dev->dev_name, DEV_NAME_LEN, "sdgmhs%d", minor);*/
    /* Not all kernels have snprintf (e.g., Embedix) */
    strncpy(mcip->dev_link.dev->dev_name, "sdgmhsgpl", DEV_NAME_LEN);
    mcip->dev_link.dev->dev_name[9] = (minor % 10) + '0';
    mcip->dev_link.dev->dev_name[10] = 0;


    client_reg.dev_info = &dev_info;
    client_reg.Attributes = INFO_IO_CLIENT;
    client_reg.EventMask = CS_EVENT_CARD_INSERTION | CS_EVENT_CARD_REMOVAL
        | CS_EVENT_RESET_PHYSICAL | CS_EVENT_CARD_RESET | CS_EVENT_PM_SUSPEND
        | CS_EVENT_PM_RESUME | CS_EVENT_EJECTION_REQUEST;
    client_reg.event_handler = &mhs_event;
    memset(&client_reg.event_callback_args, 0,
        sizeof(client_reg.event_callback_args));
    client_reg.event_callback_args.client_data = mcip;
    client_reg.Version = CS_RELEASE_CODE;

    if(CS_SUCCESS !=
          CardServices(RegisterClient, &mcip->dev_link.handle, &client_reg)) {
        kfree(mcip->dev_link.dev);
        kfree(mcip);
        printk(KERN_ERR DRIVER_NAME ": Unable to RegisterClient\n");
        return 0;
    }
    client_reg.event_callback_args.client_handle = mcip->dev_link.handle;

    /* We've been successful, fill in the information for it. */
    mcip->dev_link.priv = mcip; /* Keep track of my instance's information */

    /* Link the current instance into the list */
    if(!card_head) {
        card_tail = card_head = mcip;
        mcip->prev = 0;
        mcip->next = 0;
    } else {
        card_tail->next = mcip;
        mcip->prev = card_tail;
        card_tail = mcip;
    }

    attached = 1;

#ifdef SHARP_KBDFILTER
    {
        int sharppda_register_kbdfilter(int (*func)(unsigned char scancode,
                int down, void *context), void *context);
        int retval;
        if((retval = sharppda_register_kbdfilter(sdgmhs_filter, mcip))) {
            printk(KERN_ERR "Could not register kbdfilter: %d\n", retval);
        }
    }
#endif

    return &mcip->dev_link;
}
/* =) */

/*
 * mhs_detach:
 *      Called when a device is unbound to a socket. I am still unclear at
 *      this point about what causes this detach.
 */
static void
mhs_detach(dev_link_t *this) /* (= */
{
    int retval;
    my_card_info_t *mcip;
    int not_released;

    attached = 0;
    not_released = 0;

    if(!this) {
        printk(KERN_ERR DRIVER_NAME ": Called with null parameter\n");
        return;
    }
    if(!this->priv) {
        printk(KERN_ERR DRIVER_NAME ": Expect non-null priv\n");
        return;
    }
    mcip = this->priv;


    atomic_set(&mcip->screeching_halt, 1);

    del_timer(&mcip->txtimer);
    del_timer(&mcip->ksimtimer);
    del_timer(&mcip->scandonetimer);

    /* Wake up the neighborhood so they can return errors */
    wake_up_interruptible(&mcip->read_wait);
    wake_up_interruptible(&mcip->poll_wait);

    serial_clear(mcip); /* Turn off interrupts */
    mcip->port = 0;
    if(mcip->dev_link.state & DEV_CONFIG) {
        if(CS_SUCCESS !=
            (retval = CardServices(ReleaseConfiguration, mcip->dev_link.handle,
                    &mcip->dev_link.conf))) {
            printk(KERN_ERR DRIVER_NAME ": Could not ReleaseConfiguration: %d\n", retval);
            not_released++;
        } else {
            MSG_DEBUG((KERN_DEBUG "ReleaseConfiguration Succeeded\n"));
        }
    }
    if(mcip->got_irq) {
        if(CS_SUCCESS != CardServices(ReleaseIRQ, mcip->dev_link.handle,
                &mcip->dev_link.irq)) {
            printk(KERN_ERR DRIVER_NAME ": Could not release irq\n");
            not_released++;
        } else {
            MSG_DEBUG((KERN_DEBUG "IRQ Released\n"));
        }
    }
    if(mcip->got_io) {
        if(CS_SUCCESS != CardServices(ReleaseIO, mcip->dev_link.handle,
                &mcip->dev_link.io)) {
            printk(KERN_ERR DRIVER_NAME ": Could not release io\n");
            not_released++;
        } else {
            MSG_DEBUG((KERN_DEBUG "IO Released\n"));
        }
        mcip->got_io = 0;
    }
    mcip->dev_link.state &= ~DEV_CONFIG;
    /* Unlink it */
    if(!not_released) {
        if(mcip->prev) {
            mcip->prev->next = mcip->next;
            mcip->next->prev = mcip->prev;
        } else {
            card_head = mcip->next;
            mcip->prev = 0;
        }
    }

    {
        int i;
        for(i = 0; i < MAX_DEVICES; i++) {
            if(devices[i] == mcip) {
                devices[i] = 0;
                break;
            }
        }
        if(i == MAX_DEVICES) {
            printk(KERN_ERR DRIVER_NAME ": Unknown device[]\n");
        }
    }

    retval = CardServices(DeregisterClient, this->handle);
    switch(retval) {
        case CS_BAD_HANDLE:
            printk(KERN_ERR DRIVER_NAME ": DeregisterClient called with bad handle.\n");
            break;
        case CS_IN_USE:
            printk(KERN_ERR DRIVER_NAME ": DeregisterClient called with units in use.\n");
            break;
        case CS_SUCCESS:
            MSG_DEBUG((KERN_DEBUG DRIVER_NAME ": DeregisterClient successful\n"));
            break;
        default:
            printk(KERN_ERR DRIVER_NAME ": Unknown CardServices reply\n");
            break;
    }

#ifdef SHARP_KBDFILTER
    {
        int sharppda_unregister_kbdfilter(int (*func)(unsigned char scancode,
                int down, void *context), void *context);
        int retval;
        if((retval = sharppda_unregister_kbdfilter(sdgmhs_filter, mcip))) {
            printk(KERN_ERR "Could not unregister kbdfilter: %d\n", retval);
        }
    }
#endif

    if(!not_released && mcip) {
        kfree(mcip->dev_link.dev);
        deallocate_all_packets(mcip);
        kfree(mcip);
    }
}
/* =) */

/*
 * mhs_init:
 *      The Module entry point. Called when insmod is invoked.
 */
static int __init
mhs_init(void) /* (= */
{
    servinfo_t info;
    int retval;

    if(0 > (my_major = register_chrdev(0, DRIVER_NAME, &fops))) {
        printk(KERN_ERR DRIVER_NAME ": Could not register chrdev\n");
        return -1;
    }

    printk(KERN_INFO BANNER);
    printk(KERN_INFO "Compiled: " __DATE__ ", " __TIME__ "\n");

    if(CS_SUCCESS != CardServices(GetCardServicesInfo, &info)) {
        printk(KERN_ERR DRIVER_NAME ": Call to CardServices Failed\n");
        return -1;
    }
    if(CS_RELEASE_CODE != info.Revision) {
        printk(KERN_ERR DRIVER_NAME ": Mismatched CardServices Revision\n");
        return -1;
    }

    card_tail = card_head = 0;

    attached = 0;
    if((retval =
            register_pccard_driver(&dev_info, mhs_attach, mhs_detach))) {
        printk(KERN_ERR DRIVER_NAME ": register_pcmcia_driver failed: %d\n",
            retval);
        return -1;
    }

    printk(KERN_INFO DRIVER_NAME ": Successfully registered \"%s\"\n",
        dev_info);
    return 0;
}
/* =) */

/*
 * mhs_exit:
 *      The Module exit point. Called when rmmod is invoked.
 *      This will not be called unless the MOD_DEC_USE_COUNT is called the
 *      same number of times as MOD_INC_USE_COUNT.
 */
static void __exit
mhs_exit(void) /* (= */
{

    if(unregister_pccard_driver(&dev_info)) {
        printk(KERN_ERR DRIVER_NAME ": Error unregistering pcmcia driver\n");
    }
    if(0 != unregister_chrdev(my_major, DRIVER_NAME)) {
        printk(KERN_ERR DRIVER_NAME ": Could not unregister chrdev\n");
    }

    printk(KERN_INFO DRIVER_NAME ": Exiting\n");
} /* =) */

/*
 * C-Preprocessor magic to do the module exit and entry points
 */
module_init(mhs_init);
module_exit(mhs_exit);


/*****************************************************************************
 * Serial Port Stuff
 *****************************************************************************/
/* 16550 Registers (= */
#define REG_RBR         0 /* Read  */
#define REG_THR         0 /* Write */
#define REG_IER         1
#       define REG_IER_ERBFI    (1 << 0)
#       define REG_IER_ETBEI    (1 << 1)
#       define REG_IER_ELSI     (1 << 2)
#       define REG_IER_EDSSI    (1 << 3)
#define REG_IIR         2 /* Read  */
#       define REG_IIR_PEND     (1 << 0)
#       define REG_IIR_IRQ_MASK (7 << 1)
#       define REG_IIR_FIFO_EN1 (1 << 6)
#       define REG_IIR_FIFO_EN2 (1 << 7)
#define REG_FCR         2 /* Write */
#       define REG_FCR_FIFO_EN  (1 << 0)
#       define REG_FCR_RFIFO_RS (1 << 1)
#       define REG_FCR_TFIFO_RS (1 << 2)
#       define REG_FCR_DMA_MODE0 (0 << 3)
#       define REG_FCR_DMA_MODE1 (1 << 3)
#       define REG_FCR_RTRIGGER (3 << 6)
#define REG_LCR         3
#       define REG_LCR_WORDLEN  (3 << 0)
#       define REG_LCR_STOPBITS (1 << 2)
#       define REG_LCR_PARITYEN (1 << 3)
#       define REG_LCR_EVEN     (1 << 4)
#       define REG_LCR_STICK    (1 << 5)
#       define REG_LCR_BREAK    (1 << 6)
#       define REG_LCR_DLAB     (1 << 7)
#define REG_MCR         4
#       define REG_MCR_DTR      (1 << 0)
#       define REG_MCR_RTS      (1 << 1)
#       define REG_MCR_OUT1     (1 << 2)
#       define REG_MCR_LOOP     (1 << 4)
#define REG_LSR         5
#       define REG_LSR_DATARDY  (1 << 0)
#       define REG_LSR_OVERRUN  (1 << 1)
#       define REG_LSR_PARITY   (1 << 2)
#       define REG_LSR_FRAMING  (1 << 3)
#       define REG_LSR_BREAK    (1 << 4)
#       define REG_LSR_TXHEMPTY (1 << 5)
#       define REG_LSR_TXEMPTY  (1 << 6)
#       define REG_LSR_RFIFOERR (1 << 7)
#define REG_MSR         6
#       define REG_MSR_DELTACTS (1 << 0)
#       define REG_MSR_DELTADSR (1 << 1)
#       define REG_MSR_TERI     (1 << 2)
#       define REG_MSR_DELTADCD (1 << 3)
#       define REG_MSR_CTS      (1 << 4)
#       define REG_MSR_DSR      (1 << 5)
#       define REG_MSR_RING     (1 << 6)
#       define REG_MSR_DCD      (1 << 7)
#define REG_SCR         7
#define REG_DLL         0  /* DLAB = 1 */
#define REG_DLM         1  /* DLAB = 1 */

static void
serial_setup(my_card_info_t *mcip) /* (= */
{
    unsigned int port = mcip->port;
    unsigned long flags;

    save_flags(flags);
    cli();

    outb(0xbf, port + REG_LCR); /* Extended register access mode */
    outb(0x10, port + REG_FCR); /* EFR: Turn on enhanced mode */
    outb(0x00, port + REG_LCR); /* Turn off extended register access */
    outb(0x03, port + REG_LCR); /* Turn on 8-bit words */
    outb(0x09, port + REG_MCR); /* Interrupt enable + RTS + DTR */
    mcip->mcr = 0x09;
    outb(0x07, port + REG_IER); /* Turn on interrupts */
    outb(0x93, port + REG_LCR); /* Set DLA bit */
    outb((mcip->baud_base/mcip->baud) & 0xff, port + REG_DLL);
    outb((mcip->baud_base/mcip->baud) >> 8, port + REG_DLM);
    outb(0x03, port + REG_LCR);
    outb(0x03, port + REG_FCR); /* Reset FIFOs and enable them */

    restore_flags(flags);
} /* =) */

static void
serial_clear(my_card_info_t *mcip) /* (= */
{
    unsigned int port = mcip->port;
    outb(0, port + REG_IER);
}
/* =) */
/* =) */

/* Keyboard Simulation (= */
#define SHIFT   0x8000
#define CONTROL 0x4000
#define KEYMASK (~SHIFT & ~CONTROL)

#ifndef ZAURUS
#  define SHIFTKEY 54
#  define CONTROLKEY 0
#  define RETURNKEY 28
static u_int16_t translate[128] = { /* (= */
    -1,                 /* 0: . */
    -1,                 /* 1: . */
    -1,                 /* 2: . */
    -1,                 /* 3: . */
    -1,                 /* 4: . */
    -1,                 /* 5: . */
    -1,                 /* 6: . */
    -1,                 /* 7: . */
    14,                 /* 8: . */
    15,                 /* 9: . */
    -1,                 /* 10: . */
    -1,                 /* 11: . */
    -1,                 /* 12: . */
    -1,                 /* 13: . */
    -1,                 /* 14: . */
    -1,                 /* 15: . */
    -1,                 /* 16: . */
    -1,                 /* 17: . */
    -1,                 /* 18: . */
    -1,                 /* 19: . */
    -1,                 /* 20: . */
    -1,                 /* 21: . */
    -1,                 /* 22: . */
    -1,                 /* 23: . */
    -1,                 /* 24: . */
    -1,                 /* 25: . */
    -1,                 /* 26: . */
    -1,                 /* 27: . */
    -1,                 /* 28: . */
    -1,                 /* 29: . */
    -1,                 /* 30: . */
    -1,                 /* 31: . */
    57,                 /* 32:   */
     2 | SHIFT,         /* 33: ! */
    40 | SHIFT,         /* 34: " */
     4 | SHIFT,         /* 35: # */
     5 | SHIFT,         /* 36: $ */
     6 | SHIFT,         /* 37: % */
     8 | SHIFT,         /* 38: & */
    40,                 /* 39: ' */
    10 | SHIFT,         /* 40: ( */
    11 | SHIFT,         /* 41: ) */
     9 | SHIFT,         /* 42: * */
    13 | SHIFT,         /* 43: + */
    51,                 /* 44: , */
    12,                 /* 45: - */
    52,                 /* 46: . */
    53,                 /* 47: / */
    11,                 /* 48: 0 */
     2,                 /* 49: 1 */
     3,                 /* 50: 2 */
     4,                 /* 51: 3 */
     5,                 /* 52: 4 */
     6,                 /* 53: 5 */
     7,                 /* 54: 6 */
     8,                 /* 55: 7 */
     9,                 /* 56: 8 */
    10,                 /* 57: 9 */
    39 | SHIFT,         /* 58: : */
    39,                 /* 59: ; */
    51 | SHIFT,         /* 60: < */
    13,                 /* 61: = */
    52 | SHIFT,         /* 62: > */
    53 | SHIFT,         /* 63: ? */
     3 | SHIFT,         /* 64: @ */
    30 | SHIFT,         /* 65: A */
    48 | SHIFT,         /* 66: B */
    46 | SHIFT,         /* 67: C */
    32 | SHIFT,         /* 68: D */
    18 | SHIFT,         /* 69: E */
    33 | SHIFT,         /* 70: F */
    34 | SHIFT,         /* 71: G */
    35 | SHIFT,         /* 72: H */
    23 | SHIFT,         /* 73: I */
    36 | SHIFT,         /* 74: J */
    37 | SHIFT,         /* 75: K */
    38 | SHIFT,         /* 76: L */
    50 | SHIFT,         /* 77: M */
    49 | SHIFT,         /* 78: N */
    24 | SHIFT,         /* 79: O */
    25 | SHIFT,         /* 80: P */
    16 | SHIFT,         /* 81: Q */
    19 | SHIFT,         /* 82: R */
    31 | SHIFT,         /* 83: S */
    20 | SHIFT,         /* 84: T */
    22 | SHIFT,         /* 85: U */
    47 | SHIFT,         /* 86: V */
    17 | SHIFT,         /* 87: W */
    45 | SHIFT,         /* 88: X */
    21 | SHIFT,         /* 89: Y */
    44 | SHIFT,         /* 90: Z */
    26,                 /* 91: [ */
    43,                 /* 92: \ */
    27,                 /* 93: ] */
     7 | SHIFT,         /* 94: ^ */
    12 | SHIFT,         /* 95: _ */
    41,                 /* 96: ` */
    30,                 /* 97: a */
    48,                 /* 98: b */
    46,                 /* 99: c */
    32,                 /* 100: d */
    18,                 /* 101: e */
    33,                 /* 102: f */
    34,                 /* 103: g */
    35,                 /* 104: h */
    23,                 /* 105: i */
    36,                 /* 106: j */
    37,                 /* 107: k */
    38,                 /* 108: l */
    50,                 /* 109: m */
    49,                 /* 110: n */
    24,                 /* 111: o */
    25,                 /* 112: p */
    16,                 /* 113: q */
    19,                 /* 114: r */
    31,                 /* 115: s */
    20,                 /* 116: t */
    22,                 /* 117: u */
    47,                 /* 118: v */
    17,                 /* 119: w */
    45,                 /* 120: x */
    21,                 /* 121: y */
    44,                 /* 122: z */
    26 | SHIFT,         /* 123: { */
    43 | SHIFT,         /* 124: | */
    27 | SHIFT,         /* 125: } */
    41 | SHIFT,         /* 126: ~ */
    -1,                 /* 127: . */
}; /* =) */
#else /* Zaurus */
#  define SHIFTKEY 27
#  define CONTROLKEY 30
#  define RETURNKEY 28
#  define FUNCTION CONTROL
static u_int16_t translate[128] = { /* (= */
    -1,                 /* 0: . */
    -1,                 /* 1: . */
    -1,                 /* 2: . */
    -1,                 /* 3: . */
    -1,                 /* 4: . */
    -1,                 /* 5: . */
    -1,                 /* 6: . */
    -1,                 /* 7: . */
    31,                 /* 8: backspace */
    65,                 /* 9: tab */
    -1,                 /* 10: . */
    -1,                 /* 11: . */
    -1,                 /* 12: . */
    -1,                 /* 13: . */
    -1,                 /* 14: . */
    -1,                 /* 15: . */
    -1,                 /* 16: . */
    -1,                 /* 17: . */
    -1,                 /* 18: . */
    -1,                 /* 19: . */
    -1,                 /* 20: . */
    -1,                 /* 21: . */
    -1,                 /* 22: . */
    -1,                 /* 23: . */
    -1,                 /* 24: . */
    -1,                 /* 25: . */
    -1,                 /* 26: . */
    -1,                 /* 27: . */
    -1,                 /* 28: . */
    -1,                 /* 29: . */
    -1,                 /* 30: . */
    -1,                 /* 31: . */
    92,                 /* 32:   */
    94,                 /* 33: ! */
    72,                 /* 34: " */
    74,                 /* 35: # */
    75,                 /* 36: $ */
    76,                 /* 37: % */
    78,                 /* 38: & */
    70,                 /* 39: ' */
    80,                 /* 40: ( */
    84,                 /* 41: ) */
    79,                 /* 42: * */
    59,                 /* 43: + */
    63,                 /* 44: , */
    58,                 /* 45: - */
    64,                 /* 46: . */
    69,                 /* 47: / */
    50,                 /* 48: 0 */
    41,                 /* 49: 1 */
    42,                 /* 50: 2 */
    43,                 /* 51: 3 */
    44,                 /* 52: 4 */
    45,                 /* 53: 5 */
    46,                 /* 54: 6 */
    47,                 /* 55: 7 */
    48,                 /* 56: 8 */
    49,                 /* 57: 9 */
    73,                 /* 58: : */
    71,                 /* 59: ; */
    86,                 /* 60: < */
    83,                 /* 61: = */
    87,                 /* 62: > */
    62,                 /* 63: ? */
    61,                 /* 64: @ */
     1 | SHIFT,         /* 65: A */
     2 | SHIFT,         /* 66: B */
     3 | SHIFT,         /* 67: C */
     4 | SHIFT,         /* 68: D */
     5 | SHIFT,         /* 69: E */
     6 | SHIFT,         /* 70: F */
     7 | SHIFT,         /* 71: G */
     8 | SHIFT,         /* 72: H */
     9 | SHIFT,         /* 73: I */
    10 | SHIFT,         /* 74: J */
    11 | SHIFT,         /* 75: K */
    12 | SHIFT,         /* 76: L */
    13 | SHIFT,         /* 77: M */
    14 | SHIFT,         /* 78: N */
    15 | SHIFT,         /* 79: O */
    16 | SHIFT,         /* 80: P */
    17 | SHIFT,         /* 81: Q */
    18 | SHIFT,         /* 82: R */
    19 | SHIFT,         /* 83: S */
    20 | SHIFT,         /* 84: T */
    21 | SHIFT,         /* 85: U */
    22 | SHIFT,         /* 86: V */
    23 | SHIFT,         /* 87: W */
    24 | SHIFT,         /* 88: X */
    25 | SHIFT,         /* 89: Y */
    26 | SHIFT,         /* 90: Z */
    81 | FUNCTION | SHIFT,/* 91: [ */
    65 | SHIFT,         /* 92: \ */
    84 | SHIFT,         /* 93: ] */
    85 | SHIFT,         /* 94: ^ */
    77,                 /* 95: _ */
    32 | FUNCTION | SHIFT,/* 96: ` */
     1,                 /* 97: a */
     2,                 /* 98: b */
     3,                 /* 99: c */
     4,                 /* 100: d */
     5,                 /* 101: e */
     6,                 /* 102: f */
     7,                 /* 103: g */
     8,                 /* 104: h */
     9,                 /* 105: i */
    10,                 /* 106: j */
    11,                 /* 107: k */
    12,                 /* 108: l */
    13,                 /* 109: m */
    14,                 /* 110: n */
    15,                 /* 111: o */
    16,                 /* 112: p */
    17,                 /* 113: q */
    18,                 /* 114: r */
    19,                 /* 115: s */
    20,                 /* 116: t */
    21,                 /* 117: u */
    22,                 /* 118: v */
    23,                 /* 119: w */
    24,                 /* 120: x */
    25,                 /* 121: y */
    26,                 /* 122: z */
    86 | SHIFT,         /* 123: { */
    32,                 /* 124: | */
    87 | SHIFT,         /* 125: } */
    85,                 /* 126: ~ */
    -1,                 /* 127: . */
}; /* =) */
#endif /* =) */

static u_char
get_next_character(my_card_info_t *mcip, int *done) /* (= */
{
    u_char retval = 0;
    int got_one = 0;

    *done = 0;
next_state:
    switch(mcip->decode_state) {
        case 0:
            if(mcip->decode_packet_queue && (mcip->decode_packet_queue->idx
                        < mcip->decode_packet_queue->length)) {
                mcip->decode_packet_queue->idx = 1; /* Skip the length */
                mcip->decode_state = 1;
                /* XXX  We really need alock on the prefixes and suffixes. It
                 * is theoretically possible to change the prefix and suffix
                 * while we are doing a scan. */
                /* Fall through */
            } else {
                *done = 1;
                break;
            }
        case 1: /* At beginning of prefix */
            {
                u_char *pfx;

                pfx = mcip->pre_suf_leftoff ? mcip->pre_suf_leftoff
                    : mcip->prefix;
                /* We're at the beginning of a scan. Send the prefix */
                retval = *pfx;
                pfx++;
                if((retval == 0) || !(*pfx)) {
                    /* We finished the prefix */
                    mcip->decode_state = 10;
                    mcip->pre_suf_leftoff = 0;
                    mcip->decode_packet_queue->idx = 5; /* Skip the packet header */
                    if(retval == 0) goto next_state;
                } else {
                    /* else stay in this state */
                    mcip->pre_suf_leftoff = pfx;
                }
            }
            break;
        case 10:
            /* Send the scanned string */
            if(mcip->decode_packet_queue) {
try_again:
                if(mcip->decode_packet_queue->idx < mcip->decode_packet_queue->length) {
                    retval =
                        mcip->decode_packet_queue->buffer[mcip->decode_packet_queue->idx];
                    mcip->decode_packet_queue->idx++;
                    if(retval == '\n' || retval == '\r') {
                        goto try_again;
                    }
                    got_one = 1;
                } else {
                    /* We're done with the scanned string.  Now to the suffix */
                    mcip->decode_state = 20;
                    if(got_one == 0) {
                        goto next_state;
                    }
                }
            } else {
                printk(KERN_ERR DRIVER_NAME ": Keyboard sim with no decode data?\n");
                mcip->decode_state = 20;
            }
            break;
        case 20:
            {
                u_char *sfx;

                sfx = mcip->pre_suf_leftoff ? mcip->pre_suf_leftoff
                    : mcip->suffix;
                /* We're at the beginning of a scan. Send the suffix */
                retval = *sfx;
                sfx++;
                if((retval == 0) || !(*sfx)) {
                    /* We finished the suffix */
                    mcip->decode_state = 0;
                    mcip->pre_suf_leftoff = 0;
                    if(retval == 0) goto next_state;
                } else {
                    /* else stay in this state */
                    mcip->pre_suf_leftoff = sfx;
                }
            }
            break;
    }
    return retval;
} /* =) */

static void
simulate_keyboard_irq(unsigned long data) /* (= */
{
    my_card_info_t *mcip = (my_card_info_t *)data;
    int scancode;
    int done;

    while(1) {
        u_char new;

        if(mcip->out_of_sync) {
            return;
        }

        new = get_next_character(mcip, &done);

        if(done) {
            free_packet(mcip, dequeue_decode_packet(mcip));
            break;
        }


        if(new == '\n') {
            scancode = RETURNKEY;
        } else if(new == '\r') {
            continue;
        } else {
            scancode = translate[new];
            if(scancode == -1) {
                continue;
            }
        }

        if(scancode & SHIFT) {
            /* Shift on */
            handle_scancode(SHIFTKEY, 1);
        }
        if(scancode & CONTROL) {
            /* Control on */
            handle_scancode(CONTROLKEY, 1);
        }
        handle_scancode(scancode & KEYMASK, 1);
        handle_scancode(scancode & KEYMASK, 0);
        if(scancode & CONTROL) {
            /* Control on */
            handle_scancode(CONTROLKEY, 0);
        }
        if(scancode & SHIFT) {
            /* Shift off */
            handle_scancode(SHIFTKEY, 0);
        }
    }
} /* =) */

static void
enqueue_packets_for_ack_irq(my_card_info_t *mcip, packet_t *p) /* (= */
{
    p->next = 0;
    if(!mcip->out_of_sync) {
        unsigned long flags;
        spin_lock_irqsave(&mcip->waiting_for_ack_sl, flags);
        if(mcip->waiting_for_ack_queue) {
            mcip->waiting_for_ack_tail->next = p;
            mcip->waiting_for_ack_tail = p;
        } else {
            mcip->waiting_for_ack_queue = mcip->waiting_for_ack_tail = p;
        }
        spin_unlock_irqrestore(&mcip->waiting_for_ack_sl, flags);
    }
} /* =) */

static packet_t *
dequeue_packets_for_ack(my_card_info_t *mcip) /* (= */
{
    packet_t *retval = 0;
    if(!mcip->out_of_sync) {
        if(mcip->waiting_for_ack_queue) {
            unsigned long flags;
            spin_lock_irqsave(&mcip->waiting_for_ack_sl, flags);
            retval = (packet_t *)mcip->waiting_for_ack_queue;
            mcip->waiting_for_ack_queue = mcip->waiting_for_ack_queue->next;
            spin_unlock_irqrestore(&mcip->waiting_for_ack_sl, flags);
        }
    }
    return retval;
} /* =) */

static void
enqueue_tx_return_irq(my_card_info_t *mcip, packet_t *p) /* (= */
{
    unsigned long flags;
    p->next = 0;
    spin_lock_irqsave(&mcip->txreturn_sl, flags);
    if(mcip->txreturn_queue) {
        mcip->txreturn_tail->next = p;
        mcip->txreturn_tail = p;
    } else {
        mcip->txreturn_queue = mcip->txreturn_tail = p;
    }
    spin_unlock_irqrestore(&mcip->txreturn_sl, flags);
    wake_up_interruptible(&mcip->txreturn_wait);
} /* =) */

static packet_t *
dequeue_tx_return(my_card_info_t *mcip) /* (= */
{
    packet_t *retval = 0;
    if(mcip->txreturn_queue) {
        spin_lock(&mcip->txreturn_sl);
        retval = (packet_t *)mcip->txreturn_queue;
        mcip->txreturn_queue = retval->next;
        retval->next = 0;
        spin_unlock(&mcip->txreturn_sl);
    }
    return retval;
} /* =) */

static void
enqueue_rxpackets_irq(my_card_info_t *mcip, packet_t *p) /* (= */
{
    unsigned long flags;
    spin_lock_irqsave(&mcip->rxpacket_sl, flags);
    if(mcip->rxpackets) {
        mcip->rxtail->next = p;
        mcip->rxtail = p;
    } else {
        mcip->rxpackets = mcip->rxtail = p;
    }
    spin_unlock_irqrestore(&mcip->rxpacket_sl, flags);
} /* =) */

static packet_t *
dequeue_rxpackets_irq(my_card_info_t *mcip) /* (= */
{
    packet_t *retval = 0;
    unsigned long flags;
    if(mcip->rxpackets) {
        spin_lock_irqsave(&mcip->rxpacket_sl, flags);
        retval = (packet_t *)mcip->rxpackets;
        mcip->rxpackets = mcip->rxpackets->next;
        spin_unlock_irqrestore(&mcip->rxpacket_sl, flags);
    }
    return retval;
} /* =) */

static void
serial_transmit_irq(unsigned long data) /* (= */
{
    register my_card_info_t *mcip;
    register int can_go;
    register unsigned int port;
    volatile packet_t *pkt;
    int reschedule;
    unsigned long flags;


    can_go = 100; /* This is my tx burst size */
    mcip = (my_card_info_t *)data;
    port = mcip->port;


    switch(mcip->tx_state) {
        case IDLE:
            pkt = mcip->txacknack;
            if(!pkt) {
                pkt = mcip->txpackets;
                if(!pkt) {
                    return;
                } else {
                    mcip->tx_state = MOP_NORMAL;
                }
            } else {
                mcip->tx_state = MOP_ACKNACK;
            }
            break;
        case MOP_ACKNACK:
            pkt = mcip->txacknack;
            if(!pkt) {
                printk(KERN_ERR DRIVER_NAME ": MOP_ACKNACK with null pointer?\n");
                mcip->tx_state = IDLE;
                return;
            }
            break;
        case MOP_NORMAL:
            pkt = mcip->txpackets;
            if(!pkt) {
                printk(KERN_ERR DRIVER_NAME ": MOP_ACKNACK with null pointer?\n");
                mcip->tx_state = IDLE;
                return;
            }
            break;
        default:
            printk(KERN_ERR DRIVER_NAME ": Bad tx_state. Correcting.\n");
            mcip->tx_state = IDLE;
            return;
    }
    reschedule = 50; /* Reschedule by default by 1/50 second */

    if(!(inb(port + REG_LSR) & REG_LSR_TXEMPTY)) {
        reschedule = 50;
        can_go = 0; /* If the TX is not empty, let the timer call us back */
    }

    while(can_go--) {
        if(!(inb(mcip->port + REG_MSR) & REG_MSR_CTS)) {
            reschedule = 50;
            spin_lock_irqsave(&mcip->txrts_sl, flags);
            if(mcip->rx_state == SOP) {
                mcip->mcr |= REG_MCR_RTS;
                outb(mcip->mcr, mcip->port + REG_MCR);
            } else {
                mcip->mcr &= ~REG_MCR_RTS;
                outb(mcip->mcr, mcip->port + REG_MCR);
            }
            spin_unlock_irqrestore(&mcip->txrts_sl, flags);
            break;
        }
        outb(pkt->buffer[pkt->idx], port + REG_THR);
        pkt->idx++;
        if(pkt->idx == pkt->length) {
            /* We're done! Let the TX interrupt turn off RTS. */
            pkt->idx = 0;
            if(mcip->tx_state == MOP_ACKNACK) {
                dequeue_txacknack_irq(mcip);
            } else if(mcip->tx_state == MOP_NORMAL) {
                dequeue_transmit_irq(mcip);
            }
            mcip->tx_state = IDLE;
            if(pkt->status != HOST_ACK) {
                enqueue_packets_for_ack_irq(mcip, (packet_t *)pkt);
            } else {
                free_packet(mcip, (packet_t *)pkt);
            }
            if(!mcip->txpackets && !mcip->txacknack) {
                /* No more packets to send. We will be rescheduled when a new
                 * packet is enqueued. */
                reschedule = 0;
            }
            break;
        }
    }
    if(reschedule) {
        if(!timer_pending(&mcip->txtimer)) {
            mod_timer(&mcip->txtimer, jiffies + HZ/reschedule);
        }
    }
} /* =) */

static void
enqueue_transmit_irq(my_card_info_t *mcip, packet_t *p) /* (= */
{
    unsigned long flags;
    spin_lock_irqsave(&mcip->txpacket_sl, flags);
    if(mcip->txpackets) {
        mcip->txtail->next = p;
        mcip->txtail = p;
    } else {
        mcip->txpackets = mcip->txtail = p;
    }
    spin_unlock_irqrestore(&mcip->txpacket_sl, flags);
} /* =) */

static packet_t *
dequeue_transmit_irq(my_card_info_t *mcip) /* (= */
{
    unsigned long flags;
    packet_t *pkt = (packet_t *)mcip->txpackets;
    if(pkt) {
        spin_lock_irqsave(&mcip->txpacket_sl, flags);
        mcip->txpackets = pkt->next;
        spin_unlock_irqrestore(&mcip->txpacket_sl, flags);
        pkt->next = 0;
    }
    return pkt;
} /* =) */

static void
enqueue_decode_packet(my_card_info_t *mcip, packet_t *p) /* (= */
{
    unsigned long flags;
    spin_lock_irqsave(&mcip->decode_packet_sl, flags);
    if(mcip->decode_packet_queue) {
        mcip->decode_packet_tail->next = p;
        mcip->decode_packet_tail = p;
    } else {
        mcip->decode_packet_queue = mcip->decode_packet_tail = p;
    }
    spin_unlock_irqrestore(&mcip->decode_packet_sl, flags);
} /* =) */

static packet_t *
dequeue_decode_packet(my_card_info_t *mcip) /* (= */
{
    unsigned long flags;
    packet_t *pkt = (packet_t *)mcip->decode_packet_queue;
    if(pkt) {
        spin_lock_irqsave(&mcip->decode_packet_sl, flags);
        mcip->decode_packet_queue = pkt->next;
        spin_unlock_irqrestore(&mcip->decode_packet_sl, flags);
        pkt->next = 0;
    }
    return pkt;
} /* =) */

static void
enqueue_txacknack_irq(my_card_info_t *mcip, packet_t *p) /* (= */
{
    unsigned long flags;
    if(!mcip->out_of_sync) {
        spin_lock_irqsave(&mcip->txacknack_sl, flags);
        if(mcip->txacknack) {
            mcip->txacknacktail->next = p;
            mcip->txacknacktail = p;
        } else {
            mcip->txacknack = mcip->txacknacktail = p;
        }
        spin_unlock_irqrestore(&mcip->txacknack_sl, flags);
    }
} /* =) */

static packet_t *
dequeue_txacknack_irq(my_card_info_t *mcip) /* (= */
{
    unsigned long flags;
    packet_t *pkt = (packet_t *)mcip->txacknack;
    if(pkt) {
        spin_lock_irqsave(&mcip->txacknack_sl, flags);
        mcip->txacknack = pkt->next;
        spin_unlock_irqrestore(&mcip->txacknack_sl, flags);
        pkt->next = 0;
    }
    return pkt;
} /* =) */

static void
serial_request_to_send_irq(unsigned long data) /* (= */
{
    unsigned long flags;
    my_card_info_t *mcip = (my_card_info_t *)data;
    spin_lock_irqsave(&mcip->txrts_sl, flags);
    if(mcip->rx_state == SOP) {
        mcip->mcr |= REG_MCR_RTS;
        outb(mcip->mcr | REG_MCR_RTS, mcip->port + REG_MCR);
    }
    spin_unlock_irqrestore(&mcip->txrts_sl, flags);
    if(!timer_pending(&mcip->txtimer)) {
        /* serial_transmit_irq(data); */
        mod_timer(&mcip->txtimer, jiffies + 1);
    }
} /* =) */

/*
 * This function wakes up any system calls that may be in the wait state.
 */
static void
wake_up_the_neighborhood(unsigned long data) /* (= */
{
    my_card_info_t *mcip = (my_card_info_t *)data;

    mcip->poll_scan_done  = 1;

    wake_up_interruptible(&mcip->read_wait);
    wake_up_interruptible(&mcip->poll_wait);
} /* =) */

static void
done_with_out_of_sync(unsigned long data) /* (= */
{
    my_card_info_t *mcip = (my_card_info_t *)data;
    reset_everything(mcip);
    printk(KERN_ERR DRIVER_NAME ": Reset Complete\n");
} /* =) */

static void
reset_everything(my_card_info_t *mcip) /* (= */
{
    deallocate_all_packets(mcip);
    mcip->pre_suf_leftoff       = 0;
    mcip->rx_state              = SOP;
    mcip->tx_state              = IDLE;
    mcip->decode_state          = 0;
    mcip->txpackets             = 0;
    mcip->txtail                = 0;
    mcip->txacknack             = 0;
    mcip->txacknacktail         = 0;
    mcip->rxpackets             = 0;
    mcip->rxtail                = 0;
    mcip->rxcurrent             = 0;
    mcip->rx_last_receive       = 0;
    mcip->waiting_for_ack_queue = 0;
    mcip->waiting_for_ack_tail  = 0;
    mcip->txreturn_queue        = 0;
    mcip->txreturn_tail         = 0;
    mcip->decode_packet_queue   = 0;
    mcip->decode_packet_tail    = 0;
    mcip->last_decode           = 0;
    mcip->out_of_sync           = 0;
} /* =) */

#define ISC_CMD_ACK             0xd0
#define ISC_CMD_NAK             0xd1
#define ISC_DECODE_DATA         0xf3
#define ISC_EVENT               0xf6

/*
 * Tasklet to handle the received queue
 */
static void
process_receive_tasklet_func(unsigned long data) /* (= */
{
    register my_card_info_t *mcip;
    packet_t *p;
    mcip = (my_card_info_t *)data;

    /* LOCK rxpackets, irq */
    p = dequeue_rxpackets_irq(mcip);
    if(p) {
        switch(p->buffer[1]) {
            case ISC_CMD_ACK: /* (= */
                free_packet(mcip, (packet_t *)p);
                /* Reuse p */
                p = dequeue_packets_for_ack(mcip);
                if(!p) {
                    printk(KERN_ERR DRIVER_NAME ": ACK without command??\n");
                } else {
                    p->status = OK; /* We're OK */
                    enqueue_tx_return_irq(mcip, p);
                }
                break; /* =) */
            case ISC_CMD_NAK: /* (= */
                {
                    u_int8_t cause;

                    cause = p->buffer[4];
                    free_packet(mcip, (packet_t *)p);
                    /* Reuse p */
                    p = dequeue_packets_for_ack(mcip);
                    if(!p) {
                        printk(KERN_ERR DRIVER_NAME ": NACK without command?? cause: %d\n",
                            cause);
                        return;
                    } else {
                        p->next = 0;
                        switch(cause) {
                            case 1: /* Resend, bad checksum */
                                printk(KERN_ERR DRIVER_NAME ": NACK: Bad checksum\n");
                                p->status = RETRANSMIT;
                                break;
                            case 2: /* Don't Resend, bad context */
                                printk(KERN_ERR DRIVER_NAME ": NACK: Bad context\n");
                                p->status = BAD_CONTEXT;
                                break;
                            case 6: /* Don't Resend, denied */
                                printk(KERN_ERR DRIVER_NAME ": NACK: Denied\n");
                                p->status = DENIED;
                                break;
                            default:
                                printk(KERN_ERR DRIVER_NAME ": NACK: Unknown\n");
                                p->status = UNKNOWN;
                                break;
                        }
                        enqueue_tx_return_irq(mcip, p);
                    }
                }
                break; /* =) */
            case ISC_DECODE_DATA: /* (= */
                {
                    packet_t *ack;
                    p->next = 0;
                    enqueue_decode_packet(mcip, p);
                    mod_timer(&mcip->scandonetimer, jiffies + HZ/20);
                    if(mcip->keyboard_sim) {
                        mod_timer(&mcip->ksimtimer, jiffies + 2);
                    }
                    ack = make_ack_packet(mcip);
                    ack->status = HOST_ACK;
                    enqueue_txacknack_irq(mcip, ack);
                    serial_request_to_send_irq((unsigned long)mcip);
                }
                break; /* =) */
            case ISC_EVENT: /* (= */
                free_packet(mcip, (packet_t *)p);
                break; /* =) */
            default: /* (= */
                printk(KERN_ERR DRIVER_NAME ": Unkown message type: %d\n", p->buffer[1]);
                break; /* =) */
        }
    }
} /* =) */

static inline void
enqueue_rx_char_irq(my_card_info_t *mcip, u_int8_t c) /* (= */
{
    unsigned long last_rx;

    last_rx = mcip->rx_last_receive;
    mcip->rx_last_receive = jiffies;
    if(mcip->rx_state != SOP) {
            if(last_rx + HZ < jiffies) {
                /* We have a timeout. */
                if(mcip->rxcurrent) {
                    printk(KERN_ERR DRIVER_NAME ": Rx Timeout\n");
                    free_packet(mcip, mcip->rxcurrent);
                }
                mcip->rxcurrent = 0;
                mcip->rx_state = SOP;
            }
    }
    switch(mcip->rx_state) {
        case SOP:
            {
                mcip->rxcurrent = allocate_packet(mcip);
                if(!mcip->rxcurrent) {
                    /* Drop it on the floor */
                    /*
                     * C is the length not including the two checksum bytes.
                     * So, after this character, we have c + 2 - 1 to drop
                     * yet.
                     */
                    mcip->rx_drop_length = c + 1; /* c is the length */
                    mcip->rx_state = DROP;
                    printk(KERN_ERR DRIVER_NAME ": Unable to allocate rx packet.\n");
                    return;
                }
                mcip->rxcurrent->buffer[0] = c;
                mcip->rxcurrent->idx = 1;
                mcip->rx_cksum = 0;
                mcip->rx_cksum -= c;
                mcip->rx_state = MOP;
            }
            break;
        case MOP:
            mcip->rxcurrent->buffer[mcip->rxcurrent->idx++] = c;
            if(mcip->rxcurrent->idx == BUFFER_SIZE - 1) {
                /* Error.  Bad packet */
                free_packet(mcip, mcip->rxcurrent);
                mcip->rxcurrent = 0;
                printk(KERN_ERR DRIVER_NAME ": Bad message length. Reset.\n");
                /* How in the world do I sync up again??? */
                /* I guess reset everything. */
                /* Get a breather for 1 second */
                mcip->out_of_sync = 1;
                mod_timer(&mcip->out_of_sync_timer, jiffies + OUT_OF_SYNC_DELAY);
                return;
            }
            if(mcip->rxcurrent->idx <= mcip->rxcurrent->buffer[0]) {
                mcip->rx_cksum -= c;
            }
            if(mcip->rxcurrent->idx == mcip->rxcurrent->buffer[0] + 2) {
                /* Packet is complete */
                u_int16_t cksum;
                cksum =
                    (mcip->rxcurrent->buffer[mcip->rxcurrent->idx - 2] << 8)
                    | c;
                if(cksum != mcip->rx_cksum) {
                    printk(KERN_ERR DRIVER_NAME ": Bad RX checksum: got 0x%04x, expected 0x%04x\n",
                        cksum, mcip->rx_cksum);
                    free_packet(mcip, mcip->rxcurrent);
                    mcip->rxcurrent = 0;
                    {
                        packet_t *nack;
                        nack = make_nack_packet(mcip);
                        nack->status = HOST_ACK;
                        enqueue_txacknack_irq(mcip, nack);
                        serial_request_to_send_irq((unsigned long)mcip);
                    }
                } else {
                    /* Enqueue the packet */
                    /* LOCK rxpackets, irq */
                    mcip->rxcurrent->length = mcip->rxcurrent->buffer[0];
                    mcip->rxcurrent->idx = 0;
                    enqueue_rxpackets_irq(mcip, (packet_t *)mcip->rxcurrent);
                    mcip->rxcurrent = 0;
                    tasklet_schedule(&mcip->process_rx_tasklet);
                }
                mcip->rx_state = SOP;
            }
            break;
        case DROP:
            mcip->rx_drop_length--;
            if(mcip->rx_drop_length == 0) {
                mcip->rx_state = SOP;
            }
            break;
        case DROP_MOP:
            mcip->rxcurrent->idx++;
            if(mcip->rxcurrent->idx == BUFFER_SIZE - 1) {
                /* Error.  Bad packet */
                free_packet(mcip, mcip->rxcurrent);
                mcip->rxcurrent = 0;
                mcip->rx_state = SOP;
                printk(KERN_ERR DRIVER_NAME ": Bad message length. Reset. (DROP_MOP)\n");
                /* How in the world do I sync up again??? */
                /* I guess reset everything. */
                break;
            }
            if(mcip->rxcurrent->idx == mcip->rxcurrent->buffer[0] + 2) {
                free_packet(mcip, mcip->rxcurrent);
                mcip->rxcurrent = 0;
                mcip->rx_state = SOP;
                printk(KERN_ERR DRIVER_NAME ": Rx Timeout back to SOP\n");
            }
            break;
        default:
            mcip->rx_state = SOP;
            break;
    }
} /* =) */

/*
 *  Interrupt Service Routine
 */
static void
my_irq_handler(int irq, void *data, struct pt_regs *pt) /* (= */
{
    my_card_info_t *mcip;
    u_int retval;
    unsigned long flags;

    mcip = data;

    if(mcip->port == 0) {
        return;
    }
    if(mcip->initialized == 0) {
        outb(0, mcip->port + REG_IER);
        return;
    }
    if(atomic_read(&mcip->screeching_halt) == 1) {
        /* Not sure if I should even do this to block irqs */
        outb(0, mcip->port + REG_IER);
        return;
    }

    retval = (inb(mcip->port + REG_IIR)) & 0x3f;
    while(!(retval & REG_IIR_PEND)) {
        MSG_DEBUG((KERN_DEBUG "**IRQ: %d, IIR: %d\n**", irq, retval));
        switch(retval) {
            case 6: /* RX Line Status */
                inb(mcip->port + REG_LSR);
                break;
            case 4: /* RX Data Avail. */
            case 0xc: /* RX Timeout */
                while((inb(mcip->port + REG_LSR)) & 1) {
                    unsigned int c = inb(mcip->port+REG_RBR);
                    if(!mcip->out_of_sync) {
                        enqueue_rx_char_irq(mcip, c);
                    }
                }
                break;
            case 2: /* Can Transmit */
                /* Turn off RTS */
                spin_lock_irqsave(&mcip->txrts_sl, flags);
                mcip->mcr &= ~REG_MCR_RTS;
                outb(mcip->mcr, mcip->port + REG_MCR);
                spin_unlock_irqrestore(&mcip->txrts_sl, flags);
                break;
            case 0: /* Modem Status */
                inb(mcip->port + REG_MCR); /* Clear the interrupt source */
                break;
        }
        retval = (inb(mcip->port + REG_IIR)) & 0xf;
    }
    return;
} /* =) */

static packet_t *
do_transmit_wait_for_ack(my_card_info_t *mcip, packet_t *p) /* (= */
{
    packet_t *retval;
    signed long x;
    int timeout_count;
    int retran_count;
    DECLARE_WAITQUEUE(wait, current);

    if(!p) {
        return 0;
    }
    if(mcip->out_of_sync) {
        return 0;
    }

    retran_count = 0;
retran:
    timeout_count = 0;
try_again:
    enqueue_transmit_irq(mcip, p);
    serial_request_to_send_irq((unsigned long)mcip);

    /* Wait for the ack */
    add_wait_queue(&mcip->txreturn_wait, &wait);
    set_current_state(TASK_INTERRUPTIBLE);
    x = schedule_timeout(HZ * 2); /* Timeout is 2 seconds */
    set_current_state(TASK_RUNNING);
    remove_wait_queue(&mcip->txreturn_wait, &wait);
    if(0 == x) {
        if(mcip->out_of_sync) {
            return 0;
        }
        /* We had a timeout */
        timeout_count++;
        if(timeout_count == 3) {
            free_packet(mcip, p);
            return 0;
        }
        /* Set the retransmission bit */
        p->buffer[3] |= 1;
        p->next = 0;
        goto try_again;
        /* We're not going to try again */
        free_packet(mcip, p);
        return 0;
    }
    /* We got an ACK */
    retval = dequeue_tx_return(mcip);
    if(retval && retval->status == RETRANSMIT) {
        retval->status = OK;
        retran_count++;
        if(retran_count == 4) {
            free_packet(mcip, retval);
            return 0;
        }
        p = retval;
        goto retran;
    }
    return retval;
} /* =) */

/* Packet Management */
static packet_t *
allocate_packet(my_card_info_t *mcip) /* (= */
{
    packet_t *retval;

again:
    if(mcip->freehead) {
        retval = mcip->freehead;
        mcip->freehead = mcip->freehead->next;
        retval->length = -1;
        retval->idx    = 0;
        retval->next   = 0;
        retval->status = OK;
    } else {
        if(mcip->next_alloc ==
                sizeof(mcip->allocarray)/sizeof(mcip->allocarray[0])) {
            return 0;
        }
        mcip->allocarray[mcip->next_alloc] = mcip->freehead =
                    kmalloc(sizeof(*mcip->freehead) * 4, GFP_KERNEL);
        if(!mcip->freehead) {
            return 0;
        }
        mcip->next_alloc++;
        /* 1 */
        mcip->freetail = mcip->freehead;
        mcip->freetail->next = mcip->freetail + 1;
        /* 2 */
        mcip->freetail = mcip->freetail->next;
        mcip->freetail->next = mcip->freetail + 1;
        /* 3 */
        mcip->freetail = mcip->freetail->next;
        mcip->freetail->next = mcip->freetail + 1;
        /* 4 */
        mcip->freetail = mcip->freetail->next;
        mcip->freetail->next = 0;

        goto again;
    }
    return retval;
} /* =) */

static void
free_packet(my_card_info_t *mcip, packet_t *p) /* (= */
{
    if(!p) {
        return;
    }
    p->next = 0;
    if(mcip->freehead) {
        mcip->freetail->next = p;
        mcip->freetail = p;
    } else {
        mcip->freetail = mcip->freehead = p;
    }
} /* =) */

static void
deallocate_all_packets(my_card_info_t *mcip) /* (= */
{
    int i;
    for(i = 0; i < sizeof(mcip->allocarray)/sizeof(mcip->allocarray[0]); i++) {
        if(!mcip->allocarray[i]) break;
        kfree(mcip->allocarray[i]);
        mcip->allocarray[i] = 0;
    }
    mcip->next_alloc = 0;
} /* =) */

static packet_t *
allocate_packet_sleep(my_card_info_t *mcip, int tries) /* (= */
{

    packet_t *p;
    int count = 0;

try_again:
    count++;
    p = allocate_packet(mcip);
    if(!p) {
        if(count >= tries) {
            return 0;
        }
        interruptible_sleep_on_timeout(&mcip->misc_sleep, HZ / 20);
        goto try_again;
    }
    return p;
} /* =) */

static packet_t *
make_ack_packet(my_card_info_t *mcip) /* (= */
{
    packet_t *retval;
    u_int8_t *b;
    u_int16_t cksum;

    retval = allocate_packet_sleep(mcip, 5);
    if(retval) {
        b = retval->buffer;
        *(b++) = 0x04; /* Length */
        *(b++) = 0xd0; /* Opcode */
        *(b++) = 0x04; /* Source */
        *(b++) = 0x00; /* Status */
        cksum = 0;
        cksum -= 0x04 + 0xd0 + 0x04 + 0x00;
        *(b++) = (cksum >> 8) & 0xff; /* Checksum High */
        *(b++) = cksum & 0xff; /* Checksum Low */
        retval->length = 6;
    }
    return retval;
} /* =) */

static packet_t *
make_nack_packet(my_card_info_t *mcip) /* (= */
{
    packet_t *retval;
    u_int8_t *b;
    u_int16_t cksum;

    retval = allocate_packet_sleep(mcip, 5);
    if(retval) {
        b = retval->buffer;
        *(b++) = 0x04; /* Length */
        *(b++) = 0xd1; /* Opcode */
        *(b++) = 0x04; /* Source */
        *(b++) = 0x01; /* Status */
        cksum = 0;
        cksum -= 0x04 + 0xd1 + 0x04 + 0x01;
        *(b++) = (cksum >> 8) & 0xff; /* Checksum High */
        *(b++) = cksum & 0xff; /* Checksum Low */
        retval->length = 6;
    }
    return retval;
} /* =) */

/* vim600: set foldmethod=marker foldmarker=(=,=): */
