#define PCMCIA_DEBUG
//******************************************************************************
//	Copyright (C) USC-Information Sciences Institute
//      Copyright (C) Doe-Wan Kim <dwkim@east.isi.edu> 
//      Copyright (C) Ronald Alan Riley <rriley@east.isi.edu> 
//
//      This program is free software; you can redistribute it and/or
//      modify it under the terms of the GNU General Public License
//      as published by the Free Software Foundation; either version 2
//      of the License, or (at your option) any later version.
//
//      This program is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
//      GNU General Public License for more details.
//
//	This program has been modified from its original version for the
//	following purpose.
// 	- Added support for kernel 2.4.* 
// 	- Added support for YUV422P form
// 	- Fixed bugs in streaming video  
// 	- Removed PCI/ISA components not used by PCMCIA to make more compact
// 	- Converting from V4L2 to V4L(1) driver (partially done)
//******************************************************************************

//	Copyright (C) Jim Zajkowski <jamesez@umich.edu> (package maintainer)
//	Copyright (C) Chris Lahey <clahey@umich.edu> (original author)

//	This program is free software; you can redistribute it and/or
//	modify it under the terms of the GNU General Public License
//	as published by the Free Software Foundation; either version 2
//	of the License, or (at your option) any later version.

//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
//	GNU General Public License for more details.

//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

//	This code was taken from the Winnov Videum V4L2 driver, by Bill Dirks
//	<bdirks@pacbell.net>. His notice follows,

//	Winnov Videum	Video for Linux Two driver

//	This program is copyright 1998	by Bill Dirks

//	This program is free software; you can redistribute it and/or
//	modify it under the terms of the GNU General Public License
//	as published by the Free Software Foundation; either version
//	2 of the License, or (at your option) any later version.

//	gcc -c -O2 -Wall wnv.c

#ifdef PCMCIA_DEBUG
#define debug_msg(fmt,arg...) printk(KERN_DEBUG "wnv: " fmt,##arg)
#define err_msg(fmt,arg...) printk(KERN_ERR "wnv: " fmt,##arg)
#define info_msg(fmt,arg...) printk(KERN_INFO "wnv: " fmt,##arg)
#else
#define debug_msg(fmt,arg...)
#define err_msg(fmt,arg...)
#define info_msg(fmt,arg...)
#endif

#ifndef __KERNEL__
#define __KERNEL__
#endif
#ifndef MODULE
#define MODULE
#endif

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/devfs_fs_kernel.h>
#include <linux/vmalloc.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <asm/io.h>

// #include <pcmcia/k_compat.h>
#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/cistpl.h>
#include <pcmcia/cisreg.h>
#include <pcmcia/ds.h>
#include <videodevx/videodev.h>
//#ifdef V4L1
// //#include <linux/wrapper.h>
// #include "videodev.h"
// #else
// #include "../videodevX/videodev.h"
// #endif
#include "wnvrom.h"	// This is the Videum configuration EEPROM
static dev_info_t dev_info = "wnv_cs";
//#include <math.h>

// Video controls
#define MAXCONTROLS	4
#define VCTRL_BRIGHTNESS	0
#define VCTRL_CONTRAST		1
#define VCTRL_SATURATION	2
#define VCTRL_HUE			3

#ifdef V4L1
#define v4l2_device	video_device
#else
static struct v4l2_queryctrl videum_control[] = {
 {V4L2_CID_BRIGHTNESS,	"Brightness", 0, 15, 1, 8, V4L2_CTRL_TYPE_INTEGER},
 {V4L2_CID_CONTRAST,	"Contrast",	  0, 15, 1, 4, V4L2_CTRL_TYPE_INTEGER},
 {V4L2_CID_SATURATION,	"Saturation", 0, 15, 1, 4, V4L2_CTRL_TYPE_INTEGER},
 {V4L2_CID_HUE,			"Hue",		  0, 15, 1, 8, V4L2_CTRL_TYPE_INTEGER},
};

static int find_vctrl(int id)
{	int i;

	if (id == V4L2_CID_PRIVATE_BASE ||
		id <  V4L2_CID_BASE ||
		id >  V4L2_CID_LASTP1)
		return -EDOM;

	for (i=MAXCONTROLS; i--; )
		if (videum_control[i].id == id)
			return i;
	return -EINVAL;
}
#endif

void wnv_sleep( int milliseconds )
{	current->state = TASK_INTERRUPTIBLE; 
	schedule_timeout (HZ * milliseconds / 1000);
}

#ifndef RUN_AT
#define RUN_AT(x) (jiffies+(x))
#endif

struct videum_device;	// Forward reference

struct video_decoder
{	int	version;
	int	i2c_addr1;
	int	i2c_addr2;

	int	input;
	__u32	standards;
	__u32	standard;
	__u32	frame_period;
	__u32	decoder_stable_time;
};

// Microcode programs
struct ucode_parms
{	__u8	*buffer;
	int	source_x;
	int	source_y;
	int	source_width;
	int	source_height;
	int	field;
	int	capture_width;
	int	capture_height;
	int	dma_line;
	int	int1_line;
	int	int2_line;
	int	flags;
	int	filter;
};

// Microcode compile flags
#define COMPFLAG_NOVERTFILTER	0x00000001
#define COMPFLAG_FIELDORDER		0x00000002
#define COMPFLAG_VSDELAY		0x00000004
#define COMPFLAG_AVPRO_DMA		0x00000008
#define COMPFLAG_AVPRO_INT		0x00000010
#define COMPFLAG_HUFFMAN		0x00000020
#define COMPFLAG_MARKERCODE		0x00000040
#define COMPFLAG_EOFINT			0x00000080
#define COMPFLAG_VST_WHEN_DONE	0x00000100

//#define UC_SIZE			4096
#define UC_SIZE			0x0400

// Bus-master scatter list
struct scatter_node
{	__u32	addr;
	__u32	len;
};
//#define END_OF_SCATTER_LIST	0x80000000

// Image translations
struct lookup_rgb24
{	__u32	u_rgb[256];
	__u32	v_rgb[256];
	__u32	y_rgb[256];
	__u8	sat[1024];
};

struct lookup
{	int	type;
	int	size;
	union
	{	void			*base;	// vmalloc()
		__u16			*rgb16;
		struct lookup_rgb24	*rgb24;
	} table;
};
#define LUT_NULL	0
#define LUT_RGB555	1
#define LUT_RGB565	2
#define LUT_RGB24	3

struct translation
{	int		type;
	int		width;
	int		height;
	__u8		*in;
	__u8		*out;
	int		output_size;
	int		output_is_user;
	int		ilut;
	struct lookup	lut[4];
};
#define XLAT_NULL			0
#define XLAT_YUYV_TO_UYVY	1
#define XLAT_YUYV_TO_YUV420	2
#define XLAT_YUYV_TO_YUV422P 3
#define XLAT_YUYV_TO_GREY	4
#define XLAT_YUYV_TO_RGB555	6
#define XLAT_YUYV_TO_RGB565	7
#define XLAT_YUYV_TO_RGB24	8
#define XLAT_YUYV_TO_RGB32	9

// Per-open data for handling multiple opens on one device
struct device_open
{	int			isopen;
	struct videum_device	*dev;
};

// Streaming data buffer
struct video_q_node
{	struct video_q_node 	*forw, *back;
};
struct video_queue
{	struct video_q_node	*forw, *back;
	rwlock_t		qlock;
};
#ifdef V4L1
struct video_timecode
{	__u8	frames;
	__u8	seconds;
	__u8	minutes;
	__u8	hours;
	__u8	userbits[4];
	__u32	flags;
	__u32	type;
};
#define v4l2_timecode video_timecode
typedef __s64 stamp_t;
struct video_requestbuffers
{
	int	count;
	__u32	type;
	__u32	reserved[2];
};
#define v4l2_requestbuffers video_requestbuffers
struct vids_buffer
{	int				index;
	stamp_t			timestamp;
	__u32			type;
	__u32			offset;
	__u32			length;
	__u32			bytesused;
	__u32			flags;
	struct v4l2_timecode	timecode;
	__u32			sequence;
	__u32			reserved[3];
};
#define v4l2_buffer vids_buffer
struct video_performance
{	int	frames;
	int	framesdropped;
	__u64	bytesin;
	__u64	bytesout;
	__u32	reserved[4];
};

#define v4l2_performance video_performance
#define V4L2_STD_NTSC 11
#define V4L2_BUF_TYPE_CAPTURE		0x00000001
#define V4L2_BUF_TYPE_field		0x00001FFF  /* Type field mask  */
#define V4L2_BUF_FLAG_MAPPED	0x0001  /* Buffer is mapped (flag) */
#define V4L2_BUF_FLAG_QUEUED	0x0002	/* Buffer is queued for processing */
#define V4L2_BUF_FLAG_DONE	0x0004	/* Buffer is ready */
#define V4L2_BUF_FLAG_KEYFRAME	0x0008	/* Image is a keyframe (I-frame) */
// Map all buffers in one contiguous mmap(). Only used in VIDIOC_REQBUFS
#define V4L2_BUF_REQ_CONTIG	0x10000000
#endif
struct stream_buffer
{	struct video_q_node	qnode;
	struct v4l2_buffer	vidbuf;
	int			requested;
	__u8		*vaddress;	// vmalloc()
	struct scatter_node	*dma_list;	// get_free_page()
};
#define MAX_CAPTURE_BUFFERS	30
#define MAX_LOCKED_MEMORY	2000000

//	Videum device structure

//	One for each Videum in the system. This structure holds all the global
//	information the driver needs about each board.
typedef struct videum_device
{	struct v4l2_device	v;
	int			is_registered;
	struct device_open	open_data;

//	Per-bus index number for each Videum
	int			wavi_type;
#define WINNOV_WAVI_93		93
#define WINNOV_WAVI_95		95
#define WINNOV_WAVI_97		97
#define WINNOV_WIMP			99

//	I/O performance
	int			slave_rate;
//	Interrupts
	int			irq;
	int			ints_enabled;
	struct tq_struct	tqnode_dpc;	// for BH
	struct timer_list	tlnode;	// for polling interrupts
	wait_queue_head_t	new_video_frame;

//	Copy of the board's configuration EEPROM
	struct EEPROM		eeprom;

//	Used for I2C routines
	__u16			ctl;

//	On-board SRAM memory map
	int			sram_base;
//	int			sram_size;
	int			sram_ucode;
	int			sram_linebuf;
	int			sram_video;
	int			sram_video_size;

	struct video_decoder	videc;

	int			control[MAXCONTROLS];

//	WAVI video sampler microcode
	int			uc_loaded;
	struct ucode_parms	uc_parm;
	__u8		uc_buffer[2][UC_SIZE];
	int			uc_length[2];

//	Client Capture format and capture modes
#ifdef V4L1
	struct video_picture vPict;
	struct video_buffer vBuf;
#else
	struct v4l2_format	clientfmt;
	struct v4l2_captureparm	capture;
#endif

//	Hardware capture format	(see ucode_parms for width & height)
	int			capture_size;

//	Capture buffer
	__u8		*capture_buffer;	// vmalloc()
	int			capture_buffer_size;
//	struct scatter_node	*capture_dma_list;	// get_free_page()

//	Capture state
	int			ready_to_capture;
	int			sampler_enabled;
	int			capture_completed;
	unsigned long		time_acquired;	// millisecond time stamp
	int			streaming;
	struct stream_buffer	stream_buf[MAX_CAPTURE_BUFFERS];
	int			stream_buffers_mapped;
	struct video_queue	stream_q_capture;
	struct video_queue	stream_q_done;
	stamp_t			stream_begin;
	unsigned long	stream_last_frame;
	__u8			*stream_capture_buffer;

//	Image format conversions
	struct translation	translation2;
	__u8			*xlat_temp;	// vmalloc()

//	Performance statistics
	struct v4l2_performance perf;
 
//	PCMCIA Client data 
	dev_link_t *link;
	__u16 *mem_start;
	__u16 *data;
} videum_device;

//	Values for the microcode loaded status
#define UC_UNDEF		(-1)
#define UC_NORMAL		0

// Extreme video dimensions
#define MIN_WIDTH		32
#define MIN_HEIGHT		24
#define MAX_WIDTH		640
#define MAX_HEIGHT		240
#define MAX_FRAME_AGE		200 /* ms */
#define LOW_WATER		60 /* (%) Switch to higher quality */
#define HIGH_WATER		90 /* (%) Switch to lower quality */

typedef struct local_info_t {
	dev_node_t	node;
	int		stop;
	struct videum_device *device;
} local_info_t;

//	The Videum device structure array. This is the only global
//	variable in the module besides those used by the device probing
//	and enumeration routines (command line overrides)
#define NBOARDS		4
static struct videum_device videum[NBOARDS];
static int unit_video[NBOARDS] = { 0, 1, 2, 3, };
MODULE_PARM(unit_video, "1-"__MODULE_STRING(NBOARDS)"i");
/*
struct v4l2_device * v4l2_device_from_minor(int minor)
{	if (minor < 0 || minor >= V4L2_NUM_DEVICES)
		return NULL;
	return v4l2_device[minor];
}

struct v4l2_device * v4l2_device_from_file(struct file *file)
{	if (file == NULL)
		return NULL;
	return v4l2_device_from_minor(MINOR(file->f_dentry->d_inode->i_rdev));
}*/
#ifndef V4L1
static inline struct videum_device * videum_device_from_file(struct file *file)
{	return (struct videum_device *)v4l2_device_from_file(file);
}
#endif

//	W A V I   R E G I S T E R   A N D   S R A M   A C C E S S

//	These are the WAVI registers symbols
#include "wnvwavi.h"

// The spinlock is used to make sure device accesses are atomic.
// To avoid a compiler warning: 'wavi_lock' is unreferenced on non-SMP systems.
#define BEGIN_CRITICAL_SECTION	\
		do{unsigned long flags;spin_lock_irqsave(NULL,flags)
#define END_CRITICAL_SECTION	\
	spin_unlock_irqrestore(NULL,flags);}while(0)

//	wavi_readreg() and wavi_writereg() have regular and nolock versions. The
//	nolock versions do not use a critical section to guarantee atomicity.
//	When inside a critical section, always call the nolock functions, and
//	when not inside a critical section, always call the regular functions.
//	Failure to obey the rules will result in error, crash or system lockup.

#include "wnv_mp.h"

static __u16 wavi_readreg_nolock(struct videum_device *dev, int	reg)
{	return mp_register_read( dev, reg );
}

static __u16 wavi_writereg_nolock(struct videum_device *dev,
			 int	reg, __u16	value)
{	mp_register_write( dev, reg, value );
	return value;
}

//	For device types that use more than one I/O call 
static __u16 wavi_readreg(struct videum_device *dev, int reg)
{	__u16	x;
	BEGIN_CRITICAL_SECTION;
		x = wavi_readreg_nolock(dev, reg);
	END_CRITICAL_SECTION;
	return x;
}

//	For device types that use more than one I/O call 
static __u16 wavi_writereg(struct videum_device *dev, int reg, __u16 value)
{	BEGIN_CRITICAL_SECTION;
		wavi_writereg_nolock(dev, reg, value);
	END_CRITICAL_SECTION;
	return value;
}

static void wavi_block_read(struct videum_device *dev, int sram_address,
	 __u8 *buffer, int length)
{	BEGIN_CRITICAL_SECTION;
		wavi_writereg_nolock( dev, WAVI_HST1PTR, (__u16)(sram_address >> 4));
		mp_stream_read( dev, WAVI_HST1DAT, (__u16 *) buffer, length >> 1 );
	END_CRITICAL_SECTION;
}

static void wavi_block_write(struct videum_device *dev,
		 int sram_address, __u8 *buffer, int length)
{
	BEGIN_CRITICAL_SECTION;
		wavi_writereg_nolock(dev, WAVI_HST1PTR, (__u16)(sram_address >> 4));
		mp_stream_write( dev, WAVI_HST1DAT, (__u16 *) buffer, length >> 1 );
	END_CRITICAL_SECTION;
}

//	W A V I   F U N C T I O N S

// Put WAVI into a sensible state and do the one-time startup things
static void wavi_initialize(struct videum_device *dev)
{	int	t = wavi_readreg(dev, WAVI_MMA);
	switch (t & WAVI_FULLID)
	{	case 0x02:
			dev->wavi_type = WINNOV_WAVI_93; break;
		case 0x12:
			dev->wavi_type = WINNOV_WAVI_95; break;
		case 0x22:
			dev->wavi_type = WINNOV_WAVI_97;
			wavi_writereg(dev, WAVI_MMA, WAVI_DHRND | WAVI_MEDREN);
		break;
		case 0x03:
			dev->wavi_type = WINNOV_WIMP; break;
		default:
			dev->wavi_type = 0;
			err_msg("Unknown WAVI chip\n");
		break;
	}
	info_msg("Found WAVI-%d chip\n", dev->wavi_type);

	if (dev->wavi_type == WINNOV_WIMP)
	{	if ( ( dev->eeprom.dwHwFlags & HWFLAGS_CS4218 ) &&
		 ( dev->eeprom.dwHwFlags & HWFLAGS_AUDXSEL ) &&
		 ( dev->eeprom.dwHwFlags & HWFLAGS_XTAL_11K_NOT_PRESENT ) )
		{
		t = wavi_readreg( dev, WAVI_MMA );
		t |= WAVI_CS4218BIT;
		t |= WAVI_AUDXSEL_INT97;
		wavi_writereg( dev, WAVI_MMA, t );
		wnv_sleep( 150 );
		}

		t = wavi_readreg( dev, WAVI_CTL2 );
		t |= WAVI_DPLL_EN;
		wavi_writereg( dev, WAVI_CTL2, t );

		t |= WAVI_CCLK_EN;
		t &= ~WAVI_VCLK_EN;
		wavi_writereg( dev, WAVI_CTL2, t );
		wnv_sleep(50);
		
		t |= WAVI_MCLK_EN_CCLK;
		wavi_writereg( dev, WAVI_CTL2, t );

		t |= WAVI_VCLK_EN_MOV2;
		wavi_writereg( dev, WAVI_CTL2, t );

		if ( dev->eeprom.dwHwFlags & HWFLAGS_VCLK_INV )
		{	t |= WAVI_VCLK_INV;
			wavi_writereg( dev, WAVI_CTL2, t );
		}
		
		t = wavi_readreg( dev, WAVI_CTL2 );
		t |= WAVI_VSSBIT;
		wavi_writereg( dev, WAVI_CTL2, t );


//		Set GPIOs.

		wavi_writereg( dev, WAVI_TSTMODE, 0xFCFF );
		
		t = wavi_readreg( dev, WAVI_CTL2 );
		t &= ~WAVI_VCLK_EN;
		t |= WAVI_VCLK_EN_MOV2;
		t |= WAVI_GPIO2_OE;
		t &= ~WAVI_GPIO2;
		t &= ~WAVI_GPIO1_OE;
		t |= WAVI_GPIO0_OE;
		t &= ~WAVI_GPIO0;
		t &= ~WAVI_GPIO4;
		t &= ~WAVI_GPIO3;
		wavi_writereg( dev, WAVI_CTL2, t);
		t |= WAVI_GPIO4;
		wavi_writereg( dev, WAVI_CTL2, t );
		wnv_sleep(10);
		t &= ~WAVI_GPIO4;
		wavi_writereg( dev, WAVI_CTL2, t );
		wnv_sleep(10); 

//		Set config memory.
		if ( 0 )	// dev->eeprom.dwHwFlags & HWFLAGS_SRAM_HAIRY_MAPPING )
			wavi_writereg( dev, WAVI_MBNK, 0xF7FE );
		else
			wavi_writereg( dev, WAVI_MBNK, 0xF0F0 );
	}
// WAVI_VDF_DYCQ		0x0000  /* DYCQ */
// WAVI_VDF_YCQ		0x0020  /* YCQ */
// WAVI_VDF_YCH		0x0040  /* YCH */
// WAVI_VDF_RGB		0x0060  /* RGB */
// WAVI_VDF_DH4		0x0080  /* DH4 */
// WAVI_VDF_DH5		0x00A0  /* DH5 */
// WAVI_VDF_DH6		0x00C0  /* DH6 */
// WAVI_VDF_DH7		0x00E0  /* DH7 */
//	wavi_writereg(dev, WAVI_CTL, WAVI_VDF_RGB);
	wavi_writereg(dev, WAVI_CTL, WAVI_VDF_YCH);	//ZZZ
	wavi_writereg(dev, WAVI_ITR, 0);
	wavi_readreg(dev, WAVI_ITR);
	wavi_writereg(dev, WAVI_VOF, 0x8880);	// Offset  (Rd Bl Y) Flg =Gray->Y
	wavi_writereg(dev, WAVI_VCT, 0x4440);	// Contrast(Rd Bl Y) HSyncDel

	wavi_writereg(dev, WAVI_AUDPTR, 0);
	wavi_writereg(dev, WAVI_ATTN, 0);
	wavi_writereg(dev, WAVI_GAIN, 0);
	wavi_writereg(dev, WAVI_FREQ, 0);
	wavi_writereg(dev, WAVI_AUDMIX1, 0);
	wavi_writereg(dev, WAVI_AUDMIX2, 0);
}

static void wavi_brightness(struct videum_device *dev, int x)
{	int	t = wavi_readreg(dev, WAVI_VOF);
	wavi_writereg(dev, WAVI_VOF, (t & 0xFF0F) | ((x & 0x000F) << 4));
}

static void wavi_contrast(struct videum_device *dev, int x)
{	int	t = wavi_readreg(dev, WAVI_VCT);
	wavi_writereg(dev, WAVI_VCT, (t & 0xFF0F) | ((x & 0x000F) << 4));
}

static void wavi_saturation(struct videum_device *dev, int x)
{	int	t = wavi_readreg(dev, WAVI_VCT);
	x &= 0x000F;
	wavi_writereg(dev, WAVI_VCT, (t & 0x00FF) | (x << 8) | (x << 12));
}

static void wavi_hue(struct videum_device *dev, int x)
{	int	t = wavi_readreg(dev, WAVI_VOF);
	x &= 0x000F;
	wavi_writereg(dev, WAVI_VOF, (t & 0x00FF) | (x << 8) | (x << 12));
}

static void wavi_tone_controls(struct videum_device *dev)
{	int	*ctrl = dev->control;
	wavi_brightness(dev, ctrl[VCTRL_BRIGHTNESS]);
	wavi_contrast(dev, ctrl[VCTRL_CONTRAST]);
	wavi_saturation(dev, ctrl[VCTRL_SATURATION]);
	wavi_hue(dev, ctrl[VCTRL_HUE]);
}

static void wavi_vhsd(struct videum_device *dev, int x)
{	__u16	t = wavi_readreg(dev, WAVI_VCT);
	wavi_writereg(dev, WAVI_VCT, (t & 0xFFF0) | (x & 0x000F));
}

static void wavi_filter(struct videum_device *dev, int x)
{	__u16	t = wavi_readreg(dev, WAVI_CTL) & ~WAVI_VFSFIELD;
	wavi_writereg(dev, WAVI_CTL, t | x);
}

static void wavi_vdf(struct videum_device *dev, int x)
{	__u16	t = wavi_readreg(dev, WAVI_CTL) & ~WAVI_VDFFIELD;
	wavi_writereg(dev, WAVI_CTL, t | x);
}

static void wavi_vsc(struct videum_device *dev, int x)
{	__u16	t = wavi_readreg(dev, WAVI_CTL);
	wavi_writereg(dev, WAVI_CTL, (x) ? t | WAVI_VSCBIT : t & ~WAVI_VSCBIT);
}

static void wavi_vss(struct videum_device *dev, int x)
{	__u16	t = wavi_readreg(dev, WAVI_CTL);
	wavi_writereg(dev, WAVI_CTL, (x) ? t | WAVI_VSSBIT : t & ~WAVI_VSSBIT);
}

static int wavi_vst(struct videum_device *dev)
{	return (wavi_readreg(dev, WAVI_CTL) & WAVI_VSTBIT);
}

static void wavi_video_sampler(struct videum_device *dev, int x)
{	__u16	t = wavi_readreg(dev, WAVI_CTL);
	wavi_writereg(dev, WAVI_CTL, (x) ? t | WAVI_VSEBIT : t & ~WAVI_VSEBIT);
}

// A little integer square root routine
static int isqrt(unsigned int q)
{	int		i,r;
	unsigned int	t,b2	= 0x40000000;

	for (i = 16, r = 0; i > 0 && q; b2 >>= 2)
	{	t = ((unsigned int)r << i--) + b2;
		if (t <= q)
		{	q -= t;
			r |= 1 << i;
	}	}
	return r;
}

static unsigned long current_time_ms(void)
{	struct timeval	now;

	do_gettimeofday(&now);
	return now.tv_sec * 1000 + now.tv_usec / 1000;
}

//		2
//	  I   C   B U S   I N T E R F A C E

static __u16 in_ctl(struct videum_device *dev)
{	return wavi_readreg(dev, WAVI_CTL);
}

static void out_ctl(struct videum_device *dev)
{	wavi_writereg(dev, WAVI_CTL, dev->ctl);
}

static void i2c_delay(void)
{	udelay(2);
}

static void i2c_clock(struct videum_device *dev, int c)
{	int	t = 1000;
	if (c) dev->ctl |= WAVI_I2CBIT; else dev->ctl &= ~WAVI_I2CBIT;
	out_ctl(dev);
	if (!c) while (--t && (in_ctl(dev) & WAVI_I2CBIT));
}

static void i2c_data(struct videum_device *dev, int d)
{	if (d) dev->ctl |= WAVI_IMDBIT; else dev->ctl &= ~WAVI_IMDBIT;
	out_ctl(dev);
}
//	BUG: I2C bit read routines, polarity

static void i2c_start(struct videum_device *dev)
{	dev->ctl &= ~(WAVI_I2CBIT | WAVI_IMDBIT);
	out_ctl(dev);		i2c_delay();
	i2c_data(dev, 1);	i2c_delay();
	i2c_clock(dev, 0);	i2c_delay();
}

static int i2c_get_ack(struct videum_device *dev)
{	i2c_clock(dev, 1);	i2c_delay();
	i2c_data(dev, 0);	i2c_delay();
	i2c_clock(dev, 0);	i2c_delay();
	i2c_delay();
	return in_ctl(dev) & WAVI_IMDBIT;
}
// idle the I2C bus
static void i2c_end(struct videum_device *dev)
{	i2c_clock(dev, 1);	i2c_delay();
	i2c_data(dev, 1);	i2c_delay();
	i2c_clock(dev, 0);	i2c_delay();
	i2c_data(dev, 0);	i2c_delay();
}

static void i2c_out8(struct videum_device *dev, int	ndata)
{	int	i;

	ndata = ~ndata;
	for (i = 0; i < 8; ++i)
	{	i2c_clock(dev, 1);		i2c_delay();
		i2c_data(dev, ndata & 128);	i2c_delay();
		i2c_clock(dev, 0);		i2c_delay();
		ndata <<= 1;
}	}

static int i2c_in8(struct videum_device *dev)
{	int	i;
	int	ndata = ~0;

	dev->ctl &= ~WAVI_IMDBIT;// Don't write to h/w now
	for (i = 0; i < 8; ++i)
	{	i2c_clock(dev, 1);	i2c_delay();
		i2c_clock(dev, 0);	i2c_delay();
		ndata = (in_ctl(dev) & 1) | (ndata << 1);
	}
	return ~ndata;
}

//	Prepare I2C bus for operation
static void i2c_get_ready(struct videum_device *dev)
{	dev->ctl = in_ctl(dev);	// initialize the CTL register shadow
	i2c_end(dev);	// idle the I2C bus
}

static int i2c_error(struct videum_device *dev)
{	i2c_end(dev);
	return -1;
}

// Returns -1 on error.
static int i2c_read_reg_byte(struct videum_device *dev, int addr, int reg)
{	int	ndata;

	i2c_start(dev);
	i2c_out8(dev, addr);
	if (!i2c_get_ack(dev))
		return i2c_error(dev);	// no ACK from device
	i2c_out8(dev, reg);
	i2c_get_ack(dev);	// assume it succeeds if the first one succeeded
	i2c_end(dev);
	i2c_start(dev);
	i2c_out8(dev, addr | 1);
	if (!i2c_get_ack(dev))
		return i2c_error(dev);
	ndata = i2c_in8(dev);
	i2c_end(dev);
	return ndata;
}

static int	// -1 on error
i2c_write_reg_byte(struct videum_device *dev, int addr, int reg, int ndata)
{	i2c_start(dev);
	i2c_out8(dev, addr);
	if (!i2c_get_ack(dev))
		return i2c_error(dev);	// no ACK from device
	i2c_out8(dev, reg);
	i2c_get_ack(dev);
	i2c_out8(dev, ndata);
	i2c_get_ack(dev);
	i2c_end(dev);
	return 0;
}

 //	Read the configuration EEPROM
#define EEPROM_ADDR	0xA0

// Returns 1 = OK, 0 = error.
static int eeprom_load(struct videum_device *dev)
{	__u8	*p	= (__u8 *)&dev->eeprom;
	int	i, d;
	const int nb = sizeof(struct EEPROM);

	i2c_get_ready(dev);
	i2c_end(dev);

//	EEPROM readability/bogosity test.
	if (EEPROM_SIG != i2c_read_reg_byte(dev, EEPROM_ADDR, 0))
	{	err_msg("Couldn't read configuration EEPROM\n");
		return 0;
	}

	for (i = 0; i < nb; ++i)
	{	if (-1 == (d = i2c_read_reg_byte(dev, EEPROM_ADDR, i)))
		{	err_msg("Couldn't read all of EEPROM\n");
			break;
		}
		p[i] = (__u8)d;
	}

		// Run checksum
	for (d=i=0; i < nb; ++i)
		d += p[i];
	if ((__u8)d != 0xff)
	{	err_msg("Checksum error in configuration EEPROM\n");
		if (255 == (__u8)dev->eeprom.szBoardRev[0])
			memcpy(dev->eeprom.szBoardRev, "Rev A", 6);
		if (255 == (__u8)dev->eeprom.szProduct[0])
			memcpy(dev->eeprom.szProduct, "VideumCam AV",13);
		if (255 == (__u8)dev->eeprom.szOemName[0])
			memcpy(dev->eeprom.szOemName, "Winnov", 7);
		return 0;
	}
	info_msg("Identified: %s\n", dev->eeprom.szProduct);
	return 1;
}

//	V I D E O   D E C O D E R S

struct videc_reg_init
{	__u16	reg;
	__u16	value;
};

//==============================================================================
//	CS76x5 Video Decoder routines
//==============================================================================

static __u8 cs7665_gamma[256] = 
{	0x00, 0x0c, 0x12, 0x16, 0x1A,
	0x1D, 0x20, 0x23, 0x26, 0x29,
	0x2B, 0x2D, 0x2F, 0x32, 0x34,
	0x36, 0x38, 0x3A, 0x3B, 0x3D,
	0x3F, 0x41, 0x42, 0x44, 0x46,
  0x47, 0x49, 0x4A, 0x4A, 0x4D,
  0x4F, 0x50, 0x51, 0x53, 0x54,
  0x56, 0x57, 0x58, 0x5A, 0x5B,
  0x5C, 0x5D, 0x5F, 0x60, 0x61,
  0x62, 0x63, 0x65, 0x66, 0x67,
  0x68, 0x69, 0x6A, 0x6B, 0x6D,
  0x6E, 0x6F, 0x70, 0x71, 0x72,
  0x73, 0x74, 0x75, 0x76, 0x77,
  0x78, 0x79, 0x7A, 0x7B, 0x7C,
  0x7D, 0x7E, 0x7F, 0x80, 0x81,
  0x82, 0x83, 0x84, 0x85, 0x86,
  0x87, 0x88, 0x89, 0x8A, 0x8A,
  0x8B, 0x8C, 0x8D, 0x8E, 0x8F,
  0x90, 0x91, 0x92, 0x92, 0x93,
  0x94, 0x95, 0x96, 0x97, 0x98,
  0x98, 0x99, 0x9A, 0x9B, 0x9C,
  0x9D, 0x9D, 0x9E, 0x9F, 0xA0,
  0xA1, 0xA1, 0xA2, 0xA3, 0xA4,
  0xA5, 0xA5, 0xA6, 0xA7, 0xA8,
  0xA8, 0xA9, 0xAA, 0xAB, 0xAC,
  0xAC, 0xAD, 0xAE, 0xAF, 0xAF,
  0xB0, 0xB1, 0xB2, 0xB2, 0xB3,
  0xB4, 0xB4, 0xB5, 0xB6, 0xB7,
  0xB7, 0xB8, 0xB9, 0xBA, 0xBA,
  0xBB, 0xBC, 0xBC, 0xBD, 0xBE,
  0xBE, 0xBF, 0xC0, 0xC1, 0xC1,
  0xC2, 0xC3, 0xC3, 0xC4, 0xC5,
  0xC5, 0xC6, 0xC7, 0xC7, 0xC8,
  0xC9, 0xC9, 0xCA, 0xCB, 0xCB,
  0xCC, 0xCD, 0xCD, 0xCE, 0xCF,
  0xCF, 0xD0, 0xD1, 0xD1, 0xD2,
  0xD3, 0xD3, 0xD4, 0xD4, 0xD5,
  0xD6, 0xD6, 0xD7, 0xD8, 0xD8,
  0xD9, 0xDA, 0xDA, 0xDB, 0xDB,
  0xDC, 0xDD, 0xDD, 0xDE, 0xDE,
  0xDF, 0xE0, 0xE0, 0xE1, 0xE2,
  0xE2, 0xE3, 0xE3, 0xE4, 0xE5,
  0xE5, 0xE6, 0xE6, 0xE7, 0xE8,
  0xE8, 0xE9, 0xE9, 0xEA, 0xEB,
  0xEB, 0xEC, 0xEC, 0xED, 0xED,
  0xEE, 0xEF, 0xEF, 0xF0, 0xF0,
  0xF1, 0xF2, 0xF2, 0xF3, 0xF3,
  0xF4, 0xF4, 0xF5, 0xF6, 0xF6,
  0xF7, 0xF7, 0xF8, 0xF8, 0xF9,
  0xF9, 0xFA, 0xFB, 0xFB, 0xFC,
  0xFC, 0xFD, 0xFD, 0xFE, 0xFE,
  0xFF
};

static struct videc_reg_init init_table_cs7615[] =
{	{0x00, 0x10},
	{0x00, 0x10},
	{0x00, 0x10},
	{0x24, 0x14},
	{0x24, 0x26},
	{0x2B, 0x03},
	{0x2A, 0xF2},
	{0x41, 0x9D},
	{0x4B, 0x4D},
	{0x2C, 0x00},
	{0x29, 0x05}
};
#define NUM_CS7615_REGISTERS	\
		(sizeof(init_table_cs7615)/sizeof(struct videc_reg_init))

static struct videc_reg_init init_table_cs7665[] =
{	{0x00, 0x11},
	{0x06, 0x69},
	{0x03, 0x04},
	{0x0C, 0x07},
	{0x05, 0x04},
	{0x0A, 0x90},
	{0x0B, 0x90},
  {0x08, 0x81},
  {0x09, 0x81},
  {0x10, 0x58},
  {0x11, 0x78},
  {0x12, 0x88},
  {0x13, 0xBB},
  {0x14, 0x90},
  {0x15, 0x88},
  {0x16, 0xA8},
  {0x17, 0xB0},
  {0x18, 0x7F}
};
#define NUM_CS7665_REGISTERS	\
		(sizeof(init_table_cs7665)/sizeof(struct videc_reg_init))

static int cs7615_read_reg(struct videum_device *dev, int reg)
{	int	addr	= dev->videc.i2c_addr1;
	return i2c_read_reg_byte(dev, addr, reg);
}

static int cs7615_write_reg(struct videum_device *dev, int reg, int ndata)
{	int	addr	= dev->videc.i2c_addr1;
	return i2c_write_reg_byte(dev, addr, reg, ndata);
}

static int cs7665_read_reg(struct videum_device *dev, int reg)
{	int	addr	= dev->videc.i2c_addr2;
	return i2c_read_reg_byte(dev, addr, reg);
}

static int cs7665_write_reg(struct videum_device *dev, int reg, int ndata)
{	int	addr	= dev->videc.i2c_addr2;
	return i2c_write_reg_byte(dev, addr, reg, ndata);
}

static int cs7665_write_gamma(struct videum_device *dev, int reg,
				   int		  mask,
				   int		  count,
				   u8		   *buffer)
{	int	 i;
  
  i2c_start(dev);
  i2c_out8(dev, dev->videc.i2c_addr2);	// James,verify addr2 is correct for7665
  if (!i2c_get_ack(dev))
	{
	  i2c_end(dev);
	  return i2c_error(dev);	// no ACK from device
	}
  i2c_out8(dev, reg);
  i2c_get_ack(dev);	// assume it succeeds if the first one succeeded
  i2c_out8(dev, mask);
  i2c_get_ack(dev);	// assume it succeeds if the first one succeeded
  for (i = 0; i < count; ++i)
	{
	  i2c_out8(dev, buffer[i]);
	  if (!i2c_get_ack(dev))
	{
	  i2c_end(dev);
	  return i2c_error(dev);
	}
	}
  i2c_end(dev);
  return 0;
}

static int check_video_synchs(struct videum_device *dev)
{	__u16 CTLreg;
  __u16 block[3] = { 0xFBF4, 0xF9FA, 0xFFF5 };
  int i;

  CTLreg = wavi_readreg( dev, WAVI_CTL );
  wavi_video_sampler(dev, 0);
  wavi_block_write(dev, dev->sram_ucode, (__u8 *) block, 4);
  wavi_writereg(dev, WAVI_SPLPTR, dev->sram_ucode >> 4);
  wavi_video_sampler(dev, 1);
//	1Sec.
	for (i=10000; i-- && ((wavi_readreg(dev,WAVI_CTL) & WAVI_VSTBIT) == 0 ); )
		udelay(10);	// 10 us resolution, good enough
  wavi_video_sampler( dev, 0 );
  return (wavi_readreg( dev, WAVI_CTL ) & WAVI_VSTBIT) != 0;
}

static int ccd_initialize(struct videum_device *dev)
{	int	i,r,j;
	int variable;
	int result;
  
  info_msg("Initializing the CS7665.\n");
  
//	Video decoder information fields
	dev->videc.standards		= (1 << V4L2_STD_NTSC);
  
	for (i = 0; i < NUM_CS7665_REGISTERS; ++i)
	{	if ( init_table_cs7665[i].reg == 0xC )
		{	result = cs7665_write_gamma( dev, init_table_cs7665[i].reg,
					 init_table_cs7665[i].value,
					 sizeof(cs7665_gamma), 
					 cs7665_gamma );
			debug_msg( "Write gamma = %d.\n", result);
			continue;
		}
		if ( init_table_cs7665[i].reg == 0x00 )
		{	cs7665_write_reg(dev, init_table_cs7665[i].reg,
				init_table_cs7665[i].value);
			while( cs7665_read_reg(dev, init_table_cs7665[i].reg) > 0 )
		/* Empty loop. */;
			continue;
		}
	  for( r = 100; r > 0; r-- )
	{
	  /* FIXME: Something is missing here? A video HW check? */
	  
	  cs7665_write_reg(dev,
			   init_table_cs7665[i].reg,
			   init_table_cs7665[i].value);
	  if( (variable = cs7665_read_reg(dev, init_table_cs7665[i].reg) )
		  == init_table_cs7665[i].value )
		break;
	  wnv_sleep(20);
	}
	  if ( r <= 0 )
	{
		err_msg("Why isn't this working 7665 = %d." "reg = %d, value = %d\n", 
			NUM_CS7665_REGISTERS - i, variable, init_table_cs7665[i].value ); 
	}
	}

  variable = cs7665_read_reg( dev, 0x05 );
  if ( variable == -1 ) 
	cs7665_write_reg( dev, 0x05, variable | 0x02 );
	  

  debug_msg("Initializing the CS7615.\n");
  /*  Video decoder information fields	*/
	  
  j = 0;
  while ( j < 5 )
	{
	  for( i = 0; i < NUM_CS7615_REGISTERS; i++ )
	{
	  if ( init_table_cs7615[i].reg == 0x00 )
		{
		  cs7615_write_reg(dev, init_table_cs7615[i].reg,
				init_table_cs7615[i].value);
		  while( cs7615_read_reg(dev, init_table_cs7615[i].reg) > 0 )
		/* Empty loop. */;
		  continue;
		}
	  for( r = 100; r > 0; r-- )
		{
		  /* FIXME: Something is missing here? A video HW check? */
		  
		  cs7615_write_reg(dev,
				init_table_cs7615[i].reg,
				init_table_cs7615[i].value);
		  if( cs7615_read_reg(dev, init_table_cs7615[i].reg)
		  == init_table_cs7615[i].value )
		break;
		  wnv_sleep(20);
		}
	  if( r <= 0 )
		err_msg("Why isn't this working 7615 = %d.\n", 
			NUM_CS7615_REGISTERS - i );
	}
	  if( check_video_synchs( dev ) )
	j = 6;
	  else
	j++;
	}
  
  return 1;
}
  
static int ccd_set_standard(struct videum_device *dev, int x)
{	info_msg("ccd_set_standard type %d\n", x);

	dev->videc.frame_period = (x==V4L2_STD_NTSC) ? 333667 : 400000;
	dev->videc.standard = x;
	return 1;
}

// Probe I2C bus for video decoder and fill in the device fields
static int ccd_probe(struct videum_device *dev)
{	int	x;

	if (dev->videc.i2c_addr1 != 0)
		return 0;/*  Another device was already found  */
	dev->videc.i2c_addr1 = CS7615_ADDRESS0;
	dev->videc.i2c_addr2 = CS7665_ADDRESS;
	if (-1 == (x = cs7615_read_reg(dev, 0)))	// Version
	{	dev->videc.i2c_addr1 = CS7615_ADDRESS1;
		if (-1 == (x = cs7615_read_reg(dev, 0x0)))
		{	dev->videc.i2c_addr1 = 0;
			dev->videc.i2c_addr2 = 0;
	}	}
	if (dev->videc.i2c_addr2)
		if (-1 == (x = cs7665_read_reg( dev, CS7665_VERSION_REG )))
		{	dev->videc.i2c_addr1 = 0;
			dev->videc.i2c_addr2 = 0;
			return 0;/*  Not found  */
		}
	dev->videc.version = x;

	info_msg("Found CS7615/CS7665 (ver.%d) CCDs at %2X and %2X\n",
		 dev->videc.version, dev->videc.i2c_addr1, dev->videc.i2c_addr2 );

	return 1;/*  Found  */
}

static void set_video_input(struct videum_device *dev, int i)
{	if (i)
		return;
	wavi_vsc(dev, 0);
	wavi_vss(dev, 1);
	wavi_vhsd(dev, 4);
	wavi_tone_controls(dev);
}

//	W A V I   V I D E O   C A P T U R E   M I C R O C O D E

//	Video line capture modes
#define VMODE_NORMAL	0  /* Capture incoming pixels to VID1PTR */
#define VMODE_AVGBUF	1  /* Capture incoming pixels to VID2PTR */
#define VMODE_AVG	2  /* Average incoming with VID2PTR to VID1PTR */
#define VMODE_SKIP	(~0)  /* Fake mode: Skip whole line */

struct uc_buffer
{	__u8	*p;	/* Current position	*/
	int	len;	/* Bytes remaining	*/
};

static void compile_repeat(struct uc_buffer *buf, __u8	ui, int	n)
{	if (buf->len < n)
		n = buf->len;
	buf->len -= n;
	while (n--)
		*(buf->p)++ = ui;
}

static void compile_start_of_field(struct uc_buffer	*buf, int nfield, int vskip)
{	if (vskip < 0 || vskip > 256)
		vskip = 0;
	if (buf->len < 4 + vskip)
		return;

	*(buf->p)++ = UI_WTVB;
	if (nfield == 1)
		*(buf->p)++ = UI_WTVF1;
	else
		*(buf->p)++ = UI_WTVF2;
	*(buf->p)++ = UI_WTVB;
	*(buf->p)++ = UI_WTVA;
	compile_repeat(buf, UI_WTHS, vskip);
	buf->len -= 4 + vskip;
}

static void compile_interrupt(struct uc_buffer *buf, __u8 ui_interrupt, int	flags)
{	if (buf->len < 3)
		return;

	*(buf->p)++ = ui_interrupt;
	--buf->len;
	if (flags & COMPFLAG_AVPRO_INT)
	{
		*(buf->p)++ = UI_SVS;
		*(buf->p)++ = UI_RVS;
		buf->len -= 2;
}	}

static void compile_dma_trigger(struct uc_buffer *buf, int flags)
{	if (buf->len < 3)
		return;

	if (flags & COMPFLAG_AVPRO_DMA)
	{	*(buf->p)++ = UI_SVS;
		*(buf->p)++ = UI_SVS;
		*(buf->p)++ = UI_RVS;
		buf->len -= 3;
}	}

static void compile_set_videomode(struct uc_buffer *buf, int *videomode,
			  int	newmode)
{	int	n = (newmode - *videomode) & 3;

	if (n == 0 || buf->len < n)
		return;

	buf->len -= n;
	while (n--)
		*(buf->p)++ = UI_SVM;
	*videomode = newmode;
}

static void compile_xcrop(struct uc_buffer *buf, int x)
{	if (x < 1 || buf->len < 2)
		return;

	if (x > 960)
		x = 960;
	*((__u16 *)buf->p)++ = UI_SKIP_LOWPREC(x);
	buf->len -= 2;
}

static void compile_horizontal_decimation(int source_width, int capture_width,
	int	*hrate, int	*hclocks, int flags)
{	int npx;

	if (!source_width)
		return;
	*hclocks = source_width;
	*hrate = (capture_width << 6) / source_width;
	if (*hrate < 2)
		*hrate = 2;

//	Force the decimation ratio up if not enough pixels captured
	for (npx=0; npx < capture_width; (*hrate)++)
		npx = (*hclocks * *hrate + 32) >> 6;

//	Force the number of clocks down if too many pixels captured
	while (npx > capture_width)
		npx = ((--(*hclocks) * *hrate) + 32) >> 6;

//	Attempt to work around WAVI video sampler bugs:
//	Capture for max number of clocks that take the number of pixels we want
	while (npx == (((*hclocks + 1) * *hrate + 32) >> 6))
		++(*hclocks);
}

static void compile_line(struct uc_buffer *buf, int	hskip, __u16 ui_capture1,
	__u16 ui_capture2, int *videomode, int vmode)
{	int x;

	if (buf->len < 15)/*  No room at the inn...  */
		return;

	if (vmode == VMODE_SKIP)/*  Don't capture any pixels on this line  */
	{
		*(buf->p)++ = UI_WTHS;
		--buf->len;
		return;
	}

	compile_set_videomode(buf, videomode, vmode);

	*(buf->p)++ = UI_WTHS;
	--buf->len;

	x = hskip;
	if (vmode == VMODE_AVGBUF || vmode == VMODE_AVG)
	{/*	Rewind VID2PTR before capturing this line	*/
		*(buf->p)++ = UI_RVP2;
		--buf->len;
		--x;
	}

	compile_xcrop(buf, x);

	*((__u16 *)buf->p)++ = ui_capture1;
	buf->len -= 2;
	if (ui_capture2)
	{
		*((__u16 *)buf->p)++ = ui_capture2;
		buf->len -= 2;
	}
}

static void compile_field(struct uc_buffer *buf, int source_height,
		  int	capture_height, int	hskip, __u16 ui_capture1,
		  __u16	ui_capture2, int interrupt_line, int dma_line, int flags)
{	int	a, m;
	int	source_line_count	= 0;
	int	captured_line_count	= 0;
	int	avg_buf_full		= 0;
	int	videomode		= 0;

	a = source_height / 2;
	while (captured_line_count < capture_height)
	{
		if (a >= source_height)
		{/*	Capture this line to output buffer	*/
			a -= source_height;
			++captured_line_count;
			m = (avg_buf_full) ? VMODE_AVG : VMODE_NORMAL;
			avg_buf_full = 0;
		}
		else
		{/*	Capture to average buffer or just skip	*/
			if (avg_buf_full || (flags & COMPFLAG_NOVERTFILTER))
				m = VMODE_SKIP;
			else
			{
				m = VMODE_AVGBUF;
				avg_buf_full = 1;
			}
		}

		compile_line(buf, hskip, ui_capture1, ui_capture2,
				 &videomode, m);

		++source_line_count;
		if (source_line_count == dma_line)
			compile_dma_trigger(buf, flags);
		if (source_line_count == interrupt_line)
			compile_interrupt(buf, UI_ITR1, flags);

		a += capture_height;
	}
	compile_set_videomode(buf, &videomode, 0);
	if ((flags & COMPFLAG_HUFFMAN) && buf->len > 1)
	{
		*((__u16 *)buf->p)++ = UI_DECIMATE_LOWPREC(32, 64);
		buf->len -= 2;
	}

	while ( source_line_count < interrupt_line ||
		source_line_count < dma_line)
	{
		compile_line(buf, 0, 0, 0, 0, VMODE_SKIP);
		++source_line_count;
		if (source_line_count == dma_line)
			compile_dma_trigger(buf, flags);
		if (source_line_count == interrupt_line)
			compile_interrupt(buf, UI_ITR1, flags);
}	}

static int/*  Number of bytes of code generated  */
compile_microcode(struct videum_device *dev, struct ucode_parms *parm)
{	struct uc_buffer buffer;
	int	hrate;
	int	hclocks;
	int	field1		= 1;
	__u16	ui_capture1	= 0;
	__u16	ui_capture2	= 0;

	buffer.p = parm->buffer;
	buffer.len = UC_SIZE;

	if (parm->flags & COMPFLAG_VST_WHEN_DONE)
	{
		*(buffer.p)++ = UI_RVS;
		--buffer.len;
	}

	compile_horizontal_decimation(parm->source_width, parm->capture_width,
					  &hrate, &hclocks, parm->flags);
	if (!((parm->flags & COMPFLAG_HUFFMAN) && hrate > 32) ||
		 ((parm->flags & COMPFLAG_HUFFMAN) && hrate > 19))
		parm->flags |= COMPFLAG_NOVERTFILTER;
	if (hclocks > 960)
	{	ui_capture1 = UI_DECIMATE_LOWPREC(hrate, hclocks/2);
		ui_capture2 = UI_DECIMATE_LOWPREC(hrate, hclocks - hclocks/2);
	}
	else
		ui_capture1 = UI_DECIMATE_LOWPREC(hrate, hclocks);

//	Center the actual sample range within desired sample range.
	if (hclocks > parm->source_width)
		hclocks = parm->source_width;
	parm->source_x += (parm->source_width - hclocks) / 2;

//	Horizontal filter
	parm->filter = 0;
	if (hrate < 48)
		parm->filter |= WAVI_VFS_MED3;	// 3 tap median filter
	if (hrate < 16)
		parm->filter |= WAVI_VFS_FIR4;	// 4 pix FIR filter
	else if (hrate < 32)
		parm->filter |= WAVI_VFS_FIR2;	// 2 pix FIR filter

	field1 = parm->field;
	compile_start_of_field(&buffer, field1, parm->source_y);
	compile_field(&buffer,  
			  parm->source_height, parm->capture_height,
			  parm->source_x, ui_capture1, ui_capture2,
			  parm->int1_line, parm->dma_line,
			  parm->flags);

	if (buffer.len < 16)
	{	err_msg("Microcode buffer too small.\n");
		return 0;/*  Ran out of buffer! Oh, no!  */
	}

	if (parm->flags & COMPFLAG_VST_WHEN_DONE)
	{	*(buffer.p)++ = UI_SVS;
		--buffer.len;
	}
	*(buffer.p)++ = UI_HLT;
	--buffer.len;
	return ((UC_SIZE - buffer.len) + 3) & ~3;
}

//	I M A G E   F O R M A T   T R A N S L A T I O N

static int translate_yuyv_grey(struct translation *xlat)
{	register __u8 *esi = xlat->in;
	register __u8 *edi = xlat->out;
	register long p = xlat->height * (long) xlat->width;

	for ( ; p--; esi++)
		*(edi++) = *(esi++);
	return 1;
}

static int translate_yuyv_yuv420(struct translation *xlat)
{	int row;
	const int eax = xlat->width << 1;
	register __u8 *esi = xlat->in;
	register __u8 *edi = xlat->out;
	register long p;

//	Y plane
	for (p=xlat->height *(long)xlat->width; p--; esi++)
		*(edi++) = *(esi++);

//	U plane
	for (esi=xlat->in+1, row=xlat->height>>1; row--; esi+=eax)
		for (p=xlat->width>>1; p--; esi+=4)
			*(edi++) = *esi;

//	V plane
	for (esi=xlat->in+3, row=xlat->height>>1; row--; esi+=eax)
		for (p=xlat->width>>1; p--; esi+=4)
			*(edi++) = *esi;
	return 1;
}

static int translate_yuyv_yuv422p(struct translation *xlat)
{	register __u8 *esi = xlat->in;
	register __u8 *edi = xlat->out;
	register long p;

//	Y plane
	for (p=xlat->height *(long)xlat->width; p--; esi++)
		*(edi++) = *(esi++);

//	U plane
	esi=xlat->in+1;
	for (p=xlat->height *(long)xlat->width; p--; esi+=4)
		*(edi++) = *esi;

//	V plane
	esi=xlat->in+3;
	for (p=xlat->height *(long)xlat->width; p--; esi+=4)
		*(edi++) = *esi;

	return 1;
}

#define K12_1	 4096
#define K12_S	   12
#define K12_GU	-1409
#define K12_BU	 7258
#define K12_RV	 5743
#define K12_GV	-2925

static int translate_expand_y(int y)
{	y = (255 * y + 110) / 220;
	if (y < 0) y = 0; else if (y > 255) y = 255;
	return y;
}
static int translate_expand_c(int c)
{	c = (127 * c + 56) / 112;
	if (c < -128) c = -128; else if (c > 127) c = 127;
	return c;
}

static int translate_make_rgb16_lut(struct translation *xlat)
{	__u16	*lut;
	long	gu[32], bu[32], rv[32], gv[32];
	int	rscale[256], gscale[256], bscale[256];
	int	rrange, grange, brange;
	int	rshift, gshift, bshift;
	long	x;
	int	y, u, v;
	int	r, g, b;
	int	i, t;

	if ((xlat->type == XLAT_YUYV_TO_RGB555 &&
		 xlat->lut[0].type == LUT_RGB555) ||
		(xlat->type == XLAT_YUYV_TO_RGB565 &&
		 xlat->lut[0].type == LUT_RGB565))
		return 1;

	if (xlat->lut[0].table.base)
		vfree(xlat->lut[0].table.base);
	xlat->lut[0].table.base = vmalloc(1 << 17);
	if (xlat->lut[0].table.base == NULL)
	{	err_msg("vmalloc() failed in make_rgb16_lut\n");
		return 0;
	}
	lut = xlat->lut[0].table.rgb16;

	//  Compute all different chroma components to 8-bit precision
	for (i = 0, t = -128; t < 128; t += 8, ++i)
	{
		x = translate_expand_c(t) + 2;
		gu[i] = (K12_GU * x + K12_1/2) >> K12_S;
		bu[i] = (K12_BU * x + K12_1/2) >> K12_S;
		rv[i] = (K12_RV * x + K12_1/2) >> K12_S;
		gv[i] = (K12_GV * x + K12_1/2) >> K12_S;
	}
	//  8-bit to ?-bit scaling tables
	if (xlat->type == XLAT_YUYV_TO_RGB555)
	{
		xlat->lut[0].type = LUT_RGB555;
		rrange = grange = brange = 31;
		rshift = 10; gshift = 5; bshift = 0;
	}
	else
	{
		xlat->lut[0].type = LUT_RGB565;
		rrange = brange = 31;
		grange = 63;
		rshift = 11; gshift = 5; bshift = 0;
	}
	for (i = 0; i < 256; ++i)
	{
		rscale[i] = ((i * rrange + 127) / 255) << rshift;
		gscale[i] = ((i * grange + 127) / 255) << gshift;
		bscale[i] = ((i * brange + 127) / 255) << bshift;
	}

	//  Fill in the RGB values for each combination of YUV
	for (i = 0; i < 256; i += 4)
	{
		y = translate_expand_y(i) + 2;
		if (y > 255) y = 255;
		for (u = 0; u < 32; ++u)
			for (v = 0; v < 32; ++v)
			{
				//  Red, Green and Blue
				r = y + rv[v];
				g = y + gu[u] + gv[v];
				b = y + bu[u];
				//  Saturate
				if (r < 0) r = 0; else if (r > 254) r = 254;
				if (g < 0) g = 0; else if (g > 254) g = 254;
				if (b < 0) b = 0; else if (b > 254) b = 254;
				//  scale, shift and combine
				*lut++ = rscale[r] + gscale[g] + bscale[b];
			}
	}
	return 1;
}

static int translate_yuyv_rgb16(struct translation *xlat)
{	__u32	uv, yuv0, yuv1, dual;
	__u16	*lut;
	int	row, i;
	__u32	*src = (__u32 *)xlat->in;
	__u32	*dst = (__u32 *)xlat->out;

	if (!translate_make_rgb16_lut(xlat))
		return 0;

	lut = xlat->lut[0].table.rgb16;
	for (row = xlat->height; row; --row)
		for (i = xlat->width >> 1; i; --i)
		{	dual = *src++;
			uv   =	((dual & 0x0000F800) >> 6)
				+ ((dual & 0xF8000000) >> 27);
			yuv1 = 	  ((dual & 0x00FC0000) >> 8) + uv;
			yuv0 = 	  ((dual & 0x000000FC) << 8) + uv;
			*dst++ = ((__u32)lut[yuv1] << 16) | lut[yuv0];
		}
	return 1;
}

static int translate_make_rgb24_lut(struct translation *xlat)
{	struct lookup_rgb24	*lut;
	int			r, g, b;
	int			i;
	int			x;

	if (xlat->lut[0].type == LUT_RGB24)
		return 1;

	if (xlat->lut[0].table.base)
		vfree(xlat->lut[0].table.base);
	xlat->lut[0].table.base = vmalloc(sizeof(struct lookup_rgb24));
	if (xlat->lut[0].table.base == NULL)
	{	err_msg("vmalloc() failed in make_rgb24_lut\n");
		return 0;
	}
	xlat->lut[0].type = LUT_RGB24;
	lut = xlat->lut[0].table.rgb24;

	for (i = 0; i < 256; ++i)
	{
		x = i;		// Value is in excess-128 format
		if (x < 128)
			++x;	// Add 1 to negative values for noise rejection
		x -= 128;	// Convert to two's complement format
		x = translate_expand_c(x);

		g = (K12_GU * x + K12_1/2) >> K12_S;
		b = (K12_BU * x + K12_1/2) >> K12_S;
		lut->u_rgb[i] = ((g & 0x3FF) << 11) | (b & 0x3FF);

		r = (K12_RV * x + K12_1/2) >> K12_S;
		g = (K12_GV * x + K12_1/2) >> K12_S;
		lut->v_rgb[i] = (r << 22) | ((g & 0x3FF) << 11);
				
		x = translate_expand_y(i);
		lut->y_rgb[i] = (x << 22) | (x << 11) | x;
	}
	for (i = 0; i < 1024; ++i)
	{
		x = (i > 511) ? 0 : ((i > 255) ? 255 : i);
		lut->sat[i] = x;
	}
	return 1;
}

static int translate_yuyv_rgb24(struct translation *xlat)
{	struct lookup_rgb24	*lut;
	int			i;
	int			row;
	__u32			pela, pelb, pelc, peld;
	__u8 *src = (__u8 *)xlat->in;
	__u32 *dst = (__u32 *)xlat->out;

	if (!translate_make_rgb24_lut(xlat))
		return 0;

	lut = xlat->lut[0].table.rgb24;
	for (row = xlat->height; row; --row)
		for (i = xlat->width >> 2; i; --i)
		{
			pelb  = lut->u_rgb[src[1]]
				  + lut->v_rgb[src[3]];
			pela  = lut->y_rgb[src[0]] + pelb;
			pelb += lut->y_rgb[src[2]];
			peld  = lut->u_rgb[src[5]]
				  + lut->v_rgb[src[7]];
			pelc  = lut->y_rgb[src[4]] + peld;
			peld += lut->y_rgb[src[6]];
			src += 8;

			dst[0] = ((u32)lut->sat[pela & 0x3FF])
				   + ((u32)lut->sat[(pela >> 11) & 0x3FF] << 8)
				   + ((u32)lut->sat[pela >> 22] << 16)
				   + ((u32)lut->sat[pelb & 0x3FF] << 24);
			dst[1] = ((u32)lut->sat[(pelb >> 11) & 0x3FF])
				   + ((u32)lut->sat[pelb >> 22] << 8)
				   + ((u32)lut->sat[pelc & 0x3FF] << 16)
				   + ((u32)lut->sat[(pelc >> 11) & 0x3FF] << 24);
			dst[2] = ((u32)lut->sat[pelc >> 22])
				   + ((u32)lut->sat[peld & 0x3FF] << 8)
				   + ((u32)lut->sat[(peld >> 11) & 0x3FF] << 16)
				   + ((u32)lut->sat[peld >> 22] << 24);
			dst += 3;
		}
	return 1;
}

static int translate_yuyv_rgb32(struct translation *xlat)
{	struct lookup_rgb24	*lut;
	__u8			*src;
	__u32			*dst;
	int			i;
	int			row;
	__u32			pela, pelb;
	src = (__u8 *)xlat->in;
	dst = (__u32 *)xlat->out;

	if (!translate_make_rgb24_lut(xlat))
		return 0;
	lut = xlat->lut[0].table.rgb24;
	for (row = xlat->height; row; --row)
		for (i = xlat->width >> 1; i; --i)
		{
			pelb  = lut->u_rgb[src[1]]
				  + lut->v_rgb[src[3]];
			pela  = lut->y_rgb[src[0]] + pelb;
			pelb += lut->y_rgb[src[2]];
			src += 4;
			
			dst[0] =  (u32)lut->sat[pela & 0x3FF]
				   + ((u32)lut->sat[(pela >> 11) & 0x3FF] << 8)
				   + ((u32)lut->sat[pela >> 22] << 16);
			dst[1] =  (u32)lut->sat[pelb & 0x3FF]
				   + ((u32)lut->sat[(pelb >> 11) & 0x3FF] << 8)
				   + ((u32)lut->sat[pelb >> 22] << 16);
			dst += 2;
		}
	return 1;
}

static void translate_close(struct videum_device *dev)
{	dev->translation2.type = XLAT_NULL;
	dev->translation2.in = NULL;
	dev->translation2.out = NULL;
	dev->translation2.lut[0].type = LUT_NULL;
	if (dev->translation2.lut[0].table.base)
		vfree(dev->translation2.lut[0].table.base);
	dev->translation2.lut[0].table.base = NULL;

	if (dev->xlat_temp)
		vfree(dev->xlat_temp);
	dev->xlat_temp = NULL;
}

//	Translation 2: YUYV to client format.
static int translate_setup(struct videum_device *dev)
{	translate_close(dev);

	dev->translation2.width = dev->uc_parm.capture_width;
	dev->translation2.height = dev->uc_parm.capture_height;
#ifdef V4L1
	dev->translation2.output_size =
		(dev->vBuf.width * dev->vBuf.height * dev->vPict.depth) >> 3;
	switch (dev->vPict.palette)
	{	case VIDEO_PALETTE_YUYV:
			dev->translation2.type = XLAT_NULL;
		break;
		case VIDEO_PALETTE_GREY:
			dev->translation2.type = XLAT_YUYV_TO_GREY;
		break;
		case VIDEO_PALETTE_YUV422P:
			dev->translation2.type = XLAT_YUYV_TO_YUV422P;
		break;
		case VIDEO_PALETTE_YUV420:
			dev->translation2.type = XLAT_YUYV_TO_YUV420;
		break;
		case VIDEO_PALETTE_RGB555:
			dev->translation2.type = XLAT_YUYV_TO_RGB555;
			if (!translate_make_rgb16_lut(&dev->translation2))
				return 0;
		break;
		case VIDEO_PALETTE_RGB565:
			dev->translation2.type = XLAT_YUYV_TO_RGB565;
			if (!translate_make_rgb16_lut(&dev->translation2))
				return 0;
		break;
		case VIDEO_PALETTE_RGB24:
			dev->translation2.type = XLAT_YUYV_TO_RGB24;
			if (!translate_make_rgb24_lut(&dev->translation2))
				return 0;
		break;
		case VIDEO_PALETTE_RGB32:
			dev->translation2.type = XLAT_YUYV_TO_RGB32;
			if (!translate_make_rgb24_lut(&dev->translation2))
				return 0;
		break;
	}
#else
	dev->translation2.output_size = dev->clientfmt.fmt.pix.sizeimage;
	switch (dev->clientfmt.fmt.pix.pixelformat)
	{	case V4L2_PIX_FMT_YUYV:
			dev->translation2.type = XLAT_NULL;
		break;
		case V4L2_PIX_FMT_GREY:
			dev->translation2.type = XLAT_YUYV_TO_GREY;
		break;
		case V4L2_PIX_FMT_YVU422P:
			dev->translation2.type = XLAT_YUYV_TO_YUV422P;
		break;
		case V4L2_PIX_FMT_YUV420:
			dev->translation2.type = XLAT_YUYV_TO_YUV420;
		break;
		case V4L2_PIX_FMT_RGB555:
			dev->translation2.type = XLAT_YUYV_TO_RGB555;
			if (!translate_make_rgb16_lut(&dev->translation2))
				return 0;
		break;
		case V4L2_PIX_FMT_RGB565:
			dev->translation2.type = XLAT_YUYV_TO_RGB565;
			if (!translate_make_rgb16_lut(&dev->translation2))
				return 0;
		break;
		case V4L2_PIX_FMT_BGR24:
			dev->translation2.type = XLAT_YUYV_TO_RGB24;
			if (!translate_make_rgb24_lut(&dev->translation2))
				return 0;
		break;
		case V4L2_PIX_FMT_BGR32:
			dev->translation2.type = XLAT_YUYV_TO_RGB32;
			if (!translate_make_rgb24_lut(&dev->translation2))
				return 0;
		break;
	}
#endif
	return 1;
}

//	Translation 2: YUYV to client format
static void translate_inandout(struct videum_device *dev, __u8 *input_buffer,
		   __u8 *output_buffer, __u8 output_is_user_space)
{	dev->translation2.in = input_buffer;
	dev->translation2.out = output_buffer;
	dev->translation2.output_is_user = output_is_user_space;
}

//	Length of output image or negative error
static int translate_image(struct videum_device *dev, __u8	*input_buffer,
		__u8	*output_buffer, int	len, int output_is_user)
{	int	err;

	/* The buffer must be large enough for the whole image */
	if (len < dev->translation2.output_size)
	{
		debug_msg("Read buffer too small, %d < %d\n",
			  len, dev->translation2.output_size);
		return -EFAULT;
	}
	if (len > dev->translation2.output_size)
		len = dev->translation2.output_size;

	translate_inandout(dev, input_buffer, output_buffer, output_is_user);

//	Translation 2: YUYV to client format
	if (dev->translation2.in == dev->translation2.out)
		return len;

	if (dev->translation2.type == XLAT_NULL)
	{	if (output_is_user)
		{	err = copy_to_user(output_buffer, 
					   dev->translation2.in, len);
			len = (err) ? -EFAULT : len;
		} else
			memcpy(output_buffer, dev->translation2.in, len);
		return len;
	}

	if (output_is_user && !access_ok(VERIFY_WRITE, output_buffer, len))
		return -EFAULT;

	switch (dev->translation2.type)
	{
	case XLAT_YUYV_TO_GREY:
		translate_yuyv_grey(&dev->translation2);
		break;
	case XLAT_YUYV_TO_YUV420:
		translate_yuyv_yuv420(&dev->translation2);
		break;
	case XLAT_YUYV_TO_YUV422P:
		translate_yuyv_yuv422p(&dev->translation2);
		break;
	case XLAT_YUYV_TO_RGB555:
	case XLAT_YUYV_TO_RGB565:
		translate_yuyv_rgb16(&dev->translation2);
		break;
	case XLAT_YUYV_TO_RGB24:
		translate_yuyv_rgb24(&dev->translation2);
		break;
	case XLAT_YUYV_TO_RGB32:
		translate_yuyv_rgb32(&dev->translation2);
		break;
	}
	dev->translation2.out = NULL;
	return len;
}

//	V I D E O   C A P T U R E   F U N C T I O N S

// Supported capture formats
#ifndef V4L1
static struct v4l2_fmtdesc capfmt[] = 
{	{ 0, {"RGB-16 (5-5-5)"}, V4L2_PIX_FMT_RGB555,  0, 16, {0, 0}, },
	{ 1, {"RGB-16 (5-6-5)"}, V4L2_PIX_FMT_RGB565,  0, 16, {0, 0}, },
	{ 2, {"RGB-24 (B-G-R)"}, V4L2_PIX_FMT_BGR24,   0, 24, {0, 0}, },
	{ 3, {"RGB-32 (B-G-R-?)"}, V4L2_PIX_FMT_BGR32,   0, 32, {0, 0}, },
	{ 4, {"Greyscale-8"}, V4L2_PIX_FMT_GREY,	V4L2_FMT_CS_601YUV, 8, {0, 0}, },
	{ 5, {"YUV 4:2:2 (Y-U-Y-V)"}, V4L2_PIX_FMT_YUYV,	V4L2_FMT_CS_601YUV, 16, {0, 0}, },
	{ 6, {"YUV 4:2:0 (planar)"}, V4L2_PIX_FMT_YUV420,  V4L2_FMT_CS_601YUV, 12, {0, 0}, },
	{ 7, {"YUV 4:2:2 (planar)"}, V4L2_PIX_FMT_YVU422P,  V4L2_FMT_CS_601YUV, 16, {0, 0}, },
};
#define NUM_CAPFMT (sizeof(capfmt)/sizeof(capfmt[0]))
#endif
static void interrupt_enable(struct videum_device *dev);

// The image format has changed, width, height, pixel format.
// Decide if the format is ok or take the closest valid format.
static void capture_new_format(struct videum_device *dev)
{	int	t;
	int max_width = dev->uc_parm.source_width;
	int max_height = dev->uc_parm.source_height;
	const int max_pix = dev->sram_video_size >> 1; // capture_bpp=16
	if (dev->stream_buffers_mapped)
		return;
	dev->ready_to_capture = 0;

	dev->uc_parm.source_x = 58;
	dev->uc_parm.source_y = 20;
	dev->uc_parm.source_width = MAX_WIDTH;
	dev->uc_parm.source_height = MAX_HEIGHT;
	dev->uc_parm.field = 1; // + dev->videc.preferred_field;
#ifdef V4L1
	switch (dev->vPict.palette)
	{	case VIDEO_PALETTE_GREY:
			dev->vPict.depth = 8;
		break;
		case VIDEO_PALETTE_YUV420:
			dev->vPict.depth = 12;
		break;
		case VIDEO_PALETTE_RGB555:
		case VIDEO_PALETTE_RGB565:
		case VIDEO_PALETTE_YUYV:
		case VIDEO_PALETTE_YUV422P:
			dev->vPict.depth = 16;
		break;
		case VIDEO_PALETTE_RGB24:
			dev->vPict.depth = 24;
		break;
		case VIDEO_PALETTE_RGB32:
			dev->vPict.depth = 32;
		break;
		default:
			debug_msg("Unknown format\n");
			dev->vPict.depth = 16;
			dev->vPict.palette = VIDEO_PALETTE_RGB565;
		break;
	}
	if (dev->vBuf.width < MIN_WIDTH)
		dev->vBuf.width = MIN_WIDTH;
	if (dev->vBuf.height < MIN_HEIGHT)
		dev->vBuf.height = MIN_HEIGHT;
	dev->vBuf.width &= ~3;
	dev->vBuf.height &= ~3;

	t = isqrt((max_pix * dev->vBuf.width) / dev->vBuf.height);
	if (t < max_width)
		max_width = t;
	t = isqrt((max_pix * dev->vBuf.height) / dev->vBuf.width);
	if (t < max_height)
		max_height = t;

	if (dev->vBuf.width > max_width)
		dev->vBuf.width = max_width;
	if (dev->vBuf.height > max_height)
		dev->vBuf.height = max_height;
	dev->vBuf.width &= ~3;
	dev->vBuf.height &= ~3;

//	Normal microcode
	dev->uc_parm.capture_width = dev->vBuf.width;
	dev->uc_parm.capture_height = dev->vBuf.height;
#else
	dev->clientfmt.fmt.pix.flags = V4L2_FMT_CS_601YUV;
	switch (dev->clientfmt.fmt.pix.pixelformat)
	{	case V4L2_PIX_FMT_GREY:
			dev->clientfmt.fmt.pix.depth = 8;
		break;
		case V4L2_PIX_FMT_YUV420:
			dev->clientfmt.fmt.pix.depth = 12;
		break;
		case V4L2_PIX_FMT_RGB555:
		case V4L2_PIX_FMT_RGB565:
			dev->clientfmt.fmt.pix.flags = 0;
		case V4L2_PIX_FMT_YUYV:
		case V4L2_PIX_FMT_YVU422P:
			dev->clientfmt.fmt.pix.depth = 16;
		break;
		case V4L2_PIX_FMT_BGR24:
			dev->clientfmt.fmt.pix.depth = 24;
			dev->clientfmt.fmt.pix.flags = 0;
		break;
		case V4L2_PIX_FMT_BGR32:
			dev->clientfmt.fmt.pix.depth = 32;
			dev->clientfmt.fmt.pix.flags = 0;
		break;
		default:
			debug_msg("Unknown format\n");
			dev->clientfmt.fmt.pix.depth = 16;
			dev->clientfmt.fmt.pix.pixelformat = V4L2_PIX_FMT_RGB565;
			dev->clientfmt.fmt.pix.flags = 0;
		break;
	}
	if (dev->clientfmt.fmt.pix.width < MIN_WIDTH)
		dev->clientfmt.fmt.pix.width = MIN_WIDTH;
	if (dev->clientfmt.fmt.pix.height < MIN_HEIGHT)
		dev->clientfmt.fmt.pix.height = MIN_HEIGHT;
	dev->clientfmt.fmt.pix.width &= ~3;
	dev->clientfmt.fmt.pix.height &= ~3;

	t = isqrt((max_pix * dev->clientfmt.fmt.pix.width) / dev->clientfmt.fmt.pix.height);
	if (t < max_width)
		max_width = t;
	t = isqrt((max_pix * dev->clientfmt.fmt.pix.height) / dev->clientfmt.fmt.pix.width);
	if (t < max_height)
		max_height = t;

//printk("New_FMT: W=%d, H=%d\n", dev->clientfmt.fmt.pix.width, dev->clientfmt.fmt.pix.height);
	if (dev->clientfmt.fmt.pix.width > max_width)
		dev->clientfmt.fmt.pix.width = max_width;
	if (dev->clientfmt.fmt.pix.height > max_height)
		dev->clientfmt.fmt.pix.height = max_height;
	dev->clientfmt.fmt.pix.width &= ~3;
	dev->clientfmt.fmt.pix.height &= ~3;
//printk("New_FMT: W=%d, H=%d\n", dev->clientfmt.fmt.pix.width, dev->clientfmt.fmt.pix.height);

	dev->clientfmt.fmt.pix.sizeimage =
		(dev->clientfmt.fmt.pix.width * dev->clientfmt.fmt.pix.height *
		dev->clientfmt.fmt.pix.depth) >> 3;

//	Normal microcode
	dev->uc_parm.capture_width = dev->clientfmt.fmt.pix.width;
	dev->uc_parm.capture_height = dev->clientfmt.fmt.pix.height;
#endif
	dev->capture_size = 2 *
		dev->uc_parm.capture_width * dev->uc_parm.capture_height;
}

//	Stop the music!
static void capture_abort(struct videum_device *dev)
{	dev->sampler_enabled = 0;
	wavi_video_sampler(dev, 0);
}

static void
capture_int_line(struct videum_device *dev, struct ucode_parms *parm)
{	int	height	= parm->source_height;
	int	rate;
	int	line	= height;

	parm->int1_line = height;
	if (height == 0)
		return;
	{/*  Non-DMA case */
		if (dev->slave_rate == 0)
			return;
		rate = dev->slave_rate / 15625;
		line = dev->capture_size / rate;
		line = height - line;
		if (line < 1)
			line = 1;
		if (line > height)
			line = height;
	}
//debug_msg("Interrupt line %d\n",line);
	parm->int1_line = line;
}

static int compile_motion_microcode(struct videum_device *dev, int	uc)
{	capture_int_line(dev, &dev->uc_parm);
	dev->uc_parm.int2_line = 0;

	dev->uc_parm.flags =  0;
	dev->uc_parm.flags |= COMPFLAG_VST_WHEN_DONE;

	dev->uc_loaded = UC_UNDEF;
	dev->uc_parm.buffer = dev->uc_buffer[uc];
	dev->uc_length[uc] =
		compile_microcode(dev, &dev->uc_parm);
	if (dev->uc_length[uc] == 0)
	{/* ucode too long, no vertical filter reduces the length */
		dev->uc_parm.flags |= COMPFLAG_NOVERTFILTER;
		dev->uc_length[uc] = compile_microcode(
			dev, &dev->uc_parm);
		if (dev->uc_length[uc] == 0)
		{
			err_msg("Internal error - can't compile ucode\n");
			return 0;/* failed */
		}
	}
	return 1;/* succeeded */
}

//	Allocate buffers, compile microcode, and get everything ready to capture
//	an image, but don't start capturing yet.
static int capture_begin(struct videum_device *dev)
{	capture_abort(dev);
	if (dev->ready_to_capture)
		return dev->ready_to_capture;

	if (dev->capture_buffer_size < dev->capture_size)
	{
		if (dev->capture_buffer != NULL)
			vfree(dev->capture_buffer);
		dev->capture_buffer_size = 
			(dev->capture_size + PAGE_SIZE - 1) & ~(PAGE_SIZE - 1);
		dev->capture_buffer = (__u8 *)
			vmalloc(dev->capture_buffer_size);
		if (dev->capture_buffer == NULL)
		{	dev->capture_buffer_size = 0;
			err_msg("Can't allocate capture buffer"
				" %d bytes\n", dev->capture_size);
			return dev->ready_to_capture;
	}	}

	if (!compile_motion_microcode(dev, UC_NORMAL))
		return dev->ready_to_capture;

	if (!translate_setup(dev))
		return dev->ready_to_capture;

	interrupt_enable(dev);
	return (dev->ready_to_capture = 1);
}

//	Simple queue management
void video_q_init(struct video_queue *q)
{	if (q == NULL)
		return;
	q->qlock = RW_LOCK_UNLOCKED;
	q->forw = (struct video_q_node *)q;
	q->back = (struct video_q_node *)q;
}

void video_q_add_head(struct video_queue *q, struct video_q_node *node)
{
	unsigned long flags;
	if (q == NULL || node == NULL)
		return;
	if (q->forw == NULL || q->back == NULL)
		video_q_init(q);
	write_lock_irqsave(&(q->qlock), flags);
	node->forw = q->forw;
	node->back = (struct video_q_node *)q;
	q->forw->back = node;
	q->forw = node;
	write_unlock_irqrestore(&(q->qlock), flags);
}

void video_q_add_tail(struct video_queue *q, struct video_q_node *node)
{	unsigned long flags;
	if (q == NULL || node == NULL)
		return;
	if (q->forw == NULL || q->back == NULL)
		video_q_init(q);
	write_lock_irqsave(&(q->qlock), flags);
	node->forw = (struct video_q_node *)q;
	node->back = q->back;
	q->back->forw = node;
	q->back = node;
	write_unlock_irqrestore(&(q->qlock), flags);
}

void *video_q_del_head(struct video_queue *q)
{	unsigned long flags;
	struct video_q_node *node;
	if (q == NULL)
		return NULL;
	write_lock_irqsave(&(q->qlock), flags);
	if (q->forw == NULL || q->back == NULL ||
	    q->forw == (struct video_q_node *)q ||
	    q->back == (struct video_q_node *)q)
	{
		write_unlock_irqrestore(&(q->qlock), flags);
		return NULL;
	}
	node = q->forw;
	node->forw->back = (struct video_q_node *)q;
	q->forw = node->forw;
	node->forw = NULL;
	node->back = NULL;
	write_unlock_irqrestore(&(q->qlock), flags);
	return node;
}

void *video_q_peek_head(struct video_queue *q)
{	unsigned long flags;
	struct video_q_node *node;
	read_lock_irqsave(&(q->qlock), flags);
	if (q == NULL || q->forw == NULL || q->forw == (struct video_q_node *)q)
	{
		read_unlock_irqrestore(&(q->qlock), flags);
		return NULL;
	}
	node = q->forw;
	read_unlock_irqrestore(&(q->qlock), flags);
	return node;
}

void *video_q_yank_node(struct video_queue *q, struct video_q_node *node)
{	unsigned long flags;
	struct video_q_node *t;
	if (video_q_peek_head(q) == NULL || node == NULL)
		return NULL;
	write_lock_irqsave(&(q->qlock), flags);
	for (t = q->forw; t != (struct video_q_node *)q; t = t->forw)
		if (t == node)
		{
			node->back->forw = node->forw;
			node->forw->back = node->back;
			node->forw = NULL;
			node->back = NULL;
			write_unlock_irqrestore(&(q->qlock), flags);
			return node;
		}
	write_unlock_irqrestore(&(q->qlock), flags);
	return NULL;
}

//	Start an image capture
static void capture_grab_frame(struct videum_device *dev)
{	int			uc	= UC_NORMAL;
	struct scatter_node	*list;
//printk("grab_frame\n");
	if (dev->ready_to_capture && dev->sampler_enabled)
		return;

	capture_begin(dev);
	if (!dev->ready_to_capture)
		return;

	if (dev->uc_loaded != uc)
	{	wavi_block_write(dev, dev->sram_ucode,
				 dev->uc_buffer[uc],
				 dev->uc_length[uc]);
		wavi_filter(dev, dev->uc_parm.filter);
		dev->uc_loaded = uc;
	}
//	wavi_vdf(dev, WAVI_VDF_RGB);
	wavi_vdf(dev, WAVI_VDF_YCH);	// ZZZ

	wavi_writereg(dev, WAVI_SPLPTR, dev->sram_ucode >> 4);
	wavi_writereg(dev, WAVI_VID2PTR, dev->sram_linebuf >> 4);
	wavi_writereg(dev, WAVI_VID1PTR, dev->sram_video >> 4);

//	list = dev->capture_dma_list;	// DMA list for capture_buffer
	if (dev->streaming)
	{	struct stream_buffer	*buf;

//		Capture to capture_buffer
		dev->stream_capture_buffer = dev->capture_buffer;

//		Capture straight into streaming buffer?
		if (dev->translation2.type == XLAT_NULL)
		{
//			dev->stream_capture_buffer = NULL;	// test:
			buf = video_q_peek_head(&dev->stream_q_capture);
			if (buf != NULL)
			{
				dev->stream_capture_buffer = buf->vaddress;
				list = buf->dma_list;
			}
//			if (buf == NULL)/*test:*/
//				return;
		}
	}

	wavi_video_sampler(dev, 1);
	dev->sampler_enabled = 1;
	dev->capture_completed = 0;
}

//	STREAMING CAPTURE
static int	// 1 = success; 0 = failed
capture_queuebuffer(struct videum_device *dev, struct v4l2_buffer *vidbuf)
{	int	i = vidbuf->index;
	struct stream_buffer	*buf	= NULL;

	if (!dev->stream_buffers_mapped)
	{	debug_msg("QBUF no buffers mapped!\n");
		return 0;
	}
	if (vidbuf->type != V4L2_BUF_TYPE_CAPTURE)
	{	debug_msg("QBUF wrong buffer type!\n");
		return 0;
	}
	if (i < 0 || i >= MAX_CAPTURE_BUFFERS || !dev->stream_buf[i].requested)
	{
		debug_msg("QBUF buffer index %d is out of range\n", i);
		return 0;
	}

	buf = &dev->stream_buf[i];

	if (!(buf->vidbuf.flags & V4L2_BUF_FLAG_MAPPED))
	{
		debug_msg("QBUF buffer %d is not mapped\n", i);
		return 0;
	}
	if ((buf->vidbuf.flags & V4L2_BUF_FLAG_QUEUED))
	{
		debug_msg("QBUF buffer %d is already queued\n", i);
		return 0;
	}

	buf->vidbuf.flags &= ~V4L2_BUF_FLAG_DONE;
	video_q_add_tail(&dev->stream_q_capture, &buf->qnode);
	buf->vidbuf.flags |= V4L2_BUF_FLAG_QUEUED;
	capture_grab_frame(dev);	// does nothing if already capturing
	return 1;
}

static int/* 1 = got a buffer; 0 = no buffers */
capture_dequeuebuffer(struct videum_device *dev, struct v4l2_buffer *buf)
{	struct stream_buffer *newbuf;

	if (!dev->streaming)
	{	debug_msg("DQBUF not streaming!\n");
		return 0;
	}
	if (buf->type != V4L2_BUF_TYPE_CAPTURE)
	{	debug_msg("DQBUF wrong buffer type!\n");
		return 0;
	}
	newbuf = video_q_del_head(&dev->stream_q_done);
	if (newbuf == NULL)
	{	debug_msg("DQBUF nothing on done queue\n");
		return 0;
	}
	newbuf->vidbuf.flags &= ~V4L2_BUF_FLAG_QUEUED;

	*buf = newbuf->vidbuf;
	return 1;
}

static int capture_streamon(struct videum_device *dev, __u32 type)
{	struct stream_buffer *buf;

	if (dev->streaming || type != V4L2_BUF_TYPE_CAPTURE)
		return 0;

	capture_abort(dev);	// cancel any capture that might be in progress

//	-2 triggers start-of-stream logic in capture_interrupt()
	dev->stream_last_frame = -2;

	dev->perf.frames = 0;
	dev->perf.framesdropped = 0;
	dev->perf.bytesout = 0;
#ifndef V4L1
//	Can't capture frames faster than the video input
	if (dev->capture.timeperframe < dev->videc.frame_period)
		dev->capture.timeperframe = dev->videc.frame_period;
#endif
//	Move any leftover DONE buffers to the free pool
	while ((buf = video_q_del_head(&dev->stream_q_done)))
		buf->vidbuf.flags &= ~V4L2_BUF_FLAG_QUEUED;

//	Kick off the machine
	dev->streaming = 1;
	capture_grab_frame(dev);
	return 1;
}

static void capture_streamoff(struct videum_device *dev, __u32 type)
{
	if (!dev->streaming || type != V4L2_BUF_TYPE_CAPTURE)
		return;
	capture_abort(dev);
	dev->streaming = 0;

	/* Note: should really delay this till next capture */
	dev->perf.frames = 0;
	dev->perf.framesdropped = 0;
	dev->perf.bytesout = 0;
}

//	Read out and convert the next frame
//	returns length of data or negative for error
static int capture_imagereadout(struct videum_device	*dev,
			 __u8			*capture_buffer,
			 __u8			*output_buffer,
			 int			output_size,
			 int			output_is_user)
{	int	len		= dev->capture_size;
	int	native_len;

	{
		int	i, n, ram, block = 1024;
		__u8	*p;
		/*  Remember that ints are disabled in block_read() */
		n = (len + block - 1) / block;
		p = capture_buffer;
		ram = dev->sram_video;
		for (i = 0; i < n; ++i, ram += block, p += block)
			wavi_block_read(dev, ram, p, block);
		if (!wavi_vst(dev))
		{
			debug_msg("Slave readout underrun!\n");
			len = -1;/* don't copy to user */
		}
	}
	dev->sampler_enabled = 0;
	wavi_video_sampler(dev, 0);

	native_len = len;

	len = translate_image(dev, capture_buffer, output_buffer,
				  output_size, output_is_user);

	if (len < 0)
		return len;
	++dev->perf.frames;
	dev->perf.bytesout += len;
	return len;
}

u32 div6432(u64 a, u32 d)
{	u64 r=0xffffffff;
	u32 n;

	a += d >> 1;
	if (!d || a >= d * r)
		return (u32)r;
	if (a <= r)
		return ((u32)a) / d;
	n = ((u32)r) / d;
	r = n * (a >> 32);
	a -= r * d;
	if (a >= 0xffffffff)
	{	r += n;
		a -= n * d;
	}
	return ((u32)r) + ((u32)a) / d;
}

void masterclock_gettime100(stamp_t *curr)
{	struct timeval	t;
	do_gettimeofday(&t);
	*curr = 10 * (stamp_t)(t.tv_sec * 1000000 + t.tv_usec);
}

/*  The hardware has issued the interrupt signal, do any post-capture
 *  processing that may be necessary.
 *  [This function is called indirectly through the immediate task queue;
 *  it executes at elevated IRQL, but it is interruptible. (It's a b.h.)]
 */
static void capture_interrupt(void *v)
{	struct videum_device	*dev = (struct videum_device *)v;
	int			itr;
	struct stream_buffer	*buf;
	int			len;
	stamp_t			timestamp_rough;
	unsigned long	raw_frame_num;
	unsigned long	next_raw_frame_to_keep;
	unsigned long	stream_frame_num;
//	u64				temp64;

	itr = wavi_readreg(dev, WAVI_ITR) & (WAVI_VI1BIT | WAVI_VI2BIT);
	if (itr == 0)
		return;
	if (dev->capture_buffer == NULL ||
		!dev->sampler_enabled || !dev->ints_enabled)
	{
		err_msg("Interrupt handler aborted.\n");
		return;
	}
	dev->capture_completed = 1;

	if (!dev->streaming)
	{	dev->time_acquired = current_time_ms();
//		DMA might not have finished, but we'll check in read()
		wake_up_interruptible(&dev->new_video_frame);
		return;
	}

//	Only get here in streaming mode
	if (dev->stream_last_frame == -2)
	{	masterclock_gettime100(&dev->stream_begin);
		dev->stream_last_frame = -1;
	}

	if (dev->stream_capture_buffer == NULL)
		return;

	buf = video_q_peek_head(&dev->stream_q_capture);
//	No available buffers. Skip this frame
	if (buf == NULL)
	{	dev->sampler_enabled = 0;
		wavi_video_sampler(dev, 0);
		capture_grab_frame(dev);
		return;
	}

	masterclock_gettime100(&timestamp_rough);
	timestamp_rough -= dev->stream_begin;
//	raw_frame_num = v4l2_timestamp_divide(timestamp_rough, (unsigned long)dev->videc.frame_period);
	raw_frame_num = div6432((u64)timestamp_rough, (u32)dev->videc.frame_period);
#ifdef V4L1
	next_raw_frame_to_keep = dev->stream_last_frame + 1;
#else
	next_raw_frame_to_keep =
		div6432(((u64)(dev->stream_last_frame + 1)) * dev->capture.timeperframe,
			(u32)dev->videc.frame_period);
//		v4l2_math_div6432(temp64, dev->videc.frame_period, NULL);
#endif
	if (raw_frame_num < next_raw_frame_to_keep)
	{/*	Not time yet, don't keep this frame */
		dev->sampler_enabled = 0;
		wavi_video_sampler(dev, 0);
		capture_grab_frame(dev);
		return;
	}

	len = capture_imagereadout(dev, dev->stream_capture_buffer,
				   buf->vaddress, buf->vidbuf.length, 0);

	if (len <= 0)
	{/*	Frame no good, DMA did not finish, etc. */
		capture_grab_frame(dev);
		return;
	}

	buf->vidbuf.bytesused = len;
	buf->vidbuf.flags |= V4L2_BUF_FLAG_DONE | V4L2_BUF_FLAG_KEYFRAME;
	buf->vidbuf.timestamp = timestamp_rough;

#ifdef V4L1
	stream_frame_num = raw_frame_num;
	buf->vidbuf.timestamp = (stamp_t)(raw_frame_num * dev->videc.frame_period);
#else
//	stream_frame_num = v4l2_timestamp_correct(&buf->vidbuf.timestamp,
//		(unsigned long)dev->capture.timeperframe);
	stream_frame_num = div6432((u64)timestamp_rough,
			(u32)dev->capture.timeperframe);

	buf->vidbuf.timestamp = (stamp_t)(stream_frame_num *
		dev->capture.timeperframe);
#endif
	if (stream_frame_num > dev->stream_last_frame + 1)
	{/*	We have missed one or more frames  */
		dev->perf.framesdropped += stream_frame_num
			- dev->stream_last_frame + 1;
	}
	dev->stream_last_frame = stream_frame_num;

	buf = video_q_del_head(&dev->stream_q_capture);
	video_q_add_tail(&dev->stream_q_done, &buf->qnode);

	capture_grab_frame(dev);
	wake_up_interruptible(&dev->new_video_frame);
}

//	Read captured data into a user buffer.
//	Return: negative = error
//	0		= keep waiting
//	positive = count of bytes read successfully
static long capture_read(struct videum_device *dev, __u8	*user_buffer,
		 int	user_buffer_size)
{	int		len = user_buffer_size;
	unsigned long	now;

	if (!dev->ints_enabled)
		return -EIO;

	if (!dev->capture_completed)
	{/* No interrupt has occurred yet, or DMA didn't finish.  */
		//debug_msg("No data ready.\n");
		if (!dev->sampler_enabled)
			capture_grab_frame(dev);
		return 0;/* caller should keep waiting */
	}
	
	now = current_time_ms();
	if (now - dev->time_acquired > MAX_FRAME_AGE)
	{/* Frame in buffer is stale, get a new one */
		//debug_msg("Stale frame, re-acquiring.\n");
		dev->sampler_enabled = 0;
		wavi_video_sampler(dev, 0);
		capture_grab_frame(dev);
		return 0;/* caller should keep waiting */
	}

	len = capture_imagereadout(dev, dev->capture_buffer,
				   user_buffer, user_buffer_size, 1);
	capture_grab_frame(dev);
	return len;
}

//	Stop capturing and free all resources used for capture.
static void capture_close(struct videum_device *dev)
{	int	i;

	if (dev->streaming)
		capture_streamoff(dev, V4L2_BUF_TYPE_CAPTURE);
	capture_abort(dev);
	dev->ready_to_capture = 0;
	translate_close(dev);
//	if (dev->capture_dma_list)
//		free_page((unsigned long)dev->capture_dma_list);
//	dev->capture_dma_list = 0;
	if (dev->capture_buffer != NULL)
		vfree(dev->capture_buffer);
	dev->capture_buffer = NULL;
	dev->capture_buffer_size = 0;
	for (i = 0; i < MAX_CAPTURE_BUFFERS; ++i)
	{	dev->stream_buf[i].requested = 0;
		if (dev->stream_buf[i].vaddress)
			vfree(dev->stream_buf[i].vaddress);
		dev->stream_buf[i].vaddress = NULL;
		if (dev->stream_buf[i].dma_list)
			free_page((unsigned long)dev->stream_buf[i].dma_list);
		dev->stream_buf[i].dma_list = NULL;
	}
}

//	I N T E R R U P T   R O U T I N E S

// This function runs at interrupt time, either in response to a hardware
// interrupt, or on each timer tick if there is no hardware interrupt.
static void interrupt_handler(void *v)
{	struct videum_device *dev = (struct videum_device *)v;

	if (!dev->ints_enabled)
		return;

	/*  Call "bottom half" of handler  */
	dev->tqnode_dpc.routine = capture_interrupt;
	dev->tqnode_dpc.data = dev;
	queue_task(&dev->tqnode_dpc, &tq_immediate);
	mark_bh(IMMEDIATE_BH);

	if (dev->tlnode.function != NULL &&
		dev->ints_enabled)
	{/*	Poll again on next timer tick  */
		dev->tlnode.expires = jiffies + HZ/100;
		add_timer(&dev->tlnode);
	}
}

static void interrupt_disable(struct videum_device *dev)
{	if (!dev->ints_enabled)
		return;
	dev->ints_enabled = 0;
	wavi_writereg(dev, WAVI_ISR, 0);
		del_timer(&dev->tlnode);
	wake_up_interruptible(&dev->new_video_frame);
}

static void interrupt_enable(struct videum_device *dev)
{	if (dev->ints_enabled)
		interrupt_disable(dev);
	dev->ints_enabled = 1;

	wavi_readreg(dev, WAVI_ITR);

	wavi_writereg(dev, WAVI_ISR, WAVI_VIEBIT);

	dev->tlnode.function = NULL; /* NULL indicates h/w interrupts */
	{
		init_timer(&dev->tlnode);
		dev->tlnode.function = 
			(void(*)(unsigned long))interrupt_handler;
		dev->tlnode.data = (unsigned long)dev;
		dev->tlnode.expires = jiffies + HZ/100;
		add_timer(&dev->tlnode);
	}
	debug_msg("%s interrupts enabled\n",
		  dev->tlnode.function?"Polled":"Hardware");
}

//	M E M O R Y   M A P P I N G

static struct stream_buffer *
mmap_stream_buffer_from_offset(struct videum_device *dev, unsigned long offset)
{	int i;
	for (i=0; i < MAX_CAPTURE_BUFFERS; ++i)
		if (offset == dev->stream_buf[i].vidbuf.offset) //= i<<PAGE_SHIFT :RAR
			return &dev->stream_buf[i];
	return NULL;
}

static int
mmap_request_buffers(struct videum_device *dev, struct v4l2_requestbuffers *req)
{	int	i;
	u32	buflen;

//	can't make requests if buffers are mapped
	if (dev->stream_buffers_mapped)
		return 0;
	if (req->count < 1)
		req->count = 1;
	if (req->count > MAX_CAPTURE_BUFFERS)
		req->count = MAX_CAPTURE_BUFFERS;
	req->type = V4L2_BUF_TYPE_CAPTURE;	// only kind I know
	buflen = ~(PAGE_SIZE - 1) & (PAGE_SIZE - 1 +
#ifdef V4L1
		MAX_WIDTH * MAX_HEIGHT * 6);
//		((dev->vBuf.width * dev->vBuf.height * dev->vPict.depth) >> 3));
#else
		dev->clientfmt.fmt.pix.sizeimage);
#endif
debug_msg("Granting %d buffers\n",req->count);

//	Now initialize the buffer structures.
//	Don't allocate the buffers until they're mapped.
	for (i = 0; i < req->count; ++i)
	{	dev->stream_buf[i].requested = 1;
		dev->stream_buf[i].vidbuf.index = i;
		dev->stream_buf[i].vidbuf.type = req->type;
//		dev->stream_buf[i].vidbuf.offset = 4*i;/* anything unique */
		dev->stream_buf[i].vidbuf.offset = i<<PAGE_SHIFT;	//RAR
		dev->stream_buf[i].vidbuf.length = buflen;
		dev->stream_buf[i].vidbuf.bytesused = 0;
		dev->stream_buf[i].vidbuf.timestamp = 0;
		dev->stream_buf[i].vidbuf.flags = 0;
	}
	for (i = req->count; i < MAX_CAPTURE_BUFFERS; ++i)
		dev->stream_buf[i].requested = 0;

	return 1;
}

static void mmap_unrequest_buffers(struct videum_device *dev)
{	int	i;

	for (i = 0; i < MAX_CAPTURE_BUFFERS; ++i)
		dev->stream_buf[i].requested = 0;
}

static void mmap_vma_open(struct vm_area_struct *vma)
{	struct videum_device *dev =
#ifdef V4L1
		NULL;
#else
		videum_device_from_file(vma->vm_file);
#endif
	if (dev == NULL)
		return;
	//debug_msg("vma_open called\n");
	++dev->stream_buffers_mapped;
	//MOD_INC_USE_COUNT;
}

static void mmap_vma_close(struct vm_area_struct *vma)
{	struct videum_device *dev =
#ifdef V4L1
		NULL;
#else
		videum_device_from_file(vma->vm_file);
#endif
	struct stream_buffer *buf = mmap_stream_buffer_from_offset(dev,
			vma->vm_pgoff<<PAGE_SHIFT);

	if (dev->streaming)
	{	info_msg("Warning- munmap() called while streaming\n");
		capture_streamoff(dev, buf->vidbuf.type);
	}
	video_q_yank_node(&dev->stream_q_capture, &buf->qnode);
	video_q_yank_node(&dev->stream_q_done, &buf->qnode);

	if (buf->vaddress != NULL)
		vfree(buf->vaddress);
	buf->vaddress = NULL;
	if (buf->dma_list)
		free_page((unsigned long)buf->dma_list);
	buf->dma_list = NULL;
	buf->vidbuf.flags = 0;
debug_msg("Buffer %d deallocated\n",(int)vma->vm_pgoff<<PAGE_SHIFT);

	if (dev->stream_buffers_mapped > 0)
		--dev->stream_buffers_mapped;
	//MOD_DEC_USE_COUNT;
}

struct page *kvirt_to_pa(unsigned long adr)
{	pmd_t *pmd;
	pte_t *pte;
	pgd_t *pgd;

	pgd = pgd_offset_k(adr);
	if (!pgd_none(*pgd)) {
		pmd = pmd_offset(pgd, adr);
		if (!pmd_none(*pmd)) {
			pte = pte_offset(pmd, adr);
			if (pte_present(*pte))
				return pte_page(*pte);
		}
	}
	return (struct page *)NULL;
}

static struct page *
mmap_vma_nopage(struct vm_area_struct *vma, unsigned long address, int write)
{	struct stream_buffer	*buf;
	unsigned long		offset_into_buffer;
	struct page		*page;
	struct videum_device	*dev =
#ifdef V4L1
		NULL;
#else
		videum_device_from_file(vma->vm_file);
#endif
	if (dev == NULL)
	{	err_msg("videum_device_from_file failed in mmap_vma_nopage\n");
		return 0;
	}

	buf = mmap_stream_buffer_from_offset(dev, vma->vm_pgoff<<PAGE_SHIFT);
	if (buf == NULL)
	{	err_msg("mmap_stream_buffer_from_offset failed in mmap_vma_nopage\n");
		return 0;
	}
	offset_into_buffer = address - vma->vm_start;
	if (offset_into_buffer >= buf->vidbuf.length)
	{	err_msg("buffer too short in mmap_vma_nopage\n");
		return 0;
	}
//	page = v4l2_vmalloc_to_page(buf->vaddress + offset_into_buffer);
	page = kvirt_to_pa((unsigned long)(buf->vaddress + offset_into_buffer));
	if (page == 0)
	{	err_msg("v4l2_vmalloc_to_page() failed in mmap_vma_nopage\n");
		return 0;
	}

	atomic_inc(&page->count);
	return page;
}

static struct vm_operations_struct videum_vma_operations =
{	mmap_vma_open, mmap_vma_close, mmap_vma_nopage,
};

//	V I D E O   F O R   L I N U X   I N T E R F A C I N G
#ifdef V4L1
static int videum_open(struct video_device *v, int flags)
#else
static int videum_open(struct v4l2_device *v, int flags, void **idptr)
#endif
{	struct videum_device *dev = (struct videum_device *)v;

//	Available open_data structure
	if (dev->open_data.isopen)
		return -EBUSY;
	
	dev->perf.frames = 0;
	dev->perf.framesdropped = 0;
	dev->perf.bytesout = 0;

	dev->open_data.isopen = 1;
	dev->open_data.dev = dev;
#ifndef V4L1
	*idptr = &dev->open_data;
#endif

	dev->ready_to_capture = 0;
	dev->capture_completed = 0;
	dev->sampler_enabled = 0;
	video_q_init(&dev->stream_q_capture);
	video_q_init(&dev->stream_q_done);
	return 0;
}

#ifdef V4L1
static void videum_close(struct video_device *id)
{	struct videum_device	*dev = (struct videum_device *)id;
#else
static void videum_close(void *id)
{	struct videum_device	*dev = ((struct device_open *)id)->dev;
	((struct device_open *)id)->isopen = 0;
#endif

	debug_msg("Close\n");
	interrupt_disable(dev);
	capture_close(dev);
	if (dev->irq)
	{	free_irq(dev->irq, dev);
		dev->irq = 0;
}	}

static long
#ifdef V4L1
videum_write(struct video_device *id, const char *buf, unsigned long count,
	int noblock)
#else
videum_write(void *id, const char *buf, unsigned long count, int noblock)
#endif
{	return -EINVAL;
}

#ifdef V4L1
static int videum_ioctl(struct video_device *id, unsigned int cmd, void *arg)
{	struct videum_device *dev = (struct videum_device *)id;
#else
static int videum_ioctl(void *id, unsigned int cmd, void *arg)
{	struct videum_device *dev = ((struct device_open *)id)->dev;
#endif

//debug_msg("ioctl %d\n", _IOC_NR(cmd));
	switch(cmd)
	{
#ifdef V4L1
		case VIDIOCGCAP:
		{	struct video_capability *b = arg;
			strcpy(b->name, "Videum");
			b->type = VID_TYPE_CAPTURE; // VID_TYPE_TUNER | VID_TYPE_OVERLAY|
//						VID_TYPE_CLIPPING| VID_TYPE_FRAMERAM| VID_TYPE_SCALES;
			b->channels = 1;
			b->audios = 0;
			b->maxwidth		= MAX_WIDTH;
			b->maxheight	= MAX_HEIGHT;
			b->minwidth		= MIN_WIDTH;
			b->minheight	= MIN_HEIGHT;
			return 0;
		}
#else
		case VIDIOC_QUERYCAP:
		{	struct v4l2_capability *b = arg;
			strcpy(b->name, "Videum");
			b->type = V4L2_TYPE_CAPTURE;
			b->flags =  V4L2_FLAG_READ |
				   V4L2_FLAG_STREAMING |
				   V4L2_FLAG_SELECT;
			b->inputs = 1;
			b->outputs = 0;
			b->audios = 0;
			b->maxwidth = MAX_WIDTH;
			b->maxheight = MAX_HEIGHT;
			b->minwidth = MIN_WIDTH;
			b->minheight = MIN_HEIGHT;
			b->maxframerate = 30;
			return 0;
		}
		case VIDIOC_ENUM_CAPFMT:
		{	struct v4l2_fmtdesc *f = arg;
			if (f->index < 0 || f->index >= NUM_CAPFMT)
				return -EINVAL;
			*f = capfmt[f->index];
			return 0;
		}
#endif
#ifdef V4L1
		case VIDIOCGPICT:
		{	struct video_picture *p = arg;
			*p = dev->vPict;
			return 0;
		}
#else
		case VIDIOC_G_FMT:
		{	struct v4l2_format *fmt = arg;
			*fmt = dev->clientfmt;
			return 0;
		}
#endif
#ifdef V4L1
		case VIDIOCSPICT:
		{	struct video_picture *p = arg;
			if (p->palette != VIDEO_PALETTE_YUYV)
				return -EINVAL;
			dev->vPict = *p;
			capture_new_format(dev);
//			mmap_unrequest_buffers(dev); //ZZZ
			*p = dev->vPict;
			return 0;
		}
#else
		case VIDIOC_S_FMT:
		{	struct v4l2_format *fmt = arg;

			if (dev->stream_buffers_mapped)
				return -EPERM;
			dev->clientfmt = *fmt;
			capture_new_format(dev);
			mmap_unrequest_buffers(dev);
			*fmt = dev->clientfmt;
			return 0;
		}
#endif
#ifdef V4L1
		case VIDIOCGCHAN:
		{	struct video_channel *v = arg;
			strcpy(v->name, "MXC");
			v->flags = VIDEO_VC_AUDIO;		// VIDEO_VC_TUNER
			v->tuners = 0;
			v->type = VIDEO_TYPE_CAMERA;	// VIDEO_TYPE_TV
			v->norm = VIDEO_MODE_NTSC;		// VIDEO_MODE_PAL VIDEO_MODE_SECAM
			if (v->channel > 0)
				return -EINVAL;
			return 0;
		}
#else
		case VIDIOC_G_INPUT:
		{	*(int *)arg = 0;
			return 0;
		}
		case VIDIOC_ENUMINPUT:
		{	struct v4l2_input *vi = arg;
			if (vi->index)
				return -EINVAL;
			strcpy(vi->name, "MXC");
			vi->type = V4L2_INPUT_TYPE_CAMERA;
			vi->capability = 0;
			return 0;
		}
#endif
#ifdef V4L1
		case VIDIOCSCHAN:
		{	struct video_channel *v = arg;
		
			if (v->channel > 0)
				return -EINVAL;
			if (dev->stream_buffers_mapped)
				return -EPERM;
			return 0;
		}
#else
		case VIDIOC_S_INPUT:
		{	if (0 != *(int *)arg)
				return -EINVAL;
			return 0;
		}
#endif
//#ifdef V4L1
//		case VIDIOCGFBUF:
//		{//	struct video_buffer *v = arg;
//			*v = dev->vb;
//			return 0;
//		}
//#else
//#endif
//#ifdef V4L1
//		case VIDIOCSFBUF:
	//	{//	struct video_buffer *v = arg;
//			if(v->depth!=8 && v->depth!=15 && v->depth!=16 && 
//			   v->depth!=24 && v->depth!=32 && v->width > 16 &&
//			   v->height > 16 && v->bytesperline > 16)
//				return -EINVAL;
//			dev->vb = *v;
//			return 0;		
//		}
//#else
#ifndef V4L1
		case VIDIOC_REQBUFS:
		{	struct v4l2_requestbuffers *req = arg;
			if (dev->stream_buffers_mapped)
				return -EPERM;
			capture_begin(dev);
			if (!mmap_request_buffers(dev, req))
				return -EINVAL;
			return 0;
		}

		case VIDIOC_QUERYBUF:
		{	struct v4l2_buffer *buf = arg;
			int	i = buf->index;

			if (i < 0 || i >= MAX_CAPTURE_BUFFERS ||
				!dev->stream_buf[i].requested ||
				(buf->type & V4L2_BUF_TYPE_field) != 
				 (dev->stream_buf[i].vidbuf.type & V4L2_BUF_TYPE_field))
				return -EINVAL;
			*buf = dev->stream_buf[i].vidbuf;
			return 0;
		}

		case VIDIOC_QBUF:
		{	struct v4l2_buffer *buf = arg;
			if (!dev->stream_buffers_mapped)
				return -EINVAL;
			if (!capture_queuebuffer(dev, buf))
				return -EINVAL;
			return 0;
		}

		case VIDIOC_DQBUF:
		{	struct v4l2_buffer *buf = arg;
			if (!capture_dequeuebuffer(dev, buf))
				return -EINVAL;
			return 0;
		}

		case VIDIOC_STREAMON:
		{	__u32	type = *(__u32 *)arg;  // Bug fixed DWK/RAR 7/12/2001
			if (!capture_streamon(dev, type))
				return -EINVAL;
			return 0;
		}

		case VIDIOC_STREAMOFF:
		{	__u32	type = *(__u32 *)arg;  // Bug fixed DWK/RAR 7/12/2001
			capture_streamoff(dev, type);
			return 0;
		}
#endif
#ifdef V4L1
		case VIDIOCGTUNER:
		case VIDIOCSTUNER:
		case VIDIOCGWIN:
		case VIDIOCSWIN:
			return 0;
		case VIDIOCCAPTURE:
		{//	int *v = arg;
//			if (!dev->vb.base || !dev->vb.width || !dev->vb.height)
//				return -EINVAL;
//			if (1 == no_overlay)
//				return -EIO;
			return 0;
		}
		case VIDIOCSYNC:	// wait for a frame	//ZZZZZ
		{	int i = *(int *)arg;

//			Make sure that requested frame is in range.
			if (i < 0 || 1 < i)
			{	printk("<1>Requested frame=%d out of range!\n", i);
				return -EINVAL;
			}
			if (!dev->stream_buf[i].requested)
			{	printk("<1>Stream buffer not requested!\n");
				return -EINVAL;
			}
//			if (!dev->stream_buffers_mapped)
//			{	printk("Steam bufferes not mapped!\n");
//				return -EINVAL;
//			}
			if (!(dev->stream_buf[i].vidbuf.flags & V4L2_BUF_FLAG_MAPPED))
			{	printk("Buffer is not mapped!\n");
				return -EINVAL;
			}

//			Loop as long as the buffer is queued, but not done
			while ((dev->stream_buf[i].vidbuf.flags &
				(V4L2_BUF_FLAG_QUEUED | V4L2_BUF_FLAG_DONE))
			       == V4L2_BUF_FLAG_QUEUED)
			{
//				Error or sleep was interrupted, (=0)timeout? Shouldn't occur.
//				if (0 >= simple_select(file))
//					break;
			}
			if (!(dev->stream_buf[i].vidbuf.flags & V4L2_BUF_FLAG_DONE))	// not done
				break;
//			do
//			{
//				err = vfl->ioctl(context, VIDIOC_DQBUF, &buf2);
//			}	while (err == 0 && buf2.index != *i);

			return 0;
		}
		case VIDIOCMCAPTURE:	// capture a frame
		{	struct video_mmap	*mm = arg;
			struct stream_buffer	*buf;

//			Make sure that requested frame is in range.
			if (mm->frame < 0 || 1 < mm->frame)
			{	printk("Requested frame=%d out of range!\n", mm->frame);
				return -EINVAL;
			}
			if (!dev->stream_buf[mm->frame].requested)
			{	printk("Stream buffer not requested!\n");
				return -EINVAL;
			}
//			if (!dev->stream_buffers_mapped)
//			{	printk("Steam bufferes not mapped!\n");
//				return -EINVAL;
//			}

//			Verify format
			if (dev->vBuf.width		!= mm->width  || 
			    dev->vBuf.height	!= mm->height ||
			    dev->vPict.palette	!= mm->format)
			{	dev->vBuf.width		 = mm->width;
			    dev->vBuf.height	 = mm->height;
			    dev->vPict.palette	 = mm->format;
				printk("Changing format.\n");
				capture_new_format(dev);
			}
			buf = &dev->stream_buf[mm->frame];
			if (!(buf->vidbuf.flags & V4L2_BUF_FLAG_MAPPED))
			{	debug_msg("Buffer %d is not mapped\n", mm->frame);
				return -EINVAL;
			}
			if ((buf->vidbuf.flags & V4L2_BUF_FLAG_QUEUED))
			{	debug_msg("Buffer %d is already queued\n", mm->frame);
				return -EINVAL;
			}

			buf->vidbuf.flags &= ~V4L2_BUF_FLAG_DONE;
			video_q_add_tail(&dev->stream_q_capture, &buf->qnode);
			buf->vidbuf.flags |= V4L2_BUF_FLAG_QUEUED;
			capture_grab_frame(dev);	// does nothing if already capturing

			if (!capture_streamon(dev, V4L2_BUF_TYPE_CAPTURE))
				return -EINVAL;

			return 0;
		}
/*		case VIDIOCMCAPTURE:	// capture a frame
		{	struct video_mmap	*mm = arg;
			struct v4l2_buffer	buf2;
			struct v4l2_format	fmt2;

//			Verify format
			fmt2.type = V4L2_BUF_TYPE_CAPTURE;
			if (0 > vfl->ioctl(context, VIDIOC_G_FMT, &fmt2))
				break;
//			New capture format...
			if (mm->width != fmt2.fmt.pix.width || 
			    mm->height != fmt2.fmt.pix.height ||
			    palette_to_pixelformat(mm->format) != 
			    fmt2.fmt.pix.pixelformat)
			{	fmt2.fmt.pix.width = mm->width;
				fmt2.fmt.pix.height = mm->height;
				fmt2.fmt.pix.pixelformat =
					palette_to_pixelformat(mm->format);
				fmt2.fmt.pix.flags |= V4L2_FMT_FLAG_INTERLACED;
				if (0 > vfl->ioctl(context, VIDIOC_S_FMT, &fmt2))
					break;
			}
			buf2.index = mm->frame;
			buf2.type = V4L2_BUF_TYPE_CAPTURE;
			if (0 > vfl->ioctl(context, VIDIOC_QUERYBUF, &buf2))
				break;
			if (0 > vfl->ioctl(context, VIDIOC_QBUF, &buf2))
				break;
			vfl->ioctl(context, VIDIOC_STREAMON, &buf2.type);
		break;
		}
		case VIDIOCSYNC:	// wait for a frame
		{
			int			*i = arg;
			struct v4l2_buffer	buf2;

			buf2.index = *i;
			buf2.type = V4L2_BUF_TYPE_CAPTURE;
			err = vfl->ioctl(context, VIDIOC_QUERYBUF, &buf2);
			if (err < 0)	// No such buffer
				break;
			if (!(buf2.flags & V4L2_BUF_FLAG_MAPPED))
//				Buffer is not mapped 
			{	err = -EINVAL;
				break;
			}

//			Loop as long as the buffer is queued, but not done
			while ((buf2.flags &
				(V4L2_BUF_FLAG_QUEUED | V4L2_BUF_FLAG_DONE))
			       == V4L2_BUF_FLAG_QUEUED)
			{
//				Error or sleep was interrupted, (=0)timeout? Shouldn't occur.
				if (0 >= simple_select(file))
					break;
				vfl->ioctl(context, VIDIOC_QUERYBUF, &buf2);
			}
			if (!(buf2.flags & V4L2_BUF_FLAG_DONE))	// not done
				break;
			do
			{
				err = vfl->ioctl(context, VIDIOC_DQBUF, &buf2);
			}	while (err == 0 && buf2.index != *i);
			break;
		}
	case VIDIOCGUNIT:	// get related device minors: No translation
		break;
	case VIDIOCGCAPTURE:	// No translation, yet...
		printk(KERN_INFO"V4L2: VIDIOCGCAPTURE not implemented. "
		       "Send patches to bdirks@pacbell.net :-)\n");
		break;
	case VIDIOCSCAPTURE:	// No translation, yet...
		printk(KERN_INFO"V4L2: VIDIOCSCAPTURE not implemented. "
		       "Send patches to bdirks@pacbell.net :-)\n");
		break;
	}*/
		case VIDIOCGMBUF:
		{	struct video_mbuf *vm = arg;
			struct v4l2_requestbuffers	reqbuf2;

			memset(vm, 0, sizeof(struct video_mbuf));
			vm->frames = 2;
			vm->size = MAX_WIDTH * MAX_HEIGHT * 6;
			vm->offsets[0] = 0;
			vm->offsets[1] = vm->size / vm->frames;

			if (dev->stream_buffers_mapped)
				return -EPERM;
			capture_begin(dev);
			reqbuf2.count = 2;	// v4l always used two buffers
			reqbuf2.type = V4L2_BUF_TYPE_CAPTURE | V4L2_BUF_REQ_CONTIG;
			if (!mmap_request_buffers(dev, &reqbuf2))
				return -EINVAL;
			return 0;
		}
/*		case VIDIOCGMBUF:	// Get mmap parameters
		{	struct video_mbuf		*mbuf = arg;
			struct v4l2_requestbuffers	reqbuf2;
			struct v4l2_buffer		buf2;
			struct v4l2_format		fmt2;

//			Set the format to maximum dimensions
			fmt2.type = V4L2_BUF_TYPE_CAPTURE;
			if (vfl->ioctl(context, VIDIOC_G_FMT, &fmt2) < 0)
				break;
			fmt2.fmt.pix.width = MAX_WIDTH;
			fmt2.fmt.pix.height = MAX_HEIGHT;
			fmt2.fmt.pix.flags |= V4L2_FMT_FLAG_INTERLACED;
			if (vfl->ioctl(context, VIDIOC_S_FMT, &fmt2) < 0)
				break;
			reqbuf2.count = 2;	// v4l always used two buffers
			reqbuf2.type = V4L2_BUF_TYPE_CAPTURE | V4L2_BUF_REQ_CONTIG;
			vfl->ioctl(context, VIDIOC_REQBUFS, &reqbuf2);
			buf2.index = 0;
			buf2.type = V4L2_BUF_TYPE_CAPTURE;
			vfl->ioctl(context, VIDIOC_QUERYBUF, &buf2);
			mbuf->size = buf2.length * reqbuf2.count;
			mbuf->frames = reqbuf2.count;
			memset(mbuf->offsets, 0, sizeof(mbuf->offsets));
//			mbuf->offsets[0] = 0;
			mbuf->offsets[1] = buf2.length;
		break;
		} */
		case VIDIOCKEY:
		case VIDIOCGFREQ:
		case VIDIOCSFREQ:
		case VIDIOCGAUDIO:
		case VIDIOCSAUDIO:
		case VIDIOCGUNIT:
			return 0;
#else
	case VIDIOC_G_COMP:
	case VIDIOC_S_COMP:
	case VIDIOC_ENUM_FBUFFMT:
	case VIDIOC_G_FBUF:
	case VIDIOC_S_FBUF:
	case VIDIOC_G_WIN:
	case VIDIOC_S_WIN:
	case VIDIOC_PREVIEW:
		return -EINVAL;

	case VIDIOC_G_PERF:
	{	memcpy(arg, &dev->perf, sizeof(dev->perf));
		return 0;
	}

	case VIDIOC_G_STD:
	{	struct v4l2_standard *std = arg;
		v4l2_video_std_construct(std, dev->videc.standard, 0);
		return 0;
	}

	case VIDIOC_S_STD:
	{	struct v4l2_standard	*std = arg;
		int			id;
printk("Buff_Mapp=%d\n", dev->stream_buffers_mapped);
		if (dev->stream_buffers_mapped)
			return -EPERM;
//printk("STD=%d\n", *std);
		id = v4l2_video_std_confirm(std);
		if (!((1 << id) & dev->videc.standards))
		{	debug_msg("Bad standard: %u\n", (unsigned)id);
			return -EINVAL;
		}
		ccd_set_standard(dev, id);
		return 0;
	}

	case VIDIOC_ENUMSTD:
	{	struct v4l2_enumstd *estd = arg;
		__u32	b, i;
		if (estd->index < 0 || estd->index > 30)
			return -EINVAL;
		for (b = 1, i = 0; b < 32; ++b)
		{	if (((1 << b) & dev->videc.standards) == 0)
				continue;
			if (i == estd->index)
			{	v4l2_video_std_construct(&estd->std, b, 0);
				estd->inputs = (__u32)-1; /* all inputs */
				estd->outputs = 0;
				return 0;
			}
			++i;
		}
		return -EINVAL;
	}

		case VIDIOC_G_PARM:
		{	struct v4l2_captureparm *vp = arg;
			*vp = dev->capture;
			return 0;
		}

		case VIDIOC_S_PARM:
		{	struct v4l2_captureparm *vp = arg;
			if (vp->capturemode & ~dev->capture.capability)
				return -EINVAL;
			if (vp->timeperframe < 10000)
				return -EINVAL;
			if (vp->capturemode != dev->capture.capturemode && dev->streaming)
				return -EINVAL;

			if (vp->capturemode != dev->capture.capturemode)
			{	dev->capture.capturemode = vp->capturemode;
				capture_new_format(dev);
			}
			if ((vp->capturemode & V4L2_CAP_TIMEPERFRAME) &&
				vp->timeperframe >= dev->videc.frame_period)
				dev->capture.timeperframe = vp->timeperframe;
			else
				dev->capture.timeperframe = dev->videc.frame_period;
			return 0;
		}
		case VIDIOC_QUERYCTRL:
		{	struct v4l2_queryctrl	*qc = arg;
			int			i;
			i = find_vctrl(qc->id);
			if (i < 0)
				return -EINVAL;
			videum_control[i].category = qc->category;
			memcpy(videum_control[i].name, qc->name, sizeof(qc->name));
			*qc = videum_control[i];
			return 0;
		}

		case VIDIOC_G_CTRL:
		{	struct v4l2_control	*vc = arg;
			int			i;
			i = find_vctrl(vc->id);
			if (i < 0)
				return -EINVAL;
			vc->value = dev->control[i];
			return 0;
		}

		case VIDIOC_S_CTRL:
		{	struct v4l2_control	*vc = arg;
			int			i;
			i = find_vctrl(vc->id);
			if (i < 0)
				return -EINVAL;
			dev->control[i] = vc->value;
			wavi_tone_controls(dev);
			return 0;
		}

	case VIDIOC_G_TUNER:
	case VIDIOC_S_TUNER:
	case VIDIOC_G_FREQ:
	case VIDIOC_S_FREQ:
	case VIDIOC_G_AUDIO:
	case VIDIOC_S_AUDIO:
		return -EINVAL;
#endif
	default:
		return -ENOIOCTLCMD;
	}
	return 0;
}

#ifdef V4L1
static int
videum_mmap(struct video_device *id,const char *adr,unsigned long size)
{	struct videum_device *dev = (struct videum_device *)id;
	struct stream_buffer	*buf;
	buf = mmap_stream_buffer_from_offset(dev, (unsigned long)adr);
#else
static int videum_mmap(void *id, struct vm_area_struct *vma)
{	struct videum_device	*dev = ((struct device_open *)id)->dev;
	struct stream_buffer	*buf = 
		mmap_stream_buffer_from_offset(dev, vma->vm_pgoff<<PAGE_SHIFT);
#endif
	if (buf == NULL)
	{	printk("No such buffer\n");
		return -EINVAL;
	}
	if (!buf->requested)
	{	printk("Not requested\n");
		return -EINVAL;
	}
	if (buf->vidbuf.flags & V4L2_BUF_FLAG_MAPPED)
	{	printk("Already mapped\n");
		return -EINVAL;
	}
#ifdef V4L1
	if (buf->vidbuf.length != size)
	{	printk("Wrong length: VBlen=%d, size=%ld\n", buf->vidbuf.length, size);
#else
	if (buf->vidbuf.length != vma->vm_end - vma->vm_start)
	{	printk("Wrong length: VBlen=%d, size=%ld\n", buf->vidbuf.length, 
			vma->vm_end - vma->vm_start);
#endif
		return -EINVAL;
	}

	if (buf->vaddress != NULL)
		vfree(buf->vaddress);
	buf->vaddress = vmalloc(buf->vidbuf.length);
	if (buf->vaddress == NULL)
	{	err_msg("Couldn't allocate mmap() buffer\n");
		return -ENODEV;
	}
	buf->vidbuf.flags |= V4L2_BUF_FLAG_MAPPED;

#ifndef V4L1
//	Note: vma->vm_file will be set up by V4L2
	vma->vm_ops = &videum_vma_operations;
	if (vma->vm_ops->open)
		vma->vm_ops->open(vma);
#endif
	return 0;
}

#ifdef V4L1
static unsigned
int videum_poll(struct video_device *id, struct file *file, poll_table *table)
{	struct videum_device	*dev = (struct videum_device *)id;
#else
static int videum_poll(void *id, struct file *file, poll_table *table)
{	struct videum_device	*dev = ((struct device_open *)id)->dev;
#endif

	if (dev->streaming)
	{
		void	*node;
		node = video_q_peek_head(&dev->stream_q_done);
		if (node != NULL)
			return (POLLIN | POLLRDNORM);/* data is ready now */
		node = video_q_peek_head(&dev->stream_q_capture);
		if (node == NULL)
			return POLLERR;  /* no frames queued */
		poll_wait(file, &dev->new_video_frame, table);
		return 0;
	}

	/*  Capture is through read() call */

	if (dev->capture_completed)/* data is ready now */
		return (POLLIN | POLLRDNORM);
	capture_grab_frame(dev);/* does nothing if capture is in progress */
	if (!dev->ready_to_capture)/* Can't grab frames! */
		return POLLERR;
	poll_wait(file, &dev->new_video_frame, table);
	return 0;
}

static long
#ifdef V4L1
videum_read(struct video_device *id,char *buf,unsigned long count,int noblock)
{	struct videum_device	*dev = (struct videum_device *)id;
#else
videum_read(void *id, char *buf, unsigned long count, int noblock)
{	struct videum_device	*dev = ((struct device_open *)id)->dev;
#endif
	long	len		= 0;
	int	retries		= 2;
	long	my_timeout;
	int	 res1, res2;

	if (dev->streaming)
	{	printk("<1>Camera is streaming!");
		return -EPERM;
	}
	capture_grab_frame(dev);/* does nothing if capture is in progress */
	if (!dev->ready_to_capture)
	{	debug_msg("Can't grab frames!\n");
		return 0;
	}

	my_timeout = HZ / 4;
	while (len == 0)
	{
		if (noblock)
		{
			if (!dev->capture_completed)
				return -EAGAIN;
		}
		else
		{
			/* watch out for race condition going to sleep! */
			cli();
			if (!dev->capture_completed)
				my_timeout = interruptible_sleep_on_timeout(
					&dev->new_video_frame, my_timeout);
			sti();
		}
		if (my_timeout == 0)
		{/*	Timeout!  */
			--retries;
			if (retries == 0)
			{/*	Out of patience...  */
					res1 = wavi_readreg(dev, WAVI_ISR);
				res2 = wavi_readreg(dev, WAVI_CTL);
				debug_msg("Timeout on read. "
					  "ISR=%04X CTL=%04X\n",
					  res1, res2);
				break;
			}
			/*  Reset and restart  */
			capture_abort(dev);
			capture_grab_frame(dev);
			continue;
		}
		len = capture_read(dev, buf, count);
	}
	//debug_msg("read %d\n", (int)len);
	return len;
}

//	Remaining initialization of video decoder etc. This is only
//	done when the device is successfully identified and registered.
static int videum_init_done(struct v4l2_device *v)
{	struct videum_device *dev = (struct videum_device *)v;
	int	t;

//	Initialize video control properties	*/
#ifndef V4L1
	dev->control[VCTRL_BRIGHTNESS] =
		videum_control[VCTRL_BRIGHTNESS].default_value;
	dev->control[VCTRL_CONTRAST] =
		videum_control[VCTRL_CONTRAST].default_value;
	dev->control[VCTRL_SATURATION] =
		videum_control[VCTRL_SATURATION].default_value;
	dev->control[VCTRL_HUE] =
		videum_control[VCTRL_HUE].default_value;
#endif

//	Initialize the video decoder hardware
	ccd_initialize(dev);
	t = wavi_readreg(dev, WAVI_MMA);
	if (dev->eeprom.dwHwFlags & HWFLAGS_INV_VSYNC)
		t |=  WAVI_SAA7110;
	else
		t &= ~WAVI_SAA7110;
	wavi_writereg(dev, WAVI_MMA, t);
//	BUG: get defaults from user somehow...
	ccd_set_standard(dev, V4L2_STD_NTSC);
	set_video_input(dev, 0);

//	Default capture dimensions
#ifdef V4L1
	dev->vBuf.width = MAX_WIDTH;
	dev->vBuf.height = MAX_HEIGHT;
	dev->vPict.depth = 16;
	dev->vPict.palette = VIDEO_PALETTE_YUYV;
#else
	dev->clientfmt.fmt.pix.width = 160;
	dev->clientfmt.fmt.pix.height = 120;
	dev->clientfmt.fmt.pix.depth = 16;
	dev->clientfmt.fmt.pix.pixelformat = V4L2_PIX_FMT_RGB565;
	dev->clientfmt.fmt.pix.flags = 0;
	dev->clientfmt.fmt.pix.bytesperline = 0;
	dev->clientfmt.fmt.pix.sizeimage = 0;

//	Capture parameters 
	dev->capture.capability = V4L2_CAP_TIMEPERFRAME;
	dev->capture.capturemode = 0;
	dev->capture.extendedmode = 0;
	dev->capture.timeperframe = dev->videc.frame_period;
#endif
	capture_new_format(dev);

	return 0;
}

//==============================================================================
//	The functions below this point are only called during loading and
//	unloading of the driver.

//	D E V I C E   I N I A L I Z A T I O N   R O U T I N E S

//	These routines locate and enable the Videum, and initialize
//	the device structure. After enabling, the WAVI register access
//	functions and I2C functions will work.

//	Initialize v4l2_device fields
static int init_device_fields(struct videum_device *dev)
{
//	sprintf(dev->v.name, "Winnov Videum");
	sprintf(dev->v.name, "%s %s", dev->eeprom.szOemName, dev->eeprom.szProduct);
#ifdef V4L1
	dev->v.type = VID_TYPE_CAPTURE;
#else
	dev->v.type = V4L2_TYPE_CAPTURE;
#endif
	dev->v.minor = *unit_video;

	dev->v.open = videum_open;
	dev->v.close = videum_close;
	dev->v.read = videum_read;
	dev->v.write = videum_write;
	dev->v.ioctl = videum_ioctl;
	dev->v.mmap = videum_mmap;
	dev->v.poll = videum_poll;
	dev->v.initialize = videum_init_done;
	dev->v.priv = NULL;
	init_waitqueue_head(&dev->new_video_frame);
	return 1;
}

static int config_pcmcia_videum(struct videum_device *dev, int idev)
{	dev->sram_base = 0x00000;
//	dev->sram_size = 0x10000;
#define SR_SIZE 0x10000
	dev->irq = 0;/* We will use the jiffy timer interrupt to poll */
	return 1;
}

static void unconfig_pcmcia_videum(struct videum_device *dev)
{ }

static void unconfig_any_videum(struct videum_device *dev)
{	interrupt_disable(dev);
	capture_close(dev);
	unconfig_pcmcia_videum(dev);
	if (dev->is_registered)
	{	debug_msg("unconfig_any_videum: Calling v4l2_unregister_device.\n" );
#ifdef V4L1
		video_unregister_device(&(dev->v));
#else
		v4l2_unregister_device(&(dev->v));
#endif
		info_msg("Removed device %s\n", dev->v.name);
	}
	memset(dev, 0, sizeof(videum[0]));
}

static int config_any_videum(struct videum_device *dev)
{	static  int pcmcia	= 0;

	if (!config_pcmcia_videum(dev, pcmcia++))
		return 0;

	if (!eeprom_load(dev))
	{	err_msg("EEPROM read failure\n");
//		unconfig_any_videum(dev);
//		return 0;
	}

//	The WAVI registers are now accessible!
	wavi_initialize(dev);

//	SRAM memory map
	if (dev->eeprom.bBoardType == TYPE_VIDEUM)
		dev->sram_base = 0x10000;/* correction for early Videums */
	dev->sram_ucode		= dev->sram_base;
	dev->sram_linebuf	= dev->sram_ucode + UC_SIZE;
//	dev->sram_linebuf_size	= 0x00800;
#define SR_LB_SIZE 0x00800
	dev->sram_video		= dev->sram_linebuf;// + SR_LB_SIZE;
	dev->sram_video_size	= dev->sram_base + SR_SIZE //dev->sram_size 
						- dev->sram_video;

	init_device_fields(dev);

	if (!ccd_probe(dev))
	{	err_msg("Bad or unrecognized video decoder\n");
		/*		unconfig_any_videum(dev);
				return 0;*/
	}
	return 1;
}

//======================================================================
//	A dummy PCMCIA client driver

//	This is provided as an example of how to write an IO card client.
//	As written, it will function as a sort of generic point enabler,
//	configuring any card as that card's CIS specifies.

//	dummy_cs.c 1.10 1999/02/13 06:47:20

//	The contents of this file are subject to the Mozilla Public
//	License Version 1.0 (the "License"); you may not use this file
//	except in compliance with the License. You may obtain a copy of
//	the License at http://www.mozilla.org/MPL/

//	Software distributed under the License is distributed on an "AS
//	IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
//	implied. See the License for the specific language governing
//	rights and limitations under the License.

//	The initial developer of the original code is David A. Hinds
//	<dhinds@hyper.stanford.edu>.  Portions created by David A. Hinds
//	are Copyright (C) 1998 David A. Hinds.  All Rights Reserved.
//======================================================================

//	All the PCMCIA modules use PCMCIA_DEBUG to control debugging.  If
//	you do not define PCMCIA_DEBUG at all, all the debug code will be
//	left out.  If you compile with PCMCIA_DEBUG=0, the debug code will
//	be present but disabled -- but it can then be enabled for specific
//	modules at load time with a 'pc_debug=#' option to insmod.
#if 0
#ifdef PCMCIA_DEBUG
static int pc_debug = PCMCIA_DEBUG;
MODULE_PARM(pc_debug, "i");
#define DEBUG(n, args...) if (pc_debug>(n)) printk(KERN_DEBUG args);
#else
#define DEBUG(n, args...)
#endif
#else
#define DEBUG(n, args...) printk(KERN_DEBUG args);
static char *version = "wnv_cs.c 0.1.0 1999/03/24 (Chris Lahey)";
#endif

//====================================================================
//	Parameters that can be set with 'insmod'
//	The old way: bit map of interrupts to choose from
//	This means pick from 15, 14, 12, 11, 10, 9, 7, 5, 4, and 3
static u_int irq_mask = 0xdeb8;
//	Newer, simpler way of listing specific interrupts
static int irq_list[4] = { -1 };

MODULE_PARM(irq_mask, "i");
MODULE_PARM(irq_list, "1-4i");

//====================================================================
//	The event() function is this driver's Card Services event handler.
//	It will be called by Card Services when an appropriate card status
//	event is received.  The config() and release() entry points are
//	used to configure or release a socket, in response to card
//	insertion and ejection events.  They are invoked from the dummy
//	event handler. 

static void wnv_config(dev_link_t *link);
static void wnv_release(u_long arg);
static int wnv_event(event_t event, int priority,
			   event_callback_args_t *args);

//	The attach() and detach() entry points are used to create and destroy
//	"instances" of the driver, where each instance represents everything
//	needed to manage one actual PCMCIA card.

static dev_link_t *wnv_attach(void);
static void wnv_detach(dev_link_t *);

//	You'll also need to prototype all the functions that will actually
//	be used to talk to your device.  See 'pcmem_cs' for a good example
//	of a fully self-sufficient driver; the other drivers rely more or
//	less on other parts of the kernel.

//	A linked list of "instances" of the dummy device.  Each actual
//	PCMCIA card corresponds to one device instance, and is described
//	by one dev_link_t structure (defined in ds.h).

//	You may not want to use a linked list for this -- for example, the
//	memory card driver uses an array of dev_link_t pointers, where minor
//	device numbers are used to derive the corresponding array index.

static dev_link_t *dev_list = NULL;

//	A dev_link_t structure has fields for most things that are needed
//	to keep track of a socket, but there will usually be some device
//	specific information that also needs to be kept track of.  The
//	'priv' pointer in a dev_link_t structure can be used to point to
//	a device-specific private data structure, like this.

//	A driver needs to provide a dev_node_t structure for each device
//	on a card.  In some cases, there is only one device per card (for
//	example, ethernet cards, modems).  In other cases, there may be
//	many actual or logical devices (SCSI adapters, memory cards with
//	multiple partitions).  The dev_node_t structures need to be kept
//	in a linked list starting at the 'dev' field of a dev_link_t
//	structure.  We allocate them in the card's private data structure,
//	because they generally shouldn't be allocated dynamically.

//	In this case, we also provide a flag to indicate if a device is
//	"stopped" due to a power management event, or card ejection.  The
//	device IO routines can use a flag like this to throttle IO to a
//	card that is not ready to accept it.

//====================================================================
static void cs_error(client_handle_t handle, int func, int ret)
{	error_info_t err = { func, ret };
	CardServices(ReportError, handle, &err);
}

//======================================================================
//	dummy_attach() creates an "instance" of the driver, allocating
//	local data structures for one device.  The device is registered
//	with Card Services.
//	The dev_link structure is initialized, but we don't actually
//	configure the card at this point -- we wait until we receive a
//	card insertion event.
//======================================================================
static dev_link_t *wnv_attach(void)
{	client_reg_t client_reg;
	dev_link_t *link;
	local_info_t *local;
	int ret, i;

	DEBUG(2,"wnv_attach()\n");
//	Make sure that video0 exits
	devfs_mk_dir(NULL, "video0", NULL);

//	Initialize the dev_link_t structure
	link = kmalloc(sizeof(struct dev_link_t), GFP_KERNEL);
	memset(link, 0, sizeof(struct dev_link_t));
	link->release.function = &wnv_release;
	link->release.data = (u_long)link;

//	Interrupt setup
	link->irq.Attributes = IRQ_TYPE_EXCLUSIVE;
	link->irq.IRQInfo1 = IRQ_INFO2_VALID|IRQ_LEVEL_ID;
	if (irq_list[0] == -1)
		link->irq.IRQInfo2 = irq_mask;
	else
		for (i = 0; i < 4; i++)
			link->irq.IRQInfo2 |= 1 << irq_list[i];
	link->irq.Handler = NULL;
	
//	General socket configuration defaults can go here.  In this
//	client, we assume very little, and rely on the CIS for almost
//	everything.  In most clients, many details (i.e., number, sizes,
//	and attributes of IO windows) are fixed by the nature of the
//	device, and can be hard-wired here.
	link->conf.Attributes = 0;
	link->conf.Vcc = 50;
	link->conf.IntType = INT_MEMORY_AND_IO;

//	Allocate space for private device-specific data
	local = kmalloc(sizeof(local_info_t), GFP_KERNEL);
	memset(local, 0, sizeof(local_info_t));
	link->priv = local;
	
//	Register with Card Services
	link->next = dev_list;
	dev_list = link;
	client_reg.dev_info = &dev_info;
	client_reg.Attributes = INFO_IO_CLIENT | INFO_CARD_SHARE;
	client_reg.EventMask =
	CS_EVENT_CARD_INSERTION | CS_EVENT_CARD_REMOVAL |
	CS_EVENT_RESET_PHYSICAL | CS_EVENT_CARD_RESET |
	CS_EVENT_PM_SUSPEND | CS_EVENT_PM_RESUME;
	client_reg.event_handler = &wnv_event;
	client_reg.Version = 0x0210;
	client_reg.event_callback_args.client_data = link;
	DEBUG(2,"wnv_attach: Calling RegisterClient\n" );
	ret = CardServices(RegisterClient, &link->handle, &client_reg);
	if (ret != 0)
	{	DEBUG(0,"wnv_attach: Registration error\n" );
		cs_error(link->handle, RegisterClient, ret);
		DEBUG(2,"wnv_attach: Calling wnv_detach(%p).\n", link);
		wnv_detach(link);
		return NULL;
	}

	DEBUG(2,"Returning link=%p from wnv_attach.\n", link);
	return link;
}	// wnv_attach

//======================================================================
//	This deletes a driver "instance".  The device is de-registered
//	with Card Services.  If it has been released, all local data
//	structures are freed.  Otherwise, the structures will be freed
//	when the device is released.
//======================================================================
static void wnv_detach(dev_link_t *link)
{	dev_link_t **linkp;

	DEBUG(2,"wnv_detach(0x%p)\n", link);

//	Locate device structure
	for (linkp = &dev_list; *linkp; linkp = &(*linkp)->next)
		if (*linkp == link)
			break;
	if (*linkp == NULL)
	return;

//	If the device is currently configured and active, we won't
//	actually delete it yet.  Instead, it is marked so that when
//	the release() function is called, that will trigger a proper detach().
	if (link->state & DEV_CONFIG)
	{	DEBUG(1, "wnv_cs: detach postponed, '%s' still locked\n",
			link->dev->dev_name);
		link->state |= DEV_STALE_LINK;
		return;
	}

//	Break the link with Card Services
	if (link->handle)
		DEBUG(1, "wnv_detach: DeregisterClient=%d\n", 
			CardServices(DeregisterClient, link->handle) );
	
//	Unlink device structure, free pieces
	*linkp = link->next;
	if (link->priv)
		kfree(link->priv);
	kfree(link);

}	// wnv_detach

//======================================================================
//	wnv_config() is scheduled to run after a CARD_INSERTION event
//	is received, to configure the PCMCIA socket, and to make the
//	device available to the system.
//======================================================================
#define CS_CHECK(fn, args...) \
while ((last_ret=CardServices(last_fn=(fn),args))!=0) goto cs_failed

#define CFG_CHECK(fn, args...) \
if (CardServices(fn, args) != 0) goto next_entry

static void wnv_config(dev_link_t *link)
{//	client_handle_t handle = link->handle;
	local_info_t *dev = link->priv;
	win_req_t req;
	memreq_t map;
	int variable;
	int result;
	videum_device *videum;

	printk("wnv_config(0x%p)\n", link);
	printk( KERN_DEBUG "wnv_config: Memory Mapped\n" );
	videum =dev->device =kmalloc( sizeof( struct videum_device ), GFP_KERNEL );
	memset(videum, 0, sizeof( struct videum_device ) );
	videum->link = link;

	req.Attributes = WIN_DATA_WIDTH_16 | WIN_MEMORY_TYPE_CM | WIN_ENABLE;
	req.Base = 0;
//	req.Size = 0x2000;	///ZZZZZZZZZz
	req.Size = 0x4000;
	req.AccessSpeed = 0;
	link->win = (window_handle_t)link->handle;
	result = CardServices(RequestWindow, &link->win, &req);
	printk("wnv_config: RequestWindow=%d\n", result);
	map.Page = 0; map.CardOffset = 0;
	result = CardServices(MapMemPage, link->win, &map);
	printk("wnv_config: MapMemPage=%d\n", result);
//	videum->mem_start = ioremap(req.Base, 0x2000);
	videum->mem_start = ioremap(req.Base, 0x4000);
	videum->data = (__u16 *)(((long)videum->mem_start) + 0x1000);
//ZZZZZZZ
	if ( mp_cable_attached( videum ) )
	{	printk("wnv_config: Cable attached.\n");
		variable = 0;
		mp_power_on( videum );
		mp_reset_camera( videum );
		mp_light_on( videum );
	} else
		printk("wnv_config: Cable not attached.\n");

	config_any_videum( videum );
#ifdef V4L1
	if (video_register_device(&(videum->v), VFL_TYPE_GRABBER, -1))
#else
	if (v4l2_register_device(&(videum->v)))
#endif
		printk("Ah!\n" );
	else
		videum->is_registered = 1;

	link->state &= ~DEV_CONFIG_PENDING;

//	Configure card
	link->state |= DEV_CONFIG;

//	At this point, the dev_node_t structure(s) need to be
//	initialized and arranged in a linked list at link->dev.
	sprintf(dev->node.dev_name, "video0");
	dev->node.major = 81;
	dev->node.minor = 0;
	link->dev = &dev->node;

	return;
}	// wnv_config

//======================================================================
//	After a card is removed, wnv_release() will unregister the
//	device, and release the PCMCIA configuration.  If the device is
//	still open, this will be postponed until it is closed.
//======================================================================
static void wnv_release(u_long arg)
{	dev_link_t *link = (dev_link_t *)arg;
	local_info_t *dev = link->priv;
	int result;

	printk("wnv_release(0x%p)\n", link);

//	If the device is currently in use, we won't release until it
//	is actually closed, because until then, we can't be sure that
//	no one will try to access the device or its data structures.
	if (link->open)
	{	printk("wnv_cs: release postponed, '%s' still open\n",
			link->dev->dev_name);
		link->state |= DEV_STALE_CONFIG;
		return;
	}

//	Unlink the device chain
	link->dev = NULL;

//	In a normal driver, additional code may be needed to release
//	other kernel data structures associated with this device. 
	mp_power_off( dev->device );

	printk("Calling unconfig_any_videum\n");
	unconfig_any_videum( ((local_info_t *)link->priv)->device );

//	Don't bother checking to see if these succeed or not
	if (link->win)
	{	iounmap( ((local_info_t *)link->priv)->device->mem_start );
		result = CardServices(ReleaseWindow, link->win);
		printk("wnv_release: ReleaseWindow=%d.\n", result);
	}
	if (link->io.NumPorts1)
		CardServices(ReleaseIO, link->handle, &link->io);
	if (link->irq.AssignedIRQ)
		CardServices(ReleaseIRQ, link->handle, &link->irq);
	link->state &= ~DEV_CONFIG;

	if (link->state & DEV_STALE_LINK)
	{	printk("wnv_release: Calling wnv_detach(%p).\n", link);
		wnv_detach(link);
}	}	// wnv_release

//======================================================================
//	The card status event handler.  Mostly, this schedules other
//	stuff to run after an event is received.
//	When a CARD_REMOVAL event is received, we immediately set a
//	private flag to block future accesses to this device.  All the
//	functions that actually access the device should check this flag
//	to make sure the card is still present.
//======================================================================
static int wnv_event(event_t event, int priority, event_callback_args_t *args)
{	dev_link_t *link = args->client_data;

	printk("wnv_event(0x%06x, 0x%p)\n", event, link);

	switch (event)
	{	case CS_EVENT_CARD_REMOVAL:
			link->state &= ~DEV_PRESENT;
			if (link->state & DEV_CONFIG)
			{	((local_info_t *)link->priv)->stop = 1;
				link->release.expires = RUN_AT(HZ/20);
				add_timer(&link->release);
			}
		break;
		case CS_EVENT_CARD_INSERTION:
			link->state |= DEV_PRESENT | DEV_CONFIG_PENDING;
			wnv_config(link);
		break;
		case CS_EVENT_PM_SUSPEND:
			link->state |= DEV_SUSPEND;
//		Fall through...
		case CS_EVENT_RESET_PHYSICAL:
//			Mark the device as stopped, to block IO until later
			((local_info_t *)link->priv)->stop = 1;
			if (link->state & DEV_CONFIG)
				CardServices(ReleaseConfiguration, link->handle);
		break;
		case CS_EVENT_PM_RESUME:
			link->state &= ~DEV_SUSPEND;
//		Fall through...
		case CS_EVENT_CARD_RESET:
			if (link->state & DEV_CONFIG)
				CardServices(RequestConfiguration, link->handle, &link->conf);
			((local_info_t *)link->priv)->stop = 0;

//		In a normal driver, additional code may go here to restore
//		the device state and restart IO. 
		break;
	}
	return 0;
}	// wnv_event

int init_module(void)
{	servinfo_t serv;
	printk("%s\n", version);
	CardServices(GetCardServicesInfo, &serv);
	printk("init_module: Got Server\n");
	if (serv.Revision != CS_RELEASE_CODE)
	{	printk(KERN_NOTICE "wnv_cs: Card Services release does not match!\n");
		return -1;
	}
	printk("init_module: Server Matches\n");
	register_pcmcia_driver(&dev_info, &wnv_attach, &wnv_detach);
	printk("init_module: Driver Registered\n");
	return 0;
}

void cleanup_module(void)
{	printk("wnv_cs: unloading\n");
	unregister_pcmcia_driver(&dev_info);
	while (dev_list != NULL)
	{	if (dev_list->state & DEV_CONFIG)
			wnv_release((u_long)dev_list);
		printk("cleanup_module: Calling wnv_detach(%p).\n", dev_list);
		wnv_detach(dev_list);
}	}

MODULE_AUTHOR("Jim Zajkowski <jamesez@umich.edu>, Chris Lahey <clahey@umich.edu>");
MODULE_DESCRIPTION("Video4linux driver for Winnov Videum Traveler PCMCIA camera");
MODULE_LICENSE("GPL");

