#!/bin/bash
#
# script for driving all the other Matchbox test scripts
#
# To add a new test, write a file name 'test-<test name>.sh' to the
# directory where this script is. Other test-*.sh files can be used
# as examples. To run the test, just use testrun.sh -t <test name>

CONF_FILE="./test.conf"
ALL_TESTS=$(echo test-*.sh|sed -e 's/test-//g' -e 's/\.sh//g')


usage() {
	echo "Usage: $0 [OPTIONS]
Run matchbox regression test suite. 
Examples:
  $0 -c mysetup.conf -t interaction,windows
  $0 -d localhost:9

Parameters:
  -h, --help                display this help and exit
  -c, --config <conf file>  specify alternate conf file to ./test.conf
  -d, --display <:nro>      Use this display instead of launching Xnest
  -t, --tests  <tests>      pass a comma seperated list of tests to run.
			    Defaults to all tests

Available tests are: $ALL_TESTS

Report bugs to <mallum@handhelds.org>"
	exit 1	
}


# Process parameters
DO_TESTS=$ALL_TESTS
while [ "$1" ]; do
	case "$1" in
		--help|-h)
			usage
		;;
		--display|-d)
			shift
			USE_DPY=$1
			shift
		;;
		--config|-c)
			shift
			CONF_FILE=$1
			shift
		;;
		--tests|-t)
			shift
			DO_TESTS=$(echo $1|sed 's/,/ /')
			shift
		;;
		*)
			usage
	esac
done

if [ -z "$CONF_FILE" -o -z "$DO_TESTS" ]; then
   usage
fi

if [ ! -f $CONF_FILE ]; then
   echo "ERROR: Unable to load '$CONF_FILE'."
   exit 1
fi
source $CONF_FILE

if [ -z $(which matchbox|grep "/matchbox") ]; then
   echo "ERROR: 'matchbox' binaries missing!"
   echo
   echo "Either build/install Matchbox first or set the path correctly, e.g:"
   echo "PATH=\$PATH:$HOME/matchbox/bin $0"
   exit 1
fi

if [ -z $(which mbcontrol|grep "/mbcontrol") ]; then
   echo "ERROR: 'mbcontrol' needed for Matchbox theme setting is missing!"
   echo
   echo "Have you installed Matchbox properly?"
   exit 1
fi

if [ -z "$USE_DPY" ]; then
   # using Xnest
   if [ -z $(which Xnest|grep "/Xnest") ]; then
      echo "ERROR: 'Xnest' is missing!"
      echo
      echo "You have to install Xnest or give me a display with '-d'"
      echo "to be able to use the tests."
      exit 1
   fi
else
   # using existing display
   dpy_geom=$(xdpyinfo -display $USE_DPY|grep dimensions:)
   if [ -z "$dpy_geom" ]; then
      echo "ERROR: Can't connect to display $USE_DPY!"
      exit 1
   fi
   if [ -z "$(echo $dpy_geom|grep $XNEST_GEOM)" ]; then
      echo "ERROR: Display geometry isn't $XNEST_GEOM specified in $CONF_FILE!"
      exit 1
   fi
   DPY_NUM=$USE_DPY
fi

echo "$0 setup summary"
echo "==============================================="
echo

echo "  + Using '$CONF_FILE' for config"

if [ $VALGRIND_USAGE == "y" ]; then
   if [ -z $(which valgrind|grep "/valgrind") ]; then
      echo "ERROR: 'valgrind' not in path although it's use was requested!"
      exit 1
   fi
   VALGRIND="valgrind $VALGRIND_ARGS"
   echo "  + Running valgrind as $VALGRIND"
else
   VALGRIND=
   echo "  + Suppressing valgrind usage"
fi

echo "  + Running tests: $DO_TESTS"
if [ -z "$USE_DPY" ]; then
   echo "  + Running Xnest at $XNEST_GEOM on display $DPY_NUM"
else
   echo "  + Using display $DPY_NUM having $XNEST_GEOM geometry"
fi
echo "  + Running Matchbox using $THEME theme with:"
echo "    '$MATCHBOX_EXE $MATCHBOX_ARGS'"
echo "  + Sleep between (some) tests set to $SLEEP seconds"
echo

echo "==============================================="
echo
echo "Shall I start the tests (Y/n)"
read junk
if [ "$junk" == "n" ]; then 
   exit 
fi

# stuff needed by the test scripts
export LOGS SLEEP VALGRIND
mkdir -p $LOGS

if [ -z "$USE_DPY" ]; then
   Xnest $DPY_NUM -ac -geometry $XNEST_GEOM &
   XNESTPID=$!
   sleep 1
fi

DISPLAY=$DPY_NUM

$($VALGRIND $MATCHBOX_EXE $MATCHBOX_ARGS > $LOGS/mb-valgrind.log) &
MBPID=$!

mbcontrol -t $THEME

i=0
# initialize test states
for test in $ALL_TESTS; do
	TEST_STATE[$i]="Not run."
	i=$(($i+1))
done

function summary() 
{
	#clear	
	
	echo "================================================"
	echo
	echo "$0 finished. Test summary follows;"
	echo
	i=0
	for test in $ALL_TESTS; do
		echo -e "\t$test:\t${TEST_STATE[$i]}"
		i=$(($i+1))
	done
	echo 
	echo "Diffing valgrind output... (TODO) "
	echo "================================================"
	
	exit
}

## run the selected tests ##
i=0
for test in $ALL_TESTS; do
	do_test=$(echo $DO_TESTS|grep $test)
	if [ -z "$do_test" ]; then
		echo "Skipping test '$test'."
	else
		echo "Running test '$test'..."
		./test-$test.sh > $LOGS/test-$test.log
		RET=$?
		
		kill -0 $MBPID
		if [ $? -ne 0 ]; then 
			TEST_STATE[$i]="Failed, matchbox died???"
			summary
		fi
		
		if [ $RET -eq 0 ]; then     
			TEST_STATE[$i]="Succeeded."
		elif [ $RET -eq 2 ]; then
			TEST_STATE[$i]="Test setup problem, see $LOGS/test-$test.log for details"
		else
			TEST_STATE[$i]="Failed, see $LOGS/test-$test.log for details"
		fi
	fi
	i=$(($i+1))
done


if [ -z "$USE_DPY" ]; then
   kill $XNESTPID
   sleep 1
else
   kill -HUP $MBPID
   echo "Please close your Xnest so that all the test programs exit."
fi

summary
